# We have to override the new %%install behavior because, well... the kernel is special.
%global __spec_install_pre %{___build_pre}

# At the time of this writing (2019-03), RHEL8 packages use w2.xzdio
# compression for rpms (xz, level 2).
# Kernel has several large (hundreds of mbytes) rpms, they take ~5 mins
# to compress by single-threaded xz. Switch to threaded compression,
# and from level 2 to 3 to keep compressed sizes close to "w2" results.
#
# NB: if default compression in /usr/lib/rpm/redhat/macros ever changes,
# this one might need tweaking (e.g. if default changes to w3.xzdio,
# change below to w4T.xzdio):
%define _binary_payload w3T.xzdio

# For a kernel released for public testing, released_kernel should be 1.
# For internal testing builds during development, it should be 0.
%global released_kernel 1

%global distro_build 348

# Sign the x86_64 kernel for secure boot authentication
%ifarch x86_64 aarch64
%global signkernel 1
%else
%global signkernel 0
%endif

# Sign modules on all arches
%global signmodules 1

# Compress modules only for architectures that build modules
%ifarch noarch
%global zipmodules 0
%else
%global zipmodules 1
%endif

%if %{zipmodules}
%global zipsed -e 's/\.ko$/\.ko.xz/'
%endif

# define buildid .local

%define rpmversion 4.18.0
%define pkgrelease 348.2.1.el8_5

# allow pkg_release to have configurable %%{?dist} tag
%define specrelease 348.2.1%{?dist}

%define pkg_release %{specrelease}%{?buildid}

# libexec dir is not used by the linker, so the shared object there
# should not be exported to RPM provides
%global __provides_exclude_from ^%{_libexecdir}/kselftests

# What parts do we want to build? These are the kernels that are built IF the
# architecture allows it. All should default to 1 (enabled) and be flipped to
# 0 (disabled) by later arch-specific checks.

%define _with_kabidupchk 1
# The following build options are enabled by default.
# Use either --without <opt> in your rpmbuild command or force values
# to 0 in here to disable them.
#
# standard kernel
%define with_up        %{?_without_up:        0} %{?!_without_up:        1}
# kernel-debug
%define with_debug     %{?_without_debug:     0} %{?!_without_debug:     1}
# kernel-doc
%define with_doc       %{?_without_doc:       0} %{?!_without_doc:       1}
# realtime
# rhel doesn't incorporate PREEMPT_RT patches, but rhel-rt builds use base
# kernel spec as a template.
%define with_realtime 0
# kernel-headers
%define with_headers   %{?_without_headers:   0} %{?!_without_headers:   1}
%define with_cross_headers   %{?_without_cross_headers:   0} %{?!_without_cross_headers:   1}
# perf
%define with_perf      %{?_without_perf:      0} %{?!_without_perf:      1}
# tools
%define with_tools     %{?_without_tools:     0} %{?!_without_tools:     1}
# bpf tool
%define with_bpftool   %{?_without_bpftool:   0} %{?!_without_bpftool:   1}
# kernel-debuginfo
%define with_debuginfo %{?_without_debuginfo: 0} %{?!_without_debuginfo: 1}
# Want to build a the vsdo directories installed
%define with_vdso_install %{?_without_vdso_install: 0} %{?!_without_vdso_install: 1}
# kernel-zfcpdump (s390 specific kernel for zfcpdump)
%define with_zfcpdump  %{?_without_zfcpdump:  0} %{?!_without_zfcpdump:  1}
# kernel-abi-stablelists
%define with_kernel_abi_stablelists %{?_without_kernel_abi_stablelists: 0} %{?!_without_kernel_abi_stablelists: 1}
# internal samples and selftests
%define with_selftests %{?_without_selftests: 0} %{?!_without_selftests: 1}
#
# Additional options for user-friendly one-off kernel building:
#
# Only build the base kernel (--with baseonly):
%define with_baseonly  %{?_with_baseonly:     1} %{?!_with_baseonly:     0}
# Only build the debug kernel (--with dbgonly):
%define with_dbgonly   %{?_with_dbgonly:      1} %{?!_with_dbgonly:      0}
# Control whether we perform a compat. check against published ABI.
%define with_kabichk   %{?_without_kabichk:   0} %{?!_without_kabichk:   1}
# Control whether we perform a compat. check against DUP ABI.
%define with_kabidupchk %{?_with_kabidupchk:  1} %{?!_with_kabidupchk:   0}
#
# Control whether to run an extensive DWARF based kABI check.
# Note that this option needs to have baseline setup in SOURCE300.
%define with_kabidwchk %{?_without_kabidwchk: 0} %{?!_without_kabidwchk: 1}
%define with_kabidw_base %{?_with_kabidw_base: 1} %{?!_with_kabidw_base: 0}
#
# should we do C=1 builds with sparse
%define with_sparse    %{?_with_sparse:       1} %{?!_with_sparse:       0}
#
# Cross compile requested?
%define with_cross    %{?_with_cross:         1} %{?!_with_cross:        0}
#
# build a release kernel on rawhide
%define with_release   %{?_with_release:      1} %{?!_with_release:      0}

# The kernel tarball/base version
%define kversion 4.18

%define with_gcov %{?_with_gcov:1}%{?!_with_gcov:0}

# turn off debug kernel and kabichk for gcov builds
%if %{with_gcov}
%define with_debug 0
%define with_kabichk 0
%define with_kabidupchk 0
%define with_kabidwchk 0
%define with_kabidw_base 0
%define with_kernel_abi_stablelists 0
%endif

# turn off kABI DWARF-based check if we're generating the base dataset
%if %{with_kabidw_base}
%define with_kabidwchk 0
%endif

# kpatch_kcflags are extra compiler flags applied to base kernel
# -fdump-ipa-clones is enabled only for base kernels on selected arches
%define with_ipaclones %{?_without_ipaclones: 0} %{?!_without_ipaclones: 1}
%if %{with_ipaclones}
%ifarch x86_64 ppc64le
%define kpatch_kcflags -fdump-ipa-clones
%else
%define with_ipaclones 0
%endif
%endif

%if %{with_realtime}
# The preempt RT patch level
%global rttag %%RTTAG%%
# realtimeN
%global rtbuild %%RTBUILD%%
%define with_doc 0
%define with_headers 0
%define with_cross_headers 0
%define with_perf 0
%define with_tools 0
%define with_bpftool 0
%define with_zfcpdump 0
%define with_kabichk  0
%define with_kernel_abi_stablelists 0
%define with_bpf_samples 0
%define with_kabidw_base 0
%define with_ipaclones 0
%endif

%define make_target bzImage
%define image_install_path boot

%define KVERREL %{version}-%{release}.%{_target_cpu}
%define KVERREL_RE %(echo %KVERREL | sed 's/+/[+]/g')
%define hdrarch %_target_cpu
%define asmarch %_target_cpu

%if !%{with_debuginfo}
%define _enable_debug_packages 0
%endif
%define debuginfodir /usr/lib/debug
# Needed because we override almost everything involving build-ids
# and debuginfo generation. Currently we rely on the old alldebug setting.
%global _build_id_links alldebug

# if requested, only build base kernel
%if %{with_baseonly}
%define with_debug 0
%define with_vdso_install 0
%define with_perf 0
%define with_tools 0
%define with_bpftool 0
%define with_kernel_abi_stablelists 0
%define with_selftests 0
%define with_cross 0
%define with_cross_headers 0
%define with_ipaclones 0
%endif

# if requested, only build debug kernel
%if %{with_dbgonly}
%define with_up 0
%define with_vdso_install 0
%define with_perf 0
%define with_tools 0
%define with_bpftool 0
%define with_kernel_abi_stablelists 0
%define with_selftests 0
%define with_cross 0
%define with_cross_headers 0
%define with_ipaclones 0
%endif

# turn off kABI DUP check and DWARF-based check if kABI check is disabled
%if !%{with_kabichk}
%define with_kabidupchk 0
%define with_kabidwchk 0
%endif

# selftests require bpftool to be built
%if %{with_selftests}
%define with_bpftool 1
%endif

%ifnarch noarch
%define with_kernel_abi_stablelists 0
%endif

# Overrides for generic default options

# only package docs noarch
%ifnarch noarch
%define with_doc 0
%define doc_build_fail true
%endif

# don't build noarch kernels or headers (duh)
%ifarch noarch
%define with_up 0
%define with_headers 0
%define with_cross_headers 0
%define with_tools 0
%define with_perf 0
%define with_bpftool 0
%define with_selftests 0
%define with_debug 0
%define all_arch_configs %{name}-%{version}-*.config
%endif

# sparse blows up on ppc
%ifnarch ppc64le
%define with_sparse 0
%endif

# zfcpdump mechanism is s390 only
%ifnarch s390x
%define with_zfcpdump 0
%endif

# Per-arch tweaks

%ifarch i686
%define asmarch x86
%define hdrarch i386
%endif

%ifarch x86_64
%define asmarch x86
%define all_arch_configs %{name}-%{version}-x86_64*.config
%define kernel_image arch/x86/boot/bzImage
%endif

%ifarch ppc64le
%define asmarch powerpc
%define hdrarch powerpc
%define make_target vmlinux
%define kernel_image vmlinux
%define kernel_image_elf 1
%define all_arch_configs %{name}-%{version}-ppc64le*.config
%define kcflags -O3
%endif

%ifarch s390x
%define asmarch s390
%define hdrarch s390
%define all_arch_configs %{name}-%{version}-s390x.config
%define kernel_image arch/s390/boot/bzImage
%define vmlinux_decompressor arch/s390/boot/compressed/vmlinux
%endif

%ifarch aarch64
%define all_arch_configs %{name}-%{version}-aarch64*.config
%define asmarch arm64
%define hdrarch arm64
%define make_target Image.gz
%define kernel_image arch/arm64/boot/Image.gz
%endif

# To temporarily exclude an architecture from being built, add it to
# %%nobuildarches. Do _NOT_ use the ExclusiveArch: line, because if we
# don't build kernel-headers then the new build system will no longer let
# us use the previous build of that package -- it'll just be completely AWOL.
# Which is a BadThing(tm).

# We only build kernel-headers on the following...
%define nobuildarches i386 i686

%ifarch %nobuildarches
%define with_up 0
%define with_debug 0
%define with_debuginfo 0
%define with_perf 0
%define with_tools 0
%define with_bpftool 0
%define with_selftests 0
%define _enable_debug_packages 0
%endif

# Architectures we build tools/cpupower on
%define cpupowerarchs x86_64 ppc64le aarch64


#
# Packages that need to be installed before the kernel is, because the %%post
# scripts use them.
#
%define kernel_prereq  coreutils, systemd >= 203-2, /usr/bin/kernel-install
%define initrd_prereq  dracut >= 027


Name: kernel%{?variant}
Group: System Environment/Kernel
License: GPLv2 and Redistributable, no modification permitted
URL: http://www.kernel.org/
Version: %{rpmversion}
Release: %{pkg_release}
Summary: The Linux kernel, based on version %{version}, heavily modified with backports
%if %{with_realtime}
ExclusiveArch: x86_64
%else
# DO NOT CHANGE THE 'ExclusiveArch' LINE TO TEMPORARILY EXCLUDE AN ARCHITECTURE BUILD.
# SET %%nobuildarches (ABOVE) INSTEAD
ExclusiveArch: noarch i386 i686 x86_64 s390x aarch64 ppc64le
%endif
ExclusiveOS: Linux
%ifnarch %{nobuildarches}
Requires: %{name}-core-uname-r = %{KVERREL}%{?variant}
Requires: %{name}-modules-uname-r = %{KVERREL}%{?variant}
%if %{with_realtime}
Requires: rt-setup
%endif
%endif


#
# List the packages used during the kernel build
#
BuildRequires: kmod, patch, bash, coreutils, tar, git, which
BuildRequires: bzip2, xz, findutils, gzip, m4, perl-interpreter, perl-Carp, perl-devel, perl-generators, make, diffutils, gawk
BuildRequires: gcc, binutils, redhat-rpm-config, hmaccalc, python3-devel
BuildRequires: net-tools, hostname, bc, bison, flex, elfutils-devel, dwarves
%if %{with_doc}
BuildRequires: xmlto, asciidoc, python3-sphinx
%endif
%if %{with_sparse}
BuildRequires: sparse
%endif
%if %{with_perf}
BuildRequires: zlib-devel binutils-devel newt-devel perl(ExtUtils::Embed) bison flex xz-devel
BuildRequires: audit-libs-devel
BuildRequires: java-devel
BuildRequires: libbpf-devel
BuildRequires: libbabeltrace-devel
%ifnarch s390x
BuildRequires: numactl-devel
%endif
%ifarch aarch64
BuildRequires: opencsd-devel >= 1.0.0-2
%endif
%endif
%if %{with_tools}
BuildRequires: gettext ncurses-devel
%ifnarch s390x
BuildRequires: pciutils-devel
%endif
%endif
%if %{with_bpftool}
BuildRequires: python3-docutils
BuildRequires: zlib-devel binutils-devel
%endif
%if %{with_selftests}
BuildRequires: libcap-devel libcap-ng-devel clang llvm numactl-devel rsync libmnl-devel
%endif
BuildConflicts: rhbuildsys(DiskFree) < 500Mb
%if %{with_debuginfo}
BuildRequires: rpm-build, elfutils
BuildConflicts: rpm < 4.13.0.1-19
BuildConflicts: dwarves < 1.13
# Most of these should be enabled after more investigation
%undefine _include_minidebuginfo
%undefine _find_debuginfo_dwz_opts
%undefine _unique_build_ids
%undefine _unique_debug_names
%undefine _unique_debug_srcs
%undefine _debugsource_packages
%undefine _debuginfo_subpackages
%global _find_debuginfo_opts -r
%global _missing_build_ids_terminate_build 1
%global _no_recompute_build_ids 1
%endif
%if %{with_kabidwchk} || %{with_kabidw_base}
BuildRequires: kabi-dw
%endif

%if %{signkernel}%{signmodules}
BuildRequires: openssl openssl-devel
%if %{signkernel}
%ifarch x86_64 aarch64
BuildRequires: nss-tools
BuildRequires: pesign >= 0.10-4
%endif
%endif
%endif

%if %{with_cross}
BuildRequires: binutils-%{_build_arch}-linux-gnu, gcc-%{_build_arch}-linux-gnu
%define cross_opts CROSS_COMPILE=%{_build_arch}-linux-gnu-
%endif

# These below are required to build man pages
%if %{with_perf}
BuildRequires: xmlto
%endif
%if %{with_perf} || %{with_tools}
BuildRequires: asciidoc
%endif

Source0: linux-%{rpmversion}-%{pkgrelease}.tar.xz

Source9: x509.genkey

# Name of the packaged file containing signing key
%ifarch ppc64le
%define signing_key_filename kernel-signing-ppc.cer
%endif
%ifarch s390x
%define signing_key_filename kernel-signing-s390.cer
%endif

%if %{?released_kernel}

Source10: centossecurebootca2.der
Source11: centos-ca-secureboot.der
Source12: centossecureboot201.der
Source13: centossecureboot001.der

%define secureboot_ca_0 %{SOURCE11}
%define secureboot_ca_1 %{SOURCE10}
%ifarch x86_64 aarch64
%define secureboot_key_0 %{SOURCE13}
%define pesign_name_0 centossecureboot001
%define secureboot_key_1 %{SOURCE12}
%define pesign_name_1 centossecureboot201
%endif

# released_kernel
%else

Source11: centossecurebootca2.der
Source12: centos-ca-secureboot.der
Source13: centossecureboot201.der
Source14: centossecureboot001.der

%define secureboot_ca_0 %{SOURCE12}
%define secureboot_ca_1 %{SOURCE11}
%define secureboot_key_0 %{SOURCE14}
%define pesign_name_0 centossecureboot001
%define secureboot_key_1 %{SOURCE13}
%define pesign_name_1 centossecureboot201

# released_kernel
%endif

Source16: mod-extra.list
Source17: mod-blacklist.sh
Source18: mod-sign.sh
Source80: parallel_xz.sh
Source90: filter-x86_64.sh
Source93: filter-aarch64.sh
Source96: filter-ppc64le.sh
Source97: filter-s390x.sh
Source99: filter-modules.sh
%define modsign_cmd %{SOURCE18}

Source20: %{name}-aarch64.config
Source21: %{name}-aarch64-debug.config
Source32: %{name}-ppc64le.config
Source33: %{name}-ppc64le-debug.config
Source36: %{name}-s390x.config
Source37: %{name}-s390x-debug.config
Source38: %{name}-s390x-zfcpdump.config
Source39: %{name}-x86_64.config
Source40: %{name}-x86_64-debug.config
Source41: generate_all_configs.sh

Source42: process_configs.sh
Source43: generate_bls_conf.sh

Source44: mod-internal.list

Source100: centos-dup.x509
Source101: centos-kpatch.x509

%if %{with_kabichk}
Source200: check-kabi

Source201: Module.kabi_aarch64
Source202: Module.kabi_ppc64le
Source203: Module.kabi_s390x
Source204: Module.kabi_x86_64

Source210: Module.kabi_dup_aarch64
Source211: Module.kabi_dup_ppc64le
Source212: Module.kabi_dup_s390x
Source213: Module.kabi_dup_x86_64

Source300: kernel-abi-stablelists-%{rpmversion}-%{distro_build}.tar.bz2
Source301: kernel-kabi-dw-%{rpmversion}-%{distro_build}.tar.bz2
%endif

%if %{with_realtime}
Source400: mod-kvm.list
%endif

# Sources for kernel-tools
Source2000: cpupower.service
Source2001: cpupower.config
Source2002: kvm_stat.logrotate

Source9000: centos.pem

## Patches needed for building this package

# empty final patch to facilitate testing of kernel patches
Patch999999: linux-kernel-test.patch

# END OF PATCH DEFINITIONS

BuildRoot: %{_tmppath}/%{name}-%{KVERREL}-root

%description
This is the package which provides the Linux %{name} for CentOS.
It is based on upstream Linux at version %{version} and maintains kABI
compatibility of a set of approved symbols, however it is heavily modified with
backports and fixes pulled from newer upstream Linux %{name} releases. This means
this is not a %{version} kernel anymore: it includes several components which come
from newer upstream linux versions, while maintaining a well tested and stable
core. Some of the components/backports that may be pulled in are: changes like
updates to the core kernel (eg.: scheduler, cgroups, memory management, security
fixes and features), updates to block layer, supported filesystems, major driver
updates for supported hardware in CentOS, enhancements for
enterprise customers, etc.

#
# This macro does requires, provides, conflicts, obsoletes for a kernel package.
#	%%kernel_reqprovconf <subpackage>
# It uses any kernel_<subpackage>_conflicts and kernel_<subpackage>_obsoletes
# macros defined above.
#
%define kernel_reqprovconf \
Provides: %{name} = %{rpmversion}-%{pkg_release}\
Provides: %{name}-%{_target_cpu} = %{rpmversion}-%{pkg_release}%{?1:+%{1}}\
Provides: kernel-drm-nouveau = 16\
Provides: %{name}-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
Requires(pre): %{kernel_prereq}\
Requires(pre): %{initrd_prereq}\
Requires(pre): linux-firmware >= 20200619-99.git3890db36\
Requires(preun): systemd >= 200\
Conflicts: xfsprogs < 4.3.0-1\
Conflicts: xorg-x11-drv-vmmouse < 13.0.99\
Conflicts: kexec-tools < 2.0.20-8\
%{expand:%%{?kernel%{?1:_%{1}}_conflicts:Conflicts: %%{kernel%{?1:_%{1}}_conflicts}}}\
%{expand:%%{?kernel%{?1:_%{1}}_obsoletes:Obsoletes: %%{kernel%{?1:_%{1}}_obsoletes}}}\
%{expand:%%{?kernel%{?1:_%{1}}_provides:Provides: %%{kernel%{?1:_%{1}}_provides}}}\
# We can't let RPM do the dependencies automatic because it'll then pick up\
# a correct but undesirable perl dependency from the module headers which\
# isn't required for the kernel proper to function\
AutoReq: no\
AutoProv: yes\
%{nil}


%package doc
Summary: Various documentation bits found in the kernel source
Group: Documentation
%description doc
This package contains documentation files from the kernel
source. Various bits of information about the Linux kernel and the
device drivers shipped with it are documented in these files.

You'll want to install this package if you need a reference to the
options that can be passed to Linux kernel modules at load time.


%package headers
Summary: Header files for the Linux kernel for use by glibc
Group: Development/System
Obsoletes: glibc-kernheaders < 3.0-46
Provides: glibc-kernheaders = 3.0-46
%if "0%{?variant}"
Obsoletes: kernel-headers < %{rpmversion}-%{pkg_release}
Provides: kernel-headers = %{rpmversion}-%{pkg_release}
%endif
%description headers
Kernel-headers includes the C header files that specify the interface
between the Linux kernel and userspace libraries and programs.  The
header files define structures and constants that are needed for
building most standard programs and are also needed for rebuilding the
glibc package.

%package cross-headers
Summary: Header files for the Linux kernel for use by cross-glibc
Group: Development/System
%description cross-headers
Kernel-cross-headers includes the C header files that specify the interface
between the Linux kernel and userspace libraries and programs.  The
header files define structures and constants that are needed for
building most standard programs and are also needed for rebuilding the
cross-glibc package.


%package debuginfo-common-%{_target_cpu}
Summary: Kernel source files used by %{name}-debuginfo packages
Group: Development/Debug
Provides: installonlypkg(kernel)
%description debuginfo-common-%{_target_cpu}
This package is required by %{name}-debuginfo subpackages.
It provides the kernel source files common to all builds.

%if %{with_perf}
%package -n perf
Summary: Performance monitoring for the Linux kernel
Group: Development/System
Requires: bzip2
License: GPLv2
%description -n perf
This package contains the perf tool, which enables performance monitoring
of the Linux kernel.

%package -n perf-debuginfo
Summary: Debug information for package perf
Group: Development/Debug
Requires: %{name}-debuginfo-common-%{_target_cpu} = %{version}-%{release}
AutoReqProv: no
%description -n perf-debuginfo
This package provides debug information for the perf package.

# Note that this pattern only works right to match the .build-id
# symlinks because of the trailing nonmatching alternation and
# the leading .*, because of find-debuginfo.sh's buggy handling
# of matching the pattern against the symlinks file.
%{expand:%%global _find_debuginfo_opts %{?_find_debuginfo_opts} -p '.*%%{_bindir}/perf(\.debug)?|.*%%{_libexecdir}/perf-core/.*|.*%%{_libdir}/traceevent/plugins/.*|.*%%{_libdir}/libperf-jvmti.so(\.debug)?|XXX' -o perf-debuginfo.list}

%package -n python3-perf
Summary: Python bindings for apps which will manipulate perf events
Group: Development/Libraries
%description -n python3-perf
The python3-perf package contains a module that permits applications
written in the Python programming language to use the interface
to manipulate perf events.

%package -n python3-perf-debuginfo
Summary: Debug information for package perf python bindings
Group: Development/Debug
Requires: %{name}-debuginfo-common-%{_target_cpu} = %{version}-%{release}
AutoReqProv: no
%description -n python3-perf-debuginfo
This package provides debug information for the perf python bindings.

# the python_sitearch macro should already be defined from above
%{expand:%%global _find_debuginfo_opts %{?_find_debuginfo_opts} -p '.*%%{python3_sitearch}/perf.*so(\.debug)?|XXX' -o python3-perf-debuginfo.list}

# with_perf
%endif

%if %{with_tools}
%package -n %{name}-tools
Summary: Assortment of tools for the Linux kernel
Group: Development/System
License: GPLv2
%ifarch %{cpupowerarchs}
Provides:  cpupowerutils = 1:009-0.6.p1
Obsoletes: cpupowerutils < 1:009-0.6.p1
Provides:  cpufreq-utils = 1:009-0.6.p1
Provides:  cpufrequtils = 1:009-0.6.p1
Obsoletes: cpufreq-utils < 1:009-0.6.p1
Obsoletes: cpufrequtils < 1:009-0.6.p1
Obsoletes: cpuspeed < 1:1.5-16
Requires: %{name}-tools-libs = %{version}-%{release}
%endif
%define __requires_exclude ^%{_bindir}/python
%description -n %{name}-tools
This package contains the tools/ directory from the kernel source
and the supporting documentation.

%package -n %{name}-tools-libs
Summary: Libraries for the %{name}-tools
Group: Development/System
License: GPLv2
%description -n %{name}-tools-libs
This package contains the libraries built from the tools/ directory
from the kernel source.

%package -n %{name}-tools-libs-devel
Summary: Assortment of tools for the Linux kernel
Group: Development/System
License: GPLv2
Requires: %{name}-tools = %{version}-%{release}
%ifarch %{cpupowerarchs}
Provides:  cpupowerutils-devel = 1:009-0.6.p1
Obsoletes: cpupowerutils-devel < 1:009-0.6.p1
%endif
Requires: %{name}-tools-libs = %{version}-%{release}
Provides: %{name}-tools-devel
%description -n %{name}-tools-libs-devel
This package contains the development files for the tools/ directory from
the kernel source.

%package -n %{name}-tools-debuginfo
Summary: Debug information for package %{name}-tools
Group: Development/Debug
Requires: %{name}-debuginfo-common-%{_target_cpu} = %{version}-%{release}
AutoReqProv: no
%description -n %{name}-tools-debuginfo
This package provides debug information for package %{name}-tools.

# Note that this pattern only works right to match the .build-id
# symlinks because of the trailing nonmatching alternation and
# the leading .*, because of find-debuginfo.sh's buggy handling
# of matching the pattern against the symlinks file.
%{expand:%%global _find_debuginfo_opts %{?_find_debuginfo_opts} -p '.*%%{_bindir}/centrino-decode(\.debug)?|.*%%{_bindir}/powernow-k8-decode(\.debug)?|.*%%{_bindir}/cpupower(\.debug)?|.*%%{_libdir}/libcpupower.*|.*%%{_bindir}/turbostat(\.debug)?|.*%%{_bindir}/x86_energy_perf_policy(\.debug)?|.*%%{_bindir}/tmon(\.debug)?|.*%%{_bindir}/lsgpio(\.debug)?|.*%%{_bindir}/gpio-hammer(\.debug)?|.*%%{_bindir}/gpio-event-mon(\.debug)?|.*%%{_bindir}/iio_event_monitor(\.debug)?|.*%%{_bindir}/iio_generic_buffer(\.debug)?|.*%%{_bindir}/lsiio(\.debug)?|.*%%{_bindir}/intel-speed-select(\.debug)?|.*%%{_bindir}/page_owner_sort(\.debug)?|.*%%{_bindir}/slabinfo(\.debug)?|XXX' -o %{name}-tools-debuginfo.list}

# with_tools
%endif

%if !%{with_realtime}
%if %{with_bpftool}

%package -n bpftool
Summary: Inspection and simple manipulation of eBPF programs and maps
License: GPLv2
%description -n bpftool
This package contains the bpftool, which allows inspection and simple
manipulation of eBPF programs and maps.

%package -n bpftool-debuginfo
Summary: Debug information for package bpftool
Group: Development/Debug
Requires: %{name}-debuginfo-common-%{_target_cpu} = %{version}-%{release}
AutoReqProv: no
%description -n bpftool-debuginfo
This package provides debug information for the bpftool package.

%{expand:%%global _find_debuginfo_opts %{?_find_debuginfo_opts} -p '.*%%{_sbindir}/bpftool(\.debug)?|XXX' -o bpftool-debuginfo.list}

# with_bpftool
%endif
%endif

%if %{with_selftests}

%package selftests-internal
Summary: Kernel samples and selftests
License: GPLv2
Requires: binutils, bpftool, iproute-tc, nmap-ncat, python3
%if %{with_realtime}
Conflicts: kernel-selftests-internal
%else
Conflicts: kernel-rt-selftests-internal
%endif
%description selftests-internal
Kernel sample programs and selftests.

# Note that this pattern only works right to match the .build-id
# symlinks because of the trailing nonmatching alternation and
# the leading .*, because of find-debuginfo.sh's buggy handling
# of matching the pattern against the symlinks file.
%{expand:%%global _find_debuginfo_opts %{?_find_debuginfo_opts} -p '.*%%{_libexecdir}/(ksamples|kselftests)/.*|XXX' -o selftests-debuginfo.list}

# with_selftests
%endif

%if %{with_gcov}
%package gcov
Summary: gcov graph and source files for coverage data collection.
Group: Development/System
%description gcov
kernel-gcov includes the gcov graph and source files for gcov coverage collection.
%endif

%package -n %{name}-abi-stablelists
Summary: The CentOS kernel ABI symbol stablelists
Group: System Environment/Kernel
AutoReqProv: no
Obsoletes: %{name}-abi-whitelists < %{rpmversion}-%{pkg_release}
Provides: %{name}-abi-whitelists
%description -n %{name}-abi-stablelists
The kABI package contains information pertaining to the CentOS
kernel ABI, including lists of kernel symbols that are needed by
external Linux kernel modules, and a yum plugin to aid enforcement.

%if %{with_kabidw_base}
%package kernel-kabidw-base-internal
Summary: The baseline dataset for kABI verification using DWARF data
Group: System Environment/Kernel
AutoReqProv: no
%description kernel-kabidw-base-internal
The package contains data describing the current ABI of the CentOS
kernel, suitable for the kabi-dw tool.
%endif

#
# This macro creates a kernel-<subpackage>-debuginfo package.
#	%%kernel_debuginfo_package <subpackage>
#
%define kernel_debuginfo_package() \
%package %{?1:%{1}-}debuginfo\
Summary: Debug information for package %{name}%{?1:-%{1}}\
Group: Development/Debug\
Requires: %{name}-debuginfo-common-%{_target_cpu} = %{version}-%{release}\
Provides: %{name}%{?1:-%{1}}-debuginfo-%{_target_cpu} = %{version}-%{release}\
Provides: installonlypkg(kernel)\
AutoReqProv: no\
%description %{?1:%{1}-}debuginfo\
This package provides debug information for package %{name}%{?1:-%{1}}.\
This is required to use SystemTap with %{name}%{?1:-%{1}}-%{KVERREL}.\
%{expand:%%global _find_debuginfo_opts %{?_find_debuginfo_opts} -p '/.*/%%{KVERREL_RE}%{?1:[+]%{1}}/.*|/.*%%{KVERREL_RE}%{?1:\+%{1}}(\.debug)?' -o debuginfo%{?1}.list}\
%{nil}

#
# This macro creates a kernel-<subpackage>-devel package.
#	%%kernel_devel_package <subpackage> <pretty-name>
#
%define kernel_devel_package() \
%package %{?1:%{1}-}devel\
Summary: Development package for building kernel modules to match the %{?2:%{2} }kernel\
Group: System Environment/Kernel\
Provides: %{name}%{?1:-%{1}}-devel-%{_target_cpu} = %{version}-%{release}\
Provides: %{name}-devel-%{_target_cpu} = %{version}-%{release}%{?1:+%{1}}\
Provides: %{name}-devel-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
Provides: installonlypkg(kernel)\
AutoReqProv: no\
Requires(pre): findutils\
Requires: findutils\
Requires: perl-interpreter\
%description %{?1:%{1}-}devel\
This package provides kernel headers and makefiles sufficient to build modules\
against the %{?2:%{2} }kernel package.\
%{nil}

#
# kernel-<variant>-ipaclones-internal package
#
%define kernel_ipaclones_package() \
%package %{?1:%{1}-}ipaclones-internal\
Summary: *.ipa-clones files generated by -fdump-ipa-clones for kernel%{?1:-%{1}}\
Group: System Environment/Kernel\
AutoReqProv: no\
%description %{?1:%{1}-}ipaclones-internal\
This package provides *.ipa-clones files.\
%{nil}

#
# This macro creates a kernel-<subpackage>-modules-internal package.
#	%%kernel_modules_internal_package <subpackage> <pretty-name>
#
%define kernel_modules_internal_package() \
%package %{?1:%{1}-}modules-internal\
Summary: Extra kernel modules to match the %{?2:%{2} }kernel\
Group: System Environment/Kernel\
Provides: %{name}%{?1:-%{1}}-modules-internal-%{_target_cpu} = %{version}-%{release}\
Provides: %{name}%{?1:-%{1}}-modules-internal-%{_target_cpu} = %{version}-%{release}%{?1:+%{1}}\
Provides: %{name}%{?1:-%{1}}-modules-internal = %{version}-%{release}%{?1:+%{1}}\
Provides: installonlypkg(kernel-module)\
Provides: %{name}%{?1:-%{1}}-modules-internal-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
Requires: %{name}-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
Requires: %{name}%{?1:-%{1}}-modules-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
AutoReq: no\
AutoProv: yes\
%description %{?1:%{1}-}modules-internal\
This package provides kernel modules for the %{?2:%{2} }kernel package for CentOS internal usage.\
%{nil}

#
# This macro creates a kernel-<subpackage>-modules-extra package.
#	%%kernel_modules_extra_package <subpackage> <pretty-name>
#
%define kernel_modules_extra_package() \
%package %{?1:%{1}-}modules-extra\
Summary: Extra kernel modules to match the %{?2:%{2} }kernel\
Group: System Environment/Kernel\
Provides: %{name}%{?1:-%{1}}-modules-extra-%{_target_cpu} = %{version}-%{release}\
Provides: %{name}%{?1:-%{1}}-modules-extra-%{_target_cpu} = %{version}-%{release}%{?1:+%{1}}\
Provides: %{name}%{?1:-%{1}}-modules-extra = %{version}-%{release}%{?1:+%{1}}\
Provides: installonlypkg(kernel-module)\
Provides: %{name}%{?1:-%{1}}-modules-extra-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
Requires: %{name}-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
Requires: %{name}%{?1:-%{1}}-modules-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
AutoReq: no\
AutoProv: yes\
%description %{?1:%{1}-}modules-extra\
This package provides less commonly used kernel modules for the %{?2:%{2} }kernel package.\
%{nil}

#
# This macro creates a kernel-<subpackage>-modules package.
#	%%kernel_modules_package <subpackage> <pretty-name>
#
%define kernel_modules_package() \
%package %{?1:%{1}-}modules\
Summary: kernel modules to match the %{?2:%{2}-}core kernel\
Group: System Environment/Kernel\
Provides: %{name}%{?1:-%{1}}-modules-%{_target_cpu} = %{version}-%{release}\
Provides: %{name}-modules-%{_target_cpu} = %{version}-%{release}%{?1:+%{1}}\
Provides: %{name}-modules = %{version}-%{release}%{?1:+%{1}}\
Provides: installonlypkg(kernel-module)\
Provides: %{name}%{?1:-%{1}}-modules-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
Requires: %{name}-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
AutoReq: no\
AutoProv: yes\
%description %{?1:%{1}-}modules\
This package provides commonly used kernel modules for the %{?2:%{2}-}core kernel package.\
%{nil}

#
# this macro creates a kernel-<subpackage> meta package.
#	%%kernel_meta_package <subpackage>
#
%define kernel_meta_package() \
%package %{1}\
summary: kernel meta-package for the %{1} kernel\
group: system environment/kernel\
Requires: %{name}-%{1}-core-uname-r = %{KVERREL}%{?variant}+%{1}\
Requires: %{name}-%{1}-modules-uname-r = %{KVERREL}%{?variant}+%{1}\
%if %{with_realtime}\
Requires: rt-setup\
%endif\
Provides: installonlypkg(kernel)\
%description %{1}\
The meta-package for the %{1} kernel\
%{nil}

%if %{with_realtime}
#
# this macro creates a kernel-rt-<subpackage>-kvm package
# %%kernel_kvm_package <subpackage>
#
%define kernel_kvm_package() \
%package %{?1:%{1}-}kvm\
Summary: KVM modules for package %{name}%{?1:-%{1}}\
Group: System Environment/Kernel\
Requires: %{name}%{?1:-%{1}} = %{version}-%{release}\
Provides: installonlypkg(kernel-module)\
Provides: %{name}%{?1:-%{1}}-kvm-%{_target_cpu} = %{version}-%{release}\
AutoReq: no\
%description -n %{name}%{?1:-%{1}}-kvm\
This package provides KVM modules for package %{name}%{?1:-%{1}}.\
%{nil}
%endif

#
# This macro creates a %%{name}-<subpackage> and its -devel and -debuginfo too.
#	%%define variant_summary The Linux kernel compiled for <configuration>
#	%%kernel_variant_package [-n <pretty-name>] <subpackage>
#
%define kernel_variant_package(n:) \
%package %{?1:%{1}-}core\
Summary: %{variant_summary}\
Group: System Environment/Kernel\
Provides: %{name}-%{?1:%{1}-}core-uname-r = %{KVERREL}%{?variant}%{?1:+%{1}}\
Provides: installonlypkg(kernel)\
%{expand:%%kernel_reqprovconf}\
%if %{?1:1} %{!?1:0} \
%{expand:%%kernel_meta_package %{?1:%{1}}}\
%endif\
%{expand:%%kernel_devel_package %{?1:%{1}} %{!?-n:%{?1:%{1}}}%{?-n:%{-n*}}}\
%{expand:%%kernel_modules_package %{?1:%{1}} %{!?-n:%{?1:%{1}}}%{?-n:%{-n*}}}\
%{expand:%%kernel_modules_extra_package %{?1:%{1}} %{!?-n:%{?1:%{1}}}%{?-n:%{-n*}}}\
%{expand:%%kernel_modules_internal_package %{?1:%{1}} %{!?-n:%{?1:%{1}}}%{?-n:%{-n*}}}\
%{expand:%%kernel_debuginfo_package %{?1:%{1}}}\
%if %{with_realtime} \
%{expand:%%kernel_kvm_package %{?1:%{1}}} %{!?{-n}:%{1}}%{?{-n}:%{-n*}}}\
%endif \
%{nil}

# Now, each variant package.

%if %{with_zfcpdump}
%define variant_summary The Linux kernel compiled for zfcpdump usage
%kernel_variant_package zfcpdump
%description zfcpdump-core
The kernel package contains the Linux kernel (vmlinuz) for use by the
zfcpdump infrastructure.
# with_zfcpdump
%endif

%define variant_summary The Linux kernel compiled with extra debugging enabled
%kernel_variant_package debug
%description debug-core
The kernel package contains the Linux kernel (vmlinuz), the core of any
Linux operating system.  The kernel handles the basic functions
of the operating system:  memory allocation, process allocation, device
input and output, etc.

This variant of the kernel has numerous debugging options enabled.
It should only be installed when trying to gather additional information
on kernel bugs, as some of these options impact performance noticably.

# And finally the main -core package

%define variant_summary The Linux kernel
%kernel_variant_package
%description core
The kernel package contains the Linux kernel (vmlinuz), the core of any
Linux operating system.  The kernel handles the basic functions
of the operating system: memory allocation, process allocation, device
input and output, etc.

%if %{with_ipaclones}
%kernel_ipaclones_package
%endif

%prep
# do a few sanity-checks for --with *only builds
%if %{with_baseonly}
%if !%{with_up}
echo "Cannot build --with baseonly, up build is disabled"
exit 1
%endif
%endif

# more sanity checking; do it quietly
if [ "%{patches}" != "%%{patches}" ] ; then
  for patch in %{patches} ; do
    if [ ! -f $patch ] ; then
      echo "ERROR: Patch  ${patch##/*/}  listed in specfile but is missing"
      exit 1
    fi
  done
fi 2>/dev/null

patch_command='patch -p1 -F1 -s'
ApplyPatch()
{
  local patch=$1
  shift
  if [ ! -f $RPM_SOURCE_DIR/$patch ]; then
    exit 1
  fi
  if ! grep -E "^Patch[0-9]+: $patch\$" %{_specdir}/${RPM_PACKAGE_NAME%%%%%{?variant}}.spec ; then
    if [ "${patch:0:8}" != "patch-4." ] ; then
      echo "ERROR: Patch  $patch  not listed as a source patch in specfile"
      exit 1
    fi
  fi 2>/dev/null
  case "$patch" in
  *.bz2) bunzip2 < "$RPM_SOURCE_DIR/$patch" | $patch_command ${1+"$@"} ;;
  *.gz)  gunzip  < "$RPM_SOURCE_DIR/$patch" | $patch_command ${1+"$@"} ;;
  *.xz)  unxz    < "$RPM_SOURCE_DIR/$patch" | $patch_command ${1+"$@"} ;;
  *) $patch_command ${1+"$@"} < "$RPM_SOURCE_DIR/$patch" ;;
  esac
}

# don't apply patch if it's empty
ApplyOptionalPatch()
{
  local patch=$1
  shift
  if [ ! -f $RPM_SOURCE_DIR/$patch ]; then
    exit 1
  fi
  local C=$(wc -l $RPM_SOURCE_DIR/$patch | awk '{print $1}')
  if [ "$C" -gt 9 ]; then
    ApplyPatch $patch ${1+"$@"}
  fi
}

%setup -q -n %{name}-%{rpmversion}-%{pkgrelease} -c
cp -v %{SOURCE9000} linux-%{rpmversion}-%{pkgrelease}/certs/rhel.pem
mv linux-%{rpmversion}-%{pkgrelease} linux-%{KVERREL}

cd linux-%{KVERREL}

ApplyOptionalPatch linux-kernel-test.patch

# END OF PATCH APPLICATIONS

# Any further pre-build tree manipulations happen here.

%if %{with_realtime}
# remove the localversion-rt file since it screws around with
# the uname output
if [ -f localversion-rt ]; then
   rm -f localversion-rt
fi
%endif

chmod +x scripts/checkpatch.pl
mv COPYING COPYING-%{version}

# This Prevents scripts/setlocalversion from mucking with our version numbers.
touch .scmversion

# Do not use "ambiguous" python shebangs. RHEL 8 now has a new script
# (/usr/lib/rpm/redhat/brp-mangle-shebangs), which forces us to specify a
# "non-ambiguous" python shebang for scripts we ship in buildroot. This
# script throws an error like below:
# *** ERROR: ambiguous python shebang in /usr/bin/kvm_stat: #!/usr/bin/python. Change it to python3 (or python2) explicitly.
# We patch all sources below for which we got a report/error.
pathfix.py -i %{__python3} -p -n \
	scripts/show_delta \
	scripts/diffconfig \
	scripts/bloat-o-meter \
	tools/perf/tests/attr.py \
	tools/perf/scripts/python/stat-cpi.py \
	tools/perf/scripts/python/sched-migration.py \
	Documentation

%define make make %{?cross_opts} HOSTCFLAGS="%{?build_hostcflags}" HOSTLDFLAGS="%{?build_hostldflags}"

# only deal with configs if we are going to build for the arch
%ifnarch %nobuildarches

if [ -L configs ]; then
	rm -f configs
fi
mkdir configs
cd configs

# Drop some necessary files from the source dir into the buildroot
cp $RPM_SOURCE_DIR/kernel-*.config .
cp %{SOURCE41} .
VERSION=%{version} NAME=%{name} ./generate_all_configs.sh

# Note we need to disable these flags for cross builds because the flags
# from redhat-rpm-config assume that host == target so target arch
# flags cause issues with the host compiler.
%if !%{with_cross}
%define build_hostcflags  ${RPM_OPT_FLAGS}
%define build_hostldflags %{__global_ldflags}
%endif

# enable GCOV kernel config options if gcov is on
%if %{with_gcov}
for i in *.config
do
  sed -i 's/# CONFIG_GCOV_KERNEL is not set/CONFIG_GCOV_KERNEL=y\nCONFIG_GCOV_PROFILE_ALL=y\n/' $i
done
%endif

# Add DUP and kpatch certificates to system trusted keys for RHEL
%if %{signkernel}%{signmodules}
openssl x509 -inform der -in %{SOURCE100} -out rheldup3.pem
openssl x509 -inform der -in %{SOURCE101} -out rhelkpatch1.pem
cat rheldup3.pem rhelkpatch1.pem > ../certs/rhel.pem
%ifarch ppc64le
openssl x509 -inform der -in %{secureboot_ca_0} -out secureboot.pem
cat secureboot.pem >> ../certs/rhel.pem
%endif
for i in *.config; do
  sed -i 's@CONFIG_SYSTEM_TRUSTED_KEYS=""@CONFIG_SYSTEM_TRUSTED_KEYS="certs/rhel.pem"@' $i
done
%endif

cp %{SOURCE42} .
./process_configs.sh -w -c %{name} %{rpmversion} %{?cross_opts}

# end of kernel config
%endif

cd ..
# # End of Configs stuff

# get rid of unwanted files resulting from patch fuzz
find . \( -name "*.orig" -o -name "*~" \) -exec rm -f {} \; >/dev/null

# remove unnecessary SCM files
find . -name .gitignore -exec rm -f {} \; >/dev/null

cd ..

###
### build
###
%build

%if %{with_sparse}
%define sparse_mflags	C=1
%endif

cp_vmlinux()
{
  eu-strip --remove-comment -o "$2" "$1"
}

InitBuildVars() {
    # Initialize the kernel .config file and create some variables that are
    # needed for the actual build process.

    Flavour=$1
    Flav=${Flavour:++${Flavour}}

    # Pick the right kernel config file
    Config=%{name}-%{version}-%{_target_cpu}${Flavour:+-${Flavour}}.config
    DevelDir=/usr/src/kernels/%{KVERREL}${Flav}

    KernelVer=%{version}-%{release}.%{_target_cpu}${Flav}

    # make sure EXTRAVERSION says what we want it to say
    perl -p -i -e "s/^EXTRAVERSION.*/EXTRAVERSION = -%{release}.%{_target_cpu}${Flav}/" Makefile

    %{make} -s %{?_smp_mflags} mrproper
    cp configs/$Config .config

    %if %{signkernel}%{signmodules}
    cp %{SOURCE9} certs/.
    %endif

    Arch=`head -1 .config | cut -b 3-`
    echo USING ARCH=$Arch

    KCFLAGS="%{?kcflags}"

    # add kpatch flags for base kernel
    if [ "$Flavour" == "" ]; then
        KCFLAGS="$KCFLAGS %{?kpatch_kcflags}"
    fi
}

BuildKernel() {
    MakeTarget=$1
    KernelImage=$2
    Flavour=$4
    DoVDSO=$3
    Flav=${Flavour:++${Flavour}}
    InstallName=${5:-vmlinuz}

    DoModules=1
    if [ "$Flavour" = "zfcpdump" ]; then
	    DoModules=0
    fi

    # When the bootable image is just the ELF kernel, strip it.
    # We already copy the unstripped file into the debuginfo package.
    if [ "$KernelImage" = vmlinux ]; then
      CopyKernel=cp_vmlinux
    else
      CopyKernel=cp
    fi

    InitBuildVars $Flavour

    echo BUILDING A KERNEL FOR ${Flavour} %{_target_cpu}...

    %{make} -s ARCH=$Arch oldnoconfig >/dev/null
    %{make} -s ARCH=$Arch V=1 %{?_smp_mflags} KCFLAGS="$KCFLAGS" WITH_GCOV="%{?with_gcov}" $MakeTarget %{?sparse_mflags} %{?kernel_mflags}
    if [ $DoModules -eq 1 ]; then
	%{make} -s ARCH=$Arch V=1 %{?_smp_mflags} KCFLAGS="$KCFLAGS" WITH_GCOV="%{?with_gcov}" modules %{?sparse_mflags} || exit 1
    fi

    mkdir -p $RPM_BUILD_ROOT/%{image_install_path}
    mkdir -p $RPM_BUILD_ROOT/lib/modules/$KernelVer
%if %{with_debuginfo}
    mkdir -p $RPM_BUILD_ROOT%{debuginfodir}/%{image_install_path}
%endif

%ifarch aarch64
    %{make} -s ARCH=$Arch V=1 dtbs dtbs_install INSTALL_DTBS_PATH=$RPM_BUILD_ROOT/%{image_install_path}/dtb-$KernelVer
    cp -r $RPM_BUILD_ROOT/%{image_install_path}/dtb-$KernelVer $RPM_BUILD_ROOT/lib/modules/$KernelVer/dtb
    find arch/$Arch/boot/dts -name '*.dtb' -type f | xargs rm -f
%endif

    # Start installing the results
    install -m 644 .config $RPM_BUILD_ROOT/boot/config-$KernelVer
    install -m 644 .config $RPM_BUILD_ROOT/lib/modules/$KernelVer/config
    install -m 644 System.map $RPM_BUILD_ROOT/boot/System.map-$KernelVer
    install -m 644 System.map $RPM_BUILD_ROOT/lib/modules/$KernelVer/System.map

    # We estimate the size of the initramfs because rpm needs to take this size
    # into consideration when performing disk space calculations. (See bz #530778)
    dd if=/dev/zero of=$RPM_BUILD_ROOT/boot/initramfs-$KernelVer.img bs=1M count=20

    if [ -f arch/$Arch/boot/zImage.stub ]; then
      cp arch/$Arch/boot/zImage.stub $RPM_BUILD_ROOT/%{image_install_path}/zImage.stub-$KernelVer || :
      cp arch/$Arch/boot/zImage.stub $RPM_BUILD_ROOT/lib/modules/$KernelVer/zImage.stub-$KernelVer || :
    fi

    %if %{signkernel}
    if [ "$KernelImage" = vmlinux ]; then
        # We can't strip and sign $KernelImage in place, because
        # we need to preserve original vmlinux for debuginfo.
        # Use a copy for signing.
        $CopyKernel $KernelImage $KernelImage.tosign
        KernelImage=$KernelImage.tosign
        CopyKernel=cp
    fi

    # Sign the image if we're using EFI
    # aarch64 kernels are gziped EFI images
    KernelExtension=${KernelImage##*.}
    if [ "$KernelExtension" == "gz" ]; then
        SignImage=${KernelImage%.*}
    else
        SignImage=$KernelImage
    fi

    %ifarch x86_64 aarch64
    %pesign -s -i $SignImage -o vmlinuz.tmp -a %{secureboot_ca_0} -c %{secureboot_key_0} -n %{pesign_name_0}
    %pesign -s -i vmlinuz.tmp -o vmlinuz.signed -a %{secureboot_ca_1} -c %{secureboot_key_1} -n %{pesign_name_1}
    rm vmlinuz.tmp
    %endif
    %ifarch s390x ppc64le
    if [ -x /usr/bin/rpm-sign ]; then
	rpm-sign --key "%{pesign_name_0}" --lkmsign $SignImage --output vmlinuz.signed
    elif [ $DoModules -eq 1 ]; then
	chmod +x scripts/sign-file
	./scripts/sign-file -p sha256 certs/signing_key.pem certs/signing_key.x509 $SignImage vmlinuz.signed
    else
	mv $SignImage vmlinuz.signed
    fi
    %endif

    if [ ! -s vmlinuz.signed ]; then
        echo "pesigning failed"
        exit 1
    fi
    mv vmlinuz.signed $SignImage
    if [ "$KernelExtension" == "gz" ]; then
        gzip -f9 $SignImage
    fi
    # signkernel
    %endif

    $CopyKernel $KernelImage \
                $RPM_BUILD_ROOT/%{image_install_path}/$InstallName-$KernelVer
    chmod 755 $RPM_BUILD_ROOT/%{image_install_path}/$InstallName-$KernelVer
    cp $RPM_BUILD_ROOT/%{image_install_path}/$InstallName-$KernelVer $RPM_BUILD_ROOT/lib/modules/$KernelVer/$InstallName

    # hmac sign the kernel for FIPS
    echo "Creating hmac file: $RPM_BUILD_ROOT/%{image_install_path}/.vmlinuz-$KernelVer.hmac"
    ls -l $RPM_BUILD_ROOT/%{image_install_path}/$InstallName-$KernelVer
    sha512hmac $RPM_BUILD_ROOT/%{image_install_path}/$InstallName-$KernelVer | sed -e "s,$RPM_BUILD_ROOT,," > $RPM_BUILD_ROOT/%{image_install_path}/.vmlinuz-$KernelVer.hmac;
    cp $RPM_BUILD_ROOT/%{image_install_path}/.vmlinuz-$KernelVer.hmac $RPM_BUILD_ROOT/lib/modules/$KernelVer/.vmlinuz.hmac

    if [ $DoModules -eq 1 ]; then
	# Override $(mod-fw) because we don't want it to install any firmware
	# we'll get it from the linux-firmware package and we don't want conflicts
	%{make} -s %{?_smp_mflags} ARCH=$Arch INSTALL_MOD_PATH=$RPM_BUILD_ROOT modules_install KERNELRELEASE=$KernelVer mod-fw=
    fi

%if %{with_gcov}
    # install gcov-needed files to $BUILDROOT/$BUILD/...:
    #   gcov_info->filename is absolute path
    #   gcno references to sources can use absolute paths (e.g. in out-of-tree builds)
    #   sysfs symlink targets (set up at compile time) use absolute paths to BUILD dir
    find . \( -name '*.gcno' -o -name '*.[chS]' \) -exec install -D '{}' "$RPM_BUILD_ROOT/$(pwd)/{}" \;
%endif

    if [ $DoVDSO -ne 0 ]; then
        %{make} -s ARCH=$Arch INSTALL_MOD_PATH=$RPM_BUILD_ROOT vdso_install KERNELRELEASE=$KernelVer
        if [ ! -s ldconfig-kernel.conf ]; then
          echo > ldconfig-kernel.conf "\
    # Placeholder file, no vDSO hwcap entries used in this kernel."
        fi
        %{__install} -D -m 444 ldconfig-kernel.conf \
            $RPM_BUILD_ROOT/etc/ld.so.conf.d/%{name}-$KernelVer.conf
        rm -rf $RPM_BUILD_ROOT/lib/modules/$KernelVer/vdso/.build-id
    fi

    # And save the headers/makefiles etc for building modules against
    #
    # This all looks scary, but the end result is supposed to be:
    # * all arch relevant include/ files
    # * all Makefile/Kconfig files
    # * all script/ files

    rm -f $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    rm -f $RPM_BUILD_ROOT/lib/modules/$KernelVer/source
    mkdir -p $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    (cd $RPM_BUILD_ROOT/lib/modules/$KernelVer ; ln -s build source)
    # dirs for additional modules per module-init-tools, kbuild/modules.txt
    mkdir -p $RPM_BUILD_ROOT/lib/modules/$KernelVer/updates
    mkdir -p $RPM_BUILD_ROOT/lib/modules/$KernelVer/weak-updates
    # first copy everything
    cp --parents `find  -type f -name "Makefile*" -o -name "Kconfig*"` $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    cp Module.symvers $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    cp System.map $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    if [ -s Module.markers ]; then
      cp Module.markers $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    fi

    # create the kABI metadata for use in packaging
    # NOTENOTE: the name symvers is used by the rpm backend
    # NOTENOTE: to discover and run the /usr/lib/rpm/fileattrs/kabi.attr
    # NOTENOTE: script which dynamically adds exported kernel symbol
    # NOTENOTE: checksums to the rpm metadata provides list.
    # NOTENOTE: if you change the symvers name, update the backend too
    echo "**** GENERATING kernel ABI metadata ****"
    gzip -c9 < Module.symvers > $RPM_BUILD_ROOT/boot/symvers-$KernelVer.gz
    cp $RPM_BUILD_ROOT/boot/symvers-$KernelVer.gz $RPM_BUILD_ROOT/lib/modules/$KernelVer/symvers.gz

%if %{with_kabichk}
    echo "**** kABI checking is enabled in kernel SPEC file. ****"
    chmod 0755 $RPM_SOURCE_DIR/check-kabi
    if [ -e $RPM_SOURCE_DIR/Module.kabi_%{_target_cpu}$Flavour ]; then
        cp $RPM_SOURCE_DIR/Module.kabi_%{_target_cpu}$Flavour $RPM_BUILD_ROOT/Module.kabi
        $RPM_SOURCE_DIR/check-kabi -k $RPM_BUILD_ROOT/Module.kabi -s Module.symvers || exit 1
        # for now, don't keep it around.
        rm $RPM_BUILD_ROOT/Module.kabi
    else
        echo "**** NOTE: Cannot find reference Module.kabi file. ****"
    fi
%endif

%if %{with_kabidupchk}
    echo "**** kABI DUP checking is enabled in kernel SPEC file. ****"
    if [ -e $RPM_SOURCE_DIR/Module.kabi_dup_%{_target_cpu}$Flavour ]; then
        cp $RPM_SOURCE_DIR/Module.kabi_dup_%{_target_cpu}$Flavour $RPM_BUILD_ROOT/Module.kabi
        $RPM_SOURCE_DIR/check-kabi -k $RPM_BUILD_ROOT/Module.kabi -s Module.symvers || exit 1
        # for now, don't keep it around.
        rm $RPM_BUILD_ROOT/Module.kabi
    else
        echo "**** NOTE: Cannot find DUP reference Module.kabi file. ****"
    fi
%endif

%if %{with_kabidw_base}
    # Don't build kabi base for debug kernels
    if [ "$Flavour" != "zfcpdump" -a "$Flavour" != "debug" ]; then
        mkdir -p $RPM_BUILD_ROOT/kabi-dwarf
        tar xjvf %{SOURCE301} -C $RPM_BUILD_ROOT/kabi-dwarf

        mkdir -p $RPM_BUILD_ROOT/kabi-dwarf/stablelists
        tar xjvf %{SOURCE300} -C $RPM_BUILD_ROOT/kabi-dwarf/stablelists

        echo "**** GENERATING DWARF-based kABI baseline dataset ****"
        chmod 0755 $RPM_BUILD_ROOT/kabi-dwarf/run_kabi-dw.sh
        $RPM_BUILD_ROOT/kabi-dwarf/run_kabi-dw.sh generate \
            "$RPM_BUILD_ROOT/kabi-dwarf/stablelists/kabi-current/kabi_stablelist_%{_target_cpu}" \
            "$(pwd)" \
            "$RPM_BUILD_ROOT/kabidw-base/%{_target_cpu}${Flavour:+.${Flavour}}" || :

        rm -rf $RPM_BUILD_ROOT/kabi-dwarf
    fi
%endif

%if %{with_kabidwchk}
    if [ "$Flavour" != "zfcpdump" ]; then
        mkdir -p $RPM_BUILD_ROOT/kabi-dwarf
        tar xjvf %{SOURCE301} -C $RPM_BUILD_ROOT/kabi-dwarf
        if [ -d "$RPM_BUILD_ROOT/kabi-dwarf/base/%{_target_cpu}${Flavour:+.${Flavour}}" ]; then
            mkdir -p $RPM_BUILD_ROOT/kabi-dwarf/stablelists
            tar xjvf %{SOURCE300} -C $RPM_BUILD_ROOT/kabi-dwarf/stablelists

            echo "**** GENERATING DWARF-based kABI dataset ****"
            chmod 0755 $RPM_BUILD_ROOT/kabi-dwarf/run_kabi-dw.sh
            $RPM_BUILD_ROOT/kabi-dwarf/run_kabi-dw.sh generate \
                "$RPM_BUILD_ROOT/kabi-dwarf/stablelists/kabi-current/kabi_stablelist_%{_target_cpu}" \
                "$(pwd)" \
                "$RPM_BUILD_ROOT/kabi-dwarf/base/%{_target_cpu}${Flavour:+.${Flavour}}.tmp" || :

            echo "**** kABI DWARF-based comparison report ****"
            $RPM_BUILD_ROOT/kabi-dwarf/run_kabi-dw.sh compare \
                "$RPM_BUILD_ROOT/kabi-dwarf/base/%{_target_cpu}${Flavour:+.${Flavour}}" \
                "$RPM_BUILD_ROOT/kabi-dwarf/base/%{_target_cpu}${Flavour:+.${Flavour}}.tmp" || :
            echo "**** End of kABI DWARF-based comparison report ****"
        else
            echo "**** Baseline dataset for kABI DWARF-BASED comparison report not found ****"
        fi

        rm -rf $RPM_BUILD_ROOT/kabi-dwarf
    fi
%endif

    # then drop all but the needed Makefiles/Kconfig files
    rm -rf $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/Documentation
    rm -rf $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/scripts
    rm -rf $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/include
    cp .config $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    cp -a scripts $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    rm -rf $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/scripts/tracing
    rm -f $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/scripts/spdxcheck.py

    # Files for 'make scripts' to succeed with kernel-devel.
    mkdir -p $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/security/selinux/include
    cp -a --parents security/selinux/include/classmap.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    cp -a --parents security/selinux/include/initial_sid_to_string.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    mkdir -p $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/tools/include/tools
    cp -a --parents tools/include/tools/be_byteshift.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build
    cp -a --parents tools/include/tools/le_byteshift.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build

    if [ -f tools/objtool/objtool ]; then
      cp -a tools/objtool/objtool $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/tools/objtool/ || :
    fi
    if [ -d arch/$Arch/scripts ]; then
      cp -a arch/$Arch/scripts $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/arch/%{_arch} || :
    fi
    if [ -f arch/$Arch/*lds ]; then
      cp -a arch/$Arch/*lds $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/arch/%{_arch}/ || :
    fi
    if [ -f arch/%{asmarch}/kernel/module.lds ]; then
      cp -a --parents arch/%{asmarch}/kernel/module.lds $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    fi
    rm -f $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/scripts/*.o
    rm -f $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/scripts/*/*.o
%ifarch ppc64le
    cp -a --parents arch/powerpc/lib/crtsavres.[So] $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
%endif
    if [ -d arch/%{asmarch}/include ]; then
      cp -a --parents arch/%{asmarch}/include $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    fi
%ifarch aarch64
    # arch/arm64/include/asm/xen references arch/arm
    cp -a --parents arch/arm/include/asm/xen $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    # arch/arm64/include/asm/opcodes.h references arch/arm
    cp -a --parents arch/arm/include/asm/opcodes.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
%endif
    cp -a include $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/include
%ifarch x86_64
    # files for 'make prepare' to succeed with kernel-devel
    cp -a --parents arch/x86/entry/syscalls/syscall_32.tbl $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/entry/syscalls/syscalltbl.sh $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/entry/syscalls/syscallhdr.sh $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/entry/syscalls/syscall_64.tbl $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/tools/relocs_32.c $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/tools/relocs_64.c $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/tools/relocs.c $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/tools/relocs_common.c $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/tools/relocs.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents tools/include/tools/le_byteshift.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/purgatory/purgatory.c $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/purgatory/stack.S $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/purgatory/setup-x86_64.S $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/purgatory/entry64.S $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/boot/string.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/boot/string.c $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
    cp -a --parents arch/x86/boot/ctype.h $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/
%endif
    # Make sure the Makefile and version.h have a matching timestamp so that
    # external modules can be built
    touch -r $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/Makefile $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/include/generated/uapi/linux/version.h

    # Copy .config to include/config/auto.conf so "make prepare" is unnecessary.
    cp $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/.config $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/include/config/auto.conf

%if %{with_debuginfo}
    eu-readelf -n vmlinux | grep "Build ID" | awk '{print $NF}' > vmlinux.id
    cp vmlinux.id $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/vmlinux.id

    #
    # save the vmlinux file for kernel debugging into the kernel-debuginfo rpm
    #
    mkdir -p $RPM_BUILD_ROOT%{debuginfodir}/lib/modules/$KernelVer
    cp vmlinux $RPM_BUILD_ROOT%{debuginfodir}/lib/modules/$KernelVer
    if [ -n "%{vmlinux_decompressor}" ]; then
	    eu-readelf -n  %{vmlinux_decompressor} | grep "Build ID" | awk '{print $NF}' > vmlinux.decompressor.id
	    # Without build-id the build will fail. But for s390 the build-id
	    # wasn't added before 5.11. In case it is missing prefer not
	    # packaging the debuginfo over a build failure.
	    if [ -s vmlinux.decompressor.id ]; then
		    cp vmlinux.decompressor.id $RPM_BUILD_ROOT/lib/modules/$KernelVer/build/vmlinux.decompressor.id
		    cp %{vmlinux_decompressor} $RPM_BUILD_ROOT%{debuginfodir}/lib/modules/$KernelVer/vmlinux.decompressor
	    fi
    fi
%endif

    find $RPM_BUILD_ROOT/lib/modules/$KernelVer -name "*.ko" -type f >modnames

    # mark modules executable so that strip-to-file can strip them
    xargs --no-run-if-empty chmod u+x < modnames

    # Generate a list of modules for block and networking.

    grep -F /drivers/ modnames | xargs --no-run-if-empty nm -upA |
    sed -n 's,^.*/\([^/]*\.ko\):  *U \(.*\)$,\1 \2,p' > drivers.undef

    collect_modules_list()
    {
      sed -r -n -e "s/^([^ ]+) \\.?($2)\$/\\1/p" drivers.undef |
        LC_ALL=C sort -u > $RPM_BUILD_ROOT/lib/modules/$KernelVer/modules.$1
      if [ ! -z "$3" ]; then
        sed -r -e "/^($3)\$/d" -i $RPM_BUILD_ROOT/lib/modules/$KernelVer/modules.$1
      fi
    }

    collect_modules_list networking \
      'register_netdev|ieee80211_register_hw|usbnet_probe|phy_driver_register|rt(l_|2x00)(pci|usb)_probe|register_netdevice'
    collect_modules_list block \
      'ata_scsi_ioctl|scsi_add_host|scsi_add_host_with_dma|blk_alloc_queue|blk_init_queue|register_mtd_blktrans|scsi_esp_register|scsi_register_device_handler|blk_queue_physical_block_size' 'pktcdvd.ko|dm-mod.ko'
    collect_modules_list drm \
      'drm_open|drm_init'
    collect_modules_list modesetting \
      'drm_crtc_init'
%if %{with_realtime}
    collect_modules_list kvm \
      'kvm_init|kvmgt_init'
%endif

    # detect missing or incorrect license tags
    ( find $RPM_BUILD_ROOT/lib/modules/$KernelVer -name '*.ko' | xargs /sbin/modinfo -l | \
        grep -E -v 'GPL( v2)?$|Dual BSD/GPL$|Dual MPL/GPL$|GPL and additional rights$' ) && exit 1

    # remove files that will be auto generated by depmod at rpm -i time
    pushd $RPM_BUILD_ROOT/lib/modules/$KernelVer/
        rm -f modules.{alias*,builtin.bin,dep*,*map,symbols*,devname,softdep}
    popd

    # Identify modules in the kernel-modules-extras package
    %{SOURCE17} $RPM_BUILD_ROOT lib/modules/$KernelVer %{SOURCE16}
    # Identify modules in the kernel-modules-internal package
    %{SOURCE17} $RPM_BUILD_ROOT lib/modules/$KernelVer %{SOURCE44} internal
%if %{with_realtime}
    # Identify modules in the kernel-rt-kvm package
    %{SOURCE17} $RPM_BUILD_ROOT lib/modules/$KernelVer %{SOURCE400} kvm
%endif

    #
    # Generate the kernel-core and kernel-modules files lists
    #

    # Copy the System.map file for depmod to use, and create a backup of the
    # full module tree so we can restore it after we're done filtering
    cp System.map $RPM_BUILD_ROOT/.
    pushd $RPM_BUILD_ROOT
    mkdir restore
    cp -r lib/modules/$KernelVer/* restore/.

    # don't include anything going into kernel-modules-extra in the file lists
    xargs rm -rf < mod-extra.list
    # don't include anything going int kernel-modules-internal in the file lists
    xargs rm -rf < mod-internal.list
%if %{with_realtime}
    # don't include anything going into kernel-rt-kvm in the file lists
    xargs rm -rf < mod-kvm.list
%endif

    if [ $DoModules -eq 1 ]; then
	# Find all the module files and filter them out into the core and
	# modules lists.  This actually removes anything going into -modules
	# from the dir.
	find lib/modules/$KernelVer/kernel -name *.ko | sort -n > modules.list
	cp $RPM_SOURCE_DIR/filter-*.sh .
	%{SOURCE99} modules.list %{_target_cpu}
	rm filter-*.sh

	# Run depmod on the resulting module tree and make sure it isn't broken
	depmod -b . -aeF ./System.map $KernelVer &> depmod.out
	if [ -s depmod.out ]; then
	    echo "Depmod failure"
	    cat depmod.out
	    exit 1
	else
	    rm depmod.out
	fi
    else
	# Ensure important files/directories exist to let the packaging succeed
	echo '%%defattr(-,-,-)' > modules.list
	echo '%%defattr(-,-,-)' > k-d.list
	mkdir -p lib/modules/$KernelVer/kernel
	# Add files usually created by make modules, needed to prevent errors
	# thrown by depmod during package installation
	touch lib/modules/$KernelVer/modules.order
	touch lib/modules/$KernelVer/modules.builtin
    fi

    # remove files that will be auto generated by depmod at rpm -i time
    pushd $RPM_BUILD_ROOT/lib/modules/$KernelVer/
        rm -f modules.{alias*,builtin.bin,dep*,*map,symbols*,devname,softdep}
    popd

    # Go back and find all of the various directories in the tree.  We use this
    # for the dir lists in kernel-core
    find lib/modules/$KernelVer/kernel -mindepth 1 -type d | sort -n > module-dirs.list

    # Cleanup
    rm System.map
    cp -r restore/* lib/modules/$KernelVer/.
    rm -rf restore
    popd

    # Make sure the files lists start with absolute paths or rpmbuild fails.
    # Also add in the dir entries
    sed -e 's/^lib*/\/lib/' %{?zipsed} $RPM_BUILD_ROOT/k-d.list > ../%{name}${Flavour:+-${Flavour}}-modules.list
    sed -e 's/^lib*/%dir \/lib/' %{?zipsed} $RPM_BUILD_ROOT/module-dirs.list > ../%{name}${Flavour:+-${Flavour}}-core.list
    sed -e 's/^lib*/\/lib/' %{?zipsed} $RPM_BUILD_ROOT/modules.list >> ../%{name}${Flavour:+-${Flavour}}-core.list
    sed -e 's/^lib*/\/lib/' %{?zipsed} $RPM_BUILD_ROOT/mod-extra.list >> ../%{name}${Flavour:+-${Flavour}}-modules-extra.list
    sed -e 's/^lib*/\/lib/' %{?zipsed} $RPM_BUILD_ROOT/mod-internal.list >> ../%{name}${Flavour:+-${Flavour}}-modules-internal.list
%if %{with_realtime}
    sed -e 's/^lib*/\/lib/' %{?zipsed} $RPM_BUILD_ROOT/mod-kvm.list >> ../%{name}${Flavour:+-${Flavour}}-kvm.list
%endif

    # Cleanup
    rm -f $RPM_BUILD_ROOT/k-d.list
    rm -f $RPM_BUILD_ROOT/modules.list
    rm -f $RPM_BUILD_ROOT/module-dirs.list
    rm -f $RPM_BUILD_ROOT/mod-extra.list
    rm -f $RPM_BUILD_ROOT/mod-internal.list
%if %{with_realtime}
    rm -f $RPM_BUILD_ROOT/mod-kvm.list
%endif

%if %{signmodules}
    if [ $DoModules -eq 1 ]; then
	# Save the signing keys so we can sign the modules in __modsign_install_post
	cp certs/signing_key.pem certs/signing_key.pem.sign${Flav}
	cp certs/signing_key.x509 certs/signing_key.x509.sign${Flav}
    fi
%endif

    # Move the devel headers out of the root file system
    mkdir -p $RPM_BUILD_ROOT/usr/src/kernels
    mv $RPM_BUILD_ROOT/lib/modules/$KernelVer/build $RPM_BUILD_ROOT/$DevelDir

    # This is going to create a broken link during the build, but we don't use
    # it after this point.  We need the link to actually point to something
    # when kernel-devel is installed, and a relative link doesn't work across
    # the F17 UsrMove feature.
    ln -sf $DevelDir $RPM_BUILD_ROOT/lib/modules/$KernelVer/build

    # prune junk from kernel-devel
    find $RPM_BUILD_ROOT/usr/src/kernels -name ".*.cmd" -exec rm -f {} \;

    # build a BLS config for this kernel
    %{SOURCE43} "$KernelVer" "$RPM_BUILD_ROOT" "%{?variant}"

    # Red Hat UEFI Secure Boot CA cert, which can be used to authenticate the kernel
    mkdir -p $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer
    %ifarch x86_64 aarch64
        install -m 0644 %{secureboot_ca_0} $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer/kernel-signing-ca-20200609.cer
        install -m 0644 %{secureboot_ca_1} $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer/kernel-signing-ca-20140212.cer
        ln -s kernel-signing-ca-20200609.cer $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer/kernel-signing-ca.cer
    %else
        install -m 0644 %{secureboot_ca_0} $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer/kernel-signing-ca.cer
    %endif
    %ifarch s390x ppc64le
    if [ $DoModules -eq 1 ]; then
	if [ -x /usr/bin/rpm-sign ]; then
	    install -m 0644 %{secureboot_key_0} $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer/%{signing_key_filename}
	else
	    install -m 0644 certs/signing_key.x509.sign${Flav} $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer/kernel-signing-ca.cer
	    openssl x509 -in certs/signing_key.pem.sign${Flav} -outform der -out $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer/%{signing_key_filename}
	    chmod 0644 $RPM_BUILD_ROOT%{_datadir}/doc/kernel-keys/$KernelVer/%{signing_key_filename}
	fi
    fi
    %endif

%if %{with_ipaclones}
    MAXPROCS=$(echo %{?_smp_mflags} | sed -n 's/-j\s*\([0-9]\+\)/\1/p')
    if [ -z "$MAXPROCS" ]; then
        MAXPROCS=1
    fi
    if [ "$Flavour" == "" ]; then
        mkdir -p $RPM_BUILD_ROOT/$DevelDir-ipaclones
        find . -name '*.ipa-clones' | xargs -i{} -r -n 1 -P $MAXPROCS install -m 644 -D "{}" "$RPM_BUILD_ROOT/$DevelDir-ipaclones/{}"
    fi
%endif

}

###
# DO it...
###

# prepare directories
rm -rf $RPM_BUILD_ROOT
mkdir -p $RPM_BUILD_ROOT/boot
mkdir -p $RPM_BUILD_ROOT%{_libexecdir}

cd linux-%{KVERREL}


%if %{with_debug}
BuildKernel %make_target %kernel_image %{with_vdso_install} debug
%endif

%if %{with_zfcpdump}
BuildKernel %make_target %kernel_image %{with_vdso_install} zfcpdump
%endif

%if %{with_up}
BuildKernel %make_target %kernel_image %{with_vdso_install}
%endif

%ifnarch noarch i686
%if !%{with_debug} && !%{with_zfcpdump} && !%{with_up}
# If only building the user space tools, then initialize the build environment
# and some variables so that the various userspace tools can be built.
InitBuildVars
%endif
%endif

%ifarch aarch64
%global perf_build_extra_opts CORESIGHT=1
%endif
%global perf_make \
  make EXTRA_CFLAGS="${RPM_OPT_FLAGS}" LDFLAGS="%{__global_ldflags}" %{?cross_opts} -C tools/perf V=1 NO_PERF_READ_VDSO32=1 NO_PERF_READ_VDSOX32=1 WERROR=0 NO_LIBUNWIND=1 HAVE_CPLUS_DEMANGLE=1 NO_GTK2=1 NO_STRLCPY=1 NO_BIONIC=1 LIBBPF_DYNAMIC=1 %{?perf_build_extra_opts} prefix=%{_prefix} PYTHON=%{__python3}
%if %{with_perf}
# perf
# make sure check-headers.sh is executable
chmod +x tools/perf/check-headers.sh
%{perf_make} DESTDIR=$RPM_BUILD_ROOT all
%endif

%global tools_make \
  %{make} CFLAGS="${RPM_OPT_FLAGS}" LDFLAGS="%{__global_ldflags}" V=1

%if %{with_tools}
%ifarch %{cpupowerarchs}
# cpupower
# make sure version-gen.sh is executable.
chmod +x tools/power/cpupower/utils/version-gen.sh
%{tools_make} -C tools/power/cpupower CPUFREQ_BENCH=false DEBUG=false
%ifarch x86_64
    pushd tools/power/cpupower/debug/x86_64
    %{tools_make} centrino-decode powernow-k8-decode
    popd
%endif
%ifarch x86_64
   pushd tools/power/x86/x86_energy_perf_policy/
   %{tools_make}
   popd
   pushd tools/power/x86/turbostat
   %{tools_make}
   popd
   pushd tools/power/x86/intel-speed-select
   %{make}
   popd
%endif
%endif
pushd tools/thermal/tmon/
%{tools_make}
popd
pushd tools/iio/
%{tools_make}
popd
pushd tools/gpio/
%{tools_make}
popd
# build VM tools
pushd tools/vm/
%{tools_make} slabinfo page_owner_sort
popd
%endif

%global bpftool_make \
  make EXTRA_CFLAGS="${RPM_OPT_FLAGS}" EXTRA_LDFLAGS="%{__global_ldflags}" DESTDIR=$RPM_BUILD_ROOT V=1
%if %{with_bpftool}
pushd tools/bpf/bpftool
%{bpftool_make}
popd
%endif

%if %{with_selftests}
# Unfortunately, samples/bpf/Makefile expects that the headers are installed
# in the source tree. We installed them previously to $RPM_BUILD_ROOT/usr
# but there's no way to tell the Makefile to take them from there.
%{make} %{?_smp_mflags} headers_install
%{make} %{?_smp_mflags} ARCH=$Arch V=1 samples/bpf/

# Prevent bpf selftests to build bpftool repeatedly:
export BPFTOOL=$(pwd)/tools/bpf/bpftool/bpftool

pushd tools/testing/selftests
# We need to install here because we need to call make with ARCH set which
# doesn't seem possible to do in the install section.
%{make} %{?_smp_mflags} ARCH=$Arch V=1 TARGETS="bpf livepatch net net/forwarding net/mptcp netfilter tc-testing" FORCE_TARGETS=1 INSTALL_PATH=%{buildroot}%{_libexecdir}/kselftests install

# 'make install' for bpf is broken and upstream refuses to fix it.
# Install the needed files manually.
for dir in bpf bpf/no_alu32 bpf/progs; do
	mkdir -p %{buildroot}%{_libexecdir}/kselftests/$dir
	find $dir -maxdepth 1 -type f \( -executable -o -name '*.py' -o -name settings -o \
		-name 'btf_dump_test_case_*.c' -o \
		-name '*.o' -exec sh -c 'readelf -h "{}" | grep -q "^  Machine:.*BPF"' \; \) -print0 | \
	xargs -0 cp -t %{buildroot}%{_libexecdir}/kselftests/$dir
done
popd
export -n BPFTOOL
%endif

%if %{with_doc}
# Make the HTML pages.
make htmldocs || %{doc_build_fail}

# sometimes non-world-readable files sneak into the kernel source tree
chmod -R a=rX Documentation
find Documentation -type d | xargs chmod u+w
%endif

# In the modsign case, we do 3 things.  1) We check the "flavour" and hard
# code the value in the following invocations.  This is somewhat sub-optimal
# but we're doing this inside of an RPM macro and it isn't as easy as it
# could be because of that.  2) We restore the .tmp_versions/ directory from
# the one we saved off in BuildKernel above.  This is to make sure we're
# signing the modules we actually built/installed in that flavour.  3) We
# grab the arch and invoke mod-sign.sh command to actually sign the modules.
#
# We have to do all of those things _after_ find-debuginfo runs, otherwise
# that will strip the signature off of the modules.
#
# Don't sign modules for the zfcpdump flavour as it is monolithic.

%define __modsign_install_post \
  if [ "%{signmodules}" -eq "1" ]; then \
    if [ "%{with_debug}" -ne "0" ]; then \
      %{modsign_cmd} certs/signing_key.pem.sign+debug certs/signing_key.x509.sign+debug $RPM_BUILD_ROOT/lib/modules/%{KVERREL}+debug/ \
    fi \
    if [ "%{with_up}" -ne "0" ]; then \
      %{modsign_cmd} certs/signing_key.pem.sign certs/signing_key.x509.sign $RPM_BUILD_ROOT/lib/modules/%{KVERREL}/ \
    fi \
  fi \
  if [ "%{zipmodules}" -eq "1" ]; then \
    find $RPM_BUILD_ROOT/lib/modules/ -type f -name '*.ko' | %{SOURCE80} %{?_smp_mflags}; \
  fi \
%{nil}

###
### Special hacks for debuginfo subpackages.
###

# This macro is used by %%install, so we must redefine it before that.
%define debug_package %{nil}

%if %{with_debuginfo}

%ifnarch noarch
%global __debug_package 1
%files -f debugfiles.list debuginfo-common-%{_target_cpu}
%defattr(-,root,root)
%endif

%endif

# We don't want to package debuginfo for self-tests and samples but
# we have to delete them to avoid an error messages about unpackaged
# files.
%define __remove_unwanted_dbginfo_install_post \
  if [ "%{with_selftests}" -ne "0" ]; then \
    rm -rf $RPM_BUILD_ROOT/usr/lib/debug/usr/libexec/ksamples; \
    rm -rf $RPM_BUILD_ROOT/usr/lib/debug/usr/libexec/kselftests; \
  fi \
%{nil}

#
# Disgusting hack alert! We need to ensure we sign modules *after* all
# invocations of strip occur, which is in __debug_install_post if
# find-debuginfo.sh runs, and __os_install_post if not.
#
%define __spec_install_post \
  %{?__debug_package:%{__debug_install_post}}\
  %{__arch_install_post}\
  %{__os_install_post}\
  %{__remove_unwanted_dbginfo_install_post}\
  %{__modsign_install_post}

###
### install
###

%install

cd linux-%{KVERREL}

%if %{with_doc}
docdir=$RPM_BUILD_ROOT%{_datadir}/doc/kernel-doc-%{rpmversion}

# copy the source over
mkdir -p $docdir
tar -h -f - --exclude=man --exclude='.*' -c Documentation | tar xf - -C $docdir

# with_doc
%endif

# We have to do the headers install before the tools install because the
# kernel headers_install will remove any header files in /usr/include that
# it doesn't install itself.

%if %{with_headers}
# Install kernel headers
%{make} ARCH=%{hdrarch} INSTALL_HDR_PATH=$RPM_BUILD_ROOT/usr headers_install

find $RPM_BUILD_ROOT/usr/include \
     \( -name .install -o -name .check -o \
        -name ..install.cmd -o -name ..check.cmd \) | xargs rm -f

%endif

%if %{with_cross_headers}
HDR_ARCH_LIST='arm64 powerpc s390 x86'
mkdir -p $RPM_BUILD_ROOT/usr/tmp-headers
%{make} ARCH=%{hdrarch} HDR_ARCH_LIST="$HDR_ARCH_LIST" INSTALL_HDR_PATH=$RPM_BUILD_ROOT/usr/tmp-headers headers_install_all

find $RPM_BUILD_ROOT/usr/tmp-headers/include \
     \( -name .install -o -name .check -o \
        -name ..install.cmd -o -name ..check.cmd \) | xargs rm -f

# Copy all the architectures we care about to their respective asm directories
for arch in $HDR_ARCH_LIST ; do
mkdir -p $RPM_BUILD_ROOT/usr/${arch}-linux-gnu/include
mv $RPM_BUILD_ROOT/usr/tmp-headers/include/arch-${arch}/asm $RPM_BUILD_ROOT/usr/${arch}-linux-gnu/include/
cp -a $RPM_BUILD_ROOT/usr/tmp-headers/include/asm-generic $RPM_BUILD_ROOT/usr/${arch}-linux-gnu/include/.
done

# Remove the rest of the architectures
rm -rf $RPM_BUILD_ROOT/usr/tmp-headers/include/arch*
rm -rf $RPM_BUILD_ROOT/usr/tmp-headers/include/asm-*

# Copy the rest of the headers over
for arch in $HDR_ARCH_LIST ; do
cp -a $RPM_BUILD_ROOT/usr/tmp-headers/include/* $RPM_BUILD_ROOT/usr/${arch}-linux-gnu/include/.
done

rm -rf $RPM_BUILD_ROOT/usr/tmp-headers
%endif

%if %{with_kernel_abi_stablelists}
# kabi directory
INSTALL_KABI_PATH=$RPM_BUILD_ROOT/lib/modules/
mkdir -p $INSTALL_KABI_PATH

# install kabi releases directories
tar xjvf %{SOURCE300} -C $INSTALL_KABI_PATH
# with_kernel_abi_stablelists
%endif

%if %{with_perf}
# perf tool binary and supporting scripts/binaries
%{perf_make} DESTDIR=$RPM_BUILD_ROOT lib=%{_lib} install-bin install-traceevent-plugins
# remove the 'trace' symlink.
rm -f %{buildroot}%{_bindir}/trace

# For both of the below, yes, this should be using a macro but right now
# it's hard coded and we don't actually want it anyway right now.
# Whoever wants examples can fix it up!

# remove examples
rm -rf %{buildroot}/usr/lib/examples/perf
# remove the stray header file that somehow got packaged in examples
rm -rf %{buildroot}/usr/lib/include/perf/bpf/bpf.h

# remove perf-bpf examples
rm -rf %{buildroot}/usr/lib/perf/examples
rm -rf %{buildroot}/usr/lib/perf/include

# python-perf extension
%{perf_make} DESTDIR=$RPM_BUILD_ROOT install-python_ext

# perf man pages (note: implicit rpm magic compresses them later)
mkdir -p %{buildroot}/%{_mandir}/man1
%{perf_make} DESTDIR=$RPM_BUILD_ROOT install-man
%endif

%if %{with_tools}
%ifarch %{cpupowerarchs}
%{make} -C tools/power/cpupower DESTDIR=$RPM_BUILD_ROOT libdir=%{_libdir} mandir=%{_mandir} CPUFREQ_BENCH=false install
rm -f %{buildroot}%{_libdir}/*.{a,la}
%find_lang cpupower
mv cpupower.lang ../
%ifarch x86_64
    pushd tools/power/cpupower/debug/x86_64
    install -m755 centrino-decode %{buildroot}%{_bindir}/centrino-decode
    install -m755 powernow-k8-decode %{buildroot}%{_bindir}/powernow-k8-decode
    popd
%endif
chmod 0755 %{buildroot}%{_libdir}/libcpupower.so*
mkdir -p %{buildroot}%{_unitdir} %{buildroot}%{_sysconfdir}/sysconfig
install -m644 %{SOURCE2000} %{buildroot}%{_unitdir}/cpupower.service
install -m644 %{SOURCE2001} %{buildroot}%{_sysconfdir}/sysconfig/cpupower
%endif
%ifarch x86_64
   mkdir -p %{buildroot}%{_mandir}/man8
   pushd tools/power/x86/x86_energy_perf_policy
   %{tools_make} DESTDIR=%{buildroot} install
   popd
   pushd tools/power/x86/turbostat
   %{tools_make} DESTDIR=%{buildroot} install
   popd
   pushd tools/power/x86/intel-speed-select
   %{tools_make} CFLAGS+="-D_GNU_SOURCE -Iinclude" DESTDIR=%{buildroot} install
   popd
%endif
pushd tools/thermal/tmon
%{tools_make} INSTALL_ROOT=%{buildroot} install
popd
pushd tools/iio
%{tools_make} DESTDIR=%{buildroot} install
popd
pushd tools/gpio
%{tools_make} DESTDIR=%{buildroot} install
popd
install -m644 -D %{SOURCE2002} %{buildroot}%{_sysconfdir}/logrotate.d/kvm_stat
pushd tools/kvm/kvm_stat
make INSTALL_ROOT=%{buildroot} install-tools
make INSTALL_ROOT=%{buildroot} install-man
install -m644 -D kvm_stat.service %{buildroot}%{_unitdir}/kvm_stat.service
popd
# install VM tools
pushd tools/vm/
install -m755 slabinfo %{buildroot}%{_bindir}/slabinfo
install -m755 page_owner_sort %{buildroot}%{_bindir}/page_owner_sort
popd
%endif

%if !%{with_realtime}
%if %{with_bpftool}
pushd tools/bpf/bpftool
%{bpftool_make} prefix=%{_prefix} bash_compdir=%{_sysconfdir}/bash_completion.d/ mandir=%{_mandir} install doc-install
popd
%endif
%endif

%if %{with_selftests}
pushd samples
install -d %{buildroot}%{_libexecdir}/ksamples
# install bpf samples
pushd bpf
install -d %{buildroot}%{_libexecdir}/ksamples/bpf
find -type f -executable -exec install -m755 {} %{buildroot}%{_libexecdir}/ksamples/bpf \;
install -m755 *.sh %{buildroot}%{_libexecdir}/ksamples/bpf
# test_lwt_bpf.sh compiles test_lwt_bpf.c when run; this works only from the
# kernel tree. Just remove it.
rm %{buildroot}%{_libexecdir}/ksamples/bpf/test_lwt_bpf.sh
install -m644 *_kern.o %{buildroot}%{_libexecdir}/ksamples/bpf
install -m644 tcp_bpf.readme %{buildroot}%{_libexecdir}/ksamples/bpf
popd
# install pktgen samples
pushd pktgen
install -d %{buildroot}%{_libexecdir}/ksamples/pktgen
find . -type f -executable -exec install -m755 {} %{buildroot}%{_libexecdir}/ksamples/pktgen/{} \;
find . -type f ! -executable -exec install -m644 {} %{buildroot}%{_libexecdir}/ksamples/pktgen/{} \;
popd
popd
# install drivers/net/mlxsw selftests
pushd tools/testing/selftests/drivers/net/mlxsw
find -type d -exec install -d %{buildroot}%{_libexecdir}/kselftests/drivers/net/mlxsw/{} \;
find -type f -executable -exec install -D -m755 {} %{buildroot}%{_libexecdir}/kselftests/drivers/net/mlxsw/{} \;
find -type f ! -executable -exec install -D -m644 {} %{buildroot}%{_libexecdir}/kselftests/drivers/net/mlxsw/{} \;
popd
# install drivers/net/netdevsim selftests
pushd tools/testing/selftests/drivers/net/netdevsim
find -type d -exec install -d %{buildroot}%{_libexecdir}/kselftests/drivers/net/netdevsim/{} \;
find -type f -executable -exec install -D -m755 {} %{buildroot}%{_libexecdir}/kselftests/drivers/net/netdevsim/{} \;
find -type f ! -executable -exec install -D -m644 {} %{buildroot}%{_libexecdir}/kselftests/drivers/net/netdevsim/{} \;
popd
# install net/forwarding selftests
pushd tools/testing/selftests/net/forwarding
find -type d -exec install -d %{buildroot}%{_libexecdir}/kselftests/net/forwarding/{} \;
find -type f -executable -exec install -D -m755 {} %{buildroot}%{_libexecdir}/kselftests/net/forwarding/{} \;
find -type f ! -executable -exec install -D -m644 {} %{buildroot}%{_libexecdir}/kselftests/net/forwarding/{} \;
popd
# install net/mptcp selftests
pushd tools/testing/selftests/net/mptcp
find -type d -exec install -d %{buildroot}%{_libexecdir}/kselftests/net/mptcp/{} \;
find -type f -executable -exec install -D -m755 {} %{buildroot}%{_libexecdir}/kselftests/net/mptcp/{} \;
find -type f ! -executable -exec install -D -m644 {} %{buildroot}%{_libexecdir}/kselftests/net/mptcp/{} \;
popd
# install tc-testing selftests
pushd tools/testing/selftests/tc-testing
find -type d -exec install -d %{buildroot}%{_libexecdir}/kselftests/tc-testing/{} \;
find -type f -executable -exec install -D -m755 {} %{buildroot}%{_libexecdir}/kselftests/tc-testing/{} \;
find -type f ! -executable -exec install -D -m644 {} %{buildroot}%{_libexecdir}/kselftests/tc-testing/{} \;
popd
# install livepatch selftests
pushd tools/testing/selftests/livepatch
find -type d -exec install -d %{buildroot}%{_libexecdir}/kselftests/livepatch/{} \;
find -type f -executable -exec install -D -m755 {} %{buildroot}%{_libexecdir}/kselftests/livepatch/{} \;
find -type f ! -executable -exec install -D -m644 {} %{buildroot}%{_libexecdir}/kselftests/livepatch/{} \;
popd
# install netfilter selftests
pushd tools/testing/selftests/netfilter
find -type d -exec install -d %{buildroot}%{_libexecdir}/kselftests/netfilter/{} \;
find -type f -executable -exec install -D -m755 {} %{buildroot}%{_libexecdir}/kselftests/netfilter/{} \;
find -type f ! -executable -exec install -D -m644 {} %{buildroot}%{_libexecdir}/kselftests/netfilter/{} \;
popd
%endif

# We have to do the headers checksum calculation after the tools install because
# these might end up installing their own set of headers on top of kernel's
%if %{with_headers}
# compute a content hash to export as Provides: kernel-headers-checksum
HEADERS_CHKSUM=$(export LC_ALL=C; find $RPM_BUILD_ROOT/usr/include -type f -name "*.h" \
			! -path $RPM_BUILD_ROOT/usr/include/linux/version.h | \
		 sort | xargs cat | sha1sum - | cut -f 1 -d ' ');
# export the checksum via usr/include/linux/version.h, so the dynamic
# find-provides can grab the hash to update it accordingly
echo "#define KERNEL_HEADERS_CHECKSUM \"$HEADERS_CHKSUM\"" >> $RPM_BUILD_ROOT/usr/include/linux/version.h
%endif

###
### clean
###

%clean
rm -rf $RPM_BUILD_ROOT

###
### scripts
###

%if %{with_tools}
%post -n %{name}-tools-libs
/sbin/ldconfig

%postun -n %{name}-tools-libs
/sbin/ldconfig
%endif

#
# This macro defines a %%post script for a kernel*-devel package.
#	%%kernel_devel_post [<subpackage>]
#
%define kernel_devel_post() \
%{expand:%%post %{?1:%{1}-}devel}\
if [ -f /etc/sysconfig/kernel ]\
then\
    . /etc/sysconfig/kernel || exit $?\
fi\
if [ "$HARDLINK" != "no" -a -x /usr/sbin/hardlink ]\
then\
    (cd /usr/src/kernels/%{KVERREL}%{?1:+%{1}} &&\
     /usr/bin/find . -type f | while read f; do\
       hardlink -c /usr/src/kernels/*%{?dist}.*/$f $f\
     done)\
fi\
%{nil}

#
# This macro defines a %%post script for a kernel*-modules-extra package.
# It also defines a %%postun script that does the same thing.
#	%%kernel_modules_extra_post [<subpackage>]
#
%define kernel_modules_extra_post() \
%{expand:%%post %{?1:%{1}-}modules-extra}\
/sbin/depmod -a %{KVERREL}%{?1:+%{1}}\
%{nil}\
%{expand:%%postun %{?1:%{1}-}modules-extra}\
/sbin/depmod -a %{KVERREL}%{?1:+%{1}}\
%{nil}

#
# This macro defines a %%post script for a kernel*-modules-internal package.
# It also defines a %%postun script that does the same thing.
#	%%kernel_modules_internal_post [<subpackage>]
#
%define kernel_modules_internal_post() \
%{expand:%%post %{?1:%{1}-}modules-internal}\
/sbin/depmod -a %{KVERREL}%{?1:+%{1}}\
%{nil}\
%{expand:%%postun %{?1:%{1}-}modules-internal}\
/sbin/depmod -a %{KVERREL}%{?1:+%{1}}\
%{nil}

%if %{with_realtime}
#
# This macro defines a %%post script for a kernel*-kvm package.
# It also defines a %%postun script that does the same thing.
#	%%kernel_kvm_post [<subpackage>]
#
%define kernel_kvm_post() \
%{expand:%%post %{?1:%{1}-}kvm}\
/sbin/depmod -a %{KVERREL}%{?1:+%{1}}\
%{nil}\
%{expand:%%postun %{?1:%{1}-}kvm}\
/sbin/depmod -a %{KVERREL}%{?1:+%{1}}\
%{nil}
%endif

#
# This macro defines a %%post script for a kernel*-modules package.
# It also defines a %%postun script that does the same thing.
#	%%kernel_modules_post [<subpackage>]
#
%define kernel_modules_post() \
%{expand:%%post %{?1:%{1}-}modules}\
/sbin/depmod -a %{KVERREL}%{?1:+%{1}}\
%{nil}\
%{expand:%%postun %{?1:%{1}-}modules}\
/sbin/depmod -a %{KVERREL}%{?1:+%{1}}\
%{nil}

# This macro defines a %%posttrans script for a kernel package.
#	%%kernel_variant_posttrans [<subpackage>]
# More text can follow to go at the end of this variant's %%post.
#
%define kernel_variant_posttrans() \
%{expand:%%posttrans %{?1:%{1}-}core}\
%if !%{with_realtime}\
if [ -x %{_sbindir}/weak-modules ]\
then\
    %{_sbindir}/weak-modules --add-kernel %{KVERREL}%{?1:+%{1}} || exit $?\
fi\
%endif\
/bin/kernel-install add %{KVERREL}%{?1:+%{1}} /lib/modules/%{KVERREL}%{?1:+%{1}}/vmlinuz || exit $?\
%{nil}

#
# This macro defines a %%post script for a kernel package and its devel package.
#	%%kernel_variant_post [-v <subpackage>] [-r <replace>]
# More text can follow to go at the end of this variant's %%post.
#
%define kernel_variant_post(v:r:) \
%{expand:%%kernel_devel_post %{?-v*}}\
%{expand:%%kernel_modules_post %{?-v*}}\
%{expand:%%kernel_modules_extra_post %{?-v*}}\
%{expand:%%kernel_modules_internal_post %{?-v*}}\
%{expand:%%kernel_variant_posttrans %{?-v*}}\
%{expand:%%post %{?-v*:%{-v*}-}core}\
%{-r:\
if [ `uname -i` == "x86_64" -o `uname -i` == "i386" ] &&\
   [ -f /etc/sysconfig/kernel ]; then\
  /bin/sed -r -i -e 's/^DEFAULTKERNEL=%{-r*}$/DEFAULTKERNEL=kernel%{?-v:-%{-v*}}/' /etc/sysconfig/kernel || exit $?\
fi}\
%{nil}

#
# This macro defines a %%preun script for a kernel package.
#	%%kernel_variant_preun <subpackage>
#
%define kernel_variant_preun() \
%{expand:%%preun %{?1:%{1}-}core}\
/bin/kernel-install remove %{KVERREL}%{?1:+%{1}} /lib/modules/%{KVERREL}%{?1:+%{1}}/vmlinuz || exit $?\
%if !%{with_realtime}\
if [ -x %{_sbindir}/weak-modules ]\
then\
    %{_sbindir}/weak-modules --remove-kernel %{KVERREL}%{?1:+%{1}} || exit $?\
fi\
%endif\
%{nil}

%kernel_variant_preun
%kernel_variant_post -r kernel-smp
%if %{with_realtime}
%kernel_kvm_post
%endif

%kernel_variant_preun debug
%kernel_variant_post -v debug
%if %{with_realtime}
%kernel_kvm_post debug
%endif

%if %{with_zfcpdump}
%kernel_variant_preun zfcpdump
%kernel_variant_post -v zfcpdump
%endif

if [ -x /sbin/ldconfig ]
then
    /sbin/ldconfig -X || exit $?
fi

###
### file lists
###

%if %{with_headers}
%files headers
%defattr(-,root,root)
/usr/include/*
%endif

%if %{with_cross_headers}
%files cross-headers
%defattr(-,root,root)
/usr/*-linux-gnu/include/*
%endif

%if %{with_kernel_abi_stablelists}
%files -n kernel-abi-stablelists
%defattr(-,root,root,-)
/lib/modules/kabi-*
%endif

%if %{with_kabidw_base}
%ifarch x86_64 s390x ppc64 ppc64le aarch64
%files kernel-kabidw-base-internal
%defattr(-,root,root)
/kabidw-base/%{_target_cpu}/*
%endif
%endif

# only some architecture builds need kernel-doc
%if %{with_doc}
%files doc
%defattr(-,root,root)
%{_datadir}/doc/kernel-doc-%{rpmversion}/Documentation/*
%dir %{_datadir}/doc/kernel-doc-%{rpmversion}/Documentation
%dir %{_datadir}/doc/kernel-doc-%{rpmversion}
%endif

%if %{with_perf}
%files -n perf
%defattr(-,root,root)
%{_bindir}/perf
%{_libdir}/libperf-jvmti.so
%dir %{_libdir}/traceevent/plugins
%{_libdir}/traceevent/plugins/*
%dir %{_libexecdir}/perf-core
%{_libexecdir}/perf-core/*
%{_datadir}/perf-core/*
%{_mandir}/man[1-8]/perf*
%{_sysconfdir}/bash_completion.d/perf
%doc linux-%{KVERREL}/tools/perf/Documentation/examples.txt
%{_docdir}/perf-tip/tips.txt

%files -n python3-perf
%defattr(-,root,root)
%{python3_sitearch}/*

%if %{with_debuginfo}
%files -f perf-debuginfo.list -n perf-debuginfo
%defattr(-,root,root)

%files -f python3-perf-debuginfo.list -n python3-perf-debuginfo
%defattr(-,root,root)
%endif
# with_perf
%endif

%if %{with_tools}
%ifarch %{cpupowerarchs}
%defattr(-,root,root)
%files -n %{name}-tools -f cpupower.lang
%{_bindir}/cpupower
%{_datadir}/bash-completion/completions/cpupower
%ifarch x86_64
%{_bindir}/centrino-decode
%{_bindir}/powernow-k8-decode
%endif
%{_unitdir}/cpupower.service
%{_mandir}/man[1-8]/cpupower*
%config(noreplace) %{_sysconfdir}/sysconfig/cpupower
%ifarch x86_64
%{_bindir}/x86_energy_perf_policy
%{_mandir}/man8/x86_energy_perf_policy*
%{_bindir}/turbostat
%{_mandir}/man8/turbostat*
%{_bindir}/intel-speed-select
%endif
# !cpupowerarchs
%else
%files -n %{name}-tools
%defattr(-,root,root)
# cpupowerarchs
%endif
%{_bindir}/tmon
%{_bindir}/iio_event_monitor
%{_bindir}/iio_generic_buffer
%{_bindir}/lsiio
%{_bindir}/lsgpio
%{_bindir}/gpio-hammer
%{_bindir}/gpio-event-mon
%{_mandir}/man1/kvm_stat*
%{_bindir}/kvm_stat
%{_unitdir}/kvm_stat.service
%config(noreplace) %{_sysconfdir}/logrotate.d/kvm_stat
%{_bindir}/page_owner_sort
%{_bindir}/slabinfo

%if %{with_debuginfo}
%files -f %{name}-tools-debuginfo.list -n %{name}-tools-debuginfo
%defattr(-,root,root)
%endif

%ifarch %{cpupowerarchs}
%files -n %{name}-tools-libs
%{_libdir}/libcpupower.so.0
%{_libdir}/libcpupower.so.0.0.1

%files -n %{name}-tools-libs-devel
%{_libdir}/libcpupower.so
%{_includedir}/cpufreq.h
%endif
# with_tools
%endif

%if !%{with_realtime}
%if %{with_bpftool}
%files -n bpftool
%{_sbindir}/bpftool
%{_sysconfdir}/bash_completion.d/bpftool
%{_mandir}/man8/bpftool-cgroup.8.gz
%{_mandir}/man8/bpftool-map.8.gz
%{_mandir}/man8/bpftool-prog.8.gz
%{_mandir}/man8/bpftool-perf.8.gz
%{_mandir}/man8/bpftool.8.gz
%{_mandir}/man7/bpf-helpers.7.gz
%{_mandir}/man8/bpftool-net.8.gz
%{_mandir}/man8/bpftool-feature.8.gz
%{_mandir}/man8/bpftool-btf.8.gz
%{_mandir}/man8/bpftool-gen.8.gz
%{_mandir}/man8/bpftool-struct_ops.8.gz
%{_mandir}/man8/bpftool-iter.8.gz
%{_mandir}/man8/bpftool-link.8.gz

%if %{with_debuginfo}
%files -f bpftool-debuginfo.list -n bpftool-debuginfo
%defattr(-,root,root)
%endif
%endif
%endif

%if %{with_selftests}
%files selftests-internal
%{_libexecdir}/ksamples
%{_libexecdir}/kselftests
%endif

# empty meta-package
%ifnarch %nobuildarches noarch
%files
%defattr(-,root,root)
%endif

%if %{with_gcov}
%ifarch x86_64 s390x ppc64le aarch64
%files gcov
%defattr(-,root,root)
%{_builddir}
%endif
%endif

# This is %%{image_install_path} on an arch where that includes ELF files,
# or empty otherwise.
%define elf_image_install_path %{?kernel_image_elf:%{image_install_path}}

#
# This macro defines the %%files sections for a kernel package
# and its devel and debuginfo packages.
#	%%kernel_variant_files [-k vmlinux] <condition> <subpackage> <without_modules>
#
%define kernel_variant_files(k:) \
%if %{2}\
%{expand:%%files -f %{name}-%{?3:%{3}-}core.list %{?3:%{3}-}core}\
%defattr(-,root,root)\
%{!?_licensedir:%global license %%doc}\
%license linux-%{KVERREL}/COPYING-%{version}\
/lib/modules/%{KVERREL}%{?3:+%{3}}/%{?-k:%{-k*}}%{!?-k:vmlinuz}\
%ghost /%{image_install_path}/%{?-k:%{-k*}}%{!?-k:vmlinuz}-%{KVERREL}%{?3:+%{3}}\
/lib/modules/%{KVERREL}%{?3:+%{3}}/.vmlinuz.hmac \
%ghost /%{image_install_path}/.vmlinuz-%{KVERREL}%{?3:+%{3}}.hmac \
%ifarch aarch64\
/lib/modules/%{KVERREL}%{?3:+%{3}}/dtb \
%ghost /%{image_install_path}/dtb-%{KVERREL}%{?3:+%{3}} \
%endif\
%attr(0600, root, root) /lib/modules/%{KVERREL}%{?3:+%{3}}/System.map\
%ghost %attr(0600, root, root) /boot/System.map-%{KVERREL}%{?3:+%{3}}\
/lib/modules/%{KVERREL}%{?3:+%{3}}/symvers.gz\
/lib/modules/%{KVERREL}%{?3:+%{3}}/config\
%ghost %attr(0600, root, root) /boot/symvers-%{KVERREL}%{?3:+%{3}}.gz\
%ghost %attr(0600, root, root) /boot/initramfs-%{KVERREL}%{?3:+%{3}}.img\
%ghost %attr(0644, root, root) /boot/config-%{KVERREL}%{?3:+%{3}}\
%dir /lib/modules\
%dir /lib/modules/%{KVERREL}%{?3:+%{3}}\
%dir /lib/modules/%{KVERREL}%{?3:+%{3}}/kernel\
/lib/modules/%{KVERREL}%{?3:+%{3}}/build\
/lib/modules/%{KVERREL}%{?3:+%{3}}/source\
/lib/modules/%{KVERREL}%{?3:+%{3}}/updates\
/lib/modules/%{KVERREL}%{?3:+%{3}}/weak-updates\
/lib/modules/%{KVERREL}%{?3:+%{3}}/bls.conf\
%{_datadir}/doc/kernel-keys/%{KVERREL}%{?3:+%{3}}\
%if %{1}\
/lib/modules/%{KVERREL}%{?3:+%{3}}/vdso\
/etc/ld.so.conf.d/%{name}-%{KVERREL}%{?3:+%{3}}.conf\
%endif\
/lib/modules/%{KVERREL}%{?3:+%{3}}/modules.*\
%{expand:%%files -f %{name}-%{?3:%{3}-}modules.list %{?3:%{3}-}modules}\
%defattr(-,root,root)\
%{expand:%%files %{?3:%{3}-}devel}\
%defattr(-,root,root)\
%defverify(not mtime)\
/usr/src/kernels/%{KVERREL}%{?3:+%{3}}\
%{expand:%%files -f %{name}-%{?3:%{3}-}modules-extra.list %{?3:%{3}-}modules-extra}\
%defattr(-,root,root)\
%config(noreplace) /etc/modprobe.d/*-blacklist.conf\
%{expand:%%files -f %{name}-%{?3:%{3}-}modules-internal.list %{?3:%{3}-}modules-internal}\
%if %{with_realtime}\
%{expand:%%files -f %{name}-%{?3:%{3}-}kvm.list %{?3:%{3}-}kvm}\
%endif\
%if %{with_debuginfo}\
%ifnarch noarch\
%{expand:%%files -f debuginfo%{?3}.list %{?3:%{3}-}debuginfo}\
%defattr(-,root,root)\
%endif\
%endif\
%if %{?3:1} %{!?3:0}\
%{expand:%%files %{3}}\
%defattr(-,root,root)\
%endif\
%endif\
%{nil}

%kernel_variant_files  %{with_vdso_install} %{with_up}
%kernel_variant_files  %{with_vdso_install} %{with_debug} debug
%kernel_variant_files  %{with_vdso_install} %{with_zfcpdump} zfcpdump 1

%define kernel_variant_ipaclones(k:) \
%if %{1}\
%if %{with_ipaclones}\
%{expand:%%files %{?2:%{2}-}ipaclones-internal}\
%defattr(-,root,root)\
%defverify(not mtime)\
/usr/src/kernels/%{KVERREL}%{?2:+%{2}}-ipaclones\
%endif\
%endif\
%{nil}

%kernel_variant_ipaclones %{with_up}

# plz don't put in a version string unless you're going to tag
# and build.
#
#
%changelog
* Mon Nov 15 2021 CentOS Sources <bugs@centos.org> - 4.18.0-348.2.1.el8.centos
- Apply debranding changes

* Mon Nov 08 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-348.2.1.el8_5]
- tipc: fix size validations for the MSG_CRYPTO type (Bruno Meneguele) [2020506 2020507] {CVE-2021-43267}
- net: Fix skb->csum update in inet_proto_csum_replace16(). (Balazs Nemeth) [2005981 1975193]

* Wed Nov 03 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-348.1.1.el8_5]
- lib/timerqueue: Rely on rbtree semantics for next timer (Phil Auld) [2014677 2007099] {CVE-2021-20317}
- mm/hmm: make HMM_MIRROR an implicit option (Rafael Aquini) [2017519 1998534]
- mm/hmm: allow HMM_MIRROR on all architectures with MMU (Rafael Aquini) [2017519 1998534]
- mm: don't select MIGRATE_VMA_HELPER from HMM_MIRROR (Rafael Aquini) [2017519 1998534]
- mm: sort out the DEVICE_PRIVATE Kconfig mess (Rafael Aquini) [2017519 1998534]
- ceph: skip existing superblocks that are blocklisted or shut down when mounting (Jeffrey Layton) [2017071 2011462]
- redhat: switch release to zstream (Bruno Meneguele)

* Mon Oct 04 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-348.el8]
- drm/nouveau/fifo/ga102: initialise chid on return from channel creation (Ben Skeggs) [1997878]
- drm/nouveau/ga102-: support ttm buffer moves via copy engine (Ben Skeggs) [1997878]
- drm/nouveau/kms/tu102-: delay enabling cursor until after assign_windows (Ben Skeggs) [1997878]
- drm/nouveau/kms/nv50: workaround EFI GOP window channel format differences (Ben Skeggs) [1997878]
- drm/nouveau/disp: power down unused DP links during init (Ben Skeggs) [1997878]
- drm/nouveau: recognise GA107 (Ben Skeggs) [1997878]

* Sat Oct 02 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-347.el8]
- PCI: Mark TI C667X to avoid bus reset (Alex Williamson) [1975768]

* Fri Oct 01 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-346.el8]
- redhat: switch secureboot kernel image signing to release keys (Bruno Meneguele)
- CI: handle RT branches in a single config (Veronika Kabatova)
- CI: Fix RT check branch name (Veronika Kabatova)
- CI: Drop private CI config (Veronika Kabatova)
- CI: extend template use (Veronika Kabatova)
- Revert "Merge: mt7921e: enable new Mediatek wireless hardware" (Bruno Meneguele) [2009501]
- megaraid_sas: fix concurrent access to ISR between IRQ polling and real interrupt (Tomas Henzl) [2009022]
- scsi: megaraid_sas: mq_poll support (Tomas Henzl) [2009022]
- [PATCH v2] scsi: qla2xxx: Suppress unnecessary log messages during login (Nilesh Javali) [1982186]
- scsi: qla2xxx: Fix excessive messages during device logout (Nilesh Javali) [1982186]
- PCI: pciehp: Ignore Link Down/Up caused by DPC (Myron Stowe) [1981741]
- arm64: kpti: Fix "kpti=off" when KASLR is enabled (Mark Salter) [1979731]
- arm64: Fix CONFIG_ARCH_RANDOM=n build (Mark Salter) [1979731]
- redhat/configs: aarch64: add CONFIG_ARCH_RANDOM (Mark Salter) [1979731]
- arm64: Implement archrandom.h for ARMv8.5-RNG (Mark Salter) [1979731]
- arm64: kconfig: Fix alignment of E0PD help text (Mark Salter) [1979731]
- arm64: Use register field helper in kaslr_requires_kpti() (Mark Salter) [1979731]
- arm64: Simplify early check for broken TX1 when KASLR is enabled (Mark Salter) [1979731]
- arm64: Use a variable to store non-global mappings decision (Mark Salter) [1979731]
- arm64: Don't use KPTI where we have E0PD (Mark Salter) [1979731]
- arm64: Factor out checks for KASLR in KPTI code into separate function (Mark Salter) [1979731]
- redhat/configs: Add CONFIG_ARM64_E0PD (Mark Salter) [1979731]
- arm64: Add initial support for E0PD (Mark Salter) [1979731]
- arm64: cpufeature: Export matrix and other features to userspace (Mark Salter) [1980098]
- arm64: docs: cpu-feature-registers: Document ID_AA64PFR1_EL1 (Mark Salter) [1980098]
- docs/arm64: cpu-feature-registers: Rewrite bitfields that don't follow [e, s] (Mark Salter) [1980098]
- docs/arm64: cpu-feature-registers: Documents missing visible fields (Mark Salter) [1980098]
- arm64: Introduce system_capabilities_finalized() marker (Mark Salter) [1980098]
- arm64: entry.S: Do not preempt from IRQ before all cpufeatures are enabled (Mark Salter) [1980098]
- docs/arm64: elf_hwcaps: Document HWCAP_SB (Mark Salter) [1980098]
- docs/arm64: elf_hwcaps: sort the HWCAP{, 2} documentation by ascending value (Mark Salter) [1980098]
- arm64: cpufeature: Treat ID_AA64ZFR0_EL1 as RAZ when SVE is not enabled (Mark Salter) [1980098]
- arm64: cpufeature: Effectively expose FRINT capability to userspace (Mark Salter) [1980098]
- arm64: cpufeature: Fix feature comparison for CTR_EL0.{CWG,ERG} (Mark Salter) [1980098]
- arm64: Expose FRINT capabilities to userspace (Mark Salter) [1980098]
- arm64: Expose ARMv8.5 CondM capability to userspace (Mark Salter) [1980098]
- docs: arm64: convert perf.txt to ReST format (Mark Salter) [1980098]
- docs: arm64: convert docs to ReST and rename to .rst (Mark Salter) [1980098]
- Documentation/arm64: HugeTLB page implementation (Mark Salter) [1980098]
- Documentation/arm64/sve: Couple of improvements and typos (Mark Salter) [1980098]
- arm64: cpufeature: Fix missing ZFR0 in __read_sysreg_by_encoding() (Mark Salter) [1980098]
- arm64: Expose SVE2 features for userspace (Mark Salter) [1980098]
- arm64: Advertise ARM64_HAS_DCPODP cpu feature (Mark Salter) [1980098]
- arm64: add CVADP support to the cache maintenance helper (Mark Salter) [1980098]
- arm64: Fix minor issues with the dcache_by_line_op macro (Mark Salter) [1980098]
- arm64: Expose DC CVADP to userspace (Mark Salter) [1980098]
- arm64: Handle trapped DC CVADP (Mark Salter) [1980098]
- arm64: HWCAP: encapsulate elf_hwcap (Mark Salter) [1980098]
- arm64: HWCAP: add support for AT_HWCAP2 (Mark Salter) [1980098]
- x86/MCE/AMD, EDAC/mce_amd: Add new SMCA bank types (Aristeu Rozanski) [1965331]
- x86/MCE/AMD, EDAC/mce_amd: Remove struct smca_hwid.xec_bitmap (Aristeu Rozanski) [1965331]
- EDAC, mce_amd: Print ExtErrorCode and description on a single line (Aristeu Rozanski) [1965331]

* Thu Sep 23 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-345.el8]
- e1000e: Do not take care about recovery NVM checksum (Ken Cox) [1984558]
- qrtr: disable CONFIG_QRTR for non x86_64 archs (Íñigo Huguet) [1999642]
- ceph: fix possible null-pointer dereference in ceph_mdsmap_decode() (Jeff Layton) [1989999]
- ceph: fix dereference of null pointer cf (Jeff Layton) [1989999]
- ceph: correctly handle releasing an embedded cap flush (Jeff Layton) [1989999]
- ceph: take snap_empty_lock atomically with snaprealm refcount change (Jeff Layton) [1989999]
- ceph: don't WARN if we're still opening a session to an MDS (Jeff Layton) [1989999]
- rbd: don't hold lock_rwsem while running_list is being drained (Jeff Layton) [1989999]
- rbd: always kick acquire on "acquired" and "released" notifications (Jeff Layton) [1989999]
- ceph: take reference to req->r_parent at point of assignment (Jeff Layton) [1989999]
- ceph: eliminate ceph_async_iput() (Jeff Layton) [1989999]
- ceph: don't take s_mutex in ceph_flush_snaps (Jeff Layton) [1989999]
- ceph: don't take s_mutex in try_flush_caps (Jeff Layton) [1989999]
- ceph: don't take s_mutex or snap_rwsem in ceph_check_caps (Jeff Layton) [1989999]
- ceph: eliminate session->s_gen_ttl_lock (Jeff Layton) [1989999]
- ceph: allow ceph_put_mds_session to take NULL or ERR_PTR (Jeff Layton) [1989999]
- ceph: clean up locking annotation for ceph_get_snap_realm and __lookup_snap_realm (Jeff Layton) [1989999]
- ceph: add some lockdep assertions around snaprealm handling (Jeff Layton) [1989999]
- ceph: decoding error in ceph_update_snap_realm should return -EIO (Jeff Layton) [1989999]
- ceph: add IO size metrics support (Jeff Layton) [1989999]
- ceph: update and rename __update_latency helper to __update_stdev (Jeff Layton) [1989999]
- ceph: simplify the metrics struct (Jeff Layton) [1989999]
- libceph: fix doc warnings in cls_lock_client.c (Jeff Layton) [1989999]
- libceph: remove unnecessary ret variable in ceph_auth_init() (Jeff Layton) [1989999]
- libceph: kill ceph_none_authorizer::reply_buf (Jeff Layton) [1989999]
- ceph: make ceph_queue_cap_snap static (Jeff Layton) [1989999]
- ceph: remove bogus checks and WARN_ONs from ceph_set_page_dirty (Jeff Layton) [1989999]
- libceph: set global_id as soon as we get an auth ticket (Jeff Layton) [1989999]
- libceph: don't pass result into ac->ops->handle_reply() (Jeff Layton) [1989999]
- ceph: fix error handling in ceph_atomic_open and ceph_lookup (Jeff Layton) [1989999]
- ceph: must hold snap_rwsem when filling inode for async create (Jeff Layton) [1989999]
- libceph: Fix spelling mistakes (Jeff Layton) [1989999]
- libceph: don't set global_id until we get an auth ticket (Jeff Layton) [1989999]
- libceph: bump CephXAuthenticate encoding version (Jeff Layton) [1989999]
- ceph: don't allow access to MDS-private inodes (Jeff Layton) [1989999]
- ceph: fix up some bare fetches of i_size (Jeff Layton) [1989999]
- ceph: support getting ceph.dir.rsnaps vxattr (Jeff Layton) [1989999]
- ceph: drop pinned_page parameter from ceph_get_caps (Jeff Layton) [1989999]
- ceph: fix inode leak on getattr error in __fh_to_dentry (Jeff Layton) [1989999]
- ceph: only check pool permissions for regular files (Jeff Layton) [1989999]
- ceph: send opened files/pinned caps/opened inodes metrics to MDS daemon (Jeff Layton) [1989999]
- ceph: avoid counting the same request twice or more (Jeff Layton) [1989999]
- ceph: rename the metric helpers (Jeff Layton) [1989999]
- ceph: fix kerneldoc copypasta over ceph_start_io_direct (Jeff Layton) [1989999]
- ceph: don't use d_add in ceph_handle_snapdir (Jeff Layton) [1989999]
- ceph: don't clobber i_snap_caps on non-I_NEW inode (Jeff Layton) [1989999]
- ceph: fix fall-through warnings for Clang (Jeff Layton) [1989999]
- net: ceph: Fix a typo in osdmap.c (Jeff Layton) [1989999]
- ceph: don't allow type or device number to change on non-I_NEW inodes (Jeff Layton) [1989999]
- ceph: defer flushing the capsnap if the Fb is used (Jeff Layton) [1989999]
- ceph: allow queueing cap/snap handling after putting cap references (Jeff Layton) [1989999]
- ceph: clean up inode work queueing (Jeff Layton) [1989999]
- ceph: fix flush_snap logic after putting caps (Jeff Layton) [1989999]
- libceph: fix "Boolean result is used in bitwise operation" warning (Jeff Layton) [1989999]
- new helper: inode_wrong_type() (Jeff Layton) [1989999]
- kabi: Adding symbol single_release (fs/seq_file.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol single_open (fs/seq_file.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol seq_read (fs/seq_file.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol seq_printf (fs/seq_file.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol seq_lseek (fs/seq_file.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol unregister_chrdev_region (fs/char_dev.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol cdev_init (fs/char_dev.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol cdev_del (fs/char_dev.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol cdev_alloc (fs/char_dev.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol cdev_add (fs/char_dev.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol alloc_chrdev_region (fs/char_dev.c) (Čestmír Kalina) [1945486]
- kabi: Adding symbol pcie_capability_read_word (drivers/pci/access.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pcie_capability_read_dword (drivers/pci/access.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pcie_capability_clear_and_set_word (drivers/pci/access.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_write_config_dword (drivers/pci/access.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_write_config_byte (drivers/pci/access.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_set_power_state (drivers/pci/pci.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_read_config_dword (drivers/pci/access.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_read_config_byte (drivers/pci/access.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_irq_vector (drivers/pci/msi.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_get_device (drivers/pci/search.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_free_irq_vectors (drivers/pci/msi.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol pci_alloc_irq_vectors_affinity (drivers/pci/msi.c) (Čestmír Kalina) [1945485]
- kabi: Adding symbol kexec_crash_loaded (kernel/kexec_core.c) (Čestmír Kalina) [1945491]

* Mon Sep 20 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-344.el8]
- perf/x86/intel: Fix PEBS-via-PT reload base value for Extended PEBS (Michael Petlan) [1998051]
- perf/x86/intel/uncore: Fix Add BW copypasta (Michael Petlan) [1998051]
- perf/x86/intel/uncore: Add BW counters for GT, IA and IO breakdown (Michael Petlan) [1998051]
- Revert "ice: Add initial support framework for LAG" (Michal Schmidt) [1999016]
- net: re-initialize slow_gro flag at gro_list_prepare time (Paolo Abeni) [2002367]
- cxgb4: dont touch blocked freelist bitmap after free (Rahul Lakkireddy) [1998148]
- cxgb4vf: configure ports accessible by the VF (Rahul Lakkireddy) [1961329]
- scsi: lpfc: Fix memory leaks in error paths while issuing ELS RDF/SCR request (Dick Kennedy) [1976332]
- scsi: lpfc: Keep NDLP reference until after freeing the IOCB after ELS handling (Dick Kennedy) [1976332]
- scsi: lpfc: Move initialization of phba->poll_list earlier to avoid crash (Dick Kennedy) [1976332]

* Thu Sep 16 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-343.el8]
- rcu: Avoid unneeded function call in rcu_read_unlock() (Waiman Long) [1997500]
- mt76: connac: do not schedule mac_work if the device is not running (Íñigo Huguet) [1956419 1972045]
- mt7921e: enable module in config (Íñigo Huguet) [1956419 1972045]
- Revert tools/power/cpupower: Read energy_perf_bias from sysfs (Steve Best) [1999926]
- libnvdimm/namespace: Differentiate between probe mapping and runtime mapping (Jeff Moyer) [1795719]
- libnvdimm/pfn_dev: Don't clear device memmap area during generic namespace probe (Jeff Moyer) [1795719]
- perf/x86/intel/uncore: Clean up error handling path of iio mapping (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Fix for iio mapping on Skylake Server (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Generic support for the MMIO type of uncore blocks (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Generic support for the PCI type of uncore blocks (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Rename uncore_notifier to uncore_pci_sub_notifier (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Generic support for the MSR type of uncore blocks (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Parse uncore discovery tables (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Expose an Uncore unit to IIO PMON mapping (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Wrap the max dies calculation into an accessor (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Expose an Uncore unit to PMON mapping (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Validate MMIO address before accessing (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Record the size of mapped area (Michael Petlan) [1837330]
- perf/x86/intel/uncore: Fix oops when counting IMC uncore events on some TGL (Michael Petlan) [1837330]
- crypto: qat - remove unused macro in FW loader (Vladis Dronov) [1920086]
- crypto: qat - check return code of qat_hal_rd_rel_reg() (Vladis Dronov) [1920086]
- crypto: qat - report an error if MMP file size is too large (Vladis Dronov) [1920086]
- crypto: qat - check MMP size before writing to the SRAM (Vladis Dronov) [1920086]
- crypto: qat - return error when failing to map FW (Vladis Dronov) [1920086]
- crypto: qat - enable detection of accelerators hang (Vladis Dronov) [1920086]
- crypto: qat - Fix a double free in adf_create_ring (Vladis Dronov) [1920086]
- crypto: qat - fix error path in adf_isr_resource_alloc() (Vladis Dronov) [1920086]
- crypto: qat - ADF_STATUS_PF_RUNNING should be set after adf_dev_init (Vladis Dronov) [1920086]
- crypto: qat - don't release uninitialized resources (Vladis Dronov) [1920086]
- crypto: qat - fix use of 'dma_map_single' (Vladis Dronov) [1920086]
- crypto: qat - fix unmap invalid dma address (Vladis Dronov) [1920086]
- crypto: qat - fix spelling mistake: "messge" -> "message" (Vladis Dronov) [1920086]
- crypto: qat - reduce size of mapped region (Vladis Dronov) [1920086]
- crypto: qat - change format string and cast ring size (Vladis Dronov) [1920086]
- crypto: qat - fix potential spectre issue (Vladis Dronov) [1920086]
- crypto: qat - configure arbiter mapping based on engines enabled (Vladis Dronov) [1920086]

* Tue Sep 14 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-342.el8]
- selftest: netfilter: add test case for unreplied tcp connections (Florian Westphal) [1991523]
- netfilter: conntrack: do not renew entry stuck in tcp SYN_SENT state (Florian Westphal) [1991523]
- net/sched: store the last executed chain also for clsact egress (Davide Caratti) [1980537]
- ice: fix Tx queue iteration for Tx timestamp enablement (Ken Cox) [1999743]
- perf evsel: Add missing cloning of evsel->use_config_name (Michael Petlan) [1838635]
- perf Documentation: Document intel-hybrid support (Michael Petlan) [1838635]
- perf tests: Skip 'perf stat metrics (shadow stat) test' for hybrid (Michael Petlan) [1838635]
- perf tests: Support 'Convert perf time to TSC' test for hybrid (Michael Petlan) [1838635]
- perf tests: Support 'Session topology' test for hybrid (Michael Petlan) [1838635]
- perf tests: Support 'Parse and process metrics' test for hybrid (Michael Petlan) [1838635]
- perf tests: Support 'Track with sched_switch' test for hybrid (Michael Petlan) [1838635]
- perf tests: Skip 'Setup struct perf_event_attr' test for hybrid (Michael Petlan) [1838635]
- perf tests: Add hybrid cases for 'Roundtrip evsel->name' test (Michael Petlan) [1838635]
- perf tests: Add hybrid cases for 'Parse event definition strings' test (Michael Petlan) [1838635]
- perf record: Uniquify hybrid event name (Michael Petlan) [1838635]
- perf stat: Warn group events from different hybrid PMU (Michael Petlan) [1838635]
- perf stat: Filter out unmatched aggregation for hybrid event (Michael Petlan) [1838635]
- perf stat: Add default hybrid events (Michael Petlan) [1838635]
- perf record: Create two hybrid 'cycles' events by default (Michael Petlan) [1838635]
- perf parse-events: Support event inside hybrid pmu (Michael Petlan) [1838635]
- perf parse-events: Compare with hybrid pmu name (Michael Petlan) [1838635]
- perf parse-events: Create two hybrid raw events (Michael Petlan) [1838635]
- perf parse-events: Create two hybrid cache events (Michael Petlan) [1838635]
- perf parse-events: Create two hybrid hardware events (Michael Petlan) [1838635]
- perf stat: Uniquify hybrid event name (Michael Petlan) [1838635]
- perf pmu: Add hybrid helper functions (Michael Petlan) [1838635]
- perf pmu: Save detected hybrid pmus to a global pmu list (Michael Petlan) [1838635]
- perf pmu: Save pmu name (Michael Petlan) [1838635]
- perf pmu: Simplify arguments of __perf_pmu__new_alias (Michael Petlan) [1838635]
- perf jevents: Support unit value "cpu_core" and "cpu_atom" (Michael Petlan) [1838635]
- tools headers uapi: Update tools's copy of linux/perf_event.h (Michael Petlan) [1838635]

* Fri Sep 10 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-341.el8]
- mptcp: Only send extra TCP acks in eligible socket states (Paolo Abeni) [1997178]
- mptcp: fix possible divide by zero (Paolo Abeni) [1997178]
- mptcp: drop tx skb cache (Paolo Abeni) [1997178]
- mptcp: fix memory leak on address flush (Paolo Abeni) [1997178]
- ice: Only lock to update netdev dev_addr (Michal Schmidt) [1995868]
- ice: restart periodic outputs around time changes (Ken Cox) [1992750]
- ice: Fix perout start time rounding (Ken Cox) [1992750]
- net/sched: ets: fix crash when flipping from 'strict' to 'quantum' (Davide Caratti) [1981184]
- ovl: prevent private clone if bind mount is not allowed (Miklos Szeredi) [1993131] {CVE-2021-3732}
- gfs2: Don't call dlm after protocol is unmounted (Bob Peterson) [1997193]
- gfs2: don't stop reads while withdraw in progress (Bob Peterson) [1997193]
- gfs2: Mark journal inodes as "don't cache" (Bob Peterson) [1997193]
- bpf: bpftool: Add -fno-asynchronous-unwind-tables to BPF Clang invocation (Yauheni Kaliuta) [1997124]
- perf/x86/intel: Apply mid ACK for small core (Michael Petlan) [1838573]
- perf/x86/intel/lbr: Zero the xstate buffer on allocation (Michael Petlan) [1838573]
- perf: Fix task context PMU for Hetero (Michael Petlan) [1838573]
- perf/x86/intel: Fix fixed counter check warning for some Alder Lake (Michael Petlan) [1838573]
- perf/x86/lbr: Remove cpuc->lbr_xsave allocation from atomic context (Michael Petlan) [1838573]
- x86/fpu/xstate: Fix an xstate size check warning with architectural LBRs (Michael Petlan) [1838573]
- perf/x86/rapl: Add support for Intel Alder Lake (Michael Petlan) [1838573]
- perf/x86/cstate: Add Alder Lake CPU support (Michael Petlan) [1838573]
- perf/x86/msr: Add Alder Lake CPU support (Michael Petlan) [1838573]
- perf/x86/intel/uncore: Add Alder Lake support (Michael Petlan) [1838573]
- perf: Extend PERF_TYPE_HARDWARE and PERF_TYPE_HW_CACHE (Michael Petlan) [1838573]
- perf/x86/intel: Add Alder Lake Hybrid support (Michael Petlan) [1838573]
- perf/x86: Support filter_match callback (Michael Petlan) [1838573]
- perf/x86/intel: Add attr_update for Hybrid PMUs (Michael Petlan) [1838573]
- perf/x86: Add structures for the attributes of Hybrid PMUs (Michael Petlan) [1838573]
- perf/x86: Register hybrid PMUs (Michael Petlan) [1838573]
- perf/x86: Factor out x86_pmu_show_pmu_cap (Michael Petlan) [1838573]
- perf/x86: Remove temporary pmu assignment in event_init (Michael Petlan) [1838573]
- perf/x86/intel: Factor out intel_pmu_check_extra_regs (Michael Petlan) [1838573]
- perf/x86/intel: Factor out intel_pmu_check_event_constraints (Michael Petlan) [1838573]
- perf/x86/intel: Factor out intel_pmu_check_num_counters (Michael Petlan) [1838573]
- perf/x86: Hybrid PMU support for extra_regs (Michael Petlan) [1838573]
- perf/x86: Hybrid PMU support for event constraints (Michael Petlan) [1838573]
- perf/x86: Hybrid PMU support for hardware cache event (Michael Petlan) [1838573]
- perf/x86: Hybrid PMU support for unconstrained (Michael Petlan) [1838573]
- perf/x86: Hybrid PMU support for counters (Michael Petlan) [1838573]
- perf/x86: Hybrid PMU support for intel_ctrl (Michael Petlan) [1838573]
- perf/x86/intel: Hybrid PMU support for perf capabilities (Michael Petlan) [1838573]
- perf/x86: Track pmu in per-CPU cpu_hw_events (Michael Petlan) [1838573]
- perf/x86/intel/lbr: Support XSAVES for arch LBR read (Michael Petlan) [1838573]
- perf/x86/intel/lbr: Support XSAVES/XRSTORS for LBR context switch (Michael Petlan) [1838573]
- x86/fpu/xstate: Add helpers for LBR dynamic supervisor feature (Michael Petlan) [1838573]
- x86/fpu/xstate: Support dynamic supervisor feature for LBR (Michael Petlan) [1838573]
- x86/fpu: Use proper mask to replace full instruction mask (Michael Petlan) [1838573]
- x86/cpu: Add helper function to get the type of the current hybrid CPU (Michael Petlan) [1838573]
- x86/cpufeatures: Enumerate Intel Hybrid Technology feature bit (Michael Petlan) [1838573]
- HID: make arrays usage and value to be the same (Benjamin Tissoires) [1974942]
- ACPI: PM: s2idle: Invert Microsoft UUID entry and exit (David Arcari) [1960440]
- platform/x86: amd-pmc: Fix undefined reference to __udivdi3 (David Arcari) [1960440]
- platform/x86: amd-pmc: Fix missing unlock on error in amd_pmc_send_cmd() (David Arcari) [1960440]
- platform/x86: amd-pmc: Use return code on suspend (David Arcari) [1960440]
- platform/x86: amd-pmc: Add new acpi id for future PMC controllers (David Arcari) [1960440]
- platform/x86: amd-pmc: Add support for ACPI ID AMDI0006 (David Arcari) [1960440]
- platform/x86: amd-pmc: Add support for logging s0ix counters (David Arcari) [1960440]
- platform/x86: amd-pmc: Add support for logging SMU metrics (David Arcari) [1960440]
- platform/x86: amd-pmc: call dump registers only once (David Arcari) [1960440]
- platform/x86: amd-pmc: Fix SMU firmware reporting mechanism (David Arcari) [1960440]
- platform/x86: amd-pmc: Fix command completion code (David Arcari) [1960440]
- usb: pci-quirks: disable D3cold on xhci suspend for s2idle on AMD Renoir (David Arcari) [1960440]
- ACPI: PM: Only mark EC GPE for wakeup on Intel systems (David Arcari) [1960440]
- ACPI: PM: Adjust behavior for field problems on AMD systems (David Arcari) [1960440]
- ACPI: PM: s2idle: Add support for new Microsoft UUID (David Arcari) [1960440]
- ACPI: PM: s2idle: Add support for multiple func mask (David Arcari) [1960440]
- ACPI: PM: s2idle: Refactor common code (David Arcari) [1960440]
- ACPI: PM: s2idle: Use correct revision id (David Arcari) [1960440]
- ACPI: PM: s2idle: Add missing LPS0 functions for AMD (David Arcari) [1960440]
- lockd: Fix invalid lockowner cast after vfs_test_lock (Benjamin Coddington) [1986138]

* Thu Sep 02 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-340.el8]
- blk-mq: fix is_flush_rq (Ming Lei) [1992700]
- blk-mq: fix kernel panic during iterating over flush request (Ming Lei) [1992700]

* Fri Aug 27 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-339.el8]
- smb2: fix use-after-free in smb2_ioctl_query_info() (Ronnie Sahlberg) [1952781]
- dm crypt: Avoid percpu_counter spinlock contention in crypt_page_alloc() (Mike Snitzer) [1996854]
- md/raid10: Remove rcu_dereference when it doesn't need rcu lock to protect (Nigel Croxon) [1978115]
- scsi: csiostor: Mark known unused variable as __always_unused (Raju Rangoju) [1961333]
- scsi: csiostor: Fix wrong return value in csio_hw_prep_fw() (Raju Rangoju) [1961333]
- scsi: csiostor: Remove set but not used variable 'rln' (Raju Rangoju) [1961333]
- scsi: csiostor: Return value not required for csio_dfs_destroy (Raju Rangoju) [1961333]
- scsi: csiostor: Fix NULL check before debugfs_remove_recursive (Raju Rangoju) [1961333]
- scsi: csiostor: Don't enable IRQs too early (Raju Rangoju) [1961333]
- scsi: csiostor: Fix spelling typos (Raju Rangoju) [1961333]
- scsi: csiostor: Prefer pcie_capability_read_word() (Raju Rangoju) [1961333]
- scsi: target: cxgbit: Unmap DMA buffer before calling target_execute_cmd() (Raju Rangoju) [1961394]
- net: Use skb_frag_off accessors (Raju Rangoju) [1961394]
- net: Use skb accessors in network drivers (Raju Rangoju) [1961394]
- cxgb4/chtls/cxgbit: Keeping the max ofld immediate data size same in cxgb4 and ulds (Raju Rangoju) [1961394]
- scsi: libcxgbi: Fix a use after free in cxgbi_conn_xmit_pdu() (Raju Rangoju) [1961394]
- scsi: libcxgbi: Use kvzalloc instead of opencoded kzalloc/vzalloc (Raju Rangoju) [1961394]
- scsi: libcxgbi: Remove unnecessary NULL checks for 'tdata' pointer (Raju Rangoju) [1961394]
- scsi: cxgb4i: Remove an unnecessary NULL check for 'cconn' pointer (Raju Rangoju) [1961394]
- scsi: cxgb4i: Clean up a debug printk (Raju Rangoju) [1961394]
- scsi: cxgb4i: Fix dereference of pointer tdata before it is null checked (Raju Rangoju) [1961394]
- scsi: libcxgbi: fix NULL pointer dereference in cxgbi_device_destroy() (Raju Rangoju) [1961394]
- scsi: libcxgbi: remove unused function to stop warning (Raju Rangoju) [1961394]
- scsi: libcxgbi: add a check for NULL pointer in cxgbi_check_route() (Raju Rangoju) [1961394]
- net/chelsio: Delete drive and module versions (Raju Rangoju) [1961394]
- chelsio: Replace zero-length array with flexible-array member (Raju Rangoju) [1961394]
- [netdrv] treewide: prefix header search paths with $(srctree)/ (Raju Rangoju) [1961394]
- libcxgb: fix incorrect ppmax calculation (Raju Rangoju) [1961394]
- scsi: cxgb4i: Fix TLS dependency (Raju Rangoju) [1961394]
- [target] treewide: Use fallthrough pseudo-keyword (Raju Rangoju) [1961394]
- scsi: cxgb4i: Add support for iSCSI segmentation offload (Raju Rangoju) [1961394]
- [target] treewide: Use sizeof_field() macro (Raju Rangoju) [1961394]
- [target] treewide: replace '---help---' in Kconfig files with 'help' (Raju Rangoju) [1961394]
- scsi: cxgb4i: Remove superfluous null check (Raju Rangoju) [1961394]

* Thu Aug 26 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-338.el8]
- KVM: nSVM: avoid picking up unsupported bits from L2 in int_ctl (CVE-2021-3653) (Jon Maloy) [1985413] {CVE-2021-3653}
- KVM: nSVM: always intercept VMLOAD/VMSAVE when nested (CVE-2021-3656) (Jon Maloy) [1985430] {CVE-2021-3656}
- drm/i915/rkl: Remove require_force_probe protection (Lyude Paul) [1985159]
- drm/i915/display: support ddr5 mem types (Lyude Paul) [1992233]
- drm/i915/adl_s: Update ddi buf translation tables (Lyude Paul) [1992233]
- drm/i915/adl_s: Wa_14011765242 is also needed on A1 display stepping (Lyude Paul) [1992233]
- drm/i915/adl_s: Extend Wa_1406941453 (Lyude Paul) [1992233]
- drm/i915: Implement Wa_1508744258 (Lyude Paul) [1992233]
- drm/i915/adl_s: Fix dma_mask_size to 39 bit (Lyude Paul) [1992233]
- drm/i915: Add the missing adls vswing tables (Lyude Paul) [1992233]
- drm/i915: Add Wa_14011060649 (Lyude Paul) [1992233]
- drm/i915/adl_s: Add Interrupt Support (Lyude Paul) [1992233]
- drm/amdgpu: add another Renoir DID (Lyude Paul) [1980900]

* Wed Aug 25 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-337.el8]
- net/mlx5: Fix flow table chaining (Amir Tzin) [1987139]
- openvswitch: fix sparse warning incorrect type (Mark Gray) [1992773]
- openvswitch: fix alignment issues (Mark Gray) [1992773]
- openvswitch: update kdoc OVS_DP_ATTR_PER_CPU_PIDS (Mark Gray) [1992773]
- openvswitch: Introduce per-cpu upcall dispatch (Mark Gray) [1992773]
- KVM: X86: Expose bus lock debug exception to guest (Paul Lai) [1842322]
- KVM: X86: Add support for the emulation of DR6_BUS_LOCK bit (Paul Lai) [1842322]
- scsi: libfc: Fix array index out of bound exception (Chris Leech) [1972643]
- scsi: libfc: FDMI enhancements (Chris Leech) [1972643]
- scsi: libfc: Add FDMI-2 attributes (Chris Leech) [1972643]
- scsi: qedf: Add vendor identifier attribute (Chris Leech) [1972643]
- scsi: libfc: Initialisation of RHBA and RPA attributes (Chris Leech) [1972643]
- scsi: libfc: Correct the condition check and invalid argument passed (Chris Leech) [1972643]
- scsi: libfc: Work around -Warray-bounds warning (Chris Leech) [1972643]
- scsi: fc: FDMI enhancement (Chris Leech) [1972643]
- scsi: libfc: Move scsi/fc_encode.h to libfc (Chris Leech) [1972643]
- scsi: fc: Correct RHBA attributes length (Chris Leech) [1972643]
- block: return ELEVATOR_DISCARD_MERGE if possible (Ming Lei) [1991976]
- x86/fpu: Prevent state corruption in __fpu__restore_sig() (Terry Bowman) [1970086]
- x86/fpu: Invalidate FPU state after a failed XRSTOR from a user buffer (Terry Bowman) [1970086]
- x86/pkru: Write hardware init value to PKRU when xstate is init (Terry Bowman) [1970086]
- x86/process: Check PF_KTHREAD and not current->mm for kernel threads (Terry Bowman) [1970086]
- x86/fpu: Add address range checks to copy_user_to_xstate() (Terry Bowman) [1970086]
- selftests/x86: Test signal frame XSTATE header corruption handling (Terry Bowman) [1970086]
- Bump DRM backport version to 5.12.14 (Lyude Paul) [1944405]
- drm/i915: Use the correct max source link rate for MST (Lyude Paul) [1944405 1966599]
- drm/dp_mst: Use Extended Base Receiver Capability DPCD space (Lyude Paul) [1944405 1966599]
- drm/i915/display: Defeature PSR2 for RKL and ADL-S (Lyude Paul) [1944405]
- drm/i915/adl_s: ADL-S platform Update PCI ids for Mobile BGA (Lyude Paul) [1944405]
- drm/amdgpu: wait for moving fence after pinning (Lyude Paul) [1944405]
- drm/radeon: wait for moving fence after pinning (Lyude Paul) [1944405]
- drm/nouveau: wait for moving fence after pinning v2 (Lyude Paul) [1944405]
- radeon: use memcpy_to/fromio for UVD fw upload (Lyude Paul) [1944405]
- drm/amd/amdgpu:save psp ring wptr to avoid attack (Lyude Paul) [1944405]
- drm/amd/display: Fix potential memory leak in DMUB hw_init (Lyude Paul) [1944405]
- drm/amdgpu: refine amdgpu_fru_get_product_info (Lyude Paul) [1944405]
- drm/amd/display: Allow bandwidth validation for 0 streams. (Lyude Paul) [1944405]
- drm: Lock pointer access in drm_master_release() (Lyude Paul) [1944405]
- drm: Fix use-after-free read in drm_getunique() (Lyude Paul) [1944405]
- drm/amdgpu: make sure we unpin the UVD BO (Lyude Paul) [1944405]
- drm/amdgpu: Don't query CE and UE errors (Lyude Paul) [1944405]
- drm/amdgpu/jpeg3: add cancel_delayed_work_sync before power gate (Lyude Paul) [1944405]
- drm/amdgpu/jpeg2.5: add cancel_delayed_work_sync before power gate (Lyude Paul) [1944405]
- drm/amdgpu/vcn3: add cancel_delayed_work_sync before power gate (Lyude Paul) [1944405]
- amdgpu: fix GEM obj leak in amdgpu_display_user_framebuffer_create (Lyude Paul) [1944405]
- drm/i915/selftests: Fix return value check in live_breadcrumbs_smoketest() (Lyude Paul) [1944405]
- drm/amdgpu: stop touching sched.ready in the backend (Lyude Paul) [1944405]
- drm/amd/amdgpu: fix a potential deadlock in gpu reset (Lyude Paul) [1944405]
- drm/amdgpu: Fix a use-after-free (Lyude Paul) [1944405]
- drm/amd/amdgpu: fix refcount leak (Lyude Paul) [1944405]
- drm/amd/display: Disconnect non-DP with no EDID (Lyude Paul) [1944405]
- drm/amdgpu/jpeg2.0: add cancel_delayed_work_sync before power gate (Lyude Paul) [1944405]
- drm/amdgpu/vcn2.5: add cancel_delayed_work_sync before power gate (Lyude Paul) [1944405]
- drm/amdgpu/vcn2.0: add cancel_delayed_work_sync before power gate (Lyude Paul) [1944405]
- drm/amdkfd: correct sienna_cichlid SDMA RLC register offset error (Lyude Paul) [1944405]
- drm/amdgpu/vcn1: add cancel_delayed_work_sync before power gate (Lyude Paul) [1944405]
- drm/amd/pm: correct MGpuFanBoost setting (Lyude Paul) [1944405]
- drm/i915: Reenable LTTPR non-transparent LT mode for DPCD_REV<1.4 (Lyude Paul) [1944405]
- drm/i915/gt: Disable HiZ Raw Stall Optimization on broken gen7 (Lyude Paul) [1944405]
- dma-buf: fix unintended pin/unpin warnings (Lyude Paul) [1944405]
- drm/amdgpu: update sdma golden setting for Navi12 (Lyude Paul) [1944405]
- drm/amdgpu: update gc golden setting for Navi12 (Lyude Paul) [1944405]
- drm/amdgpu: disable 3DCGCG on picasso/raven1 to avoid compute hang (Lyude Paul) [1944405]
- drm/amdgpu: Fix GPU TLB update error when PAGE_SIZE > AMDGPU_PAGE_SIZE (Lyude Paul) [1944405]
- drm/radeon: use the dummy page for GART if needed (Lyude Paul) [1944405]
- drm/amd/display: Use the correct max downscaling value for DCN3.x family (Lyude Paul) [1944405]
- drm/i915/gem: Pin the L-shape quirked object as unshrinkable (Lyude Paul) [1944405]
- drm/ttm: Do not add non-system domain BO into swap list (Lyude Paul) [1944405]
- drm/amd/display: Fix two cursor duplication when using overlay (Lyude Paul) [1944405]
- amdgpu/pm: Prevent force of DCEFCLK on NAVI10 and SIENNA_CICHLID (Lyude Paul) [1944405]
- drm/i915/display: fix compiler warning about array overrun (Lyude Paul) [1944405]
- drm/i915: Fix crash in auto_retire (Lyude Paul) [1944405]
- drm/i915/overlay: Fix active retire callback alignment (Lyude Paul) [1944405]
- drm/i915: Read C0DRB3/C1DRB3 as 16 bits again (Lyude Paul) [1944405]
- drm/i915/gt: Fix a double free in gen8_preallocate_top_level_pdp (Lyude Paul) [1944405]
- drm/i915/dp: Use slow and wide link training for everything (Lyude Paul) [1944405]
- drm/i915: Avoid div-by-zero on gen2 (Lyude Paul) [1944405]
- drm/amd/display: Initialize attribute for hdcp_srm sysfs file (Lyude Paul) [1944405]
- drm/radeon/dpm: Disable sclk switching on Oland when two 4K 60Hz monitors are connected (Lyude Paul) [1944405]
- drm/radeon: Avoid power table parsing memory leaks (Lyude Paul) [1944405]
- drm/radeon: Fix off-by-one power_state index heap overwrite (Lyude Paul) [1944405]
- drm/amdgpu: Add mem sync flag for IB allocated by SA (Lyude Paul) [1944405]
- drm/amd/display: add handling for hdcp2 rx id list validation (Lyude Paul) [1944405]
- drm/amd/display: fixed divide by zero kernel crash during dsc enablement (Lyude Paul) [1944405]
- drm/amd/display: Force vsync flip when reconfiguring MPCC (Lyude Paul) [1944405]
- arm64: enable tlbi range instructions (Jeremy Linton) [1861872]
- arm64: tlb: Use the TLBI RANGE feature in arm64 (Jeremy Linton) [1861872]
- arm64: tlb: Detect the ARMv8.4 TLBI RANGE feature (Jeremy Linton) [1861872]
- arm64/cpufeature: Add remaining feature bits in ID_AA64ISAR0 register (Jeremy Linton) [1861872]
- arm64: tlbflush: Ensure start/end of address range are aligned to stride (Jeremy Linton) [1861872]
- arm64: Detect the ARMv8.4 TTL feature (Jeremy Linton) [1861872]
- arm64: tlbi: Set MAX_TLBI_OPS to PTRS_PER_PTE (Jeremy Linton) [1861872]

* Tue Aug 24 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-336.el8]
- bpf: Fix integer overflow involving bucket_size (Jiri Olsa) [1992588]
- bpf: Fix leakage due to insufficient speculative store bypass mitigation (Jiri Olsa) [1992588]
- bpf: Introduce BPF nospec instruction for mitigating Spectre v4 (Jiri Olsa) [1992588]
- bpf: Fix OOB read when printing XDP link fdinfo (Jiri Olsa) [1992588]
- bpf, test: fix NULL pointer dereference on invalid expected_attach_type (Jiri Olsa) [1992588]
- bpf: Fix tail_call_reachable rejection for interpreter when jit failed (Jiri Olsa) [1992588]
- bpf: Track subprog poke descriptors correctly and fix use-after-free (Jiri Olsa) [1992588]
- bpf: Fix null ptr deref with mixed tail calls and subprogs (Jiri Olsa) [1992588]
- bpf: Fix leakage under speculation on mispredicted branches (Jiri Olsa) [1992588]
- bpf: Set mac_len in bpf_skb_change_head (Jiri Olsa) [1992588]
- bpf: Prevent writable memory-mapping of read-only ringbuf pages (Jiri Olsa) [1992588]
- bpf: Fix alu32 const subreg bound tracking on bitwise operations (Jiri Olsa) [1992588]
- xsk: Fix broken Tx ring validation (Jiri Olsa) [1992588]
- xsk: Fix for xp_aligned_validate_desc() when len == chunk_size (Jiri Olsa) [1992588]
- bpf: link: Refuse non-O_RDWR flags in BPF_OBJ_GET (Jiri Olsa) [1992588]
- bpf: Refcount task stack in bpf_get_task_stack (Jiri Olsa) [1992588]
- bpf: Use NOP_ATOMIC5 instead of emit_nops(&prog, 5) for BPF_TRAMP_F_CALL_ORIG (Jiri Olsa) [1992588]
- selftest/bpf: Add a test to check trampoline freeing logic. (Jiri Olsa) [1992588]
- bpf: Fix fexit trampoline. (Jiri Olsa) [1992588]
- ftrace: Fix modify_ftrace_direct. (Jiri Olsa) [1992588]
- ftrace: Add a helper function to modify_ftrace_direct() to allow arch optimization (Jiri Olsa) [1992588]
- ftrace: Add helper find_direct_entry() to consolidate code (Jiri Olsa) [1992588]
- bpf: Fix truncation handling for mod32 dst reg wrt zero (Jiri Olsa) [1992588]
- bpf: Fix an unitialized value in bpf_iter (Jiri Olsa) [1992588]
- bpf_lru_list: Read double-checked variable once without lock (Jiri Olsa) [1992588]
- mt76: validate rx A-MSDU subframes (Íñigo Huguet) [1991459] {CVE-2020-24588 CVE-2020-26144}
- ath11k: Drop multicast fragments (Íñigo Huguet) [1991459] {CVE-2020-26145}
- ath11k: Clear the fragment cache during key install (Íñigo Huguet) [1991459] {CVE-2020-24587}
- ath10k: Validate first subframe of A-MSDU before processing the list (Íñigo Huguet) [1991459] {CVE-2020-24588 CVE-2020-26144}
- ath10k: Fix TKIP Michael MIC verification for PCIe (Íñigo Huguet) [1991459] {CVE-2020-26141}
- ath10k: drop MPDU which has discard flag set by firmware for SDIO (Íñigo Huguet) [1991459] {CVE-2020-24588}
- ath10k: drop fragments with multicast DA for SDIO (Íñigo Huguet) [1991459] {CVE-2020-26145}
- ath10k: drop fragments with multicast DA for PCIe (Íñigo Huguet) [1991459] {CVE-2020-26145}
- ath10k: add CCMP PN replay protection for fragmented frames for PCIe (Íñigo Huguet) [1991459]
- mac80211: extend protection against mixed key and fragment cache attacks (Íñigo Huguet) [1991459] {CVE-2020-24586 CVE-2020-24587}
- mac80211: do not accept/forward invalid EAPOL frames (Íñigo Huguet) [1991459] {CVE-2020-26139}
- mac80211: prevent attacks on TKIP/WEP as well (Íñigo Huguet) [1991459] {CVE-2020-26141}
- mac80211: check defrag PN against current frame (Íñigo Huguet) [1991459]
- mac80211: add fragment cache to sta_info (Íñigo Huguet) [1991459] {CVE-2020-24586 CVE-2020-24587}
- mac80211: drop A-MSDUs on old ciphers (Íñigo Huguet) [1991459] {CVE-2020-24588}
- cfg80211: mitigate A-MSDU aggregation attacks (Íñigo Huguet) [1991459] {CVE-2020-24588 CVE-2020-26144}
- mac80211: properly handle A-MSDUs that start with an RFC 1042 header (Íñigo Huguet) [1991459]
- mac80211: prevent mixed key and fragment cache attacks (Íñigo Huguet) [1991459] {CVE-2020-24586 CVE-2020-24587}
- mac80211: assure all fragments are encrypted (Íñigo Huguet) [1991459] {CVE-2020-26147}
- tipc: call tipc_wait_for_connect only when dlen is not 0 (Xin Long) [1989361]
- mptcp: remove tech preview warning (Florian Westphal) [1985120]
- tcp: consistently disable header prediction for mptcp (Florian Westphal) [1985120]
- selftests: mptcp: fix case multiple subflows limited by server (Florian Westphal) [1985120]
- selftests: mptcp: turn rp_filter off on each NIC (Florian Westphal) [1985120]
- selftests: mptcp: display proper reason to abort tests (Florian Westphal) [1985120]
- mptcp: properly account bulk freed memory (Florian Westphal) [1985120]
- mptcp: fix 'masking a bool' warning (Florian Westphal) [1985120]
- mptcp: refine mptcp_cleanup_rbuf (Florian Westphal) [1985120]
- mptcp: use fast lock for subflows when possible (Florian Westphal) [1985120]
- mptcp: avoid processing packet if a subflow reset (Florian Westphal) [1985120]
- mptcp: add sk parameter for mptcp_get_options (Florian Westphal) [1985120]
- mptcp: fix syncookie process if mptcp can not_accept new subflow (Florian Westphal) [1985120]
- mptcp: fix warning in __skb_flow_dissect() when do syn cookie for subflow join (Florian Westphal) [1985120]
- mptcp: avoid race on msk state changes (Florian Westphal) [1985120]
- mptcp: fix 32 bit DSN expansion (Florian Westphal) [1985120]
- mptcp: fix bad handling of 32 bit ack wrap-around (Florian Westphal) [1985120]
- tcp: parse mptcp options contained in reset packets (Florian Westphal) [1985120]
- ionic: count csum_none when offload enabled (Jonathan Toppins) [1991646]
- ionic: fix up dim accounting for tx and rx (Jonathan Toppins) [1991646]
- ionic: remove intr coalesce update from napi (Jonathan Toppins) [1991646]
- ionic: catch no ptp support earlier (Jonathan Toppins) [1991646]
- ionic: make all rx_mode work threadsafe (Jonathan Toppins) [1991646]
- dmaengine: idxd: Fix missing error code in idxd_cdev_open() (Jerry Snitselaar) [1990637]
- dmaengine: idxd: add missing dsa driver unregister (Jerry Snitselaar) [1990637]
- dmaengine: idxd: add engine 'struct device' missing bus type assignment (Jerry Snitselaar) [1990637]
- dmaengine: idxd: remove MSIX masking for interrupt handlers (Jerry Snitselaar) [1990637]
- dmaengine: idxd: Use cpu_feature_enabled() (Jerry Snitselaar) [1990637]
- dmaengine: idxd: enable SVA feature for IOMMU (Jerry Snitselaar) [1990637]
- dmagenine: idxd: Don't add portal offset in idxd_submit_desc (Jerry Snitselaar) [1990637]
- ethtool: strset: fix message length calculation (Balazs Nemeth) [1989003]
- net: add strict checks in netdev_name_node_alt_destroy() (Andrea Claudi) [1859038]
- net: rtnetlink: fix bugs in rtnl_alt_ifname() (Andrea Claudi) [1859038]
- net: rtnetlink: add linkprop commands to add and delete alternative ifnames (Andrea Claudi) [1859038]
- net: check all name nodes in __dev_alloc_name (Andrea Claudi) [1859038]
- net: fix a leak in register_netdevice() (Andrea Claudi) [1859038]
- tun: fix memory leak in error path (Andrea Claudi) [1859038]
- net: propagate errors correctly in register_netdevice() (Andrea Claudi) [1859038]
- net: introduce name_node struct to be used in hashlist (Andrea Claudi) [1859038]
- net: procfs: use index hashlist instead of name hashlist (Andrea Claudi) [1859038]
- configs: Enable CONFIG_CHELSIO_INLINE_CRYPTO (Raju Rangoju) [1961368]
- cxgb4/ch_ktls: Clear resources when pf4 device is removed (Raju Rangoju) [1961374]
- ch_ktls: Remove redundant variable result (Raju Rangoju) [1961374]
- ch_ktls: do not send snd_una update to TCB in middle (Raju Rangoju) [1961374]
- ch_ktls: tcb close causes tls connection failure (Raju Rangoju) [1961374]
- ch_ktls: fix device connection close (Raju Rangoju) [1961374]
- ch_ktls: Fix kernel panic (Raju Rangoju) [1961374]
- ch_ktls: fix enum-conversion warning (Raju Rangoju) [1961374]
- net: ethernet: chelsio: inline_crypto: Mundane typos fixed throughout the file chcr_ktls.c (Raju Rangoju) [1961374]
- ch_ipsec: Remove initialization of rxq related data (Raju Rangoju) [1961388]
- ch_ktls: fix build warning for ipv4-only config (Raju Rangoju) [1961374]
- ch_ktls: lock is not freed (Raju Rangoju) [1961374]
- ch_ktls: stop the txq if reaches threshold (Raju Rangoju) [1961374]
- ch_ktls: tcb update fails sometimes (Raju Rangoju) [1961374]
- ch_ktls/cxgb4: handle partial tag alone SKBs (Raju Rangoju) [1961374]
- ch_ktls: don't free skb before sending FIN (Raju Rangoju) [1961374]
- ch_ktls: packet handling prior to start marker (Raju Rangoju) [1961374]
- ch_ktls: Correction in middle record handling (Raju Rangoju) [1961374]
- ch_ktls: missing handling of header alone (Raju Rangoju) [1961374]
- ch_ktls: Correction in trimmed_len calculation (Raju Rangoju) [1961374]
- cxgb4/ch_ktls: creating skbs causes panic (Raju Rangoju) [1961374]
- ch_ktls: Update cheksum information (Raju Rangoju) [1961374]
- ch_ktls: Correction in finding correct length (Raju Rangoju) [1961374]
- cxgb4/ch_ktls: decrypted bit is not enough (Raju Rangoju) [1961374]
- cxgb4/ch_ipsec: Replace the module name to ch_ipsec from chcr (Raju Rangoju) [1961388]
- cxgb4/ch_ktls: ktls stats are added at port level (Raju Rangoju) [1961374]
- ch_ktls: Issue if connection offload fails (Raju Rangoju) [1961374]
- chelsio/chtls: Re-add dependencies on CHELSIO_T4 to fix modular CHELSIO_T4 (Raju Rangoju) [1961388]
- chelsio/chtls: CHELSIO_INLINE_CRYPTO should depend on CHELSIO_T4 (Raju Rangoju) [1961388]
- crypto: chelsio - fix minor indentation issue (Raju Rangoju) [1961368]
- crypto/chcr: move nic TLS functionality to drivers/net (Raju Rangoju) [1961368]
- cxgb4/ch_ipsec: Registering xfrmdev_ops with cxgb4 (Raju Rangoju) [1961388]
- crypto/chcr: Moving chelsio's inline ipsec functionality to /drivers/net (Raju Rangoju) [1961368]
- chelsio/chtls: separate chelsio tls driver from crypto driver (Raju Rangoju) [1961368]
- crypto: chelsio - Fix some pr_xxx messages (Raju Rangoju) [1961368]
- crypto: chelsio - Avoid some code duplication (Raju Rangoju) [1961368]
- crypto: drivers - set the flag CRYPTO_ALG_ALLOCATES_MEMORY (Raju Rangoju) [1961368]
- crypto: aead - remove useless setting of type flags (Raju Rangoju) [1961368]
- crypto: Replace zero-length array with flexible-array (Raju Rangoju) [1961368]
- [Crypto] treewide: replace '---help---' in Kconfig files with 'help' (Raju Rangoju) [1961368]
- Crypto/chcr: Checking cra_refcnt before unregistering the algorithms (Raju Rangoju) [1961368]
- Crypto/chcr: Calculate src and dst sg lengths separately for dma map (Raju Rangoju) [1961368]
- Crypto/chcr: Fixes a coccinile check error (Raju Rangoju) [1961368]
- Crypto/chcr: Fixes compilations warnings (Raju Rangoju) [1961368]
- crypto/chcr: IPV6 code needs to be in CONFIG_IPV6 (Raju Rangoju) [1961368]
- crypto: lib/sha1 - remove unnecessary includes of linux/cryptohash.h (Raju Rangoju) [1961368]
- Crypto/chcr: fix for hmac(sha) test fails (Raju Rangoju) [1961368]
- Crypto/chcr: fix for ccm(aes) failed test (Raju Rangoju) [1961368]
- Crypto/chcr: fix ctr, cbc, xts and rfc3686-ctr failed tests (Raju Rangoju) [1961368]
- crypto: chelsio - remove redundant assignment to variable error (Raju Rangoju) [1961368]
- chcr: Fix CPU hard lockup (Raju Rangoju) [1961368]
- crypto: remove CRYPTO_TFM_RES_BAD_KEY_LEN (Raju Rangoju) [1961368]
- crypto: chelsio - switch to skcipher API (Raju Rangoju) [1961368]
- crypto: chelsio - Remove VLA usage of skcipher (Raju Rangoju) [1961368]

* Mon Aug 23 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-335.el8]
- tracing: Fix bug in rb_per_cpu_empty() that might cause deadloop. (Chris von Recklinghausen) [1989485] {CVE-2021-3679}
- vmlinux.lds.h: Keep .ctors.* with .ctors (Jan Stancek) [1993208]
- spi: pxa2xx: Add support for Intel Alder Lake PCH-P (Steve Best) [1978463]
- bnxt_en: allow promiscuous mode for trusted VFs (Jonathan Toppins) [1730616]
- arm64: memory: Add missing brackets to untagged_addr() macro (Chris von Recklinghausen) [1955809]
- arm64: tags: Preserve tags for addresses translated via TTBR1 (Chris von Recklinghausen) [1955809]
- arm64: entry: Move ct_user_exit before any other exception (Chris von Recklinghausen) [1955809]
- arm64: memory: Implement __tag_set() as common function (Chris von Recklinghausen) [1955809]
- arm64: mm: Really fix sparse warning in untagged_addr() (Chris von Recklinghausen) [1955809]
- arm64: untag user pointers in access_ok and __uaccess_mask_ptr (Chris von Recklinghausen) [1955809]
- arm64/mm: fix variable 'tag' set but not used (Chris von Recklinghausen) [1955809]
- arm64: entry: SP Alignment Fault doesn't write to FAR_EL1 (Chris von Recklinghausen) [1955809]
- arm64: compat: Add separate CP15 trapping hook (Chris von Recklinghausen) [1955809]
- arm64: don't restore GPRs when context tracking (Chris von Recklinghausen) [1955809]

* Fri Aug 20 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-334.el8]
- bareudp: Fix invalid read beyond skb's linear data (Guillaume Nault) [1990938]
- perf/x86/amd: Don't touch the AMD64_EVENTSEL_HOSTONLY bit inside the guest (Michael Petlan) [1990695]
- net: dcb: Accept RTM_GETDCB messages carrying set-like DCB commands (Balazs Nemeth) [1956825]
- net: dcb: Validate netlink message in DCB handler (Balazs Nemeth) [1956825]
- xfrm: Fix RCU vs hash_resize_mutex lock inversion (Sabrina Dubroca) [1988405]
- Revert "xfrm: policy: Read seqcount outside of rcu-read side in xfrm_policy_lookup_bytype" (Sabrina Dubroca) [1988405]
- xfrm: policy: Read seqcount outside of rcu-read side in xfrm_policy_lookup_bytype (Sabrina Dubroca) [1988405]
- [s390] s390/AP: support new dynamic AP bus size limit (Claudio Imbrenda) [1974581]
- net: sched: act_mirred: Reset ct info when mirror/redirect skb (Hangbin Liu) [1980532]
- ucounts: Move max_time_namespace according to ucount_type (Alex Gladkov) [1982954]
- perf vendor events power10: Adds 24x7 nest metric events for power10 platform (Diego Domingos) [1946650]
- perf/core: fix backport of PERF_SAMPLE_WEIGHT_STRUCT (Diego Domingos) [1946650]
- perf/core: Add support for PERF_SAMPLE_CODE_PAGE_SIZE (Diego Domingos) [1946650]
- perf vendor events powerpc: Fix eventcode of power10 JSON events (Diego Domingos) [1946650]
- perf vendor events: Initial JSON/events list for power10 platform (Diego Domingos) [1946650]
- powerpc/perf: Fix sampled instruction type for larx/stcx (Diego Domingos) [1946650]
- powerpc/perf: Fix the threshold event selection for memory events in power10 (Diego Domingos) [1946650]
- perf sort: Display sort dimension p_stage_cyc only on supported archs (Diego Domingos) [1946650]
- perf tools: Support pipeline stage cycles for powerpc (Diego Domingos) [1946650]
- perf powerpc: Add support for PERF_SAMPLE_WEIGHT_STRUCT (Diego Domingos) [1946650]
- perf sort: Add dynamic headers for perf report columns (Diego Domingos) [1946650]
- powerpc/perf: Expose processor pipeline stage cycles using PERF_SAMPLE_WEIGHT_STRUCT (Diego Domingos) [1946650]
- Documentation/admin-guide: kernel-parameters: fix "disable_ddw" wording (Diego Domingos) [1946650]
- powerpc/perf: Support PERF_SAMPLE_DATA_PAGE_SIZE (Diego Domingos) [1946650]
- perf/core: Add PERF_SAMPLE_DATA_PAGE_SIZE (Diego Domingos) [1946650]
- powerpc/perf: Infrastructure to support checking of attr.config* (Diego Domingos) [1946650]
- powerpc/perf: Add platform specific check_attr_config (Diego Domingos) [1946650]
- ice: add support for auxiliary input/output pins (Jonathan Toppins) [1956913]
- ice: enable transmit timestamps for E810 devices (Jonathan Toppins) [1944818]
- ice: enable receive hardware timestamping (Jonathan Toppins) [1944818]
- ice: report the PTP clock index in ethtool .get_ts_info (Jonathan Toppins) [1944818]
- ice: register 1588 PTP clock device object for E810 devices (Jonathan Toppins) [1944818]
- ice: add low level PTP clock access functions (Jonathan Toppins) [1944818]
- ice: add support for set/get of driver-stored firmware parameters (Jonathan Toppins) [1944818]
- ice: process 1588 PTP capabilities during initialization (Jonathan Toppins) [1944818]
- ice: add support for sideband messages (Jonathan Toppins) [1944818]

* Wed Aug 18 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-333.el8]
- bnxt_en: Fix static checker warning in bnxt_fw_reset_task() (Jonathan Toppins) [1989274]
- bnxt_en: Check abort error state in bnxt_half_open_nic() (Jonathan Toppins) [1989274]
- bnxt_en: fix error path of FW reset (Jonathan Toppins) [1989274]
- bnxt_en: Add missing check for BNXT_STATE_ABORT_ERR in bnxt_fw_rset_task() (Jonathan Toppins) [1989274]
- bnxt_en: Refresh RoCE capabilities in bnxt_ulp_probe() (Jonathan Toppins) [1989274]
- bnxt_en: reject ETS settings that will starve a TC (Jonathan Toppins) [1989274]
- bnxt_en: don't disable an already disabled PCI device (Jonathan Toppins) [1989274]
- bnxt_en: Remove the read of BNXT_FW_RESET_INPROG_REG after firmware reset. (Jonathan Toppins) [1989274]
- i40e: Fix log TC creation failure when max num of queues is exceeded (Stefan Assmann) [1920274]
- i40e: Fix queue-to-TC mapping on Tx (Stefan Assmann) [1920274]
- i40e: Add additional info to PHY type error (Stefan Assmann) [1920274]
- i40e: Fix firmware LLDP agent related warning (Stefan Assmann) [1920274]
- i40e: Fix logic of disabling queues (Stefan Assmann) [1920274]
- i40e: add support for PTP external synchronization clock (Stefan Assmann) [1920274]
- i40e: improve locking of mac_filter_hash (Stefan Assmann) [1920274]
- i40e: Fix missing rtnl locking when setting up pf switch (Stefan Assmann) [1920274]
- i40e: fix PTP on 5Gb links (Stefan Assmann) [1920274]
- i40e: Fix autoneg disabling for non-10GBaseT links (Stefan Assmann) [1920274]
- i40e: Fix error handling in i40e_vsi_open (Stefan Assmann) [1920274]
- intel: Remove rcu_read_lock() around XDP program invocation (Stefan Assmann) [1920274]
- i40e: clean up packet type lookup table (Stefan Assmann) [1920274]
- i40e: add correct exception tracing for XDP (Stefan Assmann) [1920274]
- i40e: Remove LLDP frame filters (Stefan Assmann) [1920274]
- i40e: Fix PHY type identifiers for 2.5G and 5G adapters (Stefan Assmann) [1920274]
- i40e: Fix use-after-free in i40e_client_subtask() (Stefan Assmann) [1920274]
- i40e: fix broken XDP support (Stefan Assmann) [1920274]
- i40e: refactor repeated link state reporting code (Stefan Assmann) [1920274]
- i40e: optimize for XDP_REDIRECT in xsk path (Stefan Assmann) [1920274]
- i40e: fix the panic when running bpf in xdpdrv mode (Stefan Assmann) [1920274]
- i40e: Fix sparse warning: missing error code 'err' (Stefan Assmann) [1920274]
- i40e: Fix sparse error: 'vsi->netdev' could be null (Stefan Assmann) [1920274]
- i40e: Fix sparse error: uninitialized symbol 'ring' (Stefan Assmann) [1920274]
- i40e: Fix sparse errors in i40e_txrx.c (Stefan Assmann) [1920274]
- i40e: Fix display statistics for veb_tc (Stefan Assmann) [1920274]
- i40e: fix receiving of single packets in xsk zero-copy mode (Stefan Assmann) [1920274]
- i40e: Fix inconsistent indenting (Stefan Assmann) [1920274]
- i40e: Fix oops at i40e_rebuild() (Stefan Assmann) [1920274]
- i40e: Fix kernel oops when i40e driver removes VF's (Stefan Assmann) [1920274]
- i40e: Added Asym_Pause to supported link modes (Stefan Assmann) [1920274]
- net: i40e: remove repeated words (Stefan Assmann) [1920274]
- bpf, devmap: Move drop error path to devmap for XDP_REDIRECT (Stefan Assmann) [1920274]
- intel: clean up mismatched header comments (Stefan Assmann) [1920274]
- intel: Update drivers to use ethtool_sprintf (Stefan Assmann) [1920274]
- i40e: move headroom initialization to i40e_configure_rx_ring (Stefan Assmann) [1920274]
- i40e: Fix endianness conversions (Stefan Assmann) [1920274]
- i40e: Fix add TC filter for IPv6 (Stefan Assmann) [1920274]
- i40e: Fix addition of RX filters after enabling FW LLDP agent (Stefan Assmann) [1920274]
- i40e: Fix overwriting flow control settings during driver loading (Stefan Assmann) [1920274]
- i40e: Add zero-initialization of AQ command structures (Stefan Assmann) [1920274]
- i40e: Fix memory leak in i40e_probe (Stefan Assmann) [1920274]
- i40e: Fix flow for IPv6 next header (extension header) (Stefan Assmann) [1920274]
- i40e: Fix incorrect argument in call to ipv6_addr_any() (Stefan Assmann) [1920274]
- i40e: store the result of i40e_rx_offset() onto i40e_ring (Stefan Assmann) [1920274]
- i40e: Simplify the do-while allocation loop (Stefan Assmann) [1920274]
- i40e: adjust i40e_is_non_eop (Stefan Assmann) [1920274]
- i40e: drop misleading function comments (Stefan Assmann) [1920274]
- i40e: drop redundant check when setting xdp prog (Stefan Assmann) [1920274]
- i40e: remove the useless value assignment in i40e_clean_adminq_subtask (Stefan Assmann) [1920274]
- i40e: VLAN field for flow director (Stefan Assmann) [1920274]
- i40e: Add flow director support for IPv6 (Stefan Assmann) [1920274]
- i40e: Add EEE status getting & setting implementation (Stefan Assmann) [1920274]
- i40e: Fix uninitialized variable mfs_max (Stefan Assmann) [1920274]
- i40e: Add netlink callbacks support for software based DCB (Stefan Assmann) [1920274]
- i40e: Add init and default config of software based DCB (Stefan Assmann) [1920274]
- i40e: Add hardware configuration for software based DCB (Stefan Assmann) [1920274]
- i40e: Log error for oversized MTU on device (Stefan Assmann) [1920274]
- i40e: consolidate handling of XDP program actions (Stefan Assmann) [1920274]
- i40e: remove the redundant buffer info updates (Stefan Assmann) [1920274]
- i40e: remove unnecessary cleaned_count updates (Stefan Assmann) [1920274]
- i40e: remove unnecessary memory writes of the next to clean pointer (Stefan Assmann) [1920274]
- i40e: Use batched xsk Tx interfaces to increase performance (Stefan Assmann) [1920274]
- i40e: convert to new udp_tunnel infrastructure (Stefan Assmann) [1920274]
- netfilter: nf_tables: skip module reference count bump on object updates (Fernando Fernandez Mancera) [1944487]
- netfilter: nf_tables: fix unexpected EOPNOTSUPP error (Fernando Fernandez Mancera) [1944487]
- netfilter: nf_tables: Fix an Oops in nf_tables_updobj() error handling (Fernando Fernandez Mancera) [1944487]
- netfilter: nf_tables: fix possible null-pointer dereference in object update (Fernando Fernandez Mancera) [1944487]
- netfilter: nft_quota: add quota object update support (Fernando Fernandez Mancera) [1944487]
- netfilter: nf_tables: Introduce stateful object update operation (Fernando Fernandez Mancera) [1944487]
- cifs: retry lookup and readdir when EAGAIN is returned. (Ronnie Sahlberg) [1972411]
- netfilter: nf_tables: Fix dereference of null pointer flow (Florian Westphal) [1985087]
- netfilter: nf_tables: memleak in hw offload abort path (Florian Westphal) [1985087]
- netfilter: nft_tproxy: restrict support to TCP and UDP transport protocols (Florian Westphal) [1985087]
- netfilter: synproxy: Fix out of bounds when parsing TCP options (Florian Westphal) [1985087]
- netfilter: nfnetlink_cthelper: hit EBUSY on updates if size mismatches (Florian Westphal) [1985087]
- ipvs: ignore IP_VS_SVC_F_HASHED flag when adding service (Florian Westphal) [1985087]
- netfilter: nftables: avoid overflows in nft_hash_buckets() (Florian Westphal) [1985087]
- netfilter: nfnetlink_osf: Fix a missing skb_header_pointer() NULL check (Florian Westphal) [1985087]
- netfilter: nf_tables_offload: check FLOW_DISSECTOR_KEY_BASIC in VLAN transfer logic (Florian Westphal) [1985087]
- netfilter: nftables_offload: special ethertype handling for VLAN (Florian Westphal) [1985087]
- netfilter: nftables_offload: VLAN id needs host byteorder in flow dissector (Florian Westphal) [1985087]
- netfilter: nft_payload: fix C-VLAN offload support (Florian Westphal) [1985087]
- ipvs: allow connection reuse for unconfirmed conntrack (Florian Westphal) [1985087]
- net: fix GRO skb truesize update (Paolo Abeni) [1987391]
- sk_buff: avoid potentially clearing 'slow_gro' field (Paolo Abeni) [1987391]
- veth: use skb_prepare_for_gro() (Paolo Abeni) [1987391]
- skbuff: allow 'slow_gro' for skb carring sock reference (Paolo Abeni) [1987391]
- net: optimize GRO for the common case. (Paolo Abeni) [1987391]
- sk_buff: track extension status in slow_gro (Paolo Abeni) [1987391]
- sk_buff: track dst status in slow_gro (Paolo Abeni) [1987391]
- sk_buff: introduce 'slow_gro' flags (Paolo Abeni) [1987391]
- skbuff: Fix build with SKB extensions disabled (Paolo Abeni) [1987391]
- skbuff: Release nfct refcount on napi stolen or re-used skbs (Paolo Abeni) [1987391]
- selftests: net: veth: add tests for set_channel (Paolo Abeni) [1987391]
- veth: create by default nr_possible_cpus queues (Paolo Abeni) [1987391]
- veth: implement support for set_channel ethtool op (Paolo Abeni) [1987391]
- veth: factor out initialization helper (Paolo Abeni) [1987391]
- veth: always report zero combined channels (Paolo Abeni) [1987391]
- veth: Implement ethtool's get_channelis() callback (Paolo Abeni) [1987391]
- net: add GSO UDP L4 and GSO fraglists to the list of software-backed types (Paolo Abeni) [1987391]
- ice: Support RSS configure removal for AVF (Jonathan Toppins) [1946726]
- ice: Enable RSS configure for AVF (Jonathan Toppins) [1946726]
- ice: Add helper function to get the VF's VSI (Jonathan Toppins) [1946726]
- ice: remove redundant assignment to pointer vsi (Jonathan Toppins) [1946726]
- ice: Advertise virtchnl UDP segmentation offload capability (Jonathan Toppins) [1946726]
- ice: Allow ignoring opcodes on specific VF (Jonathan Toppins) [1946726]
- ice: warn about potentially malicious VFs (Jonathan Toppins) [1946726]
- ice: Consolidate VSI state and flags (Jonathan Toppins) [1946726]
- ice: Refactor ice_set/get_rss into LUT and key specific functions (Jonathan Toppins) [1946726]
- ice: Refactor get/set RSS LUT to use struct parameter (Jonathan Toppins) [1946726]
- ice: Change ice_vsi_setup_q_map() to not depend on RSS (Jonathan Toppins) [1946726]
- ice: Check FDIR program status for AVF (Jonathan Toppins) [1946726]
- ice: Add more FDIR filter type for AVF (Jonathan Toppins) [1946726]
- ice: Add GTPU FDIR filter for AVF (Jonathan Toppins) [1946726]
- ice: Add non-IP Layer2 protocol FDIR filter for AVF (Jonathan Toppins) [1946726]
- ice: Add new actions support for VF FDIR (Jonathan Toppins) [1946726]
- ice: Add FDIR pattern action parser for VF (Jonathan Toppins) [1946726]
- ice: Enable FDIR Configure for AVF (Jonathan Toppins) [1946726]
- ice: Add support for per VF ctrl VSI enabling (Jonathan Toppins) [1946726]
- ice: Enhanced IPv4 and IPv6 flow filter (Jonathan Toppins) [1946726]
- ice: Support to separate GTP-U uplink and downlink (Jonathan Toppins) [1946726]
- ice: Add more advanced protocol support in flow filter (Jonathan Toppins) [1946726]
- ice: Support non word aligned input set field (Jonathan Toppins) [1946726]
- ice: Add more basic protocol support for flow filter (Jonathan Toppins) [1946726]
- sctp: move the active_key update after sh_keys is added (Xin Long) [1986966]
- sctp: fix return value check in __sctp_rcv_asconf_lookup (Xin Long) [1986966]
- sctp: delete addr based on sin6_scope_id (Xin Long) [1986966]
- sctp: update active_key for asoc when old key is being replaced (Xin Long) [1986966]
- sctp: move 198 addresses from unusable to private scope (Xin Long) [1986966]
- sctp: add param size validation for SCTP_PARAM_SET_PRIMARY (Xin Long) [1986966]
- sctp: validate chunk size in __rcv_asconf_lookup (Xin Long) [1986966]
- sctp: add size validation when walking chunks (Xin Long) [1986966]
- sctp: validate from_addr_param return (Xin Long) [1986966]
- sctp: fix the proc_handler for sysctl encap_port (Xin Long) [1986966]
- sctp: add the missing setting for asoc encap_port (Xin Long) [1986966]
- sctp: fix a SCTP_MIB_CURRESTAB leak in sctp_sf_do_dupcook_b (Xin Long) [1986966]
- sctp: Fix out-of-bounds warning in sctp_process_asconf_param() (Xin Long) [1986966]
- sctp: Fix COMM_LOST/CANT_STR_ASSOC err reporting on big-endian platforms (Xin Long) [1986966]
- bonding: allow nesting of bonding device (Antoine Tenart) [1989099]
- bonding: avoid adding slave device with IFF_MASTER flag (Antoine Tenart) [1989099]
- ice: don't remove netdev->dev_addr from uc sync list (Ken Cox) [1873969 1961018]
- ice: Stop processing VF messages during teardown (Ken Cox) [1986451]
- ice: Prevent probing virtual functions (Ken Cox) [1952810]
- ima: extend boot_aggregate with kernel measurements (Bruno Meneguele) [1977422]
- selftest/bpf: Verifier tests for var-off access (Jiri Olsa) [1960944] {CVE-2021-29155}
- selftest/bpf: Adjust expected verifier errors (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: No need to simulate speculative domain for immediates (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Fix mask direction swap upon off reg sign change (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Wrap aux data inside bpf_sanitize_info container (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Fix pointer arithmetic mask tightening under state pruning (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Fix leakage of uninitialized bpf stack under speculation (Jiri Olsa) [1960944 1958070] {CVE-2021-29155 CVE-2021-31829}
- bpf: Fix masking negation logic upon negative dst register (Jiri Olsa) [1960944 1958070] {CVE-2021-29155 CVE-2021-31829}
- bpf: verifier: Allocate idmap scratch in verifier env (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Tighten speculative pointer arithmetic mask (Jiri Olsa) [1960944 1972496] {CVE-2021-29155 CVE-2021-33200}
- bpf: Move sanitize_val_alu out of op switch (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Refactor and streamline bounds check into helper (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Allow variable-offset stack access (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Improve verifier error messages for users (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Rework ptr_limit into alu_limit and add common error path (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Ensure off_reg has no mixed signed bounds for all types (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Move off_reg into sanitize_ptr_alu (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Use correct permission flag for mixed signed bounds arithmetic (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Add sanity check for upper ptr_limit (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Simplify alu_limit masking for pointer arithmetic (Jiri Olsa) [1960944] {CVE-2021-29155}
- bpf: Fix off-by-one for area size in creating mask to left (Jiri Olsa) [1960944] {CVE-2021-29155}
- Documentation/x86: Add ratelimit in buslock.rst (Prarit Bhargava) [1948048]
- Documentation/admin-guide: Add bus lock ratelimit (Prarit Bhargava) [1948048]
- x86/bus_lock: Set rate limit for bus lock (Prarit Bhargava) [1948048]
- Documentation/x86: Add buslock.rst (Prarit Bhargava) [1948048]
- [s390] net/smc: Ensure correct state of the socket in send path (Claudio Imbrenda) [1731026]
- [s390] net/smc: Fix ENODATA tests in smc_nl_get_fback_stats() (Claudio Imbrenda) [1731026]
- [s390] net/smc: Make SMC statistics network namespace aware (Claudio Imbrenda) [1731026]
- [s390] net/smc: Add netlink support for SMC fallback statistics (Claudio Imbrenda) [1731026]
- [s390] net/smc: Add netlink support for SMC statistics (Claudio Imbrenda) [1731026]
- [s390] net/smc: Add SMC statistics support (Claudio Imbrenda) [1731026]
- [s390] s390/ap: Fix hanging ioctl caused by wrong msg counter (Claudio Imbrenda) [1984762]
- KVM: PPC: Book3S: Fix H_RTAS rets buffer overflow (Jon Maloy) [1988226] {CVE-2021-37576}
- NFSv4: Fix an Oops in pnfs_mark_request_commit() when doing O_DIRECT (Benjamin Coddington) [1983793]
- selftests/powerpc: EBB selftest for MMCR0 control for PMU SPRs in ISA v3.1 (Desnes A. Nunes do Rosario) [1991753]
- selftests/powerpc: Fix "no_handler" EBB selftest (Desnes A. Nunes do Rosario) [1991753]
- powerpc/perf: Fix crash in perf_instruction_pointer() when ppmu is not set (Desnes A. Nunes do Rosario) [1991753]
- powerpc/pmu: Make the generic compat PMU use the architected events (Desnes A. Nunes do Rosario) [1991753]
- perf script python: Fix buffer size to report iregs in perf script (Desnes A. Nunes do Rosario) [1991753]
- ceph: reduce contention in ceph_check_delayed_caps() (Jeff Layton) [1953430]

* Mon Aug 16 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-332.el8]
- net: mac802154: Fix general protection fault (Chris von Recklinghausen) [1984571] {CVE-2021-3659}
- Bluetooth: fix the erroneous flush_work() order (Chris von Recklinghausen) [1964559] {CVE-2021-3564}
- Bluetooth: use correct lock to prevent UAF of hdev object (Chris von Recklinghausen) [1968214] {CVE-2021-3573}
- usb: ehci: Prevent missed ehci interrupts with edge-triggered MSI (Torez Smith) [1972139]
- usb: ehci: do not initialise static variables (Torez Smith) [1972139]
- usb: host: move EH SINGLE_STEP_SET_FEATURE implementation to core (Torez Smith) [1972139]
- USB: ehci: drop workaround for forced irq threading (Torez Smith) [1972139]
- usb: ehci: add spurious flag to disable overcurrent checking (Torez Smith) [1972139]
- perf test: Handle fd gaps in test__dso_data_reopen (Michael Petlan) [1871785]
- tty: vt: always invoke vc->vc_sw->con_resize callback (Maxim Levitsky) [1957611]
- iavf: fix locking of critical sections (Stefan Assmann) [1975245]
- iavf: do not override the adapter state in the watchdog task (Stefan Assmann) [1975245]
- ixgbe: Fix packet corruption due to missing DMA sync (Ken Cox) [1920269]
- ixgbe: Fix an error handling path in 'ixgbe_probe()' (Ken Cox) [1920269]
- intel: Remove rcu_read_lock() around XDP program invocation (Ken Cox) [1920269]
- ixgbe: add correct exception tracing for XDP (Ken Cox) [1920269]
- ixgbe: Fix out-bounds warning in ixgbe_host_interface_command() (Ken Cox) [1920269]
- ixgbe: reduce checker warnings (Ken Cox) [1920269]
- ixgbe: use checker safe conversions (Ken Cox) [1920269]
- ixgbe: fix large MTU request from VF (Ken Cox) [1920269]
- net: ethernet: intel: Fix a typo in the file ixgbe_dcb_nl.c (Ken Cox) [1920269]
- net: intel: Remove unused function pointer typedef ixgbe_mc_addr_itr (Ken Cox) [1920269]
- ixgbe: Support external GBE SerDes PHY BCM54616s (Ken Cox) [1920269]
- ixgbe: fix unbalanced device enable/disable in suspend/resume (Ken Cox) [1920269]
- ixgbe: Fix NULL pointer dereference in ethtool loopback test (Ken Cox) [1920269]
- ixgbe: Fix fall-through warnings for Clang (Ken Cox) [1920269]
- intel: clean up mismatched header comments (Ken Cox) [1920269]
- bpf, devmap: Move drop error path to devmap for XDP_REDIRECT (Ken Cox) [1920269]
- intel: Update drivers to use ethtool_sprintf (Ken Cox) [1920269]
- ixgbe: optimize for XDP_REDIRECT in xsk path (Ken Cox) [1920269]
- ixgbe: move headroom initialization to ixgbe_configure_rx_ring (Ken Cox) [1920269]
- ixgbe: Fix memleak in ixgbe_configure_clsu32 (Ken Cox) [1920269]
- ixgbe: fail to create xfrm offload of IPsec tunnel mode SA (Ken Cox) [1920269]
- ixgbe: store the result of ixgbe_rx_offset() onto ixgbe_ring (Ken Cox) [1920269]
- net: ixgbe: use skb_csum_is_sctp instead of protocol check (Ken Cox) [1920269]
- ixgbe: avoid premature Rx buffer reuse (Ken Cox) [1920269]
- net: remove unneeded break (Ken Cox) [1920269]
- ixgbe: add pause frame stats (Ken Cox) [1920269]
- iavf: Set RSS LUT and key in reset handle path (Ken Cox) [1910853]
- [netdrv] chelsio: Replace zero-length array with flexible-array member (Raju Rangoju) [1955208]
- cxgb4: avoid collecting SGE_QBASE regs during traffic (Raju Rangoju) [1955208]
- cxgb4: collect serial config version from register (Raju Rangoju) [1955208]
- cxgb4: remove unused vpd_cap_addr (Raju Rangoju) [1955208]
- cxgb4: remove bogus CHELSIO_VPD_UNIQUE_ID constant (Raju Rangoju) [1955208]
- cxgb4: Assign boolean values to a bool variable (Raju Rangoju) [1955208]
- cxgb4: enable interrupt based Tx completions for T5 (Raju Rangoju) [1955208]
- cxgb4: fix the panic caused by non smac rewrite (Raju Rangoju) [1955208]
- cxgb4: Fix the -Wmisleading-indentation warning (Raju Rangoju) [1955208]
- cxgb4: set up filter action after rewrites (Raju Rangoju) [1955208]
- cxgb4: handle 4-tuple PEDIT to NAT mode translation (Raju Rangoju) [1955208]
- [netdrv] net: rename flow_action_hw_stats_types* -> flow_action_hw_stats* (Raju Rangoju) [1955208]
- net: cxbg4: Remove pointless in_interrupt() check (Raju Rangoju) [1955208]
- cxgb4: Avoid log flood (Raju Rangoju) [1955208]
- cxgb4: fix memory leak during module unload (Raju Rangoju) [1955208]
- cxgb4: Fix offset when clearing filter byte counters (Raju Rangoju) [1955208]
- cxgb4: add error handlers to LE intr_handler (Raju Rangoju) [1955208]
- cxgb4: insert IPv6 filter rules in next free region (Raju Rangoju) [1955208]
- cxgb4: Fix race between loopback and normal Tx path (Raju Rangoju) [1955208]
- cxgb4: Fix work request size calculation for loopback test (Raju Rangoju) [1955208]
- cxgb4: add TC-MATCHALL IPv6 support (Raju Rangoju) [1955208]
- cxgb4: fix extracting IP addresses in TC-FLOWER rules (Raju Rangoju) [1955208]
- cxgb4: fix check for running offline ethtool selftest (Raju Rangoju) [1955208]
- cxgb4: add loopback ethtool self-test (Raju Rangoju) [1955208]
- net: cxgb4: reject unsupported coalescing params (Raju Rangoju) [1955208]
- cxgb4: use eth_zero_addr() to clear mac address (Raju Rangoju) [1955208]
- cxgb4: add missing release on skb in uld_send() (Raju Rangoju) [1955208]
- cxgb4: convert to new udp_tunnel_nic infra (Raju Rangoju) [1955208]
- cxgb4: fix all-mask IP address comparison (Raju Rangoju) [1955208]
- cxgb4: add main VI to mirror VI config replication (Raju Rangoju) [1955208]
- cxgb4: add support for mirror Rxqs (Raju Rangoju) [1955208]
- cxgb4: add mirror action to TC-MATCHALL offload (Raju Rangoju) [1955208]
- net: cxgb4: fix return error value in t4_prep_fw (Raju Rangoju) [1955208]
- cxgb4: move device dump arrays in header to C file (Raju Rangoju) [1955208]
- cxgb4: always sync access when flashing PHY firmware (Raju Rangoju) [1955208]
- cxgb4: update kernel-doc line comments (Raju Rangoju) [1955208]
- cxgb4: fix set but unused variable when DCB is disabled (Raju Rangoju) [1955208]
- cxgb4: move DCB version extern to header file (Raju Rangoju) [1955208]
- cxgb4: remove cast when saving IPv4 partial checksum (Raju Rangoju) [1955208]
- cxgb4: fix SGE queue dump destination buffer context (Raju Rangoju) [1955208]
- cxgb4: use correct type for all-mask IP address comparison (Raju Rangoju) [1955208]
- cxgb4: fix endian conversions for L4 ports in filters (Raju Rangoju) [1955208]
- cxgb4: parse TC-U32 key values and masks natively (Raju Rangoju) [1955208]
- cxgb4: use unaligned conversion for fetching timestamp (Raju Rangoju) [1955208]
- cxgb4: move PTP lock and unlock to caller in Tx path (Raju Rangoju) [1955208]
- cxgb4: move handling L2T ARP failures to caller (Raju Rangoju) [1955208]
- cxgb4: Use struct_size() helper (Raju Rangoju) [1955208]
- cxgb4: add action to steer flows to specific Rxq (Raju Rangoju) [1955208]
- cxgb4: add support to fetch ethtool n-tuple filters (Raju Rangoju) [1955208]
- cxgb4: add ethtool n-tuple filter deletion (Raju Rangoju) [1955208]
- cxgb4: add ethtool n-tuple filter insertion (Raju Rangoju) [1955208]
- [netdrv] flow_offload: check for basic action hw stats type (Raju Rangoju) [1955208]
- cxgb4: add skeleton for ethtool n-tuple filters (Raju Rangoju) [1955208]
- cxgb4: add support to read serial flash (Raju Rangoju) [1955208]
- cxgb4: add support to flash boot cfg image (Raju Rangoju) [1955208]
- cxgb4: add support to flash boot image (Raju Rangoju) [1955208]
- cxgb4: add support to flash PHY image (Raju Rangoju) [1955208]
- cxgb4: update set_flash to flash different images (Raju Rangoju) [1955208]
- cxgb4: Use kfree() instead kvfree() where appropriate (Raju Rangoju) [1955208]
- cxgb4: Use pM format specifier for MAC addresses (Raju Rangoju) [1955208]
- net: sock: fix in-kernel mark setting (Alexander Aring) [1509204]
- sock: Reset dst when changing sk_mark via setsockopt (Alexander Aring) [1509204]
- fs: dlm: fix mark setting deadlock (Alexander Aring) [1509204]
- fs: dlm: fix mark per nodeid setting (Alexander Aring) [1509204]
- fs: dlm: remove lock dependency warning (Alexander Aring) [1509204]
- fs: dlm: set skb mark per peer socket (Alexander Aring) [1509204]
- fs: dlm: set skb mark for listen socket (Alexander Aring) [1509204]
- net: sock: add sock_set_mark (Alexander Aring) [1509204]
- Bluetooth: btusb: Add support for GarfieldPeak controller (Gopal Tiwari) [1959110]
- can: gw: synchronize rcu operations before removing gw job entry (Balazs Nemeth) [1986334]
- can: bcm: fix infoleak in struct bcm_msg_head (Balazs Nemeth) [1986334]
- can: af_can: prevent potential access of uninitialized member in canfd_rcv() (Balazs Nemeth) [1956730]
- can: af_can: prevent potential access of uninitialized member in can_rcv() (Balazs Nemeth) [1956730]
- can: proc: can_remove_proc(): silence remove_proc_entry warning (Balazs Nemeth) [1956730]
- cifs: add missing parsing of backupuid (Ronnie Sahlberg) [1987126]
- cifs: use helpers when parsing uid/gid mount options and validate them (Ronnie Sahlberg) [1987126]
- kernfs: dont call d_splice_alias() under kernfs node lock (Ian Kent) [1939133]
- kernfs: use i_lock to protect concurrent inode updates (Ian Kent) [1939133]
- kernfs: switch kernfs to use an rwsem (Ian Kent) [1939133]
- kernfs: use VFS negative dentry caching (Ian Kent) [1939133]
- kernfs: add a revision to identify directory node changes (Ian Kent) [1939133]
- kernfs: move revalidate to be near lookup (Ian Kent) [1939133]

* Thu Aug 12 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-331.el8]
- mlxsw: spectrum_mr: Update egress RIF list before route's action (Ivan Vecera) [1941938]
- selftests: mlxsw: Fix mausezahn invocation in ERSPAN scale test (Ivan Vecera) [1941938]
- selftests: mlxsw: Increase the tolerance of backlog buildup (Ivan Vecera) [1941938]
- selftests: mlxsw: Return correct error code in resource scale tests (Ivan Vecera) [1941938]
- selftests: mlxsw: Remove a redundant if statement in tc_flower_scale test (Ivan Vecera) [1941938]
- selftests: mlxsw: Remove a redundant if statement in port_scale test (Ivan Vecera) [1941938]
- selftests: mlxsw: sch_red_ets: Test proper counter cleaning in ETS (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Index future FIFOs by band number (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Allocate child qdiscs dynamically (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Guard all qdisc accesses with a lock (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Track children per qdisc (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Promote backlog reduction to mlxsw_sp_qdisc_destroy() (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Track tclass_num as int, not u8 (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Drop an always-true condition (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Simplify mlxsw_sp_qdisc_compare() (Ivan Vecera) [1941938]
- mlxsw: spectrum_qdisc: Drop one argument from check_params callback (Ivan Vecera) [1941938]
- mlxsw: core: Remove critical trip points from thermal zones (Ivan Vecera) [1941938]
- selftests: mlxsw: Test vetoing of double sampling (Ivan Vecera) [1941938]
- mlxsw: spectrum: Veto sampling if already enabled on port (Ivan Vecera) [1941938]
- selftests: mlxsw: Test matchall failure with protocol match (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Perform priority checks earlier (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Convert if statements to a switch statement (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Perform protocol check earlier (Ivan Vecera) [1941938]
- selftests: forwarding: vxlan_bridge_1d: Add more ECN decap test cases (Ivan Vecera) [1941938]
- mlxsw: spectrum: Fix ECN marking in tunnel decapsulation (Ivan Vecera) [1941938]
- selftests: net: forwarding: Fix a typo (Ivan Vecera) [1941938]
- selftests: mlxsw: spectrum-2: Remove q_in_vni_veto test (Ivan Vecera) [1941938]
- selftests: forwarding: Add test for dual VxLAN bridge (Ivan Vecera) [1941938]
- mlxsw: Allow 802.1d and .1ad VxLAN bridges to coexist on Spectrum>=2 (Ivan Vecera) [1941938]
- mlxsw: Add struct mlxsw_sp_switchdev_ops per ASIC (Ivan Vecera) [1941938]
- mlxsw: spectrum: Add mlxsw_sp_port_egress_ethtype_set() (Ivan Vecera) [1941938]
- mlxsw: reg: Add Switch Port Egress VLAN EtherType Register (Ivan Vecera) [1941938]
- mlxsw: reg: Add egr_et_set field to SPVID (Ivan Vecera) [1941938]
- selftests: mlxsw: Test egress sampling limitation on Spectrum-1 only (Ivan Vecera) [1941938]
- selftests: mlxsw: Add tc sample tests for new triggers (Ivan Vecera) [1941938]
- mlxsw: spectrum_acl: Offload FLOW_ACTION_SAMPLE (Ivan Vecera) [1941938]
- mlxsw: core_acl_flex_actions: Add mirror sampler action (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Add support for egress sampling (Ivan Vecera) [1941938]
- mlxsw: spectrum: Start using sampling triggers hash table (Ivan Vecera) [1941938]
- mlxsw: spectrum: Track sampling triggers in a hash table (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Pass matchall entry to sampling operations (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Push sampling checks to per-ASIC operations (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Propagate extack further (Ivan Vecera) [1941938]
- selftests: mlxsw: Add tc sample tests (Ivan Vecera) [1941938]
- mlxsw: spectrum: Report extra metadata to psample module (Ivan Vecera) [1941938]
- mlxsw: spectrum: Remove mlxsw_sp_sample_receive() (Ivan Vecera) [1941938]
- mlxsw: spectrum: Remove unnecessary RCU read-side critical section (Ivan Vecera) [1941938]
- mlxsw: pci: Set extra metadata in skb control block (Ivan Vecera) [1941938]
- mlxsw: Create dedicated field for Rx metadata in skb control block (Ivan Vecera) [1941938]
- mlxsw: pci: Add more metadata fields to CQEv2 (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Implement sampling using mirroring (Ivan Vecera) [1941938]
- mlxsw: spectrum_trap: Split sampling traps between ASICs (Ivan Vecera) [1941938]
- mlxsw: spectrum_matchall: Split sampling support between ASICs (Ivan Vecera) [1941938]
- mlxsw: spectrum_span: Add SPAN probability rate support (Ivan Vecera) [1941938]
- mlxsw: reg: Extend mirroring registers with probability rate field (Ivan Vecera) [1941938]
- mlxsw: spectrum_span: Add SPAN session identifier support (Ivan Vecera) [1941938]
- mlxsw: Adjust some MFDE fields shift and size to fw implementation (Ivan Vecera) [1941938]
- mlxsw: core: Expose MFDE.log_ip to devlink health (Ivan Vecera) [1941938]
- mlxsw: reg: Extend MFDE register with new log_ip field (Ivan Vecera) [1941938]
- mlxsw: spectrum: Bump minimum FW version to xx.2008.2406 (Ivan Vecera) [1941938]
- mlxsw: reg: Fix comment about slot_index field in PMAOS register (Ivan Vecera) [1941938]
- mlxsw: spectrum: Reword an error message for Q-in-Q veto (Ivan Vecera) [1941938]
- mlxsw: spectrum_ethtool: Add an external speed to PTYS register (Ivan Vecera) [1941938]
- mlxsw: spectrum_span: Do not overwrite policer configuration (Ivan Vecera) [1941938]
- selftests: forwarding: Specify interface when invoking mausezahn (Ivan Vecera) [1941938]
- selftests: mlxsw: Add a scale test for physical ports (Ivan Vecera) [1941938]
- mlxsw: Register physical ports as a devlink resource (Ivan Vecera) [1941938]
- mlxsw: pci: switch from 'pci_' to 'dma_' API (Ivan Vecera) [1941938]
- mlxsw: core: Increase critical threshold for ASIC thermal zone (Ivan Vecera) [1941938]
- mlxsw: core: Add validation of transceiver temperature thresholds (Ivan Vecera) [1941938]
- selftests: mlxsw: Set headroom size of correct port (Ivan Vecera) [1941938]
- net/mlxfw: Use kzalloc for allocating only one thing (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Use eXtended mezzanine to offload IPv4 router (Ivan Vecera) [1941938]
- mlxsw: spectrum: Set KVH XLT cache mode for Spectrum2/3 (Ivan Vecera) [1941938]
- mlxsw: spectrum_router_xm: Introduce basic XM cache flushing (Ivan Vecera) [1941938]
- mlxsw: reg: Add Router LPM Cache Enable Register (Ivan Vecera) [1941938]
- mlxsw: reg: Add Router LPM Cache ML Delete Register (Ivan Vecera) [1941938]
- mlxsw: spectrum_router_xm: Implement L-value tracking for M-index (Ivan Vecera) [1941938]
- mlxsw: reg: Add XM Router M Table Register (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Introduce per-ASIC XM initialization (Ivan Vecera) [1941938]
- mlxsw: reg: Add XM Lookup Table Query Register (Ivan Vecera) [1941938]
- mlxsw: reg: Add Router XLT M select Register (Ivan Vecera) [1941938]
- mlxsw: Ignore ports that are connected to eXtended mezanine (Ivan Vecera) [1941938]
- mlxsw: pci: Obtain info about ports used by eXtended mezanine (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Introduce XM implementation of router low-level ops (Ivan Vecera) [1941938]
- mlxsw: reg: Add Router XLT Enable Register (Ivan Vecera) [1941938]
- mlxsw: reg: Add XM Direct Register (Ivan Vecera) [1941938]
- selftests: mlxsw: Add Q-in-VNI veto tests (Ivan Vecera) [1941938]
- selftests: forwarding: Add Q-in-VNI test (Ivan Vecera) [1941938]
- mlxsw: spectrum_switchdev: Allow joining VxLAN to 802.1ad bridge (Ivan Vecera) [1941938]
- mlxsw: Veto Q-in-VNI for Spectrum-1 ASIC (Ivan Vecera) [1941938]
- mlxsw: spectrum_switchdev: Use ops->vxlan_join() when adding VLAN to VxLAN device (Ivan Vecera) [1941938]
- mlxsw: spectrum_nve_vxlan: Add support for Q-in-VNI for Spectrum-2 ASIC (Ivan Vecera) [1941938]
- mlxsw: spectrum: Publish mlxsw_sp_ethtype_to_sver_type() (Ivan Vecera) [1941938]
- mlxsw: Save EtherType as part of mlxsw_sp_nve_config (Ivan Vecera) [1941938]
- mlxsw: Save EtherType as part of mlxsw_sp_nve_params (Ivan Vecera) [1941938]
- mlxsw: spectrum_switchdev: Create common function for joining VxLAN to VLAN-aware bridge (Ivan Vecera) [1941938]
- mlxsw: reg: Add support for tunnel port in SPVID register (Ivan Vecera) [1941938]
- mlxsw: reg: Add Switch Port VLAN Stacking Register (Ivan Vecera) [1941938]
- mlxsw: Use one enum for all registers that contain tunnel_port field (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Reduce mlxsw_sp_ipip_fib_entry_op_gre4() (Ivan Vecera) [1941938]
- mlxsw: spectrum: Bump minimum FW version to xx.2008.2018 (Ivan Vecera) [1941938]
- mlxsw: core_acl: Use an array instead of a struct with a zero-length array (Ivan Vecera) [1941938]
- mlxsw: spectrum_mr: Use flexible-array member instead of zero-length array (Ivan Vecera) [1941938]
- mlxsw: core: Trace EMAD events (Ivan Vecera) [1941938]
- selftests: mlxsw: Test RIF's reference count when joining a LAG (Ivan Vecera) [1941938]
- mlxsw: spectrum: Apply RIF configuration when joining a LAG (Ivan Vecera) [1941938]
- selftests: forwarding: Add QinQ veto testing (Ivan Vecera) [1941938]
- mlxsw: Add QinQ configuration vetoes (Ivan Vecera) [1941938]
- mlxsw: spectrum_switchdev: Add support of QinQ traffic (Ivan Vecera) [1941938]
- mlxsw: spectrum_switchdev: Create common functions for VLAN-aware bridge (Ivan Vecera) [1941938]
- mlxsw: Make EtherType configurable when pushing VLAN at ingress (Ivan Vecera) [1941938]
- mlxsw: spectrum: Only treat 802.1q packets as tagged packets (Ivan Vecera) [1941938]
- mlxsw: reg: Add et_vlan field to SPVID register (Ivan Vecera) [1941938]
- mlxsw: reg: Add Switch Port VLAN Classification Register (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Update adjacency index more efficiently (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Track nexthop group virtual router membership (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Rollback virtual router adjacency pointer update (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Pass virtual router parameters directly instead of pointer (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Fix error handling issue (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Resolve RIF from nexthop struct instead of neighbour (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Use loopback RIF for unresolved nexthops (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Use different trap identifier for unresolved nexthops (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Create loopback RIF during initialization (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Allow returning errors from mlxsw_sp_nexthop_group_refresh() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Add an indication if a nexthop group can be destroyed (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Re-order mlxsw_sp_nexthop6_group_get() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Set FIB entry's type based on nexthop group (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Set FIB entry's type after creating nexthop group (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Pass ifindex to mlxsw_sp_ipip_entry_find_by_decap() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Set ifindex for IPv4 nexthops (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Fix wrong kfree() in error path (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Remove outdated comment (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Consolidate mlxsw_sp_nexthop{4, 6}_type_fini() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Consolidate mlxsw_sp_nexthop{4, 6}_type_init() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Remove unused argument from mlxsw_sp_nexthop6_type_init() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Pass nexthop netdev to mlxsw_sp_nexthop4_type_init() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Pass nexthop netdev to mlxsw_sp_nexthop6_type_init() (Ivan Vecera) [1941938]
- mlxsw: spectrum_ipip: Remove overlay protocol from can_offload() callback (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Split nexthop group configuration to a different struct (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Move IPv4 FIB info into a union in nexthop group struct (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Remove unused field 'prio' from IPv4 FIB entry struct (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Store FIB info in route (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Associate neighbour table with nexthop instead of group (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Use nexthop group type in hash table key (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Add nexthop group type field (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Compare key with correct object type (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Introduce FIB entry update op (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Track FIB entry committed state and skip uncommitted on delete (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Introduce fib_entry priv for low-level ops (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Have FIB entry op context allocated for the instance (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Prepare work context for possible bulking (Ivan Vecera) [1941938]
- mlxsw: spectrum: Push RALUE packing and writing into low-level router ops (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Use RALUE pack helper from abort function (Ivan Vecera) [1941938]
- mlxsw: reg: Allow to pass NULL pointer to mlxsw_reg_ralue_pack4/6() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Pass destination IP as a pointer to mlxsw_reg_ralue_pack4() (Ivan Vecera) [1941938]
- mlxsw: spectrum: Export RALUE pack helper and use it from IPIP (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Push out RALUE pack into separate helper (Ivan Vecera) [1941938]
- mlxsw: spectrum: Propagate context from work handler containing RALUE payload (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Introduce FIB event queue instead of separate works (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Use RALUE-independent op arg (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Pass non-register proto enum to __mlxsw_sp_router_set_abort_trap() (Ivan Vecera) [1941938]
- mlxsw: spectrum_router: Introduce low-level ops and implement them for RALXX regs (Ivan Vecera) [1941938]
- mlxsw: reg: Add XRALXX Registers (Ivan Vecera) [1941938]
- hwmon: (mlxreg-fan) Add support for fan drawers capability and present registers (Ivan Vecera) [1941938]
- i2c: mux: mlxcpld: Add callback to notify mux creation completion (Ivan Vecera) [1941938]
- i2c: mux: mlxcpld: Extend supported mux number (Ivan Vecera) [1941938]
- i2c: mux: mlxcpld: Extend driver to support word address space devices (Ivan Vecera) [1941938]
- i2c: mux: mlxcpld: Get rid of adapter numbers enforcement (Ivan Vecera) [1941938]
- i2c: mux: mlxcpld: Prepare mux selection infrastructure for two-byte support (Ivan Vecera) [1941938]
- i2c: mux: mlxcpld: Convert driver to platform driver (Ivan Vecera) [1941938]
- i2c: mux: mlxcpld: Move header file out of x86 realm (Ivan Vecera) [1941938]
- platform/x86: mlxcpld: Update module license (Ivan Vecera) [1941938]
- i2c: mux: mlxcpld: Update module license (Ivan Vecera) [1941938]
- i2c: mlxcpld: Add support for I2C bus frequency setting (Ivan Vecera) [1941938]
- i2c: mlxcpld: Decrease polling time for performance improvement (Ivan Vecera) [1941938]
- i2c: mlxcpld: Update module license (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Fix item counter assignment for MSN2700/ComEx system (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Fix item counter assignment for MSN2700, MSN24xx systems (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: remove an unused variable (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Remove PSU EEPROM from MSN274x platform configuration (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Remove PSU EEPROM from default platform configuration (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Add capability field to platform FAN description (Ivan Vecera) [1941938]
- platform_data/mlxreg: Extend core platform structure (Ivan Vecera) [1941938]
- platform_data/mlxreg: Update module license (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Fix extended topology configuration for power supply units (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Remove PSU EEPROM configuration (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Extend FAN platform data description (Ivan Vecera) [1941938]
- platform_data/mlxreg: Add presence register field for FAN devices (Ivan Vecera) [1941938]
- Documentation/ABI: Add new attribute for mlxreg-io sysfs interfaces (Ivan Vecera) [1941938]
- platform/mellanox: mlxreg-io: Add support for complex attributes (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: Add more definitions for system attributes (Ivan Vecera) [1941938]
- platform_data/mlxreg: Add support for complex attributes (Ivan Vecera) [1941938]
- platform/mellanox: mlxreg-hotplug: Add environmental data to uevent (Ivan Vecera) [1941938]
- platform/mellanox: mlxreg-hotplug: Use capability register for attribute creation (Ivan Vecera) [1941938]
- platform/mellanox: mlxreg-hotplug: Modify module license (Ivan Vecera) [1941938]
- lib/string_helpers: Introduce string_upper() and string_lower() helpers (Ivan Vecera) [1941938]
- i2c: mlxcpld: check correct size of maximum RECV_LEN packet (Ivan Vecera) [1941938]
- docs: watchdog: mlx-wdt: Add description of new watchdog type 3 (Ivan Vecera) [1941938]
- watchdog: mlx-wdt: support new watchdog type with longer timeout period (Ivan Vecera) [1941938]
- platform/x86: mlx-platform: support new watchdog type with longer timeout (Ivan Vecera) [1941938]
- platform_data/mlxreg: support new watchdog type with longer timeout period (Ivan Vecera) [1941938]
- platform/mellanox: mlxreg-hotplug: convert to use i2c_new_client_device() (Ivan Vecera) [1941938]
- selftests: icmp_redirect: IPv6 PMTU info should be cleared after redirect (Hangbin Liu) [1986227]
- selftests: icmp_redirect: remove from checking for IPv6 route get (Hangbin Liu) [1986227]
- selftests: icmp_redirect: support expected failures (Hangbin Liu) [1986227]
- selftests: disable rp_filter for icmp_redirect.sh (Hangbin Liu) [1986227]
- selftests: icmp_redirect: Add support for routing via nexthop objects (Hangbin Liu) [1986227]
- selftest: Fixes for icmp_redirect test (Hangbin Liu) [1986227]
- selftests: Add redirect tests (Hangbin Liu) [1986227]
- redhat: enable vdpa_sim_net vdpa_sim_block vp_vdpa (Cindy Lu) [1975173]
- virtio/vdpa: clear the virtqueue state during probe (Cindy Lu) [1975173]
- vp_vdpa: allow set vq state to initial state after reset (Cindy Lu) [1975173]
- virtio-pci library: introduce vp_modern_get_driver_features() (Cindy Lu) [1975173]
- vdpa: support packed virtqueue for set/get_vq_state() (Cindy Lu) [1975173]
- vdpa/mlx5: Clear vq ready indication upon device reset (Cindy Lu) [1975173]
- vdpa/mlx5: Add support for doorbell bypassing (Cindy Lu) [1975173]
- vdpa/mlx5: Add support for running with virtio_vdpa (Cindy Lu) [1975173]
- vdp/mlx5: Fix setting the correct dma_device (Cindy Lu) [1975173]
- vdpa/mlx5: Support creating resources with uid == 0 (Cindy Lu) [1975173]
- vdpa/mlx5: Fix possible failure in umem size calculation (Cindy Lu) [1975173]
- vdpa/mlx5: Fix umem sizes assignments on VQ create (Cindy Lu) [1975173]
- vp_vdpa: correct the return value when fail to map notification (Cindy Lu) [1975173]
- virtio_net: Fix error handling in virtnet_restore() (Cindy Lu) [1975173]
- virtio-blk: Fix memory leak among suspend/resume procedure (Cindy Lu) [1975173]
- vDPA/ifcvf: reuse pre-defined macros for device ids and vendor ids (Cindy Lu) [1975173]
- virtio: update virtio id table, add transitional ids (Cindy Lu) [1975173]
- vDPA/ifcvf: implement doorbell mapping for ifcvf (Cindy Lu) [1975173]
- vDPA/ifcvf: record virtio notify base (Cindy Lu) [1975173]
- vdpa_sim_blk: remove duplicate include of linux/blkdev.h (Cindy Lu) [1975173]
- {net,vdpa}/mlx5: Configure interface MAC into mpfs L2 table (Cindy Lu) [1975173]
- virtio_pci_modern: correct sparse tags for notify (Cindy Lu) [1975173]
- virtio_pci_modern: __force cast the notify mapping (Cindy Lu) [1975173]
- vDPA/ifcvf: get_config_size should return dev specific config size (Cindy Lu) [1975173]
- vDPA/ifcvf: enable Intel C5000X-PL virtio-block for vDPA (Cindy Lu) [1975173]
- vDPA/ifcvf: deduce VIRTIO device ID when probe (Cindy Lu) [1975173]
- vdpa_sim_blk: add support for vdpa management tool (Cindy Lu) [1975173]
- vdpa_sim_blk: handle VIRTIO_BLK_T_GET_ID (Cindy Lu) [1975173]
- vdpa_sim_blk: implement ramdisk behaviour (Cindy Lu) [1975173]
- vdpa: add vdpa simulator for block device (Cindy Lu) [1975173]
- vhost/vdpa: Remove the restriction that only supports virtio-net devices (Cindy Lu) [1975173]
- vhost/vdpa: use get_config_size callback in vhost_vdpa_config_validate() (Cindy Lu) [1975173]
- vdpa: add get_config_size callback in vdpa_config_ops (Cindy Lu) [1975173]
- vdpa_sim: cleanup kiovs in vdpasim_free() (Cindy Lu) [1975173]
- vringh: add vringh_kiov_length() helper (Cindy Lu) [1975173]
- vringh: implement vringh_kiov_advance() (Cindy Lu) [1975173]
- vringh: explain more about cleaning riov and wiov (Cindy Lu) [1975173]
- vringh: reset kiov 'consumed' field in __vringh_iov() (Cindy Lu) [1975173]
- vringh: add 'iotlb_lock' to synchronize iotlb accesses (Cindy Lu) [1975173]
- vdpa_sim: use iova module to allocate IOVA addresses (Cindy Lu) [1975173]
- vDPA/ifcvf: deduce VIRTIO device ID from pdev ids (Cindy Lu) [1975173]
- vDPA/ifcvf: verify mandatory feature bits for vDPA (Cindy Lu) [1975173]
- vDPA/ifcvf: fetch device feature bits when probe (Cindy Lu) [1975173]
- vDPA/ifcvf: remove the version number string (Cindy Lu) [1975173]
- vDPA/ifcvf: rename original IFCVF dev ids to N3000 ids (Cindy Lu) [1975173]
- vDPA/ifcvf: enable Intel C5000X-PL virtio-net for vDPA (Cindy Lu) [1975173]
- vDPA/ifcvf: get_vendor_id returns a device specific vendor id (Cindy Lu) [1975173]
- vhost-vdpa: fix vm_flags for virtqueue doorbell mapping (Cindy Lu) [1975173]
- vp_vdpa: report doorbell address (Cindy Lu) [1975173]
- virtio-pci library: report resource address (Cindy Lu) [1975173]
- virito_pci libray: hide vp_modern_map_capability() (Cindy Lu) [1975173]
- virtio_pci_modern: hide vp_modern_get_queue_notify_off() (Cindy Lu) [1975173]
- vp_vdpa: switch to use vp_modern_map_vq_notify() (Cindy Lu) [1975173]
- virtio-pci library: switch to use vp_modern_map_vq_notify() (Cindy Lu) [1975173]
- virtio_pci_modern: introduce helper to map vq notify area (Cindy Lu) [1975173]
- virtio-net: don't allocate control_buf if not supported (Cindy Lu) [1975173]
- vdpa: Follow kdoc comment style (Cindy Lu) [1975173]
- vdpa: Follow kdoc comment style (Cindy Lu) [1975173]
- vdpa/mlx5: Enable user to add/delete vdpa device (Cindy Lu) [1975173]
- vdpa: introduce virtio pci driver (Cindy Lu) [1975173]
- vdpa/mlx5: Set err = -ENOMEM in case dma_map_sg_attrs fails (Cindy Lu) [1975173]
- vhost-vdpa: protect concurrent access to vhost device iotlb (Cindy Lu) [1975173]
- vdpa/mlx5: Fix suspend/resume index restoration (Cindy Lu) [1975173]
- vdpa/mlx5: Fix wrong use of bit numbers (Cindy Lu) [1975173]
- vdpa/mlx5: should exclude header length and fcs from mtu (Cindy Lu) [1975173]
- vhost-vdpa: set v->config_ctx to NULL if eventfd_ctx_fdget() fails (Cindy Lu) [1975173]
- vhost-vdpa: fix use-after-free of v->config_ctx (Cindy Lu) [1975173]
- vhost_vdpa: fix the missing irq_bypass_unregister_producer() invocation (Cindy Lu) [1975173]
- vdpa_sim: Skip typecasting from void* (Cindy Lu) [1975173]
- vdpa: set the virtqueue num during register (Cindy Lu) [1975173]
- vdpa/mlx5: fix param validation in mlx5_vdpa_get_config() (Cindy Lu) [1975173]
- virtio_vdpa: don't warn when fail to disable vq (Cindy Lu) [1975173]
- virtio-pci: introduce modern device module (Cindy Lu) [1975173]
- virito-pci-modern: rename map_capability() to vp_modern_map_capability() (Cindy Lu) [1975173]
- virtio-pci-modern: introduce helper to get notification offset (Cindy Lu) [1975173]
- virtio-pci-modern: introduce helper for getting queue nums (Cindy Lu) [1975173]
- virtio-pci-modern: introduce helper for setting/geting queue size (Cindy Lu) [1975173]
- virtio-pci-modern: introduce helper to set/get queue_enable (Cindy Lu) [1975173]
- virtio-pci-modern: introduce vp_modern_queue_address() (Cindy Lu) [1975173]
- virtio-pci-modern: introduce vp_modern_set_queue_vector() (Cindy Lu) [1975173]
- virtio-pci-modern: introduce vp_modern_generation() (Cindy Lu) [1975173]
- virtio-pci-modern: introduce helpers for setting and getting features (Cindy Lu) [1975173]
- virtio-pci-modern: introduce helpers for setting and getting status (Cindy Lu) [1975173]
- virtio-pci-modern: introduce helper to set config vector (Cindy Lu) [1975173]
- virtio-pci-modern: introduce vp_modern_remove() (Cindy Lu) [1975173]
- virtio-pci-modern: factor out modern device initialization logic (Cindy Lu) [1975173]
- virtio-pci: split out modern device (Cindy Lu) [1975173]
- virtio-pci: do not access iomem via struct virtio_pci_device directly (Cindy Lu) [1975173]
- vdpa_sim_net: Add support for user supported devices (Cindy Lu) [1975173]
- vdpa: Enable user to query vdpa device info (Cindy Lu) [1975173]
- vdpa: Enable a user to add and delete a vdpa device (Cindy Lu) [1975173]
- vdpa: Define vdpa mgmt device, ops and a netlink interface (Cindy Lu) [1975173]
- vdpa: Extend routine to accept vdpa device name (Cindy Lu) [1975173]
- vdpa_sim_net: Make mac address array static (Cindy Lu) [1975173]
- vdpa/mlx5: Restore the hardware used index after change map (Cindy Lu) [1975173]
- vdpa/mlx5: Fix memory key MTT population (Cindy Lu) [1975173]
- vdpa: Use simpler version of ida allocation (Cindy Lu) [1975173]
- vdpa: Add missing comment for virtqueue count (Cindy Lu) [1975173]
- vdpa: split vdpasim to core and net modules (Cindy Lu) [1975173]
- vdpa_sim: split vdpasim_virtqueue's iov field in out_iov and in_iov (Cindy Lu) [1975173]
- vdpa_sim: make vdpasim->buffer size configurable (Cindy Lu) [1975173]
- vdpa_sim: use kvmalloc to allocate vdpasim->buffer (Cindy Lu) [1975173]
- vdpa_sim: set vringh notify callback (Cindy Lu) [1975173]
- vdpa_sim: add set_config callback in vdpasim_dev_attr (Cindy Lu) [1975173]
- vdpa_sim: add get_config callback in vdpasim_dev_attr (Cindy Lu) [1975173]
- vdpa_sim: make 'config' generic and usable for any device type (Cindy Lu) [1975173]
- vdpa_sim: store parsed MAC address in a buffer (Cindy Lu) [1975173]
- vdpa_sim: add work_fn in vdpasim_dev_attr (Cindy Lu) [1975173]
- vdpa_sim: add supported_features field in vdpasim_dev_attr (Cindy Lu) [1975173]
- vdpa_sim: add device id field in vdpasim_dev_attr (Cindy Lu) [1975173]
- vdpa_sim: add struct vdpasim_dev_attr for device attributes (Cindy Lu) [1975173]
- vdpa_sim: rename vdpasim_config_ops variables (Cindy Lu) [1975173]
- vdpa_sim: make IOTLB entries limit configurable (Cindy Lu) [1975173]
- vdpa_sim: remove hard-coded virtq count (Cindy Lu) [1975173]
- vdpa_sim: remove unnecessary headers inclusion (Cindy Lu) [1975173]
- vdpa: remove unnecessary 'default n' in Kconfig entries (Cindy Lu) [1975173]
- vdpa: ifcvf: Use dma_set_mask_and_coherent to simplify code (Cindy Lu) [1975173]
- vhost_vdpa: switch to vmemdup_user() (Cindy Lu) [1975173]
- vhost: vringh: use krealloc_array() (Cindy Lu) [1975173]
- vringh: fix vringh_iov_push_*() documentation (Cindy Lu) [1975173]
- vringh: fix __vringh_iov() when riov and wiov are different (Cindy Lu) [1975173]
- net: sched: fix memory leak in tcindex_partial_destroy_work (Davide Caratti) [1987042]
- net/sched: act_ct: remove and free nf_table callbacks (Davide Caratti) [1987042]
- net: sched: fix warning in tcindex_alloc_perfect_hash (Davide Caratti) [1987042]
- pkt_sched: sch_qfq: fix qfq_change_class() error path (Davide Caratti) [1987042]
- sch_htb: fix refcount leak in htb_parent_to_leaf_offload (Davide Caratti) [1987042]
- sch_dsmark: fix a NULL deref in qdisc_reset() (Davide Caratti) [1987042]
- sch_red: fix off-by-one checks in red_check_params() (Davide Caratti) [1987042]
- ipv6: fix 'disable_policy' for fwd packets (Hangbin Liu) [1985837]
- net: ipv6: fix return value of ip6_skb_dst_mtu (Hangbin Liu) [1985837]
- ipv6: fix out-of-bound access in ip6_parse_tlv() (Hangbin Liu) [1985837]
- ipv6: exthdrs: do not blindly use init_net (Hangbin Liu) [1985837]
- ipv6: record frag_max_size in atomic fragments in input path (Hangbin Liu) [1985837]
- selftests: net: devlink_port_split.py: skip the test if no devlink device (Hangbin Liu) [1985836]
- selftests/net: bump timeout to 5 minutes (Hangbin Liu) [1985836]
- selftests: net: forwarding: Fix a typo (Hangbin Liu) [1985836]
- selftests/net: so_txtime: remove unneeded semicolon (Hangbin Liu) [1985836]
- tools: selftests: add test for changing routes with PTMU exceptions (Hangbin Liu) [1985836]
- selftests: fix the return value for UDP GRO test (Hangbin Liu) [1985836]
- selftests: pmtu.sh: improve the test result processing (Hangbin Liu) [1985836]
- selftests: pmtu.sh: use $ksft_skip for skipped return code (Hangbin Liu) [1985836]
- selftests/net: timestamping: add ptp v2 support (Hangbin Liu) [1985836]
- ixgbevf: add correct exception tracing for XDP (Ken Cox) [1920268]
- intel: call csum functions with well formatted arguments (Ken Cox) [1920268]
- net: intel: Remove unused function pointer typedef ixgbe_mc_addr_itr (Ken Cox) [1920268]
- ixgbevf: Fix fall-through warnings for Clang (Ken Cox) [1920268]
- intel: clean up mismatched header comments (Ken Cox) [1920268]
- ixgbe: fail to create xfrm offload of IPsec tunnel mode SA (Ken Cox) [1920268]
- net: ixgbevf: use skb_csum_is_sctp instead of protocol check (Ken Cox) [1920268]
- treewide: Use sizeof_field() macro (Ken Cox) [1920268]
- selftests: net: add ESP-in-UDP PMTU test (Xin Long) [1986599]
- udp: check encap socket in __udp_lib_err (Xin Long) [1986599]
- udp: fix possible user after free in error handler (Xin Long) [1986599]
- udpv6: fix possible user after free in error handler (Xin Long) [1986599]
- udp: Support for error handlers of tunnels with arbitrary destination port (Xin Long) [1986599]
- bpf, sockmap, udp: sk_prot needs inuse_idx set for proc stats (Xin Long) [1986599]
- udp: properly flush normal packet at GRO time (Xin Long) [1986599]
- udp: annotate data races around unix_sk(sk)->gso_size (Xin Long) [1986599]
- udp: fix race between close() and udp_abort() (Xin Long) [1986599]
- netfilter: conntrack: remove offload_pickup sysctl again (Florian Westphal) [1987101]
- bpf, samples: Add -fno-asynchronous-unwind-tables to BPF Clang invocation (Yauheni Kaliuta) [1977162]
- net/af_unix: fix a data-race in unix_dgram_sendmsg / unix_release_sock (Balazs Nemeth) [1986845]
- netfilter: nf_tables: fix flowtable list del corruption (Jay Shin) [1977137]
- net: qrtr: fix memory leaks (Íñigo Huguet) [1981236]
- net: qrtr: fix OOB Read in qrtr_endpoint_post (Íñigo Huguet) [1981236]
- net: qrtr: ns: Fix error return code in qrtr_ns_init() (Íñigo Huguet) [1981236]
- net: qrtr: Avoid potential use after free in MHI send (Íñigo Huguet) [1981236]
- qrtr: Convert qrtr_ports from IDR to XArray (Íñigo Huguet) [1981236]
- net: qrtr: Fix memory leak on qrtr_tx_wait failure (Íñigo Huguet) [1981236]
- net: qrtr: fix a kernel-infoleak in qrtr_recvmsg() (Íñigo Huguet) [1981236]
- net: qrtr: fix error return code of qrtr_sendmsg() (Íñigo Huguet) [1981236]
- net/qrtr: fix __netdev_alloc_skb call (Íñigo Huguet) [1981236]
- net: qrtr: Fix memory leak in qrtr_tun_open (Íñigo Huguet) [1981236]
- net/qrtr: restrict user-controlled length in qrtr_tun_write_iter() (Íñigo Huguet) [1981236]
- net: qrtr: fix null-ptr-deref in qrtr_ns_remove (Íñigo Huguet) [1981236]
- net: qrtr: Unprepare MHI channels during remove (Íñigo Huguet) [1981236]
- net: qrtr: Start MHI channels during init (Íñigo Huguet) [1981236]
- net: qrtr: Release distant nodes along the bridge node (Íñigo Huguet) [1981236]
- net: qrtr: Add GFP flags parameter to qrtr_alloc_ctrl_packet (Íñigo Huguet) [1981236]
- net: qrtr: Allow non-immediate node routing (Íñigo Huguet) [1981236]
- net: qrtr: Allow forwarded services (Íñigo Huguet) [1981236]
- net: qrtr: Fix port ID for control messages (Íñigo Huguet) [1981236]
- net: qrtr: ns: Fix the incorrect usage of rcu_read_lock() (Íñigo Huguet) [1981236]
- net: qrtr: ns: Protect radix_tree_deref_slot() using rcu read locks (Íñigo Huguet) [1981236]
- net: qrtr: check skb_put_padto() return value (Íñigo Huguet) [1981236]
- net: qrtr: fix usage of idr in port assignment to socket (Íñigo Huguet) [1981236]
- qrtr: orphan socket in qrtr_release() (Íñigo Huguet) [1981236]
- net: qrtr: Fix an out of bounds read qrtr_endpoint_post() (Íñigo Huguet) [1981236]
- net: qrtr: free flow in __qrtr_node_release (Íñigo Huguet) [1981236]
- treewide: replace '---help---' in Kconfig files with 'help' (Íñigo Huguet) [1981236]
- net: qrtr: Allocate workqueue before kernel_bind (Íñigo Huguet) [1981236]
- net: qrtr: Fix passing invalid reference to qrtr_local_enqueue() (Íñigo Huguet) [1981236]
- net: qrtr: Add MHI transport layer (Íñigo Huguet) [1981236]
- net: qrtr: Add tracepoint support (Íñigo Huguet) [1981236]
- net: qrtr: send msgs from local of same id as broadcast (Íñigo Huguet) [1981236]
- net: qrtr: Fix FIXME related to qrtr_ns_init() (Íñigo Huguet) [1981236]
- net: qrtr: Respond to HELLO message (Íñigo Huguet) [1981236]
- net: qrtr: Fix error pointer vs NULL bugs (Íñigo Huguet) [1981236]
- net: qrtr: fix spelling mistake "serivce" -> "service" (Íñigo Huguet) [1981236]
- net: qrtr: Fix the local node ID as 1 (Íñigo Huguet) [1981236]
- net: qrtr: Migrate nameservice to kernel from userspace (Íñigo Huguet) [1981236]
- net: qrtr: Remove receive worker (Íñigo Huguet) [1981236]
- net: qrtr: Make qrtr_port_lookup() use RCU (Íñigo Huguet) [1981236]
- net: qrtr: Migrate node lookup tree to spinlock (Íñigo Huguet) [1981236]
- net: qrtr: Implement outgoing flow control (Íñigo Huguet) [1981236]
- net: qrtr: Move resume-tx transmission to recvmsg (Íñigo Huguet) [1981236]
- net: qrtr: fix len of skb_put_padto in qrtr_node_enqueue (Íñigo Huguet) [1981236]
- net: qrtr: Simplify 'qrtr_tun_release()' (Íñigo Huguet) [1981236]
- net: qrtr: Stop rx_worker before freeing node (Íñigo Huguet) [1981236]
- net: qrtr: fix memort leak in qrtr_tun_write_iter (Íñigo Huguet) [1981236]
- treewide: Replace GPLv2 boilerplate/reference with SPDX - rule 284 (Íñigo Huguet) [1981236]
- treewide: Add SPDX license identifier - Makefile/Kconfig (Íñigo Huguet) [1981236]
- net: qrtr: Fix message type of outgoing packets (Íñigo Huguet) [1981236]
- net: qrtr: use protocol endiannes variable (Íñigo Huguet) [1981236]
- net: fix hangup on napi_disable for threaded napi (Paolo Abeni) [1949546]
- net: fix race between napi kthread mode and busy poll (Paolo Abeni) [1949546]
- net: add sysfs attribute to control napi threaded mode (Paolo Abeni) [1949546]
- net: implement threaded-able napi poll loop support (Paolo Abeni) [1949546]
- net: extract napi poll functionality to __napi_poll() (Paolo Abeni) [1949546]
- mfd: intel-lpss: Use devm_ioremap_uc for MMIO (Steve Best) [1986715]
- lib: devres: add a helper function for ioremap_uc (Steve Best) [1986715]
- sched/rt: Fix double enqueue caused by rt_effective_prio (Juri Lelli) [1990566]
- x86/crash: Remove crash_reserve_low_1M() (Lianbo Jiang) [1904318]
- x86/setup: Remove CONFIG_X86_RESERVE_LOW and reservelow= options (Lianbo Jiang) [1904318]
- x86/setup: Always reserve the first 1M of RAM (Lianbo Jiang) [1904318]
- powerpc/kexec: blacklist functions called in real mode for kprobe (Desnes A. Nunes do Rosario) [1987281]
- [s390] scsi: zfcp: Report port fc_security as unknown early during remote cable pull (Claudio Imbrenda) [1989910]
- x86/boot: Use common BUILD_BUG_ON (Prarit Bhargava) [1979940]
- net: phy: icplus: call phy_restore_page() when phy_select_page() fails (Petr Oros) [1955165]
- net: phy: ti: take into account all possible interrupt sources (Petr Oros) [1955165]
- net: phy: smsc: add missed clk_disable_unprepare in smsc_phy_probe() (Petr Oros) [1955165]
- net: phy: Expose phydev::dev_flags through sysfs (Petr Oros) [1955165]
- net: phy: fix save wrong speed and duplex problem if autoneg is on (Petr Oros) [1955165]
- net: phy: micrel: set soft_reset callback to genphy_soft_reset for KSZ8081 (Petr Oros) [1955165]
- net: phy: icplus: add MDI/MDIX support for IP101A/G (Petr Oros) [1955165]
- net: phy: icplus: add PHY counter for IP101G (Petr Oros) [1955165]
- net: phy: icplus: fix paged register access (Petr Oros) [1955165]
- net: phy: icplus: don't set APS_EN bit on IP101G (Petr Oros) [1955165]
- net: phy: icplus: split IP101A/G driver (Petr Oros) [1955165]
- net: phy: icplus: use the .soft_reset() of the phy-core (Petr Oros) [1955165]
- net: phy: icplus: drop address operator for functions (Petr Oros) [1955165]
- net: phy: icplus: use PHY_ID_MATCH_EXACT() for IP101A/G (Petr Oros) [1955165]
- net: phy: icplus: use PHY_ID_MATCH_MODEL() macro (Petr Oros) [1955165]
- net: phy: drop explicit genphy_read_status() op (Petr Oros) [1955165]
- net: phy: broadcom: remove BCM5482 1000Base-BX support (Petr Oros) [1955165]
- net: mdiobus: Prevent spike on MDIO bus reset signal (Petr Oros) [1955165]
- net: phy: realtek: Add support for RTL9000AA/AN (Petr Oros) [1955165]
- mdio, phy: fix -Wshadow warnings triggered by nested container_of() (Petr Oros) [1955165]
- net: phy: national: remove definition of DEBUG (Petr Oros) [1955165]
- net: phy: ar803x: disable extended next page bit (Petr Oros) [1955165]
- net: phy: smsc: fix clk error handling (Petr Oros) [1955165]
- net: phy: at803x: use phy_modify_mmd() (Petr Oros) [1955165]
- net: phy: replace mutex_is_locked with lockdep_assert_held in phylib (Petr Oros) [1955165]
- net: phy: micrel: Add KS8851 PHY support (Petr Oros) [1955165]
- net: phy: marvell: replace phy_modify() (Petr Oros) [1955165]
- net: phy: micrel: fix interrupt handling (Petr Oros) [1955165]
- net: phy: remove the .did_interrupt() and .ack_interrupt() callback (Petr Oros) [1955165]
- net: phy: qsemi: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: qsemi: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: ti: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: ti: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: national: remove the use of the .ack_interrupt() (Petr Oros) [1955165]
- net: phy: national: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: micrel: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: micrel: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: icplus: remove the use .ack_interrupt() (Petr Oros) [1955165]
- net: phy: icplus: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: intel-xway: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: intel-xway: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- mdio_bus: suppress err message for reset gpio EPROBE_DEFER (Petr Oros) [1955165]
- net: phy: don't duplicate driver name in phy_attached_print (Petr Oros) [1955165]
- net: phy: ste10Xp: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: ste10Xp: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: smsc: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: smsc: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: amd: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: amd: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: lxt: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: lxt: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: marvell: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: marvell: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: microchip: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: microchip: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: vitesse: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: vitesse: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: aquantia: do not return an error on clearing pending IRQs (Petr Oros) [1955165]
- net: phy: microchip_t1: Don't set .config_aneg (Petr Oros) [1955165]
- docs: networking: phy: s/2.5 times faster/2.5 times as fast/ (Petr Oros) [1955165]
- net: phy: realtek: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: realtek: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: add genphy_handle_interrupt_no_ack() (Petr Oros) [1955165]
- net: phy: davicom: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: davicom: implement generic .handle_interrupt() calback (Petr Oros) [1955165]
- net: phy: cicada: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: cicada: implement the generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: broadcom: remove use of ack_interrupt() (Petr Oros) [1955165]
- net: phy: broadcom: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: aquantia: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: aquantia: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: mscc: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: mscc: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: at803x: remove the use of .ack_interrupt() (Petr Oros) [1955165]
- net: phy: at803x: implement generic .handle_interrupt() callback (Petr Oros) [1955165]
- net: phy: make .ack_interrupt() optional (Petr Oros) [1955165]
- net: phy: add a shutdown procedure (Petr Oros) [1955165]
- net: phy: export phy_error and phy_trigger_machine (Petr Oros) [1955165]
- drivers: net: phy: Fix spelling in comment defalut to default (Petr Oros) [1955165]
- net: phy: marvell: add special handling of Finisar modules with 88E1111 (Petr Oros) [1955165]
- net: phylink: disable BMCR_ISOLATE in phylink_mii_c22_pcs_config (Petr Oros) [1955165]
- gpiolib: Extract gpiod_not_found() helper (Petr Oros) [1955165]
- net: phy: marvell: Use phy_read_paged() instead of open coding it (Petr Oros) [1955165]
- net: phy: dp83869: fix unsigned comparisons against less than zero values (Petr Oros) [1955165]
- net: phy: dp83869: Add speed optimization feature (Petr Oros) [1955165]
- net: phy: dp83869: support Wake on LAN (Petr Oros) [1955165]
- net: phy: dp83822: Update the fiber advertisement for speed (Petr Oros) [1955165]
- net: dp83869: Add ability to advertise Fiber connection (Petr Oros) [1955165]
- net: phy: Do not warn in phy_stop() on PHY_DOWN (Petr Oros) [1955165]
- net: phy: Avoid NPD upon phy_detach() when driver is unbound (Petr Oros) [1955165]
- net: phy: bcm7xxx: request and manage GPHY clock (Petr Oros) [1955165]
- net: phy: smsc: LAN8710/20: add phy refclk in support (Petr Oros) [1955165]
- net: phy: smsc: simplify config_init callback (Petr Oros) [1955165]
- net: phy: smsc: skip ENERGYON interrupt if disabled (Petr Oros) [1955165]
- net: phy: dp83867: Fix various styling and space issues (Petr Oros) [1955165]
- net: mdiobus: add clause 45 mdiobus write accessor (Petr Oros) [1955165]
- net: phylink: consider QSGMII interface mode in phylink_mii_c22_pcs_get_state (Petr Oros) [1955165]
- net: phylink: add helper function to decode USXGMII word (Petr Oros) [1955165]
- net: phy: DP83822: Add ability to advertise Fiber connection (Petr Oros) [1955165]
- net: dp83869: Fix RGMII internal delay configuration (Petr Oros) [1955165]
- net: mdio-mux-gpio: use devm_gpiod_get_array() (Petr Oros) [1955165]
- net: phy: add USXGMII link partner ability constants (Petr Oros) [1955165]
- net: phy: mdio-octeon: Cleanup module loading dependencies (Petr Oros) [1955165]
- net: phy: cavium: Improve __iomem mess (Petr Oros) [1955165]
- net: phy: Add a helper to return the index for of the internal delay (Petr Oros) [1955165]
- dt-bindings: net: Add RGMII internal delay for DP83869 (Petr Oros) [1955165]
- net: dp83869: Add RGMII internal delay configuration (Petr Oros) [1955165]
- net: phy: DP83822: Add setting the fixed internal delay (Petr Oros) [1955165]
- net: dp83869: Fix OF_MDIO config check (Petr Oros) [1955165]
- net: dp83867: Fix OF_MDIO config check (Petr Oros) [1955165]
- net: marvell: Fix OF_MDIO config check (Petr Oros) [1955165]
- net: phy: mscc: use mdiobus_get_phy() (Petr Oros) [1955165]
- net: phy: improve phy_driver callback handle_interrupt (Petr Oros) [1955165]

* Mon Aug 09 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-330.el8]
- net/mlx5e: Add IPsec support to uplink representor (Alaa Hleihel) [1980171 1926098]
- net/mlx5e: TC: Reserved bit 31 of REG_C1 for IPsec offload (Alaa Hleihel) [1980171 1926098]
- net/mlx5e: TC: Use bit counts for register mapping (Alaa Hleihel) [1980171 1926098]
- RDMA/mlx5: Don't access NULL-cleared mpi pointer (Alaa Hleihel) [1980171]
- RDMA/mlx5: Split mlx5_ib_update_xlt() into ODP and non-ODP cases (Alaa Hleihel) [1980171]
- ip6_tunnel: allow redirecting ip6gre and ipxip6 packets to eth devices (Guillaume Nault) [1921539]
- gre: let mac_header point to outer header only when necessary (Guillaume Nault) [1921539]
- sit: allow redirecting ip6ip, ipip and mplsip packets to eth devices (Guillaume Nault) [1921539]
- ipip: allow redirecting ipip and mplsip packets to eth devices (Guillaume Nault) [1921539]
- bareudp: allow redirecting bareudp packets to eth devices (Guillaume Nault) [1921539]
- net: handle ARPHRD_IP6GRE in dev_is_mac_header_xmit() (Guillaume Nault) [1921539]
- net: ipv4: fix memory leak in ip_mc_add1_src (Guillaume Nault) [1987605]
- net: ipv4: fix memory leak in netlbl_cipsov4_add_std (Guillaume Nault) [1987605]
- ixgbe: fix probing of multi-port devices with one MDIO (Ken Cox) [1968696]
- [s390] s390/qdio: let driver manage the QAOB (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: remove RX VLAN filter stubs in L3 driver (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: enable napi_consume_skb() for pending TX buffers (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: allocate initial TX Buffer structs with GFP_KERNEL (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: fix notification for pending buffers during teardown (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: schedule TX NAPI on QAOB completion (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: improve completion of pending TX buffers (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: fix memory leak after failed TX Buffer allocation (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: don't fake a TX completion interrupt after TX error (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: make cast type selection for af_iucv skbs robust (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: pass proto to qeth_l3_get_cast_type() (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: remove qeth_get_ip_version() (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: clean up load/remove code for disciplines (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: fix L2 header access in qeth_l3_osa_features_check() (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: fix locking for discipline setup / removal (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: fix deadlock during recovery (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: make qeth_qdio_handle_aob() more robust (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: remove QETH_QDIO_BUF_HANDLED_DELAYED state (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: don't replace a fully completed async TX buffer (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: use dev->groups for common sysfs attributes (Claudio Imbrenda) [1919250]
- [s390] s390/ccwgroup: use bus->dev_groups for bus-based sysfs attributes (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: don't call INIT_LIST_HEAD() on iob's list entry (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: fix tear down of async TX buffers (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: fix af_iucv notification race (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: improve selection of ethtool link modes (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: use QUERY OAT for initial link info (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: clean up default cases for ethtool link mode (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: set static link info during initialization (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: improve QUERY CARD INFO processing (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: tolerate error when querying card info (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: remove useless if/else (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: reduce rtnl locking for switchdev events (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: static checker cleanups (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: use netdev_name() (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: constify the disciplines (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: allow configuration of TX queues for OSA devices (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: de-magic the QIB parm area (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: keep track of wanted TX queues (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: remove forward declarations in L2 code (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: consolidate teardown code (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: consolidate online code (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: cancel cmds earlier during teardown (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: tighten ucast IP locking (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: replace deprecated simple_stroul() (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: clean up string ops in qeth_l3_parse_ipatoe() (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: relax locking for ipato config data (Claudio Imbrenda) [1919250]
- [s390] s390/qeth: don't init refcount twice for mcast IPs (Claudio Imbrenda) [1919250]
- mm, page_alloc: check for max order in hot path (Jay Shin) [1982466]
- be2net: Fix an error handling path in 'be_probe()' (Petr Oros) [1980665]
- net: emulex: benet: simplify the return expression of be_if_create() (Petr Oros) [1980665]
- treewide: Use fallthrough pseudo-keyword (Petr Oros) [1980665]
- be2net: convert to new udp_tunnel_nic infra (Petr Oros) [1980665]
- benet: use generic power management (Petr Oros) [1980665]
- treewide: replace '---help---' in Kconfig files with 'help' (Petr Oros) [1980665]
- hwmon: (k10temp) Zen3 Ryzen Desktop CPUs support (David Arcari) [1980072]
- hwmon: (k10temp) Remove support for displaying voltage and current on Zen CPUs (David Arcari) [1980072]
- hwmon: (k10temp) Add support for Zen3 CPUs (David Arcari) [1980072]
- hwmon: (k10temp) Take out debugfs code (David Arcari) [1980072]
- hwmon: (k10temp) Define SVI telemetry and current factors for Zen2 CPUs (David Arcari) [1980072]
- hwmon: (k10temp) Create common functions and macros for Zen CPU families (David Arcari) [1980072]
- hwmon: (k10temp) Add AMD family 17h model 60h PCI match (David Arcari) [1980072]
- hwmon: (k10temp) make some symbols static (David Arcari) [1980072]
- hwmon: (k10temp) Reorganize and simplify temperature support detection (David Arcari) [1980072]
- hwmon: (k10temp) Swap Tdie and Tctl on Family 17h CPUs (David Arcari) [1980072]
- hwmon: (k10temp) Display up to eight sets of CCD temperatures (David Arcari) [1980072]
- hwmon: (k10temp) Add debugfs support (David Arcari) [1980072]
- hwmon: (k10temp) Don't show temperature limits on Ryzen (Zen) CPUs (David Arcari) [1980072]
- hwmon: (k10temp) Show core and SoC current and voltages on Ryzen CPUs (David Arcari) [1980072]
- hwmon: (k10temp) Report temperatures per CPU die (David Arcari) [1980072]
- hmon: (k10temp) Convert to use devm_hwmon_device_register_with_info (David Arcari) [1980072]
- hwmon: (k10temp) Use bitops (David Arcari) [1980072]
- hwmon: (k10temp) Add support for AMD family 17h, model 70h CPUs (David Arcari) [1980072]
- treewide: Replace GPLv2 boilerplate/reference with SPDX - rule 56 (David Arcari) [1980072]
- hwmon: (k10temp) Add Hygon Dhyana support (David Arcari) [1980072]
- hwmon: (k10temp) Auto-convert to use SENSOR_DEVICE_ATTR_{RO, RW, WO} (David Arcari) [1980072]
- hwmon: (k10temp) Support all Family 15h Model 6xh and Model 7xh processors (David Arcari) [1980072]
- hwmon: k10temp: Support Threadripper 2920X, 2970WX; simplify offset table (David Arcari) [1980072]
- hwmon: (k10temp) 27C Offset needed for Threadripper2 (David Arcari) [1980072]
- x86/amd_nb: Add AMD family 17h model 60h PCI IDs (David Arcari) [1980072]
- x86/amd_nb: Add PCI device IDs for family 17h, model 70h (David Arcari) [1980072]
- x86/pci, x86/amd_nb: Add Hygon Dhyana support to PCI and northbridge (David Arcari) [1980072]
- Revert "[hwmon] hwmon: (k10temp) Add support for Zen3 CPUs" (David Arcari) [1980072]
- serial: 8250: Process sysrq at port unlock time (Prarit Bhargava) [1982016]
- serial: core: Include console.h from serial_core.h (Prarit Bhargava) [1982016]
- serial: core: Allow processing sysrq at port unlock time (Prarit Bhargava) [1982016]
- [s390] s390/dasd: add missing discipline function (Claudio Imbrenda) [1981804]
- bnxt_en: Validate vlan protocol ID on RX packets (Jonathan Toppins) [1960961]
- KVM: do not allow mapping valid but non-reference-counted pages (Jon Maloy) [1975515]
- gfs2: Fix error handling in init_statfs (Andreas Gruenbacher) [1872440]
- gfs2: Fix underflow in gfs2_page_mkwrite (Andreas Gruenbacher) [1872440]
- gfs2: Pass glock holder to gfs2_file_direct_{read,write} (Andreas Gruenbacher) [1872440]
- gfs2: Prevent direct-I/O write fallback errors from getting lost (Andreas Gruenbacher) [1872440]
- gfs2: Improve mmap write vs. truncate consistency (Andreas Gruenbacher) [1872440]
- iomap: Permit pages without an iop to enter writeback (Andreas Gruenbacher) [1872440]
- iomap: Don't create iomap_page objects for inline files (Andreas Gruenbacher) [1872440]
- gfs2: Clean up gfs2_unstuff_dinode (Andreas Gruenbacher) [1872440]
- gfs2: Unstuff before locking page in gfs2_page_mkwrite (Andreas Gruenbacher) [1872440]
- gfs2: Clean up the error handling in gfs2_page_mkwrite (Andreas Gruenbacher) [1872440]
- gfs2: Fix end-of-file handling in gfs2_page_mkwrite (Andreas Gruenbacher) [1872440]
- gfs2: Use resource group glock sharing (Andreas Gruenbacher) [1442151]
- gfs2: Allow node-wide exclusive glock sharing (Andreas Gruenbacher) [1442151]
- gfs2: Add local resource group locking (Andreas Gruenbacher) [1442151]
- gfs2: Add per-reservation reserved block accounting (Andreas Gruenbacher) [1442151]
- gfs2: Rename rs_{free -> requested} and rd_{reserved -> requested} (Andreas Gruenbacher) [1442151]
- gfs2: Check for active reservation in gfs2_release (Andreas Gruenbacher) [1442151]
- gfs2: Don't search for unreserved space twice (Andreas Gruenbacher) [1442151]
- gfs2: Only pass reservation down to gfs2_rbm_find (Andreas Gruenbacher) [1442151]
- gfs2: Also reflect single-block allocations in rgd->rd_extfail_pt (Andreas Gruenbacher) [1442151]
- gfs2: Set GBF_FULL flags when reading resource group (Andreas Gruenbacher) [1442151]
- gfs2: Don't clear GBF_FULL flags in rs_deltree (Andreas Gruenbacher) [1442151]
- Revert "gfs2: Don't reject a supposedly full bitmap if we have blocks reserved" (Andreas Gruenbacher) [1442151]
- gfs2: Minor gfs2_inplace_reserve cleanup (Andreas Gruenbacher) [1442151]
- gfs2: Get rid of unnecessary variable in gfs2_alloc_blocks (Andreas Gruenbacher) [1442151]
- gfs2: Only use struct gfs2_rbm for bitmap manipulations (Andreas Gruenbacher) [1442151]
- gfs2: Turn gfs2_rbm_incr into gfs2_rbm_add (Andreas Gruenbacher) [1872440]
- gfs2: Replace gfs2_lblk_to_dblk with gfs2_get_extent (Andreas Gruenbacher) [1872440]
- gfs2: Turn gfs2_extent_map into gfs2_{get,alloc}_extent (Andreas Gruenbacher) [1872440]
- gfs2: Add new gfs2_iomap_get helper (Andreas Gruenbacher) [1872440]
- gfs2: Make inode operations static (Andreas Gruenbacher) [1872440]
- gfs2: use iomap for buffered I/O in ordered and writeback mode (Andreas Gruenbacher) [1872440]
- gfs2: Never call gfs2_block_zero_range with an open transaction (Andreas Gruenbacher) [1872440]
- gfs2: Upgrade shared glocks for atime updates (Andreas Gruenbacher) [1872440]
- gfs2: Rework read and page fault locking (Andreas Gruenbacher) [1872440]
- fs: Add IOCB_NOIO flag for generic_file_read_iter (Andreas Gruenbacher) [1872440]
- gfs2: Remove duplicate call from gfs2_create_inode (Andreas Gruenbacher) [1872440]
- gfs2: Multi-block allocations in gfs2_page_mkwrite (Andreas Gruenbacher) [1872440]
- gfs2: implement gfs2_block_zero_range using iomap_zero_range (Andreas Gruenbacher) [1872440]
- gfs2: Add support for IOMAP_ZERO (Andreas Gruenbacher) [1872440]
- gfs2: gfs2_iomap_begin cleanup (Andreas Gruenbacher) [1872440]
- gfs2: Remove unused gfs2_iomap_alloc argument (Andreas Gruenbacher) [1872440]
- gfs2: don't use buffer_heads in gfs2_allocate_page_backing (Andreas Gruenbacher) [1872440]
- gfs2: use iomap_bmap instead of generic_block_bmap (Andreas Gruenbacher) [1872440]
- gfs2: mark stuffed_readpage static (Andreas Gruenbacher) [1872440]
- gfs2: merge gfs2_writepage_common into gfs2_writepage (Andreas Gruenbacher) [1872440]
- gfs2: merge gfs2_writeback_aops and gfs2_ordered_aops (Andreas Gruenbacher) [1872440]
- gfs2: remove the unused gfs2_stuffed_write_end function (Andreas Gruenbacher) [1872440]
- gfs2: use page_offset in gfs2_page_mkwrite (Andreas Gruenbacher) [1442151]
- gfs2: Add new sysfs file for gfs2 status (Andreas Gruenbacher) [1442151]
- gfs2: Per-revoke accounting in transactions (Andreas Gruenbacher) [1442151]
- gfs2: Rework the log space allocation logic (Andreas Gruenbacher) [1442151]
- gfs2: Minor calc_reserved cleanup (Andreas Gruenbacher) [1442151]
- gfs2: Move function gfs2_ail_empty_tr (Andreas Gruenbacher) [1442151]
- gfs2: Get rid of current_tail() (Andreas Gruenbacher) [1442151]
- gfs2: Use a tighter bound in gfs2_trans_begin (Andreas Gruenbacher) [1442151]
- gfs2: Clean up gfs2_log_reserve (Andreas Gruenbacher) [1442151]
- gfs2: Don't wait for journal flush in clean_journal (Andreas Gruenbacher) [1442151]
- gfs2: Move lock flush locking to gfs2_trans_{begin,end} (Andreas Gruenbacher) [1442151]
- gfs2: Get rid of sd_reserving_log (Andreas Gruenbacher) [1442151]
- gfs2: Clean up on-stack transactions (Andreas Gruenbacher) [1442151]
- gfs2: Use sb_start_intwrite in gfs2_ail_empty_gl (Andreas Gruenbacher) [1442151]
- gfs2: Clean up ail2_empty (Andreas Gruenbacher) [1442151]
- gfs2: Rename gfs2_{write => flush}_revokes (Andreas Gruenbacher) [1442151]
- gfs2: Minor debugging improvement (Andreas Gruenbacher) [1442151]
- gfs2: Some documentation updates (Andreas Gruenbacher) [1442151]
- gfs2: Minor gfs2_write_revokes cleanups (Andreas Gruenbacher) [1442151]
- gfs2: Simplify the buf_limit and databuf_limit definitions (Andreas Gruenbacher) [1442151]
- gfs2: Un-obfuscate function jdesc_find_i (Andreas Gruenbacher) [1442151]
- gfs2: Fix fall-through warnings for Clang (Andreas Gruenbacher) [1442151]
- gfs2: Make gfs2_setattr_simple static (Andreas Gruenbacher) [1442151]
- gfs2: Turn gfs2_meta_indirect_buffer into gfs2_meta_buffer (Andreas Gruenbacher) [1442151]
- gfs2: Remove unused variable sb_format (Andreas Gruenbacher) [1442151]
- gfs2: Fix dir.c function parameter descriptions (Andreas Gruenbacher) [1442151]
- gfs2: Eliminate gh parameter from go_xmote_bh func (Andreas Gruenbacher) [1442151]
- gfs2: don't create empty buffers for NO_CREATE (Andreas Gruenbacher) [1442151]
- gfs2: Flag a withdraw if init_threads() fails (Andreas Gruenbacher) [1442151]
- gfs2: bypass log flush if the journal is not live (Andreas Gruenbacher) [1442151]
- gfs2: bypass signal_our_withdraw if no journal (Andreas Gruenbacher) [1442151]
- gfs2: fix use-after-free in trans_drain (Andreas Gruenbacher) [1442151]
- gfs2: make function gfs2_make_fs_ro() to void type (Andreas Gruenbacher) [1442151]
- whack-a-mole: don't open-code iminor/imajor (Andreas Gruenbacher) [1442151]
- gfs2: amend SLAB_RECLAIM_ACCOUNT on gfs2 related slab cache (Andreas Gruenbacher) [1442151]
- gfs2: Remove sb_start_write from gfs2_statfs_sync (Andreas Gruenbacher) [1442151]
- gfs2: check for empty rgrp tree in gfs2_ri_update (Andreas Gruenbacher) [1442151]
- gfs2: set lockdep subclass for iopen glocks (Andreas Gruenbacher) [1442151]
- gfs2: Fix deadlock dumping resource group glocks (Andreas Gruenbacher) [1442151]
- gfs2: Wake up when sd_glock_disposal becomes zero (Andreas Gruenbacher) [1442151]
- gfs2: Split up gfs2_meta_sync into inode and rgrp versions (Andreas Gruenbacher) [1442151]
- gfs2: Don't ignore inode write errors during inode_go_sync (Andreas Gruenbacher) [1442151]
- gfs2: Ignore subsequent errors after withdraw in rgrp_go_sync (Andreas Gruenbacher) [1442151]
- gfs2: Eliminate gl_vm (Andreas Gruenbacher) [1442151]
- gfs2: Fix comments to glock_hash_walk (Andreas Gruenbacher) [1442151]
- gfs2: eliminate GLF_QUEUED flag in favor of list_empty(gl_holders) (Andreas Gruenbacher) [1442151]
- gfs2: Fix invalid block size message (Andreas Gruenbacher) [1442151]
- gfs2: add validation checks for size of superblock (Andreas Gruenbacher) [1442151]
- gfs2: use-after-free in sysfs deregistration (Andreas Gruenbacher) [1442151]
- Revert "GFS2: Prevent delete work from occurring on glocks used for create" (Andreas Gruenbacher) [1442151]
- gfs2: Don't skip dlm unlock if glock has an lvb (Andreas Gruenbacher) [1442151]
- gfs2: Lock imbalance on error path in gfs2_recover_one (Andreas Gruenbacher) [1442151]
- gfs2: keep bios separate for each journal (Bob Peterson) [1442151]
- gfs2: change from write to read lock for sd_log_flush_lock in journal replay (Bob Peterson) [1442151]
- gfs2: don't lock sd_log_flush_lock in try_rgrp_unlink (Bob Peterson) [1442151]
- gfs2: fix glock confusion in function signal_our_withdraw (Bob Peterson) [1442151]
- gfs2: Don't write log headers after file system withdraw (Andreas Gruenbacher) [1442151]
- gfs2: make gfs2_log_write_page static (Andreas Gruenbacher) [1442151]
- gfs2: fix possible reference leak in gfs2_check_blk_type (Andreas Gruenbacher) [1442151]
- gfs2: Fix NULL pointer dereference in gfs2_rgrp_dump (Andreas Gruenbacher) [1442151]
- gfs2: remove unused LBIT macros (Andreas Gruenbacher) [1925121]
- gfs2: add missing log_blocks trace points in gfs2_write_revokes (Andreas Gruenbacher) [1925121]
- gfs2: print details on transactions that aren't properly ended (Andreas Gruenbacher) [1925121]
- gfs2: fix use-after-free on transaction ail lists (Andreas Gruenbacher) [1925121]
- gfs2: new slab for transactions (Andreas Gruenbacher) [1925121]
- gfs2: initialize transaction tr_ailX_lists earlier (Andreas Gruenbacher) [1925121]
- gfs2: eliminate ssize parameter from gfs2_struct2blk (Andreas Gruenbacher) [1925121]
- gfs2: Fix error path kobject memory leak (Andreas Gruenbacher) [1925121]
- gfs2: Rename gfs2_trans_{add_unrevoke => remove_revoke} (Andreas Gruenbacher) [1925121]
- gfs2: Remove unnecessary extern declarations (Andreas Gruenbacher) [1925121]
- gfs2: Fix sign extension bug in gfs2_update_stats (Andreas Gruenbacher) [1925121]
- gfs2: Fix an incorrect gfs2_assert() (Bob Peterson) [1925121]
- gfs2: Remove vestigial bd_ops (Andreas Gruenbacher) [1925121]
- gfs2: Fix the gfs2_invalidatepage description (Andreas Gruenbacher) [1925121]
- gfs2: Clean up gfs2_is_{ordered,writeback} (Andreas Gruenbacher) [1925121]
- gfs2: Pass resource group to rgblk_free (Andreas Gruenbacher) [1925121]
- gfs2: Remove unnecessary gfs2_rlist_alloc parameter (Andreas Gruenbacher) [1925121]
- gfs2: Fix some minor typos (Andreas Gruenbacher) [1925121]
- gfs2: Remove unused RGRP_RSRV_MINBYTES definition (Andreas Gruenbacher) [1925121]
- gfs2: Move rs_{sizehint, rgd_gh} fields into the inode (Andreas Gruenbacher) [1925121]
- gfs2: Clean up out-of-bounds check in gfs2_rbm_from_block (Andreas Gruenbacher) [1925121]
- gfs2: Always check the result of gfs2_rbm_from_block (Andreas Gruenbacher) [1925121]
- gfs: no need to check return value of debugfs_create functions (Andreas Gruenbacher) [1925121]
- serial_core: switch to ->[sg]et_serial() (Artem Savkov) [1952415]

* Thu Aug 05 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-329.el8]
- tipc: fix sleeping in tipc accept routine (Xin Long) [1986549]
- tipc: fix implicit-connect for SYN+ (Xin Long) [1986549]
- net: tipc: fix FB_MTU eat two pages (Xin Long) [1986549]
- tipc: Return the correct errno code (Xin Long) [1986549]
- net/mlx5: Fix checksum issue of VXLAN and IPsec crypto offload (Alaa Hleihel) [1979901]
- net/mlx5: Optimize mlx5e_feature_checks for non IPsec packet (Alaa Hleihel) [1979901]
- bpf, selftests: Fold test_current_pid_tgid_new_ns into test_progs. (Jiri Olsa) [1869447]
- libbpf: Support BTF_KIND_FLOAT during type compatibility checks in CO-RE (Jiri Olsa) [1984486]
- bpf: Generate BTF_KIND_FLOAT when linking vmlinux (Jiri Olsa) [1984486]
- selftests/bpf: Add BTF_KIND_FLOAT to btf_dump_test_case_syntax (Jiri Olsa) [1984486]
- selftests/bpf: Add BTF_KIND_FLOAT to test_core_reloc_size (Jiri Olsa) [1984486]
- bpf: Document BTF_KIND_FLOAT in btf.rst (Jiri Olsa) [1984486]
- selftests/bpf: Add BTF_KIND_FLOAT to the existing deduplication tests (Jiri Olsa) [1984486]
- selftest/bpf: Add BTF_KIND_FLOAT tests (Jiri Olsa) [1984486]
- bpf: Add BTF_KIND_FLOAT support (Jiri Olsa) [1984486]
- selftests/bpf: Use the 25th bit in the "invalid BTF_INFO" test (Jiri Olsa) [1984486]
- tools/bpftool: Add BTF_KIND_FLOAT support (Jiri Olsa) [1984486]
- libbpf: Add BTF_KIND_FLOAT support (Jiri Olsa) [1984486]
- bpf: Add BTF_KIND_FLOAT to uapi (Jiri Olsa) [1984486]
- libbpf: Add explicit padding to btf_dump_emit_type_decl_opts (Jiri Olsa) [1984486]
- selftests/bpf: Fix test_attach_probe for powerpc uprobes (Jiri Olsa) [1931531]
- selftests/bpf: Mask bpf_csum_diff() return value to 16 bits in test_verifier (Jiri Olsa) [1942513]
- openvswitch: add trace points (Aaron Conole) [1979680]
- vxlan: add missing rcu_read_lock() in neigh_reduce() (Guillaume Nault) [1987310]
- net: ip6_tunnel: Unregister catch-all devices (Guillaume Nault) [1987310]
- net: sit: Unregister catch-all devices (Guillaume Nault) [1987310]
- cpuidle: pseries: Do not cap the CEDE0 latency in fixup_cede0_latency() (Diego Domingos) [1952809]
- cpuidle: pseries: Fixup CEDE0 latency only for POWER10 onwards (Diego Domingos) [1952809]
- mm/swap: fix for "mm: workingset: age nonresident information alongside anonymous pages" (Rafael Aquini) [1984173]
- mm/thp: remap_page() is only needed on anonymous THP (Rafael Aquini) [1984173]
- mm/compaction: fix 'limit' in fast_isolate_freepages (Rafael Aquini) [1984173]
- mm/huge_memory.c: don't discard hugepage if other processes are mapping it (Rafael Aquini) [1984173]
- slub: restore slub_debug=- behavior (Rafael Aquini) [1984173]
- mm/gup: fix try_grab_compound_head() race with split_huge_page() (Rafael Aquini) [1984173]
- mm, futex: fix shared futex pgoff on shmem huge page (Rafael Aquini) [1984173]
- x86/mm: Avoid truncating memblocks for SGX memory (Rafael Aquini) [1984173]
- mm/thp: fix vma_address() if virtual address below file offset (Rafael Aquini) [1984173]
- mm, thp: use head page in __migration_entry_wait() (Rafael Aquini) [1984173]
- mm/slub.c: include swab.h (Rafael Aquini) [1984173]
- mm/memory-failure: make sure wait for page writeback in memory_failure (Rafael Aquini) [1984173]
- mm/slub: fix redzoning for small allocations (Rafael Aquini) [1984173]
- mm, hugetlb: fix simple resv_huge_pages underflow on UFFDIO_COPY (Rafael Aquini) [1984173]
- mm/page_alloc: fix counting of free pages after take off from buddy (Rafael Aquini) [1984173]
- userfaultfd: release page in error path to avoid BUG_ON (Rafael Aquini) [1984173]
- ksm: fix potential missing rmap_item for stable_node (Rafael Aquini) [1984173]
- mm/migrate.c: fix potential indeterminate pte entry in migrate_vma_insert_page() (Rafael Aquini) [1984173]
- mm/hugeltb: handle the error case in hugetlb_fix_reserve_counts() (Rafael Aquini) [1984173]
- khugepaged: fix wrong result value for trace_mm_collapse_huge_page_isolate() (Rafael Aquini) [1984173]
- mm/memory-failure: unnecessary amount of unmapping (Rafael Aquini) [1984173]
- mm: page_alloc: ignore init_on_free=1 for debug_pagealloc=1 (Rafael Aquini) [1984173]
- mm/sparse: add the missing sparse_buffer_fini() in error branch (Rafael Aquini) [1984173]
- mm: memcontrol: fix cpuhotplug statistics flushing (Rafael Aquini) [1984173]
- x86/mem_encrypt: Correct physical address calculation in __set_clr_pte_enc() (Rafael Aquini) [1984173]
- zsmalloc: account the number of compacted pages correctly (Rafael Aquini) [1984173]
- mm/compaction: fix misbehaviors of fast_find_migrateblock() (Rafael Aquini) [1984173]
- hugetlb: fix update_and_free_page contig page struct assumption (Rafael Aquini) [1984173]
- mm/hugetlb: fix potential double free in hugetlb_register_node() error path (Rafael Aquini) [1984173]
- mm/memory.c: fix potential pte_unmap_unlock pte error (Rafael Aquini) [1984173]
- mm/memory.c: fix potential pte_unmap_unlock pte error (Rafael Aquini) [1984173]
- mm/hugetlb: fix potential missing huge page size info (Rafael Aquini) [1984173]
- x86/mm: Fix leak of pmd ptlock (Rafael Aquini) [1984173]
- mm/zswap: fix passing zero to 'PTR_ERR' warning (Rafael Aquini) [1984173]
- mm/page_isolation: do not isolate the max order page (Rafael Aquini) [1984173]
- mm,memory_failure: always pin the page in madvise_inject_error (Rafael Aquini) [1984173]
- mm/page_alloc: speed up the iteration of max_order (Rafael Aquini) [1984173]
- x86/mm/mem_encrypt: Fix definition of PMD_FLAGS_DEC_WP (Rafael Aquini) [1984173]
- x86/mm/ident_map: Check for errors from ident_pud_init() (Rafael Aquini) [1984173]
- mm, oom_adj: don't loop through tasks in __set_oom_adj when not necessary (Rafael Aquini) [1984173]
- mm/page_alloc.c: fix freeing non-compound pages (Rafael Aquini) [1984173]
- mm: khugepaged: recalculate min_free_kbytes after memory hotplug as expected by khugepaged (Rafael Aquini) [1984173]
- percpu: fix first chunk size calculation for populated bitmap (Rafael Aquini) [1984173]
- x86, fakenuma: Fix invalid starting node ID (Rafael Aquini) [1984173]
- mm: include CMA pages in lowmem_reserve at boot (Rafael Aquini) [1984173]
- mm/rodata_test.c: fix missing function declaration (Rafael Aquini) [1984173]
- mm/page_counter: fix various data races at memsw (Rafael Aquini) [1984173]
- khugepaged: retract_page_tables() remember to test exit (Rafael Aquini) [1984173]
- mm/page_counter.c: fix protection usage propagation (Rafael Aquini) [1984173]
- x86/mm/numa: Remove uninitialized_var() usage (Rafael Aquini) [1984173]
- shmem: fix possible deadlocks on shmlock_user_lock (Rafael Aquini) [1984173]
- mm: swap: use smp_mb__after_atomic() to order LRU bit set (Rafael Aquini) [1984173]
- cgroup: memcg: net: do not associate sock with unrelated cgroup (Rafael Aquini) [1984173]
- mm: fix gup_pud_range (Rafael Aquini) [1984173]
- x86/mm: Fix NX bit clearing issue in kernel_map_pages_in_pgd (Rafael Aquini) [1984173]
- mm/shmem.c: thp, shmem: fix conflict of above-47bit hint address and PMD alignment (Rafael Aquini) [1984173]
- mm: Implement no-MMU variant of vmalloc_user_node_flags (Rafael Aquini) [1984173]
- usercopy: Avoid HIGHMEM pfn warning (Rafael Aquini) [1984173]
- x86/mm: Don't exceed the valid physical address space (Rafael Aquini) [1984173]
- x86/mm/mem_encrypt: Fix erroneous sizeof() (Rafael Aquini) [1984173]
- s390/mm: fix addressing exception after suspend/resume (Rafael Aquini) [1984173]
- blk-mq: fix use-after-free in blk_mq_exit_sched (Ming Lei) [1969726]
- nvmet: use new ana_log_size instead the old one (Maurizio Lombardi) [1949777]
- nvme-multipath: fix double initialization of ANA state (Maurizio Lombardi) [1949777]
- configs: enable CONFIG_LEDS_BRIGHTNESS_HW_CHANGED (Benjamin Tissoires) [1922973]

* Tue Aug 03 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-328.el8]
- Revert "xfs: enable new inode btree counters feature" (Bill O'Donnell) [1976969]
- Revert "xfs: enable big timestamps" (Bill O'Donnell) [1976969]
- Revert "xfs: remove kmem_realloc()" (Bill O'Donnell) [1976969]
- xfs: prevent metadata files from being inactivated (Bill O'Donnell) [1976969]
- quota: widen timestamps for the fs_disk_quota structure (Bill O'Donnell) [1976969]
- cifs: fix fallocate when trying to allocate a hole. (Ronnie Sahlberg) [1866684]
- cifs: only write 64kb at a time when fallocating a small region of a file (Ronnie Sahlberg) [1866684]
- cifs: improve fallocate emulation (Ronnie Sahlberg) [1866684]
- cifs: Do not use the original cruid when following DFS links for multiuser mounts (Ronnie Sahlberg) [1812436]
- cifs: ignore relatime in smb3_fs_context_parse_monolithic() (Ronnie Sahlberg) [1812436]
- cifs: fix regression when mounting shares with prefix paths (Ronnie Sahlberg) [1812436]
- cifs: fix leak in cifs_smb3_do_mount() ctx (Ronnie Sahlberg) [1812436]
- cifs: add fs_context param to parsing helpers (Ronnie Sahlberg) [1812436]
- cifs: make fs_context error logging wrapper (Ronnie Sahlberg) [1812436]
- cifs: warn and fail if trying to use rootfs without the config option (Ronnie Sahlberg) [1812436]
- cifs: fix nodfs mount option (Ronnie Sahlberg) [1812436]
- cifs: fix handling of escaped ',' in the password mount argument (Ronnie Sahlberg) [1812436]
- cifs: Add new parameter "acregmax" for distinct file and directory metadata timeout (Ronnie Sahlberg) [1812436]
- cifs: convert revalidate of directories to using directory metadata cache timeout (Ronnie Sahlberg) [1812436]
- cifs: Add new mount parameter "acdirmax" to allow caching directory metadata (Ronnie Sahlberg) [1812436]
- cifs: fix trivial typo (Ronnie Sahlberg) [1812436]
- cifs: Set CIFS_MOUNT_USE_PREFIX_PATH flag on setting cifs_sb->prepath. (Ronnie Sahlberg) [1812436]
- cifs: In the new mount api we get the full devname as source= (Ronnie Sahlberg) [1812436]
- cifs: do not disable noperm if multiuser mount option is not provided (Ronnie Sahlberg) [1812436]
- cifs: fix dfs-links (Ronnie Sahlberg) [1812436]
- cifs: fix dfs domain referrals (Ronnie Sahlberg) [1812436]
- cifs: returning mount parm processing errors correctly (Ronnie Sahlberg) [1812436]
- cifs: fix mounts to subdirectories of target (Ronnie Sahlberg) [1812436]
- cifs: ignore auto and noauto options if given (Ronnie Sahlberg) [1812436]
- cifs: connect: style: Simplify bool comparison (Ronnie Sahlberg) [1812436]
- fs: cifs: remove unneeded variable in smb3_fs_context_dup (Ronnie Sahlberg) [1812436]
- cifs: Fix support for remount when not changing rsize/wsize (Ronnie Sahlberg) [1812436]
- cifs: handle "guest" mount parameter (Ronnie Sahlberg) [1812436]
- cifs: correct four aliased mount parms to allow use of previous names (Ronnie Sahlberg) [1812436]
- cifs: fix use after free in cifs_smb3_do_mount() (Ronnie Sahlberg) [1812436]
- cifs: fix rsize/wsize to be negotiated values (Ronnie Sahlberg) [1812436]
- cifs: update super_operations to show_devname (Ronnie Sahlberg) [1812436]
- cifs: fix uninitialized variable in smb3_fs_context_parse_param (Ronnie Sahlberg) [1812436]
- cifs: update mnt_cifs_flags during reconfigure (Ronnie Sahlberg) [1812436]
- cifs: move update of flags into a separate function (Ronnie Sahlberg) [1812436]
- cifs: remove ctx argument from cifs_setup_cifs_sb (Ronnie Sahlberg) [1812436]
- cifs: do not allow changing posix_paths during remount (Ronnie Sahlberg) [1812436]
- cifs: uncomplicate printing the iocharset parameter (Ronnie Sahlberg) [1812436]
- cifs: don't create a temp nls in cifs_setup_ipc (Ronnie Sahlberg) [1812436]
- cifs: simplify handling of cifs_sb/ctx->local_nls (Ronnie Sahlberg) [1812436]
- cifs: we do not allow changing username/password/unc/... during remount (Ronnie Sahlberg) [1812436]
- cifs: add initial reconfigure support (Ronnie Sahlberg) [1812436]
- cifs: move [brw]size from cifs_sb to cifs_sb->ctx (Ronnie Sahlberg) [1812436]
- cifs: move cifs_cleanup_volume_info[_content] to fs_context.c (Ronnie Sahlberg) [1812436]
- cifs: Add missing sentinel to smb3_fs_parameters (Ronnie Sahlberg) [1812436]
- cifs: remove actimeo from cifs_sb (Ronnie Sahlberg) [1812436]
- cifs: remove [gu]id/backup[gu]id/file_mode/dir_mode from cifs_sb (Ronnie Sahlberg) [1812436]
- cifs: minor kernel style fixes for comments (Ronnie Sahlberg) [1812436]
- linux/parser.h: add include guards (Ronnie Sahlberg) [1812436]
- cifs: Make extract_sharename function public (Ronnie Sahlberg) [1812436]
- cifs: Make extract_hostname function public (Ronnie Sahlberg) [1812436]
- cifs: get rid of cifs_sb->mountdata (Ronnie Sahlberg) [1812436]
- cifs: add an smb3_fs_context to cifs_sb (Ronnie Sahlberg) [1812436]
- cifs: remove the devname argument to cifs_compose_mount_options (Ronnie Sahlberg) [1812436]
- cifs: switch to new mount api (Ronnie Sahlberg) [1812436]
- cifs: move cifs_parse_devname to fs_context.c (Ronnie Sahlberg) [1812436]
- cifs: move the enum for cifs parameters into fs_context.h (Ronnie Sahlberg) [1812436]
- cifs: rename dup_vol to smb3_fs_context_dup and move it into fs_context.c (Ronnie Sahlberg) [1812436]
- cifs: rename smb_vol as smb3_fs_context and move it to fs_context.h (Ronnie Sahlberg) [1812436]
- cifs: Fix fall-through warnings for Clang (Ronnie Sahlberg) [1812436]
- cifs: refactor create_sd_buf() and and avoid corrupting the buffer (Ronnie Sahlberg) [1812436]
- scsi: lpfc: Fix failure to transmit ABTS on FC link (Dick Kennedy) [1964670]
- scsi: lpfc: Update lpfc version to 12.8.0.10 (Dick Kennedy) [1964670]
- scsi: lpfc: Reregister FPIN types if ELS_RDF is received from fabric controller (Dick Kennedy) [1964670]
- scsi: lpfc: Add a option to enable interlocked ABTS before job completion (Dick Kennedy) [1964670]
- scsi: lpfc: Fix crash when lpfc_sli4_hba_setup() fails to initialize the SGLs (Dick Kennedy) [1964670]
- scsi: lpfc: Ignore GID-FT response that may be received after a link flip (Dick Kennedy) [1964670]
- scsi: lpfc: Fix node handling for Fabric Controller and Domain Controller (Dick Kennedy) [1964670]
- scsi: lpfc: Fix Node recovery when driver is handling simultaneous PLOGIs (Dick Kennedy) [1964670]
- scsi: lpfc: Add ndlp kref accounting for resume RPI path (Dick Kennedy) [1964670]
- scsi: lpfc: Fix "Unexpected timeout" error in direct attach topology (Dick Kennedy) [1964670]
- scsi: lpfc: Fix non-optimized ERSP handling (Dick Kennedy) [1964670]
- scsi: lpfc: Fix unreleased RPIs when NPIV ports are created (Dick Kennedy) [1964670]
- scsi: lpfc: Remove redundant assignment to pointer temp_hdr (Dick Kennedy) [1964670]
- Documentation: add more details in tipc.rst (Xin Long) [1872308]
- net/tipc: add TIPC chapter to networking Documentation (Xin Long) [1872308]
- scsi: qla2xxx: Reserve extra IRQ vectors (Nilesh Javali) [1964834]
- tracing/hwlat: Honor the tracing_cpumask (Jerome Marchand) [1920595]
- tracing/hwlat: Drop the duplicate assignment in start_kthread() (Jerome Marchand) [1920595]
- sfc: add logs explaining XDP_TX/REDIRECT is not available (Íñigo Huguet) [1976756]
- sfc: ensure correct number of XDP queues (Íñigo Huguet) [1976756]
- sfc: fix lack of XDP TX queues - error XDP TX failed (-22) (Íñigo Huguet) [1976756]
- sfc: Remove rcu_read_lock() around XDP program invocation (Íñigo Huguet) [1976756]
- sfc: use DEVICE_ATTR_*() macro (Íñigo Huguet) [1976756]
- sfc: farch: fix compile warning in efx_farch_dimension_resources() (Íñigo Huguet) [1976756]
- sfc: don't use netif_info et al before net_device is registered (Íñigo Huguet) [1976756]
- net: stmmac: Do not enable RX FIFO overflow interrupts (Mark Salter) [1905243]
- net: stmmac: Clear receive all(RA) bit when promiscuous mode is off (Mark Salter) [1905243]
- net: stmmac: cleared __FPE_REMOVING bit in stmmac_fpe_start_wq() (Mark Salter) [1905243]
- stmmac: intel: Enable HW descriptor prefetch by default (Mark Salter) [1905243]
- net: stmmac: Add HW descriptor prefetch setting for DWMAC Core 5.20 onwards (Mark Salter) [1905243]
- net: stmmac: fix TSO and TBS feature enabling during driver open (Mark Salter) [1905243]
- stmmac: intel: set TSO/TBS TX Queues default settings (Mark Salter) [1905243]
- stmmac: intel: unlock on error path in intel_crosststamp() (Mark Salter) [1905243]
- net: stmmac: use __napi_schedule() for PREEMPT_RT (Mark Salter) [1905243]
- stmmac: intel: Enable SERDES PHY rx clk for PSE (Mark Salter) [1905243]
- stmmac: intel: Drop duplicate ID in the list of PCI device IDs (Mark Salter) [1905243]
- stmmac: intel: add cross time-stamping freq difference adjustment (Mark Salter) [1905243]
- net: stmmac: Add support for external trigger timestamping (Mark Salter) [1905243]
- net: stmmac: fix dma physical address of descriptor when display ring (Mark Salter) [1905243]
- net: stmmac: fix wrongly set buffer2 valid when sph unsupport (Mark Salter) [1905243]
- net: stmmac: arrange Tx tail pointer update to stmmac_flush_tx_descriptors (Mark Salter) [1905243]
- net: stmmac: make SPH enable/disable to be configurable (Mark Salter) [1905243]
- net: stmmac: set IRQ affinity hint for multi MSI vectors (Mark Salter) [1905243]
- net: stmmac: remove unnecessary pci_enable_msi() call (Mark Salter) [1905243]
- stmmac: intel: use managed PCI function on probe and resume (Mark Salter) [1905243]
- net: stmmac: enable MTL ECC Error Address Status Over-ride by default (Mark Salter) [1905243]
- net: ethernet: stmicro: Remove duplicate struct declaration (Mark Salter) [1905243]
- net: stmmac: fix missing unlock on error in stmmac_suspend() (Mark Salter) [1905243]
- net: stmmac: Fix kernel panic due to NULL pointer dereference of fpe_cfg (Mark Salter) [1905243]
- net: stmmac: use interrupt mode INTM=1 for multi-MSI (Mark Salter) [1905243]
- stmmac: intel: add support for multi-vector msi and msi-x (Mark Salter) [1905243]
- net: stmmac: introduce MSI Interrupt routines for mac, safety, RX & TX (Mark Salter) [1905243]
- net: stmmac: make stmmac_interrupt() function more friendly to MSI (Mark Salter) [1905243]
- net: stmmac: introduce DMA interrupt status masking per traffic direction (Mark Salter) [1905243]
- net: phy: marvell10g: Add PHY loopback support (Mark Salter) [1905243]
- net: phy: add genphy_c45_loopback (Mark Salter) [1905243]
- net: phylink: Fix phylink_err() function name error in phylink_major_config (Mark Salter) [1905243]
- net: stmmac: support FPE link partner hand-shaking procedure (Mark Salter) [1905243]
- net: stmmac: Add hardware supported cross-timestamp (Mark Salter) [1905243]
- net: stmmac: platform: fix build error with !CONFIG_PM_SLEEP (Mark Salter) [1905243]
- stmmac: intel: Add PSE and PCH PTP clock source selection (Mark Salter) [1905243]
- net: stmmac: Add EST errors into ethtool statistic (Mark Salter) [1905243]
- net: stmmac: EST interrupts handling and error reporting (Mark Salter) [1905243]
- net: stmmac: add RX frame steering based on VLAN priority in tc flower (Mark Salter) [1905243]
- net: stmmac: restructure tc implementation for RX VLAN Priority steering (Mark Salter) [1905243]
- net: stmmac: add per-queue TX & RX coalesce ethtool support (Mark Salter) [1905243]
- net: stmmac: Use hrtimer for TX coalescing (Mark Salter) [1905243]
- net: stmmac: add timestamp correction to rid CDC sync error (Mark Salter) [1905243]
- net: stmmac: add platform level clocks management (Mark Salter) [1905243]
- net: stmmac: add clocks management for gmac driver (Mark Salter) [1905243]
- stmmac: intel: add pcs-xpcs for Intel mGbE controller (Mark Salter) [1905243]
- net: stmmac: ensure phydev is attached to phylink for C37 AN (Mark Salter) [1905243]
- net: stmmac: make in-band AN mode parsing is supported for non-DT (Mark Salter) [1905243]
- net: phylink: make phylink_parse_mode() support non-DT platform (Mark Salter) [1905243]
- net: pcs: add C37 SGMII AN support for intel mGbE controller (Mark Salter) [1905243]
- net: pcs: rearrange C73 functions to prepare for C37 support later (Mark Salter) [1905243]
- net: phy: leds: Deduplicate link LED trigger registration (Mark Salter) [1905243]
- net: stmmac: Fix VLAN filter delete timeout issue in Intel mGBE SGMII (Mark Salter) [1905243]
- net: stmmac: fix incorrect DMA channel intr enable setting of EQoS v4.10 (Mark Salter) [1905243]
- net: stmmac: fix CBS idleslope and sendslope calculation (Mark Salter) [1905243]
- net: stmmac: set TxQ mode back to DCB after disabling CBS (Mark Salter) [1905243]
- net: stmmac: allow stmmac to probe for C45 PHY devices (Mark Salter) [1905243]
- net: stmmac: Enable EEE HW LPI timer with auto SW/HW switching (Mark Salter) [1905243]
- net: pcs-xpcs: depend on MDIO_BUS instead of selecting it (Mark Salter) [1905243]
- net: pcs: Move XPCS into new PCS subdirectory (Mark Salter) [1905243]
- net: stmmac: Add XLGMII support (Mark Salter) [1905243]
- net: phy: xpcs: Add XLGMII support (Mark Salter) [1905243]
- net: phy: xpcs: Restart AutoNeg if outcome was invalid (Mark Salter) [1905243]
- net: phy: xpcs: Set Link down if AutoNeg is enabled and did not finish (Mark Salter) [1905243]
- net: phy: xpcs: Reset XPCS upon probe (Mark Salter) [1905243]
- net: phy: xpcs: Return error when 10GKR link errors are found (Mark Salter) [1905243]
- stmmac/intel: mark driver as tech preview (Mark Salter) [1905243]
- redhat/configs: Add CONFIG_DWMAC_INTEL (Mark Salter) [1905243]
- stmmac: intel: Fixes clock registration error seen for multiple interfaces (Mark Salter) [1905243]
- stmmac: intel: Fix mdio bus registration issue for TGL-H/ADL-S (Mark Salter) [1905243]
- net: stmmac: Add PCI bus info to ethtool driver query output (Mark Salter) [1905243]
- stmmac: intel: Add ADL-S 1Gbps PCI IDs (Mark Salter) [1905243]
- stmmac: intel: Configure EHL PSE0 GbE and PSE1 GbE to 32 bits DMA addressing (Mark Salter) [1905243]
- stmmac: intel: Add PCI IDs for TGL-H platform (Mark Salter) [1905243]
- stmmac: intel: change all EHL/TGL to auto detect phy addr (Mark Salter) [1905243]
- net: stmmac: overwrite the dma_cap.addr64 according to HW design (Mark Salter) [1905243]
- stmmac: intel: Fix kernel panic on pci probe (Mark Salter) [1905243]
- stmmac: intel: Adding ref clock 1us tic for LPI cntr (Mark Salter) [1905243]
- net: stmmac: Add option for VLAN filter fail queue enable (Mark Salter) [1905243]
- net: stmmac: Fix clock handling on remove path (Mark Salter) [1905243]
- stmmac: intel: Place object in the Makefile according to the order (Mark Salter) [1905243]
- stmmac: intel: Fix indentation to put on one line affected code (Mark Salter) [1905243]
- stmmac: intel: Eliminate useless conditions and variables (Mark Salter) [1905243]
- stmmac: intel: Convert to use pci_alloc_irq_vectors() API (Mark Salter) [1905243]
- stmmac: intel: Remove unnecessary loop for PCI BARs (Mark Salter) [1905243]
- stmmac: intel: Fix clock handling on error and remove paths (Mark Salter) [1905243]
- net: stmmac: Enable SERDES power up/down sequence (Mark Salter) [1905243]
- net: stmmac: add EHL 2.5Gbps PCI info and PCI ID (Mark Salter) [1905243]
- net: stmmac: add EHL PSE0 & PSE1 1Gbps PCI info and PCI ID (Mark Salter) [1905243]
- net: stmmac: create dwmac-intel.c to contain all Intel platform (Mark Salter) [1905243]
- net: stmmac: Get rid of custom STMMAC_DEVICE() macro (Mark Salter) [1905243]
- net: stmmac: update pci platform data to use phy_interface (Mark Salter) [1905243]
- net: stmmac: pci: Enable TBS on GMAC5 IPK PCI entry (Mark Salter) [1905243]
- net: stmmac: pci: remove the duplicate code of set phy_mask (Mark Salter) [1905243]
- PCI: Add PCI_STD_NUM_BARS for the number of standard BARs (Mark Salter) [1905243]
- net: stmmac: pci: Add HAPS support using GMAC5 (Mark Salter) [1905243]
- net: stmmac: setup higher frequency clk support for EHL & TGL (Mark Salter) [1905243]
- net: stmmac: add EHL RGMII 1Gbps PCI info and PCI ID (Mark Salter) [1905243]
- net: stmmac: add TGL SGMII 1Gbps PCI info and PCI ID (Mark Salter) [1905243]
- net: stmmac: add EHL SGMII 1Gbps PCI info and PCI ID (Mark Salter) [1905243]
- net: stmmac: Make MDIO bus reset optional (Mark Salter) [1905243]
- stmmac: pci: Use pci_dev_id() helper (Mark Salter) [1905243]
- stmmac: pci: Fix typo in IOT2000 comment (Mark Salter) [1905243]
- stmmac: pci: Adjust IOT2000 matching (Mark Salter) [1905243]
- net: stmmac: Fix PCI module removal leak (Mark Salter) [1905243]
- net: stmmac: mark PM functions as __maybe_unused (Mark Salter) [1905243]
- ibmveth: Set CHECKSUM_PARTIAL if NULL TCP CSUM. (Diego Domingos) [1939996]
- IB/core: Add missing pin_user_pages*() usage (Waiman Long) [1972078]
- mm/gup/writeback: Fix incorrect inaccessible pages handling (Waiman Long) [1972078]
- netfilter: nf_tables: bogus EBUSY when deleting flowtable after flush (Phil Sutter) [1983245]
- ipv4: Fix device used for dst_alloc with local routes (Antoine Tenart) [1976789]
- vrf: fix a comment about loopback device (Antoine Tenart) [1976789]
- net: vrf: Fix operation not supported when set vrf mac (Antoine Tenart) [1976789]
- vrf: prevent adding upper devices (Antoine Tenart) [1976789]
- vrf: fix maximum MTU (Antoine Tenart) [1976789]
- net: vrf: Fix ping failed when vrf mtu is set to 0 (Antoine Tenart) [1976789]
- net: vrf: remove MTU limits for vrf device (Antoine Tenart) [1976789]

* Fri Jul 30 2021 Augusto Caringi <acaringi@redhat.com> [4.18.0-327.el8]
- block: fix discard request merge (Ming Lei) [1970316]
- blk-mq: update hctx->dispatch_busy in case of real scheduler (Ming Lei) [1976679]
- block: Do not pull requests from the scheduler when we cannot dispatch them (Ming Lei) [1976679]
- powerpc/pseries/mobility: handle premature return from H_JOIN (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: use struct for shared state (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: refactor node lookup during DT update (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtas: remove rtas_ibm_suspend_me_unsafe() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtas: dispatch partition migration requests to pseries (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: retry partition suspend after error (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: signal suspend cancellation to platform (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: use stop_machine for join/suspend (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: extract VASI session polling logic (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: use rtas_activate_firmware() on resume (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: error message improvements (Desnes A. Nunes do Rosario) [1979798]
- powerpc/hvcall: add token and codes for H_VASI_SIGNAL (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtas: add rtas_activate_firmware() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtas: add rtas_ibm_suspend_me() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtas: rtas_ibm_suspend_me -> rtas_ibm_suspend_me_unsafe (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtas: complete ibm,suspend-me status codes (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: Add pr_debug() for device tree changes (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries/mobility: Set pr_fmt() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries: remove obsolete memory hotplug DT notifier code (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries: remove dlpar_cpu_readd() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries: remove memory "re-add" implementation (Desnes A. Nunes do Rosario) [1979798]
- powerpc/pseries: remove prrn special case from DT update path (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove arch_update_cpu_topology (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove prrn_is_enabled() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtasd: simplify handle_rtas_event(), emit message on events (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove start/stop_topology_update() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove timed_topology_update() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: Handle extra hcall_vphn error cases (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: Suppress "VPHN is not supported" messages (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: stub out numa_update_cpu_topology() (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove vphn_enabled and prrn_enabled internal flags (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove unreachable topology workqueue code (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove unreachable topology timer code (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: make vphn_enabled, prrn_enabled flags const (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove unreachable topology update code (Desnes A. Nunes do Rosario) [1979798]
- powerpc/numa: remove ability to enable topology updates (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtas: don't online CPUs for partition suspend (Desnes A. Nunes do Rosario) [1979798]
- powerpc/rtas: Unexport rtas_online_cpus_mask, rtas_offline_cpus_mask (Desnes A. Nunes do Rosario) [1979798]
- mm/page_alloc: bail out on fatal signal during reclaim/compaction retry attempt (Aaron Tomlin) [1919765]
- RHEL: Revert "powerpc/kvm: Fix PR KVM with KUAP/MEM_KEYS enabled" (Paolo Bonzini) [1956950]
- KVM: selftests: smm_test: Test SMM enter from L2 (Paolo Bonzini) [1956950]
- KVM: nSVM: Restore nested control upon leaving SMM (Paolo Bonzini) [1956950]
- KVM: nSVM: Fix L1 state corruption upon return from SMM (Paolo Bonzini) [1956950]
- KVM: nSVM: Introduce svm_copy_vmrun_state() (Paolo Bonzini) [1956950]
- KVM: nSVM: Check that VM_HSAVE_PA MSR was set before VMRUN (Paolo Bonzini) [1956950]
- KVM: nVMX: nSVM: 'nested_run' should count guest-entry attempts that make it to guest code (Paolo Bonzini) [1956950]
- KVM: nSVM: Check the value written to MSR_VM_HSAVE_PA (Paolo Bonzini) [1956950]
- KVM: SVM: Fix sev_pin_memory() error checks in SEV migration utilities (Paolo Bonzini) [1956950]
- KVM: SVM: Return -EFAULT if copy_to_user() for SEV mig packet header fails (Paolo Bonzini) [1956950]
- KVM: SVM: add module param to control the #SMI interception (Paolo Bonzini) [1956950]
- KVM: SVM: remove INIT intercept handler (Paolo Bonzini) [1956950]
- KVM: SVM: #SMI interception must not skip the instruction (Paolo Bonzini) [1956950]
- KVM: VMX: Remove vmx_msr_index from vmx.h (Paolo Bonzini) [1956950]
- KVM: X86: Disable hardware breakpoints unconditionally before kvm_x86->run() (Paolo Bonzini) [1956950]
- KVM: x86/pmu: Clear anythread deprecated bit when 0xa leaf is unsupported on the SVM (Paolo Bonzini) [1956950]
- KVM: mmio: Fix use-after-free Read in kvm_vm_ioctl_unregister_coalesced_mmio (Paolo Bonzini) [1956950]
- KVM: SVM: Revert clearing of C-bit on GPA in #NPF handler (Paolo Bonzini) [1956950]
- KVM: x86/mmu: Do not apply HPA (memory encryption) mask to GPAs (Paolo Bonzini) [1956950]
- KVM: x86: Use kernel's x86_phys_bits to handle reduced MAXPHYADDR (Paolo Bonzini) [1956950]
- KVM: x86: Use guest MAXPHYADDR from CPUID.0x8000_0008 iff TDP is enabled (Paolo Bonzini) [1956950]
- KVM: selftests: do not require 64GB in set_memory_region_test (Paolo Bonzini) [1956950]
- KVM: selftests: Fix mapping length truncation in m{,un}map() (Paolo Bonzini) [1956950]
- KVM: x86: Force all MMUs to reinitialize if guest CPUID is modified (Paolo Bonzini) [1956950]
- Revert "KVM: x86/mmu: Drop kvm_mmu_extended_role.cr4_la57 hack" (Paolo Bonzini) [1956950]
- KVM: x86/mmu: Use MMU's role to detect CR4.SMEP value in nested NPT walk (Paolo Bonzini) [1956950]
- KVM: x86: Properly reset MMU context at vCPU RESET/INIT (Paolo Bonzini) [1956950]
- KVM: x86/mmu: Treat NX as used (not reserved) for all !TDP shadow MMUs (Paolo Bonzini) [1956950]
- KVM: x86/mmu: Remove broken WARN that fires on 32-bit KVM w/ nested EPT (Paolo Bonzini) [1956950]
- KVM: nVMX: Handle split-lock #AC exceptions that happen in L2 (Paolo Bonzini) [1956950]
- KVM: x86/mmu: Calculate and check "full" mmu_role for nested MMU (Paolo Bonzini) [1956950]
- KVM: X86: Fix x86_emulator slab cache leak (Paolo Bonzini) [1956950]
- KVM: SVM: Call SEV Guest Decommission if ASID binding fails (Paolo Bonzini) [1956950]
- KVM: x86: Immediately reset the MMU context when the SMM flag is cleared (Paolo Bonzini) [1956950]
- KVM: x86: Fix fall-through warnings for Clang (Paolo Bonzini) [1956950]
- KVM: SVM: fix doc warnings (Paolo Bonzini) [1956950]
- kvm: LAPIC: Restore guard to prevent illegal APIC register access (Paolo Bonzini) [1956950]
- kvm: fix previous commit for 32-bit builds (Paolo Bonzini) [1956950]
- kvm: avoid speculation-based attacks from out-of-range memslot accesses (Paolo Bonzini) [1956950]
- KVM: x86: Ensure liveliness of nested VM-Enter fail tracepoint message (Paolo Bonzini) [1956950]
- KVM: x86: Ensure PV TLB flush tracepoint reflects KVM behavior (Paolo Bonzini) [1956950]
- KVM: X86: MMU: Use the correct inherited permissions to get shadow page (Paolo Bonzini) [1956950]
- KVM: LAPIC: Write 0 to TMICT should also cancel vmx-preemption timer (Paolo Bonzini) [1956950]
- KVM: SVM: Fix SEV SEND_START session length & SEND_UPDATE_DATA query length after commit 238eca821cee (Paolo Bonzini) [1956950]
- KVM: X86: Kill off ctxt->ud (Paolo Bonzini) [1956950]
- KVM: X86: Fix warning caused by stale emulation context (Paolo Bonzini) [1956950]
- KVM: X86: Use kvm_get_linear_rip() in single-step and #DB/#BP interception (Paolo Bonzini) [1956950]
- KVM: x86/mmu: Fix comment mentioning skip_4k (Paolo Bonzini) [1956950]
- KVM: VMX: update vcpu posted-interrupt descriptor when assigning device (Paolo Bonzini) [1956950]
- KVM: rename KVM_REQ_PENDING_TIMER to KVM_REQ_UNBLOCK (Paolo Bonzini) [1956950]
- KVM: x86: add start_assignment hook to kvm_x86_ops (Paolo Bonzini) [1956950]
- KVM: LAPIC: Narrow the timer latency between wait_lapic_expire and world switch (Paolo Bonzini) [1956950]
- KVM: X86: Fix vCPU preempted state from guest's point of view (Paolo Bonzini) [1956950]
- KVM: X86: Bail out of direct yield in case of under-committed scenarios (Paolo Bonzini) [1956950]
- KVM: SVM: make the avic parameter a bool (Paolo Bonzini) [1956950]
- KVM: VMX: Drop unneeded CONFIG_X86_LOCAL_APIC check (Paolo Bonzini) [1956950]
- KVM: SVM: Drop unneeded CONFIG_X86_LOCAL_APIC check (Paolo Bonzini) [1956950]
- KVM: SVM: Move GHCB unmapping to fix RCU warning (Paolo Bonzini) [1956950]
- KVM: SVM: Invert user pointer casting in SEV {en,de}crypt helpers (Paolo Bonzini) [1956950]
- kvm: Cap halt polling at kvm->max_halt_poll_ns (Paolo Bonzini) [1956950]
- KVM: x86: Prevent deadlock against tk_core.seq (Paolo Bonzini) [1956950]
- KVM: x86: Cancel pvclock_gtod_work on module removal (Paolo Bonzini) [1956950]
- KVM: x86: Prevent KVM SVM from loading on kernels with 5-level paging (Paolo Bonzini) [1956950]
- KVM: SVM: Probe and load MSR_TSC_AUX regardless of RDTSCP support in host (Paolo Bonzini) [1956950]
- KVM: VMX: Disable preemption when probing user return MSRs (Paolo Bonzini) [1956950]
- KVM: x86: Move RDPID emulation intercept to its own enum (Paolo Bonzini) [1956950]
- KVM: SVM: Inject #UD on RDTSCP when it should be disabled in the guest (Paolo Bonzini) [1956950]
- KVM: x86: Emulate RDPID only if RDTSCP is supported (Paolo Bonzini) [1956950]
- KVM: VMX: Do not advertise RDPID if ENABLE_RDTSCP control is unsupported (Paolo Bonzini) [1956950]
- KVM: nSVM: always restore the L1's GIF on migration (Paolo Bonzini) [1956950]
- KVM: x86: Hoist input checks in kvm_add_msr_filter() (Paolo Bonzini) [1956950]
- KVM: nVMX: Properly pad 'struct kvm_vmx_nested_state_hdr' (Paolo Bonzini) [1956950]
- KVM/VMX: Invoke NMI non-IST entry instead of IST entry (Paolo Bonzini) [1956950]
- KVM: x86: Fix potential fput on a null source_kvm_file (Paolo Bonzini) [1956950]
- KVM: nSVM: leave the guest mode prior to loading a nested state (Paolo Bonzini) [1956950]
- KVM: nSVM: fix few bugs in the vmcb02 caching logic (Paolo Bonzini) [1956950]
- KVM: nSVM: fix a typo in svm_leave_nested (Paolo Bonzini) [1956950]
- KVM: PPC: exit halt polling on need_resched() (Paolo Bonzini) [1956950]
- kvm: exit halt polling on need_resched() as well (Paolo Bonzini) [1956950]
- KVM: LAPIC: Accurately guarantee busy wait for timer to expire when using hv_timer (Paolo Bonzini) [1956950]
- kvm/x86: Fix 'lpages' kvm stat for TDM MMU (Paolo Bonzini) [1956950]
- KVM: x86/mmu: Avoid unnecessary page table allocation in kvm_tdp_mmu_map() (Paolo Bonzini) [1956950]
- KVM: selftests: Speed up set_memory_region_test (Paolo Bonzini) [1956950]
- selftests: kvm: Fix the check of return value (Paolo Bonzini) [1956950]
- KVM: x86: Take advantage of kvm_arch_dy_has_pending_interrupt() (Paolo Bonzini) [1956950]
- KVM: SVM: Skip SEV cache flush if no ASIDs have been used (Paolo Bonzini) [1956950]
- KVM: SVM: Remove an unnecessary prototype declaration of sev_flush_asids() (Paolo Bonzini) [1956950]
- KVM: SVM: Drop redundant svm_sev_enabled() helper (Paolo Bonzini) [1956950]
- KVM: SVM: Move SEV VMCB tracking allocation to sev.c (Paolo Bonzini) [1956950]
- KVM: SVM: Explicitly check max SEV ASID during sev_hardware_setup() (Paolo Bonzini) [1956950]
- KVM: SVM: Unconditionally invoke sev_hardware_teardown() (Paolo Bonzini) [1956950]
- KVM: SVM: Enable SEV/SEV-ES functionality by default (when supported) (Paolo Bonzini) [1956950]
- KVM: SVM: Condition sev_enabled and sev_es_enabled on CONFIG_KVM_AMD_SEV=y (Paolo Bonzini) [1956950]
- KVM: SVM: Append "_enabled" to module-scoped SEV/SEV-ES control variables (Paolo Bonzini) [1956950]
- KVM: SEV: Mask CPUID[0x8000001F].eax according to supported features (Paolo Bonzini) [1956950]
- KVM: SVM: Move SEV module params/variables to sev.c (Paolo Bonzini) [1956950]
- svm/sev: Register SEV and SEV-ES ASIDs to the misc controller (Paolo Bonzini) [1956950]
- KVM: SVM: Disable SEV/SEV-ES if NPT is disabled (Paolo Bonzini) [1956950]
- KVM: SVM: Free sev_asid_bitmap during init if SEV setup fails (Paolo Bonzini) [1956950]
- KVM: SVM: Zero out the VMCB array used to track SEV ASID association (Paolo Bonzini) [1956950]
- KVM: x86: Move reverse CPUID helpers to separate header file (Paolo Bonzini) [1956950]
- KVM: x86: Rename GPR accessors to make mode-aware variants the defaults (Paolo Bonzini) [1956950]
- KVM: SVM: Use default rAX size for INVLPGA emulation (Paolo Bonzini) [1956950]
- KVM: x86/xen: Drop RAX[63:32] when processing hypercall (Paolo Bonzini) [1956950]
- KVM: nVMX: Truncate base/index GPR value on address calc in !64-bit (Paolo Bonzini) [1956950]
- KVM: nVMX: Truncate bits 63:32 of VMCS field on nested check in !64-bit (Paolo Bonzini) [1956950]
- KVM: VMX: Truncate GPR value for DR and CR reads in !64-bit mode (Paolo Bonzini) [1956950]
- KVM: SVM: Truncate GPR value for DR and CR accesses in !64-bit mode (Paolo Bonzini) [1956950]
- KVM: x86: Check CR3 GPA for validity regardless of vCPU mode (Paolo Bonzini) [1956950]
- KVM: x86: Remove emulator's broken checks on CR0/CR3/CR4 loads (Paolo Bonzini) [1956950]
- KVM: VMX: Intercept FS/GS_BASE MSR accesses for 32-bit KVM (Paolo Bonzini) [1956950]
- KVM: SVM: Delay restoration of host MSR_TSC_AUX until return to userspace (Paolo Bonzini) [1956950]
- KVM: SVM: Clear MSR_TSC_AUX[63:32] on write (Paolo Bonzini) [1956950]
- KVM: SVM: Inject #GP on guest MSR_TSC_AUX accesses if RDTSCP unsupported (Paolo Bonzini) [1956950]
- KVM: VMX: Invert the inlining of MSR interception helpers (Paolo Bonzini) [1956950]
- KVM: X86: Fix failure to boost kernel lock holder candidate in SEV-ES guests (Paolo Bonzini) [1956950]
- KVM: x86: Properly handle APF vs disabled LAPIC situation (Paolo Bonzini) [1956950]
- KVM: VMX: use EPT_VIOLATION_GVA_TRANSLATED instead of 0x100 (Paolo Bonzini) [1956950]
- KVM: x86: Support KVM VMs sharing SEV context (Paolo Bonzini) [1956950]
- KVM: x86: Remove unused function declaration (Paolo Bonzini) [1956950]
- KVM: x86/mmu: Drop trace_kvm_age_page() tracepoint (Paolo Bonzini) [1956950]
- KVM: Move prototypes for MMU notifier callbacks to generic code (Paolo Bonzini) [1956950]
- KVM: Pass MMU notifier range flags to kvm_unmap_hva_range() (Paolo Bonzini) [1956950]
- KVM: x86/mmu: preserve pending TLB flush across calls to kvm_tdp_mmu_zap_sp (Paolo Bonzini) [1956950]
- KVM: x86: disable interrupts while pvclock_gtod_sync_lock is taken (Paolo Bonzini) [1956950]
- KVM: x86: reduce pvclock_gtod_sync_lock critical sections (Paolo Bonzini) [1956950]
- kvm: x86: annotate RCU pointers (Paolo Bonzini) [1956950]
- KVM: x86: remove misplaced comment on active_mmu_pages (Paolo Bonzini) [1956950]
- x86/sev: Drop redundant and potentially misleading 'sev_enabled' (Paolo Bonzini) [1956950]
- x86/cpufeatures: Assign dedicated feature word for CPUID_0x8000001F[EAX] (Paolo Bonzini) [1956950]
- cgroup: Add misc cgroup controller (Paolo Bonzini) [1956950]
- x86: Remove duplicate TSC DEADLINE MSR definitions (Paolo Bonzini) [1956950]
- KVM: s390: allow facility 192 (vector-packed-decimal-enhancement facility 2) (Thomas Huth) [1977289]
- KVM: s390: gen_facilities: allow facilities 165, 193, 194 and 196 (Thomas Huth) [1977289]
- mpi3mr: mark as tech preview (Tomas Henzl) [1858804]
- add the mpi3mr config (Tomas Henzl) [1858804]
- scsi: mpi3mr: Fix W=1 compilation warnings (Tomas Henzl) [1858804]
- scsi: mpi3mr: Fix warnings reported by smatch (Tomas Henzl) [1858804]
- scsi: mpi3mr: Make some symbols static (Tomas Henzl) [1858804]
- scsi: mpi3mr: Fix error return code in mpi3mr_init_ioc() (Tomas Henzl) [1858804]
- scsi: mpi3mr: Fix missing unlock on error (Tomas Henzl) [1858804]
- scsi: mpi3mr: Fix error handling in mpi3mr_setup_isr() (Tomas Henzl) [1858804]
- scsi: mpi3mr: Delete unnecessary NULL check (Tomas Henzl) [1858804]
- scsi: mpi3mr: Fix a double free (Tomas Henzl) [1858804]
- scsi: mpi3mr: Fix fall-through warning for Clang (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add event handling debug prints (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add EEDP DIF DIX support (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for DSN secure firmware check (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for PM suspend and resume (Tomas Henzl) [1858804]
- scsi: mpi3mr: Wait for pending I/O completions upon detection of VD I/O timeout (Tomas Henzl) [1858804]
- scsi: mpi3mr: Print pending host I/Os for debugging (Tomas Henzl) [1858804]
- scsi: mpi3mr: Complete support for soft reset (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for threaded ISR (Tomas Henzl) [1858804]
- scsi: mpi3mr: Hardware workaround for UNMAP commands to NVMe drives (Tomas Henzl) [1858804]
- scsi: mpi3mr: Allow certain commands during pci-remove hook (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add change queue depth support (Tomas Henzl) [1858804]
- scsi: mpi3mr: Implement SCSI error handler hooks (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add bios_param SCSI host template hook (Tomas Henzl) [1858804]
- scsi: mpi3mr: Print IOC info for debugging (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for timestamp sync with firmware (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for recovering controller (Tomas Henzl) [1858804]
- scsi: mpi3mr: Additional event handling (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for PCIe device event handling (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for device add/remove event handling (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for internal watchdog thread (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add support for queue command processing (Tomas Henzl) [1858804]
- scsi: mpi3mr: Create operational request and reply queue pair (Tomas Henzl) [1858804]
- scsi: mpi3mr: Base driver code (Tomas Henzl) [1858804]
- scsi: mpi3mr: Add mpi30 Rev-R headers and Kconfig (Tomas Henzl) [1858804]
- e1000e: Fix prototype warning (Ken Cox) [1920290]
- e1000e: Mark e1000e_pm_prepare() as __maybe_unused (Ken Cox) [1920290]
- e1000e: Fix duplicate include guard (Ken Cox) [1920290]
- e1000e: Remove the runtime suspend restriction on CNP+ (Ken Cox) [1920290]
- e1000e: Leverage direct_complete to speed up s2ram (Ken Cox) [1920290]
- e1000e: Fix error handling in e1000_set_d0_lplu_state_82571 (Ken Cox) [1920290]
- e1000e: add rtnl_lock() to e1000_reset_task (Ken Cox) [1920290]
- e1000e: remove the redundant value assignment in e1000_update_nvm_checksum_spt (Ken Cox) [1920290]
- e1000e: Export S0ix flags to ethtool (Ken Cox) [1920290]
- e1000e: bump up timeout to wait when ME un-configures ULP mode (Ken Cox) [1920290]
- e1000e: Only run S0ix flows if shutdown succeeded (Ken Cox) [1920290]
- scsi: ipr: System crashes when seeing type 20 error (Desnes A. Nunes do Rosario) [1977532]
- scsi: ipr: Use scnprintf() for avoiding potential buffer overflow (Desnes A. Nunes do Rosario) [1977532]
- tipc: do not write skb_shinfo frags when doing decrytion (Xin Long) [1974627]
- bnxt_en: Call bnxt_ethtool_free() in bnxt_init_one() error path (Jonathan Toppins) [1921449]
- bnxt_en: Fix TQM fastpath ring backing store computation (Jonathan Toppins) [1921449]
- bnxt_en: Rediscover PHY capabilities after firmware reset (Jonathan Toppins) [1921449]
- bnxt_en: Fix context memory setup for 64K page size. (Jonathan Toppins) [1921449]
- bnxt_en: Include new P5 HV definition in VF check. (Jonathan Toppins) [1921449]
- bnxt_en: Add PCI IDs for Hyper-V VF devices. (Jonathan Toppins) [1921449]
- bnxt_en: Fix RX consumer index logic in the error path. (Jonathan Toppins) [1921449]
- bnxt_en: fix ternary sign extension bug in bnxt_show_temp() (Jonathan Toppins) [1921449]
- bnxt_en: reliably allocate IRQ table on reset to avoid crash (Jonathan Toppins) [1921449]
- bnxt_en: Fix race between firmware reset and driver remove. (Jonathan Toppins) [1921449]
- bnxt_en: Improve logging of error recovery settings information. (Jonathan Toppins) [1921449]
- bnxt_en: Reply to firmware's echo request async message. (Jonathan Toppins) [1921449]
- bnxt_en: Initialize "context kind" field for context memory blocks. (Jonathan Toppins) [1921449]
- bnxt_en: Add context memory initialization infrastructure. (Jonathan Toppins) [1921449]
- bnxt_en: Implement faster recovery for firmware fatal error. (Jonathan Toppins) [1921449]
- bnxt_en: selectively allocate context memories (Jonathan Toppins) [1921449]
- bnxt_en: Update firmware interface spec to 1.10.2.16. (Jonathan Toppins) [1921449]
- bnxt_en: Fix devlink info's stored fw.psid version format. (Jonathan Toppins) [1921449]
- bnxt_en: reverse order of TX disable and carrier off (Jonathan Toppins) [1921449]
- bnxt_en: Convert to use netif_level() helpers. (Jonathan Toppins) [1921449]
- bnxt_en: Do not process completion entries after fatal condition detected. (Jonathan Toppins) [1921449]
- bnxt_en: Consolidate firmware reset event logging. (Jonathan Toppins) [1921449]
- bnxt_en: Improve firmware fatal error shutdown sequence. (Jonathan Toppins) [1921449]
- bnxt_en: Modify bnxt_disable_int_sync() to be called more than once. (Jonathan Toppins) [1921449]
- bnxt_en: Add a new BNXT_STATE_NAPI_DISABLED flag to keep track of NAPI state. (Jonathan Toppins) [1921449]
- bnxt_en: Add bnxt_fw_reset_timeout() helper. (Jonathan Toppins) [1921449]
- bnxt_en: Retry open if firmware is in reset. (Jonathan Toppins) [1921449]
- bnxt_en: attempt to reinitialize after aborted reset (Jonathan Toppins) [1921449]
- bnxt_en: log firmware debug notifications (Jonathan Toppins) [1921449]
- bnxt_en: Add an upper bound for all firmware command timeouts. (Jonathan Toppins) [1921449]
- bnxt_en: Move reading VPD info after successful handshake with fw. (Jonathan Toppins) [1921449]
- bnxt_en: Retry sending the first message to firmware if it is under reset. (Jonathan Toppins) [1921449]
- bnxt_en: handle CRASH_NO_MASTER during bnxt_open() (Jonathan Toppins) [1921449]
- bnxt_en: Define macros for the various health register states. (Jonathan Toppins) [1921449]
- bnxt_en: Update firmware interface to 1.10.2.11. (Jonathan Toppins) [1921449]
- bnxt_en: Clear DEFRAG flag in firmware message when retry flashing. (Jonathan Toppins) [1921449]
- bnxt_en: Improve stats context resource accounting with RDMA driver loaded. (Jonathan Toppins) [1921449]
- bnxt_en: Use kzalloc for allocating only one thing (Jonathan Toppins) [1921449]
- bnxt_en: Check TQM rings for maximum supported value. (Jonathan Toppins) [1921449]
- bnxt_en: Fix AER recovery. (Jonathan Toppins) [1921449]
- bnxt_en: Enable batch mode when using HWRM_NVM_MODIFY to flash packages. (Jonathan Toppins) [1921449]
- bnxt_en: Retry installing FW package under NO_SPACE error condition. (Jonathan Toppins) [1921449]
- bnxt_en: Restructure bnxt_flash_package_from_fw_obj() to execute in a loop. (Jonathan Toppins) [1921449]
- bnxt_en: Rearrange the logic in bnxt_flash_package_from_fw_obj(). (Jonathan Toppins) [1921449]
- bnxt_en: Refactor bnxt_flash_nvram. (Jonathan Toppins) [1921449]
- bnxt_en: Release PCI regions when DMA mask setup fails during probe. (Jonathan Toppins) [1921449]
- bnxt_en: fix error return code in bnxt_init_board() (Jonathan Toppins) [1921449]
- bnxt_en: fix error return code in bnxt_init_one() (Jonathan Toppins) [1921449]
- bnxt_en: Avoid unnecessary NVM_GET_DEV_INFO cmd error log on VFs. (Jonathan Toppins) [1921449]
- bnxt_en: Fix counter overflow logic. (Jonathan Toppins) [1921449]
- bnxt_en: Free port stats during firmware reset. (Jonathan Toppins) [1921449]
- bnxt_en: read EEPROM A2h address using page 0 (Jonathan Toppins) [1921449]
- bnxt_en: Send HWRM_FUNC_RESET fw command unconditionally. (Jonathan Toppins) [1921449]
- bnxt_en: Check abort error state in bnxt_open_nic(). (Jonathan Toppins) [1921449]
- bnxt_en: Re-write PCI BARs after PCI fatal error. (Jonathan Toppins) [1921449]
- bnxt_en: Invoke cancel_delayed_work_sync() for PFs also. (Jonathan Toppins) [1921449]
- bnxt_en: Fix regression in workqueue cleanup logic in bnxt_remove_one(). (Jonathan Toppins) [1921449]
- bnxt_en: Add stored FW version info to devlink info_get cb. (Jonathan Toppins) [1921449]
- bnxt_en: Refactor bnxt_dl_info_get(). (Jonathan Toppins) [1921449]
- bnxt_en: Add bnxt_hwrm_nvm_get_dev_info() to query NVM info. (Jonathan Toppins) [1921449]
- bnxt_en: Log unknown link speed appropriately. (Jonathan Toppins) [1921449]
- bnxt_en: Log event_data1 and event_data2 when handling RESET_NOTIFY event. (Jonathan Toppins) [1921449]
- bnxt_en: Simplify bnxt_async_event_process(). (Jonathan Toppins) [1921449]
- bnxt_en: Set driver default message level. (Jonathan Toppins) [1921449]
- bnxt_en: Enable online self tests for multi-host/NPAR mode. (Jonathan Toppins) [1921449]
- bnxt_en: Return -EROFS to user space, if NVM writes are not permitted. (Jonathan Toppins) [1921449]
- bnxt_en: Eliminate unnecessary RX resets. (Jonathan Toppins) [1921449]
- bnxt_en: Reduce unnecessary message log during RX errors. (Jonathan Toppins) [1921449]
- bnxt_en: Add a software counter for RX ring reset. (Jonathan Toppins) [1921449]
- bnxt_en: Implement RX ring reset in response to buffer errors. (Jonathan Toppins) [1921449]
- bnxt_en: Refactor bnxt_init_one_rx_ring(). (Jonathan Toppins) [1921449]
- bnxt_en: Refactor bnxt_free_rx_skbs(). (Jonathan Toppins) [1921449]
- bnxt_en: Log FW health status info, if reset is aborted. (Jonathan Toppins) [1921449]
- bnxt_en: perform no master recovery during startup (Jonathan Toppins) [1921449]
- bnxt_en: log firmware status on firmware init failure (Jonathan Toppins) [1921449]
- bnxt_en: refactor bnxt_alloc_fw_health() (Jonathan Toppins) [1921449]
- bnxt_en: Update firmware interface spec to 1.10.1.68. (Jonathan Toppins) [1921449]
- bnxt_en: Improve preset max value for ethtool -l. (Jonathan Toppins) [1921449]
- bnxt_en: Implement ethtool set_fec_param() method. (Jonathan Toppins) [1921449]
- bnxt_en: Report Active FEC encoding during link up. (Jonathan Toppins) [1921449]
- bnxt_en: Report FEC settings to ethtool. (Jonathan Toppins) [1921449]
- bnxt_en: avoid link reset if speed is not changed (Jonathan Toppins) [1921449]
- bnxt_en: Handle ethernet link being disabled by firmware. (Jonathan Toppins) [1921449]
- bnxt_en: ethtool: support PAM4 link speeds up to 200G (Jonathan Toppins) [1921449]
- bnxt_en: add basic infrastructure to support PAM4 link speeds (Jonathan Toppins) [1921449]
- bnxt_en: refactor bnxt_get_fw_speed() (Jonathan Toppins) [1921449]
- bnxt_en: refactor code to limit speed advertising (Jonathan Toppins) [1921449]
- bnxt_en: Update firmware interface spec to 1.10.1.65. (Jonathan Toppins) [1921449]
- bnxt: add pause frame stats (Jonathan Toppins) [1921449]
- xfrm: Fix xfrm offload fallback fail case (Andrea Claudi) [1976424]
- sctp: trim optlen when it's a huge value in sctp_setsockopt (Xin Long) [1983864]
- netfilter: flowtable: Set offload timeouts according to proto values (Phil Sutter) [1979184]
- netfilter: conntrack: Introduce udp offload timeout configuration (Phil Sutter) [1979184]
- netfilter: conntrack: Introduce tcp offload timeout configuration (Phil Sutter) [1979184]
- veth: check for NAPI instead of xdp_prog before xmit of XDP frame (Paolo Abeni) [1952835]
- self-tests: add veth tests (Paolo Abeni) [1952835]
- veth: refine napi usage (Paolo Abeni) [1952835]
- veth: allow enabling NAPI even without XDP (Paolo Abeni) [1952835]
- veth: use skb_orphan_partial instead of skb_orphan (Paolo Abeni) [1952835]
- net: really orphan skbs tied to closing sk (Paolo Abeni) [1952835]
- net: let skb_orphan_partial wake-up waiters. (Paolo Abeni) [1952835]
- veth: Store queue_mapping independently of XDP prog presence (Paolo Abeni) [1952835]
- net/xfrm: Add inner_ipproto into sec_path (Petr Oros) [1981429]
- net, xdp: Introduce xdp_prepare_buff utility routine (Jiri Benc) [1983730]
- net, xdp: Introduce xdp_init_buff utility routine (Jiri Benc) [1983730]
- bridge: switchdev: Notify about VLAN protocol changes (Ivan Vecera) [1981461]
- net: bridge: br_vlan_get_pvid_rcu() should dereference the VLAN group under RCU (Ivan Vecera) [1981461]
- bridge: add br_vlan_get_proto() (Ivan Vecera) [1981461]
- bridge: add br_vlan_get_pvid_rcu() (Ivan Vecera) [1981461]
- RDMA/bnxt_re: Fix a double free in bnxt_qplib_alloc_res (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Get rid of custom module reference counting (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Create direct symbol link between bnxt modules (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Depend on bnxt ethernet driver and not blindly select it (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Fix Kconfig indentation (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Fix error return code in bnxt_qplib_cq_process_terminal() (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Move device to error state upon device crash (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Allow bigger MR creation (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Code refactor while populating user MRs (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Fix max_qp_wrs reported (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Fix entry size during SRQ create (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Fix sizeof mismatch for allocation of pbl_tbl. (Selvin Xavier) [1921454 1923026]
- RDMA/bnxt_re: Use rdma_umem_for_each_dma_block() (Selvin Xavier) [1921454 1923026]
- [s390] tools/kvm_stat: Exempt time-based counters (Claudio Imbrenda) [1981802]
- Documentation/admin-guide: intel_pstate: drop doubled word (David Arcari) [1973225]
- Documentation: fix multiple typos found in the admin-guide subdirectory (David Arcari) [1973225]
- cpufreq: intel_pstate: Fix active mode setting from command line (David Arcari) [1973225]
- cpufreq: intel_pstate: Use passive mode by default without HWP (David Arcari) [1973225]
- cpufreq: intel_pstate: Use HWP if enabled by platform firmware (David Arcari) [1973225]
- cpufreq: intel_pstate: Simplify intel_pstate_update_perf_limits() (David Arcari) [1973225]
- cpufreq: intel_pstate: Clean up frequency computations (David Arcari) [1973225]
- cpufreq/intel_pstate: Fix wrong macro conversion (David Arcari) [1973225]
- cpufreq: Convert to new X86 CPU match macros (David Arcari) [1973225]
- treewide: Replace GPLv2 boilerplate/reference with SPDX - rule 441 (David Arcari) [1973225]
- selftests/fpu: Fix debugfs_simple_attr.cocci warning (Vitaly Kuznetsov) [1919182]
- selftest/fpu: avoid clang warning (Vitaly Kuznetsov) [1919182]
- redhat/configs: Add Enable CONFIG_TEST_FPU (Vitaly Kuznetsov) [1919182]
- selftests/fpu: Add an FPU selftest (Vitaly Kuznetsov) [1919182]
- x86/fpu/64: Don't FNINIT in kernel_fpu_begin() (Vitaly Kuznetsov) [1919182]
- x86/fpu: Make the EFI FPU calling convention explicit (Vitaly Kuznetsov) [1919182]
- x86/mmx: Use KFPU_387 for MMX string operations (Vitaly Kuznetsov) [1919182]
- x86/fpu: Add kernel_fpu_begin_mask() to selectively initialize state (Vitaly Kuznetsov) [1919182]
- x86/fpu: Simplify kernel_fpu_begin() (Vitaly Kuznetsov) [1919182]
- x86/fpu: Simplify kernel_fpu_end() (Vitaly Kuznetsov) [1919182]
- x86/fpu: Don't use current->mm to check for a kthread (Vitaly Kuznetsov) [1919182]
- x86/fpu: Don't export __kernel_fpu_{begin,end}() (Vitaly Kuznetsov) [1919182]
- intel_th: pci: Add Alder Lake-P support (Tony Camuso) [1929493 1929494]

* Tue Jul 27 2021 Augusto Caringi <acaringi@redhat.com> [4.18.0-326.el8]
- can: bcm: delay release of struct bcm_op after synchronize_rcu() (Hangbin Liu) [1975059] {CVE-2021-3609}
- seq_file: Disallow extremely large seq buffer allocations (Ian Kent) [1975182]
- nvme-pci: look for StorageD3Enable on companion ACPI device instead (Gopal Tiwari) [1965660]
- prctl: exe link permission error changed from -EINVAL to -EPERM (Adrian Reber) [1960708]
- prctl: Allow local CAP_CHECKPOINT_RESTORE to change /proc/self/exe (Adrian Reber) [1960708]
- prctl_set_mm: refactor checks from validate_prctl_map (Adrian Reber) [1960708]
- proc: allow access in init userns for map_files with CAP_CHECKPOINT_RESTORE (Adrian Reber) [1960708]
- pid_namespace: use checkpoint_restore_ns_capable() for ns_last_pid (Adrian Reber) [1960708]
- capabilities: Introduce CAP_CHECKPOINT_RESTORE (Adrian Reber) [1960708]
- scripts: skip empty CAP_* strings (Adrian Reber) [1960708]
- VMCI: Release resource if the work is already queued (Cathy Avery) [1978518]
- perf vendor events arm64: Add Fujitsu A64FX pmu event (Michael Petlan) [1908109]
- perf vendor events arm64: Add more common and uarch events (Michael Petlan) [1908109]
- sunrpc: Avoid a KASAN slab-out-of-bounds bug in xdr_set_page_base() (Benjamin Coddington) [1969751]
- xfs: sync lazy sb accounting on quiesce of read-only mounts (Bill O'Donnell) [1917220]
- drm/dp_mst: Avoid to mess up payload table by ports in stale topology (Lyude Paul) [1944404]
- drm/dp_mst: Do not set proposed vcpi directly (Lyude Paul) [1944404]
- drm/qxl: add lock asserts to qxl_bo_vmap_locked + qxl_bo_vunmap_locked (Lyude Paul) [1944404 1907341]
- drm/qxl: rework cursor plane (Lyude Paul) [1944404 1907341]
- drm/qxl: move shadow handling to new qxl_prepare_shadow() (Lyude Paul) [1944404 1907341]
- drm/qxl: fix monitors object vmap (Lyude Paul) [1944404 1907341]
- drm/qxl: fix prime vmap (Lyude Paul) [1944404 1907341]
- drm/qxl: add qxl_bo_vmap/qxl_bo_vunmap (Lyude Paul) [1944404 1907341]
- drm/qxl: rename qxl_bo_kmap -> qxl_bo_vmap_locked (Lyude Paul) [1944404 1907341]
- drm/qxl: more fence wait rework (Lyude Paul) [1944404 1907341]
- drm/qxl: properly handle device init failures (Lyude Paul) [1944404 1907341]
- drm/qxl: allocate dumb buffers in ram (Lyude Paul) [1944404 1907341]
- drm/qxl: simplify qxl_fence_wait (Lyude Paul) [1944404 1907341]
- drm/qxl: properly free qxl releases (Lyude Paul) [1944404 1907341]
- drm/qxl: handle shadow in primary destroy (Lyude Paul) [1944404 1907341]
- drm/qxl: properly pin/unpin shadow (Lyude Paul) [1944404 1907341]
- drm/qxl: use drmm_mode_config_init (Lyude Paul) [1944404 1907341]
- drm/i915/icp+: Use icp_hpd_irq_setup() instead of spt_hpd_irq_setup() (Lyude Paul) [1944404]
- drm/i915/gen9_bc: Add W/A for missing STRAP config on TGP PCH + CML combos (Lyude Paul) [1944404]
- drm/i915/gen9_bc: Introduce HPD pin mappings for TGP PCH + CML combos (Lyude Paul) [1944404]
- drm/i915/gen9_bc: Introduce TGP PCH DDC pin mappings (Lyude Paul) [1944404]
- drm/i915/gen9_bc: Recognize TGP PCH + CML combos (Lyude Paul) [1944404]
- drm/i915: rename i915_rev_steppings->intel_step_info (Lyude Paul) [1944404]
- drm/i915: rename disp_stepping->display_step and gt_stepping->gt_step (Lyude Paul) [1944404]
- drm/i915: rename DISP_STEPPING->DISPLAY_STEP and GT_STEPPING->GT_STEP (Lyude Paul) [1944404]
- drm/i915: switch TGL and ADL to the new stepping scheme (Lyude Paul) [1944404]
- drm/i915: switch KBL to the new stepping scheme (Lyude Paul) [1944404]
- drm/i915: add new helpers for accessing stepping info (Lyude Paul) [1944404]
- drm/i915: split out stepping info to a new file (Lyude Paul) [1944404]
- drm/i915: Add DISPLAY_VER() and related macros (Lyude Paul) [1944404]
- drm/i915/display: Do not allow DC3CO if PSR SF is enabled (Lyude Paul) [1944404]
- drm/i915: s/dev_priv/i915/ for the remainder of DDI clock routing (Lyude Paul) [1944404]
- drm/i915: Relocate icl_sanitize_encoder_pll_mapping() (Lyude Paul) [1944404]
- drm/i915: Use .disable_clock() for pll sanitation (Lyude Paul) [1944404]
- drm/i915: Split adl-s/rkl from icl_ddi_combo_{enable,disable}_clock() (Lyude Paul) [1944404]
- drm/i915: Extract _cnl_ddi_{enable,disable}_clock() (Lyude Paul) [1944404]
- drm/i915: Sprinkle WARN(!pll) into icl/dg1 .clock_enable() (Lyude Paul) [1944404]
- drm/i915: Sprinkle a few missing locks around shared DDI clock registers (Lyude Paul) [1944404]
- drm/i915: Use intel_de_rmw() for DDI clock routing (Lyude Paul) [1944404]
- drm/i915: Extract icl+ .{enable,disable}_clock() vfuncs (Lyude Paul) [1944404]
- drm/i915: Convert DG1 over to .{enable,disable}_clock() (Lyude Paul) [1944404]
- drm/i195: Extract cnl_ddi_{enable,disable}_clock() (Lyude Paul) [1944404]
- drm/i915: Extract skl_ddi_{enable,disable}_clock() (Lyude Paul) [1944404]
- drm/i915: Extract hsw_ddi_{enable,disable}_clock() (Lyude Paul) [1944404]
- drm/i915: Introduce .{enable,disable}_clock() encoder vfuncs (Lyude Paul) [1944404]
- drm/i915: Use intel_ddi_clk_select() for FDI (Lyude Paul) [1944404]
- drm/i915/adl_s: Add GT and CTX WAs for ADL-S (Lyude Paul) [1944404]
- drm/i915/adl_s: Add display WAs for ADL-S (Lyude Paul) [1944404]
- drm/i915/display: Add HAS_D12_PLANE_MINIMIZATION (Lyude Paul) [1944404]
- drm/i915/adl_s: Re-use TGL GuC/HuC firmware (Lyude Paul) [1944404]
- drm/i915/adl_s: Add power wells (Lyude Paul) [1944404]
- drm/i915/adl_s: Update PHY_MISC programming (Lyude Paul) [1944404]
- drm/i915/adl_s: Update combo PHY master/slave relationships (Lyude Paul) [1944404]
- drm/i915/adl_s: Add vbt port and aux channel settings for adls (Lyude Paul) [1944404]
- drm/i915/adl_s: Add adl-s ddc pin mapping (Lyude Paul) [1944404]
- drm/i915/adl_s: Initialize display for ADL-S (Lyude Paul) [1944404]
- drm/i915/adl_s: Configure Port clock registers for ADL-S (Lyude Paul) [1944404]
- drm/i915/adl_s: Configure DPLL for ADL-S (Lyude Paul) [1944404]
- drm/i915/adl_s: Add PHYs for Alderlake S (Lyude Paul) [1944404]
- drm/i915/adl_s: Add PCH support (Lyude Paul) [1944404]
- drm/i915/adl_s: Add ADL-S platform info and PCI ids (Lyude Paul) [1944404]
- drm/i915/tgl: Use TGL stepping info for applying WAs (Lyude Paul) [1944404]
- redhat/configs: Update CONFIG_HSA_AMD (Lyude Paul)
- redhat/configs: Update CONFIG_DRM_AMDGPU_USERPTR (Lyude Paul)
- DRM Backport v5.11 -> v5.12.4 (Lyude Paul) [1944404]
- include/linux/kernel.h: add typeof_member() macro (Lyude Paul) [1944404]
- drm: Use USB controller's DMA mask when importing dmabufs (Lyude Paul) [1944404]
- PM: runtime: Add pm_runtime_get_if_active() (Lyude Paul) [1944404]
- PCI: Add pci_rebar_bytes_to_size() (Lyude Paul) [1944404]
- PCI: Export pci_rebar_get_possible_sizes() (Lyude Paul) [1944404]
- DRM backport v5.10->v5.11 (Lyude Paul) [1944404]
- dma-buf: Use struct dma_buf_map in dma_buf_vunmap() interfaces (Lyude Paul) [1944404]
- dma-buf: Use struct dma_buf_map in dma_buf_vmap() interfaces (Lyude Paul) [1944404]
- dma-buf: Add struct dma-buf-map for storing struct dma_buf.vaddr_ptr (Lyude Paul) [1944404]
- PCI/PM: Do not generate wakeup event when runtime resuming device (Lyude Paul) [1944404]
- PCI/PM: Rename pci_wakeup_bus() to pci_resume_bus() (Lyude Paul) [1944404]
- blktrace: make relay callbacks const (Lyude Paul) [1944404]
- drm/i915: make relay callbacks const (Lyude Paul) [1944404]
- relay: allow the use of const callback structs (Lyude Paul) [1944404]
- relay: make create_buf_file and remove_buf_file callbacks mandatory (Lyude Paul) [1944404]
- relay: require non-NULL callbacks in relay_open() (Lyude Paul) [1944404]
- relay: remove unused buf_mapped and buf_unmapped callbacks (Lyude Paul) [1944404]
- drm/edid: fix objtool warning in drm_cvt_modes() (Lyude Paul) [1944404]
- drm/edid: Fix uninitialized variable in drm_cvt_modes() (Lyude Paul) [1944404]
- DRM backport v5.9.y -> v5.10 (Lyude Paul) [1944404]
- time: Introduce jiffies64_to_msecs() (Lyude Paul) [1944404]
- virtio: add dma-buf support for exported objects (Lyude Paul) [1944404]
- objtool: Rename frame.h -> objtool.h (Lyude Paul) [1944404]
- moduleparams: Add hexint type parameter (Lyude Paul) [1944404]
- iommu/vt-d: Move intel_iommu_gfx_mapped to Intel IOMMU header (Lyude Paul) [1944404]

* Mon Jul 26 2021 Augusto Caringi <acaringi@redhat.com> [4.18.0-325.el8]
- [s390] s390/zcrypt/pkey: introduce zcrypt_wait_api_operational() function (Claudio Imbrenda) [1881833]
- [s390] s390/ap: ap bus userspace notifications for some bus conditions (Claudio Imbrenda) [1881833]
- [s390] s390/pkey: fix paes selftest failure with paes and pkey static build (Claudio Imbrenda) [1881833]
- KVM: nSVM: Avoid tainiting kernel by default (Vitaly Kuznetsov) [1982248]
- KVM: s390x: implement KVM_CAP_SET_GUEST_DEBUG2 (Thomas Huth) [1945752]
- KVM: s390: fix guarded storage control register handling (Thomas Huth) [1945752]
- s390/protvirt: fix error return code in uv_info_init() (Thomas Huth) [1945752]
- s390/uv: fix prot virt host indication compilation (Thomas Huth) [1945752]
- KVM: s390: VSIE: fix MVPG handling for prefixing and MSO (Thomas Huth) [1945752]
- KVM: s390: split kvm_s390_real_to_abs (Thomas Huth) [1945752]
- KVM: s390: VSIE: correctly handle MVPG when in VSIE (Thomas Huth) [1945752]
- KVM: s390: extend kvm_s390_shadow_fault to return entry pointer (Thomas Huth) [1945752]
- s390/uv: add prot virt guest/host indication files (Thomas Huth) [1945752]
- KVM: s390: split kvm_s390_logical_to_effective (Thomas Huth) [1945752]
- KVM: s390: Fix comment spelling in kvm_s390_vcpu_start() (Thomas Huth) [1945752]
- s390: uv: Fix sysfs max number of VCPUs reporting (Thomas Huth) [1945752]
- KVM: s390: track synchronous pfault events in kvm_stat (Thomas Huth) [1945752]
- s390/gmap: make gmap memcg aware (Thomas Huth) [1945752]
- KVM: s390: Add memcg accounting to KVM allocations (Thomas Huth) [1945752]
- s390/uv: handle destroy page legacy interface (Thomas Huth) [1945752]
- KVM: s390: pv: Mark mm as protected after the set secure parameters and improve cleanup (Thomas Huth) [1945752]
- virtio/s390: implement virtio-ccw revision 2 correctly (Thomas Huth) [1945752]
- sfc: deprecate SFC9020 family (Siena) (Íñigo Huguet) [1926379]
- NFS: Fix open coded versions of nfs_set_cache_invalid() in NFSv4 (Benjamin Coddington) [1936478]
- NFS: Fix open coded versions of nfs_set_cache_invalid() (Benjamin Coddington) [1936478]
- NFS: Clean up function nfs_mark_dir_for_revalidate() (Benjamin Coddington) [1936478]
- NFSD: Mark exports of NFS as tech-preview (Benjamin Coddington) [1972229]
- perf vendor events intel: Add missing skylake & icelake model numbers (Michael Petlan) [1971959]
- tc-testing: add test for ct DNAT tuple collision (Marcelo Ricardo Leitner) [1964578]
- tc-testing: add support for sending various scapy packets (Marcelo Ricardo Leitner) [1964578]
- tc-testing: fix list handling (Marcelo Ricardo Leitner) [1964578]
- net/sched: act_ct: handle DNAT tuple collision (Marcelo Ricardo Leitner) [1964578]
- net/packet: remove data races in fanout operations (Balazs Nemeth) [1957270]
- net/packet: remove locking from packet_rcv_has_room() (Balazs Nemeth) [1957270]
- net: packetmmap: fix only tx timestamp on request (Balazs Nemeth) [1970404]
- x86/sgx: Add TAINT_TECH_PREVIEW for virtual EPC (Bandan Das) [1977474]
- cxgb4: fix thermal zone device registration (Raju Rangoju) [1954012]
- powerpc/64s: Fix crashes when toggling stf barrier (Desnes A. Nunes do Rosario) [1964484]
- iavf: Fix an error handling path in 'iavf_probe()' (Stefan Assmann) [1920276]
- iavf: clean up packet type lookup table (Stefan Assmann) [1920276]
- iavf: redefine the magic number for FDIR GTP-U header fields (Stefan Assmann) [1920276]
- iavf: enhance the duplicated FDIR list scan handling (Stefan Assmann) [1920276]
- iavf: change the flex-byte support number to macro definition (Stefan Assmann) [1920276]
- iavf: remove duplicate free resources calls (Stefan Assmann) [1920276]
- iavf: Support for modifying SCTP RSS flow hashing (Stefan Assmann) [1920276]
- iavf: Support for modifying UDP RSS flow hashing (Stefan Assmann) [1920276]
- iavf: Support for modifying TCP RSS flow hashing (Stefan Assmann) [1920276]
- iavf: Add framework to enable ethtool RSS config (Stefan Assmann) [1920276]
- iavf: add support for UDP Segmentation Offload (Stefan Assmann) [1920276]
- intel: clean up mismatched header comments (Stefan Assmann) [1920276]
- iavf: Enable flex-bytes support (Stefan Assmann) [1920276]
- iavf: Support Ethernet Type Flow Director filters (Stefan Assmann) [1920276]
- iavf: Support IPv6 Flow Director filters (Stefan Assmann) [1920276]
- iavf: Support IPv4 Flow Director filters (Stefan Assmann) [1920276]
- iavf: Add framework to enable ethtool ntuple filters (Stefan Assmann) [1920276]
- net: intel: iavf: fix error return code of iavf_init_get_resources() (Stefan Assmann) [1920276]
- workqueue: Move the position of debug_work_activate() in __queue_work() (Waiman Long) [1977537]
- workqueue: remove useless unlock() and lock() in series (Waiman Long) [1977537]
- workqueue: Convert the pool::lock and wq_mayday_lock to raw_spinlock_t (Waiman Long) [1977537]
- workqueue: Use rcuwait for wq_manager_wait (Waiman Long) [1977537]
- workqueue: Remove unnecessary kfree() call in rcu_free_wq() (Waiman Long) [1977537]
- workqueue: Fix an use after free in init_rescuer() (Waiman Long) [1977537]
- workqueue: Use IS_ERR and PTR_ERR instead of PTR_ERR_OR_ZERO. (Waiman Long) [1977537]
- workqueue: don't use wq_select_unbound_cpu() for bound works (Waiman Long) [1977537]
- workqueue: Use pr_warn instead of pr_warning (Waiman Long) [1977537]
- workqueue: more destroy_workqueue() fixes (Waiman Long) [1977537]
- workqueue: Minor follow-ups to the rescuer destruction change (Waiman Long) [1977537]
- kernel/workqueue: Document wq_worker_last_func() argument (Waiman Long) [1977537]
- workqueue: Only unregister a registered lockdep key (Waiman Long) [1977537]
- kernel: workqueue: clarify wq_worker_last_func() caller requirements (Waiman Long) [1977537]
- workqueue: Fix spelling in source code comments (Waiman Long) [1977537]
- workqueue: fix typo in comment (Waiman Long) [1977537]
- ice: add correct exception tracing for XDP (Jonathan Toppins) [1920278]
- ice: optimize for XDP_REDIRECT in xsk path (Jonathan Toppins) [1920278]
- ice: Allow all LLDP packets from PF to Tx (Jonathan Toppins) [1920278]
- ice: report supported and advertised autoneg using PHY capabilities (Jonathan Toppins) [1920278]
- ice: handle the VF VSI rebuild failure (Jonathan Toppins) [1920278]
- ice: Fix VFR issues for AVF drivers that expect ATQLEN cleared (Jonathan Toppins) [1920278]
- ice: Fix allowing VF to request more/less queues via virtchnl (Jonathan Toppins) [1920278]
- ice: track AF_XDP ZC enabled queues in bitmap (Jonathan Toppins) [1920278]
- ice: handle increasing Tx or Rx ring sizes (Jonathan Toppins) [1920278]
- ice: Fix potential infinite loop when using u8 loop counter (Jonathan Toppins) [1920278]
- ice: fix memory leak of aRFS after resuming from suspend (Jonathan Toppins) [1920278]
- ice: Cleanup fltr list in case of allocation issues (Jonathan Toppins) [1920278]
- ice: Use port number instead of PF ID for WoL (Jonathan Toppins) [1920278]
- ice: Fix for dereference of NULL pointer (Jonathan Toppins) [1920278]
- ice: remove DCBNL_DEVRESET bit from PF state (Jonathan Toppins) [1920278]
- ice: fix memory allocation call (Jonathan Toppins) [1920278]
- ice: prevent ice_open and ice_stop during reset (Jonathan Toppins) [1920278]
- ice: Recognize 860 as iSCSI port in CEE mode (Jonathan Toppins) [1920278]
- ice: Increase control queue timeout (Jonathan Toppins) [1920278]
- ice: Continue probe on link/PHY errors (Jonathan Toppins) [1920278]
- ice: fix napi work done reporting in xsk path (Jonathan Toppins) [1920278]
- ice: update the number of available RSS queues (Jonathan Toppins) [1920278]
- ice: Fix state bits on LLDP mode switch (Jonathan Toppins) [1920278]
- ice: Account for port VLAN in VF max packet size calculation (Jonathan Toppins) [1920278]
- ice: Set trusted VF as default VSI when setting allmulti on (Jonathan Toppins) [1920278]
- ice: report correct max number of TCs (Jonathan Toppins) [1920278]
- ice: skip NULL check against XDP prog in ZC path (Jonathan Toppins) [1920278]
- ice: remove redundant checks in ice_change_mtu (Jonathan Toppins) [1920278]
- ice: move skb pointer from rx_buf to rx_ring (Jonathan Toppins) [1920278]
- ice: simplify ice_run_xdp (Jonathan Toppins) [1920278]
- ice: Improve MSI-X fallback logic (Jonathan Toppins) [1920278]
- ice: Fix trivial error message (Jonathan Toppins) [1920278]
- ice: remove unnecessary casts (Jonathan Toppins) [1920278]
- ice: Refactor DCB related variables out of the ice_port_info struct (Jonathan Toppins) [1920278]
- ice: fix writeback enable logic (Jonathan Toppins) [1920278]
- ice: Use PSM clock frequency to calculate RL profiles (Jonathan Toppins) [1920278]
- ice: create scheduler aggregator node config and move VSIs (Jonathan Toppins) [1920278 1940527 1940535 1940542] {CVE-2020-24504 CVE-2020-24503 CVE-2020-24502}
- ice: Add initial support framework for LAG (Jonathan Toppins) [1920278]
- ice: Remove xsk_buff_pool from VSI structure (Jonathan Toppins) [1920278]
- ice: implement new LLDP filter command (Jonathan Toppins) [1920278]
- ice: log message when trusted VF goes in/out of promisc mode (Jonathan Toppins) [1920278]
- ice: remove dead code (Jonathan Toppins) [1920278]
- ice: use flex_array_size where possible (Jonathan Toppins) [1920278]
- ice: Replace one-element array with flexible-array member (Jonathan Toppins) [1920278]
- ice: display stored UNDI firmware version via devlink info (Jonathan Toppins) [1920278]
- ice: display stored netlist versions via devlink info (Jonathan Toppins) [1920278]
- ice: display some stored NVM versions via devlink info (Jonathan Toppins) [1920278]
- ice: introduce function for reading from flash modules (Jonathan Toppins) [1920278]
- ice: cache NVM module bank information (Jonathan Toppins) [1920278]
- ice: introduce context struct for info report (Jonathan Toppins) [1920278]
- ice: create flash_info structure and separate NVM version (Jonathan Toppins) [1920278]
- ice: report timeout length for erasing during devlink flash (Jonathan Toppins) [1920278]
- ice: Implement flow for IPv6 next header (extension header) (Jonathan Toppins) [1920278]
- ice: fix FDir IPv6 flexbyte (Jonathan Toppins) [1920278]
- ice: drop dead code in ice_receive_skb() (Jonathan Toppins) [1920278]
- ice, xsk: Move Rx allocation out of while-loop (Jonathan Toppins) [1920278]
- ice: avoid premature Rx buffer reuse (Jonathan Toppins) [1920278]
- ice: Add space to unknown speed (Jonathan Toppins) [1920278]
- ice: join format strings to same line as ice_debug (Jonathan Toppins) [1920278]
- ice: silence static analysis warning (Jonathan Toppins) [1920278]
- ice: cleanup misleading comment (Jonathan Toppins) [1920278]
- ice: Remove vlan_ena from vsi structure (Jonathan Toppins) [1920278]
- ice: Remove gate to OROM init (Jonathan Toppins) [1920278]
- ice: Enable Support for FW Override (E82X) (Jonathan Toppins) [1920278]
- ice: don't always return an error for Get PHY Abilities AQ command (Jonathan Toppins) [1920278]
- ice: cleanup stack hog (Jonathan Toppins) [1920278]
- mm: migrate: fix missing update page_private to hugetlb_page_subpool (Chris von Recklinghausen) [1981661]
- mm,hugetlb: drop clearing of flag from prep_new_huge_page (Chris von Recklinghausen) [1981661]
- hugetlb: fix uninitialized subpool pointer (Chris von Recklinghausen) [1981661]
- include/linux/hugetlb.h: add synchronization information for new hugetlb specific flags (Chris von Recklinghausen) [1981661]
- hugetlb: convert PageHugeFreed to HPageFreed flag (Chris von Recklinghausen) [1981661]
- hugetlb: convert PageHugeTemporary() to HPageTemporary flag (Chris von Recklinghausen) [1981661]
- mm: hugetlb: fix a race between freeing and dissolving the page (Chris von Recklinghausen) [1981661]
- mm: don't let userspace spam allocations warnings (Xin Long) [1976699]
- dm writecache: make writeback pause configurable (Mike Snitzer) [1978369]
- dm writecache: pause writeback if cache full and origin being written directly (Mike Snitzer) [1978369]
- dm io tracker: factor out IO tracker (Mike Snitzer) [1978369]
- dm writecache: add optional "metadata_only" parameter (Mike Snitzer) [1978369]
- dm writecache: add "cleaner" and "max_age" to Documentation (Mike Snitzer) [1978369]
- dm writecache: write at least 4k when committing (Mike Snitzer) [1978369]
- dm writecache: have ssd writeback wait if the kcopyd workqueue is busy (Mike Snitzer) [1978369]
- dm writecache: use list_move instead of list_del_list_add in writecache_writeback() (Mike Snitzer) [1978369]
- dm writecache: commit just one block, not a full page (Mike Snitzer) [1978369]
- dm writecache: remove unused gfp_t argument from wc_add_block() (Mike Snitzer) [1978369]
- dm writecache: interrupt writeback if suspended (Mike Snitzer) [1978369]
- dm writecache: don't split bios when overwriting contiguous cache content (Mike Snitzer) [1978369]
- dm kcopyd: avoid spin_lock_irqsave from process context (Mike Snitzer) [1978369]
- dm kcopyd: avoid useless atomic operations (Mike Snitzer) [1978369]
- x86/asm: Optimize memcpy_flushcache() (Mike Snitzer) [1978369]
- IB/isert: Align target max I/O size to initiator size (Kamal Heib) [1930955]
- IB/isert: set rdma cm afonly flag (Kamal Heib) [1930955]

* Wed Jul 21 2021 Augusto Caringi <acaringi@redhat.com> [4.18.0-324.el8]
- xsk: Rollback reservation at NETDEV_TX_BUSY (Jiri Benc) [1882711]
- xsk: Fix race in SKB mode transmit with shared cq (Jiri Benc) [1882711]
- bpf: Only provide bpf_sock_from_file with CONFIG_NET (Jiri Benc) [1882711]
- xsk: Validate socket state in xsk_recvmsg, prior touching socket members (Jiri Benc) [1882711]
- samples/bpf: Fix spelling mistake "recieving" -> "receiving" (Jiri Benc) [1882711]
- samples/bpf: Sample application for eBPF load and socket creation split (Jiri Benc) [1882711]
- bpf: Adds support for setting window clamp (Jiri Benc) [1882711]
- tcp: declare tcp_mmap() only when CONFIG_MMU is set (Jiri Benc) [1882711]
- bpf: Remove trailing semicolon in macro definition (Jiri Benc) [1882711]
- bpf: Allow bpf_{s,g}etsockopt from cgroup bind{4,6} hooks (Jiri Benc) [1882711]
- bpf: Eliminate rlimit-based memory accounting infra for bpf maps (Jiri Benc) [1882711]
- bpf: Eliminate rlimit-based memory accounting for xskmap maps (Jiri Benc) [1882711]
- bpf: Eliminate rlimit-based memory accounting for sockmap and sockhash maps (Jiri Benc) [1882711]
- bpf: Refine memcg-based memory accounting for xskmap maps (Jiri Benc) [1882711]
- bpf: Refine memcg-based memory accounting for sockmap and sockhash maps (Jiri Benc) [1882711]
- net, xdp, xsk: fix __sk_mark_napi_id_once napi_id error (Jiri Benc) [1882711]
- samples/bpf: Add option to set the busy-poll budget (Jiri Benc) [1882711]
- samples/bpf: Add busy-poll support to xdpsock (Jiri Benc) [1882711]
- xsk: Propagate napi_id to XDP socket Rx path (Jiri Benc) [1882711]
- xsk: Add busy-poll support for {recv,send}msg() (Jiri Benc) [1882711]
- xsk: Check need wakeup flag in sendmsg() (Jiri Benc) [1882711]
- xsk: Add support for recvmsg() (Jiri Benc) [1882711]
- net: Add SO_BUSY_POLL_BUDGET socket option (Jiri Benc) [1882711]
- net: Introduce preferred busy-polling (Jiri Benc) [1882711]
- net: napi: use READ_ONCE()/WRITE_ONCE() (Jiri Benc) [1882711]
- net: napi: add hard irqs deferral feature (Jiri Benc) [1882711]
- fs/eventpoll.c: simply CONFIG_NET_RX_BUSY_POLL ifdefery (Jiri Benc) [1882711]
- xdp: Remove the functions xsk_map_inc and xsk_map_put (Jiri Benc) [1882711]
- xsk: Introduce batched Tx descriptor interfaces (Jiri Benc) [1882711]
- xsk: Introduce padding between more ring pointers (Jiri Benc) [1882711]
- net: page_pool: Add bulk support for ptr_ring (Jiri Benc) [1882711]
- net: xdp: Introduce bulking for xdp tx return path (Jiri Benc) [1882711]
- bpf: Rename some functions in bpf_sk_storage (Jiri Benc) [1882711]
- bpf: Folding omem_charge() into sk_storage_charge() (Jiri Benc) [1882711]
- net/smc: properly handle workqueue allocation failure (Andrea Claudi) [1955178]
- Revert "net/smc: fix a NULL pointer dereference" (Andrea Claudi) [1955178]
- KVM: selftests: evmcs_test: Test that KVM_STATE_NESTED_EVMCS is never lost (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Request to sync eVMCS from VMCS12 after migration (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Reset eVMCS clean fields data from prepare_vmcs02() (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Force enlightened VMCS sync from nested_vmx_failValid() (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Ignore 'hv_clean_fields' data when eVMCS data is copied in vmx_get_nested_state() (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Release enlightened VMCS on VMCLEAR (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Introduce 'EVMPTR_MAP_PENDING' post-migration state (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Make copy_vmcs12_to_enlightened()/copy_enlightened_to_vmcs12() return 'void' (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Release eVMCS when enlightened VMENTRY was disabled (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Don't set 'dirty_vmcs12' flag on enlightened VMPTRLD (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Use '-1' in 'hv_evmcs_vmptr' to indicate that eVMCS is not in use (Vitaly Kuznetsov) [1940837]
- KVM: selftests: evmcs_test: Check that VMCS12 is alway properly synced to eVMCS after restore (Vitaly Kuznetsov) [1940837]
- KVM: selftests: evmcs_test: Check that VMLAUNCH with bogus EVMPTR is causing #UD (Vitaly Kuznetsov) [1940837]
- KVM: nVMX: Always make an attempt to map eVMCS after migration (Vitaly Kuznetsov) [1940837]
- Bluetooth: btusb: Some Qualcomm Bluetooth adapters stop working (Gopal Tiwari) [1973909]
- KVM: x86: Unload MMU on guest TLB flush if TDP disabled to force MMU sync (Vitaly Kuznetsov) [1906634]
- scsi: sbitmap: Silence a debug kernel warning triggered by sbitmap_put() (Ewan D. Milne) [1761937]
- scsi: core: Replace sdev->device_busy with sbitmap (Ewan D. Milne) [1761937]
- scsi: core: Make sure sdev->queue_depth is <= max(shost->can_queue, 1024) (Ewan D. Milne) [1761937]
- scsi: core: Add scsi_device_busy() wrapper (Ewan D. Milne) [1761937]
- scsi: blk-mq: Return budget token from .get_budget callback (Ewan D. Milne) [1761937]
- scsi: blk-mq: Add callbacks for storing & retrieving budget token (Ewan D. Milne) [1761937]
- scsi: sbitmap: Add sbitmap_calculate_shift() helper (Ewan D. Milne) [1761937]
- scsi: sbitmap: Export sbitmap_weight (Ewan D. Milne) [1761937]
- scsi: sbitmap: Move allocation hint into sbitmap (Ewan D. Milne) [1761937]
- scsi: sbitmap: Add helpers for updating allocation hint (Ewan D. Milne) [1761937]
- scsi: sbitmap: Maintain allocation round_robin in sbitmap (Ewan D. Milne) [1761937]
- scsi: sbitmap: Remove sbitmap_clear_bit_unlock (Ewan D. Milne) [1761937]
- blk-mq: pass request queue into get/put budget callback (Ewan D. Milne) [1761937]
- dma-iommu: implement ->alloc_noncontiguous (Jerry Snitselaar) [1955767]
- dma-iommu: refactor iommu_dma_alloc_remap (Jerry Snitselaar) [1955767]
- dma-mapping: add a dma_alloc_noncontiguous API (Jerry Snitselaar) [1955767]
- dma-mapping: refactor dma_{alloc,free}_pages (Jerry Snitselaar) [1955767]
- dma-mapping: add a dma_mmap_pages helper (Jerry Snitselaar) [1955767]
- dma-mapping: remove the {alloc,free}_noncoherent methods (Jerry Snitselaar) [1955767]
- iommu/vt-d: Make unnecessarily global functions static (Jerry Snitselaar) [1955767]
- iommu/vt-d: Remove unused function declarations (Jerry Snitselaar) [1955767]
- iommu/vt-d: Remove SVM_FLAG_PRIVATE_PASID (Jerry Snitselaar) [1955767]
- iommu/vt-d: Remove svm_dev_ops (Jerry Snitselaar) [1955767]
- iommu/vt-d: Use Real PCI DMA device for IRTE (Jerry Snitselaar) [1955767]
- iommu/vt-d: Calculate and set flags for handle_mm_fault (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix build error of pasid_enable_wpe() with !X86 (Jerry Snitselaar) [1955767]
- iommu/vt-d: Avoid unnecessary cache flush in pasid entry teardown (Jerry Snitselaar) [1955767]
- iommu/vt-d: Enable write protect propagation from guest (Jerry Snitselaar) [1955767]
- iommu/vt-d: Enable write protect for supervisor SVM (Jerry Snitselaar) [1955767]
- iommu/vt-d: Report more information about invalidation errors (Jerry Snitselaar) [1955767]
- iommu/vt-d: Disable SVM when ATS/PRI/PASID are not enabled in the device (Jerry Snitselaar) [1955767]
- iommu: Stop exporting free_iova_fast() (Jerry Snitselaar) [1955767]
- iommu: Delete iommu_dma_free_cpu_cached_iovas() (Jerry Snitselaar) [1955767]
- iommu/vt-d: Remove IOVA domain rcache flushing for CPU offlining (Jerry Snitselaar) [1955767]
- iova: Add CPU hotplug handler to flush rcaches (Jerry Snitselaar) [1955767]
- iommu/iova: Improve restart logic (Jerry Snitselaar) [1955767]
- iommu/iova: Add rbtree entry helper (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Remove the unused fields for PREFETCH_CONFIG command (Jerry Snitselaar) [1955767]
- iommu/vt-d: Use user privilege for RID2PASID translation (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix sysfs leak in alloc_iommu() (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Add a check to avoid invalid iotlb sync (Jerry Snitselaar) [1955767]
- x86/events/amd/iommu: Fix invalid Perf result due to IOMMU PMC power-gating (Jerry Snitselaar) [1955767]
- iommu: Fix a boundary issue to avoid performance drop (Jerry Snitselaar) [1955767]
- iommu/vt-d: Check for allocation failure in aux_detach_device() (Jerry Snitselaar) [1955767]
- iommu/vt-d: Remove unused dma map/unmap trace events (Jerry Snitselaar) [1955767]
- iommu/amd: Clear DMA ops when switching domain (Jerry Snitselaar) [1955767]
- iommu/amd: Remove performance counter pre-initialization test (Jerry Snitselaar) [1955767]
- Revert "iommu/amd: Fix performance counter initialization" (Jerry Snitselaar) [1955767]
- iommu/dma: Resurrect the "forcedac" option (Jerry Snitselaar) [1955767]
- iommu: Check dev->iommu in iommu_dev_xxx functions (Jerry Snitselaar) [1955767]
- iommu/amd: Put newline after closing bracket in warning (Jerry Snitselaar) [1955767]
- iommu/vt-d: Force to flush iotlb before creating superpage (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix an error handling path in 'intel_prepare_irq_remapping()' (Jerry Snitselaar) [1955767]
- iommu/vt-d: Invalidate PASID cache when root/context entry changed (Jerry Snitselaar) [1955767]
- iommu/vt-d: Remove WO permissions on second-level paging entries (Jerry Snitselaar) [1955767]
- iommu/vt-d: Report the right page fault address (Jerry Snitselaar) [1955767]
- iommu/vt-d: Report right snoop capability when using FL for IOVA (Jerry Snitselaar) [1955767]
- iommu/vt-d: Don't set then clear private data in prq_event_thread() (Jerry Snitselaar) [1955767]
- iommu/vt-d: Reject unsupported page request modes (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: add bit field SFM into GERROR_ERR_MASK (Jerry Snitselaar) [1955767]
- iommu/amd: Keep track of amd_iommu_irq_remap state (Jerry Snitselaar) [1955767]
- iommu/amd: Don't call early_amd_iommu_init() when AMD IOMMU is disabled (Jerry Snitselaar) [1955767]
- iommu/amd: Move Stoney Ridge check to detect_ivrs() (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix status code for Allocate/Free PASID command (Jerry Snitselaar) [1955767]
- iommu: Don't use lazy flush for untrusted device (Jerry Snitselaar) [1955767]
- iommu/amd: Fix sleeping in atomic in increase_address_space() (Jerry Snitselaar) [1955767]
- iommu: Delete iommu_dev_has_feature() (Jerry Snitselaar) [1955767]
- iommu: Delete iommu_domain_window_disable() (Jerry Snitselaar) [1955767]
- iommu: Stop exporting iommu_map_sg_atomic() (Jerry Snitselaar) [1955767]
- iova: Stop exporting some more functions (Jerry Snitselaar) [1955767]
- iova: Delete copy_reserved_iova() (Jerry Snitselaar) [1955767]
- iova: Make has_iova_flush_queue() private (Jerry Snitselaar) [1955767]
- iommu/vt-d: Add iotlb_sync_map callback (Jerry Snitselaar) [1955767]
- iommu/vt-d: Move capability check code to cap_audit files (Jerry Snitselaar) [1921289]
- iommu/vt-d: Audit IOMMU Capabilities and add helper functions (Jerry Snitselaar) [1921289]
- iommu/vt-d: Fix 'physical' typos (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix compile error [-Werror=implicit-function-declaration] (Jerry Snitselaar) [1955767]
- iommu/vt-d: Use INVALID response code instead of FAILURE (Jerry Snitselaar) [1920726]
- iommu/vt-d: Clear PRQ overflow only when PRQ is empty (Jerry Snitselaar) [1955767]
- iommu/vt-d: Preset Access/Dirty bits for IOVA over FL (Jerry Snitselaar) [1955767]
- iommu/vt-d: Add qi_submit trace event (Jerry Snitselaar) [1955767]
- iommu/vt-d: Consolidate duplicate cache invaliation code (Jerry Snitselaar) [1955767]
- iommu/amd: Adopt IO page table framework for AMD IOMMU v1 page table (Jerry Snitselaar) [1955767]
- iommu/amd: Introduce iommu_v1_map_page and iommu_v1_unmap_page (Jerry Snitselaar) [1955767]
- iommu/amd: Introduce iommu_v1_iova_to_phys (Jerry Snitselaar) [1955767]
- iommu/amd: Refactor fetch_pte to use struct amd_io_pgtable (Jerry Snitselaar) [1955767]
- iommu/amd: Rename variables to be consistent with struct io_pgtable_ops (Jerry Snitselaar) [1955767]
- iommu/amd: Remove amd_iommu_domain_get_pgtable (Jerry Snitselaar) [1955767]
- iommu/amd: Restructure code for freeing page table (Jerry Snitselaar) [1955767]
- iommu/amd: Move IO page table related functions (Jerry Snitselaar) [1955767]
- iommu/amd: Declare functions as extern (Jerry Snitselaar) [1955767]
- iommu/amd: Convert to using amd_io_pgtable (Jerry Snitselaar) [1955767]
- iommu/amd: Move pt_root to struct amd_io_pgtable (Jerry Snitselaar) [1955767]
- iommu/amd: Prepare for generic IO page table framework (Jerry Snitselaar) [1955767]
- iommu/amd: Re-define amd_iommu_domain_encode_pgtable as inline (Jerry Snitselaar) [1955767]
- iommu/amd: remove h from printk format specifier (Jerry Snitselaar) [1955767]
- iommu/amd: Use DEFINE_SPINLOCK() for spinlock (Jerry Snitselaar) [1955767]
- iommu/amd: Remove unnecessary assignment (Jerry Snitselaar) [1955767]
- iommu: Properly pass gfp_t in _iommu_map() to avoid atomic sleeping (Jerry Snitselaar) [1955767]
- iommu/io-pgtable-arm-v7s: Add cfg as a param in some macros (Jerry Snitselaar) [1955767]
- iommu/io-pgtable-arm-v7s: Clarify LVL_SHIFT/BITS macro (Jerry Snitselaar) [1955767]
- iommu/io-pgtable-arm-v7s: Use ias to check the valid iova in unmap (Jerry Snitselaar) [1955767]
- iommu/io-pgtable: Remove TLBI_ON_MAP quirk (Jerry Snitselaar) [1955767]
- iommu/io-pgtable: Allow io_pgtable_tlb ops optional (Jerry Snitselaar) [1955767]
- iommu: Switch gather->end to the inclusive end (Jerry Snitselaar) [1955767]
- iommu: Add iova and size as parameters in iotlb_sync_map (Jerry Snitselaar) [1955767]
- iommu: Move iotlb_sync_map out from __iommu_map (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Add support for VHE (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Make BTM optional for SVA (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Split arm_smmu_tlb_inv_range() (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Use DEFINE_RES_MEM() to simplify code (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Remove the page 1 fixup (Jerry Snitselaar) [1955767]
- x86,swiotlb: Adjust SWIOTLB bounce buffer size for SEV guests (Jerry Snitselaar) [1955767]
- iommu: Check dev->iommu in dev_iommu_priv_get() before dereferencing it (Jerry Snitselaar) [1955767]
- iommu/vt-d: Do not use flush-queue when caching-mode is on (Jerry Snitselaar) [1955767]
- iommu/vt-d: Correctly check addr alignment in qi_flush_dev_iotlb_pasid() (Jerry Snitselaar) [1955767]
- iommu/amd: Use IVHD EFR for early initialization of IOMMU features (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix unaligned addresses for intel_flush_svm_range_dev() (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix ineffective devTLB invalidation for subdevices (Jerry Snitselaar) [1920750 1920749]
- iommu/vt-d: Fix general protection fault in aux_detach_device() (Jerry Snitselaar) [1920748]
- iommu/vt-d: Move intel_iommu info from struct intel_svm to struct intel_svm_dev (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix lockdep splat in sva bind()/unbind() (Jerry Snitselaar) [1955767]
- Revert "iommu: Add quirk for Intel graphic devices in map_sg" (Jerry Snitselaar) [1955767]
- iommu/intel: Fix memleak in intel_irq_remapping_alloc (Jerry Snitselaar) [1955767]
- iommu/iova: fix 'domain' typos (Jerry Snitselaar) [1955767]
- dma-iommu: remove __iommu_dma_mmap (Jerry Snitselaar) [1955767]
- iommu/io-pgtable: Remove tlb_flush_leaf (Jerry Snitselaar) [1955767]
- iommu/vt-d: Avoid GFP_ATOMIC where it is not needed (Jerry Snitselaar) [1955767]
- iommu/vt-d: Remove set but not used variable (Jerry Snitselaar) [1955767]
- iommu/vt-d: Cleanup after converting to dma-iommu ops (Jerry Snitselaar) [1955767]
- iommu/vt-d: Convert intel iommu driver to the iommu ops (Jerry Snitselaar) [1955767]
- iommu/vt-d: Don't clear GFP_DMA and GFP_DMA32 flags (Jerry Snitselaar) [1955767]
- iommu/vt-d: Update domain geometry in iommu_ops.at(de)tach_dev (Jerry Snitselaar) [1955767]
- iommu: Add quirk for Intel graphic devices in map_sg (Jerry Snitselaar) [1955767]
- swiotlb: remove the tbl_dma_addr argument to swiotlb_tbl_map_single (Jerry Snitselaar) [1955767]
- swiotlb-xen: remove start_dma_addr (Jerry Snitselaar) [1955767]
- iommu: Allow the dma-iommu api to use bounce buffers (Jerry Snitselaar) [1955767]
- iommu: Add iommu_dma_free_cpu_cached_iovas() (Jerry Snitselaar) [1955767]
- iommu: Handle freelists when using deferred flushing in iommu drivers (Jerry Snitselaar) [1955767]
- iommu/sva: Add PASID helpers (Jerry Snitselaar) [1955767]
- iommu/ioasid: Add ioasid references (Jerry Snitselaar) [1955767]
- iommu/io-pgtable-arm: Remove unused 'level' parameter from iopte_type() macro (Jerry Snitselaar) [1955767]
- iommu: Defer the early return in arm_(v7s/lpae)_map (Jerry Snitselaar) [1955767]
- iommu: Improve the performance for direct_mapping (Jerry Snitselaar) [1955767]
- iommu: return error code when it can't get group (Jerry Snitselaar) [1955767]
- iommu: Modify the description of iommu_sva_unbind_device (Jerry Snitselaar) [1955767]
- iommu: Stop exporting free_iova_mem() (Jerry Snitselaar) [1955767]
- iommu: Stop exporting alloc_iova_mem() (Jerry Snitselaar) [1955767]
- iommu: Delete split_and_remove_iova() (Jerry Snitselaar) [1955767]
- iommu: avoid taking iova_rbtree_lock twice (Jerry Snitselaar) [1955767]
- iommu/iova: Free global iova rcache on iova alloc failure (Jerry Snitselaar) [1955767]
- iommu/iova: Retry from last rb tree node if iova search fails (Jerry Snitselaar) [1955767]
- iommu: Fix htmldocs warnings in sysfs-kernel-iommu_groups (Jerry Snitselaar) [1955767]
- iommu: Document usage of "/sys/kernel/iommu_groups/<grp_id>/type" file (Jerry Snitselaar) [1838589]
- iommu: Take lock before reading iommu group default domain type (Jerry Snitselaar) [1838589]
- iommu: Add support to change default domain of an iommu group (Jerry Snitselaar) [1838589]
- iommu: Move def_domain type check for untrusted device into core (Jerry Snitselaar) [1838589]
- iommu/arm-smmu-v3: Hook up ATC invalidation to mm ops (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Implement iommu_sva_bind/unbind() (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Move non-strict mode to use io_pgtable_domain_attr (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Add support for pagetable config domain attribute (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Assign boolean values to a bool variable (Jerry Snitselaar) [1955767]
- iommu/io-pgtable-arm: Add support to use system cache (Jerry Snitselaar) [1955767]
- iommu/io-pgtable: Add a domain attribute for pagetable configuration (Jerry Snitselaar) [1955767]
- iommu/io-pgtable-arm: Support coherency for Mali LPAE (Jerry Snitselaar) [1955767]
- dma-mapping: document dma_{alloc,free}_pages (Jerry Snitselaar) [1955767]
- dma-direct: simplify the DMA_ATTR_NO_KERNEL_MAPPING handling (Jerry Snitselaar) [1955767]
- dma-direct: factor out a dma_direct_alloc_from_pool helper (Jerry Snitselaar) [1955767]
- dma-direct check for highmem pages in dma_direct_alloc_pages (Jerry Snitselaar) [1955767]
- firewire-ohci: use dma_alloc_pages (Jerry Snitselaar) [1955767]
- dma-iommu: implement ->alloc_noncoherent (Jerry Snitselaar) [1955767]
- dma-mapping: add new {alloc,free}_noncoherent dma_map_ops methods (Jerry Snitselaar) [1955767]
- dma-mapping: add a new dma_alloc_pages API (Jerry Snitselaar) [1955767]
- dma-mapping: add a new dma_alloc_noncoherent API (Jerry Snitselaar) [1955767]
- drm/nouveau/gk20a: stop setting DMA_ATTR_NON_CONSISTENT (Jerry Snitselaar) [1955767]
- mm: turn alloc_pages into an inline function (Jerry Snitselaar) [1955767]
- dma-mapping: better document dma_addr_t and DMA_MAPPING_ERROR (Jerry Snitselaar) [1955767]
- dma-mapping: move valid_dma_direction to dma-direction.h (Jerry Snitselaar) [1955767]
- dma-debug: convert comma to semicolon (Jerry Snitselaar) [1955767]
- dma-direct: Fix potential NULL pointer dereference (Jerry Snitselaar) [1955767]
- dma-mapping: move the dma_declare_coherent_memory documentation (Jerry Snitselaar) [1955767]
- dma-mapping: move dma_common_{mmap,get_sgtable} out of mapping.c (Jerry Snitselaar) [1955767]
- dma-mapping: make support for dma ops optional (Jerry Snitselaar) [1955767]
- dma-direct: rename and cleanup __phys_to_dma (Jerry Snitselaar) [1955767]
- dma-direct: remove __dma_to_phys (Jerry Snitselaar) [1955767]
- dma-direct: use phys_to_dma_direct in dma_direct_alloc (Jerry Snitselaar) [1955767]
- dma-direct: lift gfp_t manipulation out of__dma_direct_alloc_pages (Jerry Snitselaar) [1955767]
- dma-direct: remove dma_direct_{alloc,free}_pages (Jerry Snitselaar) [1955767]
- dma-debug: remove most exports (Jerry Snitselaar) [1955767]
- dma-mapping: remove the dma_dummy_ops export (Jerry Snitselaar) [1955767]
- dma-mapping: set default segment_boundary_mask to ULONG_MAX (Jerry Snitselaar) [1955767]
- dma-mapping: introduce dma_get_seg_boundary_nr_pages() (Jerry Snitselaar) [1955767]
- swiotlb: using SIZE_MAX needs limits.h included (Jerry Snitselaar) [1955767]
- swiotlb: fix "x86: Don't panic if can not alloc buffer for swiotlb" (Jerry Snitselaar) [1955767]
- swiotlb: Mark max_segment with static keyword (Jerry Snitselaar) [1955767]
- swiotlb: Declare swiotlb_late_init_with_default_size() in header (Jerry Snitselaar) [1955767]
- swiotlb: Use pa to print phys_addr_t variables (Jerry Snitselaar) [1955767]
- dma-debug: remove debug_dma_assert_idle() function (Jerry Snitselaar) [1955767]
- dma-mapping: move debug configuration options to kernel/dma (Jerry Snitselaar) [1955767]
- dma-debug: fix debug_dma_assert_idle(), use rcu_read_lock() (Jerry Snitselaar) [1955767]
- iommu: Check return of __iommu_attach_device() (Jerry Snitselaar) [1955767]
- iommu/amd: Enforce 4k mapping for certain IOMMU data structures (Jerry Snitselaar) [1955767]
- iommu: Fix a check in iommu_check_bind_data() (Jerry Snitselaar) [1955767]
- iommu/vt-d: Fix a bug for PDP check in prq_event_thread (Jerry Snitselaar) [1955767]
- iommu/vt-d: Check UAPI data processed by IOMMU core (Jerry Snitselaar) [1955767]
- iommu/uapi: Handle data and argsz filled by users (Jerry Snitselaar) [1955767]
- iommu/uapi: Rename uapi functions (Jerry Snitselaar) [1955767]
- iommu/uapi: Use named union for user data (Jerry Snitselaar) [1955767]
- iommu/uapi: Add argsz for user filled data (Jerry Snitselaar) [1955767]
- docs: IOMMU user API (Jerry Snitselaar) [1955767]
- iommu/dma: Handle init_iova_flush_queue() failure in dma-iommu path (Jerry Snitselaar) [1955767]
- iommu/iova: Replace cmpxchg with xchg in queue_iova (Jerry Snitselaar) [1955767]
- iommu/dma: Remove broken huge page handling (Jerry Snitselaar) [1955767]
- iommu: Rename iommu_tlb_* functions to iommu_iotlb_* (Jerry Snitselaar) [1955767]
- iommu/vt-d: Use device numa domain if RHSA is missing (Jerry Snitselaar) [1955767]
- iommu/vt-d: Drop kerneldoc marker from regular comment (Jerry Snitselaar) [1955767]
- iommu/amd: Re-purpose Exclusion range registers to support SNP CWWB (Jerry Snitselaar) [1955767]
- iommu/amd: Add support for RMP_PAGE_FAULT and RMP_HW_ERR (Jerry Snitselaar) [1955767]
- iommu/amd: Use 4K page for completion wait write-back semaphore (Jerry Snitselaar) [1955767]
- iommu/amd: Add missing function prototypes to fix -Wmissing-prototypes (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Add SVA device feature (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Check for SVA features (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Seize private ASID (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Share process page tables (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Move definitions to a header (Jerry Snitselaar) [1955767]
- iommu/io-pgtable-arm: Move some definitions to a header (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Ensure queue is read after updating prod pointer (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Constify some helpers (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Prepare for the adreno-smmu implementation (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Add support for split pagetables (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Pass io-pgtable config to implementation specific function (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Fix endianness annotations (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: permit users to disable msi polling (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: replace module_param_named by module_param for disable_bypass (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: replace symbolic permissions by octal permissions for module parameter (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Fix l1 stream table size in the error message (Jerry Snitselaar) [1955767]
- ACPI: Do not create new NUMA domains from ACPI static tables that are not SRAT (Jerry Snitselaar) [1955767]
- iommu/amd: Prevent NULL pointer dereference (Jerry Snitselaar) [1955767]
- treewide: Use fallthrough pseudo-keyword (Jerry Snitselaar) [1955767]
- virtio-iommu: convert to LE accessors (Jerry Snitselaar) [1955767]
- virtio_iommu: correct tags for config space fields (Jerry Snitselaar) [1955767]
- mmap locking API: use coccinelle to convert mmap_sem rwsem call sites (Jerry Snitselaar) [1955767]
- iommu/iova: Unify format of the printed messages (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Move Arm SMMU drivers into their own subdirectory (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: move TLB timeout and spin count macros (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Call configuration impl hook before consuming features (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Manage ASIDs with xarray (Jerry Snitselaar) [1955767]
- redhat/configs: Add CONFIG_ARM_SMMU_DISABLE_BYPASS_BY_DEFAULT (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Break insecure users by disabling bypass by default (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Workaround for Marvell Armada-AP806 SoC erratum #582743 (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Allow client devices to select direct mapping (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Add global/context fault implementation hooks (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: add NVIDIA implementation for ARM MMU-500 usage (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Fix trivial typo (Jerry Snitselaar) [1955767]
- iommu/arm-smmu: Update impl quirks comment (Jerry Snitselaar) [1955767]
- iommu: arm-smmu-impl: Convert to a generic reset implementation (Jerry Snitselaar) [1955767]
- iommu: arm-smmu-impl: Add sdm845 implementation hook (Jerry Snitselaar) [1955767]
- firmware/qcom_scm: Add scm call to handle smmu errata (Jerry Snitselaar) [1955767]
- firmware: qcom_scm-64: Add atomic version of qcom_scm_call (Jerry Snitselaar) [1955767]
- firmware: qcom: scm: Ensure 'a0' status code is treated as signed (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Batch context descriptor invalidation (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Write level-1 descriptors atomically (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Add support for PCI PASID (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Add second level of context descriptor table (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Prepare for handling arm_smmu_write_ctx_desc() failure (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Propagate ssid_bits (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Add support for Substream IDs (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Add context descriptor tables allocators (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Prepare arm_smmu_s1_cfg for SSID support (Jerry Snitselaar) [1955767]
- dt-bindings: document PASID property for IOMMU masters (Jerry Snitselaar) [1955767]
- iommu/arm-smmu-v3: Don't display an error when IRQ lines are missing (Jerry Snitselaar) [1955767]
- selftests: net: add UDP GRO forwarding self-tests (Paolo Abeni) [1948493]
- bareudp: allow UDP L4 GRO passthrou (Paolo Abeni) [1948493]
- geneve: allow UDP L4 GRO passthrou (Paolo Abeni) [1948493]
- vxlan: allow L4 GRO passthrough (Paolo Abeni) [1948493]
- udp: never accept GSO_FRAGLIST packets (Paolo Abeni) [1948493]
- udp: properly complete L4 GRO over UDP tunnel packet (Paolo Abeni) [1948493]
- udp: skip L4 aggregation for UDP tunnel packets (Paolo Abeni) [1948493]
- udp: fixup csum for GSO receive slow path (Paolo Abeni) [1948493]
- udp: allow forwarding of plain (non-fraglisted) UDP GRO packets (Paolo Abeni) [1948493]
- net: introduce a netdev feature for UDP GRO forwarding (Paolo Abeni) [1948493]
- [s390] s390/pkey: support CCA and EP11 secure ECC private keys (Claudio Imbrenda) [1780669]
- [s390] s390/zcrypt: Support for CCA APKA master keys (Claudio Imbrenda) [1780669]
- [s390] s390/zcrypt: remove set_fs() invocation in zcrypt device driver (Claudio Imbrenda) [1780669]
- [s390] s390/zcrypt: simplify cca_findcard2 loop code (Claudio Imbrenda) [1780669]
- cxgb4: Add new T6 PCI device id 0x6092 (Rahul Lakkireddy) [1950186]
- KVM: VMX: Don't use vcpu->run->internal.ndata as an array index (Jon Maloy) [1954219] {CVE-2021-3501}
- Bluetooth: Prevent out-of-bounds read in hci_inquiry_result_with_rssi_evt() (Gopal Tiwari) [1978598]
- Bluetooth: Prevent out-of-bounds read in hci_inquiry_result_evt() (Gopal Tiwari) [1978598]
- Bluetooth: Fix slab-out-of-bounds read in hci_extended_inquiry_result_evt() (Gopal Tiwari) [1978598]
- x86/cpu: Avoid cpuinfo-induced IPIing of idle CPUs (Waiman Long) [1970499]
- x86/cpu: Avoid cpuinfo-induced IPI pileups (Waiman Long) [1970499]
- x86/cpu: Disable frequency requests via aperfmperf IPI for nohz_full CPUs (Waiman Long) [1970499]
- x86: Convert some slow-path static_cpu_has() callers to boot_cpu_has() (Waiman Long) [1970499]
- x86/ldt: Enable LDT user-mapping for PAE (Waiman Long) [1970499]
- fanotify: fix merging marks masks with FAN_ONDIR (Miklos Szeredi) [1832099]
- sfc: avoid duplicated code in ef10_sriov (Íñigo Huguet) [1931779]
- sfc: explain that "attached" VFs only refer to Xen (Íñigo Huguet) [1931779]
- sfc: error code if SRIOV cannot be disabled (Íñigo Huguet) [1931779]
- sfc: avoid double pci_remove of VFs (Íñigo Huguet) [1931779]
- tpm: fix error return code in tpm2_get_cc_attrs_tbl() (Jerry Snitselaar) [1920812]
- tpm, tpm_tis: Reserve locality in tpm_tis_resume() (Jerry Snitselaar) [1920812]
- tpm, tpm_tis: Extend locality handling to TPM2 in tpm_tis_gen_interrupt() (Jerry Snitselaar) [1920812]
- tpm: vtpm_proxy: Avoid reading host log when using a virtual device (Jerry Snitselaar) [1920812]
- tpm: acpi: Check eventlog signature before using it (Jerry Snitselaar) [1920812]
- tpm: efi: Use local variable for calculating final log size (Jerry Snitselaar) [1920812]
- tpm: Remove unintentional dump_stack() call (Jerry Snitselaar) [1920812]
- tpm, tpm_tis: Decorate tpm_tis_gen_interrupt() with request_locality() (Jerry Snitselaar) [1920812]
- tpm, tpm_tis: Decorate tpm_get_timeouts() with request_locality() (Jerry Snitselaar) [1920812]
- KEYS: trusted: Reserve TPM for seal and unseal operations (Jerry Snitselaar) [1920812]
- tpm/ppi: Constify static struct attribute_group (Jerry Snitselaar) [1920812]
- tpm: add sysfs exports for all banks of PCR registers (Jerry Snitselaar) [1920812]
- tpm: Remove tpm_dev_wq_lock (Jerry Snitselaar) [1920812]
- tpm: Fix fall-through warnings for Clang (Jerry Snitselaar) [1920812]
- tpm_tis: Clean up locality release (Jerry Snitselaar) [1920812]
- tpm_tis: Disable interrupts on ThinkPad T490s (Jerry Snitselaar) [1920812]
- tpm_tis: Fix check_locality for correct locality acquisition (Jerry Snitselaar) [1920812]
- fuse: invalidate attrs when page writeback completes (Vivek Goyal) [1966424]
- net/sched: act_ct: remove rh_mark_used_feature mark (Marcelo Ricardo Leitner) [1973381]
- selftests: netfilter: Pass family parameter "-f" to conntrack tool (Hangbin Liu) [1979203]
- selftests: netfilter: remove unused cnt and simplify command testing (Hangbin Liu) [1979203]
- selftests: netfilter: fix nft_meta.sh error reporting (Hangbin Liu) [1979203]
- selftests: netfilter: add cpu counter check (Hangbin Liu) [1979203]
- selftests: netfilter: add meta iif/oif match test (Hangbin Liu) [1979203]
- NFS: Only change the cookie verifier if the directory page cache is empty (Benjamin Coddington) [1982825]
- NFS: Fix handling of cookie verifier in uncached_readdir() (Benjamin Coddington) [1982825]
- nfs: Subsequent READDIR calls should carry non-zero cookieverifier (Benjamin Coddington) [1982825]
- Revert "[netdrv] net/mlx5e: Rx, Update page pool numa node when changed" (Alaa Hleihel) [1915308]
- mlx5: net: zero-initialize tc skb extension on allocation (Alaa Hleihel) [1915308 1965418]
- RDMA/mlx5: Don't add slave port to unaffiliated list (Alaa Hleihel) [1915308]
- net/mlx5: Reset mkey index on creation (Alaa Hleihel) [1915308]
- net/mlx5e: Don't create devices during unload flow (Alaa Hleihel) [1915308]
- net/mlx5: DR, Fix STEv1 incorrect L3 decapsulation padding (Alaa Hleihel) [1915308]
- net/mlx5: SF_DEV, remove SF device on invalid state (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Allow setting GUID for host PF vport (Alaa Hleihel) [1915308 1967488]
- net/mlx5: E-Switch, Read PF mac address (Alaa Hleihel) [1915308 1967488]
- net/mlx5: Check that driver was probed prior attaching the device (Alaa Hleihel) [1915308]
- net/mlx5: Fix error path for set HCA defaults (Alaa Hleihel) [1915308]
- net/mlx5e: Disable TLS device offload in kdump mode (Alaa Hleihel) [1915308 1946647]
- net/mlx5e: Disable TX MPWQE in kdump mode (Alaa Hleihel) [1915308 1946647]
- net/mlx5e: kTLS, Add resiliency to RX resync failures (Alaa Hleihel) [1915308]
- net/mlx5e: TX, Inline function mlx5e_tls_handle_tx_wqe() (Alaa Hleihel) [1915308]
- net/mlx5e: TX, Inline TLS skb check (Alaa Hleihel) [1915308]
- net/mlx5e: Cleanup unused function parameter (Alaa Hleihel) [1915308]
- net/mlx5e: Remove non-essential TLS SQ state bit (Alaa Hleihel) [1915308]
- net/mlx5e: Restrict usage of mlx5e_priv in params logic functions (Alaa Hleihel) [1915308]
- net/mlx5e: Fix lost changes during code movements (Alaa Hleihel) [1915308]
- net/mlx5e: Move params logic into its dedicated file (Alaa Hleihel) [1915308]
- net/mlx5e: Pass q_counter indentifier as parameter to rq_param builders (Alaa Hleihel) [1915308]
- net/mlx5e: Block offload of outer header csum for GRE tunnel (Alaa Hleihel) [1915308]
- net/mlx5e: Block offload of outer header csum for UDP tunnels (Alaa Hleihel) [1915308]
- Revert "net/mlx5: Arm only EQs with EQEs" (Alaa Hleihel) [1915308]
- net/mlx5e: Verify dev is present in get devlink port ndo (Alaa Hleihel) [1915308]
- net/mlx5: DR, Don't use SW steering when RoCE is not supported (Alaa Hleihel) [1915308]
- net/mlx5: Consider RoCE cap before init RDMA resources (Alaa Hleihel) [1915308]
- net/mlx5e: Fix page reclaim for dead peer hairpin (Alaa Hleihel) [1915308]
- net/mlx5e: Remove dependency in IPsec initialization flows (Alaa Hleihel) [1915308]
- net/mlx5e: Fix use-after-free of encap entry in neigh update handler (Alaa Hleihel) [1915308]
- IB/mlx5: Fix initializing CQ fragments buffer (Alaa Hleihel) [1915308 1974263]
- RDMA/mlx5: Block FDB rules when not in switchdev mode (Alaa Hleihel) [1915308]
- RDMA/mlx5: Use different doorbell memory for different processes (Alaa Hleihel) [1915308]
- net/mlx5: Fix duplicate included vhca_event.h (Alaa Hleihel) [1915308]
- net/mlx5: Fix lag port remapping logic (Alaa Hleihel) [1915308]
- net/mlx5e: IPsec/rep_tc: Fix rep_tc_update_skb drops IPsec packet (Alaa Hleihel) [1915308]
- net/mlx5: DR, Create multi-destination flow table with level less than 64 (Alaa Hleihel) [1915308]
- net/mlx5e: Fix adding encap rules to slow path (Alaa Hleihel) [1915308]
- net/mlx5e: Check for needed capability for cvlan matching (Alaa Hleihel) [1915308]
- net/mlx5: Check firmware sync reset requested is set before trying to abort it (Alaa Hleihel) [1915308]
- net/mlx5e: Disable TLS offload for uplink representor (Alaa Hleihel) [1915308]
- net/mlx5e: Fix incompatible casting (Alaa Hleihel) [1915308]
- RDMA/mlx5: Fix query DCT via DEVX (Alaa Hleihel) [1915308]
- RDMA/mlx5: Recover from fatal event in dual port mode (Alaa Hleihel) [1915308]
- net/mlx5: Set term table as an unmanaged flow table (Alaa Hleihel) [1915308]
- net/mlx5e: Fix error path of updating netdev queues (Alaa Hleihel) [1915308]
- net/mlx5e: Reject mirroring on source port change encap rules (Alaa Hleihel) [1915308]
- net/mlx5e: Fix multipath lag activation (Alaa Hleihel) [1915308]
- net/mlx5e: reset XPS on error flow if netdev isn't registered yet (Alaa Hleihel) [1915308]
- net/mlx5e: Make sure fib dev exists in fib event (Alaa Hleihel) [1915308]
- net/mlx5e: Fix null deref accessing lag dev (Alaa Hleihel) [1915308]
- net/mlx5e: Fix nullptr in mlx5e_tc_add_fdb_flow() (Alaa Hleihel) [1915308]
- net/mlx5: SF, Fix show state inactive when its inactivated (Alaa Hleihel) [1915308]
- net/mlx5: Set reformat action when needed for termination rules (Alaa Hleihel) [1915308]
- net/mlx5e: Fix nullptr in add_vlan_push_action() (Alaa Hleihel) [1915308]
- {net, RDMA}/mlx5: Fix override of log_max_qp by other device (Alaa Hleihel) [1915308]
- net/mlx5: DR, Fix SQ/RQ in doorbell bitmask (Alaa Hleihel) [1915308]
- net/mlx5: Fix bit-wise and with zero (Alaa Hleihel) [1915308]
- RDMA/mlx5: Fix drop packet rule in egress table (Alaa Hleihel) [1915308]
- net/mlx5: Fix spelling mistakes in mlx5_core_info message (Alaa Hleihel) [1915308]
- net/mlx5: DR, Fixed typo in STE v0 (Alaa Hleihel) [1915308]
- RDMA/mlx5: Fix mlx5 rates to IB rates map (Alaa Hleihel) [1915308]
- RDMA/mlx5: Fix query RoCE port (Alaa Hleihel) [1915308]
- RDMA/mlx5: Zero out ODP related items in the mlx5_ib_mr (Alaa Hleihel) [1915308]
- net/mlx5e: Alloc flow spec using kvzalloc instead of kzalloc (Alaa Hleihel) [1915308]
- net/mlx5: Avoid unnecessary operation (Alaa Hleihel) [1915308]
- net/mlx5e: rep: Improve reg_cX conditions (Alaa Hleihel) [1915308]
- net/mlx5: SF, Fix return type (Alaa Hleihel) [1915308]
- net/mlx5e: mlx5_tc_ct_init does not fail (Alaa Hleihel) [1915308]
- net/mlx5: Fix indir stable stubs (Alaa Hleihel) [1915308]
- net/mlx5e: Add missing include (Alaa Hleihel) [1915308]
- net/mlx5e: fix mlx5e_tc_tun_update_header_ipv6 dummy definition (Alaa Hleihel) [1915308]
- net/mlx5e: CT, Avoid false lock dependency warning (Alaa Hleihel) [1915308]
- net/mlx5: Check returned value from health recover sequence (Alaa Hleihel) [1915308]
- net/mlx5: Don't rely on interface state bit (Alaa Hleihel) [1915308]
- net/mlx5: Remove second FW tracer check (Alaa Hleihel) [1915308]
- net/mlx5: Separate probe vs. reload flows (Alaa Hleihel) [1915308]
- net/mlx5: Remove impossible checks of interface state (Alaa Hleihel) [1915308]
- net/mlx5: Don't skip vport check (Alaa Hleihel) [1915308]
- net/mlx5e: Register nic devlink port for SF ports (Alaa Hleihel) [1915308]
- vdpa/mlx5: Retrieve BAR address suitable any function (Alaa Hleihel) [1915308]
- vdpa/mlx5: Use the correct dma device when registering memory (Alaa Hleihel) [1915308]
- net/mlx5: SF, Extend SF table for additional SF id range (Alaa Hleihel) [1915308]
- net/mlx5: SF, Split mlx5_sf_hw_table into two parts (Alaa Hleihel) [1915308]
- net/mlx5: SF, Use helpers for allocation and free (Alaa Hleihel) [1915308]
- net/mlx5: SF, Consider own vhca events of SF devices (Alaa Hleihel) [1915308]
- net/mlx5: SF, Store and use start function id (Alaa Hleihel) [1915308]
- net/mlx5: SF, Rely on hw table for SF devlink port allocation (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Consider SF ports of host PF (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Use xarray for vport number to vport and rep mapping (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Prepare to return total vports from eswitch struct (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Return eswitch max ports when eswitch is supported (Alaa Hleihel) [1915308]
- net/mlx5: SF, Reuse stored hardware function id (Alaa Hleihel) [1915308]
- net/mlx5: SF, Use device pointer directly (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Initialize eswitch acls ns when eswitch is enabled (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Move legacy code to a individual file (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Convert a macro to a helper routine (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch Make cleanup sequence mirror of init (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Make vport number u16 (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Skip querying SF enabled bits (Alaa Hleihel) [1915308]
- IB/mlx5: Set right RoCE l3 type and roce version while deleting GID (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, move QoS specific fields to existing qos struct (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, cut down mlx5_vport_info structure size by 8 bytes (Alaa Hleihel) [1915308]
- net/mlx5: Pair mutex_destory with mutex_init for rate limit table (Alaa Hleihel) [1915308]
- net/mlx5: Allocate rate limit table when rate is configured (Alaa Hleihel) [1915308]
- net/mlx5: Use helper to increment, decrement rate entry refcount (Alaa Hleihel) [1915308]
- net/mlx5: Use helpers to allocate and free rl table entries (Alaa Hleihel) [1915308]
- net/mlx5: Do not hold mutex while reading table constants (Alaa Hleihel) [1915308]
- net/mlx5: Pack mlx5_rl_entry structure (Alaa Hleihel) [1915308]
- net/mlx5: Use unsigned int for free_count (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, move QoS specific fields to existing qos struct (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, cut down mlx5_vport_info structure size by 8 bytes (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, let user to enable disable metadata (Alaa Hleihel) [1915308]
- net/mlx5e: TC, Add support to offload sample action (Alaa Hleihel) [1915308 1919649]
- net/mlx5e: TC, Handle sampled packets (Alaa Hleihel) [1915308 1919649]
- net/mlx5e: TC, Refactor tc update skb function (Alaa Hleihel) [1915308 1919649]
- net/mlx5e: TC, Add sampler restore handle API (Alaa Hleihel) [1915308 1919649]
- net/mlx5e: TC, Add sampler object API (Alaa Hleihel) [1915308 1919649]
- net/mlx5e: TC, Add sampler termination table API (Alaa Hleihel) [1915308 1919649]
- net/mlx5e: TC, Parse sample action (Alaa Hleihel) [1915308 1919649]
- net/mlx5: Instantiate separate mapping objects for FDB and NIC tables (Alaa Hleihel) [1915308 1919649]
- net/mlx5: Map register values to restore objects (Alaa Hleihel) [1915308 1919649]
- net/mlx5: E-switch, Set per vport table default group number (Alaa Hleihel) [1915308 1919649]
- net/mlx5: E-switch, Generalize per vport table API (Alaa Hleihel) [1915308 1919649]
- net/mlx5: E-switch, Rename functions to follow naming convention. (Alaa Hleihel) [1915308 1919649]
- net/mlx5: E-switch, Move vport table functions to a new file (Alaa Hleihel) [1915308 1919649]
- net/mlx5: CT: Add support for mirroring (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, Protect changing mode while adding rules (Alaa Hleihel) [1915308 1790457]
- net/mlx5: E-Switch, Change mode lock from mutex to rw semaphore (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Do not reload ethernet ports when changing eswitch mode (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Unregister eth-reps devices first (Alaa Hleihel) [1915308 1790457]
- net/mlx5: Move mlx5e hw resources into a sub object (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Register nic devlink port with switch id (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Verify dev is present in some ndos (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Use nic mode netdev ndos and ethtool ops for uplink representor (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Add offload stats ndos to nic netdev ops (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Distinguish nic and esw offload in tc setup block cb (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Allow legacy vf ndos only if in legacy mode (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: Same max num channels for both nic and uplink profiles (Alaa Hleihel) [1915308 1790457]
- net/mlx5e: fix ingress_ifindex check in mlx5e_flower_parse_meta (Alaa Hleihel) [1915308]
- net/mlx5e: Fix setting of RS FEC mode (Alaa Hleihel) [1915308]
- net/mlx5: Fix setting of devlink traps in switchdev mode (Alaa Hleihel) [1915308]
- net/mlx5: DR, Add missing vhca_id consume from STEv1 (Alaa Hleihel) [1915308]
- net/mlx5: fix kfree mismatch in indir_table.c (Alaa Hleihel) [1915308]
- net/mlx5: Fix PBMC register mapping (Alaa Hleihel) [1915308]
- net/mlx5: Fix PPLM register mapping (Alaa Hleihel) [1915308]
- net/mlx5: Fix placement of log_max_flow_counter (Alaa Hleihel) [1915308]
- net/mlx5: Fix HW spec violation configuring uplink (Alaa Hleihel) [1915308]
- net/mlx5e: Guarantee room for XSK wakeup NOP on async ICOSQ (Alaa Hleihel) [1915308]
- net/mlx5e: Consider geneve_opts for encap contexts (Alaa Hleihel) [1915308]
- net/mlx5: Don't request more than supported EQs (Alaa Hleihel) [1915308]
- net/mlx5e: kTLS, Fix RX counters atomicity (Alaa Hleihel) [1915308]
- net/mlx5e: kTLS, Fix TX counters atomicity (Alaa Hleihel) [1915308]
- net/mlx5: E-switch, Create vport miss group only if src rewrite is supported (Alaa Hleihel) [1915308 1915310]
- net/mlx5e: Fix ethtool indication of connector type (Alaa Hleihel) [1915308]
- net/mlx5: Delete auxiliary bus driver eth-rep first (Alaa Hleihel) [1915308]
- net/mlx5e: Fix mapping of ct_label zero (Alaa Hleihel) [1915308]
- net/mlx5: SF, do not use ecpu bit for vhca state processing (Alaa Hleihel) [1915308]
- net/mlx5e: Fix division by 0 in mlx5e_select_queue (Alaa Hleihel) [1915308]
- net/mlx5e: Fix error path for ethtool set-priv-flag (Alaa Hleihel) [1915308]
- net/mlx5e: Offload tuple rewrite for non-CT flows (Alaa Hleihel) [1915308]
- net/mlx5e: Allow to match on MPLS parameters only for MPLS over UDP (Alaa Hleihel) [1915308]
- net/mlx5: Add back multicast stats for uplink representor (Alaa Hleihel) [1915308]
- RDMA/mlx5: Fix typo in destroy_mkey inbox (Alaa Hleihel) [1915308]
- net/mlx5: DR, Fix potential shift wrapping of 32-bit value in STEv1 getter (Alaa Hleihel) [1915308]
- net/mlx5: SF: Fix error flow of SFs allocation flow (Alaa Hleihel) [1915308]
- net/mlx5: SF: Fix memory leak of work item (Alaa Hleihel) [1915308]
- net/mlx5: SF, Correct vhca context size (Alaa Hleihel) [1915308]
- net/mlx5e: E-switch, Fix rate calculation division (Alaa Hleihel) [1915308]
- RDMA/mlx5: Fix timestamp default mode (Alaa Hleihel) [1915308]
- net/mlx5: Set QP timestamp mode to default (Alaa Hleihel) [1915308]
- net/mlx5e: Fix error flow in change profile (Alaa Hleihel) [1915308]
- net/mlx5: Disable VF tunnel TX offload if ignore_flow_level isn't supported (Alaa Hleihel) [1915308 1915310]
- net/mlx5e: Check correct ip_version in decapsulation route resolution (Alaa Hleihel) [1915308]
- net/mlx5: Fix turn-off PPS command (Alaa Hleihel) [1915308]
- net/mlx5e: Don't match on Geneve options in case option masks are all zero (Alaa Hleihel) [1915308]
- net/mlx5e: Revert parameters on errors when changing PTP state without reset (Alaa Hleihel) [1915308]
- net/mlx5e: When changing XDP program without reset, take refs for XSK RQs (Alaa Hleihel) [1915308]
- net/mlx5e: Set PTP channel pointer explicitly to NULL (Alaa Hleihel) [1915308]
- net/mlx5e: Accumulate port PTP TX stats with other channels stats (Alaa Hleihel) [1915308]
- net/mlx5e: RX, Mind the MPWQE gaps when calculating offsets (Alaa Hleihel) [1915308]
- net/mlx5e: Enforce minimum value check for ICOSQ size (Alaa Hleihel) [1915308]
- RDMA/mlx5: Set correct kernel-doc identifier (Alaa Hleihel) [1915308]
- IB/mlx5: Add missing error code (Alaa Hleihel) [1915308]
- configs: Enable mlx5 subfunction device support (Alaa Hleihel) [1915308 1851717]
- net/mlx5: Add cyc2time HW translation mode support (Alaa Hleihel) [1915308 1919657]
- net/mlx5: Move some PPS logic into helper functions (Alaa Hleihel) [1915308 1919657]
- net/mlx5: Move all internal timer metadata into a dedicated struct (Alaa Hleihel) [1915308 1919657]
- net/mlx5: Refactor init clock function (Alaa Hleihel) [1915308 1919657]
- net/mlx5: Add register layout to support real-time time-stamp (Alaa Hleihel) [1915308 1919657]
- RDMA/mlx5: Fail QP creation if the device can not support the CQE TS (Alaa Hleihel) [1915308]
- RDMA/mlx5: Allow CQ creation without attached EQs (Alaa Hleihel) [1915308]
- net/mlx5: Add new timestamp mode bits (Alaa Hleihel) [1915308 1919657]
- net/mlx5: Remove TLS dependencies on XPS (Alaa Hleihel) [1915308]
- net/mlx5: SF, Fix error return code in mlx5_sf_dev_probe() (Alaa Hleihel) [1915308 1851717]
- net/mlx5e: Fix error return code in mlx5e_tc_esw_init() (Alaa Hleihel) [1915308 1919647]
- net/mlx5: Fix a NULL vs IS_ERR() check (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: Fix tc_tun.h to verify MLX5_ESWITCH config (Alaa Hleihel) [1915308 1919647]
- net/mlx5: Assign boolean values to a bool variable (Alaa Hleihel) [1915308]
- net/mlx5e: Fix spelling mistake "Unknouwn" -> "Unknown" (Alaa Hleihel) [1915308]
- net/mlx5e: Fix spelling mistake "channles" -> "channels" (Alaa Hleihel) [1915308]
- net/mlx5: Delete device list leftover (Alaa Hleihel) [1915308]
- net/mlx5_core: remove unused including <generated/utsrelease.h> (Alaa Hleihel) [1915308]
- net/mlx5: fix spelling mistake in Kconfig "accelaration" -> "acceleration" (Alaa Hleihel) [1915308]
- RDMA/mlx5: Support 400Gbps IB rate in mlx5 driver (Alaa Hleihel) [1915308]
- RDMA/mlx5: Cleanup the synchronize_srcu() from the ODP flow (Alaa Hleihel) [1915308]
- net/mlx5e: Handle FIB events to update tunnel endpoint device (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: Rename some encap-specific API to generic names (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: TC preparation refactoring for routing update event (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: Refactor neigh update infrastructure (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: Create route entry infrastructure (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: Extract tc tunnel encap/decap code to dedicated file (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: Match recirculated packet miss in slow table using reg_c1 (Alaa Hleihel) [1915308 1915310 1919647]
- net/mlx5e: Refactor reg_c1 usage (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: VF tunnel RX traffic offloading (Alaa Hleihel) [1915308 1915310 1919647]
- net/mlx5e: Remove redundant match on tunnel destination mac (Alaa Hleihel) [1915308 1919647]
- net/mlx5: E-Switch, Indirect table infrastructure (Alaa Hleihel) [1915308 1915310 1919647]
- net/mlx5e: Refactor tun routing helpers (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: VF tunnel TX traffic offloading (Alaa Hleihel) [1915308 1915310 1919647]
- net/mlx5: E-Switch, Refactor rule offload forward action processing (Alaa Hleihel) [1915308 1915310 1919647]
- net/mlx5e: Always set attr mdev pointer (Alaa Hleihel) [1915308 1919647]
- net/mlx5e: E-Switch, Maintain vhca_id to vport_num mapping (Alaa Hleihel) [1915308 1915310 1919647]
- net/mlx5: E-Switch, Refactor setting source port (Alaa Hleihel) [1915308 1915310 1919647]
- IB/mlx5: Use rdma_for_each_port for port iteration (Alaa Hleihel) [1915308]
- RDMA/core: Introduce and use API to read port immutable data (Alaa Hleihel) [1915308]
- IB/mlx5: Improve query port for representor port (Alaa Hleihel) [1915308]
- IB/mlx5: Avoid calling query device for reading pkey table length (Alaa Hleihel) [1915308]
- IB/mlx5: Move mlx5_port_caps from mlx5_core_dev to mlx5_ib_dev (Alaa Hleihel) [1915308]
- IB/mlx5: Support default partition key for representor port (Alaa Hleihel) [1915308]
- net/mlx5: DR, Avoid unnecessary csum recalculation on supporting devices (Alaa Hleihel) [1915308]
- net/mlx5e: CT: remove useless conversion to PTR_ERR then ERR_PTR (Alaa Hleihel) [1915308]
- net/mlx5e: accel, remove redundant space (Alaa Hleihel) [1915308]
- net/mlx5e: kTLS, Improve TLS RX workqueue scope (Alaa Hleihel) [1915308]
- net/mlx5e: remove h from printk format specifier (Alaa Hleihel) [1915308]
- net/mlx5e: Increase indirection RQ table size to 256 (Alaa Hleihel) [1915308]
- net/mlx5e: Enable napi in channel's activation stage (Alaa Hleihel) [1915308]
- net/mlx5e: Move representor neigh init into profile enable (Alaa Hleihel) [1915308]
- net/mlx5e: Avoid false lock depenency warning on tc_ht (Alaa Hleihel) [1915308]
- net/mlx5e: Move set vxlan nic info to profile init (Alaa Hleihel) [1915308]
- net/mlx5e: Move netif_carrier_off() out of mlx5e_priv_init() (Alaa Hleihel) [1915308]
- net/mlx5e: Refactor mlx5e_netdev_init/cleanup to mlx5e_priv_init/cleanup (Alaa Hleihel) [1915308]
- net/mxl5e: Add change profile method (Alaa Hleihel) [1915308]
- net/mlx5e: Separate between netdev objects and mlx5e profiles initialization (Alaa Hleihel) [1915308]
- net/mlx5: DR, Allow SW steering for sw_owner_v2 devices (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Copy all 64B whenever replacing STE in the head of miss-list (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Use HW specific logic API when writing STE (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Use the right size when writing partial STE into HW (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Add STEv1 modify header logic (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Add STEv1 action apply logic (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Add STEv1 setters and getters (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Allow native protocol support for HW STEv1 (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Add HW STEv1 match logic (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Add match STEv1 structs to ifc (Alaa Hleihel) [1915308 1851712]
- net/mlx5: DR, Fix potential shift wrapping of 32-bit value (Alaa Hleihel) [1915308 1851712]
- net/mlx5e: Enable traps according to link state (Alaa Hleihel) [1915308 1919652]
- net/mlx5e: Add listener to DMAC filter trap event (Alaa Hleihel) [1915308 1919652]
- net/mlx5e: Add listener to trap event (Alaa Hleihel) [1915308 1919652]
- net/mlx5e: Add trap entity to ETH driver (Alaa Hleihel) [1915308 1919652]
- net/mlx5e: Expose RX dma info helpers (Alaa Hleihel) [1915308 1919652]
- net/mlx5e: Add flow steering DMAC trap rule (Alaa Hleihel) [1915308 1919652]
- net/mlx5e: Add flow steering VLAN trap rule (Alaa Hleihel) [1915308 1919652]
- net/mlx5e: Optimize promiscuous mode (Alaa Hleihel) [1915308 1919652]
- net/mlx5: Notify on trap action by blocking event (Alaa Hleihel) [1915308 1919652]
- net/mlx5: Rename events notifier header (Alaa Hleihel) [1915308 1919652]
- net/mlx5: Register to devlink DMAC filter trap (Alaa Hleihel) [1915308 1919652]
- net/mlx5: Register to devlink ingress VLAN filter trap (Alaa Hleihel) [1915308 1919652]
- net/mlx5: Add support for devlink traps in mlx5 core driver (Alaa Hleihel) [1915308 1919652]
- net/mlx5e: Support HTB offload (Alaa Hleihel) [1915308]
- net/mlx5: SF, Port function state change support (Alaa Hleihel) [1915308 1851717]
- net/mlx5: SF, Add port add delete functionality (Alaa Hleihel) [1915308 1851717]
- net/mlx5: E-switch, Add eswitch helpers for SF vport (Alaa Hleihel) [1915308 1915310 1851717]
- net/mlx5: E-switch, Prepare eswitch to handle SF vport (Alaa Hleihel) [1915308 1915310 1851717]
- net/mlx5: SF, Add auxiliary device driver (Alaa Hleihel) [1915308 1851717]
- net/mlx5: SF, Add auxiliary device support (Alaa Hleihel) [1915308 1851717]
- net/mlx5: Introduce vhca state event notifier (Alaa Hleihel) [1915308 1851717]
- RDMA/hw/mlx5/qp: Demote non-conformant kernel-doc header (Alaa Hleihel) [1915308]
- RDMA/hw/mlx5/odp: Fix formatting and add missing descriptions in 'pagefault_data_segments()' (Alaa Hleihel) [1915308]
- IB/mlx5: Make function static (Alaa Hleihel) [1915308]
- net/mlx5e: IPsec, Remove unnecessary config flag usage (Alaa Hleihel) [1915308]
- net/mlx5e: IPsec, Inline feature_check fast-path function (Alaa Hleihel) [1915308]
- net/mlx5e: IPsec, Avoid unreachable return (Alaa Hleihel) [1915308]
- net/mlx5e: IPsec, Enclose csum logic under ipsec config (Alaa Hleihel) [1915308]
- net/mlx5e: CT: Remove redundant usage of zone mask (Alaa Hleihel) [1915308]
- net/mlx5e: Remove redundant initialization to null (Alaa Hleihel) [1915308]
- net/mlx5e: CT: Pass null instead of zero spec (Alaa Hleihel) [1915308]
- net/mlx5e: E-Switch, Offload all chain 0 priorities when modify header and forward action is not supported (Alaa Hleihel) [1915308]
- net/mlx5: E-Switch, use new cap as condition for mpls over udp (Alaa Hleihel) [1915308]
- net/mlx5e: Simplify condition on esw_vport_enable_qos() (Alaa Hleihel) [1915308]
- net/mlx5: Add HW definition of reg_c_preserve (Alaa Hleihel) [1915308]
- net/mlx5: DR, Move STEv0 modify header logic (Alaa Hleihel) [1915308]
- net/mlx5: DR, Add STE modify header actions per-device API (Alaa Hleihel) [1915308]
- net/mlx5: DR, Move STEv0 action apply logic (Alaa Hleihel) [1915308]
- net/mlx5: DR, Add STE tx/rx actions per-device API (Alaa Hleihel) [1915308]
- net/mlx5: DR, Move STEv0 setters and getters (Alaa Hleihel) [1915308]
- net/mlx5: DR, Add STE setters and getters per-device API (Alaa Hleihel) [1915308]
- net/mlx5: DR, Move action apply logic to dr_ste (Alaa Hleihel) [1915308]
- net/mlx5: DR, Refactor ICMP STE builder (Alaa Hleihel) [1915308]
- net/mlx5: DR, Move STEv0 look up types from mlx5_ifc_dr header (Alaa Hleihel) [1915308]
- net/mlx5: DR, Merge similar DR STE SET macros (Alaa Hleihel) [1915308]
- net/mlx5: DR, Fix STEv0 source_eswitch_owner_vhca_id support (Alaa Hleihel) [1915308]
- net/mlx5: DR, Remove unused macro definition from dr_ste (Alaa Hleihel) [1915308]
- net/mlx5: DR, Move HW STEv0 match logic to a separate file (Alaa Hleihel) [1915308]
- net/mlx5: DR, Use the new HW specific STE infrastructure (Alaa Hleihel) [1915308]
- net/mlx5: DR, Move macros from dr_ste.c to header (Alaa Hleihel) [1915308]
- net/mlx5: DR, Add infrastructure for supporting several steering formats (Alaa Hleihel) [1915308]
- ext4: handle read only external journal device (Lukas Czerner) [1860929]
- vfio/pci: Handle concurrent vma faults (Alex Williamson) [1976551]
- vfio-pci: Use io_remap_pfn_range() for PCI IO memory (Alex Williamson) [1976551]
- net: flow_dissector: Parse PTP L2 packet header (Ivan Vecera) [1971687]
- net: vlan: Add parse protocol header ops (Ivan Vecera) [1971687]
- PCI: hv: Remove bus device removal unused refcount/functions (Mohammed Gamal) [1982482]
- PCI: hv: Fix a race condition when removing the device (Mohammed Gamal) [1982482]
- RDMA/cxgb4: check for ipv6 address properly while destroying listener (Raju Rangoju) [1961313]
- RDMA/cxgb4: Fix adapter LE hash errors while destroying ipv6 listening server (Raju Rangoju) [1961313]
- RDMA: Convert sysfs device * show functions to use sysfs_emit() (Raju Rangoju) [1961313]
- RDMA/iw_cxgb4: Disable delayed ack by default (Raju Rangoju) [1961313]
- [infiniband] treewide: Remove uninitialized_var() usage (Raju Rangoju) [1961313]
- RDMA/iw_cxgb4: cleanup device debugfs entries on ULD remove (Raju Rangoju) [1961313]
- vfs: allow unprivileged whiteout creation (Miklos Szeredi) [1983608]
- Documentation: kdump: update kdump guide (Baoquan He) [1938067]
- docs: admin-guide: add kdump documentation into it (Baoquan He) [1938067]
- docs: kdump: convert docs to ReST and rename to *.rst (Baoquan He) [1938067]

* Wed Jul 14 2021 Augusto Caringi <acaringi@redhat.com> [4.18.0-323.el8]
- Revert "be2net: disable bh with spin_lock in be_process_mcc" (Petr Oros) [1973504]
- nvmet-rdma: Fix NULL deref when SEND is completed with error (Gopal Tiwari) [1959856]
- netfilter: nft_exthdr: Search chunks in SCTP packets only (Phil Sutter) [1868228]
- netfilter: nft_exthdr: Fix for unsafe packet data read (Phil Sutter) [1868228]
- netfilter: nft_exthdr: Support SCTP chunks (Phil Sutter) [1868228]
- xfs: fix deadlock retry tracepoint arguments (Bill O'Donnell) [1937116]
- xfs: fix xfs_reflink_unshare usage of filemap_write_and_wait_range (Bill O'Donnell) [1937116]
- xfs: fix return of uninitialized value in variable error (Bill O'Donnell) [1937116]
- xfs: fix scrub and remount-ro protection when running scrub (Bill O'Donnell) [1937116]
- xfs: fix xfs_trans slab cache name (Bill O'Donnell) [1937116]
- xfs: Fix a typo (Bill O'Donnell) [1937116]
- xfs: Rudimentary spelling fix (Bill O'Donnell) [1937116]
- xfs: Rudimentary typo fixes (Bill O'Donnell) [1937116]
- xfs: fix dquot scrub loop cancellation (Bill O'Donnell) [1937116]
- xfs: fix uninitialized variables in xrep_calc_ag_resblks (Bill O'Donnell) [1937116]
- xfs: fix incorrect root dquot corruption error when switching group/project quota types (Bill O'Donnell) [1937116]
- xfs: fix up build warnings when quotas are disabled (Bill O'Donnell) [1937116]
- xfs: fix chown leaking delalloc quota blocks when fssetxattr fails (Bill O'Donnell) [1937116]
- xfs: Fix assert failure in xfs_setattr_size() (Bill O'Donnell) [1937116]
- xfs: fix an ABBA deadlock in xfs_rename (Bill O'Donnell) [1937116 1966609]
- xfs: fix the forward progress assertion in xfs_iwalk_run_callbacks (Bill O'Donnell) [1937116]
- xfs: fix parent pointer scrubber bailing out on unallocated inodes (Bill O'Donnell) [1937116]
- xfs: revert "xfs: fix rmap key and record comparison functions" (Bill O'Donnell) [1937116]
- xfs: don't allow NOWAIT DIO across extent boundaries (Bill O'Donnell) [1937116]
- xfs: return corresponding errcode if xfs_initialize_perag() fail (Bill O'Donnell) [1937116]
- xfs: ensure inobt record walks always make forward progress (Bill O'Donnell) [1937116]
- xfs: fix forkoff miscalculation related to XFS_LITINO(mp) (Bill O'Donnell) [1937116]
- xfs: directory scrub should check the null bestfree entries too (Bill O'Donnell) [1937116]
- xfs: strengthen rmap record flags checking (Bill O'Donnell) [1937116]
- xfs: fix the minrecs logic when dealing with inode root child blocks (Bill O'Donnell) [1937116]
- xfs: fix a missing unlock on error in xfs_fs_map_blocks (Bill O'Donnell) [1937116]
- xfs: fix brainos in the refcount scrubber's rmap fragment processor (Bill O'Donnell) [1937116]
- xfs: fix rmap key and record comparison functions (Bill O'Donnell) [1937116]
- xfs: set the unwritten bit in rmap lookup flags in xchk_bmap_get_rmapextents (Bill O'Donnell) [1937116]
- xfs: fix flags argument to rmap lookup when converting shared file rmaps (Bill O'Donnell) [1937116]
- xfs: only flush the unshared range in xfs_reflink_unshare (Bill O'Donnell) [1937116]
- xfs: fix scrub flagging rtinherit even if there is no rt device (Bill O'Donnell) [1937116]
- xfs: fix missing CoW blocks writeback conversion retry (Bill O'Donnell) [1937116]
- xfs: set xefi_discard when creating a deferred agfl free log intent item (Bill O'Donnell) [1937116]
- xfs: cancel intents immediately if process_intents fails (Bill O'Donnell) [1937116]
- xfs: fix fallocate functions when rtextsize is larger than 1 (Bill O'Donnell) [1937116]
- xfs: fix high key handling in the rt allocator's query_range function (Bill O'Donnell) [1937116]
- xfs: annotate grabbing the realtime bitmap/summary locks in growfs (Bill O'Donnell) [1937116]
- xfs: make xfs_growfs_rt update secondary superblocks (Bill O'Donnell) [1937116]
- xfs: fix realtime bitmap/summary file truncation when growing rt volume (Bill O'Donnell) [1937116]
- xfs: fix the indent in xfs_trans_mod_dquot (Bill O'Donnell) [1937116]
- xfs: do the ASSERT for the arguments O_{u,g,p}dqpp (Bill O'Donnell) [1937116]
- xfs: fix deadlock and streamline xfs_getfsmap performance (Bill O'Donnell) [1937116]
- xfs: limit entries returned when counting fsmap records (Bill O'Donnell) [1937116]
- xfs: only relog deferred intent items if free space in the log gets low (Bill O'Donnell) [1937116]
- xfs: expose the log push threshold (Bill O'Donnell) [1937116]
- xfs: periodically relog deferred intent items (Bill O'Donnell) [1937116]
- xfs: change the order in which child and parent defer ops are finished (Bill O'Donnell) [1937116]
- xfs: fix an incore inode UAF in xfs_bui_recover (Bill O'Donnell) [1937116]
- xfs: clean up xfs_bui_item_recover iget/trans_alloc/ilock ordering (Bill O'Donnell) [1937116]
- xfs: clean up bmap intent item recovery checking (Bill O'Donnell) [1937116]
- xfs: xfs_defer_capture should absorb remaining transaction reservation (Bill O'Donnell) [1937116]
- xfs: xfs_defer_capture should absorb remaining block reservations (Bill O'Donnell) [1937116]
- xfs: proper replay of deferred ops queued during log recovery (Bill O'Donnell) [1937116]
- xfs: remove XFS_LI_RECOVERED (Bill O'Donnell) [1937116]
- xfs: remove xfs_defer_reset (Bill O'Donnell) [1937116]
- xfs: fix finobt btree block recovery ordering (Bill O'Donnell) [1937116]
- xfs: directly call xfs_generic_create() for ->create() and ->mkdir() (Bill O'Donnell) [1937116]
- xfs: avoid shared rmap operations for attr fork extents (Bill O'Donnell) [1937116]
- xfs: drop the obsolete comment on filestream locking (Bill O'Donnell) [1937116]
- xfs: code cleanup in xfs_attr_leaf_entsize_{remote,local} (Bill O'Donnell) [1937116]
- xfs: do the assert for all the log done items in xfs_trans_cancel (Bill O'Donnell) [1937116]
- xfs: remove the unused parameter id from xfs_qm_dqattach_one (Bill O'Donnell) [1937116]
- xfs: remove the redundant crc feature check in xfs_attr3_rmt_verify (Bill O'Donnell) [1937116]
- xfs: fix some comments (Bill O'Donnell) [1937116]
- xfs: remove the unnecessary xfs_dqid_t type cast (Bill O'Donnell) [1937116]
- xfs: use the existing type definition for di_projid (Bill O'Donnell) [1937116]
- xfs: remove the unused SYNCHRONIZE macro (Bill O'Donnell) [1937116]
- xfs: clean up calculation of LR header blocks (Bill O'Donnell) [1937116]
- xfs: avoid LR buffer overrun due to crafted h_len (Bill O'Donnell) [1937116]
- xfs: don't release log intent items when recovery fails (Bill O'Donnell) [1937116]
- xfs: attach inode to dquot in xfs_bui_item_recover (Bill O'Donnell) [1937116]
- xfs: log new intent items created as part of finishing recovered intent items (Bill O'Donnell) [1937116]
- xfs: check dabtree node hash values when loading child blocks (Bill O'Donnell) [1937116]
- xfs: don't free rt blocks when we're doing a REMAP bunmapi call (Bill O'Donnell) [1937116]
- xfs: Set xfs_buf's b_ops member when zeroing bitmap/summary files (Bill O'Donnell) [1937116]
- xfs: Set xfs_buf type flag when growing summary/bitmap files (Bill O'Donnell) [1937116]
- xfs: drop extra transaction roll from inode extent truncate (Bill O'Donnell) [1937116]
- xfs: don't propagate RTINHERIT -> REALTIME when there is no rtdev (Bill O'Donnell) [1937116]
- xfs: ensure that fpunch, fcollapse, and finsert operations are aligned to rt extent size (Bill O'Donnell) [1937116]
- xfs: refactor inode flags propagation code (Bill O'Donnell) [1937116]
- xfs: make sure the rt allocator doesn't run off the end (Bill O'Donnell) [1937116]
- xfs: Remove unneeded semicolon (Bill O'Donnell) [1937116]
- xfs: force the log after remapping a synchronous-writes file (Bill O'Donnell) [1937116]
- xfs: Convert xfs_attr_sf macros to inline functions (Bill O'Donnell) [1937116]
- xfs: Use variable-size array for nameval in xfs_attr_sf_entry (Bill O'Donnell) [1937116]
- xfs: Remove typedef xfs_attr_shortform_t (Bill O'Donnell) [1937116]
- xfs: remove typedef xfs_attr_sf_entry_t (Bill O'Donnell) [1937116]
- xfs: Remove kmem_zalloc_large() (Bill O'Donnell) [1937116]
- xfs: enable big timestamps (Bill O'Donnell) [1937116]
- xfs: trace timestamp limits (Bill O'Donnell) [1937116]
- xfs: widen ondisk quota expiration timestamps to handle y2038+ (Bill O'Donnell) [1937116]
- xfs: widen ondisk inode timestamps to deal with y2038+ (Bill O'Donnell) [1937116]
- xfs: redefine xfs_ictimestamp_t (Bill O'Donnell) [1937116]
- xfs: redefine xfs_timestamp_t (Bill O'Donnell) [1937116]
- xfs: move xfs_log_dinode_to_disk to the log recovery code (Bill O'Donnell) [1937116]
- xfs: refactor quota timestamp coding (Bill O'Donnell) [1937116]
- xfs: refactor default quota grace period setting code (Bill O'Donnell) [1937116]
- xfs: refactor quota expiration timer modification (Bill O'Donnell) [1937116]
- xfs: explicitly define inode timestamp range (Bill O'Donnell) [1937116]
- xfs: enable new inode btree counters feature (Bill O'Donnell) [1937116]
- xfs: support inode btree blockcounts in online repair (Bill O'Donnell) [1937116]
- xfs: support inode btree blockcounts in online scrub (Bill O'Donnell) [1937116]
- xfs: use the finobt block counts to speed up mount times (Bill O'Donnell) [1937116]
- xfs: store inode btree block counts in AGI header (Bill O'Donnell) [1937116]
- xfs: reuse _xfs_buf_read for re-reading the superblock (Bill O'Donnell) [1937116]
- xfs: remove xfs_getsb (Bill O'Donnell) [1937116]
- xfs: simplify xfs_trans_getsb (Bill O'Donnell) [1937116]
- xfs: remove xlog_recover_iodone (Bill O'Donnell) [1937116]
- xfs: clear the read/write flags later in xfs_buf_ioend (Bill O'Donnell) [1937116]
- xfs: use xfs_buf_item_relse in xfs_buf_item_done (Bill O'Donnell) [1937116]
- xfs: simplify the xfs_buf_ioend_disposition calling convention (Bill O'Donnell) [1937116]
- xfs: lift the XBF_IOEND_FAIL handling into xfs_buf_ioend_disposition (Bill O'Donnell) [1937116]
- xfs: remove xfs_buf_ioerror_retry (Bill O'Donnell) [1937116]
- xfs: refactor xfs_buf_ioerror_fail_without_retry (Bill O'Donnell) [1937116]
- xfs: fold xfs_buf_ioend_finish into xfs_ioend (Bill O'Donnell) [1937116]
- xfs: move the buffer retry logic to xfs_buf.c (Bill O'Donnell) [1937116]
- xfs: refactor xfs_buf_ioend (Bill O'Donnell) [1937116]
- xfs: mark xfs_buf_ioend static (Bill O'Donnell) [1937116]
- xfs: refactor the buf ioend disposition code (Bill O'Donnell) [1937116]
- xfs: xfs_iflock is no longer a completion (Bill O'Donnell) [1937116]
- xfs: remove kmem_realloc() (Bill O'Donnell) [1937116]
- xfs: don't update mtime on COW faults (Bill O'Donnell) [1937116]
- xfs: fix xfs_bmap_validate_extent_raw when checking attr fork of rt files (Bill O'Donnell) [1937116]
- xfs: initialize the shortform attr header padding entry (Bill O'Donnell) [1937116]
- xfs: fix off-by-one in inode alloc block reservation calculation (Bill O'Donnell) [1937116]
- xfs: finish dfops on every insert range shift iteration (Bill O'Donnell) [1937116]
- xfs: Fix UBSAN null-ptr-deref in xfs_sysfs_init (Bill O'Donnell) [1937116]
- xfs: delete duplicated words + other fixes (Bill O'Donnell) [1937116]
- xfs: Lift -ENOSPC handler from xfs_attr_leaf_addname (Bill O'Donnell) [1937116]
- xfs: Simplify xfs_attr_node_addname (Bill O'Donnell) [1937116]
- xfs: Simplify xfs_attr_leaf_addname (Bill O'Donnell) [1937116]
- xfs: Add helper function xfs_attr_node_removename_rmt (Bill O'Donnell) [1937116]
- xfs: Add helper function xfs_attr_node_removename_setup (Bill O'Donnell) [1937116]
- xfs: Add remote block helper functions (Bill O'Donnell) [1937116]
- xfs: Add helper function xfs_attr_leaf_mark_incomplete (Bill O'Donnell) [1937116]
- xfs: Add helpers xfs_attr_is_shortform and xfs_attr_set_shortform (Bill O'Donnell) [1937116]
- xfs: Remove xfs_trans_roll in xfs_attr_node_removename (Bill O'Donnell) [1937116]
- xfs: Remove unneeded xfs_trans_roll_inode calls (Bill O'Donnell) [1937116]
- xfs: Add helper function xfs_attr_node_shrink (Bill O'Donnell) [1937116]
- xfs: Pull up xfs_attr_rmtval_invalidate (Bill O'Donnell) [1937116]
- xfs: Refactor xfs_attr_rmtval_remove (Bill O'Donnell) [1937116]
- xfs: Pull up trans roll in xfs_attr3_leaf_clearflag (Bill O'Donnell) [1937116]
- xfs: Factor out xfs_attr_rmtval_invalidate (Bill O'Donnell) [1937116]
- xfs: Pull up trans roll from xfs_attr3_leaf_setflag (Bill O'Donnell) [1937116]
- xfs: Refactor xfs_attr_try_sf_addname (Bill O'Donnell) [1937116]
- xfs: Split apart xfs_attr_leaf_addname (Bill O'Donnell) [1937116]
- xfs: Pull up trans handling in xfs_attr3_leaf_flipflags (Bill O'Donnell) [1937116]
- xfs: Factor out new helper functions xfs_attr_rmtval_set (Bill O'Donnell) [1937116]
- xfs: Check for -ENOATTR or -EEXIST (Bill O'Donnell) [1937116]
- xfs: Add xfs_has_attr and subroutines (Bill O'Donnell) [1937116]
- xfs: Refactor xfs_da_state_alloc() helper (Bill O'Donnell) [1937116]
- xfs: remove xfs_zone_{alloc,zalloc} helpers (Bill O'Donnell) [1937116]
- xfs: Modify xlog_ticket_alloc() to use kernel's MM API (Bill O'Donnell) [1937116]
- xfs: Remove kmem_zone_zalloc() usage (Bill O'Donnell) [1937116]
- xfs: Remove kmem_zone_alloc() usage (Bill O'Donnell) [1937116]
- xfs: xfs_btree_staging.h: delete duplicated words (Bill O'Donnell) [1937116]
- xfs: rename the ondisk dquot d_flags to d_type (Bill O'Donnell) [1937116]
- xfs: improve ondisk dquot flags checking (Bill O'Donnell) [1937116]
- xfs: create xfs_dqtype_t to represent quota types (Bill O'Donnell) [1937116]
- xfs: replace a few open-coded XFS_DQTYPE_REC_MASK uses (Bill O'Donnell) [1937116]
- xfs: remove unnecessary quota type masking (Bill O'Donnell) [1937116]
- xfs: always use xfs_dquot_type when extracting type from a dquot (Bill O'Donnell) [1937116]
- xfs: refactor quota type testing (Bill O'Donnell) [1937116]
- xfs: remove the XFS_QM_IS[UGP]DQ macros (Bill O'Donnell) [1937116]
- xfs: refactor testing if a particular dquot is being enforced (Bill O'Donnell) [1937116]
- xfs: rename XFS_DQ_{USER,GROUP,PROJ} to XFS_DQTYPE_* (Bill O'Donnell) [1937116]
- xfs: drop the type parameter from xfs_dquot_verify (Bill O'Donnell) [1937116]
- xfs: add more dquot tracepoints (Bill O'Donnell) [1937116]
- xfs: actually bump warning counts when we send warnings (Bill O'Donnell) [1937116]
- xfs: assume the default quota limits are always set in xfs_qm_adjust_dqlimits (Bill O'Donnell) [1937116]
- xfs: refactor xfs_trans_apply_dquot_deltas (Bill O'Donnell) [1937116]
- xfs: refactor xfs_trans_dqresv (Bill O'Donnell) [1937116]
- xfs: refactor xfs_qm_scall_setqlim (Bill O'Donnell) [1937116]
- xfs: refactor quota exceeded test (Bill O'Donnell) [1937116]
- xfs: remove unnecessary arguments from quota adjust functions (Bill O'Donnell) [1937116]
- xfs: refactor default quota limits by resource (Bill O'Donnell) [1937116]
- xfs: remove qcore from incore dquots (Bill O'Donnell) [1937116]
- xfs: stop using q_core timers in the quota code (Bill O'Donnell) [1937116]
- xfs: stop using q_core warning counters in the quota code (Bill O'Donnell) [1937116]
- xfs: stop using q_core counters in the quota code (Bill O'Donnell) [1937116]
- xfs: stop using q_core limits in the quota code (Bill O'Donnell) [1937116]
- xfs: use a per-resource struct for incore dquot data (Bill O'Donnell) [1937116]
- xfs: stop using q_core.d_id in the quota code (Bill O'Donnell) [1937116]
- xfs: stop using q_core.d_flags in the quota code (Bill O'Donnell) [1937116]
- xfs: make XFS_DQUOT_CLUSTER_SIZE_FSB part of the ondisk format (Bill O'Donnell) [1937116]
- xfs: rename dquot incore state flags (Bill O'Donnell) [1937116]
- xfs: refactor quotacheck flags usage (Bill O'Donnell) [1937116]
- xfs: move the flags argument of xfs_qm_scall_trunc_qfiles to XFS_QMOPT_* (Bill O'Donnell) [1937116]
- xfs: validate ondisk/incore dquot flags (Bill O'Donnell) [1937116]
- xfs: fix inode quota reservation checks (Bill O'Donnell) [1937116]
- xfs: clear XFS_DQ_FREEING if we can't lock the dquot buffer to flush (Bill O'Donnell) [1937116]
- xfs: drain the buf delwri queue before xfsaild idles (Bill O'Donnell) [1937116]
- xfs: remove duplicated include from xfs_buf_item.c (Bill O'Donnell) [1937116]
- xfs: remove SYNC_WAIT and SYNC_TRYLOCK (Bill O'Donnell) [1937116]
- xfs: get rid of unnecessary xfs_perag_{get,put} pairs (Bill O'Donnell) [1937116]
- xfs: rtbitmap scrubber should check inode size (Bill O'Donnell) [1937116]
- xfs: rtbitmap scrubber should verify written extents (Bill O'Donnell) [1937116]
- xfs: remove xfs_inobp_check() (Bill O'Donnell) [1937116]
- xfs: factor xfs_iflush_done (Bill O'Donnell) [1937116]
- xfs: rework xfs_iflush_cluster() dirty inode iteration (Bill O'Donnell) [1937116]
- xfs: rename xfs_iflush_int() (Bill O'Donnell) [1937116]
- xfs: xfs_iflush() is no longer necessary (Bill O'Donnell) [1937116]
- xfs: attach inodes to the cluster buffer when dirtied (Bill O'Donnell) [1937116]
- xfs: rework stale inodes in xfs_ifree_cluster (Bill O'Donnell) [1937116]
- xfs: clean up inode reclaim comments (Bill O'Donnell) [1937116]
- xfs: remove SYNC_WAIT from xfs_reclaim_inodes() (Bill O'Donnell) [1937116]
- xfs: remove SYNC_TRYLOCK from inode reclaim (Bill O'Donnell) [1937116]
- xfs: don't block inode reclaim on the ILOCK (Bill O'Donnell) [1937116]
- xfs: allow multiple reclaimers per AG (Bill O'Donnell) [1937116]
- xfs: remove IO submission from xfs_reclaim_inode() (Bill O'Donnell) [1937116]
- xfs: make inode reclaim almost non-blocking (Bill O'Donnell) [1937116]
- xfs: pin inode backing buffer to the inode log item (Bill O'Donnell) [1937116]
- xfs: move xfs_clear_li_failed out of xfs_ail_delete_one() (Bill O'Donnell) [1937116]
- xfs: unwind log item error flagging (Bill O'Donnell) [1937116]
- xfs: handle buffer log item IO errors directly (Bill O'Donnell) [1937116]
- xfs: get rid of log item callbacks (Bill O'Donnell) [1937116]
- xfs: clean up the buffer iodone callback functions (Bill O'Donnell) [1937116]
- xfs: use direct calls for dquot IO completion (Bill O'Donnell) [1937116]
- xfs: make inode IO completion buffer centric (Bill O'Donnell) [1937116]
- xfs: clean up whacky buffer log item list reinit (Bill O'Donnell) [1937116]
- xfs: call xfs_buf_iodone directly (Bill O'Donnell) [1937116]
- xfs: mark log recovery buffers for completion (Bill O'Donnell) [1937116]
- xfs: mark dquot buffers in cache (Bill O'Donnell) [1937116]
- xfs: mark inode buffers in cache (Bill O'Donnell) [1937116]
- xfs: add an inode item lock (Bill O'Donnell) [1937116]
- xfs: remove logged flag from inode log item (Bill O'Donnell) [1937116]
- xfs: Don't allow logging of XFS_ISTALE inodes (Bill O'Donnell) [1937116]
- xfs: remove useless definitions in xfs_linux.h (Bill O'Donnell) [1937116]
- xfs: use MMAPLOCK around filemap_map_pages() (Bill O'Donnell) [1937116]
- xfs: move helpers that lock and unlock two inodes against userspace IO (Bill O'Donnell) [1937116]
- xfs: refactor locking and unlocking two inodes against userspace IO (Bill O'Donnell) [1937116]
- xfs: fix xfs_reflink_remap_prep calling conventions (Bill O'Donnell) [1937116]
- xfs: reflink can skip remap existing mappings (Bill O'Donnell) [1937116]
- xfs: only reserve quota blocks if we're mapping into a hole (Bill O'Donnell) [1937116]
- xfs: only reserve quota blocks for bmbt changes if we're changing the data fork (Bill O'Donnell) [1937116]
- xfs: redesign the reflink remap loop to fix blkres depletion crash (Bill O'Donnell) [1937116]
- xfs: rename xfs_bmap_is_real_extent to is_written_extent (Bill O'Donnell) [1937116]
- xfs: Couple of typo fixes in comments (Bill O'Donnell) [1937116]
- xfs: fix use-after-free on CIL context on shutdown (Bill O'Donnell) [1937116]
- xfs: flag files as supporting buffered async reads (Bill O'Donnell) [1937116]
- fs: add FMODE_BUF_RASYNC (Bill O'Donnell) [1937116]
- xfs: more lockdep whackamole with kmem_alloc* (Bill O'Donnell) [1937116 1965923]
- xfs: rearrange xfs_inode_walk_ag parameters (Bill O'Donnell) [1937116]
- xfs: straighten out all the naming around incore inode tree walks (Bill O'Donnell) [1937116]
- xfs: move xfs_inode_ag_iterator to be closer to the perag walking code (Bill O'Donnell) [1937116]
- xfs: use bool for done in xfs_inode_ag_walk (Bill O'Donnell) [1937116]
- xfs: fix inode ag walk predicate function return values (Bill O'Donnell) [1937116]
- xfs: refactor eofb matching into a single helper (Bill O'Donnell) [1937116]
- xfs: remove __xfs_icache_free_eofblocks (Bill O'Donnell) [1937116]
- xfs: remove flags argument from xfs_inode_ag_walk (Bill O'Donnell) [1937116]
- xfs: remove xfs_inode_ag_iterator_flags (Bill O'Donnell) [1937116]
- xfs: remove unused xfs_inode_ag_iterator function (Bill O'Donnell) [1937116]
- xfs: replace open-coded XFS_ICI_NO_TAG (Bill O'Donnell) [1937116]
- xfs: move eofblocks conversion function to xfs_ioctl.c (Bill O'Donnell) [1937116]
- xfs: remove the m_active_trans counter (Bill O'Donnell) [1937116]
- xfs: separate read-only variables in struct xfs_mount (Bill O'Donnell) [1937116]
- xfs: reduce free inode accounting overhead (Bill O'Donnell) [1937116]
- xfs: gut error handling in xfs_trans_unreserve_and_mod_sb() (Bill O'Donnell) [1937116]
- xfs: cleanup xfs_idestroy_fork (Bill O'Donnell) [1937116]
- xfs: move the fork format fields into struct xfs_ifork (Bill O'Donnell) [1937116]
- xfs: move the per-fork nextents fields into struct xfs_ifork (Bill O'Donnell) [1937116]
- xfs: remove xfs_ifree_local_data (Bill O'Donnell) [1937116]
- xfs: remove the XFS_DFORK_Q macro (Bill O'Donnell) [1937116]
- xfs: clean up xchk_bmap_check_rmaps usage of XFS_IFORK_Q (Bill O'Donnell) [1937116]
- xfs: remove the NULL fork handling in xfs_bmapi_read (Bill O'Donnell) [1937116]
- xfs: remove the special COW fork handling in xfs_bmapi_read (Bill O'Donnell) [1937116]
- xfs: improve local fork verification (Bill O'Donnell) [1937116]
- xfs: refactor xfs_inode_verify_forks (Bill O'Donnell) [1937116]
- xfs: remove xfs_ifork_ops (Bill O'Donnell) [1937116]
- xfs: remove xfs_iread (Bill O'Donnell) [1937116]
- xfs: don't reset i_delayed_blks in xfs_iread (Bill O'Donnell) [1937116]
- xfs: call xfs_dinode_verify from xfs_inode_from_disk (Bill O'Donnell) [1937116]
- xfs: handle unallocated inodes in xfs_inode_from_disk (Bill O'Donnell) [1937116]
- xfs: split xfs_iformat_fork (Bill O'Donnell) [1937116]
- xfs: call xfs_iformat_fork from xfs_inode_from_disk (Bill O'Donnell) [1937116]
- xfs: xfs_bmapi_read doesn't take a fork id as the last argument (Bill O'Donnell) [1937116]
- xfs: fix the warning message in xfs_validate_sb_common() (Bill O'Donnell) [1937116]
- xfs: don't allow SWAPEXT if we'd screw up quota accounting (Bill O'Donnell) [1937116]
- xfs: use ordered buffers to initialize dquot buffers during quotacheck (Bill O'Donnell) [1937116]
- xfs: don't fail verifier on empty attr3 leaf block (Bill O'Donnell) [1937116]
- xfs: Use the correct style for SPDX License Identifier (Bill O'Donnell) [1937116]
- xfs: Replace zero-length array with flexible-array (Bill O'Donnell) [1937116]
- xfs: ensure f_bfree returned by statfs() is non-negative (Bill O'Donnell) [1937116]
- xfs: remove duplicate headers (Bill O'Donnell) [1937116]
- xfs: fix unused variable warning in buffer completion on !DEBUG (Bill O'Donnell) [1937116]
- xfs: remove unnecessary includes from xfs_log_recover.c (Bill O'Donnell) [1937116]
- xfs: move log recovery buffer cancellation code to xfs_buf_item_recover.c (Bill O'Donnell) [1937116]
- xfs: hoist setting of XFS_LI_RECOVERED to caller (Bill O'Donnell) [1937116]
- xfs: refactor intent item iop_recover calls (Bill O'Donnell) [1937116]
- xfs: refactor intent item RECOVERED flag into the log item (Bill O'Donnell) [1937116]
- xfs: refactor adding recovered intent items to the log (Bill O'Donnell) [1937116]
- xfs: refactor releasing finished intents during log recovery (Bill O'Donnell) [1937116]
- xfs: refactor xlog_item_is_intent now that we're done converting (Bill O'Donnell) [1937116]
- xfs: refactor recovered BUI log item playback (Bill O'Donnell) [1937116]
- xfs: refactor recovered CUI log item playback (Bill O'Donnell) [1937116]
- xfs: refactor recovered RUI log item playback (Bill O'Donnell) [1937116]
- xfs: refactor recovered EFI log item playback (Bill O'Donnell) [1937116]
- xfs: remove log recovery quotaoff item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery BUI item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery CUI item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery RUI item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery EFI item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery icreate item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery dquot item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery inode item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery buffer item dispatch for pass2 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery item dispatch for pass1 commit functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery item dispatch for pass2 readhead functions (Bill O'Donnell) [1937116]
- xfs: refactor log recovery item sorting into a generic dispatch structure (Bill O'Donnell) [1937116]
- xfs: convert xfs_log_recover_item_t to struct xfs_log_recover_item (Bill O'Donnell) [1937116]
- xfs: remove unused iget_flags param from xfs_imap_to_bp() (Bill O'Donnell) [1937116]
- xfs: remove unused shutdown types (Bill O'Donnell) [1937116]
- xfs: random buffer write failure errortag (Bill O'Donnell) [1937116]
- xfs: remove unused iflush stale parameter (Bill O'Donnell) [1937116]
- xfs: combine xfs_trans_ail_[remove|delete]() (Bill O'Donnell) [1937116]
- xfs: drop unused shutdown parameter from xfs_trans_ail_remove() (Bill O'Donnell) [1937116]
- xfs: use delete helper for items expected to be in AIL (Bill O'Donnell) [1937116]
- xfs: acquire ->ail_lock from xfs_trans_ail_delete() (Bill O'Donnell) [1937116]
- xfs: abort consistently on dquot flush failure (Bill O'Donnell) [1937116]
- xfs: ratelimit unmount time per-buffer I/O error alert (Bill O'Donnell) [1937116]
- xfs: refactor ratelimited buffer error messages into helper (Bill O'Donnell) [1937116]
- xfs: reset buffer write failure state on successful completion (Bill O'Donnell) [1937116]
- xfs: remove unnecessary shutdown check from xfs_iflush() (Bill O'Donnell) [1937116]
- xfs: simplify inode flush error handling (Bill O'Donnell) [1937116]
- xfs: factor out buffer I/O failure code (Bill O'Donnell) [1937116]
- xfs: refactor failed buffer resubmission into xfsaild (Bill O'Donnell) [1937116]
- xfs: spell out the parameter name for ->cancel_item (Bill O'Donnell) [1937116]
- xfs: use a xfs_btree_cur for the ->finish_cleanup state (Bill O'Donnell) [1937116]
- xfs: turn dfp_done into a xfs_log_item (Bill O'Donnell) [1937116]
- xfs: refactor xfs_defer_finish_noroll (Bill O'Donnell) [1937116]
- xfs: turn dfp_intent into a xfs_log_item (Bill O'Donnell) [1937116]
- xfs: merge the ->diff_items defer op into ->create_intent (Bill O'Donnell) [1937116]
- xfs: merge the ->log_item defer op into ->create_intent (Bill O'Donnell) [1937116]
- xfs: factor out a xfs_defer_create_intent helper (Bill O'Donnell) [1937116]
- xfs: remove the xfs_inode_log_item_t typedef (Bill O'Donnell) [1937116]
- xfs: remove the xfs_efd_log_item_t typedef (Bill O'Donnell) [1937116]
- xfs: remove the xfs_efi_log_item_t typedef (Bill O'Donnell) [1937116]
- xfs: refactor xlog_recover_buffer_pass1 (Bill O'Donnell) [1937116]
- xfs: simplify xlog_recover_inode_ra_pass2 (Bill O'Donnell) [1937116]
- xfs: factor out a xlog_buf_readahead helper (Bill O'Donnell) [1937116]
- xfs: rename inode_list xlog_recover_reorder_trans (Bill O'Donnell) [1937116]
- xfs: refactor the buffer cancellation table helpers (Bill O'Donnell) [1937116]
- xfs: define printk_once variants for xfs messages (Bill O'Donnell) [1937116]
- xfs: stop CONFIG_XFS_DEBUG from changing compiler flags (Bill O'Donnell) [1937116]
- xfs: remove unnecessary check of the variable resblks in xfs_symlink (Bill O'Donnell) [1937116]
- xfs: simplify the flags setting in xfs_qm_scall_quotaon (Bill O'Donnell) [1937116]
- xfs: remove unnecessary assertion from xfs_qm_vop_create_dqattach (Bill O'Donnell) [1937116]
- xfs: remove unnecessary variable udqp from xfs_ioctl_setattr (Bill O'Donnell) [1937116]
- xfs: reserve quota inode transaction space only when needed (Bill O'Donnell) [1937116]
- xfs: combine two if statements with same condition (Bill O'Donnell) [1937116]
- xfs: trace quota allocations for all quota types (Bill O'Donnell) [1937116]
- xfs: report unrecognized log item type codes during recovery (Bill O'Donnell) [1937116]
- xfs: move inode flush to the sync workqueue (Bill O'Donnell) [1937116]
- xfs: fix partially uninitialized structure in xfs_reflink_remap_extent (Bill O'Donnell) [1937116]
- xfs: fix inode number overflow in ifree cluster helper (Bill O'Donnell) [1937116]
- xfs: remove redundant variable assignment in xfs_symlink() (Bill O'Donnell) [1937116]
- xfs: ratelimit inode flush on buffered write ENOSPC (Bill O'Donnell) [1937116]
- xfs: return locked status of inode buffer on xfsaild push (Bill O'Donnell) [1937116]
- xfs: trylock underlying buffer on dquot flush (Bill O'Donnell) [1937116]
- xfs: remove unnecessary ternary from xfs_create (Bill O'Donnell) [1937116]
- xfs: don't write a corrupt unmount record to force summary counter recalc (Bill O'Donnell) [1937116]
- xfs: factor inode lookup from xfs_ifree_cluster (Bill O'Donnell) [1937116]
- xfs: tail updates only need to occur when LSN changes (Bill O'Donnell) [1937116]
- xfs: factor common AIL item deletion code (Bill O'Donnell) [1937116]
- xfs: correctly acount for reclaimable slabs (Bill O'Donnell) [1937116]
- xfs: Improve metadata buffer reclaim accountability (Bill O'Donnell) [1937116]
- xfs: don't allow log IO to be throttled (Bill O'Donnell) [1937116]
- xfs: Throttle commits on delayed background CIL push (Bill O'Donnell) [1937116]
- xfs: Lower CIL flush limit for large logs (Bill O'Donnell) [1937116]
- xfs: remove some stale comments from the log code (Bill O'Donnell) [1937116]
- xfs: refactor unmount record writing (Bill O'Donnell) [1937116]
- xfs: merge xlog_commit_record with xlog_write_done (Bill O'Donnell) [1937116]
- xfs: split xlog_ticket_done (Bill O'Donnell) [1937116]
- xfs: kill XLOG_TIC_INITED (Bill O'Donnell) [1937116]
- xfs: refactor and split xfs_log_done() (Bill O'Donnell) [1937116]
- xfs: re-order initial space accounting checks in xlog_write (Bill O'Donnell) [1937116]
- xfs: don't try to write a start record into every iclog (Bill O'Donnell) [1937116]
- xfs: validate the realtime geometry in xfs_validate_sb_common (Bill O'Donnell) [1937116]
- xfs: shutdown on failure to add page to log bio (Bill O'Donnell) [1937116]
- xfs: directory bestfree check should release buffers (Bill O'Donnell) [1937116]
- xfs: drop all altpath buffers at the end of the sibling check (Bill O'Donnell) [1937116]
- xfs: remove xlog_state_want_sync (Bill O'Donnell) [1937116]
- xfs: move the ioerror check out of xlog_state_clean_iclog (Bill O'Donnell) [1937116]
- xfs: refactor xlog_state_clean_iclog (Bill O'Donnell) [1937116]
- xfs: remove the aborted parameter to xlog_state_done_syncing (Bill O'Donnell) [1937116]
- xfs: simplify log shutdown checking in xfs_log_release_iclog (Bill O'Donnell) [1937116]
- xfs: simplify the xfs_log_release_iclog calling convention (Bill O'Donnell) [1937116]
- xfs: factor out a xlog_wait_on_iclog helper (Bill O'Donnell) [1937116]
- xfs: merge xlog_cil_push into xlog_cil_push_work (Bill O'Donnell) [1937116]
- xfs: remove the di_version field from struct icdinode (Bill O'Donnell) [1937116]
- xfs: simplify a check in xfs_ioctl_setattr_check_cowextsize (Bill O'Donnell) [1937116]
- xfs: simplify di_flags2 inheritance in xfs_ialloc (Bill O'Donnell) [1937116]
- xfs: only check the superblock version for dinode size calculation (Bill O'Donnell) [1937116]
- xfs: add a new xfs_sb_version_has_v3inode helper (Bill O'Donnell) [1937116]
- xfs: add support for rmap btree staging cursors (Bill O'Donnell) [1937116]
- xfs: add support for refcount btree staging cursors (Bill O'Donnell) [1937116]
- xfs: add support for inode btree staging cursors (Bill O'Donnell) [1937116]
- xfs: add support for free space btree staging cursors (Bill O'Donnell) [1937116]
- xfs: support bulk loading of staged btrees (Bill O'Donnell) [1937116]
- xfs: introduce fake roots for inode-rooted btrees (Bill O'Donnell) [1937116]
- xfs: introduce fake roots for ag-rooted btrees (Bill O'Donnell) [1937116]
- xfs: replace open-coded bitmap weight logic (Bill O'Donnell) [1937116]
- xfs: rename xfs_bitmap to xbitmap (Bill O'Donnell) [1937116]
- xfs: xrep_reap_extents should not destroy the bitmap (Bill O'Donnell) [1937116]
- xfs: cleanup xfs_log_unmount_write (Bill O'Donnell) [1937116]
- xfs: remove dead code from xfs_log_unmount_write (Bill O'Donnell) [1937116]
- xfs: remove the unused return value from xfs_log_unmount_write (Bill O'Donnell) [1937116]
- xfs: remove the unused XLOG_UNMOUNT_REC_TYPE define (Bill O'Donnell) [1937116]
- xfs: mark XLOG_FORCED_SHUTDOWN as unlikely (Bill O'Donnell) [1937116]
- xfs: make the btree ag cursor private union anonymous (Bill O'Donnell) [1937116]
- xfs: make the btree cursor union members named structure (Bill O'Donnell) [1937116]
- xfs: make btree cursor private union anonymous (Bill O'Donnell) [1937116]
- xfs: rename btree cursor private btree member flags (Bill O'Donnell) [1937116]
- xfs: convert btree cursor inode-private member names (Bill O'Donnell) [1937116]
- xfs: convert btree cursor ag-private member name (Bill O'Donnell) [1937116]
- xfs: introduce new private btree cursor names (Bill O'Donnell) [1937116]
- xfs: Use scnprintf() for avoiding potential buffer overflow (Bill O'Donnell) [1937116]
- xfs: mark extended attr corrupt when lookup-by-hash fails (Bill O'Donnell) [1937116]
- xfs: mark dir corrupt when lookup-by-hash fails (Bill O'Donnell) [1937116]
- xfs: check owner of dir3 blocks (Bill O'Donnell) [1937116]
- xfs: check owner of dir3 data blocks (Bill O'Donnell) [1937116]
- xfs: check owner of dir3 free blocks (Bill O'Donnell) [1937116]
- xfs: fix buffer corruption reporting when xfs_dir3_free_header_check fails (Bill O'Donnell) [1937116]
- xfs: xfs_buf_corruption_error should take __this_address (Bill O'Donnell) [1937116]
- xfs: add a function to deal with corrupt buffers post-verifiers (Bill O'Donnell) [1937116]
- xfs: remove XFS_BUF_TO_SBP (Bill O'Donnell) [1937116]
- xfs: remove XFS_BUF_TO_AGF (Bill O'Donnell) [1937116]
- xfs: remove XFS_BUF_TO_AGI (Bill O'Donnell) [1937116]
- xfs: remove the xfs_agfl_t typedef (Bill O'Donnell) [1937116]
- xfs: remove the agfl_bno member from struct xfs_agfl (Bill O'Donnell) [1937116]
- xfs: switch xfs_attrmulti_attr_get to lazy attr buffer allocation (Bill O'Donnell) [1937116]
- xfs: only allocate the buffer size actually needed in __xfs_set_acl (Bill O'Donnell) [1937116]
- xfs: clean up bufsize alignment in xfs_ioc_attr_list (Bill O'Donnell) [1937116]
- xfs: embedded the attrlist cursor into struct xfs_attr_list_context (Bill O'Donnell) [1937116]
- xfs: remove XFS_DA_OP_INCOMPLETE (Bill O'Donnell) [1937116]
- xfs: clean up the attr flag confusion (Bill O'Donnell) [1937116]
- xfs: clean up the ATTR_REPLACE checks (Bill O'Donnell) [1937116]
- xfs: improve xfs_forget_acl (Bill O'Donnell) [1937116]
- xfs: lift cursor copy in/out into xfs_ioc_attr_list (Bill O'Donnell) [1937116]
- xfs: lift buffer allocation into xfs_ioc_attr_list (Bill O'Donnell) [1937116]
- xfs: lift common checks into xfs_ioc_attr_list (Bill O'Donnell) [1937116]
- xfs: rename xfs_attr_list_int to xfs_attr_list (Bill O'Donnell) [1937116]
- xfs: move the legacy xfs_attr_list to xfs_ioctl.c (Bill O'Donnell) [1937116]
- xfs: open code ATTR_ENTSIZE (Bill O'Donnell) [1937116]
- xfs: remove the unused ATTR_ENTRY macro (Bill O'Donnell) [1937116]
- xfs: cleanup struct xfs_attr_list_context (Bill O'Donnell) [1937116]
- xfs: factor out a xfs_attr_match helper (Bill O'Donnell) [1937116]
- xfs: replace ATTR_KERNOTIME with XFS_DA_OP_NOTIME (Bill O'Donnell) [1937116]
- xfs: remove ATTR_ALLOC and XFS_DA_OP_ALLOCVAL (Bill O'Donnell) [1937116]
- xfs: remove ATTR_KERNOVAL (Bill O'Donnell) [1937116]
- xfs: remove the xfs_inode argument to xfs_attr_get_ilocked (Bill O'Donnell) [1937116]
- xfs: pass an initialized xfs_da_args to xfs_attr_get (Bill O'Donnell) [1937116]
- xfs: pass an initialized xfs_da_args structure to xfs_attr_set (Bill O'Donnell) [1937116]
- xfs: turn xfs_da_args.value into a void pointer (Bill O'Donnell) [1937116]
- xfs: remove the MAXNAMELEN check from xfs_attr_args_init (Bill O'Donnell) [1937116]
- xfs: remove the name == NULL check from xfs_attr_args_init (Bill O'Donnell) [1937116]
- xfs: factor out a helper for a single XFS_IOC_ATTRMULTI_BY_HANDLE op (Bill O'Donnell) [1937116]
- xfs: use strndup_user in XFS_IOC_ATTRMULTI_BY_HANDLE (Bill O'Donnell) [1937116]
- xfs: merge xfs_attrmulti_attr_remove into xfs_attrmulti_attr_set (Bill O'Donnell) [1937116]
- xfs: merge xfs_attr_remove into xfs_attr_set (Bill O'Donnell) [1937116]
- xfs: remove the ATTR_INCOMPLETE flag (Bill O'Donnell) [1937116]
- xfs: reject invalid flags combinations in XFS_IOC_ATTRLIST_BY_HANDLE (Bill O'Donnell) [1937116]
- xfs: rework collapse range into an atomic operation (Bill O'Donnell) [1937116]
- xfs: rework insert range into an atomic operation (Bill O'Donnell) [1937116]
- xfs: open code insert range extent split helper (Bill O'Donnell) [1937116]
- xfs: Add missing annotation to xfs_ail_check() (Bill O'Donnell) [1937116]
- xfs: fix an undefined behaviour in _da3_path_shift (Bill O'Donnell) [1937116]
- xfs: ratelimit xfs_discard_page messages (Bill O'Donnell) [1937116]
- xfs: ratelimit xfs_buf_ioerror_alert messages (Bill O'Donnell) [1937116]
- xfs: remove the kuid/kgid conversion wrappers (Bill O'Donnell) [1937116]
- xfs: remove the icdinode di_uid/di_gid members (Bill O'Donnell) [1937116]
- xfs: ensure that the inode uid/gid match values match the icdinode ones (Bill O'Donnell) [1937116]
- xfs: improve error message when we can't allocate memory for xfs_buf (Bill O'Donnell) [1937116]
- ceph: fix test for whether we can skip read when writing beyond EOF (Jeff Layton) [1971101]
- redhat/configs: Add CONFIG_X86_SGX_KVM (Bandan Das) [1494649]
- KVM: x86: Fix implicit enum conversion goof in scattered reverse CPUID code (Bandan Das) [1494649]
- KVM: x86: Add capability to grant VM access to privileged SGX attribute (Bandan Das) [1494649]
- KVM: VMX: Enable SGX virtualization for SGX1, SGX2 and LC (Bandan Das) [1494649]
- KVM: VMX: Add ENCLS[EINIT] handler to support SGX Launch Control (LC) (Bandan Das) [1494649]
- KVM: VMX: Add emulation of SGX Launch Control LE hash MSRs (Bandan Das) [1494649]
- KVM: VMX: Add SGX ENCLS[ECREATE] handler to enforce CPUID restrictions (Bandan Das) [1494649]
- KVM: VMX: Frame in ENCLS handler for SGX virtualization (Bandan Das) [1494649]
- KVM: VMX: Add basic handling of VM-Exit from SGX enclave (Bandan Das) [1494649]
- KVM: x86: Add reverse-CPUID lookup support for scattered SGX features (Bandan Das) [1494649]
- KVM: x86: Add support for reverse CPUID lookup of scattered features (Bandan Das) [1494649]
- KVM: x86: Define new #PF SGX error code bit (Bandan Das) [1494649]
- KVM: x86: Export kvm_mmu_gva_to_gpa_{read,write}() for SGX (VMX) (Bandan Das) [1494649]
- x86/sgx: Mark sgx_vepc_vm_ops static (Bandan Das) [1494649]
- x86/sgx: Do not update sgx_nr_free_pages in sgx_setup_epc_section() (Bandan Das) [1494649]
- x86/sgx: Move provisioning device creation out of SGX driver (Bandan Das) [1494649]
- x86/sgx: Add helpers to expose ECREATE and EINIT to KVM (Bandan Das) [1494649]
- x86/sgx: Add helper to update SGX_LEPUBKEYHASHn MSRs (Bandan Das) [1494649]
- x86/sgx: Add encls_faulted() helper (Bandan Das) [1494649]
- x86/sgx: Add SGX2 ENCLS leaf definitions (EAUG, EMODPR and EMODT) (Bandan Das) [1494649]
- x86/sgx: Move ENCLS leaf definitions to sgx.h (Bandan Das) [1494649]
- x86/sgx: Expose SGX architectural definitions to the kernel (Bandan Das) [1494649]
- x86/sgx: Initialize virtual EPC driver even when SGX driver is disabled (Bandan Das) [1494649]
- x86/cpu/intel: Allow SGX virtualization without Launch Control support (Bandan Das) [1494649]
- x86/sgx: Introduce virtual EPC for use by KVM guests (Bandan Das) [1494649]
- x86/sgx: Add SGX_CHILD_PRESENT hardware error code (Bandan Das) [1494649]
- x86/sgx: Wipe out EREMOVE from sgx_free_epc_page() (Bandan Das) [1494649]
- x86/cpufeatures: Add SGX1 and SGX2 sub-features (Bandan Das) [1494649]
- x86/cpufeatures: Make SGX_LC feature bit depend on SGX bit (Bandan Das) [1494649]
- x86/sgx: Remove unnecessary kmap() from sgx_ioc_enclave_init() (Bandan Das) [1494649]
- selftests/sgx: Use getauxval() to simplify test code (Bandan Das) [1494649]
- selftests/sgx: Improve error detection and messages (Bandan Das) [1494649]
- x86/sgx: Add a basic NUMA allocation scheme to sgx_alloc_epc_page() (Bandan Das) [1494649]
- x86/sgx: Replace section->init_laundry_list with sgx_dirty_page_list (Bandan Das) [1494649]
- x86/sgx: Maintain encl->refcount for each encl->mm_list entry (Bandan Das) [1494649]
- MAINTAINERS: Add Dave Hansen as reviewer for INTEL SGX (Bandan Das) [1494649]
- x86/NUMA: Provide a range-to-target_node lookup facility (Bandan Das) [1494649]
- x86/mm: Introduce CONFIG_NUMA_KEEP_MEMINFO (Bandan Das) [1494649]
- configs: Enable CONFIG_MLXBF_GIGE on aarch64 (Alaa Hleihel) [1803489]
- Add Mellanox BlueField Gigabit Ethernet driver (Alaa Hleihel) [1803489]
- integrity: Use current_uid() in integrity_audit_message() (Richard Guy Briggs) [1957052]
- ima: AppArmor satisfies the audit rule requirements (Richard Guy Briggs) [1957052]
- IMA: Add audit log for failure conditions (Richard Guy Briggs) [1957052]
- integrity: Add errno field in audit message (Richard Guy Briggs) [1957052]
- audit: add blank line after variable declarations (Richard Guy Briggs) [1957052]
- audit: drop /proc/PID/loginuid documentation Format field (Richard Guy Briggs) [1957052]
- audit: avoid -Wempty-body warning (Richard Guy Briggs) [1957052]
- audit: document /proc/PID/sessionid (Richard Guy Briggs) [1957052]
- audit: document /proc/PID/loginuid (Richard Guy Briggs) [1957052]
- MAINTAINERS: update audit files (Richard Guy Briggs) [1957052]
- audit: further cleanup of AUDIT_FILTER_ENTRY deprecation (Richard Guy Briggs) [1957052]
- netfilter: nftables: fix a warning message in nf_tables_commit_audit_collect() (Richard Guy Briggs) [1957052]
- audit: log nftables configuration change events once per table (Richard Guy Briggs) [1957052]
- audit_alloc_mark(): don't open-code ERR_CAST() (Richard Guy Briggs) [1957052]
- make dump_common_audit_data() safe to be called from RCU pathwalk (Richard Guy Briggs) [1957052]
- new helper: d_find_alias_rcu() (Richard Guy Briggs) [1957052]
- audit: Make audit_filter_syscall() return void (Richard Guy Briggs) [1957052]
- audit: Remove leftover reference to the audit_tasklet (Richard Guy Briggs) [1957052]
- kernel/audit: convert comma to semicolon (Richard Guy Briggs) [1957052]
- dump_common_audit_data(): fix racy accesses to ->d_name (Richard Guy Briggs) [1957052]
- audit: replace atomic_add_return() (Richard Guy Briggs) [1957052]
- x86/audit: Fix a -Wmissing-prototypes warning for ia32_classify_syscall() (Richard Guy Briggs) [1957052]
- ethtool: fix kdoc attr name (Ivan Vecera) [1967261]
- ethtool: Remove link_mode param and derive link params from driver (Ivan Vecera) [1967261]
- ethtool: Add lanes parameter for ETHTOOL_LINK_MODE_10000baseR_FEC_BIT (Ivan Vecera) [1967261]
- ethtool: fix kdoc in headers (Ivan Vecera) [1967261]
- ethtool: document reserved fields in the uAPI (Ivan Vecera) [1967261]
- ethtool: un-kdocify extended link state (Ivan Vecera) [1967261]
- ethtool: document PHY tunable callbacks (Ivan Vecera) [1967261]
- docs: ethtool: fix some copy-paste errors (Ivan Vecera) [1967261]
- ethtool: fix incorrect datatype in set_eee ops (Ivan Vecera) [1967261]
- net: selftests: Add lanes setting test (Ivan Vecera) [1967261]
- mlxsw: ethtool: Pass link mode in use to ethtool (Ivan Vecera) [1967261]
- mlxsw: ethtool: Add support for setting lanes when autoneg is off (Ivan Vecera) [1967261]
- mlxsw: ethtool: Remove max lanes filtering (Ivan Vecera) [1967261]
- ethtool: Expose the number of lanes in use (Ivan Vecera) [1967261]
- ethtool: Get link mode in use instead of speed and duplex parameters (Ivan Vecera) [1967261]
- ethtool: prepare __ethtool_get_link_ksettings for subsequent changes (Ivan Vecera) [1967261]
- ethtool: Extend link modes settings uAPI with lanes (Ivan Vecera) [1967261]
- ethtool: Validate master slave configuration before rtnl_lock() (Ivan Vecera) [1967261]
- ethtool: fix string set id check (Ivan Vecera) [1967261]
- ethtool: fix stack overflow in ethnl_parse_bitset() (Ivan Vecera) [1967261]
- selftests: add ring and coalesce selftests (Ivan Vecera) [1967261]
- selftests: refactor get_netdev_name function (Ivan Vecera) [1967261]
- selftests: extract common functions in ethtool-common.sh (Ivan Vecera) [1967261]
- netdevsim: support ethtool ring and coalesce settings (Ivan Vecera) [1967261]
- netdevsim: move ethtool pause params in separate struct (Ivan Vecera) [1967261]
- ethtool: add ETHTOOL_COALESCE_ALL_PARAMS define (Ivan Vecera) [1967261]
- ethtool: netlink: add missing netdev_features_change() call (Ivan Vecera) [1967261]
- ethtool: remove unneeded semicolon (Ivan Vecera) [1967261]
- ethtool: allow netdev driver to define phy tunables (Ivan Vecera) [1967261]
- ethtool: mark netlink family as __ro_after_init (Ivan Vecera) [1967261]
- ethtool: Add 100base-FX link mode entries (Ivan Vecera) [1967261]
- ethtool: fix error handling in ethtool_phys_id (Ivan Vecera) [1967261]
- ethtool: allow flow-type ether without IP protocol field (Ivan Vecera) [1967261]
- net: phy: Uninline PHY ethtool statistics operations (Ivan Vecera) [1967261]
- net: phy: Define PHY statistics ethtool_phy_ops (Ivan Vecera) [1967261]
- net: ethtool: Remove PHYLIB direct dependency (Ivan Vecera) [1967261]
- net: phy: Register ethtool PHY operations (Ivan Vecera) [1967261]
- genksyms: Ignore module scoped _Static_assert() (Ivan Vecera) [1967261]
- net: ethtool: Introduce ethtool_phy_ops (Ivan Vecera) [1967261]
- Documentation: networking: fix ethtool-netlink table formats (Ivan Vecera) [1967261]
- ethtool: ioctl: Use array_size() in copy_to_user() (Ivan Vecera) [1967261]
- net: ethtool: cabletest: Make ethnl_act_cable_test_tdr_cfg static (Ivan Vecera) [1967261]
- SUNRPC: Handle major timeout in xprt_adjust_timeout() (Scott Mayhew) [1979070]
- gro_cells: reduce number of synchronize_net() calls (Ivan Vecera) [1959015]
- net: make sure napi_list is safe for RCU traversal (Ivan Vecera) [1959015]
- net: manage napi add/del idempotence explicitly (Ivan Vecera) [1959015]
- net: remove napi_hash_del() from driver-facing API (Ivan Vecera) [1959015]
- net: disable netpoll on fresh napis (Ivan Vecera) [1959015]
- usb: dwc3-meson-g12a: fix usb2 PHY glue init when phy0 is disabled (Torez Smith) [1942772]
- xhci: Fix 5.12 regression of missing xHC cache clearing command after a Stall (Torez Smith) [1942772]
- xhci: fix giving back URB with incorrect status regression in 5.12 (Torez Smith) [1942772]
- usb: typec: ucsi: Clear pending after acking connector change (Torez Smith) [1942772]
- net: hso: check for allocation failure in hso_create_bulk_serial_device() (Torez Smith) [1942772]
- usb: typec: tcpm: Fix wrong handling in GET_SINK_CAP (Torez Smith) [1942772]
- net: hso: fix NULL-deref on disconnect regression (Torez Smith) [1942772]
- USB: CDC-ACM: fix poison/unpoison imbalance (Torez Smith) [1942772]
- usbip: vudc: fix missing unlock on error in usbip_sockfd_store() (Torez Smith) [1942772]
- usb: xhci-mtk: improve bandwidth scheduling with TT (Torez Smith) [1942772]
- usb: xhci-mtk: remove or operator for setting schedule parameters (Torez Smith) [1942772]
- usb: typec: tcpm: Wait for vbus discharge to VSAFE0V before toggling (Torez Smith) [1942772]
- software node: Provide replacement for device_add_properties() (Torez Smith) [1942772]
- driver core: platform: Reimplement devm_platform_ioremap_resource (Torez Smith) [1942772]
- USB: usblp: initialize newChannel (Torez Smith) [1942772]
- net: hso: fix null-ptr-deref during tty device unregistration (Torez Smith) [1942772]
- usbip: synchronize event handler with sysfs code paths (Torez Smith) [1942772]
- usbip: vudc synchronize sysfs code paths (Torez Smith) [1942772]
- usbip: stub-dev synchronize sysfs code paths (Torez Smith) [1942772]
- usbip: add sysfs_lock to synchronize sysfs code paths (Torez Smith) [1942772]
- drivers: net: fix memory leak in peak_usb_create_dev (Torez Smith) [1942772]
- thunderbolt: Fix off by one in tb_port_find_retimer() (Torez Smith) [1942772]
- thunderbolt: Fix a leak in tb_retimer_add() (Torez Smith) [1942772]
- usb: dwc2: Prevent core suspend when port connection flag is 0 (Torez Smith) [1942772]
- usb: dwc2: Fix HPRT0.PrtSusp bit setting for HiKey 960 board. (Torez Smith) [1942772]
- usb: xhci-mtk: fix broken streams issue on 0.96 xHCI (Torez Smith) [1942772]
- usbip: vhci_hcd fix shift out-of-bounds in vhci_hub_control() (Torez Smith) [1942772]
- USB: quirks: ignore remote wake-up on Fibocom L850-GL LTE modem (Torez Smith) [1942772]
- USB: cdc-acm: do not log successful probe on later errors (Torez Smith) [1942772]
- USB: cdc-acm: always claim data interface (Torez Smith) [1942772]
- USB: cdc-acm: use negation for NULL checks (Torez Smith) [1942772]
- USB: cdc-acm: clean up probe error labels (Torez Smith) [1942772]
- USB: cdc-acm: drop redundant driver-data reset (Torez Smith) [1942772]
- USB: cdc-acm: drop redundant driver-data assignment (Torez Smith) [1942772]
- USB: cdc-acm: fix use-after-free after probe failure (Torez Smith) [1942772]
- USB: cdc-acm: fix double free on probe failure (Torez Smith) [1942772]
- USB: cdc-acm: downgrade message to debug (Torez Smith) [1942772]
- USB: cdc-acm: untangle a circular dependency between callback and softint (Torez Smith) [1942772]
- cdc-acm: fix BREAK rx code path adding necessary calls (Torez Smith) [1942772]
- usb: dwc3: pci: Enable dis_uX_susphy_quirk for Intel Merrifield (Torez Smith) [1942772]
- usb: dwc3: qcom: skip interconnect init for ACPI probe (Torez Smith) [1942772]
- can: peak_usb: Revert "can: peak_usb: add forgotten supported devices" (Torez Smith) [1942772]
- r8152: limit the RX buffer size of RTL8153A for USB 2.0 (Torez Smith) [1942772]
- net: cdc-phonet: fix data-interface release on probe failure (Torez Smith) [1942772]
- usb: typec: tcpm: Skip sink_cap query only when VDM sm is busy (Torez Smith) [1942772]
- usb: typec: tcpm: Invoke power_supply_changed for tcpm-source-psy- (Torez Smith) [1942772]
- usb: typec: Remove vdo[3] part of tps6598x_rx_identity_reg struct (Torez Smith) [1942772]
- usb-storage: Add quirk to defeat Kindle's automatic unload (Torez Smith) [1942772]
- usbip: Fix incorrect double assignment to udc->ud.tcp_rx (Torez Smith) [1942772]
- can: kvaser_usb: Add support for USBcan Pro 4xHS (Torez Smith) [1942772]
- can: peak_usb: add forgotten supported devices (Torez Smith) [1942772]
- xhci: Fix repeated xhci wake after suspend due to uncleared internal wake state (Torez Smith) [1942772]
- usb: xhci: Fix ASMedia ASM1042A and ASM3242 DMA addressing (Torez Smith) [1942772]
- xhci: Improve detection of device initiated wake signal. (Torez Smith) [1942772]
- usb: xhci: do not perform Soft Retry for some xHCI hosts (Torez Smith) [1942772]
- usbip: fix vudc usbip_sockfd_store races leading to gpf (Torez Smith) [1942772]
- usbip: fix vhci_hcd attach_store() races leading to gpf (Torez Smith) [1942772]
- usbip: fix stub_dev usbip_sockfd_store() races leading to gpf (Torez Smith) [1942772]
- usbip: fix vudc to check for stream socket (Torez Smith) [1942772]
- usbip: fix vhci_hcd to check for stream socket (Torez Smith) [1942772]
- usbip: fix stub_dev to check for stream socket (Torez Smith) [1942772]
- USB: usblp: fix a hang in poll() if disconnected (Torez Smith) [1942772]
- usb: dwc3: qcom: Add missing DWC3 OF node refcount decrement (Torez Smith) [1942772]
- usb: dwc3: qcom: Honor wakeup enabled/disabled state (Torez Smith) [1942772]
- usb: dwc3: qcom: add ACPI device id for sc8180x (Torez Smith) [1942772]
- Goodix Fingerprint device is not a modem (Torez Smith) [1942772]
- USB: serial: io_edgeport: fix memory leak in edge_startup (Torez Smith) [1942772]
- USB: serial: ch341: add new Product ID (Torez Smith) [1942772]
- USB: serial: cp210x: add some more GE USB IDs (Torez Smith) [1942772]
- USB: serial: cp210x: add ID for Acuity Brands nLight Air Adapter (Torez Smith) [1942772]
- net: usb: log errors to dmesg/syslog (Torez Smith) [1942772]
- net: usb: cdc_ncm: emit dev_err on error paths (Torez Smith) [1942772]
- thunderbolt: Increase runtime PM reference count on DP tunnel discovery (Torez Smith) [1942772]
- thunderbolt: Initialize HopID IDAs in tb_switch_alloc() (Torez Smith) [1942772]
- net: usb: qmi_wwan: allow qmimux add/del with master up (Torez Smith) [1942772]
- Revert "r8152: adjust the settings about MAC clock speed down for RTL8153" (Torez Smith) [1942772]
- net: usb: qmi_wwan: support ZTE P685M modem (Torez Smith) [1942772]
- r8152: spilt rtl_set_eee_plus and r8153b_green_en (Torez Smith) [1942772]
- r8152: replace netif_err with dev_err (Torez Smith) [1942772]
- r8152: check if the pointer of the function exists (Torez Smith) [1942772]
- r8152: enable U1/U2 for USB_SPEED_SUPER (Torez Smith) [1942772]
- usb: misc: usb3503: Fix logic in usb3503_init() (Torez Smith) [1942772]
- usb: Replace lkml.org links with lore (Torez Smith) [1942772]
- usb: quirks: add quirk to start video capture on ELMO L-12F document camera reliable (Torez Smith) [1942772]
- USB: quirks: sort quirk entries (Torez Smith) [1942772]
- USB: serial: drop bogus to_usb_serial_port() checks (Torez Smith) [1942772]
- USB: serial: drop if with an always false condition (Torez Smith) [1942772]
- usb: typec: tcpm: Get Sink VDO from fwnode (Torez Smith) [1942772]
- dt-bindings: connector: Add SVDM VDO properties (Torez Smith) [1942772]
- usb: typec: displayport: Fill the negotiated SVDM Version in the header (Torez Smith) [1942772]
- usb: typec: ucsi: Determine common SVDM Version (Torez Smith) [1942772]
- usb: typec: tcpm: Determine common SVDM Version (Torez Smith) [1942772]
- usb: pd: Make SVDM Version configurable in VDM header (Torez Smith) [1942772]
- usb: typec: Manage SVDM version (Torez Smith) [1942772]
- usb: dwc3: gadget: Fix dep->interval for fullspeed interrupt (Torez Smith) [1942772]
- usb: dwc3: gadget: Fix setting of DEPCFG.bInterval_m1 (Torez Smith) [1942772]
- USB: serial: option: update interface mapping for ZTE P685M (Torez Smith) [1942772]
- usb: dwc3: gadget: Track connected SSP rate and lane count (Torez Smith) [1942772]
- usb: dwc3: gadget: Implement setting of SSP rate (Torez Smith) [1942772]
- usb: dwc3: core: Check maximum_speed SSP genXxY (Torez Smith) [1942772]
- usb: common: Parse for USB SSP genXxY (Torez Smith) [1942772]
- usb: dwc2: Fix endpoint direction check in ep_from_windex (Torez Smith) [1942772]
- usb: dwc3: fix clock issue during resume in OTG mode (Torez Smith) [1942772]
- xhci: ext-caps: Use software node API with the properties (Torez Smith) [1942772]
- usb: dwc3: host: Use software node API with the properties (Torez Smith) [1942772]
- r8152: replace several functions about phy patch request (Torez Smith) [1942772]
- usb: dwc3: qcom: Constify the software node (Torez Smith) [1942772]
- usb: dwc3: haps: Constify the software node (Torez Smith) [1942772]
- usb: dwc2: pci: Drop the empty quirk function (Torez Smith) [1942772]
- thunderbolt: Add support for native USB4 _OSC (Torez Smith) [1942772]
- thunderbolt: Allow disabling XDomain protocol (Torez Smith) [1942772]
- thunderbolt: Add support for PCIe tunneling disabled (SL5) (Torez Smith) [1942772]
- thunderbolt: dma_test: Drop unnecessary include (Torez Smith) [1942772]
- thunderbolt: Add clarifying comments about USB4 terms router and adapter (Torez Smith) [1942772]
- thunderbolt: switch: Fix kernel-doc descriptions of non-static functions (Torez Smith) [1942772]
- thunderbolt: nhi: Fix kernel-doc descriptions of non-static functions (Torez Smith) [1942772]
- thunderbolt: path: Fix kernel-doc descriptions of non-static functions (Torez Smith) [1942772]
- thunderbolt: eeprom: Fix kernel-doc descriptions of non-static functions (Torez Smith) [1942772]
- thunderbolt: ctl: Fix kernel-doc descriptions of non-static functions (Torez Smith) [1942772]
- usb: pd: Reland VDO definitions of PD2.0 (Torez Smith) [1942772]
- NET: usb: qmi_wwan: Adding support for Cinterion MV31 (Torez Smith) [1942772]
- usb: pd: Update VDO definitions (Torez Smith) [1942772]
- xhci: fix bounce buffer usage for non-sg list case (Torez Smith) [1942772]
- usb: xhci-mtk: break loop when find the endpoint to drop (Torez Smith) [1942772]
- usb: typec: Return void in typec_partner_set_pd_revision (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: Enable data path when partner is USB Comm capable (Torez Smith) [1942772]
- usb: typec: tcpci: Add Callback to Usb Communication capable partner (Torez Smith) [1942772]
- usb: typec: tcpm: Add Callback to Usb Communication capable partner (Torez Smith) [1942772]
- usb: typec: tcpm: Set in_ams flag when Source caps have been received (Torez Smith) [1942772]
- usb: typec: tcpm: Handle vbus shutoff when in source mode (Torez Smith) [1942772]
- usb: typec: Add typec_partner_set_pd_revision (Torez Smith) [1942772]
- usb: typec: Provide PD Specification Revision for cable and partner (Torez Smith) [1942772]
- usb: typec: Standardize PD Revision format with Type-C Revision (Torez Smith) [1942772]
- usb: xhci-mtk: skip dropping bandwidth of unchecked endpoints (Torez Smith) [1942772]
- USB: serial: ftdi_sio: restore divisor-encoding comments (Torez Smith) [1942772]
- USB: serial: ftdi_sio: fix FTX sub-integer prescaler (Torez Smith) [1942772]
- USB: serial: cp210x: clean up auto-RTS handling (Torez Smith) [1942772]
- USB: serial: cp210x: fix RTS handling (Torez Smith) [1942772]
- USB: serial: cp210x: clean up printk zero padding (Torez Smith) [1942772]
- USB: serial: cp210x: clean up flow-control debug message (Torez Smith) [1942772]
- USB: serial: cp210x: drop shift macros (Torez Smith) [1942772]
- USB: serial: cp210x: fix modem-control handling (Torez Smith) [1942772]
- USB: serial: cp210x: suppress modem-control errors (Torez Smith) [1942772]
- xhci: handle halting transfer event properly after endpoint stop and halt raced. (Torez Smith) [1942772]
- xhci: Check for pending reset endpoint command before queueing a new one. (Torez Smith) [1942772]
- xhci: remove obsolete dequeue pointer moving code (Torez Smith) [1942772]
- xhci: introduce a new move_dequeue_past_td() function to replace old code. (Torez Smith) [1942772]
- xhci: handle stop endpoint command completion with endpoint in running state. (Torez Smith) [1942772]
- xhci: Fix halted endpoint at stop endpoint command completion (Torez Smith) [1942772]
- xhci: split handling halted endpoints into two steps (Torez Smith) [1942772]
- xhci: move and rename xhci_cleanup_halted_endpoint() (Torez Smith) [1942772]
- xhci: turn cancelled td cleanup to its own function (Torez Smith) [1942772]
- xhci: store TD status in the td struct instead of passing it along (Torez Smith) [1942772]
- xhci: use xhci_td_cleanup() helper when giving back cancelled URBs (Torez Smith) [1942772]
- xhci: move xhci_td_cleanup so it can be called by more functions (Torez Smith) [1942772]
- xhci: Add xhci_reset_halted_ep() helper function (Torez Smith) [1942772]
- xhci: flush endpoint start to reduce race risk with stop endpoint command. (Torez Smith) [1942772]
- xhci: Check link TRBs when updating ring enqueue and dequeue pointers. (Torez Smith) [1942772]
- xhci: avoid DMA double fetch when reading event trb type. (Torez Smith) [1942772]
- xhci: remove extra loop in interrupt context (Torez Smith) [1942772]
- xhci: check slot_id is valid before gathering slot info (Torez Smith) [1942772]
- xhci: prevent a theoretical endless loop while preparing rings. (Torez Smith) [1942772]
- xhci: remove xhci_stream_id_to_ring() helper (Torez Smith) [1942772]
- xhci: add xhci_virt_ep_to_ring() helper (Torez Smith) [1942772]
- xhci: check virt_dev is valid before dereferencing it (Torez Smith) [1942772]
- xhci: add xhci_get_virt_ep() helper (Torez Smith) [1942772]
- xhci: remove unused event parameter from completion handlers (Torez Smith) [1942772]
- xhci: adjust parameters passed to cleanup_halted_endpoint() (Torez Smith) [1942772]
- xhci: get isochronous ring directly from endpoint structure (Torez Smith) [1942772]
- xhci: Avoid parsing transfer events several times (Torez Smith) [1942772]
- net: qmi_wwan: Add pass through mode (Torez Smith) [1942772]
- net: usb: qmi_wwan: add qmap id sysfs file for qmimux interfaces (Torez Smith) [1942772]
- net: usb: cdc_ether: added support for Thales Cinterion PLSx3 modem family. (Torez Smith) [1942772]
- thunderbolt: Fix possible NULL pointer dereference in tb_acpi_add_link() (Torez Smith) [1942772]
- thunderbolt: switch: Fix function name in the header (Torez Smith) [1942772]
- thunderbolt: tunnel: Fix misspelling of 'receive_path' (Torez Smith) [1942772]
- thunderbolt: icm: Fix a couple of formatting issues (Torez Smith) [1942772]
- thunderbolt: switch: Demote a bunch of non-conformant kernel-doc headers (Torez Smith) [1942772]
- thunderbolt: tb: Kernel-doc function headers should document their parameters (Torez Smith) [1942772]
- thunderbolt: nhi: Demote some non-conformant kernel-doc headers (Torez Smith) [1942772]
- thunderbolt: xdomain: Fix 'tb_unregister_service_driver()'s 'drv' param (Torez Smith) [1942772]
- thunderbolt: eeprom: Demote non-conformant kernel-doc headers to standard comment blocks (Torez Smith) [1942772]
- USB: serial: mos7720: fix error code in mos7720_write() (Torez Smith) [1942772]
- thunderbolt: ctl: Demote non-conformant kernel-doc headers (Torez Smith) [1942772]
- thunderbolt: cap: Fix kernel-doc formatting issue (Torez Smith) [1942772]
- thunderbolt: dma_port: Check 'dma_port_flash_write_block()'s return value (Torez Smith) [1942772]
- USB: serial: option: Adding support for Cinterion MV31 (Torez Smith) [1942772]
- can: mcba_usb: remove h from printk format specifier (Torez Smith) [1942772]
- usbnet: fix the indentation of one code snippet (Torez Smith) [1942772]
- usb: usb251xb: Use of_device_get_match_data() (Torez Smith) [1942772]
- usb: dwc2: Make "trimming xfer length" a debug message (Torez Smith) [1942772]
- usb: dwc2: Abort transaction after errors with unknown reason (Torez Smith) [1942772]
- usb: dwc2: Do not update data length if it is 0 on inbound transfers (Torez Smith) [1942772]
- usb: xhci-mtk: fix unreleased bandwidth data (Torez Smith) [1942772]
- USB: serial: mos7840: fix error code in mos7840_write() (Torez Smith) [1942772]
- USB: usblp: don't call usb_set_interface if there's a single alt (Torez Smith) [1942772]
- USB: serial: cp210x: add pid/vid for WSDA-200-USB (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: add terminating newlines to logging (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: remove redundant assignment (Torez Smith) [1942772]
- usb: typec: tcpm: Create legacy PDOs for PD2 connection (Torez Smith) [1942772]
- net: usb: qmi_wwan: added support for Thales Cinterion PLSx3 modem family (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: Drop GPIO includes (Torez Smith) [1942772]
- usb: typec: tcpm: Respond Wait if VDM state machine is running (Torez Smith) [1942772]
- usb: typec: tcpm: Protocol Error handling (Torez Smith) [1942772]
- usb: typec: tcpm: AMS and Collision Avoidance (Torez Smith) [1942772]
- net: usb: cdc_ncm: don't spew notifications (Torez Smith) [1942772]
- usb: typec: ucsi: Add conditional dependency on USB role switch (Torez Smith) [1942772]
- usb: ch9: Add USB 3.2 SSP attributes (Torez Smith) [1942772]
- USB: serial: f81232: drop short control-transfer checks (Torez Smith) [1942772]
- USB: serial: io_ti: fix a debug-message copy-paste error (Torez Smith) [1942772]
- USB: serial: io_ti: drop short control-transfer check (Torez Smith) [1942772]
- USB: serial: upd78f0730: drop short control-transfer check (Torez Smith) [1942772]
- USB: serial: cp210x: add copyright notice (Torez Smith) [1942772]
- USB: serial: cp210x: drop unused includes (Torez Smith) [1942772]
- USB: serial: cp210x: drop short control-transfer checks (Torez Smith) [1942772]
- USB: serial: cp210x: update control-characters on every change (Torez Smith) [1942772]
- USB: serial: cp210x: set IXOFF thresholds (Torez Smith) [1942772]
- USB: serial: cp210x: add support for software flow control (Torez Smith) [1942772]
- USB: serial: pl2303: fix line-speed handling on newer chips (Torez Smith) [1942772]
- xhci: tegra: Delay for disabling LFPS detector (Torez Smith) [1942772]
- xhci: make sure TRB is fully written before giving it to the controller (Torez Smith) [1942772]
- usb: dwc3: qcom: add URS Host support for sdm845 ACPI boot (Torez Smith) [1942772]
- usb: dwc3: pci: add support for the Intel Alder Lake-P (Torez Smith) [1942772]
- usb: dwc3: pci: ID for Tiger Lake CPU (Torez Smith) [1942772]
- usb: dwc3: pci: Register a software node for the dwc3 platform device (Torez Smith) [1942772]
- thunderbolt: Add support for de-authorizing devices (Torez Smith) [1942772]
- nt: usb: USB_RTL8153_ECM should not default to y (Torez Smith) [1942772]
- USB: dwc3: document gadget_max_speed (Torez Smith) [1942772]
- usb: dwc3: gadget: Disable Vendor Test LMP Received event (Torez Smith) [1942772]
- USB: serial: cp210x: add new VID/PID for supporting Teraoka AD2000 (Torez Smith) [1942772]
- r8153_ecm: Add Lenovo Powered USB-C Hub as a fallback of r8152 (Torez Smith) [1942772]
- r8152: Add Lenovo Powered USB-C Travel Hub (Torez Smith) [1942772]
- USB: ehci: fix an interrupt calltrace error (Torez Smith) [1942772]
- ehci: fix EHCI host controller initialization sequence (Torez Smith) [1942772]
- usb: typec: ucsi: Add support for USB role switch (Torez Smith) [1942772]
- usb: dwc2: disable Link Power Management on STM32MP15 HS OTG (Torez Smith) [1942772]
- usb: dwc2: enable FS/LS PHY clock select on STM32MP15 FS OTG (Torez Smith) [1942772]
- usb: dwc2: set ahbcfg parameter for STM32MP15 OTG HS and FS (Torez Smith) [1942772]
- usb: dwc3: Simplify with dev_err_probe() (Torez Smith) [1942772]
- usb: dwc3: keystone: Simplify with dev_err_probe() (Torez Smith) [1942772]
- usb: dwc3: core: Replace devm_reset_control_array_get() (Torez Smith) [1942772]
- USB: serial: mos7720: improve OOM-handling in read_mos_reg() (Torez Smith) [1942772]
- rndis_host: set proper input size for OID_GEN_PHYSICAL_MEDIUM request (Torez Smith) [1942772]
- thunderbolt: Start lane initialization after sleep (Torez Smith) [1942772]
- thunderbolt: Constify static attribute_group structs (Torez Smith) [1942772]
- qmi_wwan: Increase headroom for QMAP SKBs (Torez Smith) [1942772]
- usb: usbip: Use DEFINE_SPINLOCK() for spinlock (Torez Smith) [1942772]
- usb: dwc3: gadget: Clear wait flag on dequeue (Torez Smith) [1942772]
- usb: typec: Send uevent for num_altmodes update (Torez Smith) [1942772]
- usb: typec: Fix copy paste error for NVIDIA alt-mode description (Torez Smith) [1942772]
- kcov, usb: hide in_serving_softirq checks in __usb_hcd_giveback_urb (Torez Smith) [1942772]
- thunderbolt: Drop duplicated 0x prefix from format string (Torez Smith) [1942772]
- net: cdc_ncm: correct overhead in delayed_ndp_size (Torez Smith) [1942772]
- net: usb: Use DEFINE_SPINLOCK() for spinlock (Torez Smith) [1942772]
- usb: uas: Add PNY USB Portable SSD to unusual_uas (Torez Smith) [1942772]
- net: usb: qmi_wwan: add Quectel EM160R-GL (Torez Smith) [1942772]
- usb: usbip: vhci_hcd: protect shift size (Torez Smith) [1942772]
- USB: usblp: fix DMA to stack (Torez Smith) [1942772]
- usb: dwc3: gadget: Preserve UDC max speed setting (Torez Smith) [1942772]
- usb: dwc3: gadget: Allow runtime suspend if UDC unbinded (Torez Smith) [1942772]
- usb: dwc3: gadget: Introduce a DWC3 VBUS draw callback (Torez Smith) [1942772]
- usb: dwc3: add imx8mp dwc3 glue layer driver (Torez Smith) [1942772]
- USB: serial: iuu_phoenix: fix DMA from stack (Torez Smith) [1942772]
- USB: serial: option: add LongSung M5710 module support (Torez Smith) [1942772]
- USB: serial: option: add Quectel EM160R-GL (Torez Smith) [1942772]
- CDC-NCM: remove "connected" log message (Torez Smith) [1942772]
- usb: dwc3: ulpi: Fix USB2.0 HS/FS/LS PHY suspend regression (Torez Smith) [1942772]
- usb: dwc3: ulpi: Replace CPU-based busyloop with Protocol-based one (Torez Smith) [1942772]
- usb: dwc3: ulpi: Use VStsDone to detect PHY regs access completion (Torez Smith) [1942772]
- usb: dwc3: drd: Improve dwc3_get_extcon() style (Torez Smith) [1942772]
- usb: dwc3: drd: Avoid error when extcon is missing (Torez Smith) [1942772]
- USB: usbtmc: Bump USBTMC_API_VERSION value (Torez Smith) [1942772]
- USB: usbtmc: Add separate USBTMC_IOCTL_GET_SRQ_STB (Torez Smith) [1942772]
- USB: usbtmc: Add USBTMC_IOCTL_GET_STB (Torez Smith) [1942772]
- USB: usbtmc: Fix reading stale status byte (Torez Smith) [1942772]
- kcov, usbip: collect coverage from vhci_rx_loop (Torez Smith) [1942772]
- USB: yurex: fix control-URB timeout handling (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: disable clk on error handling path in probe (Torez Smith) [1942772]
- USB: xhci: fix U1/U2 handling for hardware with XHCI_INTEL_HOST quirk set (Torez Smith) [1942772]
- usb: typec: intel_pmc_mux: Configure HPD first for HPD+IRQ request (Torez Smith) [1942772]
- USB: cdc-wdm: Fix use after free in service_outstanding_interrupt(). (Torez Smith) [1942772]
- USB: cdc-acm: blacklist another IR Droid device (Torez Smith) [1942772]
- thunderbolt: Use kmemdup instead of kzalloc and memcpy (Torez Smith) [1942772]
- usb: ucsi: convert comma to semicolon (Torez Smith) [1942772]
- usb: typec: tcpm: convert comma to semicolon (Torez Smith) [1942772]
- usb: typec: tcpm: Update vbus_vsafe0v on init (Torez Smith) [1942772]
- usb: typec: tcpci: Enable bleed discharge when auto discharge is enabled (Torez Smith) [1942772]
- usb: typec: Add class for plug alt mode device (Torez Smith) [1942772]
- USB: typec: tcpci: Add Bleed discharge to POWER_CONTROL definition (Torez Smith) [1942772]
- USB: typec: tcpm: Add a 30ms room for tPSSourceOn in PR_SWAP (Torez Smith) [1942772]
- USB: typec: tcpm: Fix PR_SWAP error handling (Torez Smith) [1942772]
- USB: typec: tcpm: Hard Reset after not receiving a Request (Torez Smith) [1942772]
- usb: typec: intel_pmc_mux: Configure cable generation value for USB4 (Torez Smith) [1942772]
- net: usb: convert comma to semicolon (Torez Smith) [1942772]
- USB: UAS: introduce a quirk to set no_write_same (Torez Smith) [1942772]
- USB: serial: ftdi_sio: log the CBUS GPIO validity (Torez Smith) [1942772]
- USB: serial: ftdi_sio: drop GPIO line checking dead code (Torez Smith) [1942772]
- USB: serial: ftdi_sio: report the valid GPIO lines to gpiolib (Torez Smith) [1942772]
- drivers: usb: atm: use atm_info() instead of atm_printk(KERN_INFO ... (Torez Smith) [1942772]
- drivers: usb: atm: reduce noise (Torez Smith) [1942772]
- USB: serial: option: add interface-number sanity check to flag handling (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: Enable VSAFE0V signalling (Torez Smith) [1942772]
- usb: typec: tcpci: Add support to report vSafe0V (Torez Smith) [1942772]
- usb: typec: tcpm: Introduce vsafe0v for vbus (Torez Smith) [1942772]
- usb: typec: tcpm: Clear send_discover in tcpm_check_send_discover (Torez Smith) [1942772]
- usb: typec: tcpm: Pass down negotiated rev to update retry count (Torez Smith) [1942772]
- usb: typec: intel_pmc_mux: Use correct response message bits (Torez Smith) [1942772]
- xhci: Give USB2 ports time to enter U3 in bus suspend (Torez Smith) [1942772]
- xhci-pci: Allow host runtime PM as default for Intel Maple Ridge xHCI (Torez Smith) [1942772]
- xhci-pci: Allow host runtime PM as default for Intel Alpine Ridge LP (Torez Smith) [1942772]
- usb: xhci: Use temporary buffer to consolidate SG (Torez Smith) [1942772]
- usb: xhci: Set quirk for XHCI_SG_TRB_CACHE_SIZE_QUIRK (Torez Smith) [1942772]
- USB: legotower: fix logical error in recent commit (Torez Smith) [1942772]
- USB: legousbtower: use usb_control_msg_recv() (Torez Smith) [1942772]
- USB: add RESET_RESUME quirk for Snapscan 1212 (Torez Smith) [1942772]
- usb: typec: tps6598x: Export some power supply properties (Torez Smith) [1942772]
- usb: typec: tps6598x: Select USB_ROLE_SWITCH and REGMAP_I2C (Torez Smith) [1942772]
- USB: core: drop pipe-type check from new control-message helpers (Torez Smith) [1942772]
- USB: core: return -EREMOTEIO on short usb_control_msg_recv() (Torez Smith) [1942772]
- USB: core: drop short-transfer check from usb_control_msg_send() (Torez Smith) [1942772]
- USB: apple-mfi-fastcharge: Fix use after free in probe (Torez Smith) [1942772]
- Revert "usb-storage: fix sdev->host->dma_dev" (Torez Smith) [1942772]
- Revert "uas: fix sdev->host->dma_dev" (Torez Smith) [1942772]
- Revert "uas: bump hw_max_sectors to 2048 blocks for SS or faster drives" (Torez Smith) [1942772]
- USB: serial: cp210x: clean up dtr_rts() (Torez Smith) [1942772]
- USB: serial: cp210x: refactor flow-control handling (Torez Smith) [1942772]
- USB: serial: cp210x: drop flow-control debugging (Torez Smith) [1942772]
- USB: serial: cp210x: set terminal settings on open (Torez Smith) [1942772]
- USB: serial: cp210x: clean up line-control handling (Torez Smith) [1942772]
- USB: serial: cp210x: return early on unchanged termios (Torez Smith) [1942772]
- USB: serial: kl5kusb105: fix memleak on open (Torez Smith) [1942772]
- USB: serial: ch341: sort device-id entries (Torez Smith) [1942772]
- USB: serial: ch341: add new Product ID for CH341A (Torez Smith) [1942772]
- USB: serial: option: fix Quectel BG96 matching (Torez Smith) [1942772]
- can: pcan_usb_core: fix fall-through warnings for Clang (Torez Smith) [1942772]
- usb: typec: stusb160x: fix power-opmode property with typec-power-opmode (Torez Smith) [1942772]
- USB: serial: option: add support for Thales Cinterion EXS82 (Torez Smith) [1942772]
- usb: typec: Add type sysfs attribute file for partners (Torez Smith) [1942772]
- usb: common: ulpi: Constify static attribute_group struct (Torez Smith) [1942772]
- usb: typec: Constify static attribute_group structs (Torez Smith) [1942772]
- USB: core: Constify static attribute_group structs (Torez Smith) [1942772]
- usb: typec: tcpm: Stay in SNK_TRY_WAIT_DEBOUNCE_CHECK_VBUS till Rp is seen (Torez Smith) [1942772]
- usb: typec: tcpm: Disregard vbus off while in PR_SWAP_SNK_SRC_SOURCE_ON (Torez Smith) [1942772]
- usb: typec: Expose Product Type VDOs via sysfs (Torez Smith) [1942772]
- usb: pd: DFP product types (Torez Smith) [1942772]
- usb: Fix fall-through warnings for Clang (Torez Smith) [1942772]
- usbnet: ipheth: fix connectivity with iOS 14 (Torez Smith) [1942772]
- USB: core: Change pK for __user pointers to px (Torez Smith) [1942772]
- USB: core: Fix regression in Hercules audio card (Torez Smith) [1942772]
- USB: quirks: Add USB_QUIRK_DISCONNECT_SUSPEND quirk for Lenovo A630Z TIO built-in usb-audio card (Torez Smith) [1942772]
- can: kvaser_usb: Add new Kvaser hydra devices (Torez Smith) [1942772]
- can: kvaser_usb: kvaser_usb_hydra: Add support for new device variant (Torez Smith) [1942772]
- can: kvaser_usb: Add new Kvaser Leaf v2 devices (Torez Smith) [1942772]
- can: kvaser_usb: Add USB_{LEAF,HYDRA}_PRODUCT_ID_END defines (Torez Smith) [1942772]
- USB: serial: option: add Fibocom NL668 variants (Torez Smith) [1942772]
- usb: typec: Fix num_altmodes kernel-doc error (Torez Smith) [1942772]
- r8153_ecm: avoid to be prior to r8152 driver (Torez Smith) [1942772]
- net: usb: qmi_wwan: Set DTR quirk for MR400 (Torez Smith) [1942772]
- usb: typec: Add plug num_altmodes sysfs attr (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: Fix the compatible string (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: Fix uninitialized return variable (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: Enable auto discharge disconnect (Torez Smith) [1942772]
- usb: typec: tcpci: Implement Auto discharge disconnect callbacks (Torez Smith) [1942772]
- usb: typec: tcpm: Implement enabling Auto Discharge disconnect support (Torez Smith) [1942772]
- usb: typec: tcpci_maxim: Fix vbus stuck on upon diconnecting sink (Torez Smith) [1942772]
- usb: typec: tcpci: frs sourcing vbus callback (Torez Smith) [1942772]
- usb: typec: tcpm: frs sourcing vbus callback (Torez Smith) [1942772]
- usb: typec: tcpm: Refactor logic for new-source-frs-typec-current (Torez Smith) [1942772]
- usb: typec: Add number of altmodes partner attr (Torez Smith) [1942772]
- usb: pd: Add captive Type C cable type (Torez Smith) [1942772]
- USB: apple-mfi-fastcharge: Fix kfree after failed kzalloc (Torez Smith) [1942772]
- usb: typec: Remove one bit support for the Thunderbolt rounded/non-rounded cable (Torez Smith) [1942772]
- usb: typec: intel_pmc_mux: Configure Thunderbolt cable generation bits (Torez Smith) [1942772]
- usb: typec: Correct the bit values for the Thunderbolt rounded/non-rounded cable support (Torez Smith) [1942772]
- can: kvaser_usb: kvaser_usb_hydra: Fix KCAN bittiming limits (Torez Smith) [1942772]
- cx82310_eth: fix error return code in cx82310_bind() (Torez Smith) [1942772]
- usb: typec: qcom-pmic-typec: fix builtin build errors (Torez Smith) [1942772]
- can: peak_usb: fix potential integer overflow on shift of a int (Torez Smith) [1942772]
- can: mcba_usb: mcba_usb_start_xmit(): first fill skb, then pass to can_put_echo_skb() (Torez Smith) [1942772]
- usb: hcd.h: Remove RUN_CONTEXT (Torez Smith) [1942772]
- USB: storage: avoid use of uninitialized values in error path (Torez Smith) [1942772]
- usb: fix a few cases of -Wfallthrough (Torez Smith) [1942772]
- usbnet: switch to core handling of rx/tx byte/packet counters (Torez Smith) [1942772]
- qmi_wwan: switch to core handling of rx/tx byte/packet counters (Torez Smith) [1942772]
- net: usb: fix spelling typo in cdc_ncm.c (Torez Smith) [1942772]
- usb: typec: ucsi: Work around PPM losing change information (Torez Smith) [1942772]
- usb: typec: ucsi: acpi: Always decode connector change information (Torez Smith) [1942772]
- net/usb/r8153_ecm: support ECM mode for RTL8153 (Torez Smith) [1942772]
- USB: serial: mos7720: fix parallel-port state restore (Torez Smith) [1942772]
- USB: serial: remove write wait queue (Torez Smith) [1942772]
- USB: serial: digi_acceleport: fix write-wakeup deadlocks (Torez Smith) [1942772]
- USB: serial: keyspan_pda: drop redundant usb-serial pointer (Torez Smith) [1942772]
- USB: serial: keyspan_pda: use BIT() macro (Torez Smith) [1942772]
- USB: serial: keyspan_pda: add write-fifo support (Torez Smith) [1942772]
- USB: serial: keyspan_pda: increase transmitter threshold (Torez Smith) [1942772]
- USB: serial: keyspan_pda: fix write implementation (Torez Smith) [1942772]
- USB: serial: keyspan_pda: refactor write-room handling (Torez Smith) [1942772]
- USB: serial: keyspan_pda: fix write unthrottling (Torez Smith) [1942772]
- USB: serial: keyspan_pda: fix tx-unthrottle use-after-free (Torez Smith) [1942772]
- USB: serial: keyspan_pda: fix write-wakeup use-after-free (Torez Smith) [1942772]
- USB: serial: keyspan_pda: fix stalled writes (Torez Smith) [1942772]
- USB: serial: keyspan_pda: fix write deadlock (Torez Smith) [1942772]
- USB: serial: keyspan_pda: fix dropped unthrottle interrupts (Torez Smith) [1942772]
- net: usb: r8152: Fix a couple of spelling errors in fw_phy_nc's docs (Torez Smith) [1942772]
- net: usb: lan78xx: Remove lots of set but unused 'ret' variables (Torez Smith) [1942772]
- net: usb: r8152: Provide missing documentation for some struct members (Torez Smith) [1942772]
- can: peak_usb: peak_usb_get_ts_time(): fix timestamp wrapping (Torez Smith) [1942772]
- can: peak_usb: add range checking in decode operations (Torez Smith) [1942772]
- net: usb: qmi_wwan: add Telit LE910Cx 0x1230 composition (Torez Smith) [1942772]
- kcov, usb: only collect coverage from __usb_hcd_giveback_urb in softirq (Torez Smith) [1942772]
- dt-bindings: connector: Add property to set initial current cap for FRS (Torez Smith) [1942772]
- usb: xhci: Remove in_interrupt() checks (Torez Smith) [1942772]
- usbip: Remove in_interrupt() check (Torez Smith) [1942772]
- usb: core: Replace in_interrupt() in comments (Torez Smith) [1942772]
- usb: atm: Replace in_interrupt() usage in comment (Torez Smith) [1942772]
- USB: sisusbvga: Make console support depend on BROKEN (Torez Smith) [1942772]
- usb: host: xhci-mem: remove unneeded break (Torez Smith) [1942772]
- usb: storage: freecom: remove unneeded break (Torez Smith) [1942772]
- usb: misc: iowarrior: remove unneeded break (Torez Smith) [1942772]
- usb: host: ehci-sched: add comment about find_tt() not returning error (Torez Smith) [1942772]
- usb: dwc3: ep0: Fix delay status handling (Torez Smith) [1942772]
- USB: serial: iuu_phoenix: remove unneeded break (Torez Smith) [1942772]
- USB: serial: digi_acceleport: remove in_interrupt() usage (Torez Smith) [1942772]
- usb: dwc3: pci: add support for the Intel Alder Lake-S (Torez Smith) [1942772]
- net: usb: rtl8150: don't incorrectly assign random MAC addresses (Torez Smith) [1942772]
- net: usbnet: use new function dev_fetch_sw_netstats (Torez Smith) [1942772]
- net: usb: qmi_wwan: use new function dev_fetch_sw_netstats (Torez Smith) [1942772]
- cx82310_eth: use netdev_err instead of dev_err (Torez Smith) [1942772]
- cx82310_eth: re-enable ethernet mode after router reboot (Torez Smith) [1942772]
- net: usbnet: remove driver version (Torez Smith) [1942772]
- net: usb: qmi_wwan: add Cellient MPL200 card (Torez Smith) [1942772]
- usbip: vhci_hcd: fix calling usb_hcd_giveback_urb() with irqs enabled (Torez Smith) [1942772]
- net: usb: rtl8150: set random MAC address when set_ethernet_addr() fails (Torez Smith) [1942772]
- usbnet: Use fallthrough pseudo-keyword (Torez Smith) [1942772]
- net/smscx5xx: change to of_get_mac_address() eth_platform_get_mac_address() (Torez Smith) [1942772]
- net: usb: pegasus: Proper error handing when setting pegasus' MAC address (Torez Smith) [1942772]
- usb: dwc3: gadget: Support up to max stream id (Torez Smith) [1942772]
- usb: dwc2: Fix INTR OUT transfers in DDMA mode. (Torez Smith) [1942772]
- usb: dwc3: of-simple: Add compatible string for Intel Keem Bay platform (Torez Smith) [1942772]
- net: rtl8150: Use the new usb control message API. (Torez Smith) [1942772]
- net: pegasus: Use the new usb control message API. (Torez Smith) [1942772]
- usb: dwc3: gadget: Return early if no TRB update (Torez Smith) [1942772]
- usb: dwc3: gadget: Keep TRBs in request order (Torez Smith) [1942772]
- usb: dwc3: gadget: Revise setting IOC when no TRB left (Torez Smith) [1942772]
- usb: dwc3: gadget: Look ahead when setting IOC (Torez Smith) [1942772]
- usb: dwc3: gadget: Allow restarting a transfer (Torez Smith) [1942772]
- usb: dwc3: Stop active transfers before halting the controller (Torez Smith) [1942772]
- usb: dwc3: gadget: Rename misleading function names (Torez Smith) [1942772]
- usb: dwc3: Add splitdisable quirk for Hisilicon Kirin Soc (Torez Smith) [1942772]
- usb: dwc3: gadget: Refactor preparing last TRBs (Torez Smith) [1942772]
- usb: dwc3: gadget: Set IOC if not enough for extra TRBs (Torez Smith) [1942772]
- usb: dwc3: gadget: Check for number of TRBs prepared (Torez Smith) [1942772]
- usb: dwc3: gadget: Return the number of prepared TRBs (Torez Smith) [1942772]
- usb: dwc3: ep0: Fix ZLP for OUT ep0 requests (Torez Smith) [1942772]
- usb: dwc3: gadget: Improve TRB ZLP setup (Torez Smith) [1942772]
- usb: dwc3: gadget: Refactor preparing extra TRB (Torez Smith) [1942772]
- usb: dwc3: gadget: Reclaim extra TRBs after request completion (Torez Smith) [1942772]
- usb: dwc3: gadget: Check MPS of the request length (Torez Smith) [1942772]
- usb: dwc3: gadget: Refactor ep command completion (Torez Smith) [1942772]
- usb: dwc3: allocate gadget structure dynamically (Torez Smith) [1942772]
- usb: dwc3: core: add phy cleanup for probe error handling (Torez Smith) [1942772]
- usb: dwc3: core: don't trigger runtime pm when remove driver (Torez Smith) [1942772]
- usb: dwc3: debugfs: do not queue work if try to change mode on non-drd (Torez Smith) [1942772]
- usb: dwc3: pci: Allow Elkhart Lake to utilize DSM method for PM functionality (Torez Smith) [1942772]
- usb: dwc2: add support for APM82181 USB OTG (Torez Smith) [1942772]
- usb: dwc-meson-g12a: Add support for USB on AXG SoCs (Torez Smith) [1942772]
- usb: dwc3: gadget: fix checkpatch warnings (Torez Smith) [1942772]
- usb: dwc3: core: do not queue work if dr_mode is not USB_DR_MODE_OTG (Torez Smith) [1942772]
- usb: dwc3: debugfs: fix checkpatch warnings (Torez Smith) [1942772]
- usb: dwc3: qcom: fix checkpatch warnings (Torez Smith) [1942772]
- usb: dwc3: ep0: fix checkpatch warnings (Torez Smith) [1942772]
- usb: dwc3: core: fix checkpatch warnings (Torez Smith) [1942772]
- usb: dwc3: debug: fix checkpatch warning (Torez Smith) [1942772]
- usb: dwc3: trace: fix checkpatch warnings (Torez Smith) [1942772]
- usb: dwc3: ulpi: fix checkpatch warning (Torez Smith) [1942772]
- usb: dwc3: meson: fix checkpatch errors and warnings (Torez Smith) [1942772]
- usb: dwc3: debug: fix sparse warning (Torez Smith) [1942772]
- usb: dwc3: meson: fix coccinelle WARNING (Torez Smith) [1942772]
- usb: dwc3: qcom: Add interconnect support in dwc3 driver (Torez Smith) [1942772]
- net: usb: net1080: Remove in_interrupt() comment (Torez Smith) [1942772]
- net: usb: kaweth: Remove last user of kaweth_control() (Torez Smith) [1942772]
- net: usb: kaweth: Replace kaweth_control() with usb_control_msg() (Torez Smith) [1942772]
- net: usb: ax88179_178a: add MCT usb 3.0 adapter (Torez Smith) [1942772]
- net: usb: ax88179_178a: fix missing stop entry in driver_info (Torez Smith) [1942772]
- net: usb: ax88179_178a: add Toshiba usb 3.0 adapter (Torez Smith) [1942772]
- Revert "usbip: Implement a match function to fix usbip" (Torez Smith) [1942772]
- usb: dwc2: Fix parameter type in function pointer prototype (Torez Smith) [1942772]
- usb: dwc3: core: Print warning on unsupported speed (Torez Smith) [1942772]
- usb: dwc3: core: Properly default unspecified speed (Torez Smith) [1942772]
- usb: dwc2: Add missing cleanups when usb_add_gadget_udc() fails (Torez Smith) [1942772]
- usb: dwc3: gadget: END_TRANSFER before CLEAR_STALL command (Torez Smith) [1942772]
- usb: dwc3: gadget: Resume pending requests after CLEAR_STALL (Torez Smith) [1942772]
- can: peak_usb: convert to use le32_add_cpu() (Torez Smith) [1942772]
- can: pcan_usb: add support of rxerr/txerr counters (Torez Smith) [1942772]
- can: pcan_usb: Document the commands sent to the device (Torez Smith) [1942772]
- can: mcba_usb: remove redundant initialization of variable err (Torez Smith) [1942772]
- usb: dwc2: Always disable regulators on driver teardown (Torez Smith) [1942772]
- usbip: simplify the return expression of usbip_core_init() (Torez Smith) [1942772]
- rndis_host: increase sleep time in the query-response loop (Torez Smith) [1942772]
- Revert "usb: dwc3: meson-g12a: fix shared reset control use" (Torez Smith) [1942772]
- net: usb: dm9601: Add USB ID of Keenetic Plus DSL (Torez Smith) [1942772]
- smsc95xx: add phylib support (Torez Smith) [1942772]
- smsc95xx: use usbnet->driver_priv (Torez Smith) [1942772]
- smsc95xx: remove redundant function arguments (Torez Smith) [1942772]
- net: usb: Fix uninit-was-stored issue in asix_read_phy_addr() (Torez Smith) [1942772]
- net: cdc_ncm: Fix build error (Torez Smith) [1942772]
- tools: usb: move to tools buildsystem (Torez Smith) [1942772]
- usbip: Implement a match function to fix usbip (Torez Smith) [1942772]
- usb: dwc3: gadget: Handle ZLP for sg requests (Torez Smith) [1942772]
- usb: dwc3: gadget: Fix handling ZLP (Torez Smith) [1942772]
- usb: dwc3: gadget: Don't setup more than requested (Torez Smith) [1942772]
- r8152: Use MAC address from correct device tree node (Torez Smith) [1942772]
- usb: hso: remove bogus check for EINPROGRESS (Torez Smith) [1942772]
- usb: hso: no complaint about kmalloc failure (Torez Smith) [1942772]
- hso: fix bailout in error case of probe (Torez Smith) [1942772]
- usb: dwc3: convert to devm_platform_ioremap_resource_byname (Torez Smith) [1942772]
- net: lan78xx: replace bogus endpoint lookup (Torez Smith) [1942772]
- net: lan78xx: fix transfer-buffer memory leak (Torez Smith) [1942772]
- net: lan78xx: add missing endpoint sanity check (Torez Smith) [1942772]
- usb: hso: check for return value in hso_serial_common_create() (Torez Smith) [1942772]
- Revert "usb: dwc2: override PHY input signals with usb role switch support" (Torez Smith) [1942772]
- Revert "usb: dwc2: don't use ID/Vbus detection if usb-role-switch on STM32MP15 SoCs" (Torez Smith) [1942772]
- usb: dwc3: gadget: when the started list is empty stop the active xfer (Torez Smith) [1942772]
- usb: dwc3: gadget: make starting isoc transfers more robust (Torez Smith) [1942772]
- usb: dwc3: gadget: add frame number mask (Torez Smith) [1942772]
- usb: dwc2: Fix error path in gadget registration (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: fix shared reset control use (Torez Smith) [1942772]
- usb: dwc2: don't use ID/Vbus detection if usb-role-switch on STM32MP15 SoCs (Torez Smith) [1942772]
- usb: dwc2: override PHY input signals with usb role switch support (Torez Smith) [1942772]
- net: cdc_ncm: hook into set_rx_mode to admit multicast traffic (Torez Smith) [1942772]
- net: cdc_ncm: add .ndo_set_rx_mode to cdc_ncm_netdev_ops (Torez Smith) [1942772]
- net: usbnet: export usbnet_set_rx_mode() (Torez Smith) [1942772]
- net: cdc_ether: export usbnet_cdc_update_filter (Torez Smith) [1942772]
- net: cdc_ether: use dev->intf to get interface information (Torez Smith) [1942772]
- ax88172a: fix ax88172a_unbind() failures (Torez Smith) [1942772]
- usb: dwc2: gadget: Avoid pointless read of EP control register (Torez Smith) [1942772]
- usb: dwc2: gadget: Make use of GINTMSK2 (Torez Smith) [1942772]
- usb: dwc3: Replace HTTP links with HTTPS ones (Torez Smith) [1942772]
- tools: usb: usbip: Replace HTTP links with HTTPS ones (Torez Smith) [1942772]
- usb: hso: Fix debug compile warning on sparc32 (Torez Smith) [1942772]
- usbip: Use fallthrough pseudo-keyword (Torez Smith) [1942772]
- Documentation/driver-api: usb/URB: drop doubled word (Torez Smith) [1942772]
- usb: dwc3: dwc3-qcom: Do not define 'struct acpi_device_id' when !CONFIG_ACPI (Torez Smith) [1942772]
- usb: dwc3: pci: add support for the Intel Jasper Lake (Torez Smith) [1942772]
- usb: dwc2: Fix shutdown callback in platform (Torez Smith) [1942772]
- usb: dwc3: pci: add support for the Intel Tiger Lake PCH -H variant (Torez Smith) [1942772]
- net: usb: qmi_wwan: add support for Quectel EG95 LTE modem (Torez Smith) [1942772]
- smsc95xx: avoid memory leak in smsc95xx_bind (Torez Smith) [1942772]
- smsc95xx: check return value of smsc95xx_reset (Torez Smith) [1942772]
- usb: dwc2: gadget: Remove assigned but never used 'maxsize' (Torez Smith) [1942772]
- usb: dwc3: dwc3-of-simple: Function headers are not good candidates for kerneldoc (Torez Smith) [1942772]
- usb: dwc3: dwc3-haps: Function headers are not suitable for kerneldoc (Torez Smith) [1942772]
- usb: dwc3: dwc3-omap: Do not read DMA status (Torez Smith) [1942772]
- usb: dwc3: ulpi: File headers are not doc headers (Torez Smith) [1942772]
- usb: dwc3: drd: File headers are not doc headers (Torez Smith) [1942772]
- usbnet: ipheth: fix ipheth_tx()'s return type (Torez Smith) [1942772]
- usbip: tools: add in man page how to load the client's module (Torez Smith) [1942772]
- usbip: tools: fix build error for multiple definition (Torez Smith) [1942772]
- usbip: tools: fix module name in man page (Torez Smith) [1942772]
- Revert "usb: dwc3: exynos: Add support for Exynos5422 suspend clk" (Torez Smith) [1942772]
- usb: dwc3: pci: Fix reference count leak in dwc3_pci_resume_work (Torez Smith) [1942772]
- media: USB: cdc-acm: blacklist IR Droid / IR Toy device (Torez Smith) [1942772]
- usbnet: smsc95xx: Fix use-after-free after removal (Torez Smith) [1942772]
- usb: dwc2: Postponed gadget registration to the udc class driver (Torez Smith) [1942772]
- usb: dwc2: use well defined macros for power_down (Torez Smith) [1942772]
- net: usb: ax88179_178a: fix packet alignment padding (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: fix USB2 PHY initialization on G12A and A1 SoCs (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: fix error path when fetching the reset line fails (Torez Smith) [1942772]
- net: usb: qmi_wwan: add Telit LE910C1-EUX composition (Torez Smith) [1942772]
- usb: dwc3: keystone: Turn on USB3 PHY before controller (Torez Smith) [1942772]
- usb: dwc2: Update Core Reset programming flow. (Torez Smith) [1942772]
- usb: dwc3: gadget: Check for prepared TRBs (Torez Smith) [1942772]
- usb: dwc3: Increase timeout for CmdAct cleared by device controller (Torez Smith) [1942772]
- usb: dwc3: gadget: Use SET_EP_PRIME for NoStream (Torez Smith) [1942772]
- usb: dwc3: gadget: Handle stream transfers (Torez Smith) [1942772]
- usb: dwc3: gadget: Don't prepare beyond a transfer (Torez Smith) [1942772]
- usb: dwc3: gadget: Wait for transfer completion (Torez Smith) [1942772]
- usb: dwc3: gadget: Handle XferComplete for streams (Torez Smith) [1942772]
- usb: dwc3: gadget: Enable XferComplete event (Torez Smith) [1942772]
- usb: dwc3: gadget: Refactor TRB completion handler (Torez Smith) [1942772]
- usb: dwc3: gadget: Check for in-progress END_TRANSFER (Torez Smith) [1942772]
- usb: dwc3: Get MDWIDTH for DWC_usb32 (Torez Smith) [1942772]
- usb: dwc3: Add support for DWC_usb32 IP (Torez Smith) [1942772]
- usb: dwc3: use true,false for dwc->otg_restart_host (Torez Smith) [1942772]
- usb: dwc2: gadget: move gadget resume after the core is in L0 state (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: add support for GXL and GXM SoCs (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: support the GXL/GXM DWC3 host phy disconnect (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: refactor usb init (Torez Smith) [1942772]
- usb: dwc3: gadget: Continue to process pending requests (Torez Smith) [1942772]
- r8152: support additional Microsoft Surface Ethernet Adapter variant (Torez Smith) [1942772]
- r8152: Use MAC address from device tree if available (Torez Smith) [1942772]
- net: usb: qmi_wwan: remove redundant assignment to variable status (Torez Smith) [1942772]
- net: huawei_cdc_ncm: remove redundant assignment to variable ret (Torez Smith) [1942772]
- net: usb: ax88179_178a: remove redundant assignment to variable ret (Torez Smith) [1942772]
- usb: dwc3: select USB_ROLE_SWITCH (Torez Smith) [1942772]
- usb: dwc3: pci: Enable extcon driver for Intel Merrifield (Torez Smith) [1942772]
- dwc3: Remove check for HWO flag in dwc3_gadget_ep_reclaim_trb_sg() (Torez Smith) [1942772]
- usb: hso: correct debug message (Torez Smith) [1942772]
- net: sierra_net: Remove unused inline function (Torez Smith) [1942772]
- usb: dwc3: gadget: WARN on no-resource status (Torez Smith) [1942772]
- usb: dwc3: gadget: Issue END_TRANSFER to retry isoc transfer (Torez Smith) [1942772]
- usb: dwc3: gadget: Store resource index of start cmd (Torez Smith) [1942772]
- usb: dwc3: core: Use role-switch default dr_mode (Torez Smith) [1942772]
- usb: dwc3: drd: Don't free non-existing irq (Torez Smith) [1942772]
- usb: dwc3: gadget: Properly handle failed kick_transfer (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: refactor usb2 phy init (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: check return of dwc3_meson_g12a_usb_init (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: get the reset as shared (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: handle the phy and glue registers separately (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: specify phy names in soc data (Torez Smith) [1942772]
- usb: dwc3: gadget: Refactor dwc3_gadget_ep_dequeue (Torez Smith) [1942772]
- usb: dwc3: gadget: Remove unnecessary checks (Torez Smith) [1942772]
- usb: dwc3: gadget: Give back staled requests (Torez Smith) [1942772]
- usb: dwc3: gadget: Properly handle ClearFeature(halt) (Torez Smith) [1942772]
- usb: dwc3: support continuous runtime PM with dual role (Torez Smith) [1942772]
- USB: dwc3: Use the correct style for SPDX License Identifier (Torez Smith) [1942772]
- net: usb: qmi_wwan: add support for DW5816e (Torez Smith) [1942772]
- net: usb: ax88179_178a: Implement ethtool_ops set_eeprom (Torez Smith) [1942772]
- usb: dwc3: gadget: Fix request completion check (Torez Smith) [1942772]
- USB: dwc2: Use the correct style for SPDX License Identifier (Torez Smith) [1942772]
- usb: dwc3: gadget: Do link recovery for SS and SSP (Torez Smith) [1942772]
- usb: dwc3: gadget: Properly set maxpacket limit (Torez Smith) [1942772]
- usb: dwc3: Fix GTXFIFOSIZ.TXFDEP macro name (Torez Smith) [1942772]
- Documentation: driver-api/usb/writing_usb_driver.rst Updates documentation links (Torez Smith) [1942772]
- pegasus: Remove pegasus' own workqueue (Torez Smith) [1942772]
- usb: dwc2: convert to devm_platform_get_and_ioremap_resource (Torez Smith) [1942772]
- usb: host: hisilicon: convert to devm_platform_get_and_ioremap_resource (Torez Smith) [1942772]
- usb: host: xhci-plat: convert to devm_platform_get_and_ioremap_resource (Torez Smith) [1942772]
- net: qmi_wwan: add support for ASKEY WWHC050 (Torez Smith) [1942772]
- usb: dwc3: fix up for role switch API change (Torez Smith) [1942772]
- usb: dwc2: Silence warning about supplies during deferred probe (Torez Smith) [1942772]
- usb: dwc2: Implement set_selfpowered() (Torez Smith) [1942772]
- usb: dwc3: core: don't do suspend for device mode if already suspended (Torez Smith) [1942772]
- usb: dwc3: Rework resets initialization to be more flexible (Torez Smith) [1942772]
- usb: dwc3: Rework clock initialization to be more flexible (Torez Smith) [1942772]
- usb: dwc3: Add support for role-switch-default-mode binding (Torez Smith) [1942772]
- usb: dwc3: Registering a role switch in the DRD code. (Torez Smith) [1942772]
- usb: dwc3: core: add support for disabling SS instances in park mode (Torez Smith) [1942772]
- usb: dwc3: gadget: Wrap around when skip TRBs (Torez Smith) [1942772]
- usb: dwc3: gadget: Don't clear flags before transfer ended (Torez Smith) [1942772]
- usb: dwc3: Remove kernel doc annotation where it's not needed (Torez Smith) [1942772]
- usb: dwc3: Add ACPI support for xHCI ports (Torez Smith) [1942772]
- usb: dwc3: exynos: Add support for Exynos5422 suspend clk (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: Don't use ret uninitialized in dwc3_meson_g12a_otg_init (Torez Smith) [1942772]
- usb: dwc3: trace: print enqueue/dequeue pointers too (Torez Smith) [1942772]
- usb: dwc2: add support for STM32MP15 SoCs USB OTG HS and FS (Torez Smith) [1942772]
- usb: dwc3: Add Amlogic A1 DWC3 glue (Torez Smith) [1942772]
- cdc_ncm: Fix the build warning (Torez Smith) [1942772]
- cdc_ncm: Implement the 32-bit version of NCM Transfer Block (Torez Smith) [1942772]
- r8152: reject unsupported coalescing params (Torez Smith) [1942772]
- usb: dwc3: gadget: Update chain bit correctly when using sg list (Torez Smith) [1942772]
- r8152: Replace zero-length array with flexible-array member (Torez Smith) [1942772]
- r8152: check disconnect status after long sleep (Torez Smith) [1942772]
- net: usb: qmi_wwan: restore mtu min/max values after raw_ip switch (Torez Smith) [1942772]
- USB: Replace zero-length array with flexible-array member (Torez Smith) [1942772]
- r8152: use new helper tcp_v6_gso_csum_prep (Torez Smith) [1942772]
- net: usb: cdc-phonet: Replace zero-length array with flexible-array member (Torez Smith) [1942772]
- usb: dwc3: qcom: Replace <linux/clk-provider.h> by <linux/of_clk.h> (Torez Smith) [1942772]
- usb: dwc3: gadget: Check for IOC/LST bit in TRB->ctrl fields (Torez Smith) [1942772]
- usb: dwc2: Fix in ISOC request length checking (Torez Smith) [1942772]
- usb: dwc2: Fix SET/CLEAR_FEATURE and GET_STATUS flows (Torez Smith) [1942772]
- usb: dwc3: debug: fix string position formatting mixup with ret and len (Torez Smith) [1942772]
- qmi_wwan: unconditionally reject 2 ep interfaces (Torez Smith) [1942772]
- qmi_wwan: re-add DW5821e pre-production variant (Torez Smith) [1942772]
- r8152: Add MAC passthrough support to new device (Torez Smith) [1942772]
- r8152: disable DelayPhyPwrChg (Torez Smith) [1942772]
- r8152: avoid the MCU to clear the lanwake (Torez Smith) [1942772]
- r8152: don't enable U1U2 with USB_SPEED_HIGH for RTL8153B (Torez Smith) [1942772]
- r8152: disable test IO for RTL8153B (Torez Smith) [1942772]
- r8152: Disable PLA MCU clock speed down (Torez Smith) [1942772]
- r8152: disable U2P3 for RTL8153B (Torez Smith) [1942772]
- r8152: get default setting of WOL before initializing (Torez Smith) [1942772]
- r8152: reset flow control patch when linking on for RTL8153B (Torez Smith) [1942772]
- r8152: fix runtime resume for linking change (Torez Smith) [1942772]
- usb: dwc3: pci: add ID for the Intel Comet Lake -V variant (Torez Smith) [1942772]
- net: usb: lan78xx: Add .ndo_features_check (Torez Smith) [1942772]
- usb: dwc2: Drop unlock/lock upon queueing a work item (Torez Smith) [1942772]
- usb: dwc3: gadget: Remove END_TRANSFER delay (Torez Smith) [1942772]
- usb: dwc3: gadget: Delay starting transfer (Torez Smith) [1942772]
- usb: dwc3: gadget: Check END_TRANSFER completion (Torez Smith) [1942772]
- usb: dwc3: turn off VBUS when leaving host mode (Torez Smith) [1942772]
- usb: dwc2: fix debugfs FIFO count (Torez Smith) [1942772]
- usb: dwc2: Fix IN FIFO allocation (Torez Smith) [1942772]
- usb: dwc2: Fix Stalling a Non-Isochronous OUT EP (Torez Smith) [1942772]
- usb: dwc3: gadget: Clear DCTL.ULSTCHNGREQ before set (Torez Smith) [1942772]
- usb: dwc2: Fix NULL qh in dwc2_queue_transaction (Torez Smith) [1942772]
- usb: dwc3: gadget: Set link state to RX_Detect on disconnect (Torez Smith) [1942772]
- usb: dwc3: gadget: Don't send unintended link state change (Torez Smith) [1942772]
- r8152: add missing endpoint sanity check (Torez Smith) [1942772]
- net: usb: lan78xx: limit size of local TSO packets (Torez Smith) [1942772]
- qmi_wwan: Add support for Quectel RM500Q (Torez Smith) [1942772]
- usbip: Fix unsafe unaligned pointer usage (Torez Smith) [1942772]
- usb: dwc3: use proper initializers for property entries (Torez Smith) [1942772]
- net: r8152: use skb_list_walk_safe helper for gso segments (Torez Smith) [1942772]
- net: usb: lan78xx: fix possible skb leak (Torez Smith) [1942772]
- net: ch9200: remove unnecessary return (Torez Smith) [1942772]
- net: ch9200: use __func__ in debug message (Torez Smith) [1942772]
- tools: usb: usbip: Get rid of driver name printout in README (Torez Smith) [1942772]
- can: gs_usb: gs_usb_probe(): use descriptors of current altsetting (Torez Smith) [1942772]
- can: kvaser_usb: fix interface sanity check (Torez Smith) [1942772]
- usb: dwc3: gadget: Fix request complete check (Torez Smith) [1942772]
- thunderbolt: Update documentation with the USB4 information (Torez Smith) [1942772]
- usbip: Fix error path of vhci_recv_ret_submit() (Torez Smith) [1942772]
- usbip: Fix receive error in vhci-hcd when using scatter-gather (Torez Smith) [1942772]
- net: usb: lan78xx: Fix error message format specifier (Torez Smith) [1942772]
- net: usb: lan78xx: Fix suspend/resume PHY register access error (Torez Smith) [1942772]
- usb: dwc3: pci: add ID for the Intel Comet Lake -H variant (Torez Smith) [1942772]
- usb: dwc3: ep0: Clear started flag on completion (Torez Smith) [1942772]
- usb: dwc3: gadget: Clear started flag for non-IOC (Torez Smith) [1942772]
- usb: dwc3: gadget: Fix logical condition (Torez Smith) [1942772]
- can: kvaser_usb: kvaser_usb_leaf: Fix some info-leaks to USB devices (Torez Smith) [1942772]
- can: ucan: fix non-atomic allocation in completion handler (Torez Smith) [1942772]
- net: usb: aqc111: Use the correct style for SPDX License Identifier (Torez Smith) [1942772]
- net: usbnet: Fix -Wcast-function-type (Torez Smith) [1942772]
- net: hso: Fix -Wcast-function-type (Torez Smith) [1942772]
- ax88179_178a: add ethtool_op_get_ts_info() (Torez Smith) [1942772]
- r8152: avoid to call napi_disable twice (Torez Smith) [1942772]
- r8152: Re-order napi_disable in rtl8152_close (Torez Smith) [1942772]
- ax88172a: fix information leak on short answers (Torez Smith) [1942772]
- usb: dwc2: create debugfs directory under usb root (Torez Smith) [1942772]
- usb: dwc3: create debugfs directory under usb root (Torez Smith) [1942772]
- net: cdc_ncm: Signedness bug in cdc_ncm_set_dgram_size() (Torez Smith) [1942772]
- net: usb: qmi_wwan: add support for Foxconn T77W968 LTE modules (Torez Smith) [1942772]
- usbip: Fix uninitialized symbol 'nents' in stub_recv_cmd_submit() (Torez Smith) [1942772]
- USBIP: add config dependency for SGL_ALLOC (Torez Smith) [1942772]
- net: usb: qmi_wwan: add support for DW5821e with eSIM support (Torez Smith) [1942772]
- CDC-NCM: handle incomplete transfer of MTU (Torez Smith) [1942772]
- usbip: tools: fix fd leakage in the function of read_attr_usbip_status (Torez Smith) [1942772]
- can: peak_usb: fix a potential out-of-sync while decoding packets (Torez Smith) [1942772]
- can: usb_8dev: fix use-after-free on disconnect (Torez Smith) [1942772]
- can: mcba_usb: fix use-after-free on disconnect (Torez Smith) [1942772]
- can: peak_usb: report bus recovery as well (Torez Smith) [1942772]
- can: gs_usb: gs_can_open(): prevent memory leak (Torez Smith) [1942772]
- usb: Spelling s/enpoint/endpoint/ (Torez Smith) [1942772]
- usb: dwc3: gadget: fix race when disabling ep with cancelled xfers (Torez Smith) [1942772]
- usb: dwc3: of-simple: add a shutdown (Torez Smith) [1942772]
- usbip: Fix free of unallocated memory in vhci tx (Torez Smith) [1942772]
- usbip: tools: Fix read_usb_vudc_device() error path handling (Torez Smith) [1942772]
- usb: dwc3: pci: prevent memory leak in dwc3_pci_probe (Torez Smith) [1942772]
- usb: dwc3: remove the call trace of USBx_GFLADJ (Torez Smith) [1942772]
- usb: dwc3: select CONFIG_REGMAP_MMIO (Torez Smith) [1942772]
- r8152: check the pointer rtl_fw->fw before using it (Torez Smith) [1942772]
- net: lan78xx: remove set but not used variable 'event' (Torez Smith) [1942772]
- usb: dwc3: debug: Remove newline printout (Torez Smith) [1942772]
- r8152: support firmware of PHY NC for RTL8153A (Torez Smith) [1942772]
- r8152: move r8153_patch_request forward (Torez Smith) [1942772]
- r8152: add checking fw_offset field of struct fw_mac (Torez Smith) [1942772]
- r8152: rename fw_type_1 with fw_mac (Torez Smith) [1942772]
- usb: dwc2: use a longer core rest timeout in dwc2_core_reset() (Torez Smith) [1942772]
- usb: dwc3: Disable phy suspend after power-on reset (Torez Smith) [1942772]
- r8152: add device id for Lenovo ThinkPad USB-C Dock Gen 2 (Torez Smith) [1942772]
- net: usb: lan78xx: Connect PHY before registering MAC (Torez Smith) [1942772]
- usb: hso: obey DMA rules in tiocmget (Torez Smith) [1942772]
- r8152: support request_firmware for RTL8153 (Torez Smith) [1942772]
- net: usb: sr9800: fix uninitialized local variable (Torez Smith) [1942772]
- net: usb: ax88179_178a: write mac to hardware in get_mac_addr (Torez Smith) [1942772]
- net: usb: qmi_wwan: add Telit 0x1050 composition (Torez Smith) [1942772]
- usb: dwc3: Remove dev_err() on platform_get_irq() failure (Torez Smith) [1942772]
- usb: dwc3: Switch to platform_get_irq_byname_optional() (Torez Smith) [1942772]
- r8152: Set macpassthru in reset_resume callback (Torez Smith) [1942772]
- qmi_wwan: add support for Cinterion CLS8 devices (Torez Smith) [1942772]
- usbip: clean up an indentation issue (Torez Smith) [1942772]
- usbip: vhci_hcd indicate failed message (Torez Smith) [1942772]
- r8152: Add identifier names for function pointers (Torez Smith) [1942772]
- r8152: Use guard clause and fix comment typos (Torez Smith) [1942772]
- hso: fix NULL-deref on tty open (Torez Smith) [1942772]
- net: usb: ax88179_178a: allow optionally getting mac address from device tree (Torez Smith) [1942772]
- r8152: Factor out OOB link list waits (Torez Smith) [1942772]
- usbnet: ignore endpoints with invalid wMaxPacketSize (Torez Smith) [1942772]
- cdc_ncm: fix divide-by-zero caused by invalid wMaxPacketSize (Torez Smith) [1942772]
- usbnet: sanity checking of packet sizes and device mtu (Torez Smith) [1942772]
- r8152: adjust the settings of ups flags (Torez Smith) [1942772]
- r8152: modify rtl8152_set_speed function (Torez Smith) [1942772]
- usb: dwc3: remove generic PHY calibrate() calls (Torez Smith) [1942772]
- usb: gadget: net2280: Move all "ll" registers in one structure (Torez Smith) [1942772]
- usb: dwc3: gadget: Workaround Mirosoft's BESL check (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: fix suspend resume regulator unbalanced disables (Torez Smith) [1942772]
- usb: gadget: composite: Set recommended BESL values (Torez Smith) [1942772]
- usb: dwc3: gadget: Set BESL config parameter (Torez Smith) [1942772]
- usb: dwc3: Separate field holding multiple properties (Torez Smith) [1942772]
- usb: dwc3: don't set gadget->is_otg flag (Torez Smith) [1942772]
- usb: dwc3: st: Add of_dev_put() in probe function (Torez Smith) [1942772]
- usb: dwc3: st: Add of_node_put() before return in probe function (Torez Smith) [1942772]
- r8152: add a helper function about setting EEE (Torez Smith) [1942772]
- r8152: saving the settings of EEE (Torez Smith) [1942772]
- net: usb: Delete unnecessary checks before the macro call “dev_kfree_skb” (Torez Smith) [1942772]
- r8152: divide the tx and rx bottom functions (Torez Smith) [1942772]
- r8152: fix accessing skb after napi_gro_receive (Torez Smith) [1942772]
- r8152: change rx_copybreak and rx_pending through ethtool (Torez Smith) [1942772]
- r8152: support skb_add_rx_frag (Torez Smith) [1942772]
- r8152: use alloc_pages for rx buffer (Torez Smith) [1942772]
- r8152: replace array with linking list for rx information (Torez Smith) [1942772]
- r8152: separate the rx buffer size (Torez Smith) [1942772]
- usb: dwc3: Use clk_bulk_prepare_enable() (Torez Smith) [1942772]
- usb: dwc3: Use devres to get clocks (Torez Smith) [1942772]
- usb: dwc3: remove generic PHYs forwarding for XHCI device (Torez Smith) [1942772]
- USB: usbip: convert to use dev_groups (Torez Smith) [1942772]
- usb: dwc3: Update soft-reset wait polling rate (Torez Smith) [1942772]
- usb: dwc2: gadget: Fix kill_all_requests race (Torez Smith) [1942772]
- usb: dwc3: omap: use devm_platform_ioremap_resource() to simplify code (Torez Smith) [1942772]
- usb: dwc3: keystone: use devm_platform_ioremap_resource() to simplify code (Torez Smith) [1942772]
- usb: dwc3: omap: squash include/linux/platform_data/dwc3-omap.h (Torez Smith) [1942772]
- usb: dwc3: meson-g12a: use devm_platform_ioremap_resource() to simplify code (Torez Smith) [1942772]
- usbip: tools: fix GCC8 warning for strncpy (Torez Smith) [1942772]
- usb: dwc3: Switch to use device_property_count_u32() (Torez Smith) [1942772]
- usb: dwc2: Switch to use device_property_count_u32() (Torez Smith) [1942772]
- can: kvaser_usb: Use struct_size() in alloc_candev() (Torez Smith) [1942772]
- net: usb: Merge cpu_to_le32s + memcpy to put_unaligned_le32 (Torez Smith) [1942772]
- ax88179_178a: Merge memcpy + le32_to_cpus to get_unaligned_le32 (Torez Smith) [1942772]
- usbnet: smsc75xx: Merge memcpy + le32_to_cpus to get_unaligned_le32 (Torez Smith) [1942772]
- net: lan78xx: Merge memcpy + lexx_to_cpus to get_unaligned_lexx (Torez Smith) [1942772]
- bluetooth: eliminate the potential race condition when removing the HCI controller (Gopal Tiwari) [1971488]

* Wed Jul 07 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-322.el8]
- bareudp: add NETIF_F_FRAGLIST flag for dev features (Paolo Abeni) [1977850]
- vxlan: add NETIF_F_FRAGLIST flag for dev features (Paolo Abeni) [1977850]
- geneve: add NETIF_F_FRAGLIST flag for dev features (Paolo Abeni) [1977850]
- RDMA/siw: Release xarray entry (Kamal Heib) [1971483]
- RDMA/siw: Properly check send and receive CQ pointers (Kamal Heib) [1971483]
- RDMA/siw: Fix a use after free in siw_alloc_mr (Kamal Heib) [1971483]
- RDMA/rxe: Clear all QP fields if creation failed (Kamal Heib) [1971483]
- RDMA/rxe: Fix a bug in rxe_fill_ip_info() (Kamal Heib) [1971483]
- RDMA/rxe: Remove rxe_dma_device declaration (Kamal Heib) [1971483]
- IB/hfi1: Fix error return code in parse_platform_config() (Kamal Heib) [1971483]
- IB/hfi1: Remove unused function (Kamal Heib) [1971483]
- RDMA/srpt: Fix error return code in srpt_cm_req_recv() (Kamal Heib) [1971483]
- IB/isert: Fix a use after free in isert_connect_request (Kamal Heib) [1971483]
- RDMA: Verify port when creating flow rule (Kamal Heib) [1971483]
- RDMA/uverbs: Fix a NULL vs IS_ERR() bug (Kamal Heib) [1971483]
- RDMA/core: Don't access cm_id after its destruction (Kamal Heib) [1971483]
- RDMA/core: Prevent divide-by-zero error triggered by the user (Kamal Heib) [1971483]
- RDMA/core: Add CM to restrack after successful attachment to a device (Kamal Heib) [1971483]
- RDMA/core: Unify RoCE check and re-factor code (Kamal Heib) [1971483]
- RDMA/core: Fix corrupted SL on passive side (Kamal Heib) [1971483]
- mm: vmalloc: add cond_resched() in __vunmap() (Rafael Aquini) [1978469]
- /proc/PID/smaps: Add PMD migration entry parsing (Nico Pache) [1977066]
- s390: add 3f program exception handler (Nico Pache) [1977066]
- x86/ioremap: Fix CONFIG_EFI=n build (Nico Pache) [1977066]
- mm, slub: move slub_debug static key enabling outside slab_mutex (Nico Pache) [1977066]
- mm, slub: enable slub_debug static key when creating cache with explicit debug flags (Nico Pache) [1977066]
- locking/mutex: Remove repeated declaration (Nico Pache) [1977066]
- percpu: make pcpu_nr_empty_pop_pages per chunk type (Nico Pache) [1977066]
- mm/swap: don't SetPageWorkingset unconditionally during swapin (Nico Pache) [1977066]
- s390/tlb: make cleared_pXs flags consistent with generic code (Nico Pache) [1977066]
- mm, compaction: make fast_isolate_freepages() stay within zone (Nico Pache) [1977066]
- mm/vmscan: restore zone_reclaim_mode ABI (Nico Pache) [1977066]
- mm, compaction: move high_pfn to the for loop scope (Nico Pache) [1977066]
- memblock: do not start bottom-up allocations with kernel_end (Nico Pache) [1977066]
- mm, page_alloc: disable pcplists during memory offline (Nico Pache) [1977066]
- mm, page_alloc: move draining pcplists to page isolation users (Nico Pache) [1977066]
- mm, page_alloc: cache pageset high and batch in struct zone (Nico Pache) [1977066]
- mm, page_alloc: simplify pageset_update() (Nico Pache) [1977066]
- mm, page_alloc: fix core hung in free_pcppages_bulk() (Nico Pache) [1977066]
- mm, page_alloc: remove setup_pageset() (Nico Pache) [1977066]
- mm, page_alloc: calculate pageset high and batch once per zone (Nico Pache) [1977066]
- mm, pcp: share common code between memory hotplug and percpu sysctl handler (Nico Pache) [1977066]
- mm, page_alloc: clean up pageset high and batch update (Nico Pache) [1977066]
- s390/uv: handle destroy page legacy interface (Nico Pache) [1977066]
- mm/filemap: fix filemap_map_pages for THP (Nico Pache) [1977066]
- mm: filemap: add missing FGP_ flags in kerneldoc comment for pagecache_get_page (Nico Pache) [1977066]
- mm/filemap.c: rewrite pagecache_get_page documentation (Nico Pache) [1977066]
- mm: filemap: clear idle flag for writes (Nico Pache) [1977066]
- mm: document semantics of ZONE_MOVABLE (Nico Pache) [1977066]
- mm/page_isolation: cleanup set_migratetype_isolate() (Nico Pache) [1977066]
- mm/page_isolation: drop WARN_ON_ONCE() in set_migratetype_isolate() (Nico Pache) [1977066]
- mm/page_isolation: exit early when pageblock is isolated in set_migratetype_isolate() (Nico Pache) [1977066]
- mm/page_alloc: tweak comments in has_unmovable_pages() (Nico Pache) [1977066]
- mm: reuse only-pte-mapped KSM page in do_wp_page() (Nico Pache) [1977066]
- mm/memory.c: skip spurious TLB flush for retried page fault (Nico Pache) [1977066]
- /proc/PID/smaps: consistent whitespace output format (Nico Pache) [1977066]
- mm/memcontrol.c: prevent missed memory.low load tears (Nico Pache) [1977066]
- mm, memcg: prevent mem_cgroup_protected store tearing (Nico Pache) [1977066]
- mm, memcg: prevent memory.min load/store tearing (Nico Pache) [1977066]
- mm, mempolicy: fix up gup usage in lookup_node (Nico Pache) [1977066]
- mm/page_ext.c: drop pfn_present() check when onlining (Nico Pache) [1977066]
- drivers/base/memory.c: drop pages_correctly_probed() (Nico Pache) [1977066]
- drivers/base/memory.c: drop section_count (Nico Pache) [1977066]
- include/linux/mm.h: remove dead code totalram_pages_set() (Nico Pache) [1977066]
- tools headers uapi: Sync asm-generic/mman-common.h with the kernel (Nico Pache) [1977066]
- mm/memcg: Relocate tcpmem to below memory in struct mem_cgroup (Waiman Long) [1959772]
- mm/memcg: optimize user context object stock access (Waiman Long) [1959772]
- mm/memcg: improve refill_obj_stock() performance (Waiman Long) [1959772]
- mm/memcg: cache vmstat data in percpu memcg_stock_pcp (Waiman Long) [1959772]
- mm/memcg: move mod_objcg_state() to memcontrol.c (Waiman Long) [1959772]
- mm: memcontrol: move PageMemcgKmem to the scope of CONFIG_MEMCG_KMEM (Waiman Long) [1959772]
- mm: memcontrol: inline __memcg_kmem_{un}charge() into obj_cgroup_{un}charge_pages() (Waiman Long) [1959772]
- mm: memcontrol: use obj_cgroup APIs to charge kmem pages (Waiman Long) [1959772]
- mm: memcontrol: change ug->dummy_page only if memcg changed (Waiman Long) [1959772]
- mm: memcontrol: directly access page->memcg_data in mm/page_alloc.c (Waiman Long) [1959772]
- mm: memcontrol: introduce obj_cgroup_{un}charge_pages (Waiman Long) [1959772]
- mm: memcontrol: slab: fix obtain a reference to a freeing memcg (Waiman Long) [1959772]
- mm: memmap defer init doesn't work as expected (Baoquan He) [1906271]
- mm: check nr_initialised with PAGES_PER_SECTION directly in defer_init() (Baoquan He) [1906271]
- mm/memory_hotplug: mark pageblocks MIGRATE_ISOLATE while onlining memory (Baoquan He) [1906271]
- mm, memory_hotplug: update pcp lists everytime onlining a memory block (Baoquan He) [1906271]
- mm: pass migratetype into memmap_init_zone() and move_pfn_range_to_zone() (Baoquan He) [1906271]
- nfs: account for selinux security context when deciding to share superblock (Scott Mayhew) [1927027]
- nfs: remove unneeded null check in nfs_fill_super() (Scott Mayhew) [1927027]
- lsm,selinux: add new hook to compare new mount to an existing mount (Scott Mayhew) [1927027]
- libceph: allow addrvecs with a single NONE/blank address (Jeff Layton) [1972278]
- scsi: megaraid_sas: Update driver version to 07.717.02.00-rc1 (Tomas Henzl) [1968170]
- scsi: megaraid_sas: Handle missing interrupts while re-enabling IRQs (Tomas Henzl) [1968170]
- scsi: megaraid_sas: Early detection of VD deletion through RaidMap update (Tomas Henzl) [1968170]
- scsi: megaraid_sas: Fix resource leak in case of probe failure (Tomas Henzl) [1968170]
- scsi: megaraid_sas: Send all non-RW I/Os for TYPE_ENCLOSURE device through firmware (Tomas Henzl) [1968170]
- scsi: megaraid_sas: Use DEFINE_SPINLOCK() for spinlock (Tomas Henzl) [1968170]
- scsi: scsi_devinfo: Add blacklist entry for HPE OPEN-V (Ewan D. Milne) [1927508]
- drivers: do not build devm_platform_get_and_ioremap_resource() for s390x (Jaroslav Kysela) [1917388]
- redhat: update ALSA configs and add regmap-sdw-mbq to redhat/filter-modules.sh (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: fix the return value in dpcm_apply_symmetry() (Jaroslav Kysela) [1917388]
- alsa: control_led - fix initialization in the mode show callback (Jaroslav Kysela) [1917388]
- ALSA: seq: Fix race of snd_seq_timer_open() (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs for HP ZBook Power G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: headphone and mic don't work on an Acer laptop (Jaroslav Kysela) [1917388]
- ALSA: firewire-lib: fix the context to call snd_pcm_stop_xrun() (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs for HP EliteBook 840 Aero G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs and speaker for HP EliteBook x360 1040 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs and speaker for HP Elite Dragonfly G2 (Jaroslav Kysela) [1917388]
- ASoC: rt5682: Fix the fast discharge for headset unplugging in soundwire mode (Jaroslav Kysela) [1917388]
- ASoC: tas2562: Fix TDM_CFG0_SAMPRATE values (Jaroslav Kysela) [1917388]
- ASoC: AMD Renoir: Remove fix for DMI entry on Lenovo 2020 platforms (Jaroslav Kysela) [1917388]
- ASoC: AMD Renoir - add DMI entry for Lenovo 2020 AMD platforms (Jaroslav Kysela) [1917388]
- ASoC: SOF: reset enabled_cores state at suspend (Jaroslav Kysela) [1917388]
- ASoC: fsl-asoc-card: Set .owner attribute when registering card. (Jaroslav Kysela) [1917388]
- ASoC: topology: Fix spelling mistake "vesion" -> "version" (Jaroslav Kysela) [1917388]
- ASoC: rt5659: Fix the lost powers for the HDA header (Jaroslav Kysela) [1917388]
- ASoC: core: Fix Null-point-dereference in fmt_single_name() (Jaroslav Kysela) [1917388]
- spi: pxa2xx: Drop useless comment in the pxa2xx_ssp.h (Jaroslav Kysela) [1917388]
- spi: pxa2xx: Switch to use BIT() and GENMASK() in pxa2xx_ssp.h (Jaroslav Kysela) [1917388]
- spi: pxa2xx: Update header block in pxa2xx_ssp.h (Jaroslav Kysela) [1917388]
- spi: pxa2xx: Add SSC2 and SSPSP2 SSP registers (Jaroslav Kysela) [1917388]
- ASoC: pxa: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: pxa: pxa-ssp: Remove redundant null check before clk_prepare_enable/clk_disable_unprepare (Jaroslav Kysela) [1917388]
- ASoC: mmp-sspa: drop unneeded snd_soc_dai_set_drvdata (Jaroslav Kysela) [1917388]
- ASoC: mmp-sspa: set phase two word length register (Jaroslav Kysela) [1917388]
- ASoC: pxa: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: Fix 7/8 spaces indentation in Kconfig (Jaroslav Kysela) [1917388]
- ASoC: jz4740-i2s: Remove manual DMA peripheral ID assignment (Jaroslav Kysela) [1917388]
- ASoC: jz4740-i2s: add missed checks for clk_get() (Jaroslav Kysela) [1917388]
- ASoC: Intel: fix error code cnl_set_dsp_D0() (Jaroslav Kysela) [1917388]
- ASoC: Intel: Skylake: Unassign ram_read and read_write ops (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove haswell solution (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove sst_dsp_get_thread_context (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove unused DSP interface fields (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: remove unneeded semicolon (Jaroslav Kysela) [1917388]
- catpt: Switch to use list_entry_is_head() helper (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Optimize applying user settings (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Correct clock selection for dai trigger (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Skip position update for unprepared streams (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Relax clock selection conditions (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Fix compilation when CONFIG_MODULES is disabled (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Cleanup after power routines streamlining (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Replace open coded variant of resource_intersection() (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Streamline power routines across LPT and WPT (Jaroslav Kysela) [1917388]
- ASoC: Intel: haswell: Add missing pm_ops (Jaroslav Kysela) [1917388]
- ASoC: Intel: cht_bsw_nau8824: Change SSP2-Codec DAI id to 0 (Jaroslav Kysela) [1917388]
- ASoC: Intel: cht_bsw_nau8824: Drop compress-cpu-dai bits (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5651: use semicolons rather than commas to separate statements (Jaroslav Kysela) [1917388]
- ASoC: Intel: broadwell: add missing pm_ops (Jaroslav Kysela) [1917388]
- ASoC: Intel: bdw-rt5677: add missing pm_ops (Jaroslav Kysela) [1917388]
- ASoC: Intel: broadwell: set card and driver name dynamically (Jaroslav Kysela) [1917388]
- ASoC: Intel: bdw-rt5650: Mark FE DAIs as nonatomic (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove rt5640 support for baytrail solution (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove max98090 support for baytrail solution (Jaroslav Kysela) [1917388]
- ASoC: Intel: atom: use inclusive language for SSP bclk/fsync (Jaroslav Kysela) [1917388]
- ASoC: Intel: Atom: add dynamic selection of DSP driver (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: select WANT_DEV_COREDUMP (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: add dynamic selection of DSP driver (Jaroslav Kysela) [1917388]
- ASoC: intel: SND_SOC_INTEL_KEEMBAY should depend on ARCH_KEEMBAY (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Add explicit DMADEVICES kconfig dependency (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove baytrail solution (Jaroslav Kysela) [1917388]
- ASoC: Intel: atom: Remove duplicate kconfigs (Jaroslav Kysela) [1917388]
- ASoC: remove zte zx drivers (Jaroslav Kysela) [1917388]
- ASoC: codecs: wsa881x: add missing stream rates and format (Jaroslav Kysela) [1917388]
- ASoC: wm_adsp: Remove unused control callback structure (Jaroslav Kysela) [1917388]
- ASoC: wm8994: Fix PM disable depth imbalance on error (Jaroslav Kysela) [1917388]
- ASoC: wm8903: remove useless assignments (Jaroslav Kysela) [1917388]
- ASoC: wm8523: Fix a typo in a comment (Jaroslav Kysela) [1917388]
- ASoC: wm8350: use semicolons rather than commas to separate statements (Jaroslav Kysela) [1917388]
- ASoC: wm5102: Use get_unaligned_be16() for dac_comp_coeff (Jaroslav Kysela) [1917388]
- ASoC: codecs: wcd934x: Set digital gain range correctly (Jaroslav Kysela) [1917388]
- ASoC: wcd9335: Remove unnecessary conversion to bool (Jaroslav Kysela) [1917388]
- ASoC: rt715: remove unused parameter (Jaroslav Kysela) [1917388]
- ASoC: mediatek: mt6359: add the calibration functions (Jaroslav Kysela) [1917388]
- ASoC: rt5682: clock driver must use the clock provider API (Jaroslav Kysela) [1917388]
- ASoC: rt1015: enable BCLK detection after calibration (Jaroslav Kysela) [1917388]
- ASoC: pcm512x: Add support for more data formats (Jaroslav Kysela) [1917388]
- ASoC: pcm512x: Move format check into `set_fmt()` (Jaroslav Kysela) [1917388]
- ASoC: pcm512x: Rearrange operations in `hw_params()` (Jaroslav Kysela) [1917388]
- ASoC: pcm512x: Fix not setting word length if DAIFMT_CBS_CFS (Jaroslav Kysela) [1917388]
- ASoC: mediatek: mt6359: Fix regulator_dev_lookup() fails for id "LDO_VAUD18" (Jaroslav Kysela) [1917388]
- ASoC: Remove mt6359_platform_driver_remove (Jaroslav Kysela) [1917388]
- ASoC: Fix vaud18 power leakage of mt6359 (Jaroslav Kysela) [1917388]
- ASoC: es8316: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Fix Bitclock polarity inversion (Jaroslav Kysela) [1917388]
- ASoC: cs35l33: fix an error code in probe() (Jaroslav Kysela) [1917388]
- ASoC: cs35l34: Add missing regmap use_single config (Jaroslav Kysela) [1917388]
- ASoC: wm8998: Fix PM disable depth imbalance on error (Jaroslav Kysela) [1917388]
- ASoC: codecs: wcd9335: Set digital gain range correctly (Jaroslav Kysela) [1917388]
- ASoC: TSCS454: remove unneeded semicolon (Jaroslav Kysela) [1917388]
- ASoC: TSCS42xx: remove unneeded semicolon (Jaroslav Kysela) [1917388]
- ASoC: ts3a227e: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: tlv320: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: tas571x: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: tas*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: tas2562: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: remove remnants of sirf prima/atlas audio codec (Jaroslav Kysela) [1917388]
- ASoC: rt5645: add error checking to rt5645_probe function (Jaroslav Kysela) [1917388]
- Revert "ASoC: rt5645: fix a NULL pointer dereference" (Jaroslav Kysela) [1917388]
- ASoC: rt1015: modify calibration sequence for better performance (Jaroslav Kysela) [1917388]
- ASoC: rk3328: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: pcm179x: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: pcm1789: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: nau8315: revise the power event of EN_PIN dapm widget for symmetry (Jaroslav Kysela) [1917388]
- ASoC: nau8315: add codec driver (Jaroslav Kysela) [1917388]
- ASoC: gtm601: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: es8328: Remove redundant null check before clk_disable_unprepare (Jaroslav Kysela) [1917388]
- ASoC: es7241: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: es7134: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: da9055: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: da7219: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: codecs/jz4770: Add DAPM widget to set HP out to cap-less mode (Jaroslav Kysela) [1917388]
- ASoC: codecs/jz4770: Don't change cap-couple setting in HP PMU/PMD (Jaroslav Kysela) [1917388]
- ASoC: codecs/jz4770: Adjust timeouts for cap-coupled outputs (Jaroslav Kysela) [1917388]
- ASoC: codecs/jz4770: Reset interrupt flags in bias PREPARE (Jaroslav Kysela) [1917388]
- ASoC: codecs/jz47xx: Use regmap_{set,clear}_bits (Jaroslav Kysela) [1917388]
- treewide: Use fallthrough pseudo-keyword (Jaroslav Kysela) [1917388]
- ASoC: inno_rk3036: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: hdac_hdmi: remove cancel_work_sync in runtime suspend (Jaroslav Kysela) [1917388]
- ASoC: hdmi-codec: Add RX support (Jaroslav Kysela) [1917388]
- ALSA: cx46xx: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ASoC: topology: KUnit: Convert from cpu to data format (Jaroslav Kysela) [1917388]
- ASoC: topology: KUnit: Add KUnit tests passing topology with PCM to snd_soc_tplg_component_load (Jaroslav Kysela) [1917388]
- ASoC: topology: KUnit: Add KUnit tests passing empty topology with variants to snd_soc_tplg_component_load (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: use semicolons rather than commas to separate statements (Jaroslav Kysela) [1917388]
- ASoC: tegra: Simplify with dma_set_mask_and_coherent() (Jaroslav Kysela) [1917388]
- ASoC: tegra: Don't warn on probe deferral (Jaroslav Kysela) [1917388]
- ASoC: tegra: tegra_rt5640: use devm_snd_soc_register_card() (Jaroslav Kysela) [1917388]
- ASoC: tegra: tegra_wm8753: use devm_snd_soc_register_card() (Jaroslav Kysela) [1917388]
- ASoC: tegra: trimslice.c: use devm_snd_soc_register_card() (Jaroslav Kysela) [1917388]
- ASoC: ti: davinci-evm: Remove redundant null check before clk_disable_unprepare (Jaroslav Kysela) [1917388]
- ASoC: ti: davinci: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: ti: j721e-evm: Fix compiler warning when CONFIG_OF=n (Jaroslav Kysela) [1917388]
- ASoC: ti: omap-abe-twl6040: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: ti: omap-mcsp: remove duplicate test (Jaroslav Kysela) [1917388]
- ASoC: uniphier: Simplify the return expression of uniphier_aio_startup (Jaroslav Kysela) [1917388]
- ASoC: uniphier: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: ux500: mop500: rename shadowing variable (Jaroslav Kysela) [1917388]
- ASoC: ux500: mop500: align function prototype (Jaroslav Kysela) [1917388]
- ALSA: x86: Simplify with dma_set_mask_and_coherent() (Jaroslav Kysela) [1917388]
- ASoC: cx2072x: Fix doubly definitions of Playback and Capture streams (Jaroslav Kysela) [1917388]
- ASoC: cs42l56: fix up error handling in probe (Jaroslav Kysela) [1917388]
- ASoC: cs42l52: Minor tidy up of error paths (Jaroslav Kysela) [1917388]
- ASoC: cs42l51: manage mclk shutdown delay (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Wait for PLL to lock before switching to it (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Only start PLL if it is needed (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Wait at least 150us after writing SCLK_PRESENT (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Use bclk from hw_params if set_sysclk was not called (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Report jack and button detection (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Add Capture Support (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Fix channel width support (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Always wait at least 3ms after reset (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Don't enable/disable regulator at Bias Level (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Fix mixer volume control (Jaroslav Kysela) [1917388]
- ASoC: cs*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: bd28623: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: alc5632: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: ak4458: Add MODULE_DEVICE_TABLE (Jaroslav Kysela) [1917388]
- ASoC: ak4118: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: alc5623: skip of_device_id table when !CONFIG_OF (Jaroslav Kysela) [1917388]
- ASoC: ak5558: Correct the dai name for ak5552 (Jaroslav Kysela) [1917388]
- ASoC: ak5558: change function name to ak5558_reset (Jaroslav Kysela) [1917388]
- ASoC: ak5558: Add support for ak5552 (Jaroslav Kysela) [1917388]
- ASoC: ak5558: Add MODULE_DEVICE_TABLE (Jaroslav Kysela) [1917388]
- ASoC: ak5558: mark OF related data as maybe unused (Jaroslav Kysela) [1917388]
- ASoC: adau1977: remove platform data and move micbias bindings include (Jaroslav Kysela) [1917388]
- ASoC: adau17x1: Remove redundant null check before clk_disable_unprepare (Jaroslav Kysela) [1917388]
- ASoC: adau*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: Add ADAU1372 audio CODEC support (Jaroslav Kysela) [1917388]
- ASoC: amd: fix for pcm_read() error (Jaroslav Kysela) [1917388]
- ASoC: amd: drop S24_LE format support (Jaroslav Kysela) [1917388]
- ASoC: amd: Add support for ALC1015P codec in acp3x machine driver (Jaroslav Kysela) [1917388]
- ASoC: amd: renoir: acp3x-pdm-dma: constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: amd: raven: acp3x-i2s: Constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: amd: renoir: acp3x-pdm-dma: remove unnecessary assignments (Jaroslav Kysela) [1917388]
- ASoC: amd: fix acpi dependency kernel warning (Jaroslav Kysela) [1917388]
- ASoC: amd: update spdx license for acp machine driver (Jaroslav Kysela) [1917388]
- ASoC: amd: Add support for RT5682 codec in machine driver (Jaroslav Kysela) [1917388]
- ASoC: amd: Adding DAI LINK for rt1015 codec (Jaroslav Kysela) [1917388]
- ASoC: amd: Adding support for ALC1015 codec in machine driver (Jaroslav Kysela) [1917388]
- ASoC: amd: AMD RV RT5682 should depends on CROS_EC (Jaroslav Kysela) [1917388]
- ASoC: amd: renoir: remove invalid kernel-doc comment (Jaroslav Kysela) [1917388]
- ASoC: AMD Renoir - refine DMI entries for some Lenovo products (Jaroslav Kysela) [1917388]
- ASoC: AMD Renoir - add DMI entry for Lenovo ThinkPad X395 (Jaroslav Kysela) [1917388]
- ASoC: amd: Replacing MSI with Legacy IRQ model (Jaroslav Kysela) [1917388]
- ASoC: AMD Renoir - add DMI entry for Lenovo ThinkPad E14 Gen 2 (Jaroslav Kysela) [1917388]
- ASoC: AMD Renoir - add DMI table to avoid the ACP mic probe (broken BIOS) (Jaroslav Kysela) [1917388]
- ASoC: AMD Raven/Renoir - fix the PCI probe (PCI revision) (Jaroslav Kysela) [1917388]
- ASoC: amd: change clk_get() to devm_clk_get() and add missed checks (Jaroslav Kysela) [1917388]
- ASoC: amd: Return -ENODEV for non-existing ACPI call (Jaroslav Kysela) [1917388]
- ASoC: amd: support other audio modes for raven (Jaroslav Kysela) [1917388]
- ASoC: topology: adhere to KUNIT formatting standard (Jaroslav Kysela) [1917388]
- ASoC: topology: KUnit: Add KUnit tests passing various arguments to snd_soc_tplg_component_load (Jaroslav Kysela) [1917388]
- ALSA: trident: Use DIV_ROUND_CLOSEST() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: sonicvibes: Use DIV_ROUND_CLOSEST() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: sis7019: Use DIV_ROUND_CLOSEST() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: remove unneeded break (Jaroslav Kysela) [1917388]
- ALSA: hdsp: don't disable if not enabled (Jaroslav Kysela) [1917388]
- ALSA: hdsp: hardware output loopback (Jaroslav Kysela) [1917388]
- ALSA: pci: mixart: fix kernel-doc warning (Jaroslav Kysela) [1917388]
- ALSA: maestro: Use DIV_ROUND_CLOSEST() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: lola: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: intel8x0: Fix missing check in snd_intel8x0m_create (Jaroslav Kysela) [1917388]
- ALSA: intel8x0: Don't update period unless prepared (Jaroslav Kysela) [1917388]
- ALSA: hda: Avoid spurious unsol event handling during S3/S4 (Jaroslav Kysela) [1917388]
- ALSA: ens1370: Use DIV_ROUND_CLOSEST() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: emu10k1: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: emu10k1: Use dma_set_mask_and_coherent to simplify code (Jaroslav Kysela) [1917388]
- ALSA: ctxfi: fix comment syntax in file headers (Jaroslav Kysela) [1917388]
- ALSA: ctxfi: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: ctxfi: cthw20k2: fix mask on conf to allow 4 bits (Jaroslav Kysela) [1917388]
- ALSA: bt87x: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: azt3328: Assign boolean values to a bool variable (Jaroslav Kysela) [1917388]
- ALSA: asihpi: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: dice: fix stream format for TC Electronic Konnekt Live at high sampling transfer frequency (Jaroslav Kysela) [1917388]
- ALSA: dice: disable double_pcm_frames mode for M-Audio Profire 610, 2626 and Avid M-Box 3 Pro (Jaroslav Kysela) [1917388]
- ALSA: firewire-lib: fix amdtp_packet tracepoints event for packet_index field (Jaroslav Kysela) [1917388]
- ALSA: firewire-lib: fix calculation for size of IR context payload (Jaroslav Kysela) [1917388]
- ALSA: firewire-lib: fix check for the size of isochronous packet payload (Jaroslav Kysela) [1917388]
- ALSA: bebob/oxfw: fix Kconfig entry for Mackie d.2 Pro (Jaroslav Kysela) [1917388]
- ALSA: dice: fix stream format at middle sampling rate for Alesis iO 26 (Jaroslav Kysela) [1917388]
- ALSA: bebob: enable to deliver MIDI messages for multiple ports (Jaroslav Kysela) [1917388]
- ALSA: bebob: detect the number of available MIDI ports (Jaroslav Kysela) [1917388]
- ALSA: bebob: code refactoring for stream format detection (Jaroslav Kysela) [1917388]
- ALSA: dice: fix null pointer dereference when node is disconnected (Jaroslav Kysela) [1917388]
- ALSA: fireface: fix to parse sync status register of latter protocol (Jaroslav Kysela) [1917388]
- ALSA: bebob: remove an unnecessary condition in hwdep_read() (Jaroslav Kysela) [1917388]
- ALSA: fireface: remove unnecessary condition in hwdep_read() (Jaroslav Kysela) [1917388]
- ALSA: oxfw: remove an unnecessary condition in hwdep_read() (Jaroslav Kysela) [1917388]
- ALSA: dice: add support for Lexicon I-ONIX FW810s (Jaroslav Kysela) [1917388]
- ALSA: firewire-tascam: Fix integer overflow in midi_port_work() (Jaroslav Kysela) [1917388]
- ALSA: fireface: Fix integer overflow in transmit_midi_msg() (Jaroslav Kysela) [1917388]
- ALSA: firewire: Clean up a locking issue in copy_resp_to_buf() (Jaroslav Kysela) [1917388]
- ALSA: firewire: fix comparison to bool warning (Jaroslav Kysela) [1917388]
- ALSA: firewire: Replace tasklet with work (Jaroslav Kysela) [1917388]
- ALSA: firewire: convert tasklets to use new tasklet_setup() API (Jaroslav Kysela) [1917388]
- ALSA: vx: Use roundup() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: pcsp: Fix fall-through warnings for Clang (Jaroslav Kysela) [1917388]
- ALSA: dummy: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: aloop: Fix initialization of controls (Jaroslav Kysela) [1917388]
- ALSA: aloop: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: atmel: Drop superfluous ioctl PCM ops (Jaroslav Kysela) [1917388]
- ASoC: SOF: ext_manifest: use explicit number for elem_type (Jaroslav Kysela) [1917388]
- ASoC: soc-acpi: allow for partial match in parent name (Jaroslav Kysela) [1917388]
- ASoC: soc-acpi: add helper to identify parent driver. (Jaroslav Kysela) [1917388]
- ASoC: soc-acpi: add new fields for mach_params (Jaroslav Kysela) [1917388]
- ASoC: rt5682: Add a new property for the DMIC clock driving (Jaroslav Kysela) [1917388]
- ALSA: hda: Change AZX_MAX_BUF_SIZE from 1GB to 4MB (Jaroslav Kysela) [1917388]
- soundwire: sysfs: Constify static struct attribute_group (Jaroslav Kysela) [1917388]
- ALSA: remove trailing semicolon in macro definition (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: don't send DAI_CONFIG IPC for older firmware (Jaroslav Kysela) [1917388]
- ASoC: SOF: use current DAI config during resume (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: Update ADL P to use its own descriptor (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: add missing use_acpi_target_states for TGL platforms (Jaroslav Kysela) [1917388]
- ASoC: SOF: topology: remove useless code (Jaroslav Kysela) [1917388]
- ASoC: SOF: Kconfig: fix typo of SND_SOC_SOF_PCI (Jaroslav Kysela) [1917388]
- ASOC: SOF: simplify nocodec mode (Jaroslav Kysela) [1917388]
- ASoC: SOF: pcm: export snd_pcm_dai_link_fixup (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: update set_mach_params() (Jaroslav Kysela) [1917388]
- ASoC: SOF: change signature of set_mach_params() callback (Jaroslav Kysela) [1917388]
- ASoC: SOF: add Kconfig option for probe workqueues (Jaroslav Kysela) [1917388]
- ASoC: SOF: Simplify sof_probe_complete handling for acpi/pci/of (Jaroslav Kysela) [1917388]
- ASoC: SOF: core: Add missing error prints to device probe operation (Jaroslav Kysela) [1917388]
- ASoC: SOF: match SSP config with pcm hw params (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: HDA: add hw params callback for SSP DAIs (Jaroslav Kysela) [1917388]
- ASoC: SOF: parse multiple SSP DAI and hw configs (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: drop display power on/off in D0i3 flows (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: move ELH chip info (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: APL: set shutdown callback to hda_dsp_shutdown (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: CNL: set shutdown callback to hda_dsp_shutdown (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: ICL: set shutdown callback to hda_dsp_shutdown (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: TGL: set shutdown callback to hda_dsp_shutdown (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: TGL: fix EHL ops (Jaroslav Kysela) [1917388]
- ASoC: SOF: core: harden shutdown helper (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: HDA: fix core status verification (Jaroslav Kysela) [1917388]
- ASoC: SOF: add a helper to get topology configured mclk (Jaroslav Kysela) [1917388]
- ASoC: SOF: only allocate debugfs cache buffers for IPC flood entries (Jaroslav Kysela) [1917388]
- ASoC: SOF: fix debugfs initialisation error handling (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: enable async suspend (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: turn off display power in resume (Jaroslav Kysela) [1917388]
- ASoC: SOF: intel: fix wrong poll bits in dsp power down (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: unregister DMIC device on probe error (Jaroslav Kysela) [1917388]
- ALSA: hda: move Intel SoundWire ACPI scan to dedicated module (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: SoundWire: simplify Kconfig (Jaroslav Kysela) [1917388]
- ASoC: SOF: pci: move DSP_CONFIG use to platform-specific drivers (Jaroslav Kysela) [1917388]
- ASoC: SOF: pci: split PCI into different drivers (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: add sof_icl_ops for ICL platforms (Jaroslav Kysela) [1917388]
- ASoC: SOF: ACPI: avoid reverse module dependency (Jaroslav Kysela) [1917388]
- ASoC: SOF: relax ABI checks and avoid unnecessary warnings (Jaroslav Kysela) [1917388]
- ASoC: SOF: fix runtime pm usage mismatch after probe errors (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: add dev_dbg() when DMIC number is overridden (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: HDA: don't keep a temporary variable (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: detect DMIC number in SoundWire mixed config (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: SoundWire: refine ACPI match (Jaroslav Kysela) [1917388]
- ASoC: SOF: HDA: (cosmetic) simplify hda_dsp_d0i3_work() (Jaroslav Kysela) [1917388]
- ASoC: SOF: remove unused functions (Jaroslav Kysela) [1917388]
- ASoC: SOF: fix string format for errors (Jaroslav Kysela) [1917388]
- ASoC: SOF: add missing pm debug (Jaroslav Kysela) [1917388]
- ASoC: SOF: sof-pci-dev: add missing Up-Extreme quirk (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: release display power at link_power (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: use hdac_ext fine-grained link management (Jaroslav Kysela) [1917388]
- ASoC: SOF: debug: Fix a potential issue on string buffer termination (Jaroslav Kysela) [1917388]
- ASoC: SOF: add be_hw_params_fixup() for ALH (Jaroslav Kysela) [1917388]
- ASoC: SOF: topology: Prevent NULL pointer dereference with no TLV (Jaroslav Kysela) [1917388]
- ASoC: SOF: intel: hda-loader: use snd_sof_dsp_core_power_down/up APIs (Jaroslav Kysela) [1917388]
- ASoC: SOF: Filter out unneeded core power up/downs (Jaroslav Kysela) [1917388]
- ASoC: SOF: update dsp core power status in common APIs (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda-loader: keep init cores alive (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: use snd_sof_dsp_core_power_up/down API (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: cancel D0i3 work during runtime suspend (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: Enable DMI L1 for trace (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: enable DMI L1 for D0i3-compatible streams (Jaroslav Kysela) [1917388]
- ASoC: SOF: add a pointer to download repo in case FW request fails (Jaroslav Kysela) [1917388]
- ASoC: SOF: SND_INTEL_DSP_CONFIG dependency (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: soundwire: fix select/depend unmet dependencies (Jaroslav Kysela) [1917388]
- ASoC: SOF: intel: Simplify with dma_set_mask_and_coherent() (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: initial support to AlderLake-P (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: tgl: do thorough remove at .shutdown() callback (Jaroslav Kysela) [1917388]
- ASoC: SOF: sof-pci-dev: add .shutdown() callback (Jaroslav Kysela) [1917388]
- ASoC: SOF: add snd_sof_device_shutdown() helper for shutdown (Jaroslav Kysela) [1917388]
- ASoC: SOF: add .shutdown() callback to snd_sof_dsp_ops (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: Avoid checking jack on system suspend (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: Modify existing helper to disable WAKEEN (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: hda: Resume codec to do jack detection (Jaroslav Kysela) [1917388]
- ASoC: SOF: add mutex to protect the dsp_power_state access (Jaroslav Kysela) [1917388]
- ASoC: SOF: Fix spelling mistake in Kconfig "ond" -> "and" (Jaroslav Kysela) [1917388]
- ASoC: SOF: imx: update kernel-doc description (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: add SoundWire support for ADL-S (Jaroslav Kysela) [1917388]
- ASoC: SOF: control: fix cppcheck warning in snd_sof_volume_info() (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: allow for coexistence between SOF and catpt drivers (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: allow for coexistence between SOF and Atom/SST drivers (Jaroslav Kysela) [1917388]
- ASoC: SOF: acpi: add dynamic selection of DSP driver (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: initial support for Alderlake-S (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: fix Kconfig dependency for SND_INTEL_DSP_CONFIG (Jaroslav Kysela) [1917388]
- ASoC: SOF: Intel: fix Kconfig punctuation and wording (Jaroslav Kysela) [1917388]
- ASoC: SOF: Kconfig: fix Kconfig punctuation and wording (Jaroslav Kysela) [1917388]
- ASoC: SOF: imx: fix Kconfig punctuation (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add quirk for the Lenovo Miix 3-830 tablet (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add quirk for the Glavey TM800A550L tablet (Jaroslav Kysela) [1917388]
- ASoC: Intel: KMB: Fix random noise at the HDMI output (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_sdw: add quirk for new ADL-P Rvp (Jaroslav Kysela) [1917388]
- ASoC: Intel: soc-acpi: add ADL jack-less SoundWire configurations (Jaroslav Kysela) [1917388]
- ASoC: Intel: soc-acpi: add ADL SoundWire base configurations (Jaroslav Kysela) [1917388]
- ASoC: Intel: kbl_da7219_max98927: Fix kabylake_ssp_fixup function (Jaroslav Kysela) [1917388]
- ASoC: Intel: KMB: Constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoc: Intel: board: add BE DAI link for WoV (Jaroslav Kysela) [1917388]
- ASoC: Intel: kbl: Add MST route change to kbl machine drivers (Jaroslav Kysela) [1917388]
- ASoC: Intel: cht_bsw_rt5672: Set card.components string (Jaroslav Kysela) [1917388]
- ASoC: Intel: cht_bsw_rt5672: Add support for Bay Trail CR / SSP0 (Jaroslav Kysela) [1917388]
- ASoC: Intel: Baytrail: Add quirk for the Dell Venue 10 Pro 5055 tablet (Jaroslav Kysela) [1917388]
- ASoC: Intel: Unify the thinkpad10 and aegex10 byt-match quirks (Jaroslav Kysela) [1917388]
- ASoC: Intel: Boards: cml_da7219_max98390: add capture stream for echo reference (Jaroslav Kysela) [1917388]
- ASoC: Intel: Fix a typo (Jaroslav Kysela) [1917388]
- ASoC: Intel: Fix a typo (Jaroslav Kysela) [1917388]
- ASoC: Intel: Fix a typo (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add quirk for the Chuwi Hi8 tablet (Jaroslav Kysela) [1917388]
- ASoC: Intel: add max98390 echo reference support (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_wm5102: Add jack detect support (Jaroslav Kysela) [1917388]
- ASoC: intel: atom: Remove 44100 sample-rate from the media and deep-buffer DAI descriptions (Jaroslav Kysela) [1917388]
- ASoC: intel: atom: Stop advertising non working S24LE support (Jaroslav Kysela) [1917388]
- ASoC: intel: sof_rt5682: use the topology mclk (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_rt5682: Add ALC1015Q-VB speaker amp support (Jaroslav Kysela) [1917388]
- ASoC: Intel: Skylake: skl-topology: fix -frame-larger-than (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add used AIF to the components string (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Enable jack-detect support on Asus T100TAF (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_wm5102: remove unused static variable (Jaroslav Kysela) [1917388]
- ASoC: Intel: atom: fix kernel-doc (Jaroslav Kysela) [1917388]
- ASoC: Intel: soc-acpi: remove TGL RVP mixed SoundWire/TDM config (Jaroslav Kysela) [1917388]
- ASoC: Intel: soc-acpi: remove unused TGL table with rt5682 only (Jaroslav Kysela) [1917388]
- ASoC: Intel: Skylake: Compile when any configuration is selected (Jaroslav Kysela) [1917388]
- ASoC: Intel: boards: sof-wm8804: add check for PLL setting (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Fix HP Pavilion x2 10-p0XX OVCD current threshold (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add quirk for the Acer One S1002 tablet (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5651: Add quirk for the Jumper EZpad 7 tablet (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add quirk for the Voyo Winpad A15 tablet (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add quirk for the Estar Beauty HD MID 7316R tablet (Jaroslav Kysela) [1917388]
- ASoC: Intel: soc-acpi: add ACPI matching table for HP Spectre x360 (Jaroslav Kysela) [1917388]
- ASoC: intel: sof_sdw: add trace for dai links (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_sdw: detect DMIC number based on mach params (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_sdw: add mic:dmic and cfg-mics component strings (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_sdw: add quirk for HP Spectre x360 convertible (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof-sdw: indent and add quirks consistently (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_sdw: reorganize quirks by generation (Jaroslav Kysela) [1917388]
- ASoC: Intel: boards: max98373: get dapm from cpu_dai (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_wm5102: Add machine driver for BYT/WM5102 (Jaroslav Kysela) [1917388]
- ASoC: Intel: Add DMI quirk table to soc_intel_is_byt_cr() (Jaroslav Kysela) [1917388]
- ASoC: Intel: KMB: Support IEC958 encoded PCM format (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_sdw: add missing TGL_HDMI quirk for Dell SKU 0A3E (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_sdw: add missing TGL_HDMI quirk for Dell SKU 0A32 (Jaroslav Kysela) [1917388]
- ASoC: Intel: sof_sdw: add missing TGL_HDMI quirk for Dell SKU 0A5E (Jaroslav Kysela) [1917388]
- ASoC: Intel: remove unneeded semicolon (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Wake up device before configuring SSP port (Jaroslav Kysela) [1917388]
- ASoC: Intel: remove duplicate MODULE_LICENSE/DESCRIPTION tags (Jaroslav Kysela) [1917388]
- ASoC: Intel: adl: remove sof_fw_filename setting in ADL snd_soc_acpi_mach (Jaroslav Kysela) [1917388]
- ASoC: Intel: common: add ACPI matching tables for Alder Lake (Jaroslav Kysela) [1917388]
- ASoC: Intel: Skylake: Zero snd_ctl_elem_value (Jaroslav Kysela) [1917388]
- ASoC: Intel: Skylake: skl-topology: Fix OOPs ib skl_tplg_complete (Jaroslav Kysela) [1917388]
- ASoC: intel: skl: Simplify with dma_set_mask_and_coherent() (Jaroslav Kysela) [1917388]
- ASoC: Intel: cht_bsw_nau8824: Move snd_soc_dai_set_tdm_slot call to cht_codec_fixup (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add quirk for the Mele PCG03 Mini PC (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add new BYT_RT5640_NO_INTERNAL_MIC_MAP input-mapping (Jaroslav Kysela) [1917388]
- ASoC: Intel: byt/cht: set pm ops dynamically (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove sst_pdata structure (Jaroslav Kysela) [1917388]
- ASoC: Intel: Make atom components independent of sst-dsp (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove SST-legacy specific constants (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove unused DSP operations (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove SST firmware components (Jaroslav Kysela) [1917388]
- ASoC: Intel: Remove SST ACPI component (Jaroslav Kysela) [1917388]
- ASoC: Intel: Select catpt and deprecate haswell (Jaroslav Kysela) [1917388]
- ASoC: Intel: bdw-5677: Remove haswell-solution specific code (Jaroslav Kysela) [1917388]
- ASoC: Intel: bdw-5650: Remove haswell-solution specific code (Jaroslav Kysela) [1917388]
- ASoC: Intel: broadwell: Remove haswell-solution specific code (Jaroslav Kysela) [1917388]
- ASoC: Intel: haswell: Remove haswell-solution specific code (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Simple sysfs attributes (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Event tracing (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Device driver lifecycle (Jaroslav Kysela) [1917388]
- drivers: provide devm_platform_get_and_ioremap_resource() (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: PCM operations (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Firmware loading and context restore (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Define DSP operations (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Add IPC message handlers (Jaroslav Kysela) [1917388]
- ASoC: Intel: catpt: Implement IPC protocol (Jaroslav Kysela) [1917388]
- ASoC: Intel: Add catpt base members (Jaroslav Kysela) [1917388]
- resource: Introduce resource_intersection() for overlapping resources (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add quirk for ARCHOS Cesium 140 (Jaroslav Kysela) [1917388]
- ASoC: Intel: boards: byt/cht: set card and driver name at run time (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcr_rt5640: Add new BYT_RT5640_NO_SPEAKERS quirk-flag (Jaroslav Kysela) [1917388]
- ASoC: Intel: KMB: Enable DMA transfer mode (Jaroslav Kysela) [1917388]
- ASoC: Intel: keembay: use inclusive language for bclk and fsync (Jaroslav Kysela) [1917388]
- ASoC: Intel: bytcht_es8316: Remove comment about SSP0 being untested (Jaroslav Kysela) [1917388]
- ASoC: rt1015: remove bclk_ratio (Jaroslav Kysela) [1917388]
- ASoC: cs42l73: Add missing regmap use_single config (Jaroslav Kysela) [1917388]
- ASoC: cs53l30: Add missing regmap use_single config (Jaroslav Kysela) [1917388]
- ASoC: sti-sas: add missing MODULE_DEVICE_TABLE (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Regmap must use_single_read/write (Jaroslav Kysela) [1917388]
- ASoC: rt711-sdca: fix the function number of SDCA control for feature unit 0x1E (Jaroslav Kysela) [1917388]
- ASoC: da7219: do not request a new clock consummer reference (Jaroslav Kysela) [1917388]
- ASoC: max98088: fix ni clock divider calculation (Jaroslav Kysela) [1917388]
- ASoC: rt711-sdca: add the notification when volume changed (Jaroslav Kysela) [1917388]
- ASoC: rt711-sdca: change capture switch controls (Jaroslav Kysela) [1917388]
- ASoC: da7219: properly get clk from the provider (Jaroslav Kysela) [1917388]
- ASoC: wcd934x: use the clock provider API (Jaroslav Kysela) [1917388]
- ASoC: adau17x1: Avoid overwriting CHPF (Jaroslav Kysela) [1917388]
- ASoC: ak4458: enable daisy chain (Jaroslav Kysela) [1917388]
- ASoC: rt1015p: add support on format S32_LE (Jaroslav Kysela) [1917388]
- ASoC: rt286: Generalize support for ALC3263 codec (Jaroslav Kysela) [1917388]
- ASoC: rt298: Configure combo jack for headphones (Jaroslav Kysela) [1917388]
- ASoC: rt286: Configure combo jack for headphones (Jaroslav Kysela) [1917388]
- ASoC: sigmadsp: Disable cache mechanism for readbacks (Jaroslav Kysela) [1917388]
- ASoC: rt286: Make RT286_SET_GPIO_* readable and writable (Jaroslav Kysela) [1917388]
- ASoC: rt286: Fix upper byte in DMIC2 configuration (Jaroslav Kysela) [1917388]
- ASoC: ak4458: check reset control status (Jaroslav Kysela) [1917388]
- ASoC: codecs: rt5682: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt715: remove kcontrols which no longer be used (Jaroslav Kysela) [1917388]
- ASoC: rt715: add main capture switch and main capture volume (Jaroslav Kysela) [1917388]
- ASoC: rt715: modification for code simplicity (Jaroslav Kysela) [1917388]
- ASoC: codecs: wcd934x: Fix missing IRQF_ONESHOT as only threaded handler (Jaroslav Kysela) [1917388]
- ASoC: cs35l35: remove unused including <linux/version.h> (Jaroslav Kysela) [1917388]
- ASoC: ak5558: correct reset polarity (Jaroslav Kysela) [1917388]
- ASoC: cs35l36: Fix an error handling path in 'cs35l36_i2c_probe()' (Jaroslav Kysela) [1917388]
- ASoC: cs35l35: Fix an error handling path in 'cs35l35_i2c_probe()' (Jaroslav Kysela) [1917388]
- ASoC: wcd9335: constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: rt1019: remove registers to sync with rt1019 datasheet (Jaroslav Kysela) [1917388]
- ASoC: rt711-sdca: Constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: wcd934x: constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: rt715-sdca: Constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: tas2770: Constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: cx2072x: constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: rt1019: constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: codecs: tlv320aic3x: add AIC3106 (Jaroslav Kysela) [1917388]
- ASoC: codecs: wsa881x: constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: codecs: Fix runtime PM imbalance in tas2552_probe (Jaroslav Kysela) [1917388]
- ASoC: rt1011: remove pack_id check in rt1011 (Jaroslav Kysela) [1917388]
- ASoC: codecs: tlv320aic3x: add SPI support (Jaroslav Kysela) [1917388]
- ASoC: codecs: tlv320aic3x: move I2C to separated file (Jaroslav Kysela) [1917388]
- ASoC: codecs: tlv320aic3x: rename probe function (Jaroslav Kysela) [1917388]
- ASoC: codecs: tlv320aic3x: move model definitions (Jaroslav Kysela) [1917388]
- ASoC: max98390: Add support for tx slot configuration. (Jaroslav Kysela) [1917388]
- ASoC: ak5558: Fix s/show/slow/ typo (Jaroslav Kysela) [1917388]
- ASoC: tlv320aic32x4: Register clocks before registering component (Jaroslav Kysela) [1917388]
- ASoC: tlv320aic32x4: Increase maximum register in regmap (Jaroslav Kysela) [1917388]
- ASoC: rt5670: Add a rt5670_components() helper (Jaroslav Kysela) [1917388]
- ASoC: rt5670: Add a quirk for the Dell Venue 10 Pro 5055 (Jaroslav Kysela) [1917388]
- ASoC: da732x: simplify code (Jaroslav Kysela) [1917388]
- ASoC: lm49453: fix useless assignment before return (Jaroslav Kysela) [1917388]
- ASoC: mediatek: mt6359: Fix spelling mistake "reate" -> "create" (Jaroslav Kysela) [1917388]
- ASoC: mediatek: mt6359: add MT6359 accdet jack driver (Jaroslav Kysela) [1917388]
- ASoC: max98373: Added 30ms turn on/off time delay (Jaroslav Kysela) [1917388]
- ASoC: max98373: Changed amp shutdown register as volatile (Jaroslav Kysela) [1917388]
- ASoC: wm8960: Remove bitclk relax condition in wm8960_configure_sysclk (Jaroslav Kysela) [1917388]
- ASoC: max98373: Added controls for autorestart config (Jaroslav Kysela) [1917388]
- ASoC: rt1015: Add bclk detection and dc detection (Jaroslav Kysela) [1917388]
- ASoC: wm8960: Fix wrong bclk and lrclk with pll enabled for some chips (Jaroslav Kysela) [1917388]
- ASoC: rt1019: make symbol 'rt1019_i2c_driver' static (Jaroslav Kysela) [1917388]
- ASoC: rt1019: add rt1019 amplifier driver (Jaroslav Kysela) [1917388]
- ASoC: tscs454: remove useless test on PLL disable (Jaroslav Kysela) [1917388]
- ASoC: tlv320dac33: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: tas2770: remove useless initialization (Jaroslav Kysela) [1917388]
- ASoC: tas2562: remove warning on return value (Jaroslav Kysela) [1917388]
- ASoC: tas2562: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: sti-sas: remove unused struct members (Jaroslav Kysela) [1917388]
- ASoC: sigmadsp: align function prototype (Jaroslav Kysela) [1917388]
- ASoC: pcm1681: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: nau8825: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: mt6359: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: mt6358: remove useless initializations (Jaroslav Kysela) [1917388]
- ASoC: max98090: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: hdmi-codec: remove unused spk_mask member (Jaroslav Kysela) [1917388]
- ASoC: hdmi-codec: remove useless initialization (Jaroslav Kysela) [1917388]
- ASoC: hdac_hdmi: align function arguments (Jaroslav Kysela) [1917388]
- ASoC: hdac_hdmi: remove useless initializations (Jaroslav Kysela) [1917388]
- ASoC: da7219-aad: remove useless initialization (Jaroslav Kysela) [1917388]
- ASoC: cx2070x: remove duplicate else branch (Jaroslav Kysela) [1917388]
- ASoC: cx2070x: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: adau1977: remove useless return (Jaroslav Kysela) [1917388]
- treewide: Replace GPLv2 boilerplate/reference with SPDX - rule 177 (Jaroslav Kysela) [1917388]
- ASoC: ad1836: remove useless return (Jaroslav Kysela) [1917388]
- ASoC: ab8500-codec: remove useless structure (Jaroslav Kysela) [1917388]
- ASoC: rt711: add snd_soc_component remove callback (Jaroslav Kysela) [1917388]
- ASoC: rt5659: Update MCLK rate in set_sysclk() (Jaroslav Kysela) [1917388]
- ASoC: rt5640: Rename 'Mono DAC Playback Volume' to 'DAC2 Playback Volume' (Jaroslav Kysela) [1917388]
- ASoC: rt1015p: add acpi device id for rt1015p (Jaroslav Kysela) [1917388]
- ASoC: wm_hubs: align function prototype (Jaroslav Kysela) [1917388]
- ASoC: wm_adsp: simplify return value (Jaroslav Kysela) [1917388]
- ASoC: wm8996: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: wm8994: align function prototype (Jaroslav Kysela) [1917388]
- ASoC: wm8978: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: wm8958-dsp2: rename local 'control' arrays (Jaroslav Kysela) [1917388]
- ASoC: arizona: fix function argument (Jaroslav Kysela) [1917388]
- ASoC: wm2200: remove unused structure (Jaroslav Kysela) [1917388]
- ASoC: rt5682: add delay time of workqueue to control next IRQ event (Jaroslav Kysela) [1917388]
- ASoC: rt715-sdca: Remove unused including <linux/version.h> (Jaroslav Kysela) [1917388]
- ASoC: rt715-sdca: Fix return value check in rt715_sdca_sdw_probe() (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Set clock source for both ways of stream (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Provide finer control on playback path (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Disable regulators if probe fails (Jaroslav Kysela) [1917388]
- ASoC: cs42l42: Remove power if the driver is being removed (Jaroslav Kysela) [1917388]
- ASoC: rt711-sdca: Add RT711 SDCA vendor-specific driver (Jaroslav Kysela) [1917388]
- ASoC: rt5645: The ALC3270 variant does not have a headset-mic pin (Jaroslav Kysela) [1917388]
- ASoC: rt5645: Move rt5645_platform_data to sound/soc/codecs/rt5645.c (Jaroslav Kysela) [1917388]
- ASoC: codecs/jz4770: Remove superfluous error message (Jaroslav Kysela) [1917388]
- ASoC: rt715-sdca: Add RT715 sdca vendor-specific driver (Jaroslav Kysela) [1917388]
- ASoC: wm8962: Relax bit clock divider searching (Jaroslav Kysela) [1917388]
- ASoC: rt1316: Fix return value check in rt1316_sdw_probe() (Jaroslav Kysela) [1917388]
- ASoC: sgtl5000: Fix identation of .driver elements (Jaroslav Kysela) [1917388]
- ASoC: codec: Omit superfluous error message in jz4760_codec_probe() (Jaroslav Kysela) [1917388]
- ASoC: sigmadsp-regmap: fix kernel-doc warning (Jaroslav Kysela) [1917388]
- ASoC: rt5631: fix kernel-doc warning (Jaroslav Kysela) [1917388]
- ASoC: jz4760: fix set but not used warning (Jaroslav Kysela) [1917388]
- ASoC: cs4270: fix kernel-doc (Jaroslav Kysela) [1917388]
- ASoC: rt5682: remove useless initialization (Jaroslav Kysela) [1917388]
- ASoC: rt5682: remove useless assignments (Jaroslav Kysela) [1917388]
- ASoC: rt5682: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5677: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: rt5677: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5670: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5668: remove useless assignments (Jaroslav Kysela) [1917388]
- ASoC: rt5668: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5665: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5663: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5660: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5659: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5651: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: rt5651: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5645: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt5645: use logical OR (Jaroslav Kysela) [1917388]
- ASoC: rt5640: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt1308: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt1305: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt1016: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt1015: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt1011: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: rt1011: remove redundant test (Jaroslav Kysela) [1917388]
- ASoC: rt1011: use logical OR (Jaroslav Kysela) [1917388]
- ASoC: wm8524: Do not print probe defer error (Jaroslav Kysela) [1917388]
- ASoC: codecs: nau8825: fix kernel-doc (Jaroslav Kysela) [1917388]
- ASoC: rt1316: Add RT1316 SDCA vendor-specific driver (Jaroslav Kysela) [1917388]
- ASoc: rt5631: Constify static struct coeff_clk_div (Jaroslav Kysela) [1917388]
- ASoC: rt*: Constify static struct acpi_device_id (Jaroslav Kysela) [1917388]
- ASoC: rt*: Constify static struct snd_soc_dai_ops (Jaroslav Kysela) [1917388]
- ASoC: rt*: Constify static struct sdw_slave_ops (Jaroslav Kysela) [1917388]
- ASoC: wcd934x: remove useless return (Jaroslav Kysela) [1917388]
- ASoC: wcd9335: clarify return value (Jaroslav Kysela) [1917388]
- ASoC: wcd-clsh-v2: align function prototypes (Jaroslav Kysela) [1917388]
- ASoC: codecs: wcd934x: add a sanity check in set channel map (Jaroslav Kysela) [1917388]
- ASoC: es8316: Simplify adc_pga_gain_tlv table (Jaroslav Kysela) [1917388]
- ASoC: sgtl5000: set DAP_AVC_CTRL register to correct default value on probe (Jaroslav Kysela) [1917388]
- ASoC: rt5651: Fix dac- and adc- vol-tlv values being off by a factor of 10 (Jaroslav Kysela) [1917388]
- ASoC: rt5640: Fix dac- and adc- vol-tlv values being off by a factor of 10 (Jaroslav Kysela) [1917388]
- ASoC: rt5670: Add emulated 'DAC1 Playback Switch' control (Jaroslav Kysela) [1917388]
- ASoC: rt5670: Remove ADC vol-ctrl mute bits poking from Sto1 ADC mixer settings (Jaroslav Kysela) [1917388]
- ASoC: rt5670: Remove 'HP Playback Switch' control (Jaroslav Kysela) [1917388]
- ASoC: rt5670: Remove 'OUT Channel Switch' control (Jaroslav Kysela) [1917388]
- ASoC: rt1015: fix i2c communication error (Jaroslav Kysela) [1917388]
- ASoC: codec: Add driver for JZ4760 internal codec (Jaroslav Kysela) [1917388]
- ASoC: codec/ingenic: Depend on MACH_INGENIC (Jaroslav Kysela) [1917388]
- ASoC: codec: hdmi-codec: Support IEC958 encoded PCM format (Jaroslav Kysela) [1917388]
- ASoC: rt5682: Fix panic in rt5682_jack_detect_handler happening during system shutdown (Jaroslav Kysela) [1917388]
- ASoC: rt5682: do nothing in rt5682_suspend/resume in sdw mode (Jaroslav Kysela) [1917388]
- ASoC: rt5682-sdw: cancel_work_sync() in .remove and .suspend (Jaroslav Kysela) [1917388]
- ASoC: rt711-sdw: use cancel_work_sync() for .remove (Jaroslav Kysela) [1917388]
- ASoC: rt700-sdw: use cancel_work_sync() in .remove as well as .suspend (Jaroslav Kysela) [1917388]
- ASoC: mt6359: reduce log verbosity for optional DT properties (Jaroslav Kysela) [1917388]
- ASoC: codecs: add missing max_register in regmap config (Jaroslav Kysela) [1917388]
- ASoC: cpcap: fix microphone timeslot mask (Jaroslav Kysela) [1917388]
- ASoC: rt5659: Add Kconfig prompt (Jaroslav Kysela) [1917388]
- ASoC: max98373: Fixes a typo in max98373_feedback_get (Jaroslav Kysela) [1917388]
- ASoC: ak4458: correct reset polarity (Jaroslav Kysela) [1917388]
- ASoC: rt5682: remove connection with LDO2 in DAPM graph (Jaroslav Kysela) [1917388]
- ASoC: ml26124: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: lm49453: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: inno_rk3036: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: cx2072x: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: alc5632: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: ab8500: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: lochnagar-sc: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: mt6660: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: mc13783: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: sgtl5000: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: ssm2602: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: zl38060: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: jz4740: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: ak*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: tscs*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: max*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: es*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: da*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: nau*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: rt*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: tlv*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: wm*: sync parameter naming (rate/sample_bits) (Jaroslav Kysela) [1917388]
- ASoC: remove sirf prima/atlas drivers (Jaroslav Kysela) [1917388]
- ASoC: codecs: soundwire: increase resume timeout (Jaroslav Kysela) [1917388]
- ASoC: es8316: Fix possible NULL pointer deref in es8316_disable_jack_detect() (Jaroslav Kysela) [1917388]
- ASoC: rt5682: enable fast discharge for headset unplugging (Jaroslav Kysela) [1917388]
- ASoC: rt5682: fix getting the wrong device id when the suspend_stress_test (Jaroslav Kysela) [1917388]
- ASoC: rt5645: Enable internal microphone and JD on ECS EF20 (Jaroslav Kysela) [1917388]
- ASoC: rt5645: add inv_hp_det flag (Jaroslav Kysela) [1917388]
- ASoC: rt5645: Add ACPI-defined GPIO for ECS EF20 series (Jaroslav Kysela) [1917388]
- ASoC: rt5645: Introduce mapping for ACPI-defined GPIO (Jaroslav Kysela) [1917388]
- ASoC: rt1015: re-calibrate again when resuming (Jaroslav Kysela) [1917388]
- ASoC: rt5645: Remove the redundant delay time (Jaroslav Kysela) [1917388]
- ASoC: rt1015: remove unneeded variables in rt1015_priv (Jaroslav Kysela) [1917388]
- ASoC: rt1015: refactor retry loop and rt1015_priv allocation (Jaroslav Kysela) [1917388]
- ASoC: rt1015: return error if any when setting bypass_boost (Jaroslav Kysela) [1917388]
- ASoC: rt1015: save boost_mode only if valid (Jaroslav Kysela) [1917388]
- ASoC: rt1015: sort header inclusions (Jaroslav Kysela) [1917388]
- ASoC: Intel: common: Fix some typos (Jaroslav Kysela) [1917388]
- ASoC: soc.h: remove for_each_rtd_dais_rollback() (Jaroslav Kysela) [1917388]
- ASoC: soc.h: return error if multi platform at snd_soc_fixup_dai_links_platform_name() (Jaroslav Kysela) [1917388]
- ASoC: soc.h: fixup return timing for snd_soc_fixup_dai_links_platform_name() (Jaroslav Kysela) [1917388]
- ASoC: soc.h: add asoc_link_to_cpu/codec/platform() macro (Jaroslav Kysela) [1917388]
- ASoC: add soc-jack.h (Jaroslav Kysela) [1917388]
- ASoC: soc-dai.h: Align the word of comment for SND_SOC_DAIFMT_CBC_CFC (Jaroslav Kysela) [1917388]
- ASoC: soc-core: fix always-false condition (Jaroslav Kysela) [1917388]
- ASoC: soc-core: fix signed/unsigned issue (Jaroslav Kysela) [1917388]
- ASoC: soc-acpi: remove useless initialization (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: ignore dummy-DAI at soc_pcm_params_symmetry() (Jaroslav Kysela) [1917388]
- ASoC: soc-utils: add snd_soc_component_is_dummy() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate DAI name if soc_pcm_params_symmetry() failed (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: don't use "name" on __soc_pcm_params_symmetry() macro (Jaroslav Kysela) [1917388]
- ASoC: core: Don't set platform name when of_node is set (Jaroslav Kysela) [1917388]
- ASoC: soc-compress: lock pcm_mutex to resolve lockdep error (Jaroslav Kysela) [1917388]
- ASoC: soc-core: use device_unregister() if rtd allocation failed (Jaroslav Kysela) [1917388]
- ASoC: soc-core: add comment for rtd freeing (Jaroslav Kysela) [1917388]
- ASoC: soc-component: Add snd_soc_pcm_component_ack (Jaroslav Kysela) [1917388]
- ASoC: don't indicate error message for snd_soc_[pcm_]component_xxx() (Jaroslav Kysela) [1917388]
- ASoC: don't indicate error message for snd_soc_[pcm_]dai_xxx() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: don't indicate error message for dpcm_be_dai_hw_free() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: don't indicate error message for soc_pcm_hw_free() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at dpcm_fe/be_dai_prepare() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at dpcm_fe/be_dai_hw_params() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at dpcm_fe/be_dai_startup() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at dpcm_run_update_startup/shutdown() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at dpcm_apply_symmetry() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at dpcm_be_dai_trigger() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at dpcm_path_get() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at soc_pcm_prepare() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at soc_pcm_hw_params() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: indicate error message at soc_pcm_open() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: share DPCM BE DAI stop operation (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: remove unneeded !rtd->dai_link check (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: fixup dpcm_be_dai_startup() user count (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_hw_sanity_check() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_pcm_update_symmetry() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: direct copy at snd_soc_set_runtime_hwparams() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_cpu/codec_dai_name() macro (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: check DAI activity under soc_pcm_apply_symmetry() (Jaroslav Kysela) [1917388]
- ASoC: soc-core: fix DMI handling (Jaroslav Kysela) [1917388]
- ASoC: soc-dai: fix kernel-doc (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add error log (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: remove shadowing variable (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: remove redundant assignment (Jaroslav Kysela) [1917388]
- ASoC: soc-ops: remove useless assignment (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: unpack dpcm_set_fe_runtime() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add dpcm_runtime_setup() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add dpcm_runtime_setup_fe() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: unpack dpcm_init_runtime_hw() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: remove strange format storing (Jaroslav Kysela) [1917388]
- ASoC: soc-core: Prevent warning if no DMI table is present (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: fix hw param limits calculation for multi-DAI (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: fix hwparams min/max init for dpcm (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_pcm_hw_update_format() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_pcm_hw_update_chan() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_pcm_hw_update_rate() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: change error message to debug message (Jaroslav Kysela) [1917388]
- ASoC: dapm: use component prefix when checking widget names (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: fixup snd_pcm_limit_hw_rates() timing (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: use snd_pcm_hardware at dpcm_runtime_merge_xxx() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_create_pcm() and simplify soc_new_pcm() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_get_playback_capture() and simplify soc_new_pcm() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: tidyup pcm setting (Jaroslav Kysela) [1917388]
- ASoC: soc-component: fix undefined reference to __ffssi2 (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_read/write_field() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: cleanup soc_pcm_params_symmetry() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: cleanup soc_pcm_apply_symmetry() (Jaroslav Kysela) [1917388]
- ASoC: soc-dai.h: remove symmetric_rates/samplebits (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_pcm_set_dai_params() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add dpcm_set_be_update_state() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: move dpcm_set_fe_update_state() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: Fix an uninitialized error code (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: return correct -ERRNO in failure path (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: disconnect BEs if the FE is not ready (Jaroslav Kysela) [1917388]
- ASoC: dapm: remove widget from dirty list on free (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: care trigger rollback (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: remove dpcm_do_trigger() (Jaroslav Kysela) [1917388]
- ASoC: pcm: send DAPM_STREAM_STOP event in dpcm_fe_dai_shutdown (Jaroslav Kysela) [1917388]
- ASoC: soc-core: tidyup jack.h (Jaroslav Kysela) [1917388]
- ASoC: soc-core: add soc_dapm_suspend_resume() (Jaroslav Kysela) [1917388]
- ASoC: soc-core: add soc_playback_digital_mute() (Jaroslav Kysela) [1917388]
- ASoC: soc-compress: add soc_compr_clean() and call it from soc_compr_open/free() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add mark for snd_soc_link_compr_startup/shutdown() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add mark for snd_soc_component_compr_open/free() (Jaroslav Kysela) [1917388]
- ASoC: soc-dai: add mark for snd_soc_dai_compr_startup/shutdown() (Jaroslav Kysela) [1917388]
- ASoC: soc-compress: move soc_compr_free() next to soc_compr_open() (Jaroslav Kysela) [1917388]
- ASoC: pcm: call snd_soc_dapm_stream_stop() in soc_pcm_hw_clean (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_get_metadata() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_set_metadata() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_copy() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_pointer() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_ack() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_get_codec_caps() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_get_caps() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_get_params() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_set_params() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_trigger() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_free() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add snd_soc_component_compr_open() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: Get all BEs along DAPM path (Jaroslav Kysela) [1917388]
- ASoC: soc-core: Fix component name_prefix parsing (Jaroslav Kysela) [1917388]
- ASoC: soc-compress: assume SNDRV_PCM_STREAM_xxx and SND_COMPRESS_xxx are same (Jaroslav Kysela) [1917388]
- ASoC: soc-compress: tidyup STREAM vs COMPRESS (Jaroslav Kysela) [1917388]
- ASoC: soc-topology: clarify expression (Jaroslav Kysela) [1917388]
- ASoC: sync parameter naming : rate / sample_bits (Jaroslav Kysela) [1917388]
- ASoC: topology: Check if ops is set before dereference (Jaroslav Kysela) [1917388]
- ASoC: topology: Ensure that needed parameters are set (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: add soc_pcm_hw_clean() and call it from soc_pcm_hw_params/free() (Jaroslav Kysela) [1917388]
- ASoC: soc-dai: add mark for snd_soc_dai_hw_params/free() (Jaroslav Kysela) [1917388]
- ASoC: soc-component: add mark for snd_soc_pcm_component_hw_params/free() (Jaroslav Kysela) [1917388]
- ASoC: soc-link: add mark for snd_soc_link_hw_params/free() (Jaroslav Kysela) [1917388]
- ASoC: soc-pcm: move soc_pcm_hw_free() next to soc_pcm_hw_params() (Jaroslav Kysela) [1917388]
- ASoC: dapm: use semicolons rather than commas to separate statements (Jaroslav Kysela) [1917388]
- ASoC: dmaengine: Document support for TX only or RX only streams (Jaroslav Kysela) [1917388]
- ASoC: soc-core: use devm_snd_soc_register_card() (Jaroslav Kysela) [1917388]
- ALSA: usb: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix control 'access overflow' errors from chmap (Jaroslav Kysela) [1917388]
- ALSA: line6: Fix racy initialization of LINE6 MIDI (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Disable sample read check if firmware doesn't give back (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: scarlett2: snd_scarlett_gen2_controls_create() can be static (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: scarlett2: Improve driver startup messages (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: scarlett2: Fix device hang with ehci-pci (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: fix control-request direction (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix potential out-of-bounce access in MIDI EP parser (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Validate MS endpoint descriptors (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add dB range mapping for Sennheiser Communications Headset PC 8 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Remove redundant assignment to len (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix implicit sync clearance at stopping stream (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Generic application of implicit fb to Roland/BOSS devices (Jaroslav Kysela) [1917388]
- Revert "ALSA: usb-audio: Add support for many Roland devices..." (Jaroslav Kysela) [1917388]
- ALSA: usb: midi: don't return -ENOMEM when usb_urb_ep_type_check fails (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Re-apply implicit feedback mode to Pioneer devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: DJM-750: ensure format is set (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add support for many Roland devices' implicit feedback quirks (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Apply implicit feedback mode for BOSS devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Explicitly set up the clock selector (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add MIDI quirk for Vox ToneLab EX (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Skip probe of UA-101 devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Drop implicit fb quirk entries dubbed for capture (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add implicit feeback support for the BOSS GT-1 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add error checks for usb_driver_claim_interface() calls (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Apply sample rate quirk to Logitech Connect (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Check connector value on resume (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Carve out connector value checking into a helper (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix unintentional sign extension issue (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Apply the control quirk to Plantronics headsets (Jaroslav Kysela) [1917388]
- ALSA: usb: Add Plantronics C320-M USB ctrl msg delay quirk (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix "cannot get freq eq" errors on Dell AE515 sound bar (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: fix Pioneer DJM-850 control label info (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Configure Pioneer DJM-850 samplerate (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Declare Pioneer DJM-850 mixer controls (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add Pioneer DJM-850 to quirks-table (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix Pioneer DJM devices URB_CONTROL request direction to set samplerate (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: use Corsair Virtuoso mapping for Corsair Virtuoso SE (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: generate midi streaming substream names from jack names (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: use usb headers rather than define structs locally (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Allow modifying parameters with succeeding hw_params calls (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Drop bogus dB range in too low level (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Don't abort even if the clock rate differs (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add implicit fb quirk for BOSS GP-10 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add quirk for RC-505 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Don't avoid stopping the stream at disconnection (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: More strict state change in EP (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Handle invalid running state at releasing EP (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: add mixer quirks for Pioneer DJM-900NXS2 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add DJM750 to Pioneer mixer quirk (Jaroslav Kysela) [1917388]
- ALSA: Convert strlcpy to strscpy when return value is unused (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix PCM buffer allocation in non-vmalloc mode (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Correct document for snd_usb_endpoint_free_all() (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add DJM-450 to the quirks table (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add DJM450 to Pioneer format quirk (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: workaround for iface reset issue (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix "RANGE setting not yet supported" errors (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Skip the clock selector inquiry for single connections (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix hw constraints dependencies (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add support for Pioneer DJM-750 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Avoid implicit feedback on Pioneer devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Set sample rate for all sharing EPs on UAC1 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix UAC1 rate setup for secondary endpoints (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Convert the last strlcpy() usage (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Convert remaining strlcpy() to strscpy() (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Always apply the hw constraints for implicit fb sync (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix implicit feedback sync setup for Pioneer devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Annotate the endpoint index in audioformat (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Avoid unnecessary interface re-setup (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Choose audioformat of a counter-part substream (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix the missing endpoints creations for quirks (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add quirk for BOSS AD-10 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix UBSAN warnings for MIDI jacks (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add alias entry for ASUS PRIME TRX40 PRO-S (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add VID to support native DSD reproduction on FiiO devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix potential out-of-bounds shift (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add implicit fb support for Steinberg UR22 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add support for Pioneer DJ DDJ-RR controller (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: US16x08: fix value count for level meters (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix MOTU M-Series quirks (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix quirks for other BOSS devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add implicit_fb module option (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add generic implicit fb parsing (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Factor out the implicit feedback quirk code (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Quirk for BOSS GT-001 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Show sync endpoint information in proc outputs (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Use unsigned char for iface and altsettings fields (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Replace slave/master terms (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Simplify rate_min/max and rates set up (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Unify the code for the next packet size calculation (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Drop unneeded snd_usb_substream fields (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Refactoring endpoint URB deactivation (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Use atomic_t for endpoint use_count (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Constify audioformat pointer references (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix possible stall of implicit fb packet ring-buffer (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Refactor endpoint management (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Fix EP matching for continuous rates (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Always set up the parameters after resume (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add quirk for Pioneer DJ DDJ-SR2 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Set callbacks via snd_usb_endpoint_set_callback() (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Stop both endpoints properly at error (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Simplify snd_usb_init_pitch() arguments (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Simplify snd_usb_init_sample_rate() arguments (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Don't set altsetting before initializing sample rate (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Pass snd_usb_audio object to quirk functions (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add snd_usb_get_host_interface() helper (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Drop keep_interface flag again (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Create endpoint objects at parsing phase (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Avoid doubly initialization for implicit fb (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Drop debug.h (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Simplify hw_params rules (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add hw constraint for implicit fb sync (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Move snd_usb_autoresume() call out of setup_hw_info() (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Track implicit fb sync endpoint in audioformat list (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Improve some debug prints (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Set and clear sync EP link properly (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add snd_usb_get_endpoint() helper (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Check implicit feedback EP generically for UAC2 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Check valid altsetting at parsing rates for UAC2/3 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Don't call usb_set_interface() at trigger callback (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Handle discrete rates properly in hw constraints (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add delay quirk for all Logitech USB devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Use ALC1220-VB-DT mapping for ASUS ROG Strix TRX40 mobo (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add implicit feedback quirk for Qu-16 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add implicit feedback quirk for MODX (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: add usb vendor id as DSD-capable for Khadas devices (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add implicit feedback quirk for Zoom UAC-2 (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Line6 Pod Go interface requires static clock rate quirk (Jaroslav Kysela) [1917388]
- ALSA: usb-audio: Add mixer support for Pioneer DJ DJM-250MK2 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: reset eapd coeff to default value for alc287 (Jaroslav Kysela) [1917388]
- ALSA: hda/hdmi: Cancel pending works before suspend (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132: Fix compile warning without PCI (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132: Move unsol callback setups to parser (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132: make some const arrays static, makes object smaller (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Add ZxR surround DAC setup. (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Add 8051 PLL write helper functions. (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Remove now unnecessary DSP setup functions. (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Ensure DSP is properly setup post-firmware download. (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Add 8051 exram helper functions. (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Add stream port remapping function. (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Reset codec upon initialization. (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Change Input Source enum strings. (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132 - Fix AE-5 rear headphone pincfg. (Jaroslav Kysela) [1917388]
- ALSA: hda: Reinstate runtime_allow() for all hda controllers (Jaroslav Kysela) [1917388]
- ACPI: Test for ACPI_SUCCESS rather than !ACPI_FAILURE (Jaroslav Kysela) [1917388]
- ALSA: hda: Flush pending unsolicited events before suspend (Jaroslav Kysela) [1917388]
- ALSA: hda: Re-add dropped snd_poewr_change_state() calls (Jaroslav Kysela) [1917388]
- ALSA: hda: Add missing sanity checks in PM prepare/complete callbacks (Jaroslav Kysela) [1917388]
- ALSA: hda: Separate runtime and system suspend (Jaroslav Kysela) [1917388]
- ALSA: hda: update the power_state during the direct-complete (Jaroslav Kysela) [1917388]
- ALSA: hda: Balance runtime/system PM if direct-complete is disabled (Jaroslav Kysela) [1917388]
- ALSA: hda: Refactor codec PM to use direct-complete optimization (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Set Initial DMIC volume to -26 dB (Jaroslav Kysela) [1917388]
- ALSA: hda: Fix a regression in Capture Switch mixer read (Jaroslav Kysela) [1917388]
- ALSA: hda: Add AlderLake-M PCI ID (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs and speaker for HP Zbook Fury 17 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs and speaker for HP Zbook Fury 15 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs and speaker for HP Zbook G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs for HP 855 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Chain in pop reduction fixup for ThinkStation P340 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: the bass speaker can't output sound on Yoga 9i (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Headphone volume is controlled by Front mixer (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Add some CLOVE SSIDs of ALC293 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Add fixup for HP Spectre x360 15-df0xxx (Jaroslav Kysela) [1917388]
- ALSA: hda: fixup headset for ASUS GU502 laptop (Jaroslav Kysela) [1917388]
- ALSA: hda: Fix for mute key LED for HP Pavilion 15-CK0xx (Jaroslav Kysela) [1917388]
- ALSA: hda: generic: change the DAC ctl name for LO+SPK or LO+HP (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Add fixup for HP OMEN laptop (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Fix speaker amp on HP Envy AiO 32 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Fix silent headphone output on ASUS UX430UA (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: ALC285 Thinkpad jack pin quirk is unreachable (Jaroslav Kysela) [1917388]
- ALSA: hda/conexant: Re-order CX5066 quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Remove redundant entry for ALC861 Haier/Uniwill devices (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC662 quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order remaining ALC269 quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC269 Lenovo quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC269 Sony quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC269 ASUS quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC269 Dell quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC269 Acer quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC269 HP quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC882 Clevo quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC882 Sony quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Re-order ALC882 Acer quirk table entries (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Add quirk for Intel Clevo PCx0Dx (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Use CS8409 filter to fix abnormal sounds on Bullseye (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Set Initial DMIC volume for Bullseye to -26 dB (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix static noise on ALC285 Lenovo laptops (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Enable mute/micmute LEDs and limit mic boost on EliteBook 845 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek - Headset Mic issue on HP platform (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: GA503 use same quirks as GA401 (Jaroslav Kysela) [1917388]
- ALSA: hda/hdmi: fix race in handling acomp ELD notification at resume (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs for HP ProBook 445 G7 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Add quirk for Lenovo Ideapad S740 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mic boost on Intel NUC 8 (Jaroslav Kysela) [1917388]
- ALSA: HDA: Add access description in __snd_hda_add_vmaster (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Fix speaker amp setup on Acer Aspire E1 (Jaroslav Kysela) [1917388]
- ALSA: hda/conexant: Apply quirk for another HP ZBook G5 model (Jaroslav Kysela) [1917388]
- ALSA: HDA - remove the custom implementation for the audio LED trigger (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs for HP 640 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/hdmi: fix max DP-MST dev_num for Intel TGL+ platforms (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: call alc_update_headset_mode() in hp_automute_hook (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix a determine_headset_type issue for a Dell AIO (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs for HP 850 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs for HP 440 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Make CS8409 driver more generic by using fixups. (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Fix CS42L42 Headset Mic volume control name (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Cleanup patch_cirrus.c code. (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Add error handling into CS8409 I2C functions (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: fix mute/micmute LEDs for HP 840 G8 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: apply pin quirk for XiaomiNotebook Pro (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Apply headset-mic quirks for Xiaomi Redmibook Air (Jaroslav Kysela) [1917388]
- ALSA: hda: generic: Fix the micmute led init state (Jaroslav Kysela) [1917388]
- ALSA: hda/ca0132: Add Sound BlasterX AE-5 Plus support (Jaroslav Kysela) [1917388]
- ALSA: hda: Drop the BATCH workaround for AMD controllers (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Add Headphone and Headset MIC Volume Control (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Add jack detect interrupt support from CS42L42 companion codec. (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Add support for CS8409 HDA bridge and CS42L42 companion codec. (Jaroslav Kysela) [1917388]
- ALSA: hda/cirrus: Increase AUTO_CFG_MAX_INS from 8 to 18 (Jaroslav Kysela) [1917388]
- ALSA: hda/conexant: Add quirk for mute LED control on HP ZBook G5 (Jaroslav Kysela) [1917388]
- ALSA: hda - bind headset buttons to the headphone jack (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Apply dual codec quirks for MSI Godlike X570 board (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Add quirk for Intel NUC 10 (Jaroslav Kysela) [1917388]
- ALSA: hda/hdmi: let new platforms assign the pcm slot dynamically (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Add quirk for Clevo NH55RZQ (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Enable headset mic of Acer SWIFT with ALC256 (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Quirk for HP Spectre x360 14 amp setup (Jaroslav Kysela) [1917388]
- ALSA: hda: Add another CometLake-H PCI ID (Jaroslav Kysela) [1917388]
- ALSA: hda/hdmi: Drop bogus check at closing a stream (Jaroslav Kysela) [1917388]
- ALSA: hda: Drop power save deny list entry for Clevo W65_67SB (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: modify EAPD in the ALC886 (Jaroslav Kysela) [1917388]
- ALSA: hda/via: Apply the workaround generically for Clevo machines (Jaroslav Kysela) [1917388]
- ALSA: hda/tegra: Remove unnecessary null-check from hda_tegra_runtime_resume() (Jaroslav Kysela) [1917388]
- ALSA: hda/tegra: Reset hardware (Jaroslav Kysela) [1917388]
- ALSA: hda/tegra: Use clk_bulk helpers (Jaroslav Kysela) [1917388]
- ALSA: hda: Assign boolean values to a bool variable (Jaroslav Kysela) [1917388]
- ALSA: hda: boolean values to a bool variable (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek: Enable headset of ASUS B1400CEPE with ALC256 (Jaroslav Kysela) [1917388]
- ALSA: pci: Simplify with dma_set_mask_and_coherent() (Jaroslav Kysela) [1917388]
- ALSA: hda/via: Add minimum mute flag (Jaroslav Kysela) [1917388]
- ALSA: hda/realtek - Limit int mic boost on Acer Aspire E5-575T (Jaroslav Kysela) [1917388]
- ALSA: hda: Add AlderLake-P PCI ID and HDMI codec vid (Jaroslav Kysela) [1917388]
- ALSA: hda/hdmi - enable runtime pm for CI AMD display audio (Jaroslav Kysela) [1917388]
- ALSA: hda/tegra: fix tegra-hda on tegra30 soc (Jaroslav Kysela) [1917388]
- ALSA: hda: Revert "ALSA: hda: Allow setting preallocation again for x86" (Jaroslav Kysela) [1917388]
- ALSA: hda: Fix spelling mistakes (Jaroslav Kysela) [1917388]
- ALSA: hda: ignore invalid NHLT table (Jaroslav Kysela) [1917388]
- ALSA: hda: intel-nhlt: verify config type (Jaroslav Kysela) [1917388]
- ALSA: hda: fix kernel-doc warnings (Jaroslav Kysela) [1917388]
- ALSA: hda: intel-dsp-config: add Alder Lake support (Jaroslav Kysela) [1917388]
- ALSA: hda: intel-dsp-config: Add SND_INTEL_BYT_PREFER_SOF Kconfig option (Jaroslav Kysela) [1917388]
- ALSA: hda: add link_power op to hdac_bus_ops (Jaroslav Kysela) [1917388]
- ALSA: hda: Constify static attribute_group (Jaroslav Kysela) [1917388]
- ALSA: hda: Use DIV_ROUND_UP()/roundup() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: hda: intel-dsp-config: ignore dsp_driver parameter for PCI legacy devices (Jaroslav Kysela) [1917388]
- ALSA: hda: intel-dsp-config: add Broadwell ACPI DSP driver selection (Jaroslav Kysela) [1917388]
- ALSA: hda: intel-dsp-config: add helper for ACPI DSP driver selection (Jaroslav Kysela) [1917388]
- ALSA: pcm: Fix missing check of the new non-cached buffer type (Jaroslav Kysela) [1917388]
- ALSA: pcm: use dma_can_mmap() to check if a device supports dma_mmap_* (Jaroslav Kysela) [1917388]
- ALSA: timer: Fix master timer notification (Jaroslav Kysela) [1917388]
- ALSA: control led: fix memory leak in snd_ctl_led_register (Jaroslav Kysela) [1917388]
- ALSA: control: Fix racy management of user ctl memory size account (Jaroslav Kysela) [1917388]
- ALSA: control_led - fix the stack usage (control element ops) (Jaroslav Kysela) [1917388]
- ALSA: control - double free in snd_ctl_led_init() (Jaroslav Kysela) [1917388]
- ALSA: control: Add memory consumption limit to user controls (Jaroslav Kysela) [1917388]
- ALSA: control - off by one in store_mode() (Jaroslav Kysela) [1917388]
- ALSA: control led - improve the set_led_id() parser (Jaroslav Kysela) [1917388]
- ALSA: control - add the missing prev_lops2 initialization (Jaroslav Kysela) [1917388]
- ALSA: led control - add sysfs kcontrol LED marking layer (Jaroslav Kysela) [1917388]
- ALSA: control - add sysfs support to the LED trigger module (Jaroslav Kysela) [1917388]
- ALSA: control - add generic LED trigger module as the new control layer (Jaroslav Kysela) [1917388]
- ALSA: control - add layer registration routines (Jaroslav Kysela) [1917388]
- ALSA: control - introduce snd_ctl_notify_one() helper (Jaroslav Kysela) [1917388]
- ALSA: core: remove redundant spin_lock pair in snd_card_disconnect (Jaroslav Kysela) [1917388]
- ALSA: pcm: Fix couple of typos (Jaroslav Kysela) [1917388]
- ALSA: core: avoid -Wempty-body warnings (Jaroslav Kysela) [1917388]
- ALSA: pcm: Add debug print on memory allocation failure (Jaroslav Kysela) [1917388]
- ALSA: core - add missing compress device type to /proc/asound/devices (Jaroslav Kysela) [1917388]
- ALSA: pcm: Use for_each_pcm_substream() macro (Jaroslav Kysela) [1917388]
- ALSA: pcm: Don't call sync_stop if it hasn't been stopped (Jaroslav Kysela) [1917388]
- ALSA: pcm: Assure sync with the pending stop operation at suspend (Jaroslav Kysela) [1917388]
- ALSA: pcm: Call sync_stop at disconnection (Jaroslav Kysela) [1917388]
- ASoC: dmaengine_pcm: add peripheral configuration (Jaroslav Kysela) [1917388]
- ALSA: core: Fix the debugfs removal at snd_card_free() (Jaroslav Kysela) [1917388]
- ALSA: jack: implement software jack injection via debugfs (Jaroslav Kysela) [1917388]
- ALSA: pcm: One more dependency for hw constraints (Jaroslav Kysela) [1917388]
- ALSA: seq: oss: Fix missing error check in snd_seq_oss_synth_make_info() (Jaroslav Kysela) [1917388]
- ALSA: oss: Use DIV_ROUND_CLOSEST() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: core: Use DIV_ROUND_UP() instead of open-coding it (Jaroslav Kysela) [1917388]
- ALSA: core: Remove redundant comments (Jaroslav Kysela) [1917388]
- ALSA: pcm: oss: Fix a few more UBSAN fixes (Jaroslav Kysela) [1917388]
- ALSA: pcm: Clear the full allocated memory at hw_params (Jaroslav Kysela) [1917388]
- ALSA: memalloc: Align buffer allocations in page size (Jaroslav Kysela) [1917388]
- ALSA: pcm: Remove snd_pcm_lib_preallocate_dma_free() (Jaroslav Kysela) [1917388]
- ALSA: core: memalloc: add page alignment for iram (Jaroslav Kysela) [1917388]
- ALSA: pcm: oss: Fix potential out-of-bounds shift (Jaroslav Kysela) [1917388]
- ALSA: rawmidi: Access runtime->avail always in spinlock (Jaroslav Kysela) [1917388]
- ALSA: seq: Use bool for snd_seq_queue internal flags (Jaroslav Kysela) [1917388]
- ALSA: compress: allow pause and resume during draining (Jaroslav Kysela) [1917388]
- ALSA: ctl: fix error path at adding user-defined element set (Jaroslav Kysela) [1917388]
- ALSA: seq: remove useless function (Jaroslav Kysela) [1917388]
- ALSA: fix kernel-doc markups (Jaroslav Kysela) [1917388]
- soundwire: SDCA: add helper macro to access controls (Jaroslav Kysela) [1917388]
- soundwire: add static port mapping support (Jaroslav Kysela) [1917388]
- soundwire: add definition for DPn BlockPackingMode (Jaroslav Kysela) [1917388]
- soundwire: add master quirks for bus clash and parity (Jaroslav Kysela) [1917388]
- soundwire: intel: don't return error when clock stop failed (Jaroslav Kysela) [1917388]
- soundwire: intel: Use kzalloc for allocating only one thing (Jaroslav Kysela) [1917388]
- soundwire: cadence: add status in dev_dbg 'State change' log (Jaroslav Kysela) [1917388]
- soundwire: cadence: adjust verbosity in response handling (Jaroslav Kysela) [1917388]
- soundwire: cadence: fix ACK/NAK handling (Jaroslav Kysela) [1917388]
- soundwire: cadence: reduce timeout on transactions (Jaroslav Kysela) [1917388]
- soundwire: bus: use consistent tests for return values (Jaroslav Kysela) [1917388]
- soundwire: qcom: fix handling of qcom,ports-block-pack-mode (Jaroslav Kysela) [1917388]
- soundwire: intel_init: test link->cdns (Jaroslav Kysela) [1917388]
- soundwire: qcom: handle return correctly in qcom_swrm_transport_params (Jaroslav Kysela) [1917388]
- soundwire: qcom: cleanup internal port config indexing (Jaroslav Kysela) [1917388]
- soundwire: qcom: wait for fifo space to be available before read/write (Jaroslav Kysela) [1917388]
- soundwire: qcom: add static port map support (Jaroslav Kysela) [1917388]
- soundwire: qcom: update port map allocation bit mask (Jaroslav Kysela) [1917388]
- soundwire: stream: fix memory leak in stream config error path (Jaroslav Kysela) [1917388]
- soundwire: qcom: use signed variable for error return (Jaroslav Kysela) [1917388]
- soundwire: qcom: wait for enumeration to be complete in probe (Jaroslav Kysela) [1917388]
- soundwire: qcom: add auto enumeration support (Jaroslav Kysela) [1917388]
- soundwire: export sdw_compare_devid, sdw_extract_slave_id and sdw_slave_add (Jaroslav Kysela) [1917388]
- soundwire: qcom: add support to new interrupts (Jaroslav Kysela) [1917388]
- soundwire: qcom: update register read/write routine (Jaroslav Kysela) [1917388]
- soundwire: qcom: start the clock during initialization (Jaroslav Kysela) [1917388]
- soundwire: qcom: set continue execution flag for ignored commands (Jaroslav Kysela) [1917388]
- soundwire: qcom: add support to missing transport params (Jaroslav Kysela) [1917388]
- soundwire: cadence: only prepare attached devices on clock stop (Jaroslav Kysela) [1917388]
- soundwire: generic_allocation: fix confusion between group and packing (Jaroslav Kysela) [1917388]
- soundwire: bus: Fix device found flag correctly (Jaroslav Kysela) [1917388]
- soundwire: cadence_master: fix kernel-doc (Jaroslav Kysela) [1917388]
- soundwire: stream: remove useless bus initializations (Jaroslav Kysela) [1917388]
- soundwire: stream: remove useless initialization (Jaroslav Kysela) [1917388]
- soundwire: qcom: check of_property_read status (Jaroslav Kysela) [1917388]
- soundwire: intel: remove useless readl (Jaroslav Kysela) [1917388]
- soundwire: generic_bandwidth_allocation: remove useless init (Jaroslav Kysela) [1917388]
- soundwire: bus: remove useless initialization (Jaroslav Kysela) [1917388]
- soundwire: bus: uniquify dev_err() for SCP_INT access (Jaroslav Kysela) [1917388]
- soundwire: bus: demote clock stop prepare log to dev_dbg() (Jaroslav Kysela) [1917388]
- soundwire: bus: clarify dev_err/dbg device references (Jaroslav Kysela) [1917388]
- soundwire: bus: fix confusion on device used by pm_runtime (Jaroslav Kysela) [1917388]
- soundwire: export sdw_write/read_no_pm functions (Jaroslav Kysela) [1917388]
- soundwire: bus: use no_pm IO routines for all interrupt handling (Jaroslav Kysela) [1917388]
- soundwire: bus: use sdw_write_no_pm when setting the bus scale registers (Jaroslav Kysela) [1917388]
- soundwire: bus: use sdw_update_no_pm when initializing a device (Jaroslav Kysela) [1917388]
- soundwire: return earlier if no slave is attached (Jaroslav Kysela) [1917388]
- soundwire: bus: add better dev_dbg to track complete() calls (Jaroslav Kysela) [1917388]
- soundwire: bus: add more details to track failed transfers (Jaroslav Kysela) [1917388]
- soundwire: use consistent format for Slave devID logs (Jaroslav Kysela) [1917388]
- soundwire: bus: test read status (Jaroslav Kysela) [1917388]
- soundwire: bus: use correct driver name in error messages (Jaroslav Kysela) [1917388]
- soundwire: qcom: add missing \n in dev_err() (Jaroslav Kysela) [1917388]
- soundwire: stream: add missing \n in dev_err() (Jaroslav Kysela) [1917388]
- soundwire: cadence: add missing \n in dev_err() (Jaroslav Kysela) [1917388]
- soundwire: bandwidth_allocation: add missing \n in dev_err() (Jaroslav Kysela) [1917388]
- soundwire: intel: add missing \n in dev_err() (Jaroslav Kysela) [1917388]
- soundwire: intel: add master quirks for bus clash and parity (Jaroslav Kysela) [1917388]
- soundwire: bus: handle master quirks for bus clash and parity (Jaroslav Kysela) [1917388]
- soundwire: Intel: add DMI quirk for Dell SKU 0A3E (Jaroslav Kysela) [1917388]
- soundwire: Intel: introduce DMI quirks for HP Spectre x360 Convertible (Jaroslav Kysela) [1917388]
- soundwire: add override addr ops (Jaroslav Kysela) [1917388]
- regmap/SoundWire: sdw: add support for SoundWire 1.2 MBQ (Jaroslav Kysela) [1917388]
- tracing: Fix __print_hex_dump scope (Jaroslav Kysela) [1917388]
- tracing: Use seq_buf_hex_dump() to dump buffers (Jaroslav Kysela) [1917388]
- seq_buf: Add printing formatted hex dumps (Jaroslav Kysela) [1917388]
- powerpc/64: Don't trace code that runs with the soft irq mask unreconciled (Desnes A. Nunes do Rosario) [1921631]
- powerpc/64: Disable irq restore warning for now (Desnes A. Nunes do Rosario) [1921631]
- powerpc/64s: make PACA_IRQ_HARD_DIS track MSR[EE] closely (Desnes A. Nunes do Rosario) [1921631]

* Mon Jul 05 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-321.el8]
- watchdog/hpwdt: New PCI IDs (Joseph Szczypek) [1967765]
- watchdog: hpwdt: Assign boolean values to a bool variable (Joseph Szczypek) [1967765]
- scsi: libfc: Avoid invoking response handler twice if ep is already completed (Chris Leech) [1867301]
- Bluetooth: SMP: Fail if remote and local public keys are identical (Gopal Tiwari) [1965083]
- rq-qos: fix missed wake-ups in rq_qos_throttle try two (Ming Lei) [1972111]
- mm: memcg/slab: disable cache merging for KMALLOC_NORMAL caches (Waiman Long) [1955561]
- mm: memcg/slab: create a new set of kmalloc-cg-<n> caches (Waiman Long) [1955561]
- mm: memcg/slab: properly set up gfp flags for objcg pointer array (Waiman Long) [1955561]
- mm/vmalloc.c:__vmalloc_area_node(): avoid 32-bit overflow (Rafael Aquini) [1970194]
- mm: remove the filename in the top of file comment in vmalloc.c (Rafael Aquini) [1970194]
- mm: cleanup the gfp_mask handling in __vmalloc_area_node (Rafael Aquini) [1970194]
- mm/vmalloc.c: remove unnecessary highmem_mask from parameter of gfpflags_allow_blocking() (Rafael Aquini) [1970194]
- mm/swap: fix pte_same_as_swp() not removing uffd-wp bit when compare (Chris von Recklinghausen) [1945442]
- userfaultfd: hugetlbfs: fix new flag usage in error path (Chris von Recklinghausen) [1945442]
- mm/hugetlb: fix cow where page writtable in child (Chris von Recklinghausen) [1945442]
- ovl: fix reference counting in ovl_mmap error path (Chris von Recklinghausen) [1945442]
- hugetlb: do early cow when page pinned on src mm (Chris von Recklinghausen) [1945442]
- mm: introduce page_needs_cow_for_dma() for deciding whether cow (Chris von Recklinghausen) [1945442]
- hugetlb: convert page_huge_active() HPageMigratable flag (Chris von Recklinghausen) [1945442]
- hugetlb: use page.private for hugetlb specific page flags (Chris von Recklinghausen) [1945442]
- x86/mm: Remove duplicate definition of _PAGE_PAT_LARGE (Chris von Recklinghausen) [1945442]
- mm: hugetlbfs: fix cannot migrate the fallocated HugeTLB page (Chris von Recklinghausen) [1945442]
- mm: hugetlb: fix a race between isolating and freeing page (Chris von Recklinghausen) [1945442]
- mm: hugetlb: remove VM_BUG_ON_PAGE from page_huge_active (Chris von Recklinghausen) [1945442]
- mm: introduce vma_set_file function v5 (Chris von Recklinghausen) [1945442]
- mm: mmap: fix fput in error path v2 (Chris von Recklinghausen) [1945442]
- mm/gup: prevent gup_fast from racing with COW during fork (Chris von Recklinghausen) [1945442]
- mm/gup: reorganize internal_get_user_pages_fast() (Chris von Recklinghausen) [1945442]
- mm: remove src/dst mm parameter in copy_page_range() (Chris von Recklinghausen) [1945442]
- mm: avoid early COW write protect games during fork() (Chris von Recklinghausen) [1945442]
- mm/thp: Split huge pmds/puds if they're pinned when fork() (Chris von Recklinghausen) [1945442]
- mm: Do early cow for pinned pages during fork() for ptes (Chris von Recklinghausen) [1945442]
- mm/fork: Pass new vma pointer into copy_page_range() (Chris von Recklinghausen) [1945442]
- mm: move the copy_one_pte() pte_present check into the caller (Chris von Recklinghausen) [1945442]
- mm: split out the non-present case from copy_one_pte() (Chris von Recklinghausen) [1945442]
- mm/gup: Remove enfornced COW mechanism (Chris von Recklinghausen) [1945442]
- mm/hmm.c: delete duplicated word (Chris von Recklinghausen) [1945442]
- mm/hmm: provide the page mapping order in hmm_range_fault() (Chris von Recklinghausen) [1945442]
- mmap locking API: add mmap_assert_locked() and mmap_assert_write_locked() (Chris von Recklinghausen) [1945442]
- mm/gup: introduce pin_user_pages_locked() (Chris von Recklinghausen) [1945442]
- mm/gup: introduce pin_user_pages_unlocked (Chris von Recklinghausen) [1945442]
- mm: remove the prot argument from vm_map_ram (Chris von Recklinghausen) [1945442]
- mm: remove unmap_vmap_area (Chris von Recklinghausen) [1945442]
- mm: don't return the number of pages from map_kernel_range{,_noflush} (Chris von Recklinghausen) [1945442]
- x86: fix vmap arguments in map_irq_stack (Chris von Recklinghausen) [1945442]
- mm/memory: remove unnecessary pte_devmap case in copy_one_pte() (Chris von Recklinghausen) [1945442]
- mm/hmm: remove the customizable pfn format from hmm_range_fault (Chris von Recklinghausen) [1945442]
- mm/hmm: remove HMM_PFN_SPECIAL (Chris von Recklinghausen) [1945442]
- mm/hmm: make hmm_range_fault return 0 or -1 (Chris von Recklinghausen) [1945442]
- mm/hugetlb: fix build failure with HUGETLB_PAGE but not HUGEBTLBFS (Chris von Recklinghausen) [1945442]
- mm/hmm: return error for non-vma snapshots (Chris von Recklinghausen) [1945442]
- mm/hmm: do not set pfns when returning an error code (Chris von Recklinghausen) [1945442]
- mm/hmm: do not unconditionally set pfns when returning EBUSY (Chris von Recklinghausen) [1945442]
- mm/hmm: use device_private_entry_to_pfn() (Chris von Recklinghausen) [1945442]
- mm/hmm: remove HMM_FAULT_SNAPSHOT (Chris von Recklinghausen) [1945442]
- mm/hmm: remove unused code and tidy comments (Chris von Recklinghausen) [1945442]
- mm/hmm: return the fault type from hmm_pte_need_fault() (Chris von Recklinghausen) [1945442]
- mm/hmm: remove pgmap checking for devmap pages (Chris von Recklinghausen) [1945442]
- mm/hmm: check the device private page owner in hmm_range_fault() (Chris von Recklinghausen) [1945442]
- mm: simplify device private page handling in hmm_range_fault (Chris von Recklinghausen) [1945442]
- mm: merge hmm_vma_do_fault into into hmm_vma_walk_hole_ (Chris von Recklinghausen) [1945442]
- mm/hmm: don't handle the non-fault case in hmm_vma_walk_hole_() (Chris von Recklinghausen) [1945442]
- mm/hmm: simplify hmm_vma_walk_hugetlb_entry() (Chris von Recklinghausen) [1945442]
- mm/hmm: remove the unused HMM_FAULT_ALLOW_RETRY flag (Chris von Recklinghausen) [1945442]
- mm/hmm: don't provide a stub for hmm_range_fault() (Chris von Recklinghausen) [1945442]
- mm/hmm: do not check pmd_protnone twice in hmm_vma_handle_pmd() (Chris von Recklinghausen) [1945442]
- mm/hmm: return -EFAULT when setting HMM_PFN_ERROR on requested valid pages (Chris von Recklinghausen) [1945442]
- mm/hmm: reorganize how !pte_present is handled in hmm_vma_handle_pte() (Chris von Recklinghausen) [1945442]
- mm/hmm: add missing call to hmm_range_need_fault() before returning EFAULT (Chris von Recklinghausen) [1945442]
- mm/hmm: add missing pfns set to hmm_vma_walk_pmd() (Chris von Recklinghausen) [1945442]
- mm/hmm: remove hmm_range_dma_map and hmm_range_dma_unmap (Chris von Recklinghausen) [1945442]
- mm/hmm: make full use of walk_page_range() (Chris von Recklinghausen) [1945442]
- mm/hmm: remove hmm_mirror and related (Chris von Recklinghausen) [1945442]
- mm/hmm: define the pre-processor related parts of hmm.h even if disabled (Chris von Recklinghausen) [1945442]
- mm/hmm: allow hmm_range to be used with a mmu_interval_notifier or hmm_mirror (Chris von Recklinghausen) [1945442]
- mm/hmm: hmm_range_fault() infinite loop (Chris von Recklinghausen) [1945442]
- mm/hmm: hmm_range_fault() NULL pointer bug (Chris von Recklinghausen) [1945442]
- hmm: use mmu_notifier_get/put for 'struct hmm' (Chris von Recklinghausen) [1945442]
- mm/hmm: cleanup the hmm_vma_handle_pmd stub (Chris von Recklinghausen) [1945442]
- mm/hmm: only define hmm_vma_walk_pud if needed (Chris von Recklinghausen) [1945442]
- mm/hmm: cleanup the hmm_vma_walk_hugetlb_entry stub (Chris von Recklinghausen) [1945442]
- mm/hmm: don't abuse pte_index() in hmm_vma_handle_pmd (Chris von Recklinghausen) [1945442]
- mm/hmm: comment on VM_FAULT_RETRY semantics in handle_mm_fault (Chris von Recklinghausen) [1945442]
- mm/hmm: remove the legacy hmm_pfn_* APIs (Chris von Recklinghausen) [1945442]
- mm/hmm: remove the mask variable in hmm_vma_walk_hugetlb_entry (Chris von Recklinghausen) [1945442]
- mm/hmm: remove the page_shift member from struct hmm_range (Chris von Recklinghausen) [1945442]
- mm/hmm: remove superfluous arguments from hmm_range_register (Chris von Recklinghausen) [1945442]
- mm/hmm: remove the unused vma argument to hmm_range_dma_unmap (Chris von Recklinghausen) [1945442]
- mm/hmm: remove hmm_range vma (Chris von Recklinghausen) [1945442]
- mm/hmm: remove hugetlbfs check in hmm_vma_walk_pmd (Chris von Recklinghausen) [1945442]
- mm/hmm: merge hmm_range_snapshot into hmm_range_fault (Chris von Recklinghausen) [1945442]
- mm/hmm: replace the block argument to hmm_range_fault with a flags value (Chris von Recklinghausen) [1945442]
- mm/large system hash: use vmalloc for size > MAX_ORDER when !hashdist (Chris von Recklinghausen) [1945442]
- mm/hmm: update HMM documentation (Chris von Recklinghausen) [1945442]
- mm: remove the HMM config option (Chris von Recklinghausen) [1945442]
- hugetlbfs: on restore reserve error path retain subpool reservation (Chris von Recklinghausen) [1945442]
- mm/vmalloc.c: fix potential memory leak (Chris von Recklinghausen) [1945442]
- mm/vmalloc: separate put pages and flush VM flags (Chris von Recklinghausen) [1945442]
- zsmalloc: switch from alloc_vm_area to get_vm_area (Chris von Recklinghausen) [1945442]
- mm: allow a NULL fn callback in apply_to_page_range (Chris von Recklinghausen) [1945442]
- mm: add a vmap_pfn function (Chris von Recklinghausen) [1945442]
- mm: add a VM_MAP_PUT_PAGES flag for vmap (Chris von Recklinghausen) [1945442]
- mm: update the documentation for vfree (Chris von Recklinghausen) [1945442]
- mm, slub: use kmem_cache_debug_flags() in deactivate_slab() (Chris von Recklinghausen) [1945442]
- mm, slab, slub: clear the slab_cache field when freeing page (Chris von Recklinghausen) [1945442]
- mm: slab: provide krealloc_array() (Chris von Recklinghausen) [1945442]
- mm: slab: clarify krealloc()'s behavior with __GFP_ZERO (Chris von Recklinghausen) [1945442]
- mm/slab_common.c: use list_for_each_entry in dump_unreclaimable_slab() (Chris von Recklinghausen) [1945442]
- mm/slub: make add_full() condition more explicit (Chris von Recklinghausen) [1945442]
- mm/slub: fix missing ALLOC_SLOWPATH stat when bulk alloc (Chris von Recklinghausen) [1945442]
- mm/slub.c: branch optimization in free slowpath (Chris von Recklinghausen) [1945442]
- include/linux/slab.h: fix a typo error in comment (Chris von Recklinghausen) [1945442]
- mm/vmalloc.c: fix a warning while make xmldocs (Chris von Recklinghausen) [1945442]
- mm/userfaultfd: fix memory corruption due to writeprotect (Chris von Recklinghausen) [1945442]
- mm/migrate: fixup setting UFFD_WP flag (Chris von Recklinghausen) [1945442]
- mm/rmap: fixup copying of soft dirty and uffd ptes (Chris von Recklinghausen) [1945442]
- mm/userfaultfd: disable userfaultfd-wp on x86_32 (Chris von Recklinghausen) [1945442]
- userfaultfd: selftests: fix SIGSEGV if huge mmap fails (Chris von Recklinghausen) [1945442]
- mm: do not rely on mm == current->mm in __get_user_pages_locked (Chris von Recklinghausen) [1945442]
- mm: Introduce mm_struct.has_pinned (Chris von Recklinghausen) [1945442]
- mm/gup: might_lock_read(mmap_sem) in get_user_pages_fast() (Chris von Recklinghausen) [1945442]
- mm: enforce that vmap can't map pages executable (Chris von Recklinghausen) [1945442]
- mm: remove map_vm_range (Chris von Recklinghausen) [1945442]
- mm: rename vmap_page_range to map_kernel_range (Chris von Recklinghausen) [1945442]
- mm: remove vmap_page_range_noflush and vunmap_page_range (Chris von Recklinghausen) [1945442]
- mm: only allow page table mappings for built-in zsmalloc (Chris von Recklinghausen) [1945442]
- mm: unexport unmap_kernel_range_noflush (Chris von Recklinghausen) [1945442]
- userfaultfd: selftests: add write-protect test (Chris von Recklinghausen) [1945442]
- userfaultfd: selftests: refactor statistics (Chris von Recklinghausen) [1945442]
- userfaultfd: selftest: fix compiler warning (Chris von Recklinghausen) [1945442]
- hugetlbfs: call VM_BUG_ON_PAGE earlier in free_huge_page() (Chris von Recklinghausen) [1945442]
- userfaultfd: selftest: recycle lock threads first (Chris von Recklinghausen) [1945442]
- userfaultfd: selftest: generalize read and poll (Chris von Recklinghausen) [1945442]
- userfaultfd: selftest: cleanup help messages (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: declare _UFFDIO_WRITEPROTECT conditionally (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: UFFDIO_REGISTER_MODE_WP documentation update (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: don't wake up when doing write protect (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: enabled write protection in userfaultfd API (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: add the writeprotect API to userfaultfd ioctl (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: support write protection for userfault vma range (Chris von Recklinghausen) [1945442]
- khugepaged: skip collapse if uffd-wp detected (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: support swap and page migration (Chris von Recklinghausen) [1945442]
- mm/mprotect.c: fix compilation warning because of unused 'mm' variable (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: add pmd_swp_*uffd_wp() helpers (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: drop _PAGE_UFFD_WP properly when fork (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: apply _PAGE_UFFD_WP bit (Chris von Recklinghausen) [1945442]
- mm: merge parameters for change_protection() (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: add UFFDIO_COPY_MODE_WP (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: userfaultfd_pte/huge_pmd_wp() helpers (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: add WP pagetable tracking to x86 (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: hook userfault handler to write protection fault (Chris von Recklinghausen) [1945442]
- userfaultfd: wp: add helper for writeprotect check (Chris von Recklinghausen) [1945442]
- userfaultfd: untag user pointers (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Fix typo in the Kconfig help text (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Clean up <asm/memtype.h> externs (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Rename <asm/pat.h> => <asm/memtype.h> (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Standardize on memtype_*() prefix for APIs (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Move the memtype related files to arch/x86/mm/pat/ (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Clean up PAT initialization flags (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Harmonize 'struct memtype *' local variable and function parameter use (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Simplify the free_memtype() control flow (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Create fixed width output in /sys/kernel/debug/x86/pat_memtype_list, similar to the E820 debug printouts (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Disambiguate PAT-disabled boot messages (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Update the comments in pat.c and pat_interval.c and refresh the code a bit (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Fix off-by-one bugs in interval tree search (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Rename pat_rbtree.c to pat_interval.c (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Drop the rbt_ prefix from external memtype calls (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Do not pass 'rb_root' down the memtype tree helper functions (Chris von Recklinghausen) [1945442]
- x86/mm/pat: Convert the PAT tree to a generic interval tree (Chris von Recklinghausen) [1945442]
- x86/mm: Tabulate the page table encoding definitions (Chris von Recklinghausen) [1945442]
- x86/mm: Remove the unused set_memory_wt() function (Chris von Recklinghausen) [1945442]
- x86/mm: Remove set_pages_x() and set_pages_nx() (Chris von Recklinghausen) [1945442]
- x86/mm: Remove the unused set_memory_array_*() functions (Chris von Recklinghausen) [1945442]
- x86/mm: Unexport set_memory_x() and set_memory_nx() (Chris von Recklinghausen) [1945442]
- x86/Kconfig: Fix spelling mistake "effectivness" -> "effectiveness" (Chris von Recklinghausen) [1945442]
- x86/mm: Remove unused variable 'old_pte' (Chris von Recklinghausen) [1945442]
- Revert "x86/mm/pat: Rename pat_rbtree.c to pat_interval.c" (Chris von Recklinghausen) [1945442]
- Revert "x86/mm/pat: Move the memtype related files to arch/x86/mm/pat/" (Chris von Recklinghausen) [1945442]
- ipv6: Allow the l3mdev to be a loopback (Antoine Tenart) [1973229]
- vrf: do not push non-ND strict packets with a source LLA through packet taps again (Antoine Tenart) [1965600]
- vrf: packets with lladdr src needs dst at input with orig_iif when needs strict (Antoine Tenart) [1965600]
- ipv6: Fix handling of LLA with VRF and sockets bound to VRF (Antoine Tenart) [1965600]
- ipv6: allow ping to link-local address in VRF (Antoine Tenart) [1965600]
- vrf: mark skb for multicast or link-local as enslaved to VRF (Antoine Tenart) [1965600]
- net: allow traceroute with a specified interface in a vrf (Antoine Tenart) [1965600]
- virtchnl: Enable RSS configure for AVF (Ivan Vecera) [1966954]
- virtchnl: Advertise virtchnl UDP segmentation offload capability (Ivan Vecera) [1966954]
- virtchnl: Allow ignoring opcodes on specific VF (Ivan Vecera) [1966954]
- virtchnl: Fix layout of RSS structures (Ivan Vecera) [1966954]
- virtchnl: Enable FDIR Configure for AVF (Ivan Vecera) [1966954]
- virtchnl: Add missing explicit padding to structures (Ivan Vecera) [1966954]
- virtchnl: use u8 type for a field in the virtchnl_filter struct (Ivan Vecera) [1966954]
- tools headers uapi: Sync tools/include/uapi/linux/perf_event.h (Michael Petlan) [1944685]
- tools headers uapi: Update tools's copy of linux/perf_event.h (Michael Petlan) [1944685]
- bpf: Add size arg to build_id_parse function (Michael Petlan) [1944685]
- bpf: Move stack_map_get_build_id into lib (Michael Petlan) [1944685]
- perf map: Fix error return code in maps__clone() (Michael Petlan) [1944685]
- perf ftrace: Fix access to pid in array when setting a pid filter (Michael Petlan) [1944685]
- perf auxtrace: Fix potential NULL pointer dereference (Michael Petlan) [1944685]
- perf data: Fix error return code in perf_data__create_dir() (Michael Petlan) [1944685]
- perf arm-spe: Avoid potential buffer overrun (Michael Petlan) [1944685]
- perf report: Fix wrong LBR block sorting (Michael Petlan) [1944685]
- perf inject: Fix repipe usage (Michael Petlan) [1944685]
- perf test: Change to use bash for daemon test (Michael Petlan) [1944685]
- perf record: Fix memory leak in vDSO found using ASAN (Michael Petlan) [1944685]
- perf test: Remove now useless failing sub test "BPF relocation checker" (Michael Petlan) [1944685]
- perf daemon: Return from kill functions (Michael Petlan) [1944685]
- perf daemon: Force waipid for all session on SIGCHLD delivery (Michael Petlan) [1944685]
- perf top: Fix BPF support related crash with perf_event_paranoid=3 + kptr_restrict (Michael Petlan) [1944685]
- perf pmu: Validate raw event with sysfs exported format bits (Michael Petlan) [1944685]
- perf synthetic events: Avoid write of uninitialized memory when generating PERF_RECORD_MMAP* records (Michael Petlan) [1944685]
- perf synthetic-events: Fix uninitialized 'kernel_thread' variable (Michael Petlan) [1944685]
- perf auxtrace: Fix auxtrace queue conflict (Michael Petlan) [1944685]
- perf cs-etm: Fix bitmap for option (Michael Petlan) [1944685]
- perf map: Tighten snprintf() string precision to pass gcc check on some 32-bit arches (Michael Petlan) [1944685]
- perf report: Fix -F for branch & mem modes (Michael Petlan) [1944685]
- perf tests x86: Move insn.h include to make sure it finds stddef.h (Michael Petlan) [1944685]
- perf test: Support the ins_lat check in the X86 specific test (Michael Petlan) [1944685]
- perf test: Fix sample-parsing failure on non-x86 platforms (Michael Petlan) [1944685]
- perf archive: Fix filtering of empty build-ids (Michael Petlan) [1944685]
- perf daemon: Fix compile error with Asan (Michael Petlan) [1944685]
- perf stat: Fix use-after-free when -r option is used (Michael Petlan) [1944685]
- libperf: Add perf_evlist__reset_id_hash() (Michael Petlan) [1944685]
- perf stat: Fix wrong skipping for per-die aggregation (Michael Petlan) [1944685]
- tools headers UAPI: Update tools' copy of linux/coresight-pmu.h (Michael Petlan) [1944685]
- perf test: Fix cpu and thread map leaks in perf_time_to_tsc test (Michael Petlan) [1944685]
- perf test: Fix cpu map leaks in cpu_map_print test (Michael Petlan) [1944685]
- perf test: Fix a memory leak in thread_map_remove test (Michael Petlan) [1944685]
- perf test: Fix a thread map leak in thread_map_synthesize test (Michael Petlan) [1944685]
- perf test: Fix cpu and thread map leaks in switch_tracking test (Michael Petlan) [1944685]
- perf test: Fix cpu and thread map leaks in keep_tracking test (Michael Petlan) [1944685]
- perf test: Fix cpu and thread map leaks in code_reading test (Michael Petlan) [1944685]
- perf test: Fix cpu and thread map leaks in sw_clock_freq test (Michael Petlan) [1944685]
- perf test: Fix cpu and thread map leaks in task_exit test (Michael Petlan) [1944685]
- perf test: Fix a memory leak in attr test (Michael Petlan) [1944685]
- perf test: Fix cpu and thread map leaks in basic mmap test (Michael Petlan) [1944685]
- perf tools: Fix event's PMU name parsing (Michael Petlan) [1944685]
- perf daemon: Fix running test for non root user (Michael Petlan) [1944685]
- perf daemon: Fix control fifo permissions (Michael Petlan) [1944685]
- perf build: Fix ccache usage in $(CC) when generating arch errno table (Michael Petlan) [1944685]
- perf tools: Fix documentation of verbose options (Michael Petlan) [1944685]
- perf traceevent: Ensure read cmdlines are null terminated. (Michael Petlan) [1944685]
- perf diff: Don't crash on freeing errno-session on the error path (Michael Petlan) [1944685]
- perf tools: Clean 'generated' directory used for creating the syscall table on x86 (Michael Petlan) [1944685]
- perf build: Move feature cleanup under tools/build (Michael Petlan) [1944685]
- perf tools: Cast (struct timeval).tv_sec when printing (Michael Petlan) [1944685]
- perf buildid-cache: Don't skip 16-byte build-ids (Michael Petlan) [1944685]
- perf buildid-cache: Add test for 16-byte build-id (Michael Petlan) [1944685]
- perf symbol: Remove redundant libbfd checks (Michael Petlan) [1944685]
- perf test: Output the sub testing result in cs-etm (Michael Petlan) [1944685]
- perf test: Suppress logs in cs-etm testing (Michael Petlan) [1944685]
- perf tools: Fix arm64 build error with gcc-11 (Michael Petlan) [1944685]
- perf intel-pt: Add documentation for tracing virtual machines (Michael Petlan) [1944685]
- perf intel-pt: Split VM-Entry and VM-Exit branches (Michael Petlan) [1944685]
- perf intel-pt: Adjust sample flags for VM-Exit (Michael Petlan) [1944685]
- perf intel-pt: Allow for a guest kernel address filter (Michael Petlan) [1944685]
- perf intel-pt: Support decoding of guest kernel (Michael Petlan) [1944685]
- perf machine: Factor out machine__idle_thread() (Michael Petlan) [1944685]
- perf machine: Factor out machines__find_guest() (Michael Petlan) [1944685]
- perf intel-pt: Amend decoder to track the NR flag (Michael Petlan) [1944685]
- perf intel-pt: Retain the last PIP packet payload as is (Michael Petlan) [1944685]
- perf intel_pt: Add vmlaunch and vmresume as branches (Michael Petlan) [1944685]
- perf script: Add branch types for VM-Entry and VM-Exit (Michael Petlan) [1944685]
- perf auxtrace: Automatically group aux-output events (Michael Petlan) [1944685]
- perf test: Fix unaligned access in sample parsing test (Michael Petlan) [1944685]
- perf tools: Support arch specific PERF_SAMPLE_WEIGHT_STRUCT processing (Michael Petlan) [1944685]
- perf intel-pt: Add PSB events (Michael Petlan) [1944685]
- perf intel-pt: Fix IPC with CYC threshold (Michael Petlan) [1944685]
- perf intel-pt: Fix premature IPC (Michael Petlan) [1944685]
- perf intel-pt: Fix missing CYC processing in PSB (Michael Petlan) [1944685]
- perf record: Fix continue profiling after draining the buffer (Michael Petlan) [1944685]
- perf tools: Simplify the calculation of variables (Michael Petlan) [1944685]
- perf vendor events arm64: Add JSON metrics for imx8mp DDR Perf (Michael Petlan) [1944685]
- perf vendor events arm64: Add JSON metrics for imx8mq DDR Perf (Michael Petlan) [1944685]
- perf vendor events arm64: Add JSON metrics for imx8mn DDR Perf (Michael Petlan) [1944685]
- perf vendor events arm64: Fix indentation of brackets in imx8mm metrics (Michael Petlan) [1944685]
- perf annotate: Do not jump after 'k' is pressed (Michael Petlan) [1944685]
- perf metricgroup: Remove unneeded semicolon (Michael Petlan) [1944685]
- perf tools: Add OCaml demangling (Michael Petlan) [1944685]
- tools api fs: Cache cgroupfs mount point (Michael Petlan) [1944685]
- tools api fs: Diet cgroupfs_find_mountpoint() (Michael Petlan) [1944685]
- tools api fs: Prefer cgroup v1 path in cgroupfs_find_mountpoint() (Michael Petlan) [1944685]
- perf symbols: Resolve symbols against debug file first (Michael Petlan) [1944685]
- perf probe: Fix kretprobe issue caused by GCC bug (Michael Petlan) [1944685]
- perf symbols: Fix return value when loading PE DSO (Michael Petlan) [1944685]
- perf symbols: Make dso__load_bfd_symbols() load PE files from debug cache only (Michael Petlan) [1944685]
- perf symbols: Use (long) for iterator for bfd symbols (Michael Petlan) [1944685]
- perf annotate: Fix jump parsing for C++ code. (Michael Petlan) [1944685]
- perf arm-spe: Set sample's data source field (Michael Petlan) [1944685]
- perf arm-spe: Synthesize memory event (Michael Petlan) [1944685]
- perf arm-spe: Fill address info for samples (Michael Petlan) [1944685]
- perf arm-spe: Store operation type in packet (Michael Petlan) [1944685]
- perf arm-spe: Store memory address in packet (Michael Petlan) [1944685]
- perf arm-spe: Enable sample type PERF_SAMPLE_DATA_SRC (Michael Petlan) [1944685]
- perf env: Remove unneeded internal/cpumap inclusions (Michael Petlan) [1944685]
- perf tools: Remove unused xyarray.c as it was moved to tools/lib/perf (Michael Petlan) [1944685]
- perf tools: Replace lkml.org links with lore (Michael Petlan) [1944685]
- perf tests: Add daemon 'lock' test (Michael Petlan) [1944685]
- perf tests: Add daemon 'ping' command test (Michael Petlan) [1944685]
- perf tests: Add daemon 'signal' command test (Michael Petlan) [1944685]
- perf tests: Add daemon 'stop' command test (Michael Petlan) [1944685]
- perf tests: Add daemon reconfig test (Michael Petlan) [1944685]
- perf tests: Add daemon 'list' command test (Michael Petlan) [1944685]
- perf daemon: Add examples to man page (Michael Petlan) [1944685]
- perf daemon: Add up time for daemon/session list (Michael Petlan) [1944685]
- perf daemon: Use control to stop session (Michael Petlan) [1944685]
- perf daemon: Add 'ping' command (Michael Petlan) [1944685]
- perf daemon: Set control fifo for session (Michael Petlan) [1944685]
- perf daemon: Allow only one daemon over base directory (Michael Petlan) [1944685]
- perf daemon: Add 'stop' command (Michael Petlan) [1944685]
- perf daemon: Add 'signal' command (Michael Petlan) [1944685]
- perf daemon: Add 'list' command (Michael Petlan) [1944685]
- perf daemon: Add signalfd support (Michael Petlan) [1944685]
- perf daemon: Add background support (Michael Petlan) [1944685]
- perf daemon: Add config file change check (Michael Petlan) [1944685]
- perf daemon: Add config file support (Michael Petlan) [1944685]
- perf daemon: Add client socket support (Michael Petlan) [1944685]
- perf daemon: Add server socket support (Michael Petlan) [1944685]
- perf daemon: Add base option (Michael Petlan) [1944685]
- perf daemon: Add config option (Michael Petlan) [1944685]
- perf daemon: Add daemon command (Michael Petlan) [1944685]
- perf script: Simplify bool conversion (Michael Petlan) [1944685]
- perf arm64/s390: Fix printf conversion specifier for IP addresses (Michael Petlan) [1944685]
- perf script: Support filtering by hex address (Michael Petlan) [1944685]
- perf intlist: Change 'struct intlist' int member to 'unsigned long' (Michael Petlan) [1944685]
- perf tools: Update topdown documentation for Sapphire Rapids (Michael Petlan) [1944685]
- perf stat: Support L2 Topdown events (Michael Petlan) [1944685]
- perf test: Support PERF_SAMPLE_WEIGHT_STRUCT (Michael Petlan) [1944685]
- perf report: Support instruction latency (Michael Petlan) [1944685]
- perf tools: Support PERF_SAMPLE_WEIGHT_STRUCT (Michael Petlan) [1944685]
- perf c2c: Support data block and addr block (Michael Petlan) [1944685]
- perf tools: Support data block and addr block (Michael Petlan) [1944685]
- perf tools: Support the auxiliary event (Michael Petlan) [1944685]
- perf probe: Add protection to avoid endless loop (Michael Petlan) [1944685]
- perf trace-event-info: Rename for_each_event. (Michael Petlan) [1944685]
- perf inject jit: Add namespaces support (Michael Petlan) [1944685]
- perf namespaces: Add 'in_pidns' to nsinfo struct (Michael Petlan) [1944685]
- perf tools: Use scandir() to iterate threads when synthesizing PERF_RECORD_ events (Michael Petlan) [1944685]
- perf tools: Skip PERF_RECORD_MMAP event synthesis for kernel threads (Michael Petlan) [1944685]
- perf tools: Use /proc/<PID>/task/<TID>/status for PERF_RECORD_ event synthesis (Michael Petlan) [1944685]
- perf vendor events arm64: Reference common and uarch events for A76 (Michael Petlan) [1944685]
- perf vendor events arm64: Reference common and uarch events for Ampere eMag (Michael Petlan) [1944685]
- perf vendor events arm64: Add common and uarch event JSON (Michael Petlan) [1944685]
- perf vendor events arm64: Fix Ampere eMag event typo (Michael Petlan) [1944685]
- perf script: Support DSO filter like in other perf tools (Michael Petlan) [1944685]
- perf tools: Fix DSO filtering when not finding a map for a sampled address (Michael Petlan) [1944685]
- perf stat: Add Topdown metrics events as default events (Michael Petlan) [1944685]
- perf test: Add parse-metric memory bandwidth testcase (Michael Petlan) [1944685]
- perf tools: Add 'ping' control command (Michael Petlan) [1944685]
- perf tools: Add 'stop' control command (Michael Petlan) [1944685]
- perf config: Make perf_config_global() global (Michael Petlan) [1944685]
- perf config: Make perf_config_system() global (Michael Petlan) [1944685]
- perf config: Add perf_home_perfconfig function (Michael Petlan) [1944685]
- perf debug: Add debug_set_display_time function (Michael Petlan) [1944685]
- perf config: Add config set interface (Michael Petlan) [1944685]
- perf config: Make perf_config_from_file() static (Michael Petlan) [1944685]
- perf test: Add test case for PERF_SAMPLE_CODE_PAGE_SIZE (Michael Petlan) [1944685]
- perf report: Add support for PERF_SAMPLE_CODE_PAGE_SIZE (Michael Petlan) [1944685]
- perf script: Add support for PERF_SAMPLE_CODE_PAGE_SIZE (Michael Petlan) [1944685]
- perf record: Add support for PERF_SAMPLE_CODE_PAGE_SIZE (Michael Petlan) [1944685]
- perf mem: Support data page size (Michael Petlan) [1944685]
- perf mem: Clean up output format (Michael Petlan) [1944685]
- perf cs-etm: Update ARM's CoreSight hardware tracing OpenCSD library to v1.0.0 (Michael Petlan) [1944685]
- perf c2c: Add local variables for output metrics (Michael Petlan) [1944685]
- perf c2c: Refactor node display (Michael Petlan) [1944685]
- perf c2c: Fix argument type for percent() (Michael Petlan) [1944685]
- perf c2c: Refactor display filter (Michael Petlan) [1944685]
- perf c2c: Refactor hist entry validation (Michael Petlan) [1944685]
- perf c2c: Rename for shared cache line stats (Michael Petlan) [1944685]
- perf stat: Enable counting events for BPF programs (Michael Petlan) [1944685]
- perf build: Support build BPF skeletons with perf (Michael Petlan) [1944685]
- perf record: Tweak "Lowering..." warning in record_opts__config_freq (Michael Petlan) [1944685]
- perf buildid-list: Add support for mmap2's buildid events (Michael Petlan) [1944685]
- perf buildid-cache: Add --debuginfod option to specify a server to fetch debug files (Michael Petlan) [1944685]
- perf tools: Add support to display build ids when available in PERF_RECORD_MMAP2 events (Michael Petlan) [1944685]
- perf record: Add --buildid-mmap option to enable PERF_RECORD_MMAP2's build id (Michael Petlan) [1944685]
- perf tools: Allow synthesizing the build id for kernel/modules/tasks in PERF_RECORD_MMAP2 (Michael Petlan) [1944685]
- perf tools: Allow using PERF_RECORD_MMAP2 to synthesize the kernel modules maps (Michael Petlan) [1944685]
- perf tools: Allow using PERF_RECORD_MMAP2 to synthesize the kernel map (Michael Petlan) [1944685]
- perf tools: Store build id when available in PERF_RECORD_MMAP2 metadata events (Michael Petlan) [1944685]
- perf tools: Do not swap mmap2 fields in case it contains build id (Michael Petlan) [1944685]
- perf arm64: Add argument support for SDT (Michael Petlan) [1944685]
- perf probe: Fixup Arm64 SDT arguments (Michael Petlan) [1944685]
- perf/x86/kvm: Fix Broadwell Xeon stepping in isolation_ucodes[] (Michael Petlan) [1944685]
- perf/x86/intel: Fix unchecked MSR access error caused by VLBR_EVENT (Michael Petlan) [1944685]
- perf/x86/intel: Fix a crash caused by zero PEBS status (Michael Petlan) [1944685]
- perf/x86/intel: Set PERF_ATTACH_SCHED_CB for large PEBS and LBR (Michael Petlan) [1944685]
- perf/core: Flush PMU internal buffers for per-CPU events (Michael Petlan) [1944685]
- perf/x86/rapl: Fix psys-energy event on Intel SPR platform (Michael Petlan) [1944685]
- perf/x86/rapl: Only check lower 32bits for RAPL energy counters (Michael Petlan) [1944685]
- perf/x86/rapl: Add msr mask support (Michael Petlan) [1944685]
- perf/x86/kvm: Add Cascade Lake Xeon steppings to isolation_ucodes[] (Michael Petlan) [1944685]
- perf/x86/intel: Support CPUID 10.ECX to disable fixed counters (Michael Petlan) [1944685]
- perf/x86/intel: Add perf core PMU support for Sapphire Rapids (Michael Petlan) [1944685]
- perf/x86/intel: Filter unsupported Topdown metrics event (Michael Petlan) [1944685]
- perf/x86/intel: Factor out intel_update_topdown_event() (Michael Petlan) [1944685]
- perf/core: Add PERF_SAMPLE_WEIGHT_STRUCT (Michael Petlan) [1944685]
- perf/intel: Remove Perfmon-v4 counter_freezing support (Michael Petlan) [1944685]
- perf: Add build id data in mmap2 event (Michael Petlan) [1944685]
- bpf: Remove atomics tests from test_progs (Jiri Olsa) [1874007]
- libbpf: Fixes incorrect rx_ring_setup_done (Jiri Olsa) [1874007]
- error-injection: Consolidate override function definition (Jiri Olsa) [1874007]
- selftests/bpf: Fix endianness issues in atomic tests (Jiri Olsa) [1874007]
- bpf: Fix a spelling typo in bpf_atomic_alu_string disasm (Jiri Olsa) [1874007]
- bpf, x86: Fix BPF_FETCH atomic and/or/xor with r0 as src (Jiri Olsa) [1874007]
- bpf: Explicitly zero-extend R0 after 32-bit cmpxchg (Jiri Olsa) [1874007]
- bpf: Account for BPF_FETCH in insn_has_def32() (Jiri Olsa) [1874007]
- bpf: Fix subreg optimization for BPF_FETCH (Jiri Olsa) [1874007]
- bpf: Fix the irq and nmi check in bpf_sk_storage for tracing usage (Jiri Olsa) [1874007]
- selftests/bpf: Fix a compiler warning in local_storage test (Jiri Olsa) [1874007]
- bpf: Change 'BPF_ADD' to 'BPF_AND' in print_bpf_insn() (Jiri Olsa) [1874007]
- selftests/bpf: Don't exit on failed bpf_testmod unload (Jiri Olsa) [1874007]
- bpf: Avoid old-style declaration warnings (Jiri Olsa) [1874007]
- bpf: Dont allow vmlinux BTF to be used in map_create and prog_load. (Jiri Olsa) [1874007]
- tools: Factor Clang, LLC and LLVM utils definitions (Jiri Olsa) [1874007]
- tools/runqslower: Build bpftool using HOSTCC (Jiri Olsa) [1874007]
- tools/runqslower: Enable out-of-tree build (Jiri Olsa) [1874007]
- tools/runqslower: Use Makefile.include (Jiri Olsa) [1874007]
- bpf: Expose bpf_sk_storage_* to iterator programs (Jiri Olsa) [1874007]
- tools/resolve_btfids: Warn when having multiple IDs for single type (Jiri Olsa) [1874007]
- tools/resolve_btfids: Fix some error messages (Jiri Olsa) [1874007]
- samples/bpf: Add BPF_ATOMIC_OP macro for BPF samples (Jiri Olsa) [1874007]
- samples/bpf: Fix possible hang in xdpsock with multiple threads (Jiri Olsa) [1874007]
- bpf: samples: Do not touch RLIMIT_MEMLOCK (Jiri Olsa) [1874007]
- samples/bpf: Use recvfrom() in xdpsock/l2fwd (Jiri Olsa) [1874007]
- samples/bpf: Use recvfrom() in xdpsock/rxdrop (Jiri Olsa) [1874007]
- samples: bpf: Remove bpf_load loader completely (Jiri Olsa) [1874007]
- samples: bpf: Fix lwt_len_hist reusing previous BPF map (Jiri Olsa) [1874007]
- samples: bpf: Refactor test_overhead program with libbpf (Jiri Olsa) [1874007]
- samples: bpf: Refactor task_fd_query program with libbpf (Jiri Olsa) [1874007]
- samples: bpf: Refactor test_cgrp2_sock2 program with libbpf (Jiri Olsa) [1874007]
- samples: bpf: Refactor hbm program with libbpf (Jiri Olsa) [1874007]
- samples/bpf: Increment Tx stats at sending (Jiri Olsa) [1874007]
- samples/bpf: Remove unused test_ipip.sh (Jiri Olsa) [1874007]
- samples/bpf: Remove duplicate include in hbm (Jiri Olsa) [1874007]
- bpftool: Fix compilation failure for net.o with older glibc (Jiri Olsa) [1874007]
- tools/bpftool: Auto-detect split BTFs in common cases (Jiri Olsa) [1874007]
- tools/bpftool: Emit name <anon> for anonymous BTFs (Jiri Olsa) [1874007]
- bpftool: Add {i,d}tlb_misses support for bpftool profile (Jiri Olsa) [1874007]
- tools/bpf: Always run the *-clean recipes (Jiri Olsa) [1874007]
- tools/bpf: Add bootstrap/ to .gitignore (Jiri Olsa) [1874007]
- tools/bpftool: Fix cross-build (Jiri Olsa) [1874007]
- tools/bpftool: Force clean of out-of-tree build (Jiri Olsa) [1874007]
- tools/bpftool: Add support for in-kernel and named BTF in `btf show` (Jiri Olsa) [1874007]
- bpftool: Add support for task local storage (Jiri Olsa) [1874007]
- tools/bpftool: Add bpftool support for split BTF (Jiri Olsa) [1874007]
- bpf: Unbreak BPF_PROG_TYPE_KPROBE when kprobe is called via do_int3 (Jiri Olsa) [1874007]
- bpf: Fix bpf_put_raw_tracepoint()'s use of __module_address() (Jiri Olsa) [1874007]
- bpf: Expose bpf_d_path helper to sleepable LSM hooks (Jiri Olsa) [1874007]
- bpf: Allow using bpf_sk_storage in FENTRY/FEXIT/RAW_TP (Jiri Olsa) [1874007]
- selftests/bpf: Add remaining ASSERT_xxx() variants (Jiri Olsa) [1874007]
- libbpf: Fix potential NULL pointer dereference (Jiri Olsa) [1874007]
- libbpf: Only create rx and tx XDP rings when necessary (Jiri Olsa) [1874007]
- libbpf: Ensure umem pointer is non-NULL before dereferencing (Jiri Olsa) [1874007]
- libbpf: Restore umem state after socket create failure (Jiri Olsa) [1874007]
- selftests/bpf: Fix core_reloc test runner (Jiri Olsa) [1874007]
- bpf: Fix umd memory leak in copy_process() (Jiri Olsa) [1874007]
- libbpf: Fix error path in bpf_object__elf_init() (Jiri Olsa) [1874007]
- bpf: Change inode_storage's lookup_elem return value from NULL to -EBADF (Jiri Olsa) [1874007]
- bpf: Prohibit alu ops for pointer types not defining ptr_limit (Jiri Olsa) [1874007]
- selftests/bpf: Use the last page in test_snprintf_btf on s390 (Jiri Olsa) [1874007]
- bpf: Be less specific about socket cookies guarantees (Jiri Olsa) [1874007]
- bpf: Fix 32 bit src register truncation on div/mod (Jiri Olsa) [1874007]
- bpf: Fix verifier jmp32 pruning decision logic (Jiri Olsa) [1874007]
- bpf: Fix verifier jsgt branch analysis on max bound (Jiri Olsa) [1874007]
- bpf: Check for integer overflow when using roundup_pow_of_two() (Jiri Olsa) [1874007]
- bpf, preload: Fix build when $(O) points to a relative path (Jiri Olsa) [1874007]
- bpf: Drop disabled LSM hooks from the sleepable set (Jiri Olsa) [1874007]
- bpf, cgroup: Fix problematic bounds check (Jiri Olsa) [1874007]
- bpf, cgroup: Fix optlen WARN_ON_ONCE toctou (Jiri Olsa) [1874007]
- bpf: Fix signed_{sub,add32}_overflows type handling (Jiri Olsa) [1874007]
- bpf: Fix helper bpf_map_peek_elem_proto pointing to wrong callback (Jiri Olsa) [1874007]
- selftests/bpf: Add verifier test for PTR_TO_MEM spill (Jiri Olsa) [1874007]
- bpf: Support PTR_TO_MEM{,_OR_NULL} register spilling (Jiri Olsa) [1874007]
- libbpf: Allow loading empty BTFs (Jiri Olsa) [1874007]
- bpf: Allow empty module BTFs (Jiri Olsa) [1874007]
- bpf: Don't leak memory in bpf getsockopt when optlen == 0 (Jiri Olsa) [1874007]
- bpf: Update local storage test to check handling of null ptrs (Jiri Olsa) [1874007]
- bpf: Fix typo in bpf_inode_storage.c (Jiri Olsa) [1874007]
- bpf: Add size arg to build_id_parse function (Jiri Olsa) [1874007]
- bpf: Move stack_map_get_build_id into lib (Jiri Olsa) [1874007]
- bpf: Add tests for new BPF atomic operations (Jiri Olsa) [1874007]
- bpf: Add bitwise atomic instructions (Jiri Olsa) [1874007]
- bpf: Pull out a macro for interpreting atomic ALU operations (Jiri Olsa) [1874007]
- bpf: Add instructions for atomic_[cmp]xchg (Jiri Olsa) [1874007]
- bpf: Add BPF_FETCH field / create atomic_fetch_add instruction (Jiri Olsa) [1874007]
- bpf: Move BPF_STX reserved field check into BPF_STX verifier code (Jiri Olsa) [1874007]
- bpf: Rename BPF_XADD and prepare to encode other atomics in .imm (Jiri Olsa) [1874007]
- bpf: x86: Factor out a lookup table for some ALU opcodes (Jiri Olsa) [1874007]
- bpf: x86: Factor out emission of REX byte (Jiri Olsa) [1874007]
- bpf: x86: Factor out emission of ModR/M for *(reg + off) (Jiri Olsa) [1874007]
- bpf, libbpf: Avoid unused function warning on bpf_tail_call_static (Jiri Olsa) [1874007]
- selftests/bpf: Install btf_dump test cases (Jiri Olsa) [1874007]
- selftests/bpf: Fix installation of urandom_read (Jiri Olsa) [1874007]
- selftests/bpf: Move generated test files to $(TEST_GEN_FILES) (Jiri Olsa) [1874007]
- selftests/bpf: Fix out-of-tree build (Jiri Olsa) [1874007]
- selftests/bpf: Enable cross-building (Jiri Olsa) [1874007]
- selftests/bpf: Test kernel module ksym externs (Jiri Olsa) [1874007]
- libbpf: Support kernel module ksym externs (Jiri Olsa) [1874007]
- bpf: Support BPF ksym variables in kernel modules (Jiri Olsa) [1874007]
- selftests/bpf: Sync RCU before unloading bpf_testmod (Jiri Olsa) [1874007]
- bpf: Fix a verifier message for alloc size helper arg (Jiri Olsa) [1874007]
- bpf: Clarify return value of probe str helpers (Jiri Olsa) [1874007]
- libbpf: Clarify kernel type use with USER variants of CORE reading macros (Jiri Olsa) [1874007]
- selftests/bpf: Remove duplicate include in test_lsm (Jiri Olsa) [1874007]
- bpf: Remove unnecessary <argp.h> include from preload/iterators (Jiri Olsa) [1874007]
- selftests/bpf: Add tests for user- and non-CO-RE BPF_CORE_READ() variants (Jiri Olsa) [1874007]
- libbpf: Add non-CO-RE variants of BPF_CORE_READ() macro family (Jiri Olsa) [1874007]
- selftests/bpf: Fix a compile error for BPF_F_BPRM_SECUREEXEC (Jiri Olsa) [1874007]
- bpf: Use thread_group_leader() (Jiri Olsa) [1874007]
- bpf: Save correct stopping point in file seq iteration (Jiri Olsa) [1874007]
- selftests/bpf: Work-around EBUSY errors from hashmap update/delete (Jiri Olsa) [1874007]
- bpf: Add schedule point in htab_init_buckets() (Jiri Olsa) [1874007]
- bpf: Remove unused including <linux/version.h> (Jiri Olsa) [1874007]
- selftests/bpf: Fix spelling mistake "tranmission" -> "transmission" (Jiri Olsa) [1874007]
- selftests/bpf: Clarify build error if no vmlinux (Jiri Olsa) [1874007]
- tweewide: Fix most Shebang lines (Jiri Olsa) [1874007]
- bpf: In bpf_task_fd_query use fget_task (Jiri Olsa) [1874007]
- selftests/bpf: Add a test for ptr_to_map_value on stack for helper access (Jiri Olsa) [1874007]
- bpf: Permits pointers on stack for helper calls (Jiri Olsa) [1874007]
- libbpf: Expose libbpf ring_buffer epoll_fd (Jiri Olsa) [1874007]
- selftests/bpf: Add set_attach_target() API selftest for module target (Jiri Olsa) [1874007]
- libbpf: Support modules in bpf_program__set_attach_target() API (Jiri Olsa) [1874007]
- selftests/bpf: Silence ima_setup.sh when not running in verbose mode. (Jiri Olsa) [1874007]
- selftests/bpf: Drop the need for LLVM's llc (Jiri Olsa) [1874007]
- selftests/bpf: fix bpf_testmod.ko recompilation logic (Jiri Olsa) [1874007]
- selftests/bpf: Make selftest compilation work on clang 11 (Jiri Olsa) [1874007]
- selftests/bpf: Xsk selftests - adding xdpxceiver to .gitignore (Jiri Olsa) [1874007]
- selftests/bpf: Drop tcp-{client,server}.py from Makefile (Jiri Olsa) [1874007]
- selftests/bpf: Xsk selftests - Bi-directional Sockets - SKB, DRV (Jiri Olsa) [1874007]
- selftests/bpf: Xsk selftests - Socket Teardown - SKB, DRV (Jiri Olsa) [1874007]
- selftests/bpf: Xsk selftests - DRV POLL, NOPOLL (Jiri Olsa) [1874007]
- selftests/bpf: Xsk selftests - SKB POLL, NOPOLL (Jiri Olsa) [1874007]
- selftests/bpf: Xsk selftests framework (Jiri Olsa) [1874007]
- bpf: Return -ENOTSUPP when attaching to non-kernel BTF (Jiri Olsa) [1874007]
- bpf: Propagate __user annotations properly (Jiri Olsa) [1874007]
- bpf: Avoid overflows involving hash elem_size (Jiri Olsa) [1874007]
- selftests/bpf: Test bpf_sk_storage_get in tcp iterators (Jiri Olsa) [1874007]
- selftests/bpf: Add an iterator selftest for bpf_sk_storage_get (Jiri Olsa) [1874007]
- selftests/bpf: Add an iterator selftest for bpf_sk_storage_delete (Jiri Olsa) [1874007]
- bpf: Add a bpf_sock_from_file helper (Jiri Olsa) [1874007]
- selftests/bpf: Avoid errno clobbering (Jiri Olsa) [1874007]
- selftests/bpf: Print reason when a tester could not run a program (Jiri Olsa) [1874007]
- selftests/bpf: Fix invalid use of strncat in test_sockmap (Jiri Olsa) [1874007]
- libbpf: Use memcpy instead of strncpy to please GCC (Jiri Olsa) [1874007]
- selftests/bpf: Add fentry/fexit/fmod_ret selftest for kernel module (Jiri Olsa) [1874007]
- selftests/bpf: Add tp_btf CO-RE reloc test for modules (Jiri Olsa) [1874007]
- libbpf: Support attachment of BPF tracing programs to kernel modules (Jiri Olsa) [1874007]
- libbpf: Factor out low-level BPF program loading helper (Jiri Olsa) [1874007]
- bpf: Allow to specify kernel module BTFs when attaching BPF programs (Jiri Olsa) [1874007]
- bpf: Remove hard-coded btf_vmlinux assumption from BPF verifier (Jiri Olsa) [1874007]
- selftests/bpf: Add CO-RE relocs selftest relying on kernel module BTF (Jiri Olsa) [1874007]
- selftests/bpf: Add support for marking sub-tests as skipped (Jiri Olsa) [1874007]
- selftests/bpf: Add bpf_testmod kernel module for testing (Jiri Olsa) [1874007]
- libbpf: Add kernel module BTF support for CO-RE relocations (Jiri Olsa) [1874007]
- libbpf: Refactor CO-RE relocs to not assume a single BTF object (Jiri Olsa) [1874007]
- libbpf: Add internal helper to load BTF data by FD (Jiri Olsa) [1874007]
- selftests/bpf: Add Userspace tests for TCP_WINDOW_CLAMP (Jiri Olsa) [1874007]
- bpf: Fix cold build of test_progs-no_alu32 (Jiri Olsa) [1874007]
- libbpf: Cap retries in sys_bpf_prog_load (Jiri Olsa) [1874007]
- libbpf: Sanitise map names before pinning (Jiri Olsa) [1874007]
- libbpf: Fail early when loading programs with unspecified type (Jiri Olsa) [1874007]
- selftests/bpf: Indent ima_setup.sh with tabs. (Jiri Olsa) [1874007]
- selftests/bpf: Add config dependency on BLK_DEV_LOOP (Jiri Olsa) [1874007]
- selftests/bpf: Ensure securityfs mount before writing ima policy (Jiri Olsa) [1874007]
- selftests/bpf: Update ima_setup.sh for busybox (Jiri Olsa) [1874007]
- libbpf: Separate XDP program load with xsk socket creation (Jiri Olsa) [1874007]
- selftests/bpf: Copy file using read/write in local storage test (Jiri Olsa) [1874007]
- libbpf: Add base BTF accessor (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for bpf progs (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for bpf local storage maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for stackmap maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for bpf ringbuffer (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for reuseport_array maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for queue_stack_maps maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for lpm_trie maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for hashtab maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for devmap maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for cgroup storage maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for cpumap maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for bpf_struct_ops maps (Jiri Olsa) [1874007]
- bpf: Eliminate rlimit-based memory accounting for arraymap maps (Jiri Olsa) [1874007]
- bpf: Memcg-based memory accounting for bpf local storage maps (Jiri Olsa) [1874007]
- bpf: Memcg-based memory accounting for bpf ringbuffer (Jiri Olsa) [1874007]
- bpf: Memcg-based memory accounting for lpm_trie maps (Jiri Olsa) [1874007]
- bpf: Refine memcg-based memory accounting for hashtab maps (Jiri Olsa) [1874007]
- bpf: Refine memcg-based memory accounting for devmap maps (Jiri Olsa) [1874007]
- bpf: Memcg-based memory accounting for cgroup storage maps (Jiri Olsa) [1874007]
- bpf: Refine memcg-based memory accounting for cpumap maps (Jiri Olsa) [1874007]
- bpf: Refine memcg-based memory accounting for arraymap maps (Jiri Olsa) [1874007]
- bpf: Memcg-based memory accounting for bpf maps (Jiri Olsa) [1874007]
- bpf: Prepare for memcg-based memory accounting for bpf maps (Jiri Olsa) [1874007]
- bpf: Memcg-based memory accounting for bpf progs (Jiri Olsa) [1874007]
- selftests/bpf: Extend bind{4,6} programs with a call to bpf_setsockopt (Jiri Olsa) [1874007]
- selftests/bpf: Rewrite test_sock_addr bind bpf into C (Jiri Olsa) [1874007]
- selftests/bpf: Fix flavored variants of test_ima (Jiri Olsa) [1874007]
- libbpf: Replace size_t with __u32 in xsk interfaces (Jiri Olsa) [1874007]
- bpf: Add a selftest for bpf_ima_inode_hash (Jiri Olsa) [1874007]
- bpf: Add a BPF helper for getting the IMA hash of an inode (Jiri Olsa) [1874007]
- libbpf: Add support for canceling cached_cons advance (Jiri Olsa) [1874007]
- bpf: Refactor check_cfg to use a structured loop. (Jiri Olsa) [1874007]
- selftest/bpf: Fix rst formatting in readme (Jiri Olsa) [1874007]
- selftest/bpf: Fix link in readme (Jiri Olsa) [1874007]
- bpf: Simplify task_file_seq_get_next() (Jiri Olsa) [1874007]
- selftests/bpf: Mark tests that require unaligned memory access (Jiri Olsa) [1874007]
- selftests/bpf: Avoid running unprivileged tests with alignment requirements (Jiri Olsa) [1874007]
- selftests/bpf: Fix broken riscv build (Jiri Olsa) [1874007]
- bpf: Add bpf_ktime_get_coarse_ns helper (Jiri Olsa) [1874007]
- bpf: Add tests for bpf_bprm_opts_set helper (Jiri Olsa) [1874007]
- bpf: Add bpf_bprm_opts_set helper (Jiri Olsa) [1874007]
- libbpf: bpf__find_by_name[_kind] should use btf__get_nr_types() (Jiri Olsa) [1874007]
- selftest/bpf: Fix IPV6FR handling in flow dissector (Jiri Olsa) [1874007]
- bpf: Augment the set of sleepable LSM hooks (Jiri Olsa) [1874007]
- bpf: selftest: Use bpf_sk_storage in FENTRY/FEXIT/RAW_TP (Jiri Olsa) [1874007]
- selftests/bpf: Add asm tests for pkt vs pkt_end comparison. (Jiri Olsa) [1874007]
- selftests/bpf: Add skb_pkt_end test (Jiri Olsa) [1874007]
- bpf: Support for pointers beyond pkt_end. (Jiri Olsa) [1874007]
- bpf: Compile out btf_parse_module() if module BTF is not enabled (Jiri Olsa) [1874007]
- selftest/bpf: Add missed ip6ip6 test back (Jiri Olsa) [1874007]
- bpf: Keep module's btf_data_size intact after load (Jiri Olsa) [1874007]
- bpf: Sanitize BTF data pointer after module is loaded (Jiri Olsa) [1874007]
- bpf: Load and verify kernel module BTFs (Jiri Olsa) [1874007]
- bpf: Assign ID to vmlinux BTF and return extra info for BTF in GET_OBJ_INFO (Jiri Olsa) [1874007]
- bpf: Add in-kernel split BTF support (Jiri Olsa) [1874007]
- bpf, btf: Remove the duplicate btf_ids.h include (Jiri Olsa) [1874007]
- selftests/bpf: Fix selftest build with old libc (Jiri Olsa) [1874007]
- bpf: Exercise syscall operations for inode and sk storage (Jiri Olsa) [1874007]
- bpf: Add tests for task_local_storage (Jiri Olsa) [1874007]
- bpf: Update selftests for local_storage to use vmlinux.h (Jiri Olsa) [1874007]
- bpf: Implement get_current_task_btf and RET_PTR_TO_BTF_ID (Jiri Olsa) [1874007]
- libbpf: Add support for task local storage (Jiri Olsa) [1874007]
- bpf: Implement task local storage (Jiri Olsa) [1874007]
- bpf: Allow LSM programs to use bpf spin locks (Jiri Olsa) [1874007]
- bpf: Lift hashtab key_size limit (Jiri Olsa) [1874007]
- selftests/bpf: Add split BTF dedup selftests (Jiri Olsa) [1874007]
- libbpf: Accomodate DWARF/compiler bug with duplicated identical arrays (Jiri Olsa) [1874007]
- libbpf: Support BTF dedup of split BTFs (Jiri Olsa) [1874007]
- libbpf: Fix BTF data layout checks and allow empty BTF (Jiri Olsa) [1874007]
- selftests/bpf: Add checking of raw type dump in BTF writer APIs selftests (Jiri Olsa) [1874007]
- selftests/bpf: Add split BTF basic test (Jiri Olsa) [1874007]
- libbpf: Implement basic split BTF support (Jiri Olsa) [1874007]
- libbpf: Unify and speed up BTF string deduplication (Jiri Olsa) [1874007]
- selftest/bpf: Relax btf_dedup test checks (Jiri Olsa) [1874007]
- libbpf: Factor out common operations in BTF writing APIs (Jiri Olsa) [1874007]
- selftest/bpf: Use global variables instead of maps for test_tcpbpf_kern (Jiri Olsa) [1874007]
- selftests/bpf: Migrate tcpbpf_user.c to use BPF skeleton (Jiri Olsa) [1874007]
- selftests/bpf: Replace EXPECT_EQ with ASSERT_EQ and refactor verify_results (Jiri Olsa) [1874007]
- selftests/bpf: Drop python client/server in favor of threads (Jiri Olsa) [1874007]
- selftests/bpf: Move test_tcppbf_user into test_progs (Jiri Olsa) [1874007]
- bpf: Fix error path in htab_map_alloc() (Jiri Olsa) [1874007]
- bpf: Avoid hashtab deadlock with map_locked (Jiri Olsa) [1874007]
- bpf: Use separate lockdep class for each hashtab (Jiri Olsa) [1874007]
- fault-injection: handle EI_ETYPE_TRUE (Jiri Olsa) [1874007]
- vfs, fdtable: Add fget_task helper (Jiri Olsa) [1874007]
- [s390] vfio-pci/zdev: Add zPCI capabilities to VFIO_DEVICE_GET_INFO (Claudio Imbrenda) [1858844]
- [s390] vfio: Introduce capability definitions for VFIO_DEVICE_GET_INFO (Claudio Imbrenda) [1858844]
- [s390] s390/pci: track whether util_str is valid in the zpci_dev (Claudio Imbrenda) [1858844]
- [s390] s390/pci: stash version in the zpci_dev (Claudio Imbrenda) [1858844]

* Thu Jul 01 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-320.el8]
- Enable CRYPTO_SHA512 and _SSSE3 so these are available in FIPS mode (Vladis Dronov) [1973106]
- crypto: drbg - self test for HMAC(SHA-512) (Vladis Dronov) [1973106]
- xfrm: xfrm_state_mtu should return at least 1280 for ipv6 (Sabrina Dubroca) [1919786]
- xfrm: remove get_mtu indirection from xfrm_type (Sabrina Dubroca) [1919786]
- netfilter: move endif to correct location (Florian Westphal) [1971991]
- netfilter: nft_compat: remove flush counter optimization (Florian Westphal) [1974323]
- mptcp: fix soft lookup in subflow_error_report() (Florian Westphal) [1936368]
- selftests: mptcp: enable syncookie only in absence of reorders (Florian Westphal) [1936368]
- mptcp: do not warn on bad input from the network (Florian Westphal) [1936368]
- mptcp: wake-up readers only for in sequence data (Florian Westphal) [1936368]
- mptcp: try harder to borrow memory from subflow under pressure (Florian Westphal) [1936368]
- mptcp: Fix out of bounds when parsing TCP options (Florian Westphal) [1936368]
- selftests: mptcp_connect: add SO_TIMESTAMPNS cmsg support (Florian Westphal) [1936368]
- mptcp: receive path cmsg support (Florian Westphal) [1936368]
- tcp: export timestamp helpers for mptcp (Florian Westphal) [1936368]
- mptcp: setsockopt: handle SOL_SOCKET in one place only (Florian Westphal) [1936368]
- mptcp: sockopt: propagate timestamp request to subflows (Florian Westphal) [1936368]
- sock: expose so_timestamping options for mptcp (Florian Westphal) [1936368]
- mptcp: restrict values of 'enabled' sysctl (Florian Westphal) [1936368]
- mptcp: support SYSCTL only if enabled (Florian Westphal) [1936368]
- mptcp: make sure flag signal is set when add addr with port (Florian Westphal) [1936368]
- mptcp: remove redundant initialization in pm_nl_init_net() (Florian Westphal) [1936368]
- mptcp: generate subflow hmac after mptcp_finish_join() (Florian Westphal) [1936368]
- mptcp: using TOKEN_MAX_RETRIES instead of magic number (Florian Westphal) [1936368]
- mptcp: fix pr_debug in mptcp_token_new_connect (Florian Westphal) [1936368]
- mptcp: update selftest for fallback due to OoO (Florian Westphal) [1936368]
- mptcp: do not reset MP_CAPABLE subflow on mapping errors (Florian Westphal) [1936368]
- mptcp: always parse mptcp options for MPC reqsk (Florian Westphal) [1936368]
- mptcp: fix sk_forward_memory corruption on retransmission (Florian Westphal) [1936368]
- mptcp: validate 'id' when stopping the ADD_ADDR retransmit timer (Florian Westphal) [1936368]
- mptcp: avoid error message on infinite mapping (Florian Westphal) [1936368]
- mptcp: drop unconditional pr_warn on bad opt (Florian Westphal) [1936368]
- mptcp: avoid OOB access in setsockopt() (Florian Westphal) [1936368]
- mptcp: fix data stream corruption (Florian Westphal) [1936368]
- mptcp: fix splat when closing unaccepted socket (Florian Westphal) [1936368]
- selftests: mptcp: add a test case for MSG_PEEK (Florian Westphal) [1936368]
- mptcp: add MSG_PEEK support (Florian Westphal) [1936368]
- mptcp: ignore unsupported msg flags (Florian Westphal) [1936368]
- mptcp: implement MSG_TRUNC support (Florian Westphal) [1936368]
- mptcp: implement dummy MSG_ERRQUEUE support (Florian Westphal) [1936368]
- mptcp: Retransmit DATA_FIN (Florian Westphal) [1936368]
- selftests: mptcp: add packet mark test case (Florian Westphal) [1936368]
- selftests: mptcp: add the net device name testcase (Florian Westphal) [1936368]
- selftests: mptcp: dump more info on mpjoin errors (Florian Westphal) [1936368]
- selftests: mptcp: init nstat history (Florian Westphal) [1936368]
- selftests: mptcp: launch mptcp_connect with timeout (Florian Westphal) [1936368]
- selftests: mptcp: remove id 0 address testcases (Florian Westphal) [1936368]
- selftests: mptcp: add addr argument for del_addr (Florian Westphal) [1936368]
- selftests: mptcp: avoid calling pm_nl_ctl with bad IDs (Florian Westphal) [1936368]
- selftests: mptcp: signal addresses testcases (Florian Westphal) [1936368]
- selftests: mptcp: timeout testcases for multi addresses (Florian Westphal) [1936368]
- selftests: mptcp: add cfg_do_w for cfg_remove (Florian Westphal) [1936368]
- selftests: mptcp: Restore packet capture option in join tests (Florian Westphal) [1936368]
- selftests: mptcp: add testcases for removing addrs (Florian Westphal) [1936368]
- selftests: mptcp: set addr id for removing testcases (Florian Westphal) [1936368]
- selftests: mptcp: add invert argument for chk_rm_nr (Florian Westphal) [1936368]
- selftests: mptcp: fail if not enough SYN/3rd ACK (Florian Westphal) [1936368]
- selftests: mptcp: display warnings on one line (Florian Westphal) [1936368]
- selftests: mptcp: fix ACKRX debug message (Florian Westphal) [1936368]
- selftests: mptcp: dump more info on errors (Florian Westphal) [1936368]
- selftests: mptcp: add command line arguments for mptcp_join.sh (Florian Westphal) [1936368]
- selftests: mptcp: add testcases for ADD_ADDR with port (Florian Westphal) [1936368]
- selftests: mptcp: add port argument for pm_nl_ctl (Florian Westphal) [1936368]
- selftests: mptcp: add testcases for newly added addresses (Florian Westphal) [1936368]
- selftests: mptcp: use minus values for removing address numbers (Florian Westphal) [1936368]
- selftests: increase timeout to 10 min (Florian Westphal) [1936368]
- selftests: mptcp: add IPv4-mapped IPv6 testcases (Florian Westphal) [1936368]
- mptcp: use mptcp_for_each_subflow in mptcp_close (Florian Westphal) [1936368]
- mptcp: add tracepoint in subflow_check_data_avail (Florian Westphal) [1936368]
- mptcp: add tracepoint in ack_update_msk (Florian Westphal) [1936368]
- mptcp: add tracepoint in get_mapping_status (Florian Westphal) [1936368]
- mptcp: add tracepoint in mptcp_subflow_get_send (Florian Westphal) [1936368]
- mptcp: export mptcp_subflow_active (Florian Westphal) [1936368]
- mptcp: fix format specifiers for unsigned int (Florian Westphal) [1936368]
- mptcp: sockopt: add TCP_CONGESTION and TCP_INFO (Florian Westphal) [1936368]
- mptcp: setsockopt: SO_DEBUG and no-op options (Florian Westphal) [1936368]
- mptcp: setsockopt: add SO_INCOMING_CPU (Florian Westphal) [1936368]
- mptcp: setsockopt: add SO_MARK support (Florian Westphal) [1936368]
- mptcp: setsockopt: support SO_LINGER (Florian Westphal) [1936368]
- mptcp: setsockopt: handle receive/send buffer and device bind (Florian Westphal) [1936368]
- mptcp: setsockopt: handle SO_KEEPALIVE and SO_PRIORITY (Florian Westphal) [1936368]
- mptcp: tag sequence_seq with socket state (Florian Westphal) [1936368]
- mptcp: add skeleton to sync msk socket options to subflows (Florian Westphal) [1936368]
- mptcp: only admit explicitly supported sockopt (Florian Westphal) [1936368]
- mptcp: move sockopt function into a new file (Florian Westphal) [1936368]
- mptcp: drop all sub-options except ADD_ADDR when the echo bit is set (Florian Westphal) [1936368]
- mptcp: unify add_addr(6)_generate_hmac (Florian Westphal) [1936368]
- mptcp: drop MPTCP_ADDR_IPVERSION_4/6 (Florian Westphal) [1936368]
- mptcp: use mptcp_addr_info in mptcp_options_received (Florian Westphal) [1936368]
- mptcp: drop OPTION_MPTCP_ADD_ADDR6 (Florian Westphal) [1936368]
- mptcp: use mptcp_addr_info in mptcp_out_options (Florian Westphal) [1936368]
- mptcp: move flags and ifindex out of mptcp_addr_info (Florian Westphal) [1936368]
- mptcp: add mptcp reset option support (Florian Westphal) [1936368]
- mptcp: remove unneeded check on first subflow (Florian Westphal) [1936368]
- mptcp: add active MPC mibs (Florian Westphal) [1936368]
- mptcp: add mib for token creation fallback (Florian Westphal) [1936368]
- mptcp: remove id 0 address (Florian Westphal) [1936368]
- mptcp: unify RM_ADDR and RM_SUBFLOW receiving (Florian Westphal) [1936368]
- mptcp: remove all subflows involving id 0 address (Florian Westphal) [1936368]
- mptcp: subflow.c: Fix a typo (Florian Westphal) [1936368]
- mptcp: rename mptcp_pm_nl_add_addr_send_ack (Florian Westphal) [1936368]
- mptcp: send ack for rm_addr (Florian Westphal) [1936368]
- mptcp: drop useless addr_signal clear (Florian Westphal) [1936368]
- mptcp: move to next addr when subflow creation fail (Florian Westphal) [1936368]
- mptcp: export lookup_anno_list_by_saddr (Florian Westphal) [1936368]
- mptcp: move to next addr when timeout (Florian Westphal) [1936368]
- mptcp: drop unused subflow in mptcp_pm_subflow_established (Florian Westphal) [1936368]
- mptcp: skip connecting the connected address (Florian Westphal) [1936368]
- mptcp: drop argument port from mptcp_pm_announce_addr (Florian Westphal) [1936368]
- mptcp: clean-up the rtx path (Florian Westphal) [1936368]
- mptcp: fix ADD_ADDR HMAC in case port is specified (Florian Westphal) [1936368]
- mptcp: remove a list of addrs when flushing (Florian Westphal) [1936368]
- mptcp: remove multi addresses and subflows in PM (Florian Westphal) [1936368]
- mptcp: remove multi subflows in PM (Florian Westphal) [1936368]
- mptcp: remove multi addresses in PM (Florian Westphal) [1936368]
- mptcp: add rm_list_rx in mptcp_pm_data (Florian Westphal) [1936368]
- mptcp: add rm_list in mptcp_options_received (Florian Westphal) [1936368]
- mptcp: add rm_list_tx in mptcp_pm_data (Florian Westphal) [1936368]
- mptcp: add rm_list in mptcp_out_options (Florian Westphal) [1936368]
- mptcp: fix bit MPTCP_PUSH_PENDING tests (Florian Westphal) [1936368]
- mptcp: fix length of ADD_ADDR with port sub-option (Florian Westphal) [1936368]
- mptcp: free resources when the port number is mismatched (Florian Westphal) [1936368]
- mptcp: fix missing wakeup (Florian Westphal) [1936368]
- mptcp: fix race in release_cb (Florian Westphal) [1936368]
- mptcp: factor out __mptcp_retrans helper() (Florian Westphal) [1936368]
- mptcp: reset 'first' and ack_hint on subflow close (Florian Westphal) [1936368]
- mptcp: dispose initial struct socket when its subflow is closed (Florian Westphal) [1936368]
- mptcp: fix memory accounting on allocation error (Florian Westphal) [1936368]
- mptcp: put subflow sock on connect error (Florian Westphal) [1936368]
- mptcp: reset last_snd on subflow close (Florian Westphal) [1936368]
- mptcp: do not wakeup listener for MPJ subflows (Florian Westphal) [1936368]
- mptcp: fix DATA_FIN generation on early shutdown (Florian Westphal) [1936368]
- mptcp: fix DATA_FIN processing for orphaned sockets (Florian Westphal) [1936368]
- mptcp: add local addr info in mptcp_info (Florian Westphal) [1936368]
- mptcp: add netlink event support (Florian Westphal) [1936368]
- mptcp: avoid lock_fast usage in accept path (Florian Westphal) [1936368]
- netlink: don't call ->netlink_bind with table lock held (Florian Westphal) [1936368]
- genetlink: restrict upcoming mptcp netlink events to CAP_NET_ADMIN (Florian Westphal) [1936368]
- mptcp: pass subflow socket to a few helpers (Florian Westphal) [1936368]
- mptcp: move subflow close loop after sk close check (Florian Westphal) [1936368]
- mptcp: schedule worker when subflow is closed (Florian Westphal) [1936368]
- mptcp: split __mptcp_close_ssk helper (Florian Westphal) [1936368]
- mptcp: move pm netlink work into pm_netlink (Florian Westphal) [1936368]
- mptcp: add a missing retransmission timer scheduling (Florian Westphal) [1936368]
- mptcp: better msk receive window updates (Florian Westphal) [1936368]
- mptcp: init mptcp request socket earlier (Florian Westphal) [1936368]
- mptcp: fix spurious retransmissions (Florian Westphal) [1936368]
- mptcp: fix poll after shutdown (Florian Westphal) [1936368]
- mptcp: deliver ssk errors to msk (Florian Westphal) [1936368]
- mptcp: pm: add lockdep assertions (Florian Westphal) [1936368]
- mptcp: add the mibs for ADD_ADDR with port (Florian Westphal) [1936368]
- mptcp: deal with MPTCP_PM_ADDR_ATTR_PORT in PM netlink (Florian Westphal) [1936368]
- mptcp: enable use_port when invoke addresses_equal (Florian Westphal) [1936368]
- mptcp: add port number check for MP_JOIN (Florian Westphal) [1936368]
- mptcp: add a new helper subflow_req_create_thmac (Florian Westphal) [1936368]
- mptcp: drop unused skb in subflow_token_join_request (Florian Westphal) [1936368]
- mptcp: create the listening socket for new port (Florian Westphal) [1936368]
- mptcp: create subflow or signal addr for newly added address (Florian Westphal) [1936368]
- mptcp: drop *_max fields in mptcp_pm_data (Florian Westphal) [1936368]
- mptcp: use WRITE_ONCE for the pernet *_max (Florian Westphal) [1936368]
- mptcp: pm nl: reduce variable scope (Florian Westphal) [1936368]
- mptcp: pm nl: support IPv4 mapped in v6 addresses (Florian Westphal) [1936368]
- mptcp: support MPJoin with IPv4 mapped in v6 sk (Florian Westphal) [1936368]
- mptcp: use sha256() instead of open coding (Florian Westphal) [1936368]
- crypto: lib/sha256 - add sha256() function (Florian Westphal) [1936368]
- mptcp: use SHA256_BLOCK_SIZE, not SHA_MESSAGE_BYTES (Florian Westphal) [1936368]
- xprtrdma: Pad optimization, revisited (Steve Dickson) [1926957]
- ibmvnic: remove set but not used variable 'netdev' (Diego Domingos) [1922193]
- net/ibmvnic: Remove tests of member address (Diego Domingos) [1922193]
- ibmvnic: remove duplicate napi_schedule call in do_reset function (Diego Domingos) [1922193]
- ibmvnic: avoid calling napi_disable() twice (Diego Domingos) [1922193]
- ibmvnic: queue reset work in system_long_wq (Diego Domingos) [1922193]
- ibmvnic: correctly use dev_consume/free_skb_irq (Diego Domingos) [1922193]
- ibmvnic: improve failover sysfs entry (Diego Domingos) [1922193]
- ibmvnic: print adapter state as a string (Diego Domingos) [1922193]
- ibmvnic: print reset reason as a string (Diego Domingos) [1922193]
- ibmvnic: clean up the remaining debugfs data structures (Diego Domingos) [1922193]
- ibmvnic: Use 'skb_frag_address()' instead of hand coding it (Diego Domingos) [1922193]
- ibmvnic: remove excessive irqsave (Diego Domingos) [1922193]
- ibmvnic: always store valid MAC address (Diego Domingos) [1922193]
- ibmvnic: Fix possibly uninitialized old_num_tx_queues variable warning. (Diego Domingos) [1922193]
- vio: make remove callback return void (Diego Domingos) [1922193]
- tty: hvcs: Drop unnecessary if block (Diego Domingos) [1922193]
- ibmvnic: serialize access to work queue on remove (Diego Domingos) [1922193]
- ibmvnic: skip send_request_unmap for timeout reset (Diego Domingos) [1922193]
- ibmvnic: add memory barrier to protect long term buffer (Diego Domingos) [1922193]
- ibmvnic: substitute mb() with dma_wmb() for send_*crq* functions (Diego Domingos) [1922193]
- ibmvnic: prefer strscpy over strlcpy (Diego Domingos) [1922193]
- ibmvnic: remove unused spinlock_t stats_lock definition (Diego Domingos) [1922193]
- ibmvnic: add comments for spinlock_t definitions (Diego Domingos) [1922193]
- ibmvnic: fix miscellaneous checks (Diego Domingos) [1922193]
- ibmvnic: avoid multiple line dereference (Diego Domingos) [1922193]
- ibmvnic: fix braces (Diego Domingos) [1922193]
- ibmvnic: fix block comments (Diego Domingos) [1922193]
- ibmvnic: prefer 'unsigned long' over 'unsigned long int' (Diego Domingos) [1922193]
- ibmvnic: remove unnecessary rmb() inside ibmvnic_poll (Diego Domingos) [1922193]
- ibmvnic: rework to ensure SCRQ entry reads are properly ordered (Diego Domingos) [1922193]
- ibmvnic: Ensure that CRQ entry read are correctly ordered (Diego Domingos) [1922193]
- net: ethernet: ibm: ibmvnic: Fix some kernel-doc misdemeanours (Diego Domingos) [1922193]
- powerpc/hmi: Fix kernel hang when TB is in error state. (Gustavo Walbon) [1924091]
- powerpc/eeh: Fix EEH handling for hugepages in ioremap space. (Gustavo Walbon) [1924090]
- powerpc/pseries: Only register vio drivers if vio bus exists (Gustavo Walbon) [1924090]
- powerpc/pci: Remove unimplemented prototypes (Gustavo Walbon) [1924090]
- powerpc/powernv/pci: Use kzalloc() for phb related allocations (Gustavo Walbon) [1924090]
- powerpc/pasemi: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/embedded6xx/mve5100: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/embedded6xx/mpc7448: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/embedded6xx/linkstation: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/embedded6xx/holly: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/chrp: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/amigaone: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/83xx: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/82xx/*: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/52xx/mpc5200_simple: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/52xx/media5200: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/52xx/lite5200: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/52xx/efika: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/512x: Move PHB discovery (Gustavo Walbon) [1924090]
- powerpc/pci: Move PHB discovery for PCI_DN using platforms (Gustavo Walbon) [1924090]
- powerpc/maple: Fix declaration made after definition (Gustavo Walbon) [1924090]
- powerpc/pci: Add ppc_md.discover_phbs() (Gustavo Walbon) [1924090]
- powerpc/pci: Delete traverse_pci_dn() (Gustavo Walbon) [1924090]
- powerpc/eeh: Rework pci_dev lookup in debugfs attributes (Gustavo Walbon) [1924090]
- selftests/powerpc: Add VF recovery tests (Gustavo Walbon) [1924090]
- powerpc/pseries/eeh: Make pseries_send_allow_unfreeze() static (Gustavo Walbon) [1924090]
- powerpc/pseries/eeh: Make pseries_pcibios_bus_add_device() static (Gustavo Walbon) [1924090]
- selftests/powerpc: Make the test check in eeh-basic.sh posix compliant (Gustavo Walbon) [1924090]
- powernv/pci: Print an error when device enable is blocked (Gustavo Walbon) [1924090]
- selftests/powerpc/eeh: disable kselftest timeout setting for eeh-basic (Gustavo Walbon) [1924090]
- powerpc/eeh_cache: Fix a possible debugfs deadlock (Gustavo Walbon) [1924090]
- powerpc/eeh: Fix eeh_dev_check_failure() for PE#0 (Gustavo Walbon) [1924090]
- selftests/powerpc: Fix eeh-basic.sh exit codes (Gustavo Walbon) [1924090]
- powerpc/pseries/eeh: Fix use of uninitialised variable (Gustavo Walbon) [1924090]
- powerpc/eeh: Delete eeh_pe->config_addr (Gustavo Walbon) [1924090]
- powerpc/eeh: Clean up PE addressing (Gustavo Walbon) [1924090]
- powerpc/pseries/eeh: Allow zero to be a valid PE configuration address (Gustavo Walbon) [1924090]
- powerpc/pseries/eeh: Rework device EEH PE determination (Gustavo Walbon) [1924090]
- powerpc/pseries/eeh: Clean up pe_config_addr lookups (Gustavo Walbon) [1924090]
- powerpc/eeh: Move EEH initialisation to an arch initcall (Gustavo Walbon) [1924090]
- powerpc/eeh: Delete eeh_ops->init (Gustavo Walbon) [1924090]
- powerpc/pseries: Stop using eeh_ops->init() (Gustavo Walbon) [1924090]
- powerpc/powernv: Stop using eeh_ops->init() (Gustavo Walbon) [1924090]
- powerpc/eeh: Rework EEH initialisation (Gustavo Walbon) [1924090]
- powerpc/pseries/eeh: Fix dumb linebreaks (Gustavo Walbon) [1924090]
- selftests/powerpc: Squash spurious errors due to device removal (Gustavo Walbon) [1924090]
- powerpc/eeh: Move PE tree setup into the platform (Gustavo Walbon) [1924090]
- powerpc/eeh: Drop pdn use in eeh_pe_tree_insert() (Gustavo Walbon) [1924090]
- powerpc/eeh: Rename eeh_{add_to|remove_from}_parent_pe() (Gustavo Walbon) [1924090]
- powerpc/eeh: Remove class code field from edev (Gustavo Walbon) [1924090]
- powerpc/eeh: Remove spurious use of pci_dn in eeh_dump_dev_log (Gustavo Walbon) [1924090]
- powerpc/eeh: Pass eeh_dev to eeh_ops->{read|write}_config() (Gustavo Walbon) [1924090]
- powerpc/eeh: Pass eeh_dev to eeh_ops->resume_notify() (Gustavo Walbon) [1924090]
- powerpc/eeh: Pass eeh_dev to eeh_ops->restore_config() (Gustavo Walbon) [1924090]
- powerpc/eeh: Remove VF config space restoration (Gustavo Walbon) [1924090]
- powerpc/eeh: Kill off eeh_ops->get_pe_addr() (Gustavo Walbon) [1924090]
- powerpc/pseries: Stop using pdn->pe_number (Gustavo Walbon) [1924090]
- powerpc/eeh: Move vf_index out of pci_dn and into eeh_dev (Gustavo Walbon) [1924090]
- powerpc/eeh: Remove eeh_dev.c (Gustavo Walbon) [1924090]
- powerpc/eeh: Remove eeh_dev_phb_init_dynamic() (Gustavo Walbon) [1924090]
- powerpc/powernv: Make pnv_pci_sriov_enable() and friends static (Gustavo Walbon) [1924090]
- powerpc/pseries: Make vio and ibmebus initcalls pseries specific (Gustavo Walbon) [1924090]
- powerpc/eeh: Release EEH device state synchronously (Gustavo Walbon) [1924090]
- powerpc/eeh: Rework eeh_ops->probe() (Gustavo Walbon) [1924090]
- powerpc/eeh: Make early EEH init pseries specific (Gustavo Walbon) [1924090]
- powerpc/eeh: Remove PHB check in probe (Gustavo Walbon) [1924090]
- powerpc/eeh: Do early EEH init only when required (Gustavo Walbon) [1924090]
- powerpc/eeh: Only dump stack once if an MMIO loop is detected (Gustavo Walbon) [1924090]
- powernv/pci: Add a debugfs entry to dump PHB's IODA PE state (Gustavo Walbon) [1924090]
- powernv/pci: Allow any write trigger the diag dump (Gustavo Walbon) [1924090]
- powernv/pci: Use pnv_phb as the private data for debugfs entries (Gustavo Walbon) [1924090]
- powerpc/powernv/npu: Fix debugfs_simple_attr.cocci warnings (Gustavo Walbon) [1924090]
- powerpc/pcidn: Warn when sriov pci_dn management is used incorrectly (Gustavo Walbon) [1924090]
- powerpc/pcidn: Make VF pci_dn management CONFIG_PCI_IOV specific (Gustavo Walbon) [1924090]
- powerpc/sriov: Remove VF eeh_dev state when disabling SR-IOV (Gustavo Walbon) [1924090]
- powerpc/eeh_sysfs: Make clearing EEH_DEV_SYSFS saner (Gustavo Walbon) [1924090]
- powerpc/eeh_sysfs: Remove double pci_dn lookup. (Gustavo Walbon) [1924090]
- powerpc/eeh_sysfs: ifdef pseries sr-iov sysfs properties (Gustavo Walbon) [1924090]
- powerpc/eeh_cache: Don't use pci_dn when inserting new ranges (Gustavo Walbon) [1924090]
- powerpc/eeh: differentiate duplicate detection message (Gustavo Walbon) [1924090]
- powerpc/eeh: Handle hugepages in ioremap space (Gustavo Walbon) [1924090]
- powerpc/powernv-eeh: Consisely desribe what this file does (Gustavo Walbon) [1924090]
- powerpc/eeh_cache: fix a W=1 kernel-doc warning (Gustavo Walbon) [1924090]
- powerpc/eeh_cache: Add a way to dump the EEH address cache (Gustavo Walbon) [1924090]
- powerpc/powernv: Escalate reset when IODA reset fails (Gustavo Walbon) [1924090]
- powerpc/powernv/eeh/npu: Fix uninitialized variables in opal_pci_eeh_freeze_status (Gustavo Walbon) [1924090]
- powerpc: eeh_event: convert semaphore to completion (Gustavo Walbon) [1924090]
- powerpc/eeh: Declare pci_ers_result_name() as static (Gustavo Walbon) [1924090]
- Bluetooth: verify AMP hci_chan before amp_destroy (Gopal Tiwari) [1962546] {CVE-2021-33034}
- NFSv4.2 fix handling of sr_eof in SEEK's reply (Steve Dickson) [1948015]
- SUNRPC: Fix null pointer dereference in svc_rqst_free() (Benjamin Coddington) [1952008]
- NFSv4: Use sequence counter with associated spinlock (Benjamin Coddington) [1952008]
- NFSv4.2: fix return value of _nfs4_get_security_label() (Benjamin Coddington) [1952008]
- NFSD: fix error handling in NFSv4.0 callbacks (Benjamin Coddington) [1952008]
- NFS: Don't gratuitously clear the inode cache when lookup failed (Benjamin Coddington) [1952008]
- NFS: Don't revalidate the directory permissions on a lookup failure (Benjamin Coddington) [1952008]
- nfsd: register pernet ops last, unregister first (Benjamin Coddington) [1952008]
- SUNRPC: Fix general protection fault in trace_rpc_xdr_overflow() (Benjamin Coddington) [1952008]
- lockd: fix access beyond unterminated strings in prints (Benjamin Coddington) [1952008]
- tty: Protect disc_data in n_tty_close and n_tty_flush_buffer (Rafael Aquini) [1879861]
- hwmon: (amd_energy) Add AMD family 19h model 30h x86 match (Terry Bowman) [1949280]

* Mon Jun 28 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-319.el8]
- redhat/configs: Re-enable dptf_power module (Prarit Bhargava) [1962349]
- net/sched: act_ct: Offload connections with commit action (Marcelo Ricardo Leitner) [1965817]
- netfilter: flowtable: Remove redundant hw refresh bit (Marcelo Ricardo Leitner) [1965817]
- inet_diag: add support for tw_mark (Xin Long) [1972967]
- mm: thp: fix MADV_REMOVE deadlock on shmem THP (Waiman Long) [1974659] {CVE-2020-29368}
- mm: thp: make the THP mapcount atomic against __split_huge_pmd_locked() (Waiman Long) [1974659] {CVE-2020-29368}
- mmap locking API: add mmap_lock_is_contended() (Waiman Long) [1970177]
- mm: madvise: fix vma user-after-free (Waiman Long) [1970177]
- khugepaged: collapse_pte_mapped_thp() protect the pmd lock (Waiman Long) [1970177]
- khugepaged: collapse_pte_mapped_thp() flush the right range (Waiman Long) [1970177]
- mm/mmap.c: close race between munmap() and expand_upwards()/downwards() (Waiman Long) [1970177]
- mm: document warning in move_normal_pmd() and make it warn only once (Waiman Long) [1970177]
- mmap locking API: convert mmap_sem comments (Waiman Long) [1970177]
- mmap locking API: convert mmap_sem API comments (Waiman Long) [1970177]
- mmap locking API: rename mmap_sem to mmap_lock (Waiman Long) [1970177]
- mmap locking API: add mmap_assert_locked() and mmap_assert_write_locked() (Waiman Long) [1970177]
- mmap locking API: add MMAP_LOCK_INITIALIZER (Waiman Long) [1970177]
- mmap locking API: convert nested write lock sites (Waiman Long) [1970177]
- mmap locking API: convert mmap_sem call sites missed by coccinelle (Waiman Long) [1970177]
- mmap locking API: use coccinelle to convert mmap_sem rwsem call sites (Waiman Long) [1970177]
- MMU notifier: use the new mmap locking API (Waiman Long) [1970177]
- mm/gup.c: further document vma_permits_fault() (Waiman Long) [1970177]
- mm/gup.c: update the documentation (Waiman Long) [1970177]
- mm/ksm: fix NULL pointer dereference when KSM zero page is enabled (Waiman Long) [1970177]
- proc: inline vma_stop into m_stop (Waiman Long) [1970177]
- mm: remove CONFIG_TRANSPARENT_HUGE_PAGECACHE (Waiman Long) [1970177]
- fs/io_uring: set FOLL_PIN via pin_user_pages() (Waiman Long) [1970177]
- shmem: pin the file in shmem_fault() if mmap_sem is dropped (Waiman Long) [1970177]
- mm/gup.c: fix comments of __get_user_pages() and get_user_pages_remote() (Waiman Long) [1970177]
- mm/init-mm.c: include <linux/mman.h> for vm_committed_as_batch (Waiman Long) [1970177]
- lib/generic-radix-tree.c: add kmemleak annotations (Waiman Long) [1970177]
- mm: Remove BUG_ON mmap_sem not held from xxx_trans_huge_lock() (Waiman Long) [1970177]
- lib/generic-radix-tree.c: make 2 functions static inline (Waiman Long) [1970177]
- x86/mm: Clean up the pmd_read_atomic() comments (Waiman Long) [1970177]
- x86/mm: Fix function name typo in pmd_read_atomic() comment (Waiman Long) [1970177]
- khugepaged: enable collapse pmd for pte-mapped THP (Waiman Long) [1970177]
- mm: Handle MADV_WILLNEED through vfs_fadvise() (Waiman Long) [1970177]
- pagewalk: use lockdep_assert_held for locking validation (Waiman Long) [1970177]
- mm: use down_read_killable for locking mmap_sem in access_remote_vm (Waiman Long) [1970177]
- proc: use down_read_killable mmap_sem for /proc/pid/map_files (Waiman Long) [1970177]
- proc: use down_read_killable mmap_sem for /proc/pid/clear_refs (Waiman Long) [1970177]
- proc: use down_read_killable mmap_sem for /proc/pid/pagemap (Waiman Long) [1970177]
- proc: use down_read_killable mmap_sem for /proc/pid/smaps_rollup (Waiman Long) [1970177]
- proc: use down_read_killable mmap_sem for /proc/pid/maps (Waiman Long) [1970177]
- mm/filemap.c: correct the comment about VM_FAULT_RETRY (Waiman Long) [1970177]
- mm: mmu_gather: remove __tlb_reset_range() for force flush (Waiman Long) [1970177]
- arm64/mm: Drop mmap_sem before calling __do_kernel_fault() (Waiman Long) [1970177]
- docs/vm: Minor editorial changes in the THP and hugetlbfs (Waiman Long) [1970177]
- x86/mpx, mm/core: Fix recursive munmap() corruption (Waiman Long) [1970177]
- doc: mm: migration doesn't use FOLL_SPLIT anymore (Waiman Long) [1970177]
- filemap: add a comment about FAULT_FLAG_RETRY_NOWAIT behavior (Waiman Long) [1970177]
- proc: commit to genradix (Waiman Long) [1970177]
- generic radix trees (Waiman Long) [1970177]
- mm: swap: add comment for swap_vma_readahead (Waiman Long) [1970177]
- x86/mm: Make set_pmd_at() paravirt aware (Waiman Long) [1970177]
- mm: select HAVE_MOVE_PMD on x86 for faster mremap (Waiman Long) [1970177]
- mm: speed up mremap by 20x on large regions (Waiman Long) [1970177]
- mm/mmap.c: remove verify_mm_writelocked() (Waiman Long) [1970177]
- x86/fault: Check user_mode(regs) when avoiding an mmap_sem deadlock (Waiman Long) [1970177]
- mm: brk: downgrade mmap_sem to read when shrinking (Waiman Long) [1970177]
- mm: mremap: downgrade mmap_sem to read when shrinking (Waiman Long) [1970177]
- mm: unmap VM_PFNMAP mappings with optimized path (Waiman Long) [1970177]
- mm: unmap VM_HUGETLB mappings with optimized path (Waiman Long) [1970177]
- mm: mmap: zap pages with read mmap_sem in munmap (Waiman Long) [1970177]
- x86/mm: Fix exception table comments (Waiman Long) [1970177]
- ACPI: x86: Call acpi_boot_table_init() after acpi_table_upgrade() (Mark Langsdorf) [1946282]
- acpi/drivers/thermal: Remove TRIPS_NONE cooling device binding (Mark Langsdorf) [1946282]
- thermal/drivers/acpi: Use hot and critical ops (Mark Langsdorf) [1946282]
- thermal/core: Add critical and hot ops (Mark Langsdorf) [1946282]
- thermal/core: Emit a warning if the thermal zone is updated without ops (Mark Langsdorf) [1946282]
- ACPI: scan: Fix _STA getting called on devices with unmet dependencies (Mark Langsdorf) [1946282]
- ACPI: tables: x86: Reserve memory occupied by ACPI tables (Mark Langsdorf) [1946282]
- ACPICA: Always create namespace nodes using acpi_ns_create_node() (Mark Langsdorf) [1946282]
- ACPI: platform: Add balanced-performance platform profile (Mark Langsdorf) [1946282]
- ACPI: platform: Hide ACPI_PLATFORM_PROFILE option (Mark Langsdorf) [1946282]
- ACPICA: Remove some code duplication from acpi_ev_address_space_dispatch (Mark Langsdorf) [1946282]
- ACPICA: Fix race in generic_serial_bus (I2C) and GPIO op_region parameter handling (Mark Langsdorf) [1946282]
- ACPI: property: Satisfy kernel doc validator (part 2) (Mark Langsdorf) [1946282]
- ACPI: property: Satisfy kernel doc validator (part 1) (Mark Langsdorf) [1946282]
- ACPI: property: Make acpi_node_prop_read() static (Mark Langsdorf) [1946282]
- ACPI: property: Remove dead code (Mark Langsdorf) [1946282]
- iommu/amd: Fix performance counter initialization (Mark Langsdorf) [1946282]
- ACPI: property: Fix fwnode string properties matching (Mark Langsdorf) [1946282]
- ACPI / NUMA: add a stub function for node_to_pxm() (Mark Langsdorf) [1946282]
- ACPI: OSL: Rework acpi_check_resource_conflict() (Mark Langsdorf) [1946282]
- ACPI: APEI: ERST: remove unneeded semicolon (Mark Langsdorf) [1946282]
- iommu/vt-d: Add new enum value and structure for SATC (Mark Langsdorf) [1946282]
- ACPI: Add support for native USB4 control _OSC (Mark Langsdorf) [1946282]
- ACPI: Execute platform _OSC also with query bit clear (Mark Langsdorf) [1946282]
- redhat/configs: Add CONFIG_ACPI_FPDT (Mark Langsdorf) [1946282]
- ACPI: tables: introduce support for FPDT table (Mark Langsdorf) [1946282]
- ACPI: APEI: Add is_generic_error() to identify GHES sources (Mark Langsdorf) [1946282]
- ACPI: platform-profile: Fix possible deadlock in platform_profile_remove() (Mark Langsdorf) [1946282]
- ACPI: platform-profile: Introduce object pointers to callbacks (Mark Langsdorf) [1946282]
- PCI/ACPI: Clarify message about _OSC failure (Mark Langsdorf) [1946282]
- PCI/ACPI: Remove unnecessary osc_lock (Mark Langsdorf) [1946282]
- PCI/ACPI: Make acpi_pci_osc_control_set() static (Mark Langsdorf) [1946282]
- media: ACPI / bus: Add acpi_dev_get_next_match_dev() and helper macro (Mark Langsdorf) [1946282]
- ACPI: scan: Adjust white space in acpi_device_add() (Mark Langsdorf) [1946282]
- ACPI: scan: Rearrange memory allocation in acpi_device_add() (Mark Langsdorf) [1946282]
- ACPI: platform-profile: Drop const qualifier for cur_profile (Mark Langsdorf) [1946282]
- ACPI: configfs: add missing check after configfs_register_default_group() (Mark Langsdorf) [1946282]
- ACPI: CPPC: initialise vaddr pointers to NULL (Mark Langsdorf) [1946282]
- ACPI: CPPC: add __iomem annotation to generic_comm_base pointer (Mark Langsdorf) [1946282]
- ACPI: CPPC: remove __iomem annotation for cpc_reg's address (Mark Langsdorf) [1946282]
- ACPI: Use DEVICE_ATTR_<RW|RO|WO> macros (Mark Langsdorf) [1946282]
- ACPICA: Update version to 20210105 (Mark Langsdorf) [1946282]
- ACPICA: Updated all copyrights to 2021 (Mark Langsdorf) [1946282]
- ACPICA: Remove the VRTC table (Mark Langsdorf) [1946282]
- ACPICA: Remove the MTMR (Mid-Timer) table (Mark Langsdorf) [1946282]
- ACPICA: Update version to 20201217 (Mark Langsdorf) [1946282]
- ACPICA: add type casts for string functions (Mark Langsdorf) [1946282]
- ACPICA: fix -Wfallthrough (Mark Langsdorf) [1946282]
- ACPICA: Clean up exception code class checks (Mark Langsdorf) [1946282]
- ACPICA: Fix exception code class checks (Mark Langsdorf) [1946282]
- redhat/configs: Add CONFIG_ACPI_PLATFORM_PROFILE (Mark Langsdorf) [1946282]
- ACPI: platform: Add platform profile support (Mark Langsdorf) [1946282]
- NTB/msi: Use irq_has_action() (Myron Stowe) [1933153]
- NTB: Use struct_size() helper in devm_kzalloc() (Myron Stowe) [1933153]
- ntb: intel: Fix memleak in intel_ntb_pci_probe (Myron Stowe) [1933153]
- NTB: hw: amd: fix an issue about leak system resources (Myron Stowe) [1933153]
- ntb: hw: remove the code that sets the DMA mask (Myron Stowe) [1933153]
- [s390] s390/qdio: remove 'merge_pending' mechanism (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: improve handling of PENDING buffers for QEBSM devices (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: rework q->qdio_error indication (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: inline qdio_kick_handler() (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: track time of last data IRQ for each device (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: make thinint registration symmetric (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: adopt new tasklet API (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: remove qdio_inbound_q_moved() wrapper (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: remove Input tasklet code (Claudio Imbrenda) [1919251]
- [s390] scsi: zfcp: Do not set COMMAND_COMPLETE (Claudio Imbrenda) [1919251]
- [s390] scsi: zfcp: Handle event-lost notification for Version Change events (Claudio Imbrenda) [1919251]
- [s390] scsi: zfcp: Process Version Change events (Claudio Imbrenda) [1919251]
- [s390] scsi: zfcp: Clarify & assert the stat_lock locking in zfcp_qdio_send() (Claudio Imbrenda) [1919251]
- [s390] scsi: zfcp: Remove orphaned function declarations (Claudio Imbrenda) [1919251]
- [s390] scsi: zfcp: Lift Input Queue tasklet from qdio (Claudio Imbrenda) [1919251]
- [s390] scsi: zfcp: Clarify access to erp_action in zfcp_fsf_req_complete() (Claudio Imbrenda) [1919251]
- [s390] scsi: zfcp: Use list_first_entry_or_null() in zfcp_erp_thread() (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: always use dev_name() for device name in QIB (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: clean up QDR setup (Claudio Imbrenda) [1919251]
- [s390] s390/qdio: make qdio_handle_aobs() more robust (Claudio Imbrenda) [1919251]
- [s390] s390/cpumf: disable preemption when accessing per-cpu variable (Claudio Imbrenda) [1925527]
- [s390] s390/cpumf: rename header file to hwctrset.h (Claudio Imbrenda) [1925527]
- [s390] s390/cpumf: remove 60 seconds read limit (Claudio Imbrenda) [1925527]
- [s390] s390/cpumf: Add support for complete counter set extraction (Claudio Imbrenda) [1925527]

* Fri Jun 25 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-318.el8]
- KVM: X86: hyper-v: Task srcu lock when accessing kvm_memslots() (Vitaly Kuznetsov) [1904570]
- KVM: nSVM: remove a warning about vmcb01 VM exit reason (Vitaly Kuznetsov) [1904570]
- KVM: x86: Consolidate guest enter/exit logic to common helpers (Vitaly Kuznetsov) [1904570]
- KVM: x86: Defer vtime accounting 'til after IRQ handling (Vitaly Kuznetsov) [1904570]
- context_tracking: Move guest exit vtime accounting to separate helpers (Vitaly Kuznetsov) [1904570]
- context_tracking: Move guest exit context tracking to separate helpers (Vitaly Kuznetsov) [1904570]
- context_tracking: Make guest_enter/exit() .noinstr ready (Vitaly Kuznetsov) [1904570]
- sched/cputime: Rename vtime_account_system() to vtime_account_kernel() (Vitaly Kuznetsov) [1904570]
- x86/kvm/svm: Move guest enter/exit into .noinstr.text (Vitaly Kuznetsov) [1904570]
- x86/kvm/vmx: Move guest enter/exit into .noinstr.text (Vitaly Kuznetsov) [1904570]
- x86/kvm/svm: Add hardirq tracing on guest enter/exit (Vitaly Kuznetsov) [1904570]
- x86/kvm/vmx: Add hardirq tracing to guest enter/exit (Vitaly Kuznetsov) [1904570]
- x86/kvm: Move context tracking where it belongs (Vitaly Kuznetsov) [1904570]
- Move some more variables to kernel job templates (Michael Hofmann)
- Use a different branch name for realtime pipelines (Michael Hofmann)
- block: mark queue init done at the end of blk_register_queue (Ming Lei) [1963766]
- block: fix race between adding/removing rq qos and normal IO (Ming Lei) [1963766]
- netfilter: x_tables: Use correct memory barriers. (Phil Sutter) [1949090]
- Revert "netfilter: x_tables: Switch synchronization to RCU" (Phil Sutter) [1949090]
- vfio/pci/nvlink2: Do not attempt NPU2 setup on POWER8NVL NPU (Gustavo Walbon) [1891589]
- powerpc/powernv/npu: Do not attempt NPU2 setup on POWER8NVL NPU (Gustavo Walbon) [1891589]
- cpupower: Add cpuid cap flag for MSR_AMD_HWCR support (Steve Best) [1949161]
- cpupower: Remove family arg to decode_pstates() (Steve Best) [1949161]
- cpupower: Condense pstate enabled bit checks in decode_pstates() (Steve Best) [1949161]
- cpupower: Update family checks when decoding HW pstates (Steve Best) [1949161]
- cpupower: Remove unused pscur variable. (Steve Best) [1949161]
- cpupower: Add CPUPOWER_CAP_AMD_HW_PSTATE cpuid caps flag (Steve Best) [1949161]
- cpupower: Correct macro name for CPB caps flag (Steve Best) [1949161]
- cpupower: Update msr_pstate union struct naming (Steve Best) [1949161]
- cpupower: add Makefile dependencies for install targets (Steve Best) [1949161]
- tools/power/cpupower: Read energy_perf_bias from sysfs (Steve Best) [1949161]
- tools: Avoid comma separated statements (Steve Best) [1949161]
- cpupower: speed up generating git version string (Steve Best) [1949161]
- cpupowerutils: fix spelling mistake "dependant" -> "dependent" (Steve Best) [1949161]
- ext4: fix ext4_error_err save negative errno into superblock (Lukas Czerner) [1964928]
- ext4: do not set SB_ACTIVE in ext4_orphan_cleanup() (Lukas Czerner) [1964928]
- ext4: fix check to prevent false positive report of incorrect used inodes (Lukas Czerner) [1964928]
- ext4: fix potential error in ext4_do_update_inode (Lukas Czerner) [1964928]
- ext4: do not try to set xattr into ea_inode if value is empty (Lukas Czerner) [1964928]
- ext4: fix potential htree index checksum corruption (Lukas Czerner) [1964928]
- ext4: don't leak old mountpoint samples (Lukas Czerner) [1964928]
- ext4: don't remount read-only with errors=continue on reboot (Lukas Czerner) [1964928]
- ext4: fix deadlock with fs freezing and EA inodes (Lukas Czerner) [1964928]
- ext4: fix a memory leak of ext4_free_data (Lukas Czerner) [1964928]
- ext4: find old entry again if failed to rename whiteout (Lukas Czerner) [1945503]
- ext4: fix bug for rename with RENAME_WHITEOUT (Lukas Czerner) [1945503]
- mount: fix mounting of detached mounts onto targets that reside on shared mounts (Carlos Maiolino) [1959172]
- fs: fix lazytime expiration handling in __writeback_single_inode() (Carlos Maiolino) [1936260]
- tty: Don't hold ldisc lock in tty_reopen() if ldisc present (Waiman Long) [1968271]
- tty/ldsem: Add lockdep asserts for ldisc_sem (Waiman Long) [1968271]
- tty: Simplify tty->count math in tty_reopen() (Waiman Long) [1968271]
- tty: Don't block on IO when ldisc change is pending (Waiman Long) [1968271]
- tty: Hold tty_ldisc_lock() during tty_reopen() (Waiman Long) [1968271]
- tty: Drop tty->count on tty_reopen() failure (Waiman Long) [1968271]
- net: udp: Add support for getsockopt(..., ..., UDP_GRO, ..., ...); (Xin Long) [1951679]
- net: Fix gro aggregation for udp encaps with zero csum (Xin Long) [1951679]
- udp: ipv4: manipulate network header of NATed UDP GRO fraglist (Xin Long) [1951679]
- udp: not remove the CRC flag from dev features when need_csum is false (Xin Long) [1951679]
- udp: mask TOS bits in udp_v4_early_demux() (Xin Long) [1951679]
- net: fix use-after-free when UDP GRO with shared fraglist (Xin Long) [1951679]
- x86, sched: Treat Intel SNC topology as default, COD as exception (David Arcari) [1920040]
- dmaengine: idxd: iax bus removal (Jerry Snitselaar) [1955744]
- dmaengine: idxd: fix cdev setup and free device lifetime issues (Jerry Snitselaar) [1955744]
- dmaengine: idxd: fix group conf_dev lifetime (Jerry Snitselaar) [1955744]
- dmaengine: idxd: fix engine conf_dev lifetime (Jerry Snitselaar) [1955744]
- dmaengine: idxd: fix wq conf_dev 'struct device' lifetime (Jerry Snitselaar) [1955744]
- dmaengine: idxd: fix idxd conf_dev 'struct device' lifetime (Jerry Snitselaar) [1955744]
- dmaengine: idxd: use ida for device instance enumeration (Jerry Snitselaar) [1955744]
- dmaengine: idxd: removal of pcim managed mmio mapping (Jerry Snitselaar) [1955744]
- dmaengine: idxd: cleanup pci interrupt vector allocation management (Jerry Snitselaar) [1955744]
- dmaengine: idxd: fix dma device lifetime (Jerry Snitselaar) [1955744]
- dmaengine: idxd: Fix potential null dereference on pointer status (Jerry Snitselaar) [1927070]
- dmaengine: idxd: fix wq cleanup of WQCFG registers (Jerry Snitselaar) [1955744]
- dmaengine: idxd: clear MSIX permission entry on shutdown (Jerry Snitselaar) [1955744]
- dmaengine: idxd: fix wq size store permission state (Jerry Snitselaar) [1955744]
- dmaengine: idxd: fix opcap sysfs attribute output (Jerry Snitselaar) [1955744]
- dmaengine: idxd: Fix clobbering of SWERR overflow bit on writeback (Jerry Snitselaar) [1955744]
- dmaengine: idxd: add module parameter to force disable of SVA (Jerry Snitselaar) [1920734]
- dmaengine: idxd: set DMA channel to be private (Jerry Snitselaar) [1920759]
- dma: idxd: use DEFINE_MUTEX() for mutex lock (Jerry Snitselaar) [1955744]
- dmaengine: idxd: check device state before issue command (Jerry Snitselaar) [1927070]
- dmaengine: move channel device_node deletion to driver (Jerry Snitselaar) [1920721]
- dmaengine: idxd: fix misc interrupt completion (Jerry Snitselaar) [1920757]
- dmaengine: idxd: Fix list corruption in description completion (Jerry Snitselaar) [1920746]
- dmaengine: idxd: add IAX configuration support in the IDXD driver (Jerry Snitselaar) [1837233]
- dmaengine: idxd: add ATS disable knob for work queues (Jerry Snitselaar) [1921291]
- dmaengine: idxd: define table offset multiplier (Jerry Snitselaar) [1921290]
- dmaengine: idxd: Update calculation of group offset to be more readable (Jerry Snitselaar) [1955744]

* Wed Jun 23 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-317.el8]
- redhat/config: Double MAX_LOCKDEP_ENTRIES (Waiman Long) [1945937]
- dm writecache: flush origin device when writing and cache is full (Mike Snitzer) [1970513]
- dm snapshot: properly fix a crash when an origin has no snapshots (Mike Snitzer) [1970513]
- dm verity: fix require_signatures module_param permissions (Mike Snitzer) [1970513]
- dm snapshot: fix crash with transient storage and zero chunk size (Mike Snitzer) [1970513]
- dm rq: fix double free of blk_mq_tag_set in dev remove after table load fails (Mike Snitzer) [1970513]
- dm integrity: increase RECALC_SECTORS to improve recalculate speed (Mike Snitzer) [1970513]
- dm integrity: don't re-write metadata if discarding same blocks (Mike Snitzer) [1970513]
- dm raid: fix inconclusive reshape layout on fast raid4/5/6 table reload sequences (Mike Snitzer) [1970513]
- dm raid: fix fall-through warning in rs_check_takeover() for Clang (Mike Snitzer) [1970513]
- dm integrity: fix missing goto in bitmap_flush_interval error handling (Mike Snitzer) [1970513]
- dm space map common: fix division bug in sm_ll_find_free_block() (Mike Snitzer) [1970513]
- dm persistent data: packed struct should have an aligned() attribute too (Mike Snitzer) [1970513]
- dm btree spine: remove paranoid node_check call in node_prep_for_write() (Mike Snitzer) [1970513]
- dm space map disk: remove redundant calls to sm_disk_get_nr_free() (Mike Snitzer) [1970513]
- dm persistent data: remove unused return from exit_shadow_spine() (Mike Snitzer) [1970513]
- dm verity: allow only one error handling mode (Mike Snitzer) [1970513]
- dm: remove useless loop in __split_and_process_bio (Mike Snitzer) [1970513]
- redhat/configs: Update CONFIG_PINCTRL_AMD (David Arcari) [1964588]
- ionic: fix ptp support config breakage (Jonathan Toppins) [1951826]
- ionic: return -EFAULT if copy_to_user() fails (Jonathan Toppins) [1951826]
- ionic: git_ts_info bit shifters (Jonathan Toppins) [1951826]
- ionic: extend ts_config set locking (Jonathan Toppins) [1951826]
- ionic: add ts_config replay (Jonathan Toppins) [1951826]
- ionic: ignore EBUSY on queue start (Jonathan Toppins) [1951826]
- ionic: re-start ptp after queues up (Jonathan Toppins) [1951826]
- ionic: add SKBTX_IN_PROGRESS (Jonathan Toppins) [1951826]
- ionic: check for valid tx_mode on SKBTX_HW_TSTAMP xmit (Jonathan Toppins) [1951826]
- ionic: remove unnecessary compat ifdef (Jonathan Toppins) [1951826]
- ionic: fix up a couple of code style nits (Jonathan Toppins) [1951826]
- ionic: advertise support for hardware timestamps (Jonathan Toppins) [1951826]
- ionic: ethtool ptp stats (Jonathan Toppins) [1951826]
- ionic: add ethtool support for PTP (Jonathan Toppins) [1951826]
- ionic: add and enable tx and rx timestamp handling (Jonathan Toppins) [1951826]
- ethtool: add timestamping related string sets (Jonathan Toppins) [1951826]
- net: Introduce peer to peer one step PTP time stamping. (Jonathan Toppins) [1951826]
- ionic: set up hw timestamp queues (Jonathan Toppins) [1951826]
- ionic: add rx filtering for hw timestamp steering (Jonathan Toppins) [1951826]
- ionic: link in the new hw timestamp code (Jonathan Toppins) [1951826]
- ionic: add hw timestamp support files (Jonathan Toppins) [1951826]
- ionic: split adminq post and wait calls (Jonathan Toppins) [1951826]
- ionic: add hw timestamp structs to interface (Jonathan Toppins) [1951826]
- ionic: add handling of larger descriptors (Jonathan Toppins) [1951826]
- ionic: add new queue features to interface (Jonathan Toppins) [1951826]
- ionic: pull per-q stats work out of queue loops (Jonathan Toppins) [1951826]
- ionic: avoid races in ionic_heartbeat_check (Jonathan Toppins) [1951826]
- ionic: fix sizeof usage (Jonathan Toppins) [1951826]
- ionic: count dma errors (Jonathan Toppins) [1951826]
- ionic: protect adminq from early destroy (Jonathan Toppins) [1951826]
- ionic: stop watchdog when in broken state (Jonathan Toppins) [1951826]
- ionic: block actions during fw reset (Jonathan Toppins) [1951826]
- ionic: update ethtool support bits for BASET (Jonathan Toppins) [1951826]
- ionic: fix unchecked reference (Jonathan Toppins) [1951826]
- ionic: simplify the intr_index use in txq_init (Jonathan Toppins) [1951826]
- ionic: code cleanup details (Jonathan Toppins) [1951826]
- ionic: Update driver to use ethtool_sprintf (Jonathan Toppins) [1951826]
- ionic: aggregate Tx byte counting calls (Jonathan Toppins) [1951826]
- ionic: simplify tx clean (Jonathan Toppins) [1951826]
- ionic: generic tx skb mapping (Jonathan Toppins) [1951826]
- ionic: simplify TSO descriptor mapping (Jonathan Toppins) [1951826]
- ionic: simplify use of completion types (Jonathan Toppins) [1951826]
- ionic: rebuild debugfs on qcq swap (Jonathan Toppins) [1951826]
- ionic: simplify rx skb alloc (Jonathan Toppins) [1951826]
- ionic: optimize fastpath struct usage (Jonathan Toppins) [1951826]
- ionic: implement Rx page reuse (Jonathan Toppins) [1951826]
- ionic: move rx_page_alloc and free (Jonathan Toppins) [1951826]
- drivers: net: ionic: simplify the return expression of ionic_set_rxfh() (Jonathan Toppins) [1951826]
- ionic: change mtu after queues are stopped (Jonathan Toppins) [1951826]
- ionic: remove some unnecessary oom messages (Jonathan Toppins) [1951826]
- ionic: useful names for booleans (Jonathan Toppins) [1951826]
- ionic: change set_rx_mode from_ndo to can_sleep (Jonathan Toppins) [1951826]
- ionic: flatten calls to ionic_lif_rx_mode (Jonathan Toppins) [1951826]
- ionic: batch rx buffer refilling (Jonathan Toppins) [1951826]
- ionic: add lif quiesce (Jonathan Toppins) [1951826]
- ionic: check for link after netdev registration (Jonathan Toppins) [1951826]
- ionic: check port ptr before use (Jonathan Toppins) [1951826]
- platform/x86: hp-wireless: add AMD's hardware id to the supported list (David Arcari) [1960472]
- treewide: Replace GPLv2 boilerplate/reference with SPDX - rule 1 (David Arcari) [1960472]
- net/bond: revert deletion of driver and module versions (Jarod Wilson) [1955702]
- net:sfc: fix non-freed irq in legacy irq mode (Íñigo Huguet) [1906728]
- devlink: Extend SF port attributes to have external attribute (Petr Oros) [1959002]
- netfilter: xt_SECMARK: add new revision to fix structure layout (Phil Sutter) [1961714]
- xfrm: ipcomp: remove unnecessary get_cpu() (Sabrina Dubroca) [1946885]
- platform/x86: intel_pmc_core: Add support for Alder Lake PCH-P (David Arcari) [1838624]
- scsi: qedf: Update the max_id value in host structure (Nilesh Javali) [1954876]
- scsi: qedf: Add pointer checks in qedf_update_link_speed() (Nilesh Javali) [1871687]
- [s390] s390/nvme: support firmware-assisted dump to NVMe disks (Claudio Imbrenda) [1847447]
- [s390] s390: nvme dump support (Claudio Imbrenda) [1847447]
- [s390] s390/boot: add build-id to decompressor (Philipp Rudo) [1897093]
- genirq: Reduce irqdebug cacheline bouncing (Gustavo Walbon) [1957729]

* Mon Jun 21 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-316.el8]
- CI: Merge MR, baseline and private configs (Michael Hofmann)
- RDMA/rxe: Fix failure during driver load (Kamal Heib) [1930263]
- xfrm: remove the fragment check for ipv6 beet mode (Xin Long) [1965185]
- SUNRPC: Fix NFS READs that start at non-page-aligned offsets (Benjamin Coddington) [1950303]
- pNFS/NFSv4: Improve rejection of out-of-order layouts (Benjamin Coddington) [1950303]
- pNFS/NFSv4: Update the layout barrier when we schedule a layoutreturn (Benjamin Coddington) [1950303]
- pNFS/NFSv4: Try to return invalid layout in pnfs_layout_process() (Benjamin Coddington) [1950303]
- SUNRPC: Move the svc_xdr_recvfrom tracepoint again (Benjamin Coddington) [1950303]
- nfsd4: readdirplus shouldn't return parent of export (Benjamin Coddington) [1950303]
- NFS: nfs_igrab_and_active must first reference the superblock (Benjamin Coddington) [1950303]
- NFS: nfs_delegation_find_inode_server must first reference the superblock (Benjamin Coddington) [1950303]
- NFS/pNFS: Fix a leak of the layout 'plh_outstanding' counter (Benjamin Coddington) [1950303]
- NFS/pNFS: Don't leak DS commits in pnfs_generic_retry_commit() (Benjamin Coddington) [1950303]
- NFS/pNFS: Don't call pnfs_free_bucket_lseg() before removing the request (Benjamin Coddington) [1950303]
- pNFS: Stricter ordering of layoutget and layoutreturn (Benjamin Coddington) [1950303]
- pNFS: Clean up pnfs_layoutreturn_free_lsegs() (Benjamin Coddington) [1950303]
- pNFS: We want return-on-close to complete when evicting the inode (Benjamin Coddington) [1950303]
- pNFS: Mark layout for return if return-on-close was not sent (Benjamin Coddington) [1950303]
- net: sunrpc: interpret the return value of kstrtou32 correctly (Benjamin Coddington) [1950303]
- SUNRPC: Handle TCP socket sends with kernel_sendpage() again (Benjamin Coddington) [1950303]
- NFS/pNFS: Fix a typo in ff_layout_resend_pnfs_read() (Benjamin Coddington) [1950303]
- pNFS/flexfiles: Avoid spurious layout returns in ff_layout_choose_ds_for_read (Benjamin Coddington) [1950303]
- NFSv4/pnfs: Add tracing for the deviceid cache (Benjamin Coddington) [1950303]
- fs/lockd: convert comma to semicolon (Benjamin Coddington) [1950303]
- SUNRPC: When expanding the buffer, we may need grow the sparse pages (Benjamin Coddington) [1950303]
- SUNRPC: Cleanup - constify a number of xdr_buf helpers (Benjamin Coddington) [1950303]
- SUNRPC: Clean up open coded setting of the xdr_stream 'nwords' field (Benjamin Coddington) [1950303]
- SUNRPC: _copy_to/from_pages() now check for zero length (Benjamin Coddington) [1950303]
- SUNRPC: Cleanup xdr_shrink_bufhead() (Benjamin Coddington) [1950303]
- SUNRPC: Fix xdr_expand_hole() (Benjamin Coddington) [1950303]
- SUNRPC: Fixes for xdr_align_data() (Benjamin Coddington) [1950303]
- SUNRPC: _shift_data_left/right_pages should check the shift length (Benjamin Coddington) [1950303]
- NFSv4.1: use BITS_PER_LONG macro in nfs4session.h (Benjamin Coddington) [1950303]
- xprtrdma: Fix XDRBUF_SPARSE_PAGES support (Benjamin Coddington) [1950303]
- sunrpc: fix xs_read_xdr_buf for partial pages receive (Benjamin Coddington) [1950303]
- Revert "nfsd4: support change_attr_type attribute" (Benjamin Coddington) [1950303]
- nfsd4: don't query change attribute in v2/v3 case (Benjamin Coddington) [1950303]
- nfsd: minor nfsd4_change_attribute cleanup (Benjamin Coddington) [1950303]
- nfsd: simplify nfsd4_change_info (Benjamin Coddington) [1950303]
- nfsd: only call inode_query_iversion in the I_VERSION case (Benjamin Coddington) [1950303]
- nfs_common: need lock during iterate through the list (Benjamin Coddington) [1950303]
- nfsd: Fix message level for normal termination (Benjamin Coddington) [1950303]
- NFS: switch nfsiod to be an UNBOUND workqueue. (Benjamin Coddington) [1950303]
- lockd: don't use interval-based rebinding over TCP (Benjamin Coddington) [1950303]
- net: sunrpc: Fix 'snprintf' return value check in 'do_xprt_debugfs' (Benjamin Coddington) [1950303]
- NFSv4: Refactor to use user namespaces for nfs4idmap (Benjamin Coddington) [1950303]
- NFS: NFSv2/NFSv3: Use cred from fs_context during mount (Benjamin Coddington) [1950303]
- NFSv4: Fix a pNFS layout related use-after-free race when freeing the inode (Benjamin Coddington) [1950303]
- SUNRPC: Fix up xdr_set_page() (Benjamin Coddington) [1950303]
- SUNRPC: Fix open coded xdr_stream_remaining() (Benjamin Coddington) [1950303]
- pNFS: Clean up open coded xdr string decoding (Benjamin Coddington) [1950303]
- SUNRPC: Fix up open coded kmemdup_nul() (Benjamin Coddington) [1950303]
- pNFS/flexfiles: Fix up layoutstats reporting for non-TCP transports (Benjamin Coddington) [1950303]
- NFSv4/pNFS: Store the transport type in struct nfs4_pnfs_ds_addr (Benjamin Coddington) [1950303]
- pNFS: Add helpers for allocation/free of struct nfs4_pnfs_ds_addr (Benjamin Coddington) [1950303]
- NFSv4/pNFS: Use connections to a DS that are all of the same protocol family (Benjamin Coddington) [1950303]
- SUNRPC: Remove unused function xprt_load_transport() (Benjamin Coddington) [1950303]
- NFS: Switch mount code to use xprt_find_transport_ident() (Benjamin Coddington) [1950303]
- SUNRPC: Add a helper to return the transport identifier given a netid (Benjamin Coddington) [1950303]
- SUNRPC: Close a race with transport setup and module put (Benjamin Coddington) [1950303]
- SUNRPC: xprt_load_transport() needs to support the netid "rdma6" (Benjamin Coddington) [1950303]
- SUNRPC: rpc_wake_up() should wake up tasks in the correct order (Benjamin Coddington) [1950303]
- NFSv4: Observe the NFS_MOUNT_SOFTREVAL flag in _nfs4_proc_lookupp (Benjamin Coddington) [1950303]
- NFSv3: Add emulation of the lookupp() operation (Benjamin Coddington) [1950303]
- NFSv3: Refactor nfs3_proc_lookup() to split out the dentry (Benjamin Coddington) [1950303]
- SUNRPC: Remove XDRBUF_SPARSE_PAGES flag in gss_proxy upcall (Benjamin Coddington) [1950303]
- NFSD: Add tracepoints in nfsd4_decode/encode_compound() (Benjamin Coddington) [1950303]
- NFSD: Add tracepoints in nfsd_dispatch() (Benjamin Coddington) [1950303]
- NFSD: Add common helpers to decode void args and encode void results (Benjamin Coddington) [1950303]
- SUNRPC: Prepare for xdr_stream-style decoding on the server-side (Benjamin Coddington) [1950303]
- SUNRPC: Add xdr_set_scratch_page() and xdr_reset_scratch_buffer() (Benjamin Coddington) [1950303]
- NFSD: Add SPDX header for fs/nfsd/trace.c (Benjamin Coddington) [1950303]
- SUNRPC: Move the svc_xdr_recvfrom() tracepoint (Benjamin Coddington) [1950303]
- NFSD: Add SPDX header for fs/nfsd/trace.c (Benjamin Coddington) [1950303]
- NFSD: A semicolon is not needed after a switch statement. (Benjamin Coddington) [1950303]
- svcrdma: support multiple Read chunks per RPC (Benjamin Coddington) [1950303]
- svcrdma: Use the new parsed chunk list when pulling Read chunks (Benjamin Coddington) [1950303]
- svcrdma: Rename info::ri_chunklen (Benjamin Coddington) [1950303]
- svcrdma: Clean up chunk tracepoints (Benjamin Coddington) [1950303]
- svcrdma: Remove chunk list pointers (Benjamin Coddington) [1950303]
- svcrdma: Support multiple Write chunks in svc_rdma_send_reply_chunk (Benjamin Coddington) [1950303]
- svcrdma: Support multiple Write chunks in svc_rdma_map_reply_msg() (Benjamin Coddington) [1950303]
- svcrdma: Support multiple write chunks when pulling up (Benjamin Coddington) [1950303]
- svcrdma: Use parsed chunk lists to encode Reply transport headers (Benjamin Coddington) [1950303]
- svcrdma: Use parsed chunk lists to construct RDMA Writes (Benjamin Coddington) [1950303]
- svcrdma: Use parsed chunk lists to detect reverse direction replies (Benjamin Coddington) [1950303]
- svcrdma: Use parsed chunk lists to derive the inv_rkey (Benjamin Coddington) [1950303]
- svcrdma: Add a "parsed chunk list" data structure (Benjamin Coddington) [1950303]
- svcrdma: Clean up svc_rdma_encode_reply_chunk() (Benjamin Coddington) [1950303]
- svcrdma: Post RDMA Writes while XDR encoding replies (Benjamin Coddington) [1950303]
- NFSD: Invoke svc_encode_result_payload() in "read" NFSD encoders (Benjamin Coddington) [1950303]
- SUNRPC: Rename svc_encode_read_payload() (Benjamin Coddington) [1950303]
- svcrdma: Refactor the RDMA Write path (Benjamin Coddington) [1950303]
- svcrdma: Const-ify the xdr_buf arguments (Benjamin Coddington) [1950303]
- SUNRPC: Adjust synopsis of xdr_buf_subsegment() (Benjamin Coddington) [1950303]
- svcrdma: Catch another Reply chunk overflow case (Benjamin Coddington) [1950303]
- xprtrdma: Micro-optimize MR DMA-unmapping (Benjamin Coddington) [1950303]
- xprtrdma: Move rpcrdma_mr_put() (Benjamin Coddington) [1950303]
- xprtrdma: Trace unmap_sync calls (Benjamin Coddington) [1950303]
- xprtrdma: Display the task ID when reporting MR events (Benjamin Coddington) [1950303]
- xprtrdma: Clean up trace_xprtrdma_nomrs() (Benjamin Coddington) [1950303]
- xprtrdma: Clean up xprtrdma callback tracepoints (Benjamin Coddington) [1950303]
- xprtrdma: Clean up tracepoints in the reply path (Benjamin Coddington) [1950303]
- xprtrdma: Clean up reply parsing error tracepoints (Benjamin Coddington) [1950303]
- xprtrdma: Clean up trace_xprtrdma_post_linv (Benjamin Coddington) [1950303]
- xprtrdma: Introduce FRWR completion IDs (Benjamin Coddington) [1950303]
- xprtrdma: Introduce Send completion IDs (Benjamin Coddington) [1950303]
- xprtrdma: Introduce Receive completion IDs (Benjamin Coddington) [1950303]
- xprtrdma: Replace dprintk call sites in ERR_CHUNK path (Benjamin Coddington) [1950303]
- xprtrdma: Fix a BUG when tracing is enabled with NFSv4.1 on RDMA (Benjamin Coddington) [1950303]
- drm: virtio: fix common struct sg_table related issues (Eric Auger) [1971821]
- net/mlx5e: Release skb in case of failure in tc update skb (Alaa Hleihel) [1915307]
- net/mlx5: Release devlink object if adev fails (Alaa Hleihel) [1915307]
- net/mlx5: Fix compilation warning for 32-bit platform (Alaa Hleihel) [1915307]
- net/mlx5: Use effective interrupt affinity (Alaa Hleihel) [1915307]
- net/mlx5: Replace irq_to_desc() abuse (Alaa Hleihel) [1915307]
- RDMA/mlx5: Remove unneeded semicolon (Alaa Hleihel) [1915307]
- net: mlx5: convert comma to semicolon (Alaa Hleihel) [1915307]
- net/mlx5e: Fill mlx5e_create_cq_param in a function (Alaa Hleihel) [1915307]
- net/mlx5e: Split between RX/TX tunnel FW support indication (Alaa Hleihel) [1915307]
- net/mlx5: Arm only EQs with EQEs (Alaa Hleihel) [1915307]
- net/mlx5e: Remove duplicated include (Alaa Hleihel) [1915307]
- net/mlx5e: Add TX port timestamp support (Alaa Hleihel) [1915307 1919646]
- net/mlx5e: Add TX PTP port object support (Alaa Hleihel) [1915307 1919646]
- net/mlx5e: Move MLX5E_RX_ERR_CQE macro (Alaa Hleihel) [1915307 1919646]
- net/mlx5e: Split SW group counters update function (Alaa Hleihel) [1915307 1919646]
- net/mlx5e: Change skb fifo push/pop API to be used without SQ (Alaa Hleihel) [1915307 1919646]
- net/mlx5e: Allow SQ outside of channel context (Alaa Hleihel) [1915307 1919646]
- net/mlx5e: Allow RQ outside of channel context (Alaa Hleihel) [1915307 1919646]
- net/mlx5e: Allow CQ outside of channel context (Alaa Hleihel) [1915307 1919646]
- net/mlx5e: Free drop RQ in a dedicated function (Alaa Hleihel) [1915307]
- RDMA/mlx5: Remove IB representors dead code (Alaa Hleihel) [1915307]
- net/mlx5: Simplify eswitch mode check (Alaa Hleihel) [1915307]
- net/mlx5: Delete custom device management logic (Alaa Hleihel) [1915307]
- RDMA/mlx5: Convert mlx5_ib to use auxiliary bus (Alaa Hleihel) [1915307]
- net/mlx5e: Connect ethernet part to auxiliary bus (Alaa Hleihel) [1915307]
- vdpa/mlx5: Connect mlx5_vdpa to auxiliary bus (Alaa Hleihel) [1915307]
- net/mlx5: Register mlx5 devices to auxiliary virtual bus (Alaa Hleihel) [1915307]
- vdpa/mlx5: Make hardware definitions visible to all mlx5 devices (Alaa Hleihel) [1915307]
- net/mlx5_core: Clean driver version and name (Alaa Hleihel) [1915307]
- net/mlx5: Treat host PF vport as other (non eswitch manager) vport (Alaa Hleihel) [1915307]
- net/mlx5: Export steering related functions (Alaa Hleihel) [1915307]
- net/mlx5: Expose other function ifc bits (Alaa Hleihel) [1915307]
- net/mlx5: Expose IP-in-IP TX and RX capability bits (Alaa Hleihel) [1915307]
- net/mlx5: Update the hardware interface definition for vhca state (Alaa Hleihel) [1915307]
- net/mlx5: Avoid exposing driver internal command helpers (Alaa Hleihel) [1915307]
- net/mlx5: Add ts_cqe_to_dest_cqn related bits (Alaa Hleihel) [1915307]
- net/mlx5: Add misc4 to mlx5_ifc_fte_match_param_bits (Alaa Hleihel) [1915307]
- net/mlx5: Check dr mask size against mlx5_match_param size (Alaa Hleihel) [1915307]
- net/mlx5: Add sampler destination type (Alaa Hleihel) [1915307]
- net/mlx5: Add sample offload hardware bits and structures (Alaa Hleihel) [1915307]
- RDMA/mlx5: Use PCI device for dma mappings (Alaa Hleihel) [1915307]
- RDMA/mlx5: Silence the overflow warning while building offset mask (Alaa Hleihel) [1915307]
- RDMA/mlx5: Check for ERR_PTR from uverbs_zalloc() (Alaa Hleihel) [1915307]
- RDMA/mlx5: Enable querying AH for XRC QP types (Alaa Hleihel) [1915307]
- RDMA/mlx5: Lower setting the umem's PAS for SRQ (Alaa Hleihel) [1915307]
- RDMA/mlx5: Use ib_umem_find_best_pgsz() for devx (Alaa Hleihel) [1915307]
- RDMA/mlx5: mlx5_umem_find_best_quantized_pgoff() for CQ (Alaa Hleihel) [1915307]
- RDMA/mlx5: Use mlx5_umem_find_best_quantized_pgoff() for QP (Alaa Hleihel) [1915307]
- RDMA/mlx5: Directly compute the PAS list for raw QP RQ's (Alaa Hleihel) [1915307]
- RDMA/mlx5: Use mlx5_umem_find_best_quantized_pgoff() for WQ (Alaa Hleihel) [1915307]
- RDMA/mlx5: Use ib_umem_find_best_pgoff() for SRQ (Alaa Hleihel) [1915307]
- net: mlx5: Replace in_irq() usage (Alaa Hleihel) [1915307]
- net/mlx5: Cleanup kernel-doc warnings (Alaa Hleihel) [1915307]
- net/mlx5e: Validate stop_room size upon user input (Alaa Hleihel) [1915307]
- net/mlx5: DR, Free unused buddy ICM memory (Alaa Hleihel) [1915307]
- net/mlx5: DR, ICM memory pools sync optimization (Alaa Hleihel) [1915307]
- net/mlx5: DR, Sync chunks only during free (Alaa Hleihel) [1915307]
- net/mlx5: DR, Handle ICM memory via buddy allocation instead of buckets (Alaa Hleihel) [1915307]
- net/mlx5: DR, Add buddy allocator utilities (Alaa Hleihel) [1915307]
- net/mlx5: DR, Rename matcher functions to be more HW agnostic (Alaa Hleihel) [1915307]
- net/mlx5: DR, Rename builders HW specific names (Alaa Hleihel) [1915307]
- net/mlx5: DR, Remove unused member of action struct (Alaa Hleihel) [1915307]
- IB/mlx5: Add support for NDR link speed (Alaa Hleihel) [1915307 1926596]
- RDMA/mlx5: Use ib_umem_find_best_pgsz() for mkc's (Alaa Hleihel) [1915307]
- RDMA/mlx5: Split the WR setup out of mlx5_ib_update_xlt() (Alaa Hleihel) [1915307]
- RDMA/mlx5: Move xlt_emergency_page_mutex into mr.c (Alaa Hleihel) [1915307]
- RDMA/mlx5: Change mlx5_ib_populate_pas() to use rdma_for_each_block() (Alaa Hleihel) [1915307]
- RDMA/mlx5: Remove npages from mlx5_ib_cont_pages() (Alaa Hleihel) [1915307]
- RDMA/mlx5: Remove ncont from mlx5_ib_cont_pages() (Alaa Hleihel) [1915307]
- RDMA/mlx5: Remove order from mlx5_ib_cont_pages() (Alaa Hleihel) [1915307]
- RDMA/mlx5: Move mlx5_ib_cont_pages() to the creation of the mlx5_ib_mr (Alaa Hleihel) [1915307]
- RDMA/mlx5: Remove mlx5_ib_mr->order (Alaa Hleihel) [1915307]
- net/mlx5: fix error return code in mlx5e_tc_nic_init() (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5e: Fix IPsec packet drop by mlx5e_tc_update_skb (Alaa Hleihel) [1915306 1915307]
- net/mlx5e: Use spin_lock_bh for async_icosq_lock (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Replace zero-length array with flexible-array member (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Fix uininitialized pointer read on pointer attr (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Add support for devlink reload limit no reset (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Add support for fw live patch event (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Add devlink param enable_remote_dev_reset support (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Add support for devlink reload action fw activate (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Handle sync reset abort event (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Handle sync reset now event (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Handle sync reset request event (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Set cap for pci sync for fw update event (Alaa Hleihel) [1915306 1915307]
- net/mlx5: Add functions to set/query MFRL register (Alaa Hleihel) [1915306 1915307]
- net/mlx5e: Fix potential null pointer dereference (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5: Fix dereference on pointer attr after null check (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5: Use dma device access helper (Alaa Hleihel) [1915306 1915307]
- mlx5: cross-tree: phase out dma_zalloc_coherent() (Alaa Hleihel) [1915306 1915307]
- net/mlx5: E-Switch, Support flow source for local vport (Alaa Hleihel) [1915306 1915307]
- net/mlx5: E-switch, Move devlink eswitch ports closer to eswitch (Alaa Hleihel) [1915306 1915307]
- net/mlx5: E-switch, Use helper function to load unload representor (Alaa Hleihel) [1915306 1915307]
- net/mlx5: E-switch, Add helper to check egress ACL need (Alaa Hleihel) [1915306 1915307]
- net/mlx5: E-switch, Use PF num in metadata reg c0 (Alaa Hleihel) [1915306 1915307]
- net/mlx5: DR, Add support for rule creation with flow source hint (Alaa Hleihel) [1915306 1915307]
- net/mlx5: DR, Call ste_builder directly with tag pointer (Alaa Hleihel) [1915306 1915307]
- net/mlx5: DR, Remove unneeded local variable (Alaa Hleihel) [1915306 1915307]
- net/mlx5: DR, Remove unneeded vlan check from L2 builder (Alaa Hleihel) [1915306 1915307]
- net/mlx5: DR, Remove unneeded check from source port builder (Alaa Hleihel) [1915306 1915307]
- net/mlx5: DR, Replace the check for valid STE entry (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Delete not needed GSI QP signal QP type (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Change GSI QP to have same creation flow like other QPs (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Reuse existing fields in parent QP storage object (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Embed GSI QP into general mlx5_ib QP (Alaa Hleihel) [1915306 1915307]
- net/mlx5: remove unreachable return (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5e: Use kfree() to free fd->g in accel_fs_tcp_create_groups() (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5e: Keep direct reference to mlx5_core_dev in tc ct (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5e: Support CT offload for tc nic flows (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5e: rework ct offload init messages (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5e: Add tc chains offload support for nic flows (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5: Refactor tc flow attributes structure (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5e: Split nic tc flow allocation and creation (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5e: Tc nic flows to use mlx5_chains flow tables (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5: Allow ft level ignore for nic rx tables (Alaa Hleihel) [1915306 1915307 1919642]
- net/mlx5: Refactor multi chains and prios support (Alaa Hleihel) [1915306 1915307 1919642]
- RDMA/mlx5: Don't call to restrack recursively (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Expose TIR and QP ICM address for sw_owner_v2 devices (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Allow DM allocation for sw_owner_v2 enabled devices (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Add sw_owner_v2 bit capability (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Delete duplicated mlx5_ptys_width enum (Alaa Hleihel) [1915306 1915307]
- net/mlx5: IPsec: make spdxcheck.py happy (Alaa Hleihel) [1915306 1915307]
- mlx5: add pause frame stats (Alaa Hleihel) [1915306 1915307]
- net/mlx5e: Add CQE compression support for multi-strides packets (Alaa Hleihel) [1915306 1915307]
- net/mlx5e: Add support for tc trap (Alaa Hleihel) [1915306 1915307 1919643]
- net/mlx5: E-Switch, Use vport metadata matching by default (Alaa Hleihel) [1915306 1915307]
- net/mlx5: remove erroneous fallthrough (Alaa Hleihel) [1915306 1915307]
- IB/mlx5: Add DCT RoCE LAG support (Alaa Hleihel) [1915306 1915307]
- IB/mlx5: Add tx_affinity support for DCI QP (Alaa Hleihel) [1915306 1915307]
- net/mlx5e: RX, Add a prefetch command for small L1_CACHE_BYTES (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Enable sniffer when device is in switchdev mode (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Add new IB rates support (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Replace open-coded offsetofend() macro (Alaa Hleihel) [1915306 1915307]
- RDMA/mlx5: Simplify multiple else-if cases with switch keyword (Alaa Hleihel) [1915306 1915307]
- net/mlx5e: Link non uplink representors to PCI device (Alaa Hleihel) [1915306 1915307 1959367]
- perf stat: Use nftw() instead of ftw() (Michael Petlan) [1944684]
- perf unwind: Set userdata for all __report_module() paths (Michael Petlan) [1944684]
- perf debug: Move debug initialization earlier (Michael Petlan) [1944684]
- perf script: Fix overrun issue for dynamically-allocated PMU type number (Michael Petlan) [1944684]
- perf metricgroup: Fix system PMU metrics (Michael Petlan) [1944684]
- perf metricgroup: Fix for metrics containing duration_time (Michael Petlan) [1944684]
- perf evlist: Fix id index for heterogeneous systems (Michael Petlan) [1944684]
- perf inject: Correct event attribute sizes (Michael Petlan) [1944684]
- perf intel-pt: Fix 'CPU too large' error (Michael Petlan) [1944684]
- perf stat: Take cgroups into account for shadow stats (Michael Petlan) [1944684]
- perf stat: Introduce struct runtime_stat_data (Michael Petlan) [1944684]
- libperf tests: Fail when failing to get a tracepoint id (Michael Petlan) [1944684]
- libperf tests: If a test fails return non-zero (Michael Petlan) [1944684]
- libperf tests: Avoid uninitialized variable warning (Michael Petlan) [1944684]
- perf test: Fix shadow stat test for non-bash shells (Michael Petlan) [1944684]
- perf bpf examples: Fix bpf.h header include directive in 5sec.c example (Michael Petlan) [1944684]
- perf probe: Fix memory leak when synthesizing SDT probes (Michael Petlan) [1944684]
- perf stat aggregation: Add separate thread member (Michael Petlan) [1944684]
- perf stat aggregation: Add separate core member (Michael Petlan) [1944684]
- perf stat aggregation: Add separate die member (Michael Petlan) [1944684]
- perf stat aggregation: Add separate socket member (Michael Petlan) [1944684]
- perf stat aggregation: Add separate node member (Michael Petlan) [1944684]
- perf stat aggregation: Start using cpu_aggr_id in map (Michael Petlan) [1944684]
- perf cpumap: Drop in cpu_aggr_map struct (Michael Petlan) [1944684]
- perf cpumap: Add new map type for aggregation (Michael Petlan) [1944684]
- perf stat: Replace aggregation ID with a struct (Michael Petlan) [1944684]
- perf cpumap: Add new struct for cpu aggregation (Michael Petlan) [1944684]
- perf cpumap: Use existing allocator to avoid using malloc (Michael Petlan) [1944684]
- perf tests: Improve topology test to check all aggregation types (Michael Petlan) [1944684]
- perf powerpc: Move syscall.tbl check to check-headers.sh (Michael Petlan) [1944684]
- tools arch x86: Sync the msr-index.h copy with the kernel sources (Michael Petlan) [1944684]
- perf mem: Factor out a function to generate sort order (Michael Petlan) [1944684]
- perf sort: Add sort option for data page size (Michael Petlan) [1944684]
- perf script: Support data page size (Michael Petlan) [1944684]
- tools headers UAPI: Sync linux/stat.h with the kernel sources (Michael Petlan) [1944684]
- tools: Factor HOSTCC, HOSTLD, HOSTAR definitions (Michael Petlan) [1944684]
- perf config: Fix example command in manpage to conform to syntax specified in the SYNOPSIS section. (Michael Petlan) [1944684]
- perf test: Make sample-parsing test aware of PERF_SAMPLE_{CODE,DATA}_PAGE_SIZE (Michael Petlan) [1944684]
- perf tools: Add support to read build id from compressed elf (Michael Petlan) [1944684]
- perf debug: Add debug_set_file function (Michael Petlan) [1944684]
- perf evlist: Support pipe mode display (Michael Petlan) [1944684]
- perf report: Support --header-only for pipe mode (Michael Petlan) [1944684]
- perf vendor events: Add JSON metrics for imx8mm DDR Perf (Michael Petlan) [1944684]
- perf metricgroup: Support adding metrics for system PMUs (Michael Petlan) [1944684]
- perf metricgroup: Support printing metric groups for system PMUs (Michael Petlan) [1944684]
- perf metricgroup: Split up metricgroup__print() (Michael Petlan) [1944684]
- perf metricgroup: Fix metrics using aliases covering multiple PMUs (Michael Petlan) [1944684]
- perf evlist: Change evlist__splice_list_tail() ordering (Michael Petlan) [1944684]
- perf pmu: Add pmu_add_sys_aliases() (Michael Petlan) [1944684]
- perf pmu: Add pmu_id() (Michael Petlan) [1944684]
- perf jevents: Add support for system events tables (Michael Petlan) [1944684]
- perf jevents: Add support for an extra directory level (Michael Petlan) [1944684]
- perf evsel: Emit warning about kernel not supporting the data page size sample_type bit (Michael Petlan) [1944684]
- perf record: Support new sample type for data page size (Michael Petlan) [1944684]
- perf unwind: Fix separate debug info files when using elfutils' libdw's unwinder (Michael Petlan) [1944684]
- perf record: Fix memory leak when using '--user-regs=?' to list registers (Michael Petlan) [1944684]
- tools headers UAPI: Update tools's copy of linux/perf_event.h (Michael Petlan) [1944684]
- tools build: Add missing libcap to test-all.bin target (Michael Petlan) [1944684]
- perf test: Fix metric parsing test (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' record methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' diff methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' nr_threads method (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' deliver event method (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' header methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' raw samples methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' mmap pages parsing method (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' event attribute config methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for alternative 'struct evlist' constructors (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' event selection methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' event group methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' create maps methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' print methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' evsel list methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' pause/resume methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' id_pos methods (Michael Petlan) [1944684]
- perf tools: Add aarch64 registers to --user-regs (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' tracking event methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' browser methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' sample id lookup methods (Michael Petlan) [1944684]
- perf evlist: Ditch unused set/reset sample_bit methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' sample parsing methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' sideband thread methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' 'filter' methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' stats methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' 'workload' methods (Michael Petlan) [1944684]
- perf evlist: Use the right prefix for 'struct evlist' methods: evlist__set_leader() (Michael Petlan) [1944684]
- perf evsel: Convert last 'struct evsel' methods to the right evsel__ prefix (Michael Petlan) [1944684]
- perf test: Add shadow stat test (Michael Petlan) [1944684]
- perf build-id: Add build_id_cache__add function (Michael Petlan) [1944684]
- perf build-id: Add __perf_session__cache_build_ids function (Michael Petlan) [1944684]
- perf build-id: Use machine__for_each_dso in perf_session__cache_build_ids (Michael Petlan) [1944684]
- perf data: Add is_perf_data function (Michael Petlan) [1944684]
- perf symbols: Try to load vmlinux from buildid database (Michael Petlan) [1944684]
- perf tools: Use struct extra_kernel_map in machine__process_kernel_mmap_event (Michael Petlan) [1944684]
- perf build-id: Add check for existing link in buildid dir (Michael Petlan) [1944684]
- perf tools: Add filename__decompress function (Michael Petlan) [1944684]
- perf tools: Add build_id__is_defined function (Michael Petlan) [1944684]
- tools lib: Adopt memchr_inv() from kernel (Michael Petlan) [1944684]
- perf arm-spe: Add support for ARMv8.3-SPE (Michael Petlan) [1944684]
- perf arm_spe: Decode memory tagging properties (Michael Petlan) [1944684]
- perf arm-spe: Add more sub classes for operation packet (Michael Petlan) [1944684]
- perf arm-spe: Refactor operation packet handling (Michael Petlan) [1944684]
- perf arm-spe: Add new function arm_spe_pkt_desc_op_type() (Michael Petlan) [1944684]
- perf arm-spe: Remove size condition checking for events (Michael Petlan) [1944684]
- perf arm-spe: Refactor event type handling (Michael Petlan) [1944684]
- perf arm-spe: Add new function arm_spe_pkt_desc_event() (Michael Petlan) [1944684]
- perf arm-spe: Refactor counter packet handling (Michael Petlan) [1944684]
- perf arm-spe: Add new function arm_spe_pkt_desc_counter() (Michael Petlan) [1944684]
- perf arm-spe: Refactor context packet handling (Michael Petlan) [1944684]
- perf arm_spe: Fixup top byte for data virtual address (Michael Petlan) [1944684]
- perf arm-spe: Refactor address packet handling (Michael Petlan) [1944684]
- perf arm-spe: Add new function arm_spe_pkt_desc_addr() (Michael Petlan) [1944684]
- perf arm-spe: Refactor packet header parsing (Michael Petlan) [1944684]
- perf arm-spe: Refactor printing string to buffer (Michael Petlan) [1944684]
- perf expr: Force encapsulation on expr_id_data (Michael Petlan) [1944684]
- perf vendor events: Update Skylake client events to v50 (Michael Petlan) [1944684]
- perf data: Allow to use stdio functions for pipe mode (Michael Petlan) [1944684]
- perf arm-spe: Fix packet length handling (Michael Petlan) [1944684]
- perf arm-spe: Refactor arm_spe_get_events() (Michael Petlan) [1944684]
- perf arm-spe: Refactor payload size calculation (Michael Petlan) [1944684]
- perf arm-spe: Fix a typo in comment (Michael Petlan) [1944684]
- perf arm-spe: Include bitops.h for BIT() macro (Michael Petlan) [1944684]
- perf mem: Support ARM SPE events (Michael Petlan) [1944684]
- perf c2c: Support AUX trace (Michael Petlan) [1944684]
- perf mem: Support AUX trace (Michael Petlan) [1944684]
- perf auxtrace: Add itrace option '-M' for memory events (Michael Petlan) [1944684]
- perf mem: Only initialize memory event for recording (Michael Petlan) [1944684]
- perf c2c: Support memory event PERF_MEM_EVENTS__LOAD_STORE (Michael Petlan) [1944684]
- perf mem: Support new memory event PERF_MEM_EVENTS__LOAD_STORE (Michael Petlan) [1944684]
- perf mem: Introduce weak function perf_mem_events__ptr() (Michael Petlan) [1944684]
- perf mem: Search event name with more flexible path (Michael Petlan) [1944684]
- perf jevents: Add test for arch std events (Michael Petlan) [1944684]
- perf jevents: Tidy error handling (Michael Petlan) [1944684]
- perf trace beauty: Allow header files in a different path (Michael Petlan) [1944684]
- perf stat: Add --quiet option (Michael Petlan) [1944684]
- perf stat: Support regex pattern in --for-each-cgroup (Michael Petlan) [1944684]
- perf kvm: Add kvm-stat for arm64 (Michael Petlan) [1944684]
- perf env: Conditionally compile BPF support code on having HAVE_LIBBPF_SUPPORT (Michael Petlan) [1944684]
- perf annotate: Move bpf header inclusion to inside HAVE_LIBBPF_SUPPORT (Michael Petlan) [1944684]
- perf tests: Skip the llvm and bpf tests if HAVE_LIBBPF_SUPPORT isn't defined (Michael Petlan) [1944684]
- perf bpf: Enclose libbpf.h include within HAVE_LIBBPF_SUPPORT (Michael Petlan) [1944684]
- perf test: Implement skip_reason callback for watchpoint tests (Michael Petlan) [1944684]
- perf tests tsc: Add checking helper is_supported() (Michael Petlan) [1944684]
- perf tests tsc: Make tsc testing as a common testing (Michael Petlan) [1944684]
- perf mem2node: Improve warning if detected no memory nodes (Michael Petlan) [1944684]
- perf version: Add a feature for libpfm4 (Michael Petlan) [1944684]
- perf annotate mips: Add perf arch instructions annotate handlers (Michael Petlan) [1944684]
- perf/x86/intel: Add Tremont Topdown support (Michael Petlan) [1944684]
- perf/x86: Fix fall-through warnings for Clang (Michael Petlan) [1944684]
- perf/x86/intel/lbr: Fix the return type of get_lbr_cycles() (Michael Petlan) [1944684]
- perf/x86/intel: Fix rtm_abort_event encoding on Ice Lake (Michael Petlan) [1944684]
- perf: Break deadlock involving exec_update_mutex (Michael Petlan) [1944684]
- perf/x86/intel: Add event constraint for CYCLE_ACTIVITY.STALLS_MEM_ANY (Michael Petlan) [1944684]
- perf/x86: Avoid TIF_IA32 when checking 64bit mode (Michael Petlan) [1944684]
- bpf, ringbuf: Deny reserve of buffers larger than ringbuf (Wander Lairson Costa) [1968591]
- bpf: Fix propagation of 32 bit unsigned bounds from 64 bit bounds (Wander Lairson Costa) [1965938]
- tools/power turbostat: Support Alder Lake Mobile (Steve Best) [1962291]
- ACPI: PM: Add ACPI ID of Alder Lake Fan (Steve Best) [1961617]
- powercap/intel_rapl: add support for AlderLake Mobile (Steve Best) [1961816]
- platform/x86: intel_pmc_core: Uninitialized data in pmc_core_lpm_latch_mode_write() (David Arcari) [1783557]
- platform/x86: intel_pmc_core: add ACPI dependency (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Fix "unsigned 'ret' is never less than zero" smatch warning (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Add LTR registers for Tiger Lake (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Add option to set/clear LPM mode (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Add requirements file to debugfs (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Get LPM requirements for Tiger Lake (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Show LPM residency in microseconds (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Handle sub-states generically (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Remove global struct pmc_dev (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Don't use global pmcdev in quirks (David Arcari) [1783557]
- platform/x86: intel_pmc_core: export platform global reset bits via etr3 sysfs file (David Arcari) [1783557]
- MAINTAINERS: Update maintainers for pmc_core driver (David Arcari) [1783557]
- MAINTAINERS: intel_pmc_core: Update MAINTAINERS (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Ignore GBE LTR on Tiger Lake platforms (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Assign boolean values to a bool variable (David Arcari) [1783557]
- platform/x86: intel_pmc_core: fix: Replace dev_dbg macro with dev_info() (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Clean up: Remove the duplicate comments and reorganize (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Fix the slp_s0 counter displayed value (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Fix TigerLake power gating status map (David Arcari) [1783557]
- platform/x86: pmc_core: Use descriptive names for LPM registers (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Change Jasper Lake S0ix debug reg map back to ICL (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Add Atom based Jasper Lake (JSL) platform support (David Arcari) [1783557]
- platform/x86: intel_pmc_core: Add Intel Elkhart Lake support (David Arcari) [1783557]
- [s390] s390/ipl: support NVMe IPL kernel parameters (Claudio Imbrenda) [1963120]
- [s390] net/smc: use memcpy instead of snprintf to avoid out of bounds read (Claudio Imbrenda) [1919249]
- [s390] smc: fix out of bound access in smc_nl_get_sys_info() (Claudio Imbrenda) [1919249]
- [s390] net/smc: fix access to parent of an ib device (Claudio Imbrenda) [1919249]
- [s390] s390/vtime: fix increased steal time accounting (Claudio Imbrenda) [1963075]
- [s390] tools/kvm_stat: Add restart delay (Claudio Imbrenda) [1963082]
- [s390] s390/dasd: fix hanging IO request during DASD driver unbind (Claudio Imbrenda) [1963077]
- [s390] s390/dasd: fix hanging DASD driver unbind (Claudio Imbrenda) [1963077]

* Thu Jun 17 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-315.el8]
- Add New KUNIT Tests to Redhat kernel-modules-internal (Nico Pache) [1939309]
- kunit: lib: adhear to KUNIT formatting standard (Nico Pache) [1939309]
- kunit: mptcp: adhere to KUNIT formatting standard (Nico Pache) [1939309]
- ASoC: topology: adhere to KUNIT formatting standard (Nico Pache) [1939309]
- kunit: software node: adhear to KUNIT formatting standard (Nico Pache) [1939309]
- resource: provide meaningful MODULE_LICENSE() in test suite (Nico Pache) [1939309]
- resource: Add test cases for new resource API (Nico Pache) [1939309]
- resource: Introduce resource_intersection() for overlapping resources (Nico Pache) [1939309]
- resource: Introduce resource_union() for overlapping resources (Nico Pache) [1939309]
- mm/resource: Use resource_overlaps() to simplify region_intersects() (Nico Pache) [1939309]
- kernel.h: split out min()/max() et al. helpers (Nico Pache) [1939309]
- lib/math/rational.c: fix possible incorrect result from rational fractions helper (Nico Pache) [1939309]
- lib: Move mathematic helpers to separate folder (Nico Pache) [1939309]
- lib/cmdline: remove an unneeded local variable in next_arg() (Nico Pache) [1939309]
- lib/cmdline: Allow get_options() to take 0 to validate the input (Nico Pache) [1939309]
- lib/cmdline: Update documentation to reflect behaviour (Nico Pache) [1939309]
- lib/cmdline_kunit: add a new test case for get_options() (Nico Pache) [1939309]
- ext4: add .kunitconfig fragment to enable ext4-specific tests (Nico Pache) [1939309]
- ext: EXT4_KUNIT_TESTS should depend on EXT4_FS instead of selecting it (Nico Pache) [1939309]
- kunit: tool: fix unintentional statefulness in run_kernel() (Nico Pache) [1939309]
- kunit: tool: add support for filtering suites by glob (Nico Pache) [1939309]
- kunit: add kunit.filter_glob cmdline option to filter suites (Nico Pache) [1939309]
- kunit: don't show `1 == 1` in failed assertion messages (Nico Pache) [1939309]
- kunit: make kunit_tool accept optional path to .kunitconfig fragment (Nico Pache) [1939309]
- Documentation: kunit: add tips.rst for small examples (Nico Pache) [1939309]
- KUnit: Docs: make start.rst example Kconfig follow style.rst (Nico Pache) [1939309]
- kunit: tool: simplify kconfig is_subset_of() logic (Nico Pache) [1939309]
- minor: kunit: tool: fix unit test so it can run from non-root dir (Nico Pache) [1939309]
- kunit: tool: use `with open()` in unit test (Nico Pache) [1939309]
- kunit: tool: stop using bare asserts in unit test (Nico Pache) [1939309]
- kunit: tool: fix unit test cleanup handling (Nico Pache) [1939309]
- ASoC: topology: KUnit: Convert from cpu to data format (Nico Pache) [1939309]
- drivers/base: build kunit tests without structleak plugin (Nico Pache) [1939309]
- ASoC: topology: KUnit: Add KUnit tests passing topology with PCM to snd_soc_tplg_component_load (Nico Pache) [1939309]
- ASoC: topology: KUnit: Add KUnit tests passing empty topology with variants to snd_soc_tplg_component_load (Nico Pache) [1939309]
- ASoC: topology: KUnit: Add KUnit tests passing various arguments to snd_soc_tplg_component_load (Nico Pache) [1939309]
- ASoC: topology: Check if ops is set before dereference (Nico Pache) [1939309]
- ASoC: topology: Ensure that needed parameters are set (Nico Pache) [1939309]
- kunit: tool: move kunitconfig parsing into __init__, make it optional (Nico Pache) [1939309]
- kunit: tool: fix minor typing issue with None status (Nico Pache) [1939309]
- kunit: tool: surface and address more typing issues (Nico Pache) [1939309]
- Documentation: kunit: include example of a parameterized test (Nico Pache) [1939309]
- kunit: tool: Fix spelling of "diagnostic" in kunit_parser (Nico Pache) [1939309]
- kunit: tool: Force the use of the 'tty' console for UML (Nico Pache) [1939309]
- lib/cmdline_kunit: add a new test suite for cmdline API (Nico Pache) [1939309]
- lib/cmdline: allow NULL to be an output for get_option() (Nico Pache) [1939309]
- lib/cmdline: fix get_option() for strings starting with hyphen (Nico Pache) [1939309]
- lib/cmdline.c: mark expected switch fall-throughs (Nico Pache) [1939309]
- lib/test_bits.c: add tests of GENMASK (Nico Pache) [1939309]
- linux/bits.h: add compile time sanity check of GENMASK inputs (Nico Pache) [1939309]
- linux/build_bug.h: change type to int (Nico Pache) [1939309]
- fs: ext4: Modify inode-test.c to use KUnit parameterized testing feature (Nico Pache) [1939309]
- kunit: Support for Parameterized Testing (Nico Pache) [1939309]
- kunit: kunit_tool: Correctly parse diagnostic messages (Nico Pache) [1939309]
- Documentation: kunit: provide guidance for testing many inputs (Nico Pache) [1939309]
- kunit: Introduce get_file_path() helper (Nico Pache) [1939309]
- kunit: fix display of failed expectations for strings (Nico Pache) [1939309]
- kunit: tool: fix extra trailing \n in raw + parsed test output (Nico Pache) [1939309]
- kunit: tool: print out stderr from make (like build warnings) (Nico Pache) [1939309]
- KUnit: Docs: usage: wording fixes (Nico Pache) [1939309]
- KUnit: Docs: style: fix some Kconfig example issues (Nico Pache) [1939309]
- KUnit: Docs: fix a wording typo (Nico Pache) [1939309]
- kunit: Do not pollute source directory with generated files (test.log) (Nico Pache) [1939309]
- kunit: Do not pollute source directory with generated files (.kunitconfig) (Nico Pache) [1939309]
- kunit: tool: fix pre-existing python type annotation errors (Nico Pache) [1939309]
- kunit: Fix kunit.py parse subcommand (use null build_dir) (Nico Pache) [1939309]
- kunit: test: fix remaining kernel-doc warnings (Nico Pache) [1939309]
- lib: kunit: Fix compilation test when using TEST_BIT_FIELD_COMPILE (Nico Pache) [1939309]
- kunit: test.h: fix a bad kernel-doc markup (Nico Pache) [1939309]
- kunit: test.h: solve kernel-doc warnings (Nico Pache) [1939309]
- lib: kunit: add bitfield test conversion to KUnit (Nico Pache) [1939309]
- Documentation: kunit: add a brief blurb about kunit_test_suite (Nico Pache) [1939309]
- kunit: test: add test plan to KUnit TAP format (Nico Pache) [1939309]
- init: main: add KUnit to kernel init (Nico Pache) [1939309]
- init: unify opening /dev/console as stdin/stdout/stderr (Nico Pache) [1939309]
- kunit: test: create a single centralized executor for all tests (Nico Pache) [1939309]
- Documentation: test.h - fix warnings (Nico Pache) [1939309]
- vmlinux.lds.h: add linker section for KUnit test suites (Nico Pache) [1939309]
- Documentation: kunit: Add naming guidelines (Nico Pache) [1939309]
- kunit: tool: fix display of make errors (Nico Pache) [1939309]
- kunit: tool: handle when .kunit exists but .kunitconfig does not (Nico Pache) [1939309]
- kunit: tool: fix --alltests flag (Nico Pache) [1939309]
- kunit: tool: allow generating test results in JSON (Nico Pache) [1939309]
- kunit: tool: fix running kunit_tool from outside kernel tree (Nico Pache) [1939309]
- lib: add linear ranges helpers (Nico Pache) [1939309]
- PCI/RCEC: Fix RCiEP device to RCEC association (Myron Stowe) [1895942]
- PCI/portdrv: Report reset for frozen channel (Myron Stowe) [1895942]
- PCI/AER: Specify the type of Port that was reset (Myron Stowe) [1895942]
- PCI/ERR: Retain status from error notification (Myron Stowe) [1895942]
- PCI/AER: Clear AER status from Root Port when resetting Downstream Port (Myron Stowe) [1895942]
- PCI/ERR: Clear status of the reporting device (Myron Stowe) [1895942]
- PCI/AER: Add RCEC AER error injection support (Myron Stowe) [1895942]
- PCI/PME: Add pcie_walk_rcec() to RCEC PME handling (Myron Stowe) [1895942]
- PCI/AER: Add pcie_walk_rcec() to RCEC AER handling (Myron Stowe) [1895942]
- PCI/ERR: Recover from RCiEP AER errors (Myron Stowe) [1895942]
- PCI/ERR: Add pcie_link_rcec() to associate RCiEPs (Myron Stowe) [1895942]
- PCI/ERR: Recover from RCEC AER errors (Myron Stowe) [1895942]
- PCI/ERR: Clear AER status only when we control AER (Myron Stowe) [1895942]
- PCI/ERR: Add pci_walk_bridge() to pcie_do_recovery() (Myron Stowe) [1895942]
- PCI/ERR: Avoid negated conditional for clarity (Myron Stowe) [1895942]
- PCI/ERR: Use "bridge" for clarity in pcie_do_recovery() (Myron Stowe) [1895942]
- PCI/ERR: Simplify by computing pci_pcie_type() once (Myron Stowe) [1895942]
- PCI/ERR: Simplify by using pci_upstream_bridge() (Myron Stowe) [1895942]
- PCI/ERR: Rename reset_link() to reset_subordinates() (Myron Stowe) [1895942]
- PCI/ERR: Cache RCEC EA Capability offset in pci_init_capabilities() (Myron Stowe) [1895942]
- PCI/ERR: Bind RCEC devices to the Root Port driver (Myron Stowe) [1895942]
- PCI/AER: Write AER Capability only when we control it (Myron Stowe) [1895942]
- PCI: Export pcie_has_flr() (Myron Stowe) [1895942]
- xfs: don't reuse busy extents on extent trim (Brian Foster) [1960380]
- nvme: add 'kato' sysfs attribute (Gopal Tiwari) [1970260]
- scsi: qla2xxx: Update version to 10.02.00.106-k (Nilesh Javali) [1925465]
- scsi: qla2xxx: Add marginal path handling support (Nilesh Javali) [1925465]
- scsi: qla2xxx: Do logout even if fabric scan retries got exhausted (Nilesh Javali) [1925465]
- scsi: qla2xxx: Update default AER debug mask (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix mailbox recovery during PCIe error (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix crash in PCIe error handling (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix RISC RESET completion polling (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix crash in qla2xxx_mqueuecommand() (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix use after free in bsg (Nilesh Javali) [1925465]
- scsi: qla2xxx: Consolidate zio threshold setting for both FCP & NVMe (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix stuck session (Nilesh Javali) [1925465]
- scsi: qla2xxx: Add H:C:T info in the log message for fc ports (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix IOPS drop seen in some adapters (Nilesh Javali) [1925465]
- scsi: qla2xxx: Check kzalloc() return value (Nilesh Javali) [1925465]
- scsi: qla2xxx: Always check the return value of qla24xx_get_isp_stats() (Nilesh Javali) [1925465]
- scsi: qla2xxx: Simplify qla8044_minidump_process_control() (Nilesh Javali) [1925465]
- scsi: qla2xxx: Suppress Coverity complaints about dseg_r* (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix endianness annotations (Nilesh Javali) [1925465]
- scsi: qla2xxx: Constify struct qla_tgt_func_tmpl (Nilesh Javali) [1925465]
- scsi: qla2xxx: Use dma_pool_zalloc() (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix a couple of misdocumented functions (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix incorrectly named function qla8044_check_temp() (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix a couple of misnamed functions (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix some incorrect formatting/spelling issues (Nilesh Javali) [1925465]
- scsi: qla2xxx: Replace __qla2x00_marker()'s missing underscores (Nilesh Javali) [1925465]
- scsi: Revert "qla2xxx: Make sure that aborted commands are freed" (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix broken #endif placement (Nilesh Javali) [1925465]
- scsi: qla2xxx: Simplify if statement (Nilesh Javali) [1925465]
- scsi: qla2xxx: Simplify the calculation of variables (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix some memory corruption (Nilesh Javali) [1925465]
- scsi: qla2xxx: Remove redundant NULL check (Nilesh Javali) [1925465]
- scsi: qla2xxx: Remove unnecessary NULL check (Nilesh Javali) [1925465]
- scsi: qla2xxx: Assign boolean values to a bool variable (Nilesh Javali) [1925465]
- scsi: qla2xxx: fc_remote_port_chkready() returns a SCSI result value (Nilesh Javali) [1925465]
- scsi: qla2xxx: Fix description for parameter ql2xenforce_iocb_limit (Nilesh Javali) [1925465]
- scsi: qla2xxx: Update version to 10.02.00.105-k (Nilesh Javali) [1925465]
- scsi: qla2xxx: Enable NVMe CONF (BIT_7) when enabling SLER (Nilesh Javali) [1925465]
- scsi: qla2xxx: Wait for ABTS response on I/O timeouts for NVMe (Nilesh Javali) [1925465]
- scsi: qla2xxx: Move some messages from debug to normal log level (Nilesh Javali) [1925465]
- scsi: qla2xxx: Add error counters to debugfs node (Nilesh Javali) [1925465]
- scsi: qla2xxx: Implementation to get and manage host, target stats and initiator port (Nilesh Javali) [1925465]
- scsi: ibmvfc: Reinit target retries (Steve Best) [1965010]
- scsi: ibmvfc: Avoid move login if fast fail is enabled (Steve Best) [1965010]
- scsi: ibmvfc: Handle move login failure (Steve Best) [1965010]
- scsi: ibmvfc: Fix invalid state machine BUG_ON() (Steve Best) [1965010]
- scsi: qedf: Do not put host in qedf_vport_create() unconditionally (Nilesh Javali) [1899384]
- scsi: storvsc: Enable scatterlist entry lengths > 4Kbytes (Cathy Avery) [1941664]
- scsi: storvsc: Parameterize number hardware queues (Cathy Avery) [1941664]
- scsi: storvsc: Return DID_ERROR for invalid commands (Cathy Avery) [1941664]
- scsi: storvsc: Validate length of incoming packet in storvsc_on_channel_callback() (Cathy Avery) [1941664]
- scsi: storvsc: Resolve data race in storvsc_probe() (Cathy Avery) [1941664]
- scsi: storvsc: Fix max_outstanding_req_per_channel for Win8 and newer (Cathy Avery) [1941664]
- scsi: storvsc: Use vmbus_requestor to generate transaction IDs for VMBus hardening (Cathy Avery) [1941664]
- RDMA/ipoib: Fix warning caused by destroying non-initial netns (Kamal Heib) [1961815]
- can: dev: Move device back to init netns on owning netns delete (Kamal Heib) [1961815]
- net: ena: remove extra words from comments (Petr Oros) [1955155]
- net: ena: fix inaccurate print type (Petr Oros) [1955155]
- ena: Update driver to use ethtool_sprintf (Petr Oros) [1955155]
- docs: net: ena: Fix ena_start_xmit() function name typo (Petr Oros) [1955155]
- ethernet: amazon: ena: A typo fix in the file ena_com.h (Petr Oros) [1955155]
- net: ena: Update XDP verdict upon failure (Petr Oros) [1955155]
- net: ena: introduce ndo_xdp_xmit() function for XDP_REDIRECT (Petr Oros) [1955155]
- net: ena: use xdp_return_frame() to free xdp frames (Petr Oros) [1955155]
- net: ena: introduce XDP redirect implementation (Petr Oros) [1955155]
- net: ena: use xdp_frame in XDP TX flow (Petr Oros) [1955155]
- net: ena: aggregate stats increase into a function (Petr Oros) [1955155]
- net: ena: fix coding style nits (Petr Oros) [1955155]
- net: ena: store values in their appropriate variables types (Petr Oros) [1955155]
- net: ena: add device distinct log prefix to files (Petr Oros) [1955155]
- net: ena: use constant value for net_device allocation (Petr Oros) [1955155]
- net: ena: set initial DMA width to avoid intel iommu issue (Petr Oros) [1955155]
- net: ena: update ena documentation (Petr Oros) [1955155]
- net: ena: Fix all static chekers' warnings (Petr Oros) [1955155]
- net: ena: Change RSS related macros and variables names (Petr Oros) [1955155]
- net: ena: Remove redundant print of placement policy (Petr Oros) [1955155]
- net: ena: Capitalize all log strings and improve code readability (Petr Oros) [1955155]
- net: ena: Change log message to netif/dev function (Petr Oros) [1955155]
- net: ena: Change license into format to SPDX in all files (Petr Oros) [1955155]
- net: ena: xdp: add queue counters for xdp actions (Petr Oros) [1955155]
- net: ena: ethtool: add stats printing to XDP queues (Petr Oros) [1955155]
- net: ena: ethtool: Add new device statistics (Petr Oros) [1955155]
- net: ena: ethtool: convert stat_offset to 64 bit resolution (Petr Oros) [1955155]
- net: ena: Make missed_tx stat incremental (Petr Oros) [1955155]
- net: ena: Prevent reset after device destruction (Petr Oros) [1955155]
- treewide: replace '---help---' in Kconfig files with 'help' (Petr Oros) [1955155]
- treewide: Add SPDX license identifier - Makefile/Kconfig (Petr Oros) [1955155]
- cross-tree: phase out dma_zalloc_coherent() (Petr Oros) [1955155]
- net: ena: fix compilation error in xtensa architecture (Petr Oros) [1955155]
- docs: networking: reorganize driver documentation again (Petr Oros) [1955155]
- docs: networking: device drivers: convert amazon/ena.txt to ReST (Petr Oros) [1955155]
- net: ena: documentation: update ena.txt (Petr Oros) [1955155]
- net: documentation: build a directory structure for drivers (Petr Oros) [1955155]
- xfrm: interface: fix ipv4 pmtu check to honor ip header df (Sabrina Dubroca) [1931942]
- vti: fix ipv4 pmtu check to honor ip header df (Sabrina Dubroca) [1931942]
- vti6: fix ipv4 pmtu check to honor ip header df (Sabrina Dubroca) [1931942]
- net/sched: act_ct: Fix ct template allocation for zone 0 (Marcelo Ricardo Leitner) [1881824]
- uio: Fix use-after-free in uio_unregister_device() (Myron Stowe) [1897285]
- uio: free uio id after uio file node is freed (Myron Stowe) [1897285]
- drivers: uio: remove redundant assignment to variable retval (Myron Stowe) [1897285]
- driver: uio: fix possible use-after-free in __uio_register_device (Myron Stowe) [1897285]
- driver: uio: fix possible memory leak in __uio_register_device (Myron Stowe) [1897285]
- uio: remove redundant check (Myron Stowe) [1897285]
- uio: fix potential memory leak in error case (Myron Stowe) [1897285]
- uio: dismiss waiters on device unregistration (Myron Stowe) [1897285]
- uio: Fix an Oops on load (Myron Stowe) [1897285]
- uio: make symbol 'uio_class_registered' static (Myron Stowe) [1897285]
- uio: convert to vm_fault_t (Myron Stowe) [1897285]
- uio: ensure class is registered before devices (Myron Stowe) [1897285]
- uio: potential double frees if __uio_register_device() fails (Myron Stowe) [1897285]
- uio: fix possible circular locking dependency (Myron Stowe) [1897285]
- uio: fix wrong return value from uio_mmap() (Myron Stowe) [1897285]
- uio: add SPDX license tag (Myron Stowe) [1897285]
- net: zero-initialize tc skb extension on allocation (Ivan Vecera) [1946986]
- net: sched: fix tx action reschedule issue with stopped queue (Ivan Vecera) [1946986]
- net: sched: fix tx action rescheduling issue during deactivation (Ivan Vecera) [1946986]
- net: sched: fix packet stuck problem for lockless qdisc (Ivan Vecera) [1946986]
- net/sched: act_ct: Remove redundant ct get and check (Ivan Vecera) [1946986]
- net: sched: tapr: prevent cycle_time == 0 in parse_taprio_schedule (Ivan Vecera) [1946986]
- tc-testing: add simple action test to verify batch change cleanup (Ivan Vecera) [1946986]
- tc-testing: add simple action test to verify batch add cleanup (Ivan Vecera) [1946986]
- tc-testing: add simple action change test (Ivan Vecera) [1946986]
- selftests: forwarding: Add tc-police tests for packets per second (Ivan Vecera) [1946986]
- selftests: tc-testing: add action police selftest for packets per second (Ivan Vecera) [1946986]
- net: sched: Mundane typo fixes (Ivan Vecera) [1946986]
- net/sched: cls_flower: use nla_get_be32 for TCA_FLOWER_KEY_FLAGS (Ivan Vecera) [1946986]
- net/sched: cls_flower: use ntohs for struct flow_dissector_key_ports (Ivan Vecera) [1946986]
- taprio: Handle short intervals and large packets (Ivan Vecera) [1946986]
- net/sched: cls_flower: fix only mask bit check in the validate_ct_state (Ivan Vecera) [1946986]
- treewide: Remove uninitialized_var() usage (Ivan Vecera) [1946986]
- lockdep: Provide dummy forward declaration of *_is_held() helpers (Ivan Vecera) [1946986]
- net: sched: Remove broken definitions and un-hide for !LOCKDEP (Ivan Vecera) [1946986]
- net: sched: sch_teql: fix null-pointer dereference (Ivan Vecera) [1946986]
- net: sched: fix err handler in tcf_action_init() (Ivan Vecera) [1946986]
- net: sched: fix action overwrite reference counting (Ivan Vecera) [1946986]
- Revert "net: sched: bump refcount for new action in ACT replace mode" (Ivan Vecera) [1946986]
- net: cls_api: Fix uninitialised struct field bo->unlocked_driver_cb (Ivan Vecera) [1946986]
- net: sched: bump refcount for new action in ACT replace mode (Ivan Vecera) [1946986]
- net/sched: act_api: fix miss set post_ct for ovs after do conntrack in act_ct (Ivan Vecera) [1946986]
- net/sched: act_police: add support for packet-per-second policing (Ivan Vecera) [1946986]
- flow_offload: reject configuration of packet-per-second policing in offload drivers (Ivan Vecera) [1946986]
- flow_offload: add support for packet-per-second policing (Ivan Vecera) [1946986]
- net: sched: validate stab values (Ivan Vecera) [1946986]
- net: add a helper to avoid issues with HW TX timestamping and SO_TXTIME (Ivan Vecera) [1946986]
- sched: act_sample: Implement stats_update callback (Ivan Vecera) [1946986]
- net: sched: avoid duplicates in classes dump (Ivan Vecera) [1946986]
- net/sched: cls_flower: validate ct_state for invalid and reply flags (Ivan Vecera) [1946986]
- net: sched: fix police ext initialization (Ivan Vecera) [1946986]
- selftests: tc: Add generic mpls matching support for tc-flower (Ivan Vecera) [1946986]
- selftests: tc: Add basic mpls_* matching support for tc-flower (Ivan Vecera) [1946986]
- selftests: tc-testing: u32: Add tests covering sample option (Ivan Vecera) [1946986]
- net: sched: Return the correct errno code (Ivan Vecera) [1946986]
- net: sched: replaced invalid qdisc tree flush helper in qdisc_replace (Ivan Vecera) [1946986]
- selftests: forwarding: Fix spelling mistake "succeded" -> "succeeded" (Ivan Vecera) [1946986]
- taprio: boolean values to a bool variable (Ivan Vecera) [1946986]
- net_sched: fix RTNL deadlock again caused by request_module() (Ivan Vecera) [1946986]
- cls_flower: call nla_ok() before nla_next() (Ivan Vecera) [1946986]
- net/sched: sch_taprio: ensure to reset/destroy all child qdiscs (Ivan Vecera) [1946986]
- net/sched: sch_taprio: reset child qdiscs before freeing them (Ivan Vecera) [1946986]
- net: sched: incorrect Kconfig dependencies on Netfilter modules (Ivan Vecera) [1946986]
- treewide: replace '---help---' in Kconfig files with 'help' (Ivan Vecera) [1946986]
- net/sched: cls_u32: simplify the return expression of u32_reoffload_knode() (Ivan Vecera) [1946986]
- net: sched: fix spelling mistake in Kconfig "trys" -> "tries" (Ivan Vecera) [1946986]
- net/sched: fq_pie: initialize timer earlier in fq_pie_init() (Ivan Vecera) [1946986]
- net: sched: remove redundant 'rtnl_held' argument (Ivan Vecera) [1946986]
- selftests: tc-testing: enable CONFIG_NET_SCH_RED as a module (Ivan Vecera) [1946986]
- net: sched: alias action flags with TCA_ACT_ prefix (Ivan Vecera) [1946986]
- treewide: rename nla_strlcpy to nla_strscpy. (Ivan Vecera) [1946986]
- Modify return value of nla_strlcpy to match that of strscpy. (Ivan Vecera) [1946986]
- Fix unefficient call to memset before memcpu in nla_strlcpy. (Ivan Vecera) [1946986]
- net: sched: fix misspellings using misspell-fixer tool (Ivan Vecera) [1946986]
- selftest: fix flower terse dump tests (Ivan Vecera) [1946986]
- net: sched: implement action-specific terse dump (Ivan Vecera) [1946986]
- net: cls_api: remove unneeded local variable in tc_dump_chain() (Ivan Vecera) [1946986]
- openvswitch: fix send of uninitialized stack memory in ct limit reply (Antoine Tenart) [1950022]
- net/sched: act_api: fix miss set post_ct for ovs after do conntrack in act_ct (Antoine Tenart) [1950022]
- xsk: Clear pool even for inactive queues (Jiri Benc) [1882710]
- xsk: Fix memory leak for failed bind (Jiri Benc) [1882710]
- xsk: Return error code if force_zc is set (Jiri Benc) [1882710]
- xsk: Change the tx writeable condition (Jiri Benc) [1882710]
- xsk: Replace datagram_poll by sock_poll_wait (Jiri Benc) [1882710]
- xdp: Handle MEM_TYPE_XSK_BUFF_POOL correctly in xdp_return_buff() (Jiri Benc) [1882710]
- xsk: Fix incorrect netdev reference count (Jiri Benc) [1882710]
- xsk: Fix umem cleanup bug at socket destruct (Jiri Benc) [1882710]
- netdevsim: set .owner to THIS_MODULE (Jiri Benc) [1882710]
- bpf: selftest: Use static globals in tcp_hdr_options and btf_skc_cls_ingress (Jiri Benc) [1882710]
- xsk: Fix possible memory leak at socket close (Jiri Benc) [1882710]
- samples/bpf: Set rlimit for memlock to infinity in all samples (Jiri Benc) [1882710]
- bpf, selftests: Extend test_tc_redirect to use modified bpf_redirect_neigh() (Jiri Benc) [1882710]
- bpf: Fix bpf_redirect_neigh helper api to support supplying nexthop (Jiri Benc) [1882710]
- bpf, selftest: Fix flaky tcp_hdr_options test when adding addr to lo (Jiri Benc) [1882710]
- bpf, selftests: Add redirect_peer selftest (Jiri Benc) [1882710]
- bpf, selftests: Make redirect_neigh test more extensible (Jiri Benc) [1882710]
- bpf: Add redirect_peer helper (Jiri Benc) [1882710]
- bpf: Improve bpf_redirect_neigh helper description (Jiri Benc) [1882710]
- bpf: Always return target ifindex in bpf_fib_lookup (Jiri Benc) [1882710]
- xsk: Introduce padding between ring pointers (Jiri Benc) [1882710]
- xsk: Remove internal DMA headers (Jiri Benc) [1882710]
- bpf: selftest: Ensure the child sk inherited all bpf_sock_ops_cb_flags (Jiri Benc) [1882710]
- bpf: tcp: Do not limit cb_flags when creating child sk from listen sk (Jiri Benc) [1882710]
- bpf, selftests: Add redirect_neigh selftest (Jiri Benc) [1882710]
- bpf: Add redirect_neigh helper as redirect drop-in (Jiri Benc) [1882710]
- tools, bpf: Synchronise BPF UAPI header with tools (Jiri Benc) [1882710]
- bpf: Fix formatting in documentation for BPF helpers (Jiri Benc) [1882710]
- xsk: Fix a documentation mistake in xsk_queue.h (Jiri Benc) [1882710]
- xsk: Fix possible crash in socket_release when out-of-memory (Jiri Benc) [1882710]
- xsk: Fix refcount warning in xp_dma_map (Jiri Benc) [1882710]
- tcp: Only init congestion control if not initialized already (Jiri Benc) [1882710]
- xsk: Fix null check on error return path (Jiri Benc) [1882710]
- xsk: Fix possible segfault at xskmap entry insertion (Jiri Benc) [1882710]
- xsk: Fix possible segfault in xsk umem diagnostics (Jiri Benc) [1882710]
- xsk: Fix use-after-free in failed shared_umem bind (Jiri Benc) [1882710]
- xsk: Documentation for XDP_SHARED_UMEM between queues and netdevs (Jiri Benc) [1882710]
- xsk: Add shared umem support between devices (Jiri Benc) [1882710]
- xsk: Add shared umem support between queue ids (Jiri Benc) [1882710]
- xsk: i40e: ice: ixgbe: mlx5: Test for dma_need_sync earlier for better performance (Jiri Benc) [1882710]
- xsk: Rearrange internal structs for better performance (Jiri Benc) [1882710]
- xsk: Enable sharing of dma mappings (Jiri Benc) [1882710]
- xsk: Move addrs from buffer pool to umem (Jiri Benc) [1882710]
- xsk: Move xsk_tx_list and its lock to buffer pool (Jiri Benc) [1882710]
- xsk: Move queue_id, dev and need_wakeup to buffer pool (Jiri Benc) [1882710]
- xsk: Move fill and completion rings to buffer pool (Jiri Benc) [1882710]
- xsk: Create and free buffer pool independently from umem (Jiri Benc) [1882710]
- xsk: i40e: ice: ixgbe: mlx5: Rename xsk zero-copy driver interfaces (Jiri Benc) [1882710]
- xsk: i40e: ice: ixgbe: mlx5: Pass buffer pool to driver instead of umem (Jiri Benc) [1882710]
- tcp: bpf: Optionally store mac header in TCP_SAVE_SYN (Jiri Benc) [1882710]
- bpf: selftests: Tcp header options (Jiri Benc) [1882710]
- bpf: selftests: Add fastopen_connect to network_helpers (Jiri Benc) [1882710]
- bpf: tcp: Allow bpf prog to write and parse TCP header option (Jiri Benc) [1882710]
- bpf: sock_ops: Change some members of sock_ops_kern from u32 to u8 (Jiri Benc) [1882710]
- bpf: tcp: Add bpf_skops_hdr_opt_len() and bpf_skops_write_hdr_opt() (Jiri Benc) [1882710]
- bpf: tcp: Add bpf_skops_parse_hdr() (Jiri Benc) [1882710]
- bpf: tcp: Add bpf_skops_established() (Jiri Benc) [1882710]
- tcp: Add saw_unknown to struct tcp_options_received (Jiri Benc) [1882710]
- tcp: bpf: Add TCP_BPF_RTO_MIN for bpf_setsockopt (Jiri Benc) [1882710]
- tcp: bpf: Add TCP_BPF_DELACK_MAX setsockopt (Jiri Benc) [1882710]
- tcp: Use a struct to represent a saved_syn (Jiri Benc) [1882710]
- tcp: tcp_init_buffer_space can be static (Jiri Benc) [1882710]
- tcp: refactor setting the initial congestion window (Jiri Benc) [1882710]
- tcp: fix code style in tcp_recvmsg() (Jiri Benc) [1882710]
- tcp: duplicate clear of icsk_backoff (Jiri Benc) [1882710]
- tcp: move rx_opt & syn_data_acked init to tcp_disconnect() (Jiri Benc) [1882710]
- tcp: move tp->rack init to tcp_disconnect() (Jiri Benc) [1882710]
- tcp: move app_limited init to tcp_disconnect() (Jiri Benc) [1882710]
- tcp: move retrans_out, sacked_out, tlp_high_seq, last_oow_ack_time init to tcp_disconnect() (Jiri Benc) [1882710]
- tcp: do not clear urg_data in tcp_create_openreq_child (Jiri Benc) [1882710]
- tcp: move snd_cwnd & snd_cwnd_cnt init to tcp_disconnect() (Jiri Benc) [1882710]
- tcp: move mdev_us init to tcp_disconnect() (Jiri Benc) [1882710]
- tcp: do not clear srtt_us in tcp_create_openreq_child (Jiri Benc) [1882710]
- tcp: do not clear packets_out in tcp_create_openreq_child() (Jiri Benc) [1882710]
- tcp: move icsk_rto init to tcp_disconnect() (Jiri Benc) [1882710]
- tcp: do not set snd_ssthresh in tcp_create_openreq_child() (Jiri Benc) [1882710]
- net-veth: Add type safety to veth_xdp_to_ptr() and veth_ptr_to_xdp() (Jiri Benc) [1882710]
- net-tun: Eliminate two tun/xdp related function calls from vhost-net (Jiri Benc) [1882710]
- net-tun: Add type safety to tun_xdp_to_ptr() and tun_ptr_to_xdp() (Jiri Benc) [1882710]
- ipv4: Add helpers for neigh lookup for nexthop (Jiri Benc) [1882710]
- net: bpf: remove XDP_QUERY_XSK_UMEM enumerator (Jiri Benc) [1882710]
- tipc: better validate user input in tipc_nl_retrieve_key() (Xin Long) [1950093]
- tipc: simplify the finalize work queue (Xin Long) [1958456]
- tipc: wait and exit until all work queues are done (Xin Long) [1958456]
- [s390] KVM: s390: diag9c (directed yield) forwarding (Claudio Imbrenda) [1874105]
- kvm/svm: Reenable nesting (Dr. David Alan Gilbert) [1790949]
- powerpc/papr_scm: Reduce error severity if nvdimm stats inaccessible (Steve Best) [1965193]
- Revert "[redhat] redhat/configs: Update CONFIG_ARM64_CNP" (Mark Salter) [1949533]
- arm64: kernel: disable CNP on Carmel (Mark Salter) [1949533]

* Mon Jun 14 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-314.el8]
- redhat/configs: Enable CONFIG_MICROSOFT_MANA=m (Vitaly Kuznetsov) [1957820]
- net: mana: Use int to check the return value of mana_gd_poll_cq() (Vitaly Kuznetsov) [1957820]
- net: mana: fix PCI_HYPERV dependency (Vitaly Kuznetsov) [1957820]
- net: mana: remove redundant initialization of variable err (Vitaly Kuznetsov) [1957820]
- net: mana: Add a driver for Microsoft Azure Network Adapter (MANA) (Vitaly Kuznetsov) [1957820]
- hv_netvsc: Make netvsc/VF binding check both MAC and serial number (Vitaly Kuznetsov) [1957820]
- net: evaluate net.ipvX.conf.all.ignore_routes_with_linkdown (Balazs Nemeth) [1957223]
- net: evaluate net.ipv4.conf.all.proxy_arp_pvlan (Balazs Nemeth) [1957223]
- netfilter: nf_log_syslog: Unset bridge logger in pernet exit (Phil Sutter) [1938214]
- netfilter: nft_log: perform module load from nf_tables (Phil Sutter) [1938214]
- netfilter: nft_dynset: fix timeouts later than 23 days (Phil Sutter) [1938214]
- netfilter: nft_compat: make sure xtables destructors have run (Phil Sutter) [1938214]
- netfilter: nftables: Add __printf() attribute (Phil Sutter) [1938214]
- netfilter: nf_log: add module softdeps (Phil Sutter) [1938214]
- netfilter: nf_log_common: merge with nf_log_syslog (Phil Sutter) [1938214]
- netfilter: nf_log_bridge: merge with nf_log_syslog (Phil Sutter) [1938214]
- netfilter: nf_log_netdev: merge with nf_log_syslog (Phil Sutter) [1938214]
- netfilter: nf_log_ipv6: merge with nf_log_syslog (Phil Sutter) [1938214]
- netfilter: move inline nf_ip6_ext_hdr() function to a more appropriate header. (Phil Sutter) [1938214]
- ipv6: Use ipv6_authlen for len (Phil Sutter) [1938214]
- netfilter: nf_log_arp: merge with nf_log_syslog (Phil Sutter) [1938214]
- netfilter: nf_log_ipv4: rename to nf_log_syslog (Phil Sutter) [1938214]
- net/sched: cls_api: increase max_reclassify_loop (Davide Caratti) [1955136]
- vsock/vmci: Remove redundant assignment to err (Stefano Garzarella) [1956861]
- vsock/virtio: free queued packets when closing socket (Stefano Garzarella) [1956861]
- net: vsock: Fix a typo (Stefano Garzarella) [1956861]
- selinux: vsock: Set SID for socket returned by accept() (Stefano Garzarella) [1956861]
- af_vsock: Assign the vsock transport considering the vsock address flags (Stefano Garzarella) [1956861]
- af_vsock: Set VMADDR_FLAG_TO_HOST flag on the receive path (Stefano Garzarella) [1956861]
- vsock_addr: Check for supported flag values (Stefano Garzarella) [1956861]
- vm_sockets: Add VMADDR_FLAG_TO_HOST vsock flag (Stefano Garzarella) [1956861]
- vm_sockets: Add flags field in the vsock address data structure (Stefano Garzarella) [1956861]
- hv_sock: Remove the accept port restriction (Stefano Garzarella) [1956861]
- hv_sock: use HV_HYP_PAGE_SIZE for Hyper-V communication (Stefano Garzarella) [1956861]
- net: add a helper to avoid issues with HW TX timestamping and SO_TXTIME (Corinna Vinschen) [1958988]
- net: ensure mac header is set in virtio_net_hdr_to_skb() (Balazs Nemeth) [1955441]
- devlink: Fix dmac_filter trap name, align to its documentation (Petr Oros) [1935701]
- devlink: Add DMAC filter generic packet trap (Petr Oros) [1935701]
- devlink: Add blackhole_nexthop trap (Petr Oros) [1935701]
- stm class: Fix module init return on allocation failure (Jiri Olsa) [1907791]
- stm class: ftrace: Use different channel accroding to CPU (Jiri Olsa) [1907791]
- stm class: ftrace: Enable supported trace export flag (Jiri Olsa) [1907791]
- stm class: ftrace: Change dependency to TRACING (Jiri Olsa) [1907791]
- stm class: Replace zero-length array with flexible-array (Jiri Olsa) [1907791]
- stm class: sys-t: Fix the use of time_after() (Jiri Olsa) [1907791]
- stm class: Lose the protocol driver when dropping its reference (Jiri Olsa) [1907791]
- stm class: Fix a double free of stm_source_device (Jiri Olsa) [1907791]
- stm class: Fix channel bitmap on 32-bit systems (Jiri Olsa) [1907791]
- stm class: Fix channel free in stm output free path (Jiri Olsa) [1907791]
- stm class: Prevent division by zero (Jiri Olsa) [1907791]
- stm class: Fix an endless loop in channel allocation (Jiri Olsa) [1907791]
- stm class: Fix a module refcount leak in policy creation error path (Jiri Olsa) [1907791]
- stm class: Use memcat_p() (Jiri Olsa) [1907791]
- stm class: heartbeat: Fix whitespace (Jiri Olsa) [1907791]
- stm class: p_sys-t: Add support for CLOCKSYNC packets (Jiri Olsa) [1907791]
- stm class: Add MIPI SyS-T protocol support (Jiri Olsa) [1907791]
- stm class: Switch over to the protocol driver (Jiri Olsa) [1907791]
- stm class: Factor out default framing protocol (Jiri Olsa) [1907791]
- stm class: Add a helper for writing data packets (Jiri Olsa) [1907791]
- stm class: Introduce framing protocol drivers (Jiri Olsa) [1907791]
- stm class: Clean up stp_configfs_init (Jiri Olsa) [1907791]
- stm class: Clarify configfs root type/operations names (Jiri Olsa) [1907791]
- stm class: Rework policy node fallback (Jiri Olsa) [1907791]
- lib: Add memcat_p(): paste 2 pointer arrays together (Jiri Olsa) [1907791]
- tracing: Add trace_export support for trace_marker (Jiri Olsa) [1907791]
- tracing: Add trace_export support for event trace (Jiri Olsa) [1907791]
- tracing: Add flag to control different traces (Jiri Olsa) [1907791]
- tracing: Make function ‘ftrace_exports’ static (Jiri Olsa) [1907791]
- tipc: increment the tmp aead refcnt before attaching it (Xin Long) [1931312]
- tipc: skb_linearize the head skb when reassembling msgs (Xin Long) [1912376]
- scsi: target: core: Fix warning on realtime kernels (Maurizio Lombardi) [1956416]
- scsi: bnx2fc: Fix a typo (Nilesh Javali) [1921617]
- scsi: bnx2fc: Fix misnaming of bnx2fc_free_session_resc() (Nilesh Javali) [1921617]
- scsi: bnx2fc: Fix typo in bnx2fc_indicate_kcqe() (Nilesh Javali) [1921617]
- scsi: bnx2fc: Fix Kconfig warning & CNIC build errors (Nilesh Javali) [1921617]
- scsi: bnx2fc: Fix comparison to bool warning (Nilesh Javali) [1921617]
- scsi: bnx2fc: Remove unneeded semicolon (Nilesh Javali) [1921617]
- scsi: bnx2fc: Make a bunch of symbols static in bnx2fc_fcoe.c (Nilesh Javali) [1921617]
- scsi: bnx2fc: Fix spelling mistake "couldnt" -> "couldn't" (Nilesh Javali) [1921617]
- scsi: bnx2fc: Demote obvious misuse of kerneldoc to standard comment blocks (Nilesh Javali) [1921617]
- scsi: bnx2fc: Fix a couple of bitrotted function documentation headers (Nilesh Javali) [1921617]
- scsi: bnx2fc: Repair a range of kerneldoc issues (Nilesh Javali) [1921617]
- scsi: bnx2fc: Removal of unused variables (Nilesh Javali) [1921617]
- scsi: bnx2fc: Remove unneeded semicolon in bnx2fc_fcoe.c (Nilesh Javali) [1921617]
- scsi: bnx2fc: Add missing annotation for bnx2fc_abts_cleanup() (Nilesh Javali) [1921617]
- tools/power turbostat: Enable tsc_tweak for Elkhart Lake and Jasper Lake (Puneet Sethi) [1920704]
- selinux: Allow context mounts for unpriviliged overlayfs (Miklos Szeredi) [1956486]
- md-cluster: fix use-after-free issue when removing rdev (Nigel Croxon) [1956355]
- md/bitmap: wait for external bitmap writes to complete during tear down (Nigel Croxon) [1956355]
- md: do not return existing mddevs from mddev_find_or_alloc (Nigel Croxon) [1956355]
- md: refactor mddev_find_or_alloc (Nigel Croxon) [1956355]
- md: factor out a mddev_alloc_unit helper from mddev_find (Nigel Croxon) [1956355]
- md: split mddev_find (Nigel Croxon) [1956355]
- md: factor out a mddev_find_locked helper from mddev_find (Nigel Croxon) [1956355]
- md: md_open returns -EBUSY when entering racing area (Nigel Croxon) [1956355]
- md: Fix missing unused status line of /proc/mdstat (Nigel Croxon) [1956355]
- md/raid10: improve discard request for far layout (Xiao Ni) [1956355]
- md/raid10: improve raid10 discard request (Nigel Croxon) [1956355]
- md/raid10: pull the code that wait for blocked dev into one function (Xiao Ni) [1956355]
- md/raid10: extend r10bio devs to raid disks (Xiao Ni) [1956355]
- md: add md_submit_discard_bio() for submitting discard bio (Nigel Croxon) [1956355]
- leds: trigger: fix potential deadlock with libata (Waiman Long) [1963373]
- x86/kvm: Unify kvm_pv_guest_cpu_reboot() with kvm_guest_cpu_offline() (Lenny Szubowicz) [1934273]
- x86/kvm: Disable all PV features on crash (Lenny Szubowicz) [1934273]
- x86/kvm: Disable kvmclock on all CPUs on shutdown (Lenny Szubowicz) [1934273]
- x86/kvm: Teardown PV features on boot CPU as well (Lenny Szubowicz) [1934273]
- x86/kvm: Fix pr_info() for async PF setup/teardown (Lenny Szubowicz) [1934273]
- RDMA/qedr: Fix error return code in qedr_iw_connect() (Manish Chopra) [1914922]
- qede: Use 'skb_add_rx_frag()' instead of hand coding it (Manish Chopra) [1914922]
- qede: Remove a erroneous ++ in 'qede_rx_build_jumbo()' (Manish Chopra) [1914922]
- qede: fix offload for IPIP tunnel packets (Manish Chopra) [1914922]
- RDMA/qedr: iWARP invalid(zero) doorbell address fix (Manish Chopra) [1914922]
- qed: fix ILT configuration of SRC block (Manish Chopra) [1914922]
- qed: fix error return code in qed_iwarp_ll2_start() (Manish Chopra) [1914922]
- RDMA/qedr: Fix resource leak in qedr_create_qp (Manish Chopra) [1914922]
- qede: make driver reliable on unload after failures (Manish Chopra) [1914922]
- net: qed: Remove unnecessary cast (Manish Chopra) [1914922]
- nvme: fix NULL derefence in nvme_ctrl_fast_io_fail_tmo_show/store (Gopal Tiwari) [1935135]
- nvme: export fast_io_fail_tmo to sysfs (Gopal Tiwari) [1935135]
- EDAC: enable CONFIG_EDAC_IGEN6 (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC/igen6: ecclog_llist can be static (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC/i10nm: Add Intel Sapphire Rapids server support (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC: Add DDR5 new memory type (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC/i10nm: Use readl() to access MMIO registers (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC/igen6: Add debugfs interface for Intel client SoC EDAC driver (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC/igen6: Add EDAC driver for Intel client SoCs using IBECC (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC: Add three new memory types (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC, {skx,i10nm}: Use CPU stepping macro to pass configurations (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC/skx: Use the mcmtr register to retrieve close_pg/bank_xor_enable (Aristeu Rozanski) [1837389 1783539 1861479]
- x86/cpu: Add a X86_MATCH_INTEL_FAM6_MODEL_STEPPINGS() macro (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC: Add missing MEM_LRDDR4 entry in edac_mem_types[] (Aristeu Rozanski) [1837389 1783539 1861479]
- x86: export x86_match_cpu_v2 (Aristeu Rozanski) [1837389 1783539 1861479]
- EDAC/amd64: Fix PCI component registration (Aristeu Rozanski) [1918583]
- EDAC/mce_amd: Use struct cpuinfo_x86.cpu_die_id for AMD NodeId (Aristeu Rozanski) [1918583]
- x86/CPU/AMD: Remove amd_get_nb_id() (Aristeu Rozanski) [1918583]
- EDAC/mce_amd: Make fam_ops static global (Aristeu Rozanski) [1918583]
- x86/MCE/AMD: Allow Reserved types to be overwritten in smca_banks[] (Aristeu Rozanski) [1918583]
- x86/MCE/AMD: Do not use rdmsr_safe_on_cpu() in smca_configure() (Aristeu Rozanski) [1918583]
- EDAC/amd64: Get rid of the ECC disabled long message (Aristeu Rozanski) [1918583]
- EDAC/amd64: Check for memory before fully initializing an instance (Aristeu Rozanski) [1918583]
- EDAC/amd64: Use cached data when checking for ECC (Aristeu Rozanski) [1918583]
- x86/MCE: Make the number of MCA banks a per-CPU variable (Aristeu Rozanski) [1918583]
- x86/MCE/AMD: Don't cache block addresses on SMCA systems (Aristeu Rozanski) [1918583]
- x86/MCE: Make mce_banks a per-CPU array (Aristeu Rozanski) [1918583]
- x86/MCE: Make struct mce_banks[] static (Aristeu Rozanski) [1918583]
- x86/MCE/AMD: Don't report L1 BTB MCA errors on some family 17h models (Aristeu Rozanski) [1918583]
- EDAC/mce_amd: Decode MCA_STATUS in bit definition order (Aristeu Rozanski) [1918583]
- EDAC/mce_amd: Decode MCA_STATUS[Scrub] bit (Aristeu Rozanski) [1918583]
- EDAC, mce_amd: Match error descriptions to latest documentation (Aristeu Rozanski) [1918583]
- x86/MCE/AMD: Carve out the MC4_MISC thresholding quirk (Aristeu Rozanski) [1918583]
- x86/MCE/AMD: Turn off MC4_MISC thresholding on all family 0x15 models (Aristeu Rozanski) [1918583]
- powerpc/64s: Fix unrelocated interrupt trampoline address test (Waiman Long) [1966462]
- intel_th: pci: Add Alder Lake CPU support (Jiri Olsa) [1840246]
- intel_th: pci: Add Alder Lake-S support (Jiri Olsa) [1840246]
- mfd: intel_pmt: Add support for DG1 (David Arcari) [1783144]
- mfd: intel_pmt: Fix nuisance messages and handling of disabled capabilities (David Arcari) [1783144]
- redhat/configs: Add CONFIG_INTEL_PMT_CRASHLOG (David Arcari) [1783144]
- redhat/configs: Add CONFIG_INTEL_PMT_TELEMETRY (David Arcari) [1783144]
- redhat/configs: Add CONFIG_MFD_INTEL_PMT (David Arcari) [1783144]
- platform/x86: intel_pmt_crashlog: Add dependency on MFD_INTEL_PMT (David Arcari) [1783144]
- platform/x86: intel_pmt_telemetry: Add dependency on MFD_INTEL_PMT (David Arcari) [1783144]
- platform/x86: intel_pmt: Make INTEL_PMT_CLASS non-user-selectable (David Arcari) [1783144]
- platform/x86: intel_pmt_crashlog: Fix incorrect macros (David Arcari) [1783144]
- platform/x86: intel_pmt_class: Initial resource to 0 (David Arcari) [1783144]
- mfd: Standardise MFD_CELL_* helper names (David Arcari) [1783144]
- platform/x86: pmt: Fix a potential Oops on error in probe (David Arcari) [1783144]
- platform/x86: Intel PMT Crashlog capability driver (David Arcari) [1783144]
- platform/x86: Intel PMT Telemetry capability driver (David Arcari) [1783144]
- platform/x86: Intel PMT class driver (David Arcari) [1783144]
- mfd: Intel Platform Monitoring Technology support (David Arcari) [1783144]
- mfd: mfd-core: Ensure disabled devices are ignored without error (David Arcari) [1783144]
- mfd: core: Fix double-free in mfd_remove_devices_fn() (David Arcari) [1783144]
- mfd: mfd-core: Add mechanism for removal of a subset of children (David Arcari) [1783144]
- mfd: core: Add OF_MFD_CELL_REG() helper (David Arcari) [1783144]
- mfd: core: Fix formatting of MFD helpers (David Arcari) [1783144]
- mfd: core: Make a best effort attempt to match devices with the correct of_nodes (David Arcari) [1783144]
- mfd: mfd-core: Complete kerneldoc header for devm_mfd_add_devices() (David Arcari) [1783144]
- mfd: Constify properties in mfd_cell (David Arcari) [1783144]
- mfd: mfd-core: Honour Device Tree's request to disable a child-device (David Arcari) [1783144]
- mfd: mfd-core: Move pdev->mfd_cell creation back into mfd_add_device() (David Arcari) [1783144]
- mfd: mfd-core: Remove usage counting for .{en,dis}able() call-backs (David Arcari) [1783144]
- mfd: mfd-core: Protect against NULL call-back function pointer (David Arcari) [1783144]
- mfd: mfd-core: Remove mfd_clone_cell() (David Arcari) [1783144]
- mfd: Provide MACRO to declare commonly defined MFD cell attributes (David Arcari) [1783144]
- mfd: core: Set fwnode for created devices (David Arcari) [1783144]
- treewide: Replace GPLv2 boilerplate/reference with SPDX - rule 500 (David Arcari) [1783144]
- mfd: mfd-core: Document mfd_add_devices() (David Arcari) [1783144]
- PCI: Add defines for Designated Vendor-Specific Extended Capability (David Arcari) [1783144]
- tick/nohz: Update idle_exittime on actual idle exit (Phil Auld) [1962632]
- tick/nohz: Remove superflous check for CONFIG_VIRT_CPU_ACCOUNTING_NATIVE (Phil Auld) [1962632]
- tick/nohz: Conditionally restart tick on idle exit (Phil Auld) [1962632]
- x86, sched: Fix the AMD CPPC maximum performance value on certain AMD Ryzen generations (Phil Auld) [1960687]
- pinctrl: devicetree: Avoid taking direct reference to device name string (Aristeu Rozanski) [1922904] {CVE-2020-0427}
- redhat: ppc64: enable CONFIG_PPC_RTAS_FILTER (Aristeu Rozanski) [1906444] {CVE-2020-27777}
- powerpc/rtas: Fix typo of ibm,open-errinjct in RTAS filter (Aristeu Rozanski) [1906444] {CVE-2020-27777}
- powerpc/rtas: Restrict RTAS requests from userspace (Aristeu Rozanski) [1906444] {CVE-2020-27777}

* Thu Jun 10 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-313.el8]
- docs: vxlan: add info about device features (Petr Oros) [1941739]
- selftests: net: add a test for static UDP tunnel ports (Petr Oros) [1941739]
- netdevsim: support the static IANA VXLAN port flag (Petr Oros) [1941739]
- selftests: net: add a test for shared UDP tunnel info tables (Petr Oros) [1941739]
- netdevsim: shared UDP tunnel port table support (Petr Oros) [1941739]
- netdevsim: add warnings on unexpected UDP tunnel port errors (Petr Oros) [1941739]
- udp_tunnel: add the ability to share port tables (Petr Oros) [1941739]
- docs: networking: convert vxlan.txt to ReST (Petr Oros) [1941739]
- net: Treat __napi_schedule_irqoff() as __napi_schedule() on PREEMPT_RT (Ivan Vecera) [1960528]
- ethtool: fix incorrect datatype in set_eee ops (Balazs Nemeth) [1956698]
- ethtool: fix missing NLM_F_MULTI flag when dumping (Antoine Tenart) [1953847]
- y2038: remove CONFIG_64BIT_TIME (Waiman Long) [1965360]
- net/mlx4: Fix EEPROM dump support (Alaa Hleihel) [1915305]
- net/mlx4_en: update moderation when config reset (Alaa Hleihel) [1915305]
- net/mlx4_core: Add missed mlx4_free_cmd_mailbox() (Alaa Hleihel) [1915305]
- IB/mlx4: Use port iterator and validation APIs (Alaa Hleihel) [1915305]
- net/mlx4: Use effective interrupt affinity (Alaa Hleihel) [1915302 1915305]
- net/mlx4: Replace irq_to_desc() abuse (Alaa Hleihel) [1915302 1915305]
- net/mlx4: simplify the return expression of mlx4_init_srq_table() (Alaa Hleihel) [1915302 1915305]
- RDMA/mlx4: Remove bogus dev_base_lock usage (Alaa Hleihel) [1915302 1915305]
- net/mlx4: simplify the return expression of mlx4_init_cq_table() (Alaa Hleihel) [1915302 1915305]
- net/mlx4: Remove unused #define MAX_MSIX_P_PORT (Alaa Hleihel) [1915302 1915305]
- RDMA/mlx4: Enable querying AH for XRC QP types (Alaa Hleihel) [1915302 1915305]
- IB/mlx4: Fix fall-through warnings for Clang (Alaa Hleihel) [1915302 1915305]
- net/mlx4_en: Remove unused performance counters (Alaa Hleihel) [1915302 1915305]
- net/mlx4: Assign boolean values to a bool variable (Alaa Hleihel) [1915302 1915305]
- net/mlx4: Cleanup kernel-doc warnings (Alaa Hleihel) [1915302 1915305]
- net/mlx4_core : remove unneeded semicolon (Alaa Hleihel) [1915302 1915305]
- net/mlx4_core: Fix init_hca fields offset (Alaa Hleihel) [1915301 1915305]
- mlx4: handle non-napi callers to napi_poll (Alaa Hleihel) [1915301 1915305]
- mlx4: drivers/net/ethernet: clean up unused assignments (Alaa Hleihel) [1915301 1915305]
- mlx4: add pause frame stats (Alaa Hleihel) [1915301 1915305]
- mlx4: make sure to always set the port type (Alaa Hleihel) [1915301 1915305]
- net/mlx4_en: RX, Add a prefetch command for small L1_CACHE_BYTES (Alaa Hleihel) [1915301 1915305]
- cnic: remove redundant assignment to variable ret (Nilesh Javali) [1921619]
- net: cnic: fix spelling mistake "reserverd" -> "reserved" (Nilesh Javali) [1921619]
- cnic: Explicitly initialize all reference counts to 0. (Nilesh Javali) [1921619]
- cfg80211: fix locking in netlink owner interface destruction (Jarod Wilson) [1937053]
- brcmfmac: Avoid GFP_ATOMIC when GFP_KERNEL is enough (Jarod Wilson) [1937053]
- ath11k: fix warning in ath11k_mhi_config (Jarod Wilson) [1937053]
- ath11k: qmi: Fix spelling mistake "requeqst" -> "request" (Jarod Wilson) [1937053]
- ath10k: Fix ath10k_wmi_tlv_op_pull_peer_stats_info() unlock without lock (Jarod Wilson) [1937053]
- ath10k: Fix a use after free in ath10k_htc_send_bundle (Jarod Wilson) [1937053]
- ath9k: Fix error check in ath9k_hw_read_revisions() for PCI devices (Jarod Wilson) [1937053]
- mt76: mt7921: reinit wpdma during drv_own if necessary (Jarod Wilson) [1937053]
- mt76: move mt76_token_init in mt76_alloc_device (Jarod Wilson) [1937053]
- mt76: mt7921: fix a precision vs width bug in printk (Jarod Wilson) [1937053]
- mt76: mt7915: fix a precision vs width bug in printk (Jarod Wilson) [1937053]
- mt76: mt7615: fix a precision vs width bug in printk (Jarod Wilson) [1937053]
- mt76: mt7921: introduce mt7921_mcu_sta_add routine (Jarod Wilson) [1937053]
- mt76: mt7921: mt7921_stop should put device in fw_own state (Jarod Wilson) [1937053]
- mt76: connac: unschedule mac_work before going to sleep (Jarod Wilson) [1937053]
- mt76: mt7663: add awake and doze time accounting (Jarod Wilson) [1937053]
- mt76: mt7921: improve doze opportunity (Jarod Wilson) [1937053]
- mt76: mt7921: get rid of mcu_reset function pointer (Jarod Wilson) [1937053]
- mt76: mt7915: do not read rf value from efuse in flash mode (Jarod Wilson) [1937053]
- mt76: move token utilities in mt76 common module (Jarod Wilson) [1937053]
- mt76: move token_lock, token and token_count in mt76_dev (Jarod Wilson) [1937053]
- mt76: mt7921: fix possible invalid register access (Jarod Wilson) [1937053]
- mt76: mt7921: enable deep sleep when the device suspends (Jarod Wilson) [1937053]
- mt76: connac: introduce mt76_connac_mcu_set_deep_sleep utility (Jarod Wilson) [1937053]
- mt76: mt7921: introduce mt7921_wpdma_reinit_cond utility routine (Jarod Wilson) [1937053]
- mt76: mt7921: introduce mt7921_dma_{enable,disable} utilities (Jarod Wilson) [1937053]
- mt76: mt7921: introduce mt7921_wpdma_reset utility routine (Jarod Wilson) [1937053]
- mt76: mt7921: move mt7921_dma_reset in dma.c (Jarod Wilson) [1937053]
- mt76: mt7615: Fix a dereference of pointer sta before it is null checked (Jarod Wilson) [1937053]
- mt76: mt7921: enable sw interrupts (Jarod Wilson) [1937053]
- mt76: mt7921: add awake and doze time accounting (Jarod Wilson) [1937053]
- mt76: remove MT76_STATE_PM in tx path (Jarod Wilson) [1937053]
- mt76: connac: use waitqueue for runtime-pm (Jarod Wilson) [1937053]
- mt76: connac: do not run mt76_txq_schedule_all directly (Jarod Wilson) [1937053]
- mt76: mt7615: rely on pm refcounting in mt7615_led_set_config (Jarod Wilson) [1937053]
- mt76: connac: alaways wake the device before scanning (Jarod Wilson) [1937053]
- mt76: mt7921: get rid of useless MT76_STATE_PM in mt7921_mac_work (Jarod Wilson) [1937053]
- mt76: connac: remove MT76_STATE_PM in mac_tx_free (Jarod Wilson) [1937053]
- mt76: connac: check wake refcount in mcu_fw_pmctrl (Jarod Wilson) [1937053]
- mt76: connac: unschedule ps_work in mt76_connac_pm_wake (Jarod Wilson) [1937053]
- mt76: mt7663: rely on mt76_connac_pm_ref/mt76_connac_pm_unref in tx/rx napi (Jarod Wilson) [1937053]
- mt76: mt7921: rely on mt76_connac_pm_ref/mt76_connac_pm_unref in tx/rx napi (Jarod Wilson) [1937053]
- mt76: dma: add the capability to define a custom rx napi poll routine (Jarod Wilson) [1937053]
- mt76: mt7663: rely on mt76_connac_pm_ref/mt76_connac_pm_unref in tx path (Jarod Wilson) [1937053]
- mt76: mt7921: rely on mt76_connac_pm_ref/mt76_connac_pm_unref in tx path (Jarod Wilson) [1937053]
- mt76: connac: introduce wake counter for fw_pmctrl synchronization (Jarod Wilson) [1937053]
- mt76: mt7663: fix a race between mt7615_mcu_drv_pmctrl and mt7615_mcu_fw_pmctrl (Jarod Wilson) [1937053]
- mt76: mt7921: fix a race between mt7921_mcu_drv_pmctrl and mt7921_mcu_fw_pmctrl (Jarod Wilson) [1937053]
- mt76: mt7921: remove leftover function declaration (Jarod Wilson) [1937053]
- mt76: connac: move mcu_update_arp_filter in mt76_connac module (Jarod Wilson) [1937053]
- mt76: mt7921: do not use 0 as NULL pointer (Jarod Wilson) [1937053]
- mt76: mt7915: directly read per-rate tx power from registers (Jarod Wilson) [1937053]
- mt76: mt7915: rework the flow of txpower setting (Jarod Wilson) [1937053]
- mt76: mt7915: add support for DT rate power limits (Jarod Wilson) [1937053]
- mt76: mt7921: abort uncompleted scan by wifi reset (Jarod Wilson) [1937053]
- mt76: mt7921: add wifisys reset support in debugfs (Jarod Wilson) [1937053]
- mt76: mt7921: run mt7921_mcu_fw_log_2_host holding mt76 mutex (Jarod Wilson) [1937053]
- mt76: improve mcu error logging (Jarod Wilson) [1937053]
- mt76: mt7921: move hw configuration in mt7921_register_device (Jarod Wilson) [1937053]
- mt76: mt7915: add support for applying pre-calibration data (Jarod Wilson) [1937053]
- mt76: mt7615: load ROM patch before checking patch semaphore status (Jarod Wilson) [1937053]
- mt76: mt7615: fix entering driver-own state on mt7663 (Jarod Wilson) [1937053]
- mt76: mt7615: fix hardware error recovery for mt7663 (Jarod Wilson) [1937053]
- mt76: mt7921: add dumping Tx power table (Jarod Wilson) [1937053]
- mt76: introduce single-sku support for mt7663/mt7921 (Jarod Wilson) [1937053]
- mt76: mt7615: do not use mt7615 single-sku values for mt7663 (Jarod Wilson) [1937053]
- mt76: mt7615: implement support for using DT rate power limits (Jarod Wilson) [1937053]
- mt76: extend DT rate power limits to support 11ax devices (Jarod Wilson) [1937053]
- mt76: add functions for parsing rate power limits from DT (Jarod Wilson) [1937053]
- mt76: flush tx status queue on DMA reset (Jarod Wilson) [1937053]
- mt76: mt7921: add mt7921_dma_cleanup in mt7921_unregister_device (Jarod Wilson) [1937053]
- mt76: mt7615: use ieee80211_free_txskb() in mt7615_tx_token_put() (Jarod Wilson) [1937053]
- mt76: mt7615: only free skbs after mt7615_dma_reset() when reset happens (Jarod Wilson) [1937053]
- mt76: mt7915: only free skbs after mt7915_dma_reset() when reset happens (Jarod Wilson) [1937053]
- mt76: mt7915: fix memleak when mt7915_unregister_device() (Jarod Wilson) [1937053]
- mt76: mt7615: fix memleak when mt7615_unregister_device() (Jarod Wilson) [1937053]
- mt76: mt7915: fix rate setting of tx descriptor in testmode (Jarod Wilson) [1937053]
- mt76: mt7915: rework mt7915_tm_set_tx_len() (Jarod Wilson) [1937053]
- mt76: testmode: add support to send larger packet (Jarod Wilson) [1937053]
- rtlwifi: implement set_tim by update beacon content (Jarod Wilson) [1937053]
- rtw88: refine napi deinit flow (Jarod Wilson) [1937053]
- rtw88: Fix potential unrecoverable tx queue stop (Jarod Wilson) [1937053]
- rtw88: 8821c: Don't set RX_FLAG_DECRYPTED if packet has no encryption (Jarod Wilson) [1937053]
- rtw88: 8822c: debug: allow debugfs to enable/disable TXGAPK (Jarod Wilson) [1937053]
- rtw88: 8822c: Add gap-k calibration to improve long range performance (Jarod Wilson) [1937053]
- rtw88: 8822c: reorder macro position according to the register number (Jarod Wilson) [1937053]
- iwlwifi: Fix softirq/hardirq disabling in iwl_pcie_gen2_enqueue_hcmd() (Jarod Wilson) [1937053]
- cfg80211: scan: drop entry from hidden_list on overflow (Jarod Wilson) [1937053]
- wireless: fix spelling of A-MSDU in HE capabilities (Jarod Wilson) [1937053]
- wireless: align HE capabilities A-MPDU Length Exponent Extension (Jarod Wilson) [1937053]
- wireless: align some HE capabilities with the spec (Jarod Wilson) [1937053]
- mac80211: drop the connection if firmware crashed while in CSA (Jarod Wilson) [1937053]
- nl80211: Add new RSNXE related nl80211 extended features (Jarod Wilson) [1937053]
- mac80211: properly drop the connection in case of invalid CSA IE (Jarod Wilson) [1937053]
- mac80211: make ieee80211_vif_to_wdev work when the vif isn't in the driver (Jarod Wilson) [1937053]
- nl80211/cfg80211: add a flag to negotiate for LMR feedback in NDP ranging (Jarod Wilson) [1937053]
- ieee80211: add the values of ranging parameters max LTF total field (Jarod Wilson) [1937053]
- mac80211: aes_cmac: check crypto_shash_setkey() return value (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: remove extraneous indentation on if statement (Jarod Wilson) [1937053]
- mac80211: properly process TXQ management frames (Jarod Wilson) [1937053]
- cfg80211: constify ieee80211_get_response_rate return (Jarod Wilson) [1937053]
- cfg80211: fix a few kernel-doc warnings (Jarod Wilson) [1937053]
- cfg80211: fix an htmldoc warning (Jarod Wilson) [1937053]
- rtw88: 8822c: add CFO tracking (Jarod Wilson) [1937053]
- iwlwifi: pcie: don't enable BHs with IRQs disabled (Jarod Wilson) [1937053]
- rtw88: Fix array overrun in rtw_get_tx_power_params() (Jarod Wilson) [1937053]
- brcmfmac: Remove duplicate struct declaration (Jarod Wilson) [1937053]
- rtw88: update statistics to fw for fine-tuning performance (Jarod Wilson) [1937053]
- brcmfmac: A typo fix (Jarod Wilson) [1937053]
- rtl8xxxu: Simplify locking of a skb list accesses (Jarod Wilson) [1937053]
- mwifiex: Remove unneeded variable: "ret" (Jarod Wilson) [1937053]
- rtl8xxxu: Fix fall-through warnings for Clang (Jarod Wilson) [1937053]
- mwifiex: don't print SSID to logs (Jarod Wilson) [1937053]
- rtlwifi: rtl8192de: Use DEFINE_SPINLOCK() for spinlock (Jarod Wilson) [1937053]
- rtlwifi: Simplify locking of a skb list accesses (Jarod Wilson) [1937053]
- rtlwifi: remove rtl_get_tid_h (Jarod Wilson) [1937053]
- rtlwifi: rtl8188ee: remove redundant assignment of variable rtlpriv->btcoexist.reg_bt_sco (Jarod Wilson) [1937053]
- rtlwifi: remove redundant assignment to variable err (Jarod Wilson) [1937053]
- rtlwifi: Few mundane typo fixes (Jarod Wilson) [1937053]
- rtlwifi: 8821ae: upgrade PHY and RF parameters (Jarod Wilson) [1937053]
- iwlwifi: dbg: disable ini debug in 9000 family and below (Jarod Wilson) [1937053]
- iwlwifi: pcie: Change ma product string name (Jarod Wilson) [1937053]
- iwlwifi: fw: print out trigger delay when collecting data (Jarod Wilson) [1937053]
- iwlwifi: trans/pcie: defer transport initialisation (Jarod Wilson) [1937053]
- iwlwifi: bump FW API to 63 for AX devices (Jarod Wilson) [1937053]
- iwlwifi: mvm: don't WARN if we can't remove a time event (Jarod Wilson) [1937053]
- iwlwifi: mvm: don't disconnect immediately if we don't hear beacons after CSA (Jarod Wilson) [1937053]
- iwlwifi: rs-fw: don't support stbc for HE 160 (Jarod Wilson) [1937053]
- iwlwifi: warn on SKB free w/o op-mode (Jarod Wilson) [1937053]
- iwlwifi: mvm: add support for version 3 of LARI_CONFIG_CHANGE command. (Jarod Wilson) [1937053]
- iwlwifi: pcie: add ISR debug info for msix debug (Jarod Wilson) [1937053]
- iwlwifi: pcie: merge napi_poll_msix functions (Jarod Wilson) [1937053]
- iwlwifi: mvm: don't lock mutex in RCU critical section (Jarod Wilson) [1937053]
- iwlwifi: mvm: remove PS from lower rates. (Jarod Wilson) [1937053]
- iwlwifi: mvm: umac error table mismatch (Jarod Wilson) [1937053]
- iwlwifi: queue: avoid memory leak in reset flow (Jarod Wilson) [1937053]
- iwlwifi: don't warn if we can't wait for empty tx queues (Jarod Wilson) [1937053]
- iwlwifi: remove remaining software checksum code (Jarod Wilson) [1937053]
- iwlwifi: remove TCM events (Jarod Wilson) [1937053]
- iwlwifi: mvm: Use IWL_INFO in fw_reset_handshake() (Jarod Wilson) [1937053]
- iwlwifi: mvm: refactor ACPI DSM evaluation function (Jarod Wilson) [1937053]
- iwlwifi: mvm: support BIOS enable/disable for 11ax in Ukraine (Jarod Wilson) [1937053]
- iwlwifi: move iwl_configure_rxq to be used by other op_modes (Jarod Wilson) [1937053]
- iwlwifi: add new so-gf device (Jarod Wilson) [1937053]
- iwlwifi: mvm: enable PPAG in China (Jarod Wilson) [1937053]
- iwlwifi: mvm: Add support for 6GHz passive scan (Jarod Wilson) [1937053]
- iwlwifi: add ax201 killer device (Jarod Wilson) [1937053]
- iwlwifi: mvm: when associated with PMF, use protected NDP ranging negotiation (Jarod Wilson) [1937053]
- iwlwifi: mvm: clean up queue sync implementation (Jarod Wilson) [1937053]
- iwlwifi: mvm: write queue_sync_state only for sync (Jarod Wilson) [1937053]
- iwlwifi: mvm: responder: support responder config command version 8 (Jarod Wilson) [1937053]
- iwlwifi: mvm: support range request command version 12 (Jarod Wilson) [1937053]
- iwlwifi: pcie: make cfg vs. trans_cfg more robust (Jarod Wilson) [1937053]
- iwlwifi: pcie: clear only FH bits handle in the interrupt (Jarod Wilson) [1937053]
- iwlwifi: add 160Mhz to killer 1550 name (Jarod Wilson) [1937053]
- iwlwifi: change name to AX 211 and 411 family (Jarod Wilson) [1937053]
- iwlwifi: change step in so-gf struct (Jarod Wilson) [1937053]
- iwlwifi: pcie: Add support for Bz Family (Jarod Wilson) [1937053]
- iwlwifi: mvm: don't allow CSA if we haven't been fully associated (Jarod Wilson) [1937053]
- iwlwifi: pcie: normally grab NIC access for inflight-hcmd (Jarod Wilson) [1937053]
- iwlwifi: pcie: avoid unnecessarily taking spinlock (Jarod Wilson) [1937053]
- iwlwifi: mvm: enable TX on new CSA channel before disconnecting (Jarod Wilson) [1937053]
- mt76: mt7921: add rcu section in mt7921_mcu_tx_rate_report (Jarod Wilson) [1937053]
- mt76: mt7921: introduce MCU_EVENT_LP_INFO event parsing (Jarod Wilson) [1937053]
- mt76: mt7615: always add rx header translation tlv when adding stations (Jarod Wilson) [1937053]
- mt76: mt7615: add missing SPDX tag in mmio.c (Jarod Wilson) [1937053]
- mt76: mt7915: add mmio.c (Jarod Wilson) [1937053]
- mt76: report Rx timestamp (Jarod Wilson) [1937053]
- mt76: mt7921: remove 80+80 MHz support capabilities (Jarod Wilson) [1937053]
- mt76: fix potential DMA mapping leak (Jarod Wilson) [1937053]
- mt76: mt7921: reduce the data latency during hw scan (Jarod Wilson) [1937053]
- mt76: mt7921: fix the insmod hangs (Jarod Wilson) [1937053]
- mt76: mt7921: fix kernel crash when the firmware fails to download (Jarod Wilson) [1937053]
- mt76: mt7921: fix the dwell time control (Jarod Wilson) [1937053]
- mt76: mt7921: fix inappropriate WoW setup with the missing ARP informaiton (Jarod Wilson) [1937053]
- mt76: mt7921: introduce MT_WFDMA_DUMMY_CR definition (Jarod Wilson) [1937053]
- mt76: mt7915: add wifi subsystem reset (Jarod Wilson) [1937053]
- mt76: mt7915: fix mt7915_mcu_add_beacon (Jarod Wilson) [1937053]
- mt76: mt7615: fix .add_beacon_offload() (Jarod Wilson) [1937053]
- mt76: mt7921: rework mt7921_mcu_debug_msg_event routine (Jarod Wilson) [1937053]
- mt76: mt7921: always wake the device in mt7921_remove_interface (Jarod Wilson) [1937053]
- mt76: mt7921: fix key set/delete issue (Jarod Wilson) [1937053]
- mt76: mt7915: fix txpower init for TSSI off chips (Jarod Wilson) [1937053]
- mt76: mt7915: limit firmware log message printk to buffer length (Jarod Wilson) [1937053]
- mt76: mt7615: limit firmware log message printk to buffer length (Jarod Wilson) [1937053]
- mt76: mt7615: fix chip reset on MT7622 and MT7663e (Jarod Wilson) [1937053]
- mt76: mt7915: cleanup mcu tx queue in mt7915_dma_reset() (Jarod Wilson) [1937053]
- mt76: mt7915: keep mcu_add_bss_info enabled till interface removal (Jarod Wilson) [1937053]
- mt76: mt7615: keep mcu_add_bss_info enabled till interface removal (Jarod Wilson) [1937053]
- mt76: mt7622: trigger hif interrupt for system reset (Jarod Wilson) [1937053]
- mt76: mt7615: cleanup mcu tx queue in mt7615_dma_reset() (Jarod Wilson) [1937053]
- mt76: mt7615: only enable DFS test knobs for mt7615 (Jarod Wilson) [1937053]
- mt76: mt7921: properly configure rcpi adding a sta to the fw (Jarod Wilson) [1937053]
- mt76: connac: introduce mt76_sta_cmd_info data structure (Jarod Wilson) [1937053]
- mt76: mt7921: get rid of mt7921_mac_wtbl_lmac_addr (Jarod Wilson) [1937053]
- mt76: mt7915: refresh repeater entry MAC address when setting BSSID (Jarod Wilson) [1937053]
- mt76: mt7915: fix key set/delete issue (Jarod Wilson) [1937053]
- mt76: mt7921: remove duplicated macros in mcu.h (Jarod Wilson) [1937053]
- mt76: mt7921: remove redundant check on type (Jarod Wilson) [1937053]
- mt76: mt7921: remove leftovers from dbdc configuration (Jarod Wilson) [1937053]
- mt76: mt7921: add wifi reset support (Jarod Wilson) [1937053]
- mt76: dma: export mt76_dma_rx_cleanup routine (Jarod Wilson) [1937053]
- mt76: dma: introduce mt76_dma_queue_reset routine (Jarod Wilson) [1937053]
- mt76: mt7921: introduce __mt7921_start utility routine (Jarod Wilson) [1937053]
- mt76: mt7921: introduce mt7921_run_firmware utility routine. (Jarod Wilson) [1937053]
- mt76: introduce mcu_reset function pointer in mt76_mcu_ops structure (Jarod Wilson) [1937053]
- mt76: mt7921: reduce mcu timeouts for suspend, offload and hif_ctrl msg (Jarod Wilson) [1937053]
- mt76: mt7915: fix possible deadlock while mt7915_register_ext_phy() (Jarod Wilson) [1937053]
- mt76: mt7615: add missing capabilities for DBDC (Jarod Wilson) [1937053]
- mt76: mt7663s: fix the possible device hang in high traffic (Jarod Wilson) [1937053]
- mt76: mt7663s: make all of packets 4-bytes aligned in sdio tx aggregation (Jarod Wilson) [1937053]
- mt76: mt7663: fix when beacon filter is being applied (Jarod Wilson) [1937053]
- mt76: mt7615: check mcu returned values in mt7615_ops (Jarod Wilson) [1937053]
- mt76: mt7915: check mcu returned values in mt7915_ops (Jarod Wilson) [1937053]
- mt76: mt7915: fix txrate reporting (Jarod Wilson) [1937053]
- mt76: mt7915: fix rxrate reporting (Jarod Wilson) [1937053]
- mt76: mt76x0u: Add support for TP-Link T2UHP(UN) v1 (Jarod Wilson) [1937053]
- mt76: mt7915: fix PHY mode for DBDC (Jarod Wilson) [1937053]
- mt76: mt7915: stop ext_phy queue when mac reset happens (Jarod Wilson) [1937053]
- mt76: mt7915: fix CSA notification for DBDC (Jarod Wilson) [1937053]
- mt76: mt7615: stop ext_phy queue when mac reset happens (Jarod Wilson) [1937053]
- mt76: mt7615: fix CSA notification for DBDC (Jarod Wilson) [1937053]
- mt76: mt7915: add missing capabilities for DBDC (Jarod Wilson) [1937053]
- mt76: mt7921: check mcu returned values in mt7921_start (Jarod Wilson) [1937053]
- mt76: mt7921: fix the base of the dynamic remap (Jarod Wilson) [1937053]
- mt76: mt7921: fix the base of PCIe interrupt (Jarod Wilson) [1937053]
- mt76: mt7921: get rid of mt7921_sta_rc_update routine (Jarod Wilson) [1937053]
- mt76: check return value of mt76_txq_send_burst in mt76_txq_schedule_list (Jarod Wilson) [1937053]
- mt76: connac: fix kernel warning adding monitor interface (Jarod Wilson) [1937053]
- mt76: mt7915: fix mib stats counter reporting to mac80211 (Jarod Wilson) [1937053]
- mt76: mt7615: fix mib stats counter reporting to mac80211 (Jarod Wilson) [1937053]
- mt76: mt7615: remove hdr->fw_ver check (Jarod Wilson) [1937053]
- mt76: mt7615: fix TSF configuration (Jarod Wilson) [1937053]
- mt76: mt7921: fix stats register definitions (Jarod Wilson) [1937053]
- mt76: mt7921: remove unneeded semicolon (Jarod Wilson) [1937053]
- mt76: mt7915: fix aggr len debugfs node (Jarod Wilson) [1937053]
- mt76: mt7921: fix aggr length histogram (Jarod Wilson) [1937053]
- mt76: mt7615: fix memory leak in mt7615_coredump_work (Jarod Wilson) [1937053]
- mt76: mt7615: add support for rx decapsulation offload (Jarod Wilson) [1937053]
- mt76: mt7615: add rx checksum offload support (Jarod Wilson) [1937053]
- mt76: mt7615: enable hw rx-amsdu de-aggregation (Jarod Wilson) [1937053]
- mt76: connac: update sched_scan cmd usage (Jarod Wilson) [1937053]
- mt76: mt7921: add flush operation (Jarod Wilson) [1937053]
- mt76: mt7921: fixup rx bitrate statistics (Jarod Wilson) [1937053]
- mt76: connac: fix up the setting for ht40 mode in mt76_connac_mcu_uni_add_bss (Jarod Wilson) [1937053]
- mt76: mt7921: switch to new api for hardware beacon filter (Jarod Wilson) [1937053]
- mt76: mt7921: fix memory leak in mt7921_coredump_work (Jarod Wilson) [1937053]
- mt76: mt7921: fix suspend/resume sequence (Jarod Wilson) [1937053]
- mt76: use PCI_VENDOR_ID_MEDIATEK to avoid open coded (Jarod Wilson) [1937053]
- mt76: mt7915: always check return value from mt7915_mcu_alloc_wtbl_req (Jarod Wilson) [1937053]
- mt76: connac: always check return value from mt76_connac_mcu_alloc_wtbl_req (Jarod Wilson) [1937053]
- mt76: mt76x0: disable GTK offloading (Jarod Wilson) [1937053]
- mt76: always use WTBL_MAX_SIZE for tlv allocation (Jarod Wilson) [1937053]
- mt76: mt7921: removed unused definitions in mcu.h (Jarod Wilson) [1937053]
- mt76: mt7921: remove unnecessary variable (Jarod Wilson) [1937053]
- mt76: mt7921: enable random mac addr during scanning (Jarod Wilson) [1937053]
- mt76: mt7615: support loading EEPROM for MT7613BE (Jarod Wilson) [1937053]
- mt76: mt7915: fix tx skb dma unmap (Jarod Wilson) [1937053]
- mt76: mt7615: fix tx skb dma unmap (Jarod Wilson) [1937053]
- mt76: mt7615: fix key set/delete issues (Jarod Wilson) [1937053]
- mt76: mt7915: add support for rx decapsulation offload (Jarod Wilson) [1937053]
- mt76: mt7915: add rx checksum offload support (Jarod Wilson) [1937053]
- mt76: mt7915: enable hw rx-amsdu de-aggregation (Jarod Wilson) [1937053]
- mt76: add support for 802.3 rx frames (Jarod Wilson) [1937053]
- cfg80211: Remove wrong RNR IE validation check (Jarod Wilson) [1937053]
- mac80211: bail out if cipher schemes are invalid (Jarod Wilson) [1937053]
- mac80211: clear the beacon's CRC after channel switch (Jarod Wilson) [1937053]
- mac80211: Set priority and queue mapping for injected frames (Jarod Wilson) [1937053]
- mac80211: Allow concurrent monitor iface and ethernet rx decap (Jarod Wilson) [1937053]
- nl80211: Add interface to indicate TDLS peer's HE capability (Jarod Wilson) [1937053]
- cfg80211: allow specifying a reason for hw_rfkill (Jarod Wilson) [1937053]
- nl80211: better document CMD_ROAM behavior (Jarod Wilson) [1937053]
- mac80211: set sk_pacing_shift for 802.3 txpath (Jarod Wilson) [1937053]
- mac80211: don't apply flow control on management frames (Jarod Wilson) [1937053]
- nl80211: Add missing line in nl80211_fils_discovery_policy (Jarod Wilson) [1937053]
- mac80211: remove redundant assignment of variable result (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: remove unused variable 'mg' in minstrel_ht_next_jump_rate() (Jarod Wilson) [1937053]
- cfg80211: regulatory: use DEFINE_SPINLOCK() for spinlock (Jarod Wilson) [1937053]
- mac80211_hwsim: use DEFINE_SPINLOCK() for spinlock (Jarod Wilson) [1937053]
- rfkill: use DEFINE_SPINLOCK() for spinlock (Jarod Wilson) [1937053]
- mt7601u: enable TDLS support (Jarod Wilson) [1937053]
- mt7601u: fix always true expression (Jarod Wilson) [1937053]
- rtw88: Fix an error code in rtw_debugfs_set_rsvd_page() (Jarod Wilson) [1937053]
- rtw88: coex: fix A2DP stutters while WL busy + WL scan (Jarod Wilson) [1937053]
- rtw88: 8822c: add LC calibration for RTL8822C (Jarod Wilson) [1937053]
- rtw88: 8822c: update tx power limit table to RF v40.1 (Jarod Wilson) [1937053]
- rtw88: fix DIG min setting (Jarod Wilson) [1937053]
- rtw88: add flush hci support (Jarod Wilson) [1937053]
- rtw88: 8822c: support FW crash dump when FW crash (Jarod Wilson) [1937053]
- rtw88: remove unnecessary variable (Jarod Wilson) [1937053]
- rtw88: coex: add power off setting (Jarod Wilson) [1937053]
- ath11k: Add support for STA to handle beacon miss (Jarod Wilson) [1937053]
- ath11k: Update signal filled flag during sta_statistics drv op (Jarod Wilson) [1937053]
- ath10k: skip the wait for completion to recovery in shutdown path (Jarod Wilson) [1937053]
- ath11k: fix thermal temperature read (Jarod Wilson) [1937053]
- ath11k: fix potential wmi_mgmt_tx_queue race condition (Jarod Wilson) [1937053]
- ath11k: qmi: cosmetic changes to error messages (Jarod Wilson) [1937053]
- ath11k: qmi: add more debug messages (Jarod Wilson) [1937053]
- Revert "ath9k: fix ath_tx_process_buffer() potential null ptr dereference" (Jarod Wilson) [1937053]
- ath11k: print hardware name and version during initialisation (Jarod Wilson) [1937053]
- ath11k: Enable radar detection for 160MHz secondary segment (Jarod Wilson) [1937053]
- ath11k: Fix sounding dimension config in HE cap (Jarod Wilson) [1937053]
- ath11k: add qcn9074 pci device support (Jarod Wilson) [1937053]
- ath11k: add extended interrupt support for QCN9074 (Jarod Wilson) [1937053]
- ath11k: add CE interrupt support for QCN9074 (Jarod Wilson) [1937053]
- ath11k: add data path support for QCN9074 (Jarod Wilson) [1937053]
- ath11k: add hal support for QCN9074 (Jarod Wilson) [1937053]
- ath11k: add static window support for register access (Jarod Wilson) [1937053]
- ath11k: Add qcn9074 mhi controller config (Jarod Wilson) [1937053]
- ath11k: Update memory segment count for qcn9074 (Jarod Wilson) [1937053]
- ath11k: qmi: increase the number of fw segments (Jarod Wilson) [1937053]
- ath11k: Move qmi service_ins_id to hw_params (Jarod Wilson) [1937053]
- ath11k: Refactor ath11k_msi_config (Jarod Wilson) [1937053]
- ath11k: debugfs: Fix spelling mistake "Opportunies" -> "Opportunities" (Jarod Wilson) [1937053]
- ath9k: fix ath_tx_process_buffer() potential null ptr dereference (Jarod Wilson) [1937053]
- nl80211: fix beacon head validation (Jarod Wilson) [1937053]
- nl80211: fix potential leak of ACL params (Jarod Wilson) [1937053]
- cfg80211: check S1G beacon compat element length (Jarod Wilson) [1937053]
- cfg80211: remove WARN_ON() in cfg80211_sme_connect (Jarod Wilson) [1937053]
- mac80211: fix time-is-after bug in mlme (Jarod Wilson) [1937053]
- mac80211: fix TXQ AC confusion (Jarod Wilson) [1937053]
- rfkill: revert back to old userspace API by default (Jarod Wilson) [1937053]
- mac80211: clear sta->fast_rx when STA removed from 4-addr VLAN (Jarod Wilson) [1937053]
- iwlwifi: mvm: fix beacon protection checks (Jarod Wilson) [1937053]
- iwlwifi: mvm: rfi: don't lock mvm->mutex when sending config command (Jarod Wilson) [1937053]
- iwlwifi: pcie: add support for So-F devices (Jarod Wilson) [1937053]
- iwlwifi: fw: fix notification wait locking (Jarod Wilson) [1937053]
- iwlwifi: add support for Qu with AX201 device (Jarod Wilson) [1937053]
- iwlwifi: pcie: properly set LTR workarounds on 22000 devices (Jarod Wilson) [1937053]
- iwlwifi: fix 11ax disabled bit in the regulatory capability flags (Jarod Wilson) [1937053]
- brcmfmac: p2p: Fix recently introduced deadlock issue (Jarod Wilson) [1937053]
- mt76: mt7921: fix airtime reporting (Jarod Wilson) [1937053]
- iwlwifi: Fix softirq/hardirq disabling in iwl_pcie_enqueue_hcmd() (Jarod Wilson) [1937053]
- mt76: mt7615: remove redundant dev_err call in mt7622_wmac_probe() (Jarod Wilson) [1937053]
- mt76: Convert to DEFINE_SHOW_ATTRIBUTE (Jarod Wilson) [1937053]
- reg.c: Fix a spello (Jarod Wilson) [1937053]
- mac80211: cfg.c: A typo fix (Jarod Wilson) [1937053]
- wireless: remove never implemented MODULE_SUPPORTED_DEVICE (Jarod Wilson) [1937053]
- wireless/nl80211: fix wdev_id may be used uninitialized (Jarod Wilson) [1937053]
- nl80211: fix locking for wireless device netns change (Jarod Wilson) [1937053]
- iwlwifi: don't call netif_napi_add() with rxq->lock held (was Re: Lockdep warning in iwl_pcie_rx_handle()) (Jarod Wilson) [1937053]
- iwlwifi: fix ARCH=i386 compilation warnings (Jarod Wilson) [1937053]
- iwlwifi: mvm: add terminate entry for dmi_system_id tables (Jarod Wilson) [1937053]
- mt76: mt7915: fix unused 'mode' variable (Jarod Wilson) [1937053]
- mt76: dma: do not report truncated frames to mac80211 (Jarod Wilson) [1937053]
- mt76: mt7921: remove incorrect error handling (Jarod Wilson) [1937053]
- iwlwifi: pcie: fix iwl_so_trans_cfg link error when CONFIG_IWLMVM is disabled (Jarod Wilson) [1937053]
- ath11k: fix AP mode for QCA6390 (Jarod Wilson) [1937053]
- ath11k: qmi: use pad to format dma_addr_t (Jarod Wilson) [1937053]
- iwlwifi: avoid crash on unsupported debug collection (Jarod Wilson) [1937053]
- mt76: mt7915: only modify tx buffer list after allocating tx token id (Jarod Wilson) [1937053]
- mt76: fix tx skb error handling in mt76_dma_tx_queue_skb (Jarod Wilson) [1937053]
- ath9k: fix transmitting to stations in dynamic SMPS mode (Jarod Wilson) [1937053]
- redhat: Enable ath11k PCI device support (Jarod Wilson) [1937053]
- soc: qcom: Do not depend on ARCH_QCOM for QMI helpers (Jarod Wilson) [1937053]
- net: qrtr: Do not depend on ARCH_QCOM (Jarod Wilson) [1937053]
- b43: N-PHY: Fix the update of coef for the PHY revision >= 3case (Jarod Wilson) [1937053]
- nl80211: add documentation for HT/VHT/HE disable attributes (Jarod Wilson) [1937053]
- cfg80211/mac80211: Support disabling HE mode (Jarod Wilson) [1937053]
- mac80211: add STBC encoding to ieee80211_parse_tx_radiotap (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: remove sample rate switching code for constrained devices (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: show sampling rates in debugfs (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: significantly redesign the rate probing strategy (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: reduce the need to sample slower rates (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: update total packets counter in tx status path (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: use bitfields to encode rate indexes (Jarod Wilson) [1937053]
- cfg80211: initialize reg_rule in __freq_reg_info() (Jarod Wilson) [1937053]
- mac80211: fix potential overflow when multiplying to u32 integers (Jarod Wilson) [1937053]
- mac80211: enable QoS support for nl80211 ctrl port (Jarod Wilson) [1937053]
- cfg80211: remove unused callback (Jarod Wilson) [1937053]
- rtw88: 8822c: update RF_B (2/2) parameter tables to v60 (Jarod Wilson) [1937053]
- rtw88: 8822c: update RF_B (1/2) parameter tables to v60 (Jarod Wilson) [1937053]
- rtw88: 8822c: update RF_A parameter tables to v60 (Jarod Wilson) [1937053]
- rtw88: 8822c: update MAC/BB parameter tables to v60 (Jarod Wilson) [1937053]
- rtw88: replace tx tasklet with work queue (Jarod Wilson) [1937053]
- rtw88: add napi support (Jarod Wilson) [1937053]
- rtw88: add rts condition (Jarod Wilson) [1937053]
- rtw88: add dynamic rrsr configuration (Jarod Wilson) [1937053]
- iwlwifi: remove incorrect comment in pnvm (Jarod Wilson) [1937053]
- ath11k: qmi: add debug message for allocated memory segment addresses and sizes (Jarod Wilson) [1937053]
- ath11k: pci: remove experimental warning (Jarod Wilson) [1937053]
- ath10k: hold RCU lock when calling ieee80211_find_sta_by_ifaddr() (Jarod Wilson) [1937053]
- ath10k: change ath10k_offchan_tx_work() peer present msg to a warn (Jarod Wilson) [1937053]
- ath9k: fix data bus crash when setting nf_override via debugfs (Jarod Wilson) [1937053]
- ath11k: add support to configure spatial reuse parameter set (Jarod Wilson) [1937053]
- ath10k: restore tx sk_buff of htt header for SDIO (Jarod Wilson) [1937053]
- brcmsmac: Fix the spelling configation to configuration in the file d11.h (Jarod Wilson) [1937053]
- iwlwifi: bump FW API to 62 for AX devices (Jarod Wilson) [1937053]
- iwlwifi: pnvm: implement reading PNVM from UEFI (Jarod Wilson) [1937053]
- iwlwifi: pnvm: move file loading code to a separate function (Jarod Wilson) [1937053]
- iwlwifi: pnvm: increment the pointer before checking the TLV (Jarod Wilson) [1937053]
- iwlwifi: pcie: define FW_RESET_TIMEOUT for clarity (Jarod Wilson) [1937053]
- iwlwifi: pnvm: set the PNVM again if it was already loaded (Jarod Wilson) [1937053]
- iwlwifi: mvm: global PM mode does not reset after FW crash (Jarod Wilson) [1937053]
- iwlwifi: mvm: reduce the print severity of failing getting NIC temp (Jarod Wilson) [1937053]
- iwlwifi: mvm: get NVM later in the mvm_start flow (Jarod Wilson) [1937053]
- iwlmvm: set properly NIC_NOT_ACK_ENABLED flag (Jarod Wilson) [1937053]
- iwlwifi: remove max_ht_ampdu_exponent config parameter (Jarod Wilson) [1937053]
- iwlwifi: remove max_vht_ampdu_exponent config parameter (Jarod Wilson) [1937053]
- iwlwifi: mvm: Check ret code for iwl_mvm_load_nvm_to_nic (Jarod Wilson) [1937053]
- iwlwifi: mvm: don't check if CSA event is running before removing (Jarod Wilson) [1937053]
- iwlwifi: mvm: Support SCAN_CFG_CMD version 5 (Jarod Wilson) [1937053]
- iwlwifi: mvm: isolate the get nvm flow (Jarod Wilson) [1937053]
- iwlwifi: mvm: simplify iwl_mvm_dbgfs_register (Jarod Wilson) [1937053]
- iwlwifi: mvm: register to mac80211 last (Jarod Wilson) [1937053]
- iwlwifi: acpi: add support for DSM RFI (Jarod Wilson) [1937053]
- iwlwifi: mvm: add RFI-M support (Jarod Wilson) [1937053]
- iwlwifi:mvm: Add support for version 2 of the LARI_CONFIG_CHANGE command. (Jarod Wilson) [1937053]
- iwlwifi: pcie: don't crash when rx queues aren't allocated in interrupt (Jarod Wilson) [1937053]
- iwlwifi: correction of group-id once sending REPLY_ERROR (Jarod Wilson) [1937053]
- iwlwifi: pcie: add AX201 and AX211 radio modules for Ma devices (Jarod Wilson) [1937053]
- iwlwifi: pcie: add CDB bit to the device configuration parsing (Jarod Wilson) [1937053]
- iwlwifi: acpi: don't return valid pointer as an ERR_PTR (Jarod Wilson) [1937053]
- iwlwifi: queue: add fake tx time point (Jarod Wilson) [1937053]
- iwlwifi: remove flags argument for nic_access (Jarod Wilson) [1937053]
- iwlwifi: declare support for triggered SU/MU beamforming feedback (Jarod Wilson) [1937053]
- iwlwifi: dbg: add op_mode callback for collecting debug data. (Jarod Wilson) [1937053]
- iwlwifi: api: clean up some documentation/bits (Jarod Wilson) [1937053]
- iwlwifi: dbg: remove unsupported regions (Jarod Wilson) [1937053]
- iwlwifi: pcie: Change Ma device ID (Jarod Wilson) [1937053]
- iwlwifi: when HW has rate offload don't look at control field (Jarod Wilson) [1937053]
- iwlwifi: pcie: NULLify pointers after free (Jarod Wilson) [1937053]
- iwlwifi: mvm: assign SAR table revision to the command later (Jarod Wilson) [1937053]
- iwlwifi: mvm: remove useless iwl_mvm_resume_d3() function (Jarod Wilson) [1937053]
- iwlwifi: mvm: enhance a print in CSA flows (Jarod Wilson) [1937053]
- iwlwifi: mvm: send stored PPAG command instead of local (Jarod Wilson) [1937053]
- iwlwifi: mvm: store PPAG enabled/disabled flag properly (Jarod Wilson) [1937053]
- iwlwifi: mvm: fix the type we use in the PPAG table validity checks (Jarod Wilson) [1937053]
- iwlwifi: acpi: fix PPAG table sizes (Jarod Wilson) [1937053]
- iwlwifi: pcie: don't disable interrupts for reg_lock (Jarod Wilson) [1937053]
- iwlwifi: pcie: add a few missing entries for So with Hr (Jarod Wilson) [1937053]
- iwlwifi: dbg: Mark ucode tlv data as const (Jarod Wilson) [1937053]
- iwlwifi: add new cards for So and Qu family (Jarod Wilson) [1937053]
- ath11k: fix a locking bug in ath11k_mac_op_start() (Jarod Wilson) [1937053]
- rtlwifi: rtl8821ae: phy: Simplify bool comparison (Jarod Wilson) [1937053]
- rtlwifi: rtl8192se: Simplify bool comparison (Jarod Wilson) [1937053]
- brcmfmac: add support for CQM RSSI notifications (Jarod Wilson) [1937053]
- ath10k: Add new debug level for sta related logs (Jarod Wilson) [1937053]
- ath11k: Update tx descriptor search index properly (Jarod Wilson) [1937053]
- ath10k: Fix lockdep assertion warning in ath10k_sta_statistics (Jarod Wilson) [1937053]
- ath10k: Fix suspicious RCU usage warning in ath10k_wmi_tlv_parse_peer_stats_info() (Jarod Wilson) [1937053]
- wl3501: fix alignment constraints (Jarod Wilson) [1937053]
- rt2800usb: add Sweex LW163V2 id's (Jarod Wilson) [1937053]
- rt2x00: remove duplicate word and fix typo in comment (Jarod Wilson) [1937053]
- mwifiex: Report connected BSS with cfg80211_connect_bss() (Jarod Wilson) [1937053]
- rtl8xxxu: remove unused assignment value (Jarod Wilson) [1937053]
- atmel: at76c50x: use DEFINE_MUTEX() for mutex lock (Jarod Wilson) [1937053]
- brcmsmac: fix alignment constraints (Jarod Wilson) [1937053]
- brcmfmac: Add DMI nvram filename quirk for Voyo winpad A15 tablet (Jarod Wilson) [1937053]
- brcmfmac: Add DMI nvram filename quirk for Predia Basic tablet (Jarod Wilson) [1937053]
- rtw88: 8821c: support RFE type2 wifi NIC (Jarod Wilson) [1937053]
- rtw88: 8821c: Correct CCK RSSI (Jarod Wilson) [1937053]
- rtw88: coex: 8821c: correct antenna switch function (Jarod Wilson) [1937053]
- rtlwifi: rtl8192se: remove redundant initialization of variable rtstatus (Jarod Wilson) [1937053]
- iwlwifi: pcie: Disable softirqs during Rx queue init (Jarod Wilson) [1937053]
- iwlwifi: bump FW API to 61 for AX devices (Jarod Wilson) [1937053]
- iwlwifi: mvm: add Asus to the PPAG approved list (Jarod Wilson) [1937053]
- iwlwifi: mvm: add Microsoft to the PPAG approved list (Jarod Wilson) [1937053]
- iwlwifi: mvm: add Samsung to the PPAG approved list (Jarod Wilson) [1937053]
- iwlwifi: mvm: add HP to the PPAG approved list (Jarod Wilson) [1937053]
- iwlwifi: mvm: implement approved list for the PPAG feature (Jarod Wilson) [1937053]
- iwlwifi: mvm: set enabled in the PPAG command properly (Jarod Wilson) [1937053]
- iwlwifi: mvm: add debugfs entry to trigger a dump as any time-point (Jarod Wilson) [1937053]
- iwlwifi: mvm: add tx fail time point (Jarod Wilson) [1937053]
- iwlwifi: fwrt: add suspend/resume time point (Jarod Wilson) [1937053]
- iwlwifi: mvm: add triggers for MLME events (Jarod Wilson) [1937053]
- iwlwifi: mvm: add IML/ROM information for other HW families (Jarod Wilson) [1937053]
- iwlwifi: mvm: fix CSA AP side (Jarod Wilson) [1937053]
- iwlwifi: mvm: make iwl_mvm_tt_temp_changed() static (Jarod Wilson) [1937053]
- iwlwifi: mvm: cancel the scan delayed work when scan is aborted (Jarod Wilson) [1937053]
- iwlwifi: pcie: add support for SnJ with Hr1 (Jarod Wilson) [1937053]
- iwlwifi: mvm: move early time-point before nvm_init in non-unified (Jarod Wilson) [1937053]
- iwlwifi: add support for SnJ with Jf devices (Jarod Wilson) [1937053]
- iwlwifi: move SnJ and So rules to the new tables (Jarod Wilson) [1937053]
- iwlwifi: mvm: slightly clean up rs_fw_set_supp_rates() (Jarod Wilson) [1937053]
- iwlwifi: fw api: make hdr a zero-size array again (Jarod Wilson) [1937053]
- iwlwifi: bump FW API to 60 for AX devices (Jarod Wilson) [1937053]
- iwlwifi: mvm: advertise BIGTK client support if available (Jarod Wilson) [1937053]
- iwlwifi: always allow maximum A-MSDU on newer devices (Jarod Wilson) [1937053]
- iwlwifi: mvm: debugfs: check length precisely in inject_packet (Jarod Wilson) [1937053]
- iwlwifi: mvm: simplify TX power setting (Jarod Wilson) [1937053]
- iwlwifi: tx: move handing sync/async host command to trans (Jarod Wilson) [1937053]
- iwlwifi: mvm: add explicit check for non-data frames in get Tx rate (Jarod Wilson) [1937053]
- iwlwifi: mvm: debugfs for phy-integration-ver (Jarod Wilson) [1937053]
- iwlwifi: parse phy integration string from FW TLV (Jarod Wilson) [1937053]
- iwlwifi: mvm: csa: do not abort CSA before disconnect (Jarod Wilson) [1937053]
- iwlwifi: mvm: don't send commands during suspend\resume transition (Jarod Wilson) [1937053]
- iwlwifi: mvm: don't check system_pm_mode without mutex held (Jarod Wilson) [1937053]
- iwlwifi: remove TRANS_PM_OPS (Jarod Wilson) [1937053]
- iwlwifi: dbg: dump paged memory from index 1 (Jarod Wilson) [1937053]
- iwl-trans: iwlwifi: move sync NMI logic to trans (Jarod Wilson) [1937053]
- iwlwifi: pcie: properly implement NAPI (Jarod Wilson) [1937053]
- iwlwifi: mvm: add support for new flush queue response (Jarod Wilson) [1937053]
- iwlwifi: mvm: handle CCA-EXT delay firmware notification (Jarod Wilson) [1937053]
- iwlwifi: mvm: scan: fix scheduled scan restart handling (Jarod Wilson) [1937053]
- iwlwifi: mvm: remove debugfs injection limitations (Jarod Wilson) [1937053]
- iwlwifi: mvm: check more notification sizes (Jarod Wilson) [1937053]
- iwlwifi: mvm: add notification size checks (Jarod Wilson) [1937053]
- ath11k: remove h from printk format specifier (Jarod Wilson) [1937053]
- ath10k: remove h from printk format specifier (Jarod Wilson) [1937053]
- cfg80211: fix netdev registration deadlock (Jarod Wilson) [1937053]
- mac80211: fix station rate table updates on assoc (Jarod Wilson) [1937053]
- mt76: mt7663: introduce coredump support (Jarod Wilson) [1937053]
- mt76: mt7921: add coredump support (Jarod Wilson) [1937053]
- mt76: mt7921: enable MSI interrupts (Jarod Wilson) [1937053]
- mt76: mt7921: introduce regdomain notifier support (Jarod Wilson) [1937053]
- mt76: mt7921: introduce Runtime PM support (Jarod Wilson) [1937053]
- mt76: mt7921: rely on mt76_connac_mcu module for suspend and WoW support (Jarod Wilson) [1937053]
- mt76: mt7921: rely on mt76_connac_mcu module for sched_scan and hw_scan (Jarod Wilson) [1937053]
- mt76: mt7921: rely on mt76_connac_mcu common library (Jarod Wilson) [1937053]
- mt76: mt7921: introduce PM support (Jarod Wilson) [1937053]
- mt76: mt7921: introduce beacon_loss mcu event (Jarod Wilson) [1937053]
- mt76: mt7921: introduce support for hardware beacon filter (Jarod Wilson) [1937053]
- mt76: mt7921: introduce 802.11 PS support in sta mode (Jarod Wilson) [1937053]
- mt76: mt7921: introduce schedule scan support (Jarod Wilson) [1937053]
- mt76: mt7921: add debugfs support (Jarod Wilson) [1937053]
- mt76: mt7921: introduce mt7921e support (Jarod Wilson) [1937053]
- mt76: mt7921: add ieee80211_ops (Jarod Wilson) [1937053]
- mt76: mt7921: add EEPROM support (Jarod Wilson) [1937053]
- mt76: mt7921: add DMA support (Jarod Wilson) [1937053]
- mt76: mt7921: add MCU support (Jarod Wilson) [1937053]
- mt76: mt7921: add MAC support (Jarod Wilson) [1937053]
- wireless: Fix "ordering" comment typos (Jarod Wilson) [1937053]
- mt76: mt76_connac: move pm utility routines in mt76_connac_lib module (Jarod Wilson) [1937053]
- mt76: mt76_connac: move pm data struct in mt76_connac.h (Jarod Wilson) [1937053]
- mt76: mt76_connac: move WoW and suspend code in mt76_connac_mcu module (Jarod Wilson) [1937053]
- mt76: mt76_connac: move hw_scan and sched_scan routine in mt76_connac_mcu module (Jarod Wilson) [1937053]
- mt76: mt76_connac: create mcu library (Jarod Wilson) [1937053]
- mt76: introduce mt76_vif data structure (Jarod Wilson) [1937053]
- cfg80211: call cfg80211_destroy_ifaces() with wiphy lock held (Jarod Wilson) [1937053]
- wext: call cfg80211_set_encryption() with wiphy lock held (Jarod Wilson) [1937053]
- wext: call cfg80211_change_iface() with wiphy lock held (Jarod Wilson) [1937053]
- nl80211: call cfg80211_dev_rename() under RTNL (Jarod Wilson) [1937053]
- mt76: dma: fix a possible memory leak in mt76_add_fragment() (Jarod Wilson) [1937053]
- ath9k: fix build error with LEDS_CLASS=m (Jarod Wilson) [1937053]
- ath10k: fix wmi mgmt tx queue full due to race condition (Jarod Wilson) [1937053]
- ath10k: pass the ssid info to get the correct bss entity (Jarod Wilson) [1937053]
- ath10k: allow dynamic SAR power limits via common API (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: fix regression in the max_prob_rate fix (Jarod Wilson) [1937053]
- mt76: mt7615: reduce VHT maximum MPDU length (Jarod Wilson) [1937053]
- mt76: reduce q->lock hold time (Jarod Wilson) [1937053]
- mt76: usb: process URBs with status EPROTO properly (Jarod Wilson) [1937053]
- mt76: move vif_mask back from mt76_phy to mt76_dev (Jarod Wilson) [1937053]
- mt76: mt7915: make vif index per adapter instead of per band (Jarod Wilson) [1937053]
- mt76: mt7915: add support for using a secondary PCIe link for gen1 (Jarod Wilson) [1937053]
- mt76: fix crash on tearing down ext phy (Jarod Wilson) [1937053]
- mt76: mt7915: bring up the WA event rx queue for band1 (Jarod Wilson) [1937053]
- mt76: mt7615: unify init work (Jarod Wilson) [1937053]
- mt76: mt7915: support TxBF for DBDC (Jarod Wilson) [1937053]
- mt76: mt7915: Remove unneeded semicolon (Jarod Wilson) [1937053]
- mt76: mt7615: set mcu country code in mt7615_mcu_set_channel_domain() (Jarod Wilson) [1937053]
- mt76: mt7915: fix eeprom DBDC band selection (Jarod Wilson) [1937053]
- mt76: mt7915: fix eeprom parsing for DBDC (Jarod Wilson) [1937053]
- iwlwifi: provide gso_type to GSO packets (Jarod Wilson) [1937053]
- mt76: mt7915: disable RED support in the WA firmware (Jarod Wilson) [1937053]
- mt76: mt7915: rework mcu API (Jarod Wilson) [1937053]
- mt76: mt7915: do not set DRR group for stations (Jarod Wilson) [1937053]
- mt76: mt7915: ensure that init work completes before starting the device (Jarod Wilson) [1937053]
- mt76: mt7603: fix ED/CCA monitoring with single-stream devices (Jarod Wilson) [1937053]
- mt76: mt7915: add implicit Tx beamforming support (Jarod Wilson) [1937053]
- mt76: mt7915: simplify peer's TxBF capability check (Jarod Wilson) [1937053]
- mt76: mt7615: mt7915: disable txpower sku when testmode enabled (Jarod Wilson) [1937053]
- mt76: mt7915: add support for continuous tx in testmode (Jarod Wilson) [1937053]
- mt76: mt7915: rework set state part in testmode (Jarod Wilson) [1937053]
- mt76: testmode: add a new state for continuous tx (Jarod Wilson) [1937053]
- mt76: mt7915: clean hw queue before starting new testmode tx (Jarod Wilson) [1937053]
- mt76: mt7915: calculate new packet length when tx_time is set in testmode (Jarod Wilson) [1937053]
- mt76: mt7915: add support for ipg in testmode (Jarod Wilson) [1937053]
- mt76: mt7915: split edca update function (Jarod Wilson) [1937053]
- mt76: testmode: make tx queued limit adjustable (Jarod Wilson) [1937053]
- mt76: testmode: add attributes for ipg related parameters (Jarod Wilson) [1937053]
- mt76: testmode: add support to set user-defined spe index (Jarod Wilson) [1937053]
- mt76: mt7915: force ldpc for bw larger than 20MHz in testmode (Jarod Wilson) [1937053]
- mt76: move chainmask in mt76_phy (Jarod Wilson) [1937053]
- mt76: move mac_work in mt76_core module (Jarod Wilson) [1937053]
- mt76: mt7615: move testmode data from dev to phy (Jarod Wilson) [1937053]
- mt76: mt7915: move testmode data from dev to phy (Jarod Wilson) [1937053]
- mt76: testmode: move mtd part to mt76_dev (Jarod Wilson) [1937053]
- mt76: testmode: introduce dbdc support (Jarod Wilson) [1937053]
- mt76: mt7915: add partial add_bss_info command on testmode init (Jarod Wilson) [1937053]
- mt76: mt7915: drop zero-length packet to avoid Tx hang (Jarod Wilson) [1937053]
- mt76: mt7915: simplify mt7915_mcu_send_message routine (Jarod Wilson) [1937053]
- mt76: mt7915: fix endianness warning in mt7915_mcu_set_radar_th (Jarod Wilson) [1937053]
- mt76: mt7915: add support for flash mode (Jarod Wilson) [1937053]
- mt76: mt7915: run mt7915_configure_filter holding mt76 mutex (Jarod Wilson) [1937053]
- mt76: mt7915: convert comma to semicolon (Jarod Wilson) [1937053]
- mt76: mt7615: convert comma to semicolon (Jarod Wilson) [1937053]
- mt76: mt7615: reset token when mac_reset happens (Jarod Wilson) [1937053]
- mt76: mt7915: reset token when mac_reset happens (Jarod Wilson) [1937053]
- mt76: mt7915: fix MT_CIPHER_BIP_CMAC_128 setkey (Jarod Wilson) [1937053]
- mt76: mt7615: add vif check in mt7615_update_vif_beacon() (Jarod Wilson) [1937053]
- mt76: mt7915: add vif check in mt7915_update_vif_beacon() (Jarod Wilson) [1937053]
- ath11k: add ieee80211_unregister_hw to avoid kernel crash caused by NULL pointer (Jarod Wilson) [1937053]
- mac80211: pause TX while changing interface type (Jarod Wilson) [1937053]
- wext: fix NULL-ptr-dereference with cfg80211's lack of commit() (Jarod Wilson) [1937053]
- cfg80211: avoid holding the RTNL when calling the driver (Jarod Wilson) [1937053]
- iwl4965: do not process non-QOS frames on txq->sched_retry path (Jarod Wilson) [1937053]
- mt7601u: process tx URBs with status EPROTO properly (Jarod Wilson) [1937053]
- mt7601u: use ieee80211_rx_list to pass frames to the network stack as a batch (Jarod Wilson) [1937053]
- rtw88: 8723de: adjust the LTR setting (Jarod Wilson) [1937053]
- rtlwifi: rtl8821ae: fix bool comparison in expressions (Jarod Wilson) [1937053]
- rtlwifi: rtl8192se: fix bool comparison in expressions (Jarod Wilson) [1937053]
- rtlwifi: rtl8188ee: fix bool comparison in expressions (Jarod Wilson) [1937053]
- rtlwifi: rtl8192c-common: fix bool comparison in expressions (Jarod Wilson) [1937053]
- rtlwifi: rtl_pci: fix bool comparison in expressions (Jarod Wilson) [1937053]
- mt7601u: fix kernel crash unplugging the device (Jarod Wilson) [1937053]
- iwlwifi: queue: bail out on invalid freeing (Jarod Wilson) [1937053]
- iwlwifi: mvm: guard against device removal in reprobe (Jarod Wilson) [1937053]
- iwlwifi: Fix IWL_SUBDEVICE_NO_160 macro to use the correct bit. (Jarod Wilson) [1937053]
- iwlwifi: mvm: clear IN_D3 after wowlan status cmd (Jarod Wilson) [1937053]
- iwlwifi: pcie: add rules to match Qu with Hr2 (Jarod Wilson) [1937053]
- iwlwifi: mvm: invalidate IDs of internal stations at mvm start (Jarod Wilson) [1937053]
- iwlwifi: mvm: fix the return type for DSM functions 1 and 2 (Jarod Wilson) [1937053]
- iwlwifi: pcie: reschedule in long-running memory reads (Jarod Wilson) [1937053]
- iwlwifi: pcie: use jiffies for memory read spin time limit (Jarod Wilson) [1937053]
- iwlwifi: pcie: fix context info memory leak (Jarod Wilson) [1937053]
- iwlwifi: pcie: add a NULL check in iwl_pcie_txq_unmap (Jarod Wilson) [1937053]
- iwlwifi: pcie: set LTR on more devices (Jarod Wilson) [1937053]
- iwlwifi: queue: don't crash if txq->entries is NULL (Jarod Wilson) [1937053]
- iwlwifi: fix the NMI flow for old devices (Jarod Wilson) [1937053]
- iwlwifi: pnvm: don't try to load after failures (Jarod Wilson) [1937053]
- iwlwifi: pnvm: don't skip everything when not reloading (Jarod Wilson) [1937053]
- iwlwifi: pcie: avoid potential PNVM leaks (Jarod Wilson) [1937053]
- iwlwifi: mvm: take mutex for calling iwl_mvm_get_sync_time() (Jarod Wilson) [1937053]
- iwlwifi: mvm: skip power command when unbinding vif during CSA (Jarod Wilson) [1937053]
- cfg80211: change netdev registration/unregistration semantics (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: fix rounding error in throughput calculation (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: increase stats update interval (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: fix max probability rate selection (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: improve sample rate selection (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: improve ampdu length estimation (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: remove old ewma based rate average code (Jarod Wilson) [1937053]
- mac80211: remove legacy minstrel rate control (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: add support for OFDM rates on non-HT clients (Jarod Wilson) [1937053]
- mac80211: minstrel_ht: clean up CCK code (Jarod Wilson) [1937053]
- mac80211: introduce aql_enable node in debugfs (Jarod Wilson) [1937053]
- cfg80211: Add phyrate conversion support for extended MCS in 60GHz band (Jarod Wilson) [1937053]
- cfg80211: add VHT rate entries for MCS-10 and MCS-11 (Jarod Wilson) [1937053]
- mac80211: reduce peer HE MCS/NSS to own capabilities (Jarod Wilson) [1937053]
- mac80211: remove NSS number of 160MHz if not support 160MHz for HE (Jarod Wilson) [1937053]
- mac80211: 160MHz with extended NSS BW in CSA (Jarod Wilson) [1937053]
- mac80211: add LDPC encoding to ieee80211_parse_tx_radiotap (Jarod Wilson) [1937053]
- mac80211: add rx decapsulation offload support (Jarod Wilson) [1937053]
- ath10k: remove unused struct ath10k::dev_type (Jarod Wilson) [1937053]
- ath11k: remove duplicate function declaration (Jarod Wilson) [1937053]
- ath10k: increase rx buffer size to 2048 (Jarod Wilson) [1937053]
- ath10k: sanitity check for ep connectivity (Jarod Wilson) [1937053]
- mt76: mt7663s: fix rx buffer refcounting (Jarod Wilson) [1937053]
- mt7601u: fix rx buffer refcounting (Jarod Wilson) [1937053]
- mac80211: check if atf has been disabled in __ieee80211_schedule_txq (Jarod Wilson) [1937053]
- mac80211: do not drop tx nulldata packets on encrypted links (Jarod Wilson) [1937053]
- mac80211: fix encryption key selection for 802.3 xmit (Jarod Wilson) [1937053]
- mac80211: fix fast-rx encryption check (Jarod Wilson) [1937053]
- mac80211: fix incorrect strlen of .write in debugfs (Jarod Wilson) [1937053]
- cfg80211: fix a kerneldoc markup (Jarod Wilson) [1937053]
- brcmfmac: clear EAP/association status bits on linkdown events (Jarod Wilson) [1937053]
- brcmfmac: Delete useless kfree code (Jarod Wilson) [1937053]
- mt7601u: check the status of device in calibration (Jarod Wilson) [1937053]
- mt7601u: process URBs in status EPROTO properly (Jarod Wilson) [1937053]
- brcmfmac: support BCM4365E with 43666 ChipCommon chip ID (Jarod Wilson) [1937053]
- mwifiex: pcie: Drop bogus __refdata annotation (Jarod Wilson) [1937053]
- rtw88: Simplify bool comparison (Jarod Wilson) [1937053]
- rtw88: coex: set 4 slot TDMA for BT link and WL busy (Jarod Wilson) [1937053]
- rtw88: 8821c: apply CCK PD level which calculates from dynamic mechanism (Jarod Wilson) [1937053]
- rtw88: reduce the log level for failure of tx report (Jarod Wilson) [1937053]
- rtw88: Delete useless kfree code (Jarod Wilson) [1937053]
- iwlwifi: dbg: Don't touch the tlv data (Jarod Wilson) [1937053]
- mt76: Fix queue ID variable types after mcu queue split (Jarod Wilson) [1937053]
- cfg80211: Save the regulatory domain with a lock (Jarod Wilson) [1937053]
- cfg80211/mac80211: fix kernel-doc for SAR APIs (Jarod Wilson) [1937053]
- mt76: mt7915: fix MESH ifdef block (Jarod Wilson) [1937053]
- mt76: mt76s: fix NULL pointer dereference in mt76s_process_tx_queue (Jarod Wilson) [1937053]
- mt76: sdio: remove wake logic in mt76s_process_tx_queue (Jarod Wilson) [1937053]
- mt76: usb: remove wake logic in mt76u_status_worker (Jarod Wilson) [1937053]
- ath11k: dp: clean up a variable name (Jarod Wilson) [1937053]
- ath11k: pci: remove unnecessary mask in ath11k_pci_enable_ltssm() (Jarod Wilson) [1937053]
- ath11k: pci: disable ASPM L0sLs before downloading firmware (Jarod Wilson) [1937053]
- ath11k: qmi: try to allocate a big block of DMA memory first (Jarod Wilson) [1937053]
- rtlwifi: rise completion at the last step of firmware callback (Jarod Wilson) [1937053]
- mt76: mt76u: fix NULL pointer dereference in mt76u_status_worker (Jarod Wilson) [1937053]
- ath10k: prevent deinitializing NAPI twice (Jarod Wilson) [1937053]
- ath9k: Postpone key cache entry deletion for TXQ frames reference it (Jarod Wilson) [1937053]
- ath: Modify ath_key_delete() to not need full key entry (Jarod Wilson) [1937053]
- ath: Export ath_hw_keysetmac() (Jarod Wilson) [1937053]
- ath9k: Clear key cache explicitly on disabling hardware (Jarod Wilson) [1937053]
- ath: Use safer key clearing with key cache entries (Jarod Wilson) [1937053]
- ath10k: Remove voltage regulator votes during wifi disable (Jarod Wilson) [1937053]
- ath10k: Fix error handling in case of CE pipe init failure (Jarod Wilson) [1937053]
- ath11k: Fix ath11k_pci_fix_l1ss() (Jarod Wilson) [1937053]
- ath11k: Fix error code in ath11k_core_suspend() (Jarod Wilson) [1937053]
- ath11k: start vdev if a bss peer is already created (Jarod Wilson) [1937053]
- ath11k: fix crash caused by NULL rx_channel (Jarod Wilson) [1937053]
- ath11k: add missing null check on allocated skb (Jarod Wilson) [1937053]
- ath9k: make relay callbacks const (Jarod Wilson) [1937053]
- ath11k: make relay callbacks const (Jarod Wilson) [1937053]
- ath10k: make relay callbacks const (Jarod Wilson) [1937053]
- ath11k: implement suspend for QCA6390 PCI devices (Jarod Wilson) [1937053]
- ath11k: hif: add ce irq enable and disable functions (Jarod Wilson) [1937053]
- ath11k: implement WoW enable and wakeup commands (Jarod Wilson) [1937053]
- ath11k: set credit_update flag for flow controlled ep only (Jarod Wilson) [1937053]
- ath11k: dp: stop rx pktlog before suspend (Jarod Wilson) [1937053]
- ath11k: htc: implement suspend handling (Jarod Wilson) [1937053]
- ath11k: htc: remove unused struct ath11k_htc_ops (Jarod Wilson) [1937053]
- ath11k: pci: read select_window register to ensure write is finished (Jarod Wilson) [1937053]
- ath11k: hif: implement suspend and resume functions (Jarod Wilson) [1937053]
- ath11k: mhi: hook suspend and resume (Jarod Wilson) [1937053]
- ath11k: Fix incorrect tlvs in scan start command (Jarod Wilson) [1937053]
- ath11k: pci: disable VDD4BLOW (Jarod Wilson) [1937053]
- ath11k: pci: fix L1ss clock unstable problem (Jarod Wilson) [1937053]
- ath11k: pci: fix hot reset stability issues (Jarod Wilson) [1937053]
- ath11k: put hw to DBS using WMI_PDEV_SET_HW_MODE_CMDID (Jarod Wilson) [1937053]
- ath11k: mhi: print a warning if firmware crashed (Jarod Wilson) [1937053]
- ath11k: use MHI provided APIs to allocate and free MHI controller (Jarod Wilson) [1937053]
- ath10k: add atomic protection for device recovery (Jarod Wilson) [1937053]
- ath10k: add option for chip-id based BDF selection (Jarod Wilson) [1937053]
- mt76: remove unused variable q (Jarod Wilson) [1937053]
- mac80211: add ieee80211_set_sar_specs (Jarod Wilson) [1937053]
- nl80211: add common API to configure SAR power limitations (Jarod Wilson) [1937053]
- mac80211: fix a mistake check for rx_stats update (Jarod Wilson) [1937053]
- mac80211: mlme: save ssid info to ieee80211_bss_conf while assoc (Jarod Wilson) [1937053]
- mac80211: Update rate control on channel change (Jarod Wilson) [1937053]
- mac80211: don't filter out beacons once we start CSA (Jarod Wilson) [1937053]
- mac80211: Fix calculation of minimal channel width (Jarod Wilson) [1937053]
- mac80211: ignore country element TX power on 6 GHz (Jarod Wilson) [1937053]
- mac80211: use bitfield helpers for BA session action frames (Jarod Wilson) [1937053]
- mac80211: support Rx timestamp calculation for all preamble types (Jarod Wilson) [1937053]
- mac80211: don't set set TDLS STA bandwidth wider than possible (Jarod Wilson) [1937053]
- mac80211: support driver-based disconnect with reconnect hint (Jarod Wilson) [1937053]
- cfg80211: support immediate reconnect request hint (Jarod Wilson) [1937053]
- mac80211: use struct assignment for he_obss_pd (Jarod Wilson) [1937053]
- cfg80211: remove struct ieee80211_he_bss_color (Jarod Wilson) [1937053]
- nl80211: validate key indexes for cfg80211_registered_device (Jarod Wilson) [1937053]
- cfg80211: include block-tx flag in channel switch started event (Jarod Wilson) [1937053]
- mac80211: disallow band-switch during CSA (Jarod Wilson) [1937053]
- ieee80211: update reduced neighbor report TBTT info length (Jarod Wilson) [1937053]
- cfg80211: Save the regulatory domain when setting custom regulatory (Jarod Wilson) [1937053]
- nl80211: always accept scan request with the duration set (Jarod Wilson) [1937053]
- cfg80211: Update TSF and TSF BSSID for multi BSS (Jarod Wilson) [1937053]
- cfg80211: scan PSC channels in case of scan with wildcard SSID (Jarod Wilson) [1937053]
- mac80211: Skip entries with SAE H2E only membership selector (Jarod Wilson) [1937053]
- cfg80211: Parse SAE H2E only membership selector (Jarod Wilson) [1937053]
- mac80211: support MIC error/replay detected counters driver update (Jarod Wilson) [1937053]
- mac80211: he: remove non-bss-conf fields from bss_conf (Jarod Wilson) [1937053]
- mac80211: remove trailing semicolon in macro definitions (Jarod Wilson) [1937053]
- nl80211: Fix fall-through warnings for Clang (Jarod Wilson) [1937053]
- mac80211: Fix fall-through warnings for Clang (Jarod Wilson) [1937053]
- cfg80211: Fix fall-through warnings for Clang (Jarod Wilson) [1937053]
- cfg80211: fix callback type mismatches in wext-compat (Jarod Wilson) [1937053]
- net: wireless: make a const array static, makes object smaller (Jarod Wilson) [1937053]
- net: mac80211: use core API for updating TX/RX stats (Jarod Wilson) [1937053]
- rfkill: add a reason to the HW rfkill state (Jarod Wilson) [1937053]
- rtlwifi: rtl8192de: fix ofdm power compensation (Jarod Wilson) [1937053]
- ath9k_htc: adhere to the DONT_REORDER transmit flag (Jarod Wilson) [1937053]
- ath11k: mesh: add support for 256 bitmap in blockack frames in 11ax (Jarod Wilson) [1937053]
- ath11k: support TXOP duration based RTS threshold (Jarod Wilson) [1937053]
- ath11k: fix incorrect wmi param for configuring HE operation (Jarod Wilson) [1937053]
- ath11k: unlock on error path in ath11k_mac_op_add_interface() (Jarod Wilson) [1937053]
- ath11k: fix rmmod failure if qmi sequence fails (Jarod Wilson) [1937053]
- rtw88: Fix fall-through warnings for Clang (Jarod Wilson) [1937053]
- rt2x00: Fix fall-through warnings for Clang (Jarod Wilson) [1937053]
- ath11k: pci: add MODULE_FIRMWARE macros (Jarod Wilson) [1937053]
- rtw88: reduce polling time of IQ calibration (Jarod Wilson) [1937053]
- rtw88: fix multiple definition of rtw_pm_ops (Jarod Wilson) [1937053]
- brcmfmac: remove redundant assignment to pointer 'entry' (Jarod Wilson) [1937053]
- rtw88: declare hw supports ch 144 (Jarod Wilson) [1937053]
- rtw88: coex: fix missing unitialization of variable 'interval' (Jarod Wilson) [1937053]
- mwifiex: change license text of Makefile and README from MARVELL to NXP (Jarod Wilson) [1937053]
- rtw88: pci: Add prototypes for .probe, .remove and .shutdown (Jarod Wilson) [1937053]
- ath9k: remove trailing semicolon in macro definition (Jarod Wilson) [1937053]
- ath11k: Ignore resetting peer auth flag in peer assoc cmd (Jarod Wilson) [1937053]
- ath11k: add 64bit check before reading msi high addr (Jarod Wilson) [1937053]
- ath10k: fix a check patch warning returnNonBoolInBooleanFunction of sdio.c (Jarod Wilson) [1937053]
- mac80211: mesh: fix mesh_pathtbl_init() error path (Jarod Wilson) [1937053]
- mt76: mt7615: Fix fall-through warnings for Clang (Jarod Wilson) [1937053]
- mt76: mt7915: fix ht mcs in mt7915_mcu_get_rx_rate() (Jarod Wilson) [1937053]
- mt76: attempt to free up more room when filling the tx queue (Jarod Wilson) [1937053]
- mt76: mt7915: stop queues when running out of tx tokens (Jarod Wilson) [1937053]
- mt76: improve tx queue stop/wake (Jarod Wilson) [1937053]
- mt76: mt7915: fix memory leak in mt7915_mcu_get_rx_rate() (Jarod Wilson) [1937053]
- mt76: mt7615: fix rdd mcu cmd endianness (Jarod Wilson) [1937053]
- mt76: mt7915: fix endian issues (Jarod Wilson) [1937053]
- mt76: mt7915: get rid of dbdc debugfs knob (Jarod Wilson) [1937053]
- mt76: mt7915: introduce dbdc support (Jarod Wilson) [1937053]
- mt76: move hw mac_addr in mt76_phy (Jarod Wilson) [1937053]
- mt76: move band allocation in mt76_register_phy (Jarod Wilson) [1937053]
- mt76: rely on mt76_phy in mt76_init_sband_2g and mt76_init_sband_5g (Jarod Wilson) [1937053]
- mt76: move band capabilities in mt76_phy (Jarod Wilson) [1937053]
- mt76: move tx hw data queues in mt76_phy (Jarod Wilson) [1937053]
- mt76: move mcu queues to mt76_dev q_mcu array (Jarod Wilson) [1937053]
- mt76: rely on mt76_queue in tx_queue_skb_raw signature (Jarod Wilson) [1937053]
- mt76: introduce mt76_init_mcu_queue utility routine (Jarod Wilson) [1937053]
- mt76: rely on mt76_queue in tx_queue_skb signature (Jarod Wilson) [1937053]
- mt76: dma: rely on mt76_queue in mt76_dma_tx_cleanup signature (Jarod Wilson) [1937053]
- mt76: mt7663s: rely on mt76_queue in mt7663s_tx_run_queue signature (Jarod Wilson) [1937053]
- mt76: sdio: rely on mt76_queue in mt76s_process_tx_queue signature (Jarod Wilson) [1937053]
- mt76: sdio: introduce mt76s_alloc_tx_queue (Jarod Wilson) [1937053]
- mt76: move mt76_init_tx_queue in common code (Jarod Wilson) [1937053]
- mt76: mt7915: rely on eeprom definitions (Jarod Wilson) [1937053]
- mt76: mt7615: refactor usb/sdio rate code (Jarod Wilson) [1937053]
- mt76: mt7615: support 16 interfaces (Jarod Wilson) [1937053]
- mt76: mt7915: remove unused mt7915_mcu_bss_sync_tlv() (Jarod Wilson) [1937053]
- mt76: mt7915: use BIT_ULL for omac_idx (Jarod Wilson) [1937053]
- mt76: mt7915: make mt7915_eeprom_read static (Jarod Wilson) [1937053]
- mt76: mt7915: add support to set tx frequency offset in testmode (Jarod Wilson) [1937053]
- mt76: mt7915: add support to set txpower in testmode (Jarod Wilson) [1937053]
- mt76: mt7915: implement testmode rx support (Jarod Wilson) [1937053]
- mt76: mt7915: implement testmode tx support (Jarod Wilson) [1937053]
- mt76: testmode: add support for HE rate modes (Jarod Wilson) [1937053]
- mt76: mt7915: fix tx rate related fields in tx descriptor (Jarod Wilson) [1937053]
- mt76: testmode: add support for LTF and GI combinations for HE mode (Jarod Wilson) [1937053]
- mt76: testmode: add tx_rate_stbc parameter (Jarod Wilson) [1937053]
- mt76: testmode: add snr attribute in rx statistics (Jarod Wilson) [1937053]
- mt76: testmode: switch ib and wb rssi to array type for per-antenna report (Jarod Wilson) [1937053]
- mt76: mt76u: use dedicated thread for status work (Jarod Wilson) [1937053]
- mt76: mt76u: rely on woker APIs for rx work (Jarod Wilson) [1937053]
- mt76: mt7615: run key configuration in mt7615_set_key for usb/sdio devices (Jarod Wilson) [1937053]
- mt76: fix tkip configuration for mt7615/7663 devices (Jarod Wilson) [1937053]
- mt76: switch to wep sw crypto for mt7615/mt7915 (Jarod Wilson) [1937053]
- mt76: mt7663s: introduce WoW support via GPIO (Jarod Wilson) [1937053]
- mt76: move mt76_mcu_send_firmware in common module (Jarod Wilson) [1937053]
- mt76: mt7603: add additional EEPROM chip ID (Jarod Wilson) [1937053]
- mt76: fix memory leak if device probing fails (Jarod Wilson) [1937053]
- mt76: mt7915: fix sparse warning cast from restricted __le16 (Jarod Wilson) [1937053]
- mt76: dma: fix possible deadlock running mt76_dma_cleanup (Jarod Wilson) [1937053]
- mt76: set fops_tx_stats.owner to THIS_MODULE (Jarod Wilson) [1937053]
- mt76: mt7915: rename mt7915_mcu_get_rate_info to mt7915_mcu_get_tx_rate (Jarod Wilson) [1937053]
- mt76: mt7915: update ppe threshold (Jarod Wilson) [1937053]
- mt76: mt7915: set fops_sta_stats.owner to THIS_MODULE (Jarod Wilson) [1937053]
- mt76: sdio: get rid of sched.lock (Jarod Wilson) [1937053]
- mt76: mt7663s: fix a possible ple quota underflow (Jarod Wilson) [1937053]
- mt76: mt7663s: get rid of mt7663s_sta_add (Jarod Wilson) [1937053]
- mt76: mt7615: introduce quota debugfs node for mt7663s (Jarod Wilson) [1937053]
- mt76: mt7615: enable beacon filtering by default for offload fw (Jarod Wilson) [1937053]
- mt76: add back the SUPPORTS_REORDERING_BUFFER flag (Jarod Wilson) [1937053]
- mt76: mt7915: query station rx rate from firmware (Jarod Wilson) [1937053]
- mt76: mt7915: move eeprom parsing out of mt7915_mcu_parse_response (Jarod Wilson) [1937053]
- mt76: implement functions to get the response skb for MCU calls (Jarod Wilson) [1937053]
- mt76: mt7603: switch to .mcu_skb_send_msg (Jarod Wilson) [1937053]
- mt76: make mcu_ops->mcu_send_msg optional (Jarod Wilson) [1937053]
- mt76: move waiting and locking out of mcu_ops->mcu_skb_send_msg (Jarod Wilson) [1937053]
- mt76: move mcu timeout handling to .mcu_parse_response (Jarod Wilson) [1937053]
- mt76: implement .mcu_parse_response in struct mt76_mcu_ops (Jarod Wilson) [1937053]
- mt76: rename __mt76_mcu_skb_send_msg to mt76_mcu_skb_send_msg (Jarod Wilson) [1937053]
- mt76: rename __mt76_mcu_send_msg to mt76_mcu_send_msg (Jarod Wilson) [1937053]
- mt76: mt7915: disable OFDMA/MU-MIMO UL (Jarod Wilson) [1937053]
- mt76: mt7915: fix DRR sta bss group index (Jarod Wilson) [1937053]
- mt76: mt7915: use napi_consume_skb to bulk-free tx skbs (Jarod Wilson) [1937053]
- mt76: mt7915: fix processing txfree events (Jarod Wilson) [1937053]
- mt76: mt7915: support 32 station interfaces (Jarod Wilson) [1937053]
- mt76: do not set NEEDS_UNIQUE_STA_ADDR for 7615 and 7915 (Jarod Wilson) [1937053]
- mt76: mt7615: add debugfs knob for setting extended local mac addresses (Jarod Wilson) [1937053]
- mt76: use ieee80211_rx_list to pass frames to the network stack as a batch (Jarod Wilson) [1937053]
- mt76: mt7915: add encap offload for 4-address mode stations (Jarod Wilson) [1937053]
- mt76: mt7915: add 802.11 encap offload support (Jarod Wilson) [1937053]
- mt76: sdio: convert {status/net}_work to mt76_worker (Jarod Wilson) [1937053]
- mt76: mt7663s: disable interrupt during txrx_worker processing (Jarod Wilson) [1937053]
- mt76: mt7663s: convert txrx_work to mt76_worker (Jarod Wilson) [1937053]
- mt76: mt7663s: move tx/rx processing in the same txrx workqueue (Jarod Wilson) [1937053]
- mt76: mt7615: retry if mt7615_mcu_init returns -EAGAIN (Jarod Wilson) [1937053]
- mt76: mt7915: fix VHT LDPC capability (Jarod Wilson) [1937053]
- mt76: mt7915: measure channel noise and report it via survey (Jarod Wilson) [1937053]
- mac80211: set SDATA_STATE_RUNNING for monitor interfaces (Jarod Wilson) [1937053]
- cfg80211: initialize rekey_data (Jarod Wilson) [1937053]
- mac80211: fix return value of ieee80211_chandef_he_6ghz_oper (Jarod Wilson) [1937053]
- rtw88: debug: Fix uninitialized memory in debugfs code (Jarod Wilson) [1937053]
- brcmfmac: expose firmware config files through modinfo (Jarod Wilson) [1937053]
- rtw88: coex: add feature to enhance HID coexistence performance (Jarod Wilson) [1937053]
- rtw88: coex: upgrade coexistence A2DP mechanism (Jarod Wilson) [1937053]
- rtw88: coex: add action for coexistence in hardware initial (Jarod Wilson) [1937053]
- rtw88: coex: add function to avoid cck lock (Jarod Wilson) [1937053]
- rtw88: coex: change the coexistence mechanism for WLAN connected (Jarod Wilson) [1937053]
- rtw88: coex: change the coexistence mechanism for HID (Jarod Wilson) [1937053]
- rtw88: coex: update AFH information while in free-run mode (Jarod Wilson) [1937053]
- rtw88: coex: update the mechanism for A2DP + PAN (Jarod Wilson) [1937053]
- rtw88: coex: add debug message (Jarod Wilson) [1937053]
- rtw88: coex: run coexistence when WLAN entering/leaving LPS (Jarod Wilson) [1937053]
- Revert "rtl8xxxu: Add Buffalo WI-U3-866D to list of supported devices" (Jarod Wilson) [1937053]
- mt76: usb: fix crash on device removal (Jarod Wilson) [1937053]
- ath11k: dp_rx: fix monitor status dma unmap direction (Jarod Wilson) [1937053]
- ath10k: Constify static qmi structs (Jarod Wilson) [1937053]
- ath10k: Release some resources in an error handling path (Jarod Wilson) [1937053]
- ath10k: Fix an error handling path (Jarod Wilson) [1937053]
- ath10k: Fix the parsing error in service available event (Jarod Wilson) [1937053]
- ath11k: Fix an error handling path (Jarod Wilson) [1937053]
- ath11k: Build check size of ath11k_skb_cb (Jarod Wilson) [1937053]
- ath11k: Reset ath11k_skb_cb before setting new flags (Jarod Wilson) [1937053]
- ath11k: Don't cast ath11k_skb_cb to ieee80211_tx_info.control (Jarod Wilson) [1937053]
- ath11k: remove "ath11k_mac_get_ar_vdev_stop_status" references (Jarod Wilson) [1937053]
- ath11k: peer delete synchronization with firmware (Jarod Wilson) [1937053]
- ath11k: vdev delete synchronization with firmware (Jarod Wilson) [1937053]
- mwifiex: Remove duplicated REG_PORT definition (Jarod Wilson) [1937053]
- rtlwifi: rtl8723ae: avoid accessing the data mapped to streaming DMA (Jarod Wilson) [1937053]
- rtlwifi: rtl8192de: avoid accessing the data mapped to streaming DMA (Jarod Wilson) [1937053]
- rtlwifi: rtl8192ce: avoid accessing the data mapped to streaming DMA (Jarod Wilson) [1937053]
- rtlwifi: rtl8188ee: avoid accessing the data mapped to streaming DMA (Jarod Wilson) [1937053]
- mwifiex: Fix fall-through warnings for Clang (Jarod Wilson) [1937053]
- brcmsmac: ampdu: Check BA window size before checking block ack (Jarod Wilson) [1937053]
- brcmfmac: Fix incorrect type in assignment (Jarod Wilson) [1937053]
- rtlwifi: rtl8192de: remove the useless value assignment (Jarod Wilson) [1937053]
- brcmfmac: fix error return code in brcmf_cfg80211_connect() (Jarod Wilson) [1937053]
- rtw88: wow: print key type when failing (Jarod Wilson) [1937053]
- rtw88: coex: change the decode method from firmware (Jarod Wilson) [1937053]
- rtw88: coex: remove unnecessary WLAN slot extend (Jarod Wilson) [1937053]
- rtw88: coex: remove write scan bit to scoreboard in scan and connect notify (Jarod Wilson) [1937053]
- rtw88: coex: fix BT performance drop during initial/power-on step (Jarod Wilson) [1937053]
- rtw88: coex: Change antenna setting to enhance free-run performance (Jarod Wilson) [1937053]
- rtw88: coex: update the TDMA parameter when leave LPS (Jarod Wilson) [1937053]
- rtw88: coex: add the mechanism for RF4CE (Jarod Wilson) [1937053]
- rtw88: coex: Add force flag for coexistence table function (Jarod Wilson) [1937053]
- rtw88: coex: add write scoreboard action when WLAN in critical procedure (Jarod Wilson) [1937053]
- rtw88: coex: remove unnecessary feature/function (Jarod Wilson) [1937053]
- rtw88: coex: update TDMA settings for different beacon interval (Jarod Wilson) [1937053]
- rtw88: add CCK_PD debug log (Jarod Wilson) [1937053]
- rtw88: 8723d: add cck pd seetings (Jarod Wilson) [1937053]
- ath11k: Fix the rx_filter flag setting for peer rssi stats (Jarod Wilson) [1937053]
- ath10k: add target IRAM recovery feature support (Jarod Wilson) [1937053]
- ath11k: Fix beamformee STS in HE cap (Jarod Wilson) [1937053]
- ath11k: add processor_id based ring_selector logic (Jarod Wilson) [1937053]
- mac80211: free sta in sta_info_insert_finish() on errors (Jarod Wilson) [1937053]
- mac80211: minstrel: fix tx status processing corner case (Jarod Wilson) [1937053]
- mac80211: minstrel: remove deferred sampling code (Jarod Wilson) [1937053]
- mac80211: fix memory leak on filtered powersave frames (Jarod Wilson) [1937053]
- rfkill: Fix use-after-free in rfkill_resume() (Jarod Wilson) [1937053]
- nl80211: fix kernel-doc warning in the new SAE attribute (Jarod Wilson) [1937053]
- cfg80211: remove WDS code (Jarod Wilson) [1937053]
- mac80211: remove WDS-related code (Jarod Wilson) [1937053]
- rt2x00: remove WDS code (Jarod Wilson) [1937053]
- b43: remove WDS code (Jarod Wilson) [1937053]
- ath9k: remove WDS code (Jarod Wilson) [1937053]
- rtw88: coex: simplify the setting and condition about WLAN TX limitation (Jarod Wilson) [1937053]
- rtw88: coex: add debug message (Jarod Wilson) [1937053]
- rtw88: coex: update WLAN 5G AFH parameter for 8822b (Jarod Wilson) [1937053]
- rtw88: coex: change the parameter for A2DP when WLAN connecting (Jarod Wilson) [1937053]
- rtw88: coex: modified for BT info notify (Jarod Wilson) [1937053]
- rtw88: coex: add separate flag for manual control (Jarod Wilson) [1937053]
- rtw88: coex: Modify the timing of set_ant_path/set_rf_para (Jarod Wilson) [1937053]
- rtw88: coex: coding style adjustment (Jarod Wilson) [1937053]
- rtw88: coex: reduce magic number (Jarod Wilson) [1937053]
- rtw88: coex: update coex parameter to improve A2DP quality (Jarod Wilson) [1937053]
- rtw88: coex: fixed some wrong register definition and setting (Jarod Wilson) [1937053]
- rtlwifi: fix spelling typo of workaround (Jarod Wilson) [1937053]
- mwifiex: pcie: skip cancel_work_sync() on reset failure path (Jarod Wilson) [1937053]
- mwifiex: update comment for shutdown_sw()/reinit_sw() to reflect current state (Jarod Wilson) [1937053]
- mwifiex: fix mwifiex_shutdown_sw() causing sw reset failure (Jarod Wilson) [1937053]
- ath11k: Handle errors if peer creation fails (Jarod Wilson) [1937053]
- ath9k: work around false-positive gcc warning (Jarod Wilson) [1937053]
- wireless: remove CONFIG_WIRELESS_WDS (Jarod Wilson) [1937053]
- brcmfmac: Fix memory leak for unpaired brcmf_{alloc/free} (Jarod Wilson) [1937053]
- rtlwifi: rtl8821ae: Place braces around empty if() body (Jarod Wilson) [1937053]
- rtlwifi: rtl8821ae: phy: Remove a couple of unused variables (Jarod Wilson) [1937053]
- rtlwifi: rtl8723be: Remove set but unused variable 'cck_highpwr' (Jarod Wilson) [1937053]
- rtlwifi: halbtc8821a2ant: Remove a bunch of unused variables (Jarod Wilson) [1937053]
- rtlwifi: rtl8188ee: Remove set but unused variable 'reg_ea4' (Jarod Wilson) [1937053]
- rtlwifi: rtl8723be: Remove set but unused variable 'lc_cal' (Jarod Wilson) [1937053]
- rtlwifi: halbtc8821a1ant: Remove set but unused variable 'wifi_rssi_state' (Jarod Wilson) [1937053]
- rtlwifi: phy: Remove set but unused variable 'bbvalue' (Jarod Wilson) [1937053]
- rtlwifi: halbtc8723b2ant: Remove a bunch of set but unused variables (Jarod Wilson) [1937053]
- rtlwifi: rtl8192cu: trx: Demote clear abuse of kernel-doc format (Jarod Wilson) [1937053]
- rtlwifi: rtl8192cu: mac: Fix some missing/ill-documented function parameters (Jarod Wilson) [1937053]
- rtw88: rtw8822c: Remove unused variable 'corr_val' (Jarod Wilson) [1937053]
- mwifiex: pcie: Remove a couple of unchecked 'ret's (Jarod Wilson) [1937053]
- wl1251: cmd: Rename 'len' to 'buf_len' in the documentation (Jarod Wilson) [1937053]
- brcmfmac: fweh: Add missing description for 'gfp' (Jarod Wilson) [1937053]
- brcmfmac: pcie: Provide description for missing function parameter 'devinfo' (Jarod Wilson) [1937053]
- brcmfmac: bcmsdh: Fix description for function parameter 'pktlist' (Jarod Wilson) [1937053]
- rt2x00: save survey for every channel visited (Jarod Wilson) [1937053]
- rtlwifi: Remove in_interrupt() usage in halbtc_send_bt_mp_operation() (Jarod Wilson) [1937053]
- rtlwifi: Remove in_interrupt() usage in is_any_client_connect_to_ap(). (Jarod Wilson) [1937053]
- rtl8xxxu: Add Buffalo WI-U3-866D to list of supported devices (Jarod Wilson) [1937053]
- rtw88: decide lps deep mode from firmware feature. (Jarod Wilson) [1937053]
- rtw88: add C2H response for checking firmware leave lps (Jarod Wilson) [1937053]
- rtw88: store firmware feature in firmware header (Jarod Wilson) [1937053]
- rtw88: sync the power state between driver and firmware (Jarod Wilson) [1937053]
- rtw88: coex: separate BLE HID profile from BLE profile (Jarod Wilson) [1937053]
- rtlwifi: Fix non-canonical address access issues (Jarod Wilson) [1937053]
- brcmfmac: fix SDIO access for big-endian host (Jarod Wilson) [1937053]
- rtw88: remove extraneous 'const' qualifier (Jarod Wilson) [1937053]
- rtlwifi: fix -Wpointer-sign warning (Jarod Wilson) [1937053]
- ath11k: fix ZERO address in probe request (Jarod Wilson) [1937053]
- ath9k: dynack: Demote non-compliant function header (Jarod Wilson) [1937053]
- ath9k: ar5008_phy: Demote half completed function headers (Jarod Wilson) [1937053]
- ath9k: ar9003_2p2_initvals: Remove unused const variables (Jarod Wilson) [1937053]
- ath9k: ar9485_initvals: Remove unused const variable 'ar9485_fast_clock_1_1_baseband_postamble' (Jarod Wilson) [1937053]
- ath9k: ar9340_initvals: Remove unused const variable 'ar9340Modes_ub124_tx_gain_table_1p0' (Jarod Wilson) [1937053]
- ath9k: ar9330_1p1_initvals: Remove unused const variable 'ar9331_common_tx_gain_offset1_1' (Jarod Wilson) [1937053]
- ath: dfs_pri_detector: Demote zero/half completed kernel-doc headers (Jarod Wilson) [1937053]
- ath: dfs_pattern_detector: Fix some function kernel-doc headers (Jarod Wilson) [1937053]
- ath: regd: Provide description for ath_reg_apply_ir_flags's 'reg' param (Jarod Wilson) [1937053]
- ath11k: Add new dfs region name for JP (Jarod Wilson) [1937053]
- ath11k: fix wmi init configuration (Jarod Wilson) [1937053]
- ath11k: Fix the hal descriptor mask (Jarod Wilson) [1937053]
- ath11k: Fix single phy hw mode (Jarod Wilson) [1937053]
- ath11k: Fix number of rules in filtered ETSI regdomain (Jarod Wilson) [1937053]
- ath11k: Remove unused param from wmi_mgmt_params (Jarod Wilson) [1937053]
- ath11k: Initialize complete alpha2 for regulatory change (Jarod Wilson) [1937053]
- ath11k: cold boot calibration support (Jarod Wilson) [1937053]
- ath11k: search DT for qcom,ath11k-calibration-variant (Jarod Wilson) [1937053]
- ath11k: Remove unnecessary data sync to cpu on monitor buffer (Jarod Wilson) [1937053]
- ath10k: cancel rx worker in hif_stop for SDIO (Jarod Wilson) [1937053]
- ath10k: Don't iterate over not-sdata-in-driver interfaces. (Jarod Wilson) [1937053]
- ath10k: fix compilation warning (Jarod Wilson) [1937053]
- mac80211: assure that certain drivers adhere to DONT_REORDER flag (Jarod Wilson) [1937053]
- mac80211: don't overwrite QoS TID of injected frames (Jarod Wilson) [1937053]
- mac80211: adhere to Tx control flag that prevents frame reordering (Jarod Wilson) [1937053]
- mac80211: add radiotap flag to assure frames are not reordered (Jarod Wilson) [1937053]
- mac80211: save HE oper info in BSS config for mesh (Jarod Wilson) [1937053]
- cfg80211: add support to configure HE MCS for beacon rate (Jarod Wilson) [1937053]
- nl80211: fix beacon tx rate mask validation (Jarod Wilson) [1937053]
- cfg80211: Add support to calculate and report 4096-QAM HE rates (Jarod Wilson) [1937053]
- cfg80211: Add support to configure SAE PWE value to drivers (Jarod Wilson) [1937053]
- ieee80211: Add definition for WFA DPP (Jarod Wilson) [1937053]
- mac80211: use semicolons rather than commas to separate statements (Jarod Wilson) [1937053]
- ath11k: FILS discovery and unsolicited broadcast probe response support (Jarod Wilson) [1937053]
- ath10k: sdio: remove redundant check in for loop (Jarod Wilson) [1937053]
- mac80211: add KCOV remote annotations to incoming frame processing (Jarod Wilson) [1937053]
- rtw88: fix fw_fifo_addr check (Jarod Wilson) [1937053]
- mac80211: don't require VHT elements for HE on 2.4 GHz (Jarod Wilson) [1937053]
- cfg80211: regulatory: Fix inconsistent format argument (Jarod Wilson) [1937053]
- mac80211: fix kernel-doc markups (Jarod Wilson) [1937053]
- mac80211: always wind down STA state (Jarod Wilson) [1937053]
- cfg80211: initialize wdev data earlier (Jarod Wilson) [1937053]
- mac80211: fix use of skb payload instead of header (Jarod Wilson) [1937053]
- mac80211: fix regression where EAPOL frames were sent in plaintext (Jarod Wilson) [1937053]
- ath11k: remove repeated words in comments and warnings (Jarod Wilson) [1937053]
- ath10k: ath10k_pci_init_irq(): workaround for checkpatch fallthrough warning (Jarod Wilson) [1937053]
- ath10k: remove repeated words in comments (Jarod Wilson) [1937053]
- nl80211: docs: add a description for s1g_cap parameter (Jarod Wilson) [1937053]
- mac80211: use new function dev_fetch_sw_netstats (Jarod Wilson) [1937053]
- rtlwifi: rtl8192se: remove duplicated legacy_httxpowerdiff (Jarod Wilson) [1937053]
- ath11k: Fix memory leak on error path (Jarod Wilson) [1937053]
- mac80211: copy configured beacon tx rate to driver (Jarod Wilson) [1937053]
- cfg80211: only allow S1G channels on S1G band (Jarod Wilson) [1937053]
- mac80211: initialize last_rate for S1G STAs (Jarod Wilson) [1937053]
- mac80211: handle lack of sband->bitrates in rates (Jarod Wilson) [1937053]
- mac80211: avoid processing non-S1G elements on S1G band (Jarod Wilson) [1937053]
- nl80211: fix non-split wiphy information (Jarod Wilson) [1937053]
- nl80211: reduce non-split wiphy dump size (Jarod Wilson) [1937053]
- ath11k: remove unnecessary casts to u32 (Jarod Wilson) [1937053]
- ath11k: enable idle power save mode (Jarod Wilson) [1937053]
- ath11k: start a timer to update HP for CE pipe 4 (Jarod Wilson) [1937053]
- ath11k: start a timer to update REO cmd ring (Jarod Wilson) [1937053]
- ath11k: start a timer to update TCL HP (Jarod Wilson) [1937053]
- ath11k: set WMI pipe credit to 1 for QCA6390 (Jarod Wilson) [1937053]
- ath11k: enable shadow register configuration and access (Jarod Wilson) [1937053]
- ath11k: read and write registers below unwindowed address (Jarod Wilson) [1937053]
- ath11k: debugfs: fix crash during rmmod (Jarod Wilson) [1937053]
- ath11k: fix warning caused by lockdep_assert_held (Jarod Wilson) [1937053]
- ath11k: mac: remove unused conf_mutex to solve a deadlock (Jarod Wilson) [1937053]
- ath11k: pci: fix rmmod crash (Jarod Wilson) [1937053]
- ath11k: add packet log support for QCA6390 (Jarod Wilson) [1937053]
- ath11k: Use GFP_ATOMIC instead of GFP_KERNEL in idr_alloc (Jarod Wilson) [1937053]
- ath11k: Use GFP_ATOMIC instead of GFP_KERNEL in ath11k_dp_htt_get_ppdu_desc (Jarod Wilson) [1937053]
- ath11k: change to disable softirqs for ath11k_regd_update to solve deadlock (Jarod Wilson) [1937053]
- ath11k: disable monitor mode on QCA6390 (Jarod Wilson) [1937053]
- ath11k: pci: check TCSR_SOC_HW_VERSION (Jarod Wilson) [1937053]
- ath11k: add interface_modes to hw_params (Jarod Wilson) [1937053]
- ath11k: fix AP mode for QCA6390 (Jarod Wilson) [1937053]
- ath11k: support loading ELF board files (Jarod Wilson) [1937053]
- ath11k: Correctly check errors for calls to debugfs_create_dir() (Jarod Wilson) [1937053]
- ath11k: mac: fix parenthesis alignment (Jarod Wilson) [1937053]
- rtw88: pci: Power cycle device during shutdown (Jarod Wilson) [1937053]
- brcmfmac: Fix warning message after dongle setup failed (Jarod Wilson) [1937053]
- brcmfmac: Fix warning when hitting FW crash with flow control feature (Jarod Wilson) [1937053]
- net: rtlwifi: Replace in_interrupt() for context detection (Jarod Wilson) [1937053]
- net: rtlwifi: Remove in_interrupt() from debug macro (Jarod Wilson) [1937053]
- net: rtlwifi: Remove void* casts related to delayed work (Jarod Wilson) [1937053]
- net: mwifiex: Use netif_rx_any_context(). (Jarod Wilson) [1937053]
- net: iwlwifi: Remove in_interrupt() from tracing macro. (Jarod Wilson) [1937053]
- net: ipw2x00,iwlegacy,iwlwifi: Remove in_interrupt() from debug macros (Jarod Wilson) [1937053]
- net: brcmfmac: Convey allocation mode as argument (Jarod Wilson) [1937053]
- net: brcmfmac: Convey execution context via argument to brcmf_netif_rx() (Jarod Wilson) [1937053]
- net: brcmfmac: Replace in_interrupt() (Jarod Wilson) [1937053]
- ath11k: Move non-fatal warn logs to dbg level (Jarod Wilson) [1937053]
- ath9k: Remove set but not used variable (Jarod Wilson) [1937053]
- rtw88: show current regulatory in tx power table (Jarod Wilson) [1937053]
- rtw88: add dump fw crash log (Jarod Wilson) [1937053]
- rtw88: add dump firmware fifo support (Jarod Wilson) [1937053]
- rtw88: handle and recover when firmware crash (Jarod Wilson) [1937053]
- rtw88: increse the size of rx buffer size (Jarod Wilson) [1937053]
- nl80211: extend support to config spatial reuse parameter set (Jarod Wilson) [1937053]
- mac80211: Support not iterating over not-sdata-in-driver ifaces (Jarod Wilson) [1937053]
- mac80211: fix some more kernel-doc in mesh (Jarod Wilson) [1937053]
- cfg80211: regulatory: remove a bogus initialization (Jarod Wilson) [1937053]
- mac80211: fix regression in sta connection monitor (Jarod Wilson) [1937053]
- nl80211: include frequency offset in survey info (Jarod Wilson) [1937053]
- mac80211: support S1G association (Jarod Wilson) [1937053]
- mac80211: receive and process S1G beacons (Jarod Wilson) [1937053]
- mac80211: avoid rate init for S1G band (Jarod Wilson) [1937053]
- mac80211: handle S1G low rates (Jarod Wilson) [1937053]
- mac80211: don't calculate duration for S1G (Jarod Wilson) [1937053]
- mac80211: encode listen interval for S1G (Jarod Wilson) [1937053]
- cfg80211: handle Association Response from S1G STA (Jarod Wilson) [1937053]
- mac80211: convert S1G beacon to scan results (Jarod Wilson) [1937053]
- cfg80211: parse S1G Operation element for BSS channel (Jarod Wilson) [1937053]
- cfg80211: convert S1G beacon to scan results (Jarod Wilson) [1937053]
- mac80211: support S1G STA capabilities (Jarod Wilson) [1937053]
- nl80211: support S1G capability overrides in assoc (Jarod Wilson) [1937053]
- mac80211: s1g: choose scanning width based on frequency (Jarod Wilson) [1937053]
- mac80211: get correct default channel width for S1G (Jarod Wilson) [1937053]
- wireless: radiotap: fix some kernel-doc (Jarod Wilson) [1937053]
- mac80211: fix some missing kernel-doc (Jarod Wilson) [1937053]
- mac80211: Inform AP when returning operating channel (Jarod Wilson) [1937053]
- ath11k: fix undefined reference to 'ath11k_debugfs_htt_ext_stats_handler' (Jarod Wilson) [1937053]
- mt76: mt7663s: remove max_tx_fragments limitation (Jarod Wilson) [1937053]
- mt76: Convert to DEFINE_SHOW_ATTRIBUTE (Jarod Wilson) [1937053]
- mt76: mt7915: add offchannel condition in switch channel command (Jarod Wilson) [1937053]
- mt76: mt7915: convert to use le16_add_cpu() (Jarod Wilson) [1937053]
- mt76: Fix unsigned expressions compared with zero (Jarod Wilson) [1937053]
- mt76: mt7915: fix possible memory leak in mt7915_mcu_add_beacon (Jarod Wilson) [1937053]
- mt76: mt76x0: Move tables used only by init.c to their own header file (Jarod Wilson) [1937053]
- mt76: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- mt76: mt7615: unlock dfs bands (Jarod Wilson) [1937053]
- mt76: mt7663: check isr read return value in mt7663s_rx_work (Jarod Wilson) [1937053]
- mt76: mt7663s: introduce sdio tx aggregation (Jarod Wilson) [1937053]
- mt76: mt7663s: fix possible quota leak in mt7663s_refill_sched_quota (Jarod Wilson) [1937053]
- mt76: move pad estimation out of mt76_skb_adjust_pad (Jarod Wilson) [1937053]
- mt76: mt7663s: introduce __mt7663s_xmit_queue routine (Jarod Wilson) [1937053]
- mt76: mt7663s: split mt7663s_tx_update_sched in mt7663s_tx_{pick,update}_quota (Jarod Wilson) [1937053]
- mt76: mt7663s: do not use altx for ctl/mgmt traffic (Jarod Wilson) [1937053]
- mt76: mt7622: fix fw hang on mt7622 (Jarod Wilson) [1937053]
- mt76: mt7615: fix VHT LDPC capability (Jarod Wilson) [1937053]
- mt76: mt7615: Remove set but unused variable 'index' (Jarod Wilson) [1937053]
- mt76: remove retry_q from struct mt76_txq and related code (Jarod Wilson) [1937053]
- mt76: move txwi handling code to dma.c, since it is mmio specific (Jarod Wilson) [1937053]
- mt76: mt7915: fix queue/tid mapping for airtime reporting (Jarod Wilson) [1937053]
- mt76: mt7915: simplify mt7915_lmac_mapping (Jarod Wilson) [1937053]
- mt76: dma: cache dma map address/len in struct mt76_queue_entry (Jarod Wilson) [1937053]
- mt76: mt7915: fix HE BSS info (Jarod Wilson) [1937053]
- mt76: convert from tx tasklet to tx worker thread (Jarod Wilson) [1937053]
- mt76: add utility functions for deferring work to a kernel thread (Jarod Wilson) [1937053]
- mt76: testmode: add a limit for queued tx_frames packets (Jarod Wilson) [1937053]
- mt76: mt7615: fix antenna selection for testmode tx_frames (Jarod Wilson) [1937053]
- mt76: mt7615: fix MT_ANT_SWITCH_CON register definition (Jarod Wilson) [1937053]
- mt76: mt7915: fix unexpected firmware mode (Jarod Wilson) [1937053]
- mt76: mt76x02: tune tx ring size (Jarod Wilson) [1937053]
- mt76: mt7603: tune tx ring size (Jarod Wilson) [1937053]
- mt76: remove struct mt76_sw_queue (Jarod Wilson) [1937053]
- mt76: rely on AQL for burst size limits on tx queueing (Jarod Wilson) [1937053]
- mt76: remove swq from struct mt76_sw_queue (Jarod Wilson) [1937053]
- mt76: remove qid argument to drv->tx_complete_skb (Jarod Wilson) [1937053]
- mt76: unify queue tx cleanup code (Jarod Wilson) [1937053]
- mt76: sdio: fix use of q->head and q->tail (Jarod Wilson) [1937053]
- mt76: usb: fix use of q->head and q->tail (Jarod Wilson) [1937053]
- mt76: mt7603: check for single-stream EEPROM configuration (Jarod Wilson) [1937053]
- mt76: add memory barrier to DMA queue kick (Jarod Wilson) [1937053]
- mt76: mt7915: add support for accessing mapped registers via bus ops (Jarod Wilson) [1937053]
- mt76: mt7615: significantly reduce interrupt load (Jarod Wilson) [1937053]
- mt76: mt7915: significantly reduce interrupt load (Jarod Wilson) [1937053]
- mt76: mt7915: schedule tx tasklet in mt7915_mac_tx_free (Jarod Wilson) [1937053]
- mt76: dma: update q->queued immediately on cleanup (Jarod Wilson) [1937053]
- mt76: mt7915: optimize mt7915_mac_sta_poll (Jarod Wilson) [1937053]
- mt76: mt7615: fix reading airtime statistics (Jarod Wilson) [1937053]
- mt76: mt7663u: fix dma header initialization (Jarod Wilson) [1937053]
- mt76: fix a possible NULL pointer dereference in mt76_testmode_dump (Jarod Wilson) [1937053]
- mt76: mt7615: fix a possible NULL pointer dereference in mt7615_pm_wake_work (Jarod Wilson) [1937053]
- mt76: mt7615: fix possible memory leak in mt7615_tm_set_tx_power (Jarod Wilson) [1937053]
- mt76: mt7663s: fix unable to handle kernel paging request (Jarod Wilson) [1937053]
- mt76: mt7663s: fix resume failure (Jarod Wilson) [1937053]
- mt76: mt7663s: use NULL instead of 0 in sdio code (Jarod Wilson) [1937053]
- mt76: mt7615: release mutex in mt7615_reset_test_set (Jarod Wilson) [1937053]
- mt76: mt7915: add Tx A-MSDU offloading support (Jarod Wilson) [1937053]
- mt76: mt7915: add missing flags in WMM parameter settings (Jarod Wilson) [1937053]
- mt76: mt7915: simplify aggregation session check (Jarod Wilson) [1937053]
- mt76: mt7615: remove mtxq->agg_ssn assignment (Jarod Wilson) [1937053]
- mt76: move mt76_check_agg_ssn to driver tx_prepare calls (Jarod Wilson) [1937053]
- mt76: mt7915: enable offloading of sequence number assignment (Jarod Wilson) [1937053]
- mt76: mt7915: increase tx retry count (Jarod Wilson) [1937053]
- mt76: mt7915: clean up station stats polling and rate control update (Jarod Wilson) [1937053]
- mt76: mt7915: do not do any work in napi poll after calling napi_complete_done() (Jarod Wilson) [1937053]
- mt76: mt7615: do not do any work in napi poll after calling napi_complete_done() (Jarod Wilson) [1937053]
- mt76: mt76x02: clean up and fix interrupt masking in the irq handler (Jarod Wilson) [1937053]
- mt76: mt7615: only clear unmasked interrupts in irq tasklet (Jarod Wilson) [1937053]
- mt76: mt7915: clean up and fix interrupt masking in the irq handler (Jarod Wilson) [1937053]
- mt76: set interrupt mask register to 0 before requesting irq (Jarod Wilson) [1937053]
- mt76: fix double DMA unmap of the first buffer on 7615/7915 (Jarod Wilson) [1937053]
- mt76: mt7915: fix crash on tx rate report for invalid stations (Jarod Wilson) [1937053]
- mt76: mt7915: enable U-APSD on AP side (Jarod Wilson) [1937053]
- mt76: mt76s: get rid of unused variable (Jarod Wilson) [1937053]
- mt76: mt76s: move tx/rx processing in 2 separate works (Jarod Wilson) [1937053]
- mt76: mt76s: move status processing in txrx wq (Jarod Wilson) [1937053]
- mt76: mt7663s: move rx processing in txrx wq (Jarod Wilson) [1937053]
- mt76: mt76s: move tx processing in a dedicated wq (Jarod Wilson) [1937053]
- mt76: mt76s: fix oom in mt76s_tx_queue_skb_raw (Jarod Wilson) [1937053]
- mt76: mt7615: reschedule runtime-pm receiving a tx interrupt (Jarod Wilson) [1937053]
- mt76: do not inject packets if MT76_STATE_PM is set (Jarod Wilson) [1937053]
- mt76: mt7615: hold mt76 lock queueing wd in mt7615_queue_key_update (Jarod Wilson) [1937053]
- mt76: mt7663s: move drv_own/fw_own in mt7615_mcu_ops (Jarod Wilson) [1937053]
- mt76: mt7615: move drv_own/fw_own in mt7615_mcu_ops (Jarod Wilson) [1937053]
- mt76: mt7615: register ext_phy if DBDC is detected (Jarod Wilson) [1937053]
- brcmfmac: check return value of driver_for_each_device() (Jarod Wilson) [1937053]
- ath11k: Remove unused function ath11k_htc_restore_tx_skb() (Jarod Wilson) [1937053]
- ath11k: remove redundant num_keep_alive_pattern assignment (Jarod Wilson) [1937053]
- ath11k: wmi: remove redundant configuration values from init (Jarod Wilson) [1937053]
- ath11k: Add support spectral scan for IPQ6018 (Jarod Wilson) [1937053]
- ath11k: debugfs: move some function declarations to correct header files (Jarod Wilson) [1937053]
- ath11k: rename debug_htt_stats.[c|h] to debugfs_htt_stats.[c|h] (Jarod Wilson) [1937053]
- ath11k: debugfs: use ath11k_debugfs_ prefix (Jarod Wilson) [1937053]
- ath11k: refactor debugfs code into debugfs.c (Jarod Wilson) [1937053]
- ath10k: Use bdf calibration variant for snoc targets (Jarod Wilson) [1937053]
- rtlwifi: rtl8723be: use true,false for bool variable large_cfo_hit (Jarod Wilson) [1937053]
- rtlwifi: rtl8821ae: use true,false for bool variable large_cfo_hit (Jarod Wilson) [1937053]
- rtlwifi: rtl8192ee: use true,false for bool variable large_cfo_hit (Jarod Wilson) [1937053]
- rtlwifi: Use ffs in <foo>_phy_calculate_bit_shift (Jarod Wilson) [1937053]
- mt7601u: Convert to DEFINE_SHOW_ATTRIBUTE (Jarod Wilson) [1937053]
- ath11k: Remove rproc references from common core layer (Jarod Wilson) [1937053]
- ath9k: hif_usb: fix race condition between usb_get_urb() and usb_kill_anchored_urbs() (Jarod Wilson) [1937053]
- rtlwifi: rtl8723be: fix comparison to bool warning in hw.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8192de: fix comparison to bool warning in hw.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8192ce: fix comparison to bool warning in hw.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8192cu: fix comparison to bool warning in hw.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8821ae: fix comparison to bool warning in phy.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8821ae: fix comparison to bool warning in hw.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8192cu: fix comparison to bool warning in mac.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8192c: fix comparison to bool warning in phy_common.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8192ee: fix comparison to bool warning in hw.c (Jarod Wilson) [1937053]
- mac80211: fix some encapsulation offload kernel-doc (Jarod Wilson) [1937053]
- cfg80211: add missing kernel-doc for S1G band capabilities (Jarod Wilson) [1937053]
- mac80211: Unsolicited broadcast probe response support (Jarod Wilson) [1937053]
- nl80211: Unsolicited broadcast probe response support (Jarod Wilson) [1937053]
- mac80211: Add FILS discovery support (Jarod Wilson) [1937053]
- nl80211: Add FILS discovery support (Jarod Wilson) [1937053]
- mac80211: allow bigger A-MSDU sizes in VHT, even if HT is limited (Jarod Wilson) [1937053]
- nl80211: support setting S1G channels (Jarod Wilson) [1937053]
- nl80211: correctly validate S1G beacon head (Jarod Wilson) [1937053]
- cfg80211: regulatory: handle S1G channels (Jarod Wilson) [1937053]
- nl80211: advertise supported channel width in S1G (Jarod Wilson) [1937053]
- ieee80211: redefine S1G bits with GENMASK (Jarod Wilson) [1937053]
- mac80211: reorganize code to remove a forward declaration (Jarod Wilson) [1937053]
- mac80211: extend ieee80211_tx_status_ext to support bulk free (Jarod Wilson) [1937053]
- mac80211: support using ieee80211_tx_status_ext to free skbs without status info (Jarod Wilson) [1937053]
- mac80211: unify 802.3 (offload) and 802.11 tx status codepath (Jarod Wilson) [1937053]
- mac80211: optimize station connection monitor (Jarod Wilson) [1937053]
- mac80211: notify the driver when a sta uses 4-address mode (Jarod Wilson) [1937053]
- mac80211: swap NEED_TXPROCESSING and HW_80211_ENCAP tx flags (Jarod Wilson) [1937053]
- mac80211: remove tx status call to ieee80211_sta_register_airtime (Jarod Wilson) [1937053]
- mac80211: reduce duplication in tx status functions (Jarod Wilson) [1937053]
- mac80211: rework tx encapsulation offload API (Jarod Wilson) [1937053]
- mac80211: set info->control.hw_key for encap offload packets (Jarod Wilson) [1937053]
- mac80211: skip encap offload for tx multicast/control packets (Jarod Wilson) [1937053]
- mac80211: check and refresh aggregation session in encap offload tx (Jarod Wilson) [1937053]
- mac80211: add missing queue/hash initialization to 802.3 xmit (Jarod Wilson) [1937053]
- cfg80211: add more comments for ap_isolate in bss_parameters (Jarod Wilson) [1937053]
- lib80211: Remove unused macro DRV_NAME (Jarod Wilson) [1937053]
- rtlwifi: rtl8723ae: fix comparison pointer to bool warning in phy.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8723ae: fix comparison pointer to bool warning in trx.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8723ae: fix comparison pointer to bool warning in rf.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8188ee: fix comparison pointer to bool warning in hw.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8188ee: fix comparison pointer to bool warning in trx.c (Jarod Wilson) [1937053]
- rtlwifi: rtl8188ee: fix comparison pointer to bool warning in phy.c (Jarod Wilson) [1937053]
- rtw88: rtw8822c: eliminate code duplication, use native swap() function (Jarod Wilson) [1937053]
- brcmsmac: main: Eliminate empty brcms_c_down_del_timer() (Jarod Wilson) [1937053]
- brcmsmac: phy_lcn: Remove unused variable 'lcnphy_rx_iqcomp_table_rev0' (Jarod Wilson) [1937053]
- brcmsmac: phytbl_lcn: Remove unused array 'dot11lcn_gain_tbl_rev1' (Jarod Wilson) [1937053]
- brcmfmac: support SAE authentication offload in AP mode (Jarod Wilson) [1937053]
- brcmfmac: support 4-way handshake offloading for WPA/WPA2-PSK in AP mode (Jarod Wilson) [1937053]
- ath10k: Remove unused macro ATH10K_ROC_TIMEOUT_HZ (Jarod Wilson) [1937053]
- ath11k: Remove unused inline function htt_htt_stats_debug_dump() (Jarod Wilson) [1937053]
- ath11k: fix link error when CONFIG_REMOTEPROC is disabled (Jarod Wilson) [1937053]
- ath11k: remove calling ath11k_init_hw_params() second time (Jarod Wilson) [1937053]
- ath11k: add raw mode and software crypto support (Jarod Wilson) [1937053]
- ath11k: add ipq6018 support (Jarod Wilson) [1937053]
- ath11k: move target ce configs to hw_params (Jarod Wilson) [1937053]
- net: wireless: drop bogus CRYPTO_xxx Kconfig selects (Jarod Wilson) [1937053]
- rtlwifi: switch from 'pci_' to 'dma_' API (Jarod Wilson) [1937053]
- brcmsmac: fix memory leak in wlc_phy_attach_lcnphy (Jarod Wilson) [1937053]
- rtl8xxxu: prevent potential memory leak (Jarod Wilson) [1937053]
- rtlwifi: rtl8723ae: Delete a stray tab (Jarod Wilson) [1937053]
- ath11k: Add peer max mpdu parameter in peer assoc command (Jarod Wilson) [1937053]
- ath10k: Add support for chain1 regulator supply voting (Jarod Wilson) [1937053]
- brcmfmac: set F2 SDIO block size to 128 bytes for BCM4329 (Jarod Wilson) [1937053]
- brcmfmac: drop chip id from debug messages (Jarod Wilson) [1937053]
- brcmfmac: increase F2 watermark for BCM4329 (Jarod Wilson) [1937053]
- mwifiex: sdio: Fix -Wunused-const-variable warnings (Jarod Wilson) [1937053]
- mwifiex: wmm: Fix -Wunused-const-variable warnings (Jarod Wilson) [1937053]
- mt7601u: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- rtlwifi: Remove temporary definition of RT_TRACE (Jarod Wilson) [1937053]
- rtlwifi: rtl8821ae: Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- brcmsmac: phytbl_n: Remove a few unused arrays (Jarod Wilson) [1937053]
- brcmsmac: phytbl_lcn: Remove unused array 'dot11lcnphytbl_rx_gain_info_rev1' (Jarod Wilson) [1937053]
- brcmsmac: phy_n: Remove a bunch of unused variables (Jarod Wilson) [1937053]
- brcmsmac: phy_lcn: Remove a bunch of unused variables (Jarod Wilson) [1937053]
- prism54: isl_ioctl: Remove unused variable 'j' (Jarod Wilson) [1937053]
- rtw88: debug: Remove unused variables 'val' (Jarod Wilson) [1937053]
- brcmfmac: fwsignal: Finish documenting 'brcmf_fws_mac_descriptor' (Jarod Wilson) [1937053]
- brcmsmac: phy_cmn: Remove a unused variables 'vbat' and 'temp' (Jarod Wilson) [1937053]
- brcmfmac: p2p: Fix a couple of function headers (Jarod Wilson) [1937053]
- brcmsmac: ampdu: Remove a couple set but unused variables (Jarod Wilson) [1937053]
- mwifiex: pcie: Move tables to the only place they're used (Jarod Wilson) [1937053]
- ath10k: Get rid of "per_ce_irq" hw param (Jarod Wilson) [1937053]
- ath10k: Keep track of which interrupts fired, don't poll them (Jarod Wilson) [1937053]
- ath10k: Add interrupt summary based CE processing (Jarod Wilson) [1937053]
- rtlwifi: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- rtl818x_pci: switch from 'pci_' to 'dma_' API (Jarod Wilson) [1937053]
- p54: avoid accessing the data mapped to streaming DMA (Jarod Wilson) [1937053]
- rtlwifi: rtl8192c: phy_common: Remove unused variable 'bbvalue' (Jarod Wilson) [1937053]
- brcmfmac: fwsignal: Remove unused variable 'brcmf_fws_prio2fifo' (Jarod Wilson) [1937053]
- b43: phy_ht: Remove 9 year old TODO (Jarod Wilson) [1937053]
- brcmfmac: btcoex: Update 'brcmf_btcoex_state' and demote others (Jarod Wilson) [1937053]
- brcmfmac: firmware: Demote seemingly unintentional kernel-doc header (Jarod Wilson) [1937053]
- brcmsmac: main: Remove a bunch of unused variables (Jarod Wilson) [1937053]
- brcmfmac: p2p: Fix a bunch of function docs (Jarod Wilson) [1937053]
- brcmsmac: ampdu: Remove a bunch of unused variables (Jarod Wilson) [1937053]
- brcmfmac: p2p: Deal with set but unused variables (Jarod Wilson) [1937053]
- mt7601u: phy: Fix misnaming when documented function parameter 'dac' (Jarod Wilson) [1937053]
- rtlwifi: rtl8723-common: Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8723be Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8723ae Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8192se Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8192ee: Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8192de: Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8192cu: Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8192ce: Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8192-common: Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: rtl8188ee: Rename RT_TRACE to rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: btcoexist: Replace RT_TRACE with rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: Replace RT_TRACE with rtl_dbg (Jarod Wilson) [1937053]
- rtlwifi: Start changing RT_TRACE into rtl_dbg (Jarod Wilson) [1937053]
- ath10k: wmi: Use struct_size() helper in ath10k_wmi_alloc_skb() (Jarod Wilson) [1937053]
- ath11k: fix missing error check on call to ath11k_pci_get_user_msi_assignment (Jarod Wilson) [1937053]
- ath11k: fix spelling mistake "moniter" -> "monitor" (Jarod Wilson) [1937053]
- nl80211: support SAE authentication offload in AP mode (Jarod Wilson) [1937053]
- atmel: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- b43: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- brcmfmac: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- rt2x00: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- b43: phy_n: Add empty braces around empty statements (Jarod Wilson) [1937053]
- b43: phy_common: Demote non-conformant kerneldoc header (Jarod Wilson) [1937053]
- brcmsmac: mac80211_if: Demote a few non-conformant kerneldoc headers (Jarod Wilson) [1937053]
- brcmfmac: fweh: Fix docrot related function documentation issues (Jarod Wilson) [1937053]
- brcmfmac: fweh: Remove set but unused variable 'err' (Jarod Wilson) [1937053]
- b43: main: Add braces around empty statements (Jarod Wilson) [1937053]
- atmel: Demote non-kerneldoc header to standard comment block (Jarod Wilson) [1937053]
- nl80211: rename csa counter attributes countdown counters (Jarod Wilson) [1937053]
- nl80211: add support for setting fixed HE rate/gi/ltf (Jarod Wilson) [1937053]
- ath11k: return -ENOMEM on allocation failure (Jarod Wilson) [1937053]
- ath11k: return error if firmware request fails (Jarod Wilson) [1937053]
- ath9k: Do not select MAC80211_LEDS by default (Jarod Wilson) [1937053]
- ath9k: ar5008_initvals: Move ar5416Bank{0,1,2,3,7} to where they are used (Jarod Wilson) [1937053]
- ath9k: ar5008_initvals: Remove unused table entirely (Jarod Wilson) [1937053]
- ath9k: ar9001_initvals: Remove unused array 'ar5416Bank6_9100' (Jarod Wilson) [1937053]
- ath9k: ar9002_initvals: Remove unused array 'ar9280PciePhy_clkreq_off_L1_9280' (Jarod Wilson) [1937053]
- ath9k_htc: Do not select MAC80211_LEDS by default (Jarod Wilson) [1937053]
- rtw88: 8822c: update tx power limit tables to RF v20.1 (Jarod Wilson) [1937053]
- rtw88: use read_poll_timeout_atomic() for poll loop (Jarod Wilson) [1937053]
- rtw88: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- rtw88: switch from 'pci_' to 'dma_' API (Jarod Wilson) [1937053]
- mwifiex: don't call del_timer_sync() on uninitialized timer (Jarod Wilson) [1937053]
- mwifiex: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- mwifiex: Clean up some err and dbg messages (Jarod Wilson) [1937053]
- mwifiex: switch from 'pci_' to 'dma_' API (Jarod Wilson) [1937053]
- nl80211: fix PORT_AUTHORIZED wording to reflect behavior (Jarod Wilson) [1937053]
- cfg80211: add helper fn for adjacent rule channels (Jarod Wilson) [1937053]
- cfg80211: add helper fn for single rule channels (Jarod Wilson) [1937053]
- nl80211: use NLA_POLICY_RANGE(NLA_BINARY, ...) for a few attributes (Jarod Wilson) [1937053]
- nl80211: clean up code/policy a bit (Jarod Wilson) [1937053]
- net: wireless: wext_compat.c: delete duplicated word (Jarod Wilson) [1937053]
- net: wireless: sme.c: delete duplicated word (Jarod Wilson) [1937053]
- net: wireless: scan.c: delete or fix duplicated words (Jarod Wilson) [1937053]
- net: wireless: reg.c: delete duplicated words + fix punctuation (Jarod Wilson) [1937053]
- net: wireless: delete duplicated word + fix grammar (Jarod Wilson) [1937053]
- net: mac80211: mesh.h: delete duplicated word (Jarod Wilson) [1937053]
- net: mac80211: agg-rx.c: fix duplicated words (Jarod Wilson) [1937053]
- net: ath11k: constify ath11k_thermal_ops (Jarod Wilson) [1937053]
- ath10k: Add new api to support reset TID config (Jarod Wilson) [1937053]
- ath10k: Add new api to support TID specific configuration (Jarod Wilson) [1937053]
- ath10k: Move rate mask validation function up in the file (Jarod Wilson) [1937053]
- ath10k: Add wmi command support for station specific TID config (Jarod Wilson) [1937053]
- ath10k: sdio: add firmware coredump support (Jarod Wilson) [1937053]
- ath10k: add bus type for each layout of coredump (Jarod Wilson) [1937053]
- netlink/wireless: consistently use NLA_POLICY_MIN_LEN() (Jarod Wilson) [1937053]
- netlink/wireless: consistently use NLA_POLICY_EXACT_LEN() (Jarod Wilson) [1937053]
- rndis_wlan: tighten check of rndis_query_oid return (Jarod Wilson) [1937053]
- rtw88: fix spelling mistake: "unsupport" -> "unsupported" (Jarod Wilson) [1937053]
- brcmfmac: use *ph to print small buffer (Jarod Wilson) [1937053]
- brcm80211: fix possible memleak in brcmf_proto_msgbuf_attach (Jarod Wilson) [1937053]
- ath11k: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- ath10k: fix VHT NSS calculation when STBC is enabled (Jarod Wilson) [1937053]
- ath10k: Register shutdown handler (Jarod Wilson) [1937053]
- ath11k: reset MHI during power down and power up (Jarod Wilson) [1937053]
- ath11k: use TCL_DATA_RING_0 for QCA6390 (Jarod Wilson) [1937053]
- ath11k: process both lmac rings for QCA6390 (Jarod Wilson) [1937053]
- ath11k: assign correct search flag and type for QCA6390 (Jarod Wilson) [1937053]
- ath11k: delay vdev_start for QCA6390 (Jarod Wilson) [1937053]
- ath11k: refine the phy_id check in ath11k_reg_chan_list_event (Jarod Wilson) [1937053]
- ath11k: setup QCA6390 rings for both rxdmas (Jarod Wilson) [1937053]
- ath11k: don't initialize rxdma1 related ring (Jarod Wilson) [1937053]
- ath11k: enable DP interrupt setup for QCA6390 (Jarod Wilson) [1937053]
- ath11k: dp: redefine peer_map and peer_unmap (Jarod Wilson) [1937053]
- ath11k: wmi: put hardware to DBS mode (Jarod Wilson) [1937053]
- ath11k: initialize wmi config based on hw_params (Jarod Wilson) [1937053]
- ath11k: force single pdev only for QCA6390 (Jarod Wilson) [1937053]
- ath11k: disable CE interrupt before hif start (Jarod Wilson) [1937053]
- ath11k: ce: get msi_addr and msi_data before srng setup (Jarod Wilson) [1937053]
- ath11k: hal: assign msi_addr and msi_data to srng (Jarod Wilson) [1937053]
- ath11k: ce: remove CE_COUNT() macro (Jarod Wilson) [1937053]
- ath11k: ce: remove host_ce_config_wlan macro (Jarod Wilson) [1937053]
- ath11k: ce: support different CE configurations (Jarod Wilson) [1937053]
- ath11k: hal: cleanup dynamic register macros (Jarod Wilson) [1937053]
- ath11k: hal: create register values dynamically (Jarod Wilson) [1937053]
- ath10k: move enable_pll_clk call to ath10k_core_start() (Jarod Wilson) [1937053]
- ath10k: Fix the size used in a 'dma_free_coherent()' call in an error handling path (Jarod Wilson) [1937053]
- ath10k: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- ath9k: add NL80211_EXT_FEATURE_CAN_REPLACE_PTK0 support (Jarod Wilson) [1937053]
- ath11k: hal: create hw_srng_config dynamically (Jarod Wilson) [1937053]
- ath11k: enable internal sleep clock (Jarod Wilson) [1937053]
- ath11k: fix KASAN warning of ath11k_qmi_wlanfw_wlan_cfg_send (Jarod Wilson) [1937053]
- ath11k: fix memory OOB access in qmi_decode (Jarod Wilson) [1937053]
- ath11k: allocate smaller chunks of memory for firmware (Jarod Wilson) [1937053]
- ath11k: setup ce tasklet for control path (Jarod Wilson) [1937053]
- ath11k: configure copy engine msi address in CE srng (Jarod Wilson) [1937053]
- ath11k: pci: add read32() and write32() hif operations (Jarod Wilson) [1937053]
- ath11k: fill appropriate QMI service instance id for QCA6390 (Jarod Wilson) [1937053]
- ath11k: add board file support for PCI devices (Jarod Wilson) [1937053]
- ath11k: add support for m3 firmware (Jarod Wilson) [1937053]
- ath11k: use remoteproc only with AHB devices (Jarod Wilson) [1937053]
- ath11k: pci: add HAL, CE and core initialisation (Jarod Wilson) [1937053]
- ath11k: register MHI controller device for QCA6390 (Jarod Wilson) [1937053]
- ath11k: implement ath11k_core_pre_init() (Jarod Wilson) [1937053]
- ath11k: pci: add MSI config initialisation (Jarod Wilson) [1937053]
- ath11k: pci: setup resources (Jarod Wilson) [1937053]
- ath11k: add simple PCI client driver for QCA6390 chipset (Jarod Wilson) [1937053]
- ath11k: move ring mask definitions to hw_params (Jarod Wilson) [1937053]
- ath11k: add hw_params entry for QCA6390 (Jarod Wilson) [1937053]
- ath11k: do not depend on ARCH_QCOM for ath11k (Jarod Wilson) [1937053]
- ath10k: correct the array index from mcs index for HT mode for QCA6174 (Jarod Wilson) [1937053]
- ath10k: enable supports_peer_stats_info for QCA6174 PCI devices (Jarod Wilson) [1937053]
- ath10k: remove return for NL80211_STA_INFO_TX_BITRATE (Jarod Wilson) [1937053]
- ath10k: add wmi service peer stat info for wmi tlv (Jarod Wilson) [1937053]
- ath10k: start recovery process when payload length exceeds max htc length for sdio (Jarod Wilson) [1937053]
- ath: drop unnecessary list_empty (Jarod Wilson) [1937053]
- ath9k: Use fallthrough pseudo-keyword (Jarod Wilson) [1937053]
- ath9k: Fix typo in function name (Jarod Wilson) [1937053]
- ath9k: Replace HTTP links with HTTPS ones (Jarod Wilson) [1937053]
- ath10k: check idx validity in __ath10k_htt_rx_ring_fill_n() (Jarod Wilson) [1937053]
- ath9k: Check the return value of pcie_capability_read_*() (Jarod Wilson) [1937053]
- ath11k: remove define ATH11K_QMI_DEFAULT_CAL_FILE_NAME (Jarod Wilson) [1937053]
- ath11k: don't use defines in hw_params (Jarod Wilson) [1937053]
- ath11k: qmi: cleanup info messages (Jarod Wilson) [1937053]
- ath11k: remove useless info messages (Jarod Wilson) [1937053]
- ath11k: change ath11k_core_fetch_board_data_api_n() to use ath11k_core_create_firmware_path() (Jarod Wilson) [1937053]
- ath11k: don't use defines for hw specific firmware directories (Jarod Wilson) [1937053]
- ath11k: create a common function to request all firmware files (Jarod Wilson) [1937053]
- ath11k: Add bdf-addr in hw_params (Jarod Wilson) [1937053]
- ath11k: add hw_ops for pdev id to hw_mac mapping (Jarod Wilson) [1937053]
- ath11k: define max_radios in hw_params (Jarod Wilson) [1937053]
- ath11k: convert ath11k_hw_params to an array (Jarod Wilson) [1937053]
- ath11k: ahb: call ath11k_core_init() before irq configuration (Jarod Wilson) [1937053]
- ath10k: pci: fix memcpy size of bmi response (Jarod Wilson) [1937053]
- ath11k: rename default board file (Jarod Wilson) [1937053]
- ath11k: update firmware files read path (Jarod Wilson) [1937053]
- ath10k: provide survey info as accumulated data (Jarod Wilson) [1937053]
- ath10k: fix retry packets update in station dump (Jarod Wilson) [1937053]
- ath10k: Wait until copy complete is actually done before completing (Jarod Wilson) [1937053]
- mac80211_hwsim: add 6GHz channels (Jarod Wilson) [1937053]
- mac80211_hwsim: indicate support for S1G (Jarod Wilson) [1937053]
- mac80211_hwsim: write TSF timestamp correctly to S1G beacon (Jarod Wilson) [1937053]
- mac80211_hwsim: fix typo in kernel-doc (Jarod Wilson) [1937053]
- mac80211: rename csa counters to countdown counters (Jarod Wilson) [1937053]
- mac80211_hwsim: report the WIPHY_FLAG_SUPPORTS_5_10_MHZ capability (Jarod Wilson) [1937053]
- mac80211_hwsim: Claim support for setting Beacon frame TX legacy rate (Jarod Wilson) [1937053]
- mac80211_hwsim: Advertise support for multicast RX registration (Jarod Wilson) [1937053]
- mac80211_hwsim: indicate in IBSS that we have transmitted beacons (Jarod Wilson) [1937053]
- mac80211_hwsim: notify wmediumd of used MAC addresses (Jarod Wilson) [1937053]
- mac80211_hwsim: use GFP_ATOMIC under spin lock (Jarod Wilson) [1937053]
- mac80211_hwsim: Use kstrndup() in place of kasprintf() (Jarod Wilson) [1937053]
- mac80211_hwsim: add frame transmission support over virtio (Jarod Wilson) [1937053]
- mac80211_hwsim: enable Beacon protection (Jarod Wilson) [1937053]
- mac80211_hwsim: add power save support (Jarod Wilson) [1937053]
- mac80211_hwsim: remove maximum TX power (Jarod Wilson) [1937053]
- mac80211_hwsim: use DEFINE_DEBUGFS_ATTRIBUTE to define debugfs fops (Jarod Wilson) [1937053]
- mac80211_hwsim: add support for OCB (Jarod Wilson) [1937053]
- mac80211_hwsim: add more 5GHz channels, 5/10 MHz support (Jarod Wilson) [1937053]
- mac80211_hwsim: fix incorrect dev_alloc_name failure goto (Jarod Wilson) [1937053]
- mac80211_hwsim: Register support for HE meshpoint (Jarod Wilson) [1937053]
- mac80211_hwsim: fill boottime_ns in netlink RX path (Jarod Wilson) [1937053]
- mac80211_hwsim: Fix possible null-pointer dereferences in hwsim_dump_radio_nl() (Jarod Wilson) [1937053]
- mac80211_hwsim: Extended Key ID API update (Jarod Wilson) [1937053]
- mac80211_hwsim: Fix a typo in the name of function 'mac80211_hswim_he_capab()' (Jarod Wilson) [1937053]
- mac80211_hwsim: Extended Key ID support (Jarod Wilson) [1937053]
- mac80211_hwsim: set p2p device interface support indication (Jarod Wilson) [1937053]
- mac80211_hwsim: calculate if_combination.max_interfaces (Jarod Wilson) [1937053]
- mac80211_hwsim: propagate genlmsg_reply return code (Jarod Wilson) [1937053]
- mac80211_hwsim: Declare support for Multi-BSSID (Jarod Wilson) [1937053]
- mac80211_hwsim: Support boottime in scan results (Jarod Wilson) [1937053]
- mac80211_hwsim: check that n_limits makes sense (Jarod Wilson) [1937053]
- mac80211_hwsim: fix overwriting of if_combination (Jarod Wilson) [1937053]
- mac80211_hwsim: fixes kernel crash during mac80211_hwsim init (Jarod Wilson) [1937053]
- mac80211_hwsim: allow configurable cipher types (Jarod Wilson) [1937053]
- mac80211_hwsim: allow setting iftype support (Jarod Wilson) [1937053]
- mac80211_hwsim: move HWSIM_ATTR_RADIO_NAME parsing last (Jarod Wilson) [1937053]
- mac80211_hwsim: fix module init error paths for netlink (Jarod Wilson) [1937053]
- mac80211_hwsim: drop now unused work-queue from hwsim (Jarod Wilson) [1937053]
- mac80211_hwsim: do not omit multicast announce of first added radio (Jarod Wilson) [1937053]
- mac80211_hwsim: fix race in radio destruction from netlink notifier (Jarod Wilson) [1937053]
- mac80211_hwsim: fix locking when iterating radios during ns exit (Jarod Wilson) [1937053]
- wireless: align to draft 11ax D3.0 (Jarod Wilson) [1937053]
- mac80211_hwsim: correct use of IEEE80211_VHT_CAP_RXSTBC_X (Jarod Wilson) [1937053]
- mac80211_hwsim: require at least one channel (Jarod Wilson) [1937053]
- mac80211_hwsim: Add support for HE (Jarod Wilson) [1937053]
- net/fq_impl: do not maintain a backlog-sorted list of flows (Jarod Wilson) [1937053]
- net/fq_impl: drop get_default_func, move default flow to fq_tin (Jarod Wilson) [1937053]
- net/fq_impl: bulk-free packets from a flow on overmemory (Jarod Wilson) [1937053]
- net/fq_impl: use skb_get_hash instead of skb_get_hash_perturb (Jarod Wilson) [1937053]
- net/fq_impl: Switch to kvmalloc() for memory allocation (Jarod Wilson) [1937053]
- mac80211: set NETIF_F_LLTX when using intermediate tx queues (Jarod Wilson) [1937053]
- mac80211: calculate hash for fq without holding fq->lock in itxq enqueue (Jarod Wilson) [1937053]
- nvme-tcp: fix misuse of __smp_processor_id with preemption enabled (Gopal Tiwari) [1964227]
- powerpc/smp: Cache CPU to chip lookup (Daniel Henrique Barboza) [1934421]
- Revert "powerpc/topology: Update topology_core_cpumask" (Daniel Henrique Barboza) [1934421]
- powerpc/smp: Reintroduce cpu_core_mask (Daniel Henrique Barboza) [1934421]
- scsi: smartpqi: Remove unused functions (Don Brace) [1952154]
- scsi: smartpqi: Fix device pointer variable reference static checker issue (Don Brace) [1952154]
- scsi: smartpqi: Fix blocks_per_row static checker issue (Don Brace) [1952154]
- scsi: smartpqi: Update version to 2.1.8-045 (Don Brace) [1952154]
- scsi: smartpqi: Add new PCI IDs (Don Brace) [1952154]
- scsi: smartpqi: Correct system hangs when resuming from hibernation (Don Brace) [1952154]
- scsi: smartpqi: Update enclosure identifier in sysfs (Don Brace) [1952154]
- scsi: smartpqi: Add additional logging for LUN resets (Don Brace) [1952154]
- scsi: smartpqi: Update SAS initiator_port_protocols and target_port_protocols (Don Brace) [1952154]
- scsi: smartpqi: Add phy ID support for the physical drives (Don Brace) [1952154]
- scsi: smartpqi: Convert snprintf() to scnprintf() (Don Brace) [1952154]
- scsi: smartpqi: Fix driver synchronization issues (Don Brace) [1952154]
- scsi: smartpqi: Update device scan operations (Don Brace) [1952154]
- scsi: smartpqi: Update OFA management (Don Brace) [1952154]
- scsi: smartpqi: Update RAID bypass handling (Don Brace) [1952154]
- scsi: smartpqi: Update suspend/resume and shutdown (Don Brace) [1952154]
- scsi: smartpqi: Synchronize device resets with mutex (Don Brace) [1952154]
- scsi: smartpqi: Update soft reset management for OFA (Don Brace) [1952154]
- scsi: smartpqi: Update event handler (Don Brace) [1952154]
- scsi: smartpqi: Add support for wwid (Don Brace) [1952154]
- scsi: smartpqi: Remove timeouts from internal cmds (Don Brace) [1952154]
- scsi: smartpqi: Disable WRITE SAME for HBA NVMe disks (Don Brace) [1952154]
- scsi: smartpqi: Add host level stream detection enable (Don Brace) [1952154]
- scsi: smartpqi: Add stream detection (Don Brace) [1952154]
- scsi: smartpqi: Align code with oob driver (Don Brace) [1952154]
- scsi: smartpqi: Add support for long firmware version (Don Brace) [1952154]
- scsi: smartpqi: Add support for BMIC sense feature cmd and feature bits (Don Brace) [1952154]
- scsi: smartpqi: Add support for RAID1 writes (Don Brace) [1952154]
- scsi: smartpqi: Add support for RAID5 and RAID6 writes (Don Brace) [1952154]
- scsi: smartpqi: Refactor scatterlist code (Don Brace) [1952154]
- scsi: smartpqi: Refactor aio submission code (Don Brace) [1952154]
- scsi: smartpqi: Add support for new product ids (Don Brace) [1952154]
- scsi: smartpqi: Correct request leakage during reset operations (Don Brace) [1952154]
- scsi: smartpqi: Use host-wide tag space (Don Brace) [1952154]
- smartpqi: Use fallthrough pseudo-keyword (Don Brace) [1952154]
- scsi: smartpqi: Support device deletion via sysfs (Don Brace) [1952154]
- scsi: smartpqi: Use scnprintf() for avoiding potential buffer overflow (Don Brace) [1952154]
- smartpqi: remove ioremap_nocache and devm_ioremap_nocache (Don Brace) [1952154]
- smartpqi: Use sizeof_field() macro (Don Brace) [1952154]
- smartpqi: phase out dma_zalloc_coherent() (Don Brace) [1952154]
- scsi: smartpqi: add support for huawei controllers (Don Brace) [1952154]
- block: only update parent bi_status when bio fail (Ming Lei) [1945457]
- kbuild: trace functions in subdirectories of lib/ (Joe Lawrence) [1942601]
- kbuild: introduce ccflags-remove-y and asflags-remove-y (Joe Lawrence) [1942601]
- redhat/configs: Add CONFIG_AMD_PMC (David Arcari) [1961755]
- platform/x86: amd-pmc: put device on error paths (David Arcari) [1961755]
- platform/x86: amd-pmc: Fix CONFIG_DEBUG_FS check (David Arcari) [1961755]
- platform/x86: amd-pmc: Add AMD platform support for S2Idle (David Arcari) [1961755]
- crypto: DRBG - switch to HMAC SHA512 DRBG as default DRBG (Vladis Dronov) [1956808]
- x86/asm: Annotate movdir64b()'s dst argument with __iomem (Jerry Snitselaar) [1920747]
- x86/asm: Add a missing __iomem annotation in enqcmds() (Jerry Snitselaar) [1920747]

* Wed Jun 02 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-312.el8]
- fm10k: amend removal of MODULE_VERSION (Vladis Dronov) [1955730]
- hv_netvsc: Make netvsc/VF binding check both MAC and serial number (Mohammed Gamal) [1940927]
- hv_netvsc: Add error handling while switching data path (Mohammed Gamal) [1940927]
- netvsc: Update driver to use ethtool_sprintf (Mohammed Gamal) [1940927]
- hv_netvsc: Add a comment clarifying batching logic (Mohammed Gamal) [1940927]
- hv_netvsc: Fix validation in netvsc_linkstatus_callback() (Mohammed Gamal) [1940927]
- hv_netvsc: Restrict configurations on isolated guests (Mohammed Gamal) [1940927]
- hv_netvsc: Load and store the proper (NBL_HASH_INFO) per-packet info (Mohammed Gamal) [1940927]
- hv_netvsc: Allocate the recv_buf buffers after NVSP_MSG1_TYPE_SEND_RECV_BUF (Mohammed Gamal) [1940927]
- hv_netvsc: Reset the RSC count if NVSP_STAT_FAIL in netvsc_receive() (Mohammed Gamal) [1940927]
- hv_netvsc: Copy packets sent by Hyper-V out of the receive buffer (Mohammed Gamal) [1940927]
- net/hyperv: use skb_vlan_tag_*() helpers (Mohammed Gamal) [1940927]
- net/vlan: introduce skb_vlan_tag_get_cfi() helper (Mohammed Gamal) [1940927]
- hv_netvsc: Add (more) validation for untrusted Hyper-V values (Mohammed Gamal) [1940927]
- hv_netvsc: Process NETDEV_GOING_DOWN on VF hot remove (Mohammed Gamal) [1940927]
- hv_netvsc: Wait for completion on request SWITCH_DATA_PATH (Mohammed Gamal) [1940927]
- hv_netvsc: Check VF datapath when sending traffic to VF (Mohammed Gamal) [1940927]
- hv_netvsc: Validate number of allocated sub-channels (Mohammed Gamal) [1940927]
- hv_netvsc: Use vmbus_requestor to generate transaction IDs for VMBus hardening (Mohammed Gamal) [1940927]
- hv_netvsc: Use HV_HYP_PAGE_SIZE for Hyper-V communication (Mohammed Gamal) [1940927]
- scsi: fnic: Use scsi_host_busy_iter() to traverse commands (Ewan D. Milne) [1949250]
- scsi: fnic: Kill 'exclude_id' argument to fnic_cleanup_io() (Ewan D. Milne) [1949250]
- net: vlan: Avoid using BUG() in vlan_proto_idx() (Davide Caratti) [1964456]
- sch_htb: fix null pointer dereference on a null new_q (Ivan Vecera) [1935703]
- sch_htb: Fix offload cleanup in htb_destroy on htb_init failure (Ivan Vecera) [1935703]
- sch_htb: Fix select_queue for non-offload mode (Ivan Vecera) [1935703]
- sch_htb: Stats for offloaded HTB (Ivan Vecera) [1935703]
- sch_htb: Hierarchical QoS hardware offload (Ivan Vecera) [1935703]
- net: sched: Add extack to Qdisc_class_ops.delete (Ivan Vecera) [1935703]
- net: sched: Add multi-queue support to sch_tree_lock (Ivan Vecera) [1935703]
- capabilities: require CAP_SETFCAP to map uid 0 (Alexey Gladkov) [1961202]
- locking/qrwlock: Cleanup queued_write_lock_slowpath() (Waiman Long) [1954361]
- locking/rtmutex: Clean up signal handling in __rt_mutex_slowlock() (Waiman Long) [1954361]
- locking/rtmutex: Restrict the trylock WARN_ON() to debug (Waiman Long) [1954361]
- locking/rtmutex: Fix misleading comment in rt_mutex_postunlock() (Waiman Long) [1954361]
- locking/rtmutex: Consolidate the fast/slowpath invocation (Waiman Long) [1954361]
- locking/rtmutex: Make text section and inlining consistent (Waiman Long) [1954361]
- locking/rtmutex: Move debug functions as inlines into common header (Waiman Long) [1954361]
- locking/rtmutex: Decrapify __rt_mutex_init() (Waiman Long) [1954361]
- locking/rtmutex: Remove pointless CONFIG_RT_MUTEXES=n stubs (Waiman Long) [1954361]
- locking/rtmutex: Inline chainwalk depth check (Waiman Long) [1954361]
- locking/rtmutex: Move rt_mutex_debug_task_free() to rtmutex.c (Waiman Long) [1954361]
- locking/rtmutex: Remove empty and unused debug stubs (Waiman Long) [1954361]
- locking/rtmutex: Consolidate rt_mutex_init() (Waiman Long) [1954361]
- locking/rtmutex: Remove output from deadlock detector (Waiman Long) [1954361]
- locking/rtmutex: Remove rtmutex deadlock tester leftovers (Waiman Long) [1954361]
- locking/rtmutex: Remove rt_mutex_timed_lock() (Waiman Long) [1954361]
- kernel: delete repeated words in comments (Waiman Long) [1954361]
- rbtree, rtmutex: Use rb_add_cached() (Waiman Long) [1954361]
- rbtree: Add generic add and find helpers (Waiman Long) [1954361]
- locking/rtmutex: Remove unused rt_mutex_cmpxchg_relaxed() (Waiman Long) [1954361]
- lib/rbtree: avoid generating code twice for the cached versions (Waiman Long) [1954361]
- locking/rtmutex: Fix the preprocessor logic with normal #ifdef #else #endif (Waiman Long) [1954361]
- kernel/futex: Explicitly document pi_lock for pi_state owner fixup (Waiman Long) [1954361]
- kernel/futex: Move hb unlock out of unqueue_me_pi() (Waiman Long) [1954361]
- kernel/futex: Make futex_wait_requeue_pi() only call fixup_owner() (Waiman Long) [1954361]
- kernel/futex: Kill rt_mutex_next_owner() (Waiman Long) [1954361]
- futex: Remove unused empty compat_exit_robust_list() (Waiman Long) [1954361]
- futex: Handle transient "ownerless" rtmutex state correctly (Waiman Long) [1954361]
- futex: Convert to use the preferred 'fallthrough' macro (Waiman Long) [1954361]
- futex: get rid of a kernel-docs build warning (Waiman Long) [1954361]
- futex: arch_futex_atomic_op_inuser() calling conventions change (Waiman Long) [1954361]
- s390: Use arch_local_irq_{save,restore}() in early boot code (Waiman Long) [1954361]
- lockdep: Add a missing initialization hint to the "INFO: Trying to register non-static key" message (Waiman Long) [1954361]
- locking: Fix typos in comments (Waiman Long) [1954361]
- locking/lockdep: Update the comment for __lock_release() (Waiman Long) [1954361]
- lockdep: Allow tuning tracing capacity constants. (Waiman Long) [1954361]
- lockdep: Address clang -Wformat warning printing for hd (Waiman Long) [1954361]
- lockdep: Add lockdep lock state defines (Waiman Long) [1954361]
- lockdep: Add lockdep_assert_not_held() (Waiman Long) [1954361]
- lockdep: Noinstr annotate warn_bogus_irq_restore() (Waiman Long) [1954361]
- locking/lockdep: Avoid unmatched unlock (Waiman Long) [1954361]
- locking/rwsem: Remove empty rwsem.h (Waiman Long) [1954361]
- lockdep: report broken irq restoration (Waiman Long) [1954361]
- locking/lockdep: Avoid noinstr warning for DEBUG_LOCKDEP (Waiman Long) [1954361]
- locking/lockdep: Exclude local_lock_t from IRQ inversions (Waiman Long) [1954361]
- locking/lockdep: Clean up check_redundant() a bit (Waiman Long) [1954361]
- locking/lockdep: Add a skip() function to __bfs() (Waiman Long) [1954361]
- locking/lockdep: Mark local_lock_t (Waiman Long) [1954361]
- lockdep: Put graph lock/unlock under lock_recursion protection (Waiman Long) [1954361]
- locking/rtmutex: Add missing kernel-doc markup (Waiman Long) [1954361]
- lockdep: Avoid to modify chain keys in validate_chain() (Waiman Long) [1954361]
- lockdep: Fix nr_unused_locks accounting (Waiman Long) [1954361]
- lockdep: Optimize the memory usage of circular queue (Waiman Long) [1954361]
- lockdep: Take read/write status in consideration when generate chainkey (Waiman Long) [1954361]
- lockdep/selftest: Add a R-L/L-W test case specific to chain cache behavior (Waiman Long) [1954361]
- lockdep: Add recursive read locks into dependency graph (Waiman Long) [1954361]
- lockdep: Fix recursive read lock related safe->unsafe detection (Waiman Long) [1954361]
- lockdep: Adjust check_redundant() for recursive read change (Waiman Long) [1954361]
- lockdep: Support deadlock detection for recursive read locks in check_noncircular() (Waiman Long) [1954361]
- lockdep: Make __bfs(.match) return bool (Waiman Long) [1954361]
- lockdep: Extend __bfs() to work with multiple types of dependencies (Waiman Long) [1954361]
- lockdep: Introduce lock_list::dep (Waiman Long) [1954361]
- lockdep: Reduce the size of lock_list::distance (Waiman Long) [1954361]
- lockdep: Make __bfs() visit every dependency until a match (Waiman Long) [1954361]
- lockdep: Demagic the return value of BFS (Waiman Long) [1954361]
- lockdep/Documention: Recursive read lock detection reasoning (Waiman Long) [1954361]
- locking: More accurate annotations for read_lock() (Waiman Long) [1954361]
- Documentation/locking/locktypes: Fix local_locks documentation (Waiman Long) [1954361]
- treewide: Remove uninitialized_var() usage (Waiman Long) [1954361]
- lockdep: Use this_cpu_ptr instead of get_cpu_var stats (Waiman Long) [1954361]
- KVM: arm/arm64: Fix KVM_VGIC_V3_ADDR_TYPE_REDIST read (Gavin Shan) [1939323]
- KVM: arm64: Don't advertise FEAT_SPE to guests (Gavin Shan) [1939323]
- KVM: arm64: Don't print warning when trapping SPE registers (Gavin Shan) [1939323]
- KVM: arm64: Fully zero the vcpu state on reset (Gavin Shan) [1939323]
- KVM: arm64: Clarify vcpu reset behaviour (Gavin Shan) [1939323]
- Documentation: KVM: Document KVM_GUESTDBG_USE_HW control flag for arm64 (Gavin Shan) [1939323]
- KVM: arm64: Initialize VCPU mdcr_el2 before loading it (Gavin Shan) [1939323]
- KVM: arm64: Handle access to TRFCR_EL1 (Gavin Shan) [1939323]
- arm64: Add TRFCR_ELx definitions (Gavin Shan) [1939323]
- KVM: arm64: Unify trap handlers injecting an UNDEF (Gavin Shan) [1939323]
- KVM: arm64: Use find_vma_intersection() (Gavin Shan) [1939323]
- KVM: arm64: Hide kvm_mmu_wp_memory_region() (Gavin Shan) [1939323]
- KVM: selftests: vgic_init kvm selftests fixup (Gavin Shan) [1939323]
- KVM: selftests: aarch64/vgic-v3 init sequence tests (Gavin Shan) [1939323]
- KVM: arm64: vgic-v3: Expose GICR_TYPER.Last for userspace (Gavin Shan) [1939323]
- kvm: arm64: vgic-v3: Introduce vgic_v3_free_redist_region() (Gavin Shan) [1939323]
- KVM: arm64: Simplify argument passing to vgic_uaccess_[read|write] (Gavin Shan) [1939323]
- docs: kvm: devices/arm-vgic-v3: enhance KVM_DEV_ARM_VGIC_CTRL_INIT doc (Gavin Shan) [1939323]
- KVM: arm/arm64: vgic: Reset base address on kvm_vgic_dist_destroy() (Gavin Shan) [1939323]
- KVM: arm64: vgic-v3: Fix error handling in vgic_v3_set_redist_base() (Gavin Shan) [1939323]
- KVM: arm64: Fix KVM_VGIC_V3_ADDR_TYPE_REDIST_REGION read (Gavin Shan) [1939323]
- KVM: arm64: vgic-v3: Fix some error codes when setting RDIST base (Gavin Shan) [1939323]
- KVM: arm64: Disable guest access to trace filter controls (Gavin Shan) [1939323]
- KVM: arm64: Fix CPU interface MMIO compatibility detection (Gavin Shan) [1939323]
- KVM: arm64: Hide system instruction access to Trace registers (Gavin Shan) [1939323]
- KVM: arm64: Fix exclusive limit for IPA size (Gavin Shan) [1939323]
- KVM: arm64: Ensure I-cache isolation between vcpus of a same VM (Gavin Shan) [1939323]
- KVM: arm64: nvhe: Save the SPE context early (Gavin Shan) [1939323]
- KVM: arm64: Correct spelling of DBGDIDR register (Gavin Shan) [1939323]
- KVM: arm64: Fix AArch32 PMUv3 capping (Gavin Shan) [1939323]
- KVM: arm64: Fix missing RES1 in emulation of DBGBIDR (Gavin Shan) [1939323]
- KVM: arm64: Remove spurious semicolon in reg_to_encoding() (Gavin Shan) [1939323]
- KVM: arm64: Fix hyp_cpu_pm_{init,exit} __init annotation (Gavin Shan) [1939323]
- KVM: arm64: Workaround firmware wrongly advertising GICv2-on-v3 compatibility (Gavin Shan) [1939323]
- KVM: arm64: Rename __vgic_v3_get_ich_vtr_el2() to __vgic_v3_get_gic_config() (Gavin Shan) [1939323]
- KVM: arm64: Some fixes of PV-time interface document (Gavin Shan) [1939323]
- KVM: arm64: CSSELR_EL1 max is 13 (Gavin Shan) [1939323]
- idr test suite: Improve reporting from idr_find_test_1 (Chris von Recklinghausen) [1946304]
- idr test suite: Create anchor before launching throbber (Chris von Recklinghausen) [1946304]
- idr test suite: Take RCU read lock in idr_find_test_1 (Chris von Recklinghausen) [1946304]
- radix tree test suite: Register the main thread with the RCU library (Chris von Recklinghausen) [1946304]
- radix tree test suite: Fix compilation (Chris von Recklinghausen) [1946304]
- XArray: Add xa_limit_16b (Chris von Recklinghausen) [1946304]
- XArray: Fix splitting to non-zero orders (Chris von Recklinghausen) [1946304]
- XArray: Fix split documentation (Chris von Recklinghausen) [1946304]
- PCI/P2PDMA: Drop double zeroing for sg_init_table() (Myron Stowe) [1955693]
- PCI: Simplify bool comparisons (Myron Stowe) [1955693]
- PCI: endpoint: Use "NULL" instead of "0" as a NULL pointer (Myron Stowe) [1955693]
- PCI: Simplify pci_dev_reset_slot_function() (Myron Stowe) [1955693]
- PCI: Update mmap-related #ifdef comments (Myron Stowe) [1955693]
- PCI/LINK: Print IRQ number used by port (Myron Stowe) [1955693]
- PCI/IOV: Simplify pci-pf-stub with module_pci_driver() (Myron Stowe) [1955693]
- PCI: Use scnprintf(), not snprintf(), in sysfs "show" functions (Myron Stowe) [1955693]
- x86/PCI: Fix intel_mid_pci.c build error when ACPI is not enabled (Myron Stowe) [1955693]
- PCI: Remove unnecessary header includes (Myron Stowe) [1955693]
- PCI/PM: Revert "PCI/PM: Apply D2 delay as milliseconds, not microseconds" (Myron Stowe) [1955693]
- PCI/PM: Remove unused PCI_PM_BUS_WAIT (Myron Stowe) [1955693]
- PCI: shpchp: Remove unused 'rc' assignment (Myron Stowe) [1955693]
- PCI: pciehp: Reduce noisiness on hot removal (Myron Stowe) [1955693 1812902]
- PCI: rpadlpar: Use for_each_child_of_node() and for_each_node_by_name() (Myron Stowe) [1955693]
- PCI: Add Kconfig options for MPS/MRRS strategy (Myron Stowe) [1955693]
- PCI: Add schedule point in pci_read_config() (Myron Stowe) [1955693]
- PCI/ASPM: Remove struct pcie_link_state.l1ss (Myron Stowe) [1955693]
- PCI/ASPM: Remove struct aspm_register_info.l1ss_cap (Myron Stowe) [1955693]
- PCI/ASPM: Pass L1SS Capabilities value, not struct aspm_register_info (Myron Stowe) [1955693]
- PCI/ASPM: Remove struct aspm_register_info.l1ss_ctl1 (Myron Stowe) [1955693]
- PCI/ASPM: Remove struct aspm_register_info.l1ss_ctl2 (unused) (Myron Stowe) [1955693]
- PCI/ASPM: Remove struct aspm_register_info.l1ss_cap_ptr (Myron Stowe) [1955693]
- PCI/ASPM: Remove struct aspm_register_info.latency_encoding (Myron Stowe) [1955693]
- PCI/ASPM: Remove struct aspm_register_info.enabled (Myron Stowe) [1955693]
- PCI/ASPM: Remove struct aspm_register_info.support (Myron Stowe) [1955693]
- PCI/ASPM: Use 'parent' and 'child' for readability (Myron Stowe) [1955693]
- PCI/ASPM: Move LTR path check to where it's used (Myron Stowe) [1955693]
- PCI/ASPM: Move pci_clear_and_set_dword() earlier (Myron Stowe) [1955693]
- PCI/ACS: Enable Translation Blocking for external devices (Myron Stowe) [1955693]
- scsi: qedi: Remove redundant assignment to variable err (Nilesh Javali) [1920855]
- scsi: qedi: Fix error return code of qedi_alloc_global_queues() (Nilesh Javali) [1920855]
- scsi: qedi: Fix missing destroy_workqueue() on error in __qedi_probe (Nilesh Javali) [1920855]
- scsi: qedf: Simplify bool comparison (Nilesh Javali) [1921608]
- bpf, sockmap: Fix incorrect fwd_alloc accounting (Yauheni Kaliuta) [1874006]
- bpf: Local storage helpers should check nullness of owner ptr passed (Yauheni Kaliuta) [1874006]
- bpf, inode_storage: Put file handler if no storage was found (Yauheni Kaliuta) [1874006]
- kbuild: explicitly specify the build id style (Yauheni Kaliuta) [1874006]
- bpf: Reject too big ctx_size_in for raw_tp test run (Yauheni Kaliuta) [1874006]
- bpf: Fix enum names for bpf_this_cpu_ptr() and bpf_per_cpu_ptr() helpers (Yauheni Kaliuta) [1874006]
- bpf: Don't rely on GCC __attribute__((optimize)) to disable GCSE (Jiri Benc) [1874006]
- bpf: Fix tests for local_storage (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix "dubious pointer arithmetic" test (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix array access with signed variable test (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add test for signed 32-bit bound check bug (Yauheni Kaliuta) [1874006]
- bpf: Fix propagation of 32-bit signed bounds from 64-bit bounds. (Yauheni Kaliuta) [1874006]
- lwt_bpf: Replace preempt_disable() with migrate_disable() (Yauheni Kaliuta) [1874006]
- lwt: Disable BH too in run_lwt_bpf() (Yauheni Kaliuta) [1874006]
- selftests/bpf: Drain ringbuf samples at the end of test (Yauheni Kaliuta) [1874006]
- libbpf: Fix ring_buffer__poll() to return number of consumed samples (Yauheni Kaliuta) [1874006]
- bpf: Add MAINTAINERS entry for BPF LSM (Yauheni Kaliuta) [1874006]
- bpftool: Fix error return value in build_btf_type_table (Yauheni Kaliuta) [1874006]
- fail_function: Remove a redundant mutex unlock (Yauheni Kaliuta) [1874006]
- selftest/bpf: Test bpf_probe_read_user_str() strips trailing bytes after NUL (Yauheni Kaliuta) [1874006]
- lib/strncpy_from_user.c: Mask out bytes after NUL terminator. (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Avoid failures from skb_to_sgvec when skb has frag_list (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Handle memory acct if skb_verdict prog redirects to self (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Avoid returning unneeded EAGAIN when redirecting to self (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Use truesize with sk_rmem_schedule() (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Ensure SO_RCVBUF memory is observed on ingress redirect (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Fix partial copy_page_to_iter so progress can still be made (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix error return code in run_getsockopt_test() (Yauheni Kaliuta) [1874006]
- bpf: Relax return code check for subprograms (Yauheni Kaliuta) [1874006]
- tools, bpftool: Add missing close before bpftool net attach exit (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix unused attribute usage in subprogs_unused test (Yauheni Kaliuta) [1874006]
- bpf: Fix unsigned 'datasec_id' compared with zero in check_pseudo_btf_id (Yauheni Kaliuta) [1874006]
- bpf: Fix passing zero to PTR_ERR() in bpf_btf_printf_prepare (Yauheni Kaliuta) [1874006]
- libbpf: Don't attempt to load unused subprog as an entry-point BPF program (Yauheni Kaliuta) [1874006]
- bpf: Update verification logic for LSM programs (Yauheni Kaliuta) [1874006]
- bpf: Zero-fill re-used per-cpu map element (Yauheni Kaliuta) [1874006]
- bpf: BPF_PRELOAD depends on BPF_SYSCALL (Yauheni Kaliuta) [1874006]
- libbpf: Fix possible use after free in xsk_socket__delete (Yauheni Kaliuta) [1874006]
- libbpf: Fix null dereference in xsk_socket__delete (Yauheni Kaliuta) [1874006]
- libbpf, hashmap: Fix undefined behavior in hash_bits (Yauheni Kaliuta) [1874006]
- tools, bpftool: Remove two unused variables. (Yauheni Kaliuta) [1874006]
- tools, bpftool: Avoid array index warnings. (Yauheni Kaliuta) [1874006]
- bpf: Add struct bpf_redir_neigh forward declaration to BPF helper defs (Yauheni Kaliuta) [1874006]
- bpf: Fix -Wshadow warnings (Yauheni Kaliuta) [1874006]
- bpf, libbpf: Guard bpf inline asm from bpf_tail_call_static (Yauheni Kaliuta) [1874006]
- bpf: selftest: Ensure the return value of the bpf_per_cpu_ptr() must be checked (Yauheni Kaliuta) [1874006]
- bpf: selftest: Ensure the return value of bpf_skc_to helpers must be checked (Yauheni Kaliuta) [1874006]
- bpf: Remove unneeded break (Yauheni Kaliuta) [1874006]
- net, sockmap: Don't call bpf_prog_put() on NULL pointer (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Add locking annotations to iterator (Yauheni Kaliuta) [1874006]
- bpf: Fix register equivalence tracking. (Yauheni Kaliuta) [1874006]
- bpf, selftests: Add three new sockmap tests for verdict only programs (Yauheni Kaliuta) [1874006]
- bpf, selftests: Add option to test_sockmap to omit adding parser program (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Allow skipping sk_skb parser program (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Check skb_verdict and skb_parser programs explicitly (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Add memory accounting so skbs on ingress lists are visible (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Remove skb_orphan and let normal skb_kfree do cleanup (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Remove dropped data on errors in redirect case (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Remove skb_set_owner_w wmem will be taken later from sendpage (Yauheni Kaliuta) [1874006]
- bpf, sockmap: On receive programs try to fast track SK_PASS ingress (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Skb verdict SK_PASS to self already checked rmem limits (Yauheni Kaliuta) [1874006]
- bpf: Migrate from patchwork.ozlabs.org to patchwork.kernel.org. (Yauheni Kaliuta) [1874006]
- samples: bpf: Refactor XDP kern program maps with BTF-defined map (Yauheni Kaliuta) [1874006]
- samples: bpf: Replace attach_tracepoint() to attach() in xdp_redirect_cpu (Yauheni Kaliuta) [1874006]
- samples: bpf: Refactor xdp_monitor with libbpf (Yauheni Kaliuta) [1874006]
- bpf, selftests: Add test for different array inner map size (Yauheni Kaliuta) [1874006]
- bpf: Allow for map-in-map with dynamic inner array map entries (Yauheni Kaliuta) [1874006]
- selftests/bpf: Asm tests for the verifier regalloc tracking. (Yauheni Kaliuta) [1874006]
- bpf: Fix selftest compilation on clang 11 (Yauheni Kaliuta) [1874006]
- selftest/bpf: Fix profiler test using CO-RE relocation for enums (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add profiler test (Yauheni Kaliuta) [1874006]
- bpf: Track spill/fill of bounded scalars. (Yauheni Kaliuta) [1874006]
- bpf: Propagate scalar ranges through register assignments. (Yauheni Kaliuta) [1874006]
- bpf: Add tcp_notsent_lowat bpf setsockopt (Yauheni Kaliuta) [1874006]
- selftests/bpf: Validate libbpf's auto-sizing of LD/ST/STX instructions (Yauheni Kaliuta) [1874006]
- libbpf: Allow specifying both ELF and raw BTF for CO-RE BTF override (Yauheni Kaliuta) [1874006]
- libbpf: Support safe subset of load/store instruction resizing with CO-RE (Yauheni Kaliuta) [1874006]
- libbpf: Skip CO-RE relocations for not loaded BPF programs (Yauheni Kaliuta) [1874006]
- libbpf: Fix compatibility problem in xsk_socket__create (Yauheni Kaliuta) [1874006]
- bpf: Fix typo in uapi/linux/bpf.h (Yauheni Kaliuta) [1874006]
- bpf: Fix build failure for kernel/trace/bpf_trace.c with CONFIG_NET=n (Yauheni Kaliuta) [1874006]
- kernel/bpf/verifier: Fix build when NET is not enabled (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix test_verifier after introducing resolve_pseudo_ldimm64 (Yauheni Kaliuta) [1874006]
- samples: bpf: Driver interrupt statistics in xdpsock (Yauheni Kaliuta) [1874006]
- samples: bpf: Count syscalls in xdpsock (Yauheni Kaliuta) [1874006]
- samples: bpf: Split xdpsock stats into new struct (Yauheni Kaliuta) [1874006]
- samples/bpf: Fix a compilation error with fallthrough marking (Yauheni Kaliuta) [1874006]
- samples/bpf: Change Makefile to cope with latest llvm (Yauheni Kaliuta) [1874006]
- bpf, libbpf: Use valid btf in bpf_program__set_attach_target (Yauheni Kaliuta) [1874006]
- selftest/bpf: Test pinning map with reused map fd (Yauheni Kaliuta) [1874006]
- libbpf: Check if pin_path was set even map fd exist (Yauheni Kaliuta) [1874006]
- libbpf: Close map fd if init map slots failed (Yauheni Kaliuta) [1874006]
- bpf: Use raw_spin_trylock() for pcpu_freelist_push/pop in NMI (Yauheni Kaliuta) [1874006]
- bpf, verifier: Use fallthrough pseudo-keyword (Yauheni Kaliuta) [1874006]
- bpf: Deref map in BPF_PROG_BIND_MAP when it's already used (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Update selftests to use skb_adjust_room (Yauheni Kaliuta) [1874006]
- bpf, sockmap: Add skb_adjust_room to pop bytes off ingress payload (Yauheni Kaliuta) [1874006]
- bpf/selftests: Test for bpf_per_cpu_ptr() and bpf_this_cpu_ptr() (Yauheni Kaliuta) [1874006]
- bpf: Introducte bpf_this_cpu_ptr() (Yauheni Kaliuta) [1874006]
- bpf: Introduce bpf_per_cpu_ptr() (Yauheni Kaliuta) [1874006]
- selftests/bpf: Ksyms_btf to test typed ksyms (Yauheni Kaliuta) [1874006]
- bpf/libbpf: BTF support for typed ksyms (Yauheni Kaliuta) [1874006]
- bpf: Introduce pseudo_btf_id (Yauheni Kaliuta) [1874006]
- selftests/bpf: Properly initialize linfo in sockmap_basic (Yauheni Kaliuta) [1874006]
- selftests/bpf: Initialize duration in xdp_noinline.c (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add tests for BPF_F_PRESERVE_ELEMS (Yauheni Kaliuta) [1874006]
- bpf: Introduce BPF_F_PRESERVE_ELEMS for perf event array (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix alignment of .BTF_ids (Yauheni Kaliuta) [1874006]
- selftests/bpf: Test "incremental" btf_dump in C format (Yauheni Kaliuta) [1874006]
- libbpf: Make btf_dump work with modifiable BTF (Yauheni Kaliuta) [1874006]
- bpf, selftests: Use bpf_tail_call_static where appropriate (Yauheni Kaliuta) [1874006]
- bpf, libbpf: Add bpf_tail_call_static helper for bpf programs (Yauheni Kaliuta) [1874006]
- bpf, net: Rework cookie generator as per-cpu one (Yauheni Kaliuta) [1874006]
- bpf: Add classid helper only based on skb->sk (Yauheni Kaliuta) [1874006]
- bpf: fix raw_tp test run in preempt kernel (Yauheni Kaliuta) [1874006]
- libbpf: Compile in PIC mode only for shared library case (Yauheni Kaliuta) [1874006]
- libbpf: Compile libbpf under -O2 level by default and catch extra warnings (Yauheni Kaliuta) [1874006]
- bpf: x64: Do not emit sub/add 0, rsp when !stack_depth (Yauheni Kaliuta) [1874006]
- bpf, x64: Drop "pop rcx" instruction on BPF JIT epilogue (Yauheni Kaliuta) [1874006]
- bpf, selftests: Fix warning in snprintf_btf where system() call unchecked (Yauheni Kaliuta) [1874006]
- selftests/bpf: Test BTF's handling of endianness (Yauheni Kaliuta) [1874006]
- selftests/bpf: Move and extend ASSERT_xxx() testing macros (Yauheni Kaliuta) [1874006]
- selftests: Make sure all 'skel' variables are declared static (Yauheni Kaliuta) [1874006]
- selftests/bpf_iter: Don't fail test due to missing __builtin_btf_type_id (Yauheni Kaliuta) [1874006]
- bpf/preload: Make sure Makefile cleans up after itself, and add .gitignore (Yauheni Kaliuta) [1874006]
- selftests/bpf: Ensure snprintf_btf/bpf_iter tests compatibility with old vmlinux.h (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix unused-result warning in snprintf_btf.c (Yauheni Kaliuta) [1874006]
- bpf, selftests: Fix cast to smaller integer type 'int' warning in raw_tp (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add test for bpf_seq_printf_btf helper (Yauheni Kaliuta) [1874006]
- bpf: Add bpf_seq_printf_btf helper (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix overflow tests to reflect iter size increase (Yauheni Kaliuta) [1874006]
- bpf: Bump iter seq size to support BTF representation of large data structures (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add bpf_snprintf_btf helper tests (Yauheni Kaliuta) [1874006]
- bpf: Add bpf_snprintf_btf helper (Yauheni Kaliuta) [1874006]
- bpf: Move to generic BTF show support, apply it to seq files/strings (Yauheni Kaliuta) [1874006]
- bpf: Provide function to get vmlinux BTF information (Yauheni Kaliuta) [1874006]
- selftest: bpf: Test copying a sockmap and sockhash (Yauheni Kaliuta) [1874006]
- selftests: bpf: Remove shared header from sockmap iter test (Yauheni Kaliuta) [1874006]
- selftests: bpf: Add helper to compare socket cookies (Yauheni Kaliuta) [1874006]
- bpf: sockmap: Enable map_update_elem from bpf_iter (Yauheni Kaliuta) [1874006]
- bpf, cpumap: Remove rcpu pointer from cpu_map_build_skb signature (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add raw_tp_test_run (Yauheni Kaliuta) [1874006]
- libbpf: Support test run of raw tracepoint programs (Yauheni Kaliuta) [1874006]
- bpf: Enable BPF_PROG_TEST_RUN for raw_tracepoint (Yauheni Kaliuta) [1874006]
- bpf: Add comment to document BTF type PTR_TO_BTF_ID_OR_NULL (Yauheni Kaliuta) [1874006]
- bpf: Add AND verifier test case where 32bit and 64bit bounds differ (Yauheni Kaliuta) [1874006]
- bpf, verifier: Remove redundant var_off.value ops in scalar known reg cases (Yauheni Kaliuta) [1874006]
- bpf: selftest: Add test_btf_skc_cls_ingress (Yauheni Kaliuta) [1874006]
- bpf: selftest: Remove enum tcp_ca_state from bpf_tcp_helpers.h (Yauheni Kaliuta) [1874006]
- bpf: selftest: Use bpf_skc_to_tcp_sock() in the sock_fields test (Yauheni Kaliuta) [1874006]
- bpf: selftest: Use network_helpers in the sock_fields test (Yauheni Kaliuta) [1874006]
- bpf: selftest: Adapt sock_fields test to use skel and global variables (Yauheni Kaliuta) [1874006]
- bpf: selftest: Move sock_fields test into test_progs (Yauheni Kaliuta) [1874006]
- bpf: selftest: Add ref_tracking verifier test for bpf_skc casting (Yauheni Kaliuta) [1874006]
- bpf: Change bpf_sk_assign to accept ARG_PTR_TO_BTF_ID_SOCK_COMMON (Yauheni Kaliuta) [1874006]
- bpf: Change bpf_tcp_*_syncookie to accept ARG_PTR_TO_BTF_ID_SOCK_COMMON (Yauheni Kaliuta) [1874006]
- bpf: Change bpf_sk_storage_*() to accept ARG_PTR_TO_BTF_ID_SOCK_COMMON (Yauheni Kaliuta) [1874006]
- bpf: Change bpf_sk_release and bpf_sk_*cgroup_id to accept ARG_PTR_TO_BTF_ID_SOCK_COMMON (Yauheni Kaliuta) [1874006]
- bpf: Enable bpf_skc_to_* sock casting helper to networking prog type (Yauheni Kaliuta) [1874006]
- bpf: Move the PTR_TO_BTF_ID check to check_reg_type() (Yauheni Kaliuta) [1874006]
- Revert "bpf: Fix potential call bpf_link_free() in atomic context" (Yauheni Kaliuta) [1874006]
- tools resolve_btfids: Always force HOSTARCH (Yauheni Kaliuta) [1874006]
- bpf: Check CONFIG_BPF option for resolve_btfids (Yauheni Kaliuta) [1874006]
- bpf: Explicitly size compatible_reg_types (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix stat probe in d_path test (Yauheni Kaliuta) [1874006]
- bpf: Using rcu_read_lock for bpf_sk_storage_map iterator (Yauheni Kaliuta) [1874006]
- bpf: Use a table to drive helper arg type checks (Yauheni Kaliuta) [1874006]
- bpf: Hoist type checking for nullable arg types (Yauheni Kaliuta) [1874006]
- bpf: Check ARG_PTR_TO_SPINLOCK register type in check_func_arg (Yauheni Kaliuta) [1874006]
- bpf: Set meta->raw_mode for pointers close to use (Yauheni Kaliuta) [1874006]
- bpf: Make context access check generic (Yauheni Kaliuta) [1874006]
- bpf: Make reference tracking generic (Yauheni Kaliuta) [1874006]
- bpf: Make BTF pointer type checking generic (Yauheni Kaliuta) [1874006]
- bpf: Allow specifying a BTF ID per argument in function protos (Yauheni Kaliuta) [1874006]
- btf: Add BTF_ID_LIST_SINGLE macro (Yauheni Kaliuta) [1874006]
- bpf: Check scalar or invalid register in check_helper_mem_access (Yauheni Kaliuta) [1874006]
- btf: Make btf_set_contains take a const pointer (Yauheni Kaliuta) [1874006]
- bpf: Fix potential call bpf_link_free() in atomic context (Yauheni Kaliuta) [1874006]
- bpf: Use hlist_add_head_rcu when linking to local_storage (Yauheni Kaliuta) [1874006]
- samples/bpf: Fix test_map_in_map on s390 (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix endianness issue in test_sockopt_sk (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix endianness issue in sk_assign (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add tailcall_bpf2bpf tests (Yauheni Kaliuta) [1874006]
- bpf: Add abnormal return checks. (Yauheni Kaliuta) [1874006]
- bpf: allow for tailcalls in BPF subprograms for x64 JIT (Yauheni Kaliuta) [1874006]
- bpf, x64: rework pro/epilogue and tailcall handling in JIT (Yauheni Kaliuta) [1874006]
- bpf: Limit caller's stack depth 256 for subprogs with tailcalls (Yauheni Kaliuta) [1874006]
- bpf: rename poke descriptor's 'ip' member to 'tailcall_target' (Yauheni Kaliuta) [1874006]
- bpf: propagate poke descriptors to subprograms (Yauheni Kaliuta) [1874006]
- bpf, x64: use rcx instead of rax for tail call retpolines (Yauheni Kaliuta) [1874006]
- selftests/bpf: Merge most of test_btf into test_progs (Yauheni Kaliuta) [1874006]
- selftests/bpf: Test load and dump metadata with btftool and skel (Yauheni Kaliuta) [1874006]
- bpftool: Support dumping metadata (Yauheni Kaliuta) [1874006]
- libbpf: Add BPF_PROG_BIND_MAP syscall and use it on .rodata section (Yauheni Kaliuta) [1874006]
- bpf: Add BPF_PROG_BIND_MAP syscall (Yauheni Kaliuta) [1874006]
- bpf: Mutex protect used_maps array and count (Yauheni Kaliuta) [1874006]
- libbpf: Fix a compilation error with xsk.c for ubuntu 16.04 (Yauheni Kaliuta) [1874006]
- samples/bpf: Add quiet option to xdpsock (Yauheni Kaliuta) [1874006]
- samples: bpf: Add an option for printing extra statistics in xdpsock (Yauheni Kaliuta) [1874006]
- samples/bpf: Fix possible deadlock in xdpsock (Yauheni Kaliuta) [1874006]
- samples/bpf: Fix one packet sending in xdpsock (Yauheni Kaliuta) [1874006]
- s390/bpf: Fix multiple tail calls (Yauheni Kaliuta) [1874006]
- tcp: Simplify tcp_set_congestion_control() load=false case (Yauheni Kaliuta) [1874006]
- tcp: simplify _bpf_setsockopt(): Remove flags argument (Yauheni Kaliuta) [1874006]
- tcp: simplify tcp_set_congestion_control(): Always reinitialize (Yauheni Kaliuta) [1874006]
- tcp: Simplify EBPF TCP_CONGESTION to always init CC (Yauheni Kaliuta) [1874006]
- bpftool: Fix build failure (Yauheni Kaliuta) [1874006]
- tools: bpftool: Automate generation for "SEE ALSO" sections in man pages (Yauheni Kaliuta) [1874006]
- bpf: Fix comment for helper bpf_current_task_under_cgroup() (Yauheni Kaliuta) [1874006]
- selftests/bpf: Define string const as global for test_sysctl_prog.c (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix test_ksyms on non-SMP kernels (Yauheni Kaliuta) [1874006]
- tools: bpftool: Add "inner_map" to "bpftool map create" outer maps (Yauheni Kaliuta) [1874006]
- tools: bpftool: Keep errors for map-of-map dumps if distinct from ENOENT (Yauheni Kaliuta) [1874006]
- tools: bpftool: Clean up function to dump map entry (Yauheni Kaliuta) [1874006]
- selftests: bpf: Test iterating a sockmap (Yauheni Kaliuta) [1874006]
- net: Allow iterating sockmap and sockhash (Yauheni Kaliuta) [1874006]
- net: sockmap: Remove unnecessary sk_fullsock checks (Yauheni Kaliuta) [1874006]
- tools: bpftool: Include common options from separate file (Yauheni Kaliuta) [1874006]
- tools: bpftool: Print optional built-in features along with version (Yauheni Kaliuta) [1874006]
- selftests, bpftool: Add bpftool (and eBPF helpers) documentation build (Yauheni Kaliuta) [1874006]
- tools: bpftool: Log info-level messages when building bpftool man pages (Yauheni Kaliuta) [1874006]
- bpf: Remove duplicate headers (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix test_sysctl_loop{1, 2} failure due to clang change (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add test for map_ptr arithmetic (Yauheni Kaliuta) [1874006]
- bpf: Permit map_ptr arithmetic with opcode add and offset 0 (Yauheni Kaliuta) [1874006]
- tools, bpf: Synchronise BPF UAPI header with tools (Yauheni Kaliuta) [1874006]
- bpf: Fix formatting in documentation for BPF helpers (Yauheni Kaliuta) [1874006]
- tools: bpftool: Fix formatting in bpftool-link documentation (Yauheni Kaliuta) [1874006]
- samples, bpf: Add xsk_fwd test file to .gitignore (Yauheni Kaliuta) [1874006]
- samples, bpf: Replace bpf_program__title() with bpf_program__section_name() (Yauheni Kaliuta) [1874006]
- libbpf: Fix potential multiplication overflow (Yauheni Kaliuta) [1874006]
- libbpf: Fix another __u64 cast in printf (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix check in global_data_init. (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add __noinline variant of cls_redirect selftest (Yauheni Kaliuta) [1874006]
- selftests/bpf: Modernize xdp_noinline test w/ skeleton and __noinline (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add subprogs to pyperf, strobemeta, and l4lb_noinline tests (Yauheni Kaliuta) [1874006]
- tools/bpftool: Replace bpf_program__title() with bpf_program__section_name() (Yauheni Kaliuta) [1874006]
- selftests/bpf: Don't use deprecated libbpf APIs (Yauheni Kaliuta) [1874006]
- libbpf: Deprecate notion of BPF program "title" in favor of "section name" (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add selftest for multi-prog sections and bpf-to-bpf calls (Yauheni Kaliuta) [1874006]
- libbpf: Add multi-prog section support for struct_ops (Yauheni Kaliuta) [1874006]
- libbpf: Implement generalized .BTF.ext func/line info adjustment (Yauheni Kaliuta) [1874006]
- libbpf: Make RELO_CALL work for multi-prog sections and sub-program calls (Yauheni Kaliuta) [1874006]
- libbpf: Support CO-RE relocations for multi-prog sections (Yauheni Kaliuta) [1874006]
- libbpf: Parse multi-function sections into multiple BPF programs (Yauheni Kaliuta) [1874006]
- libbpf: Ensure ELF symbols table is found before further ELF processing (Yauheni Kaliuta) [1874006]
- selftests/bpf: Test task_file iterator without visiting pthreads (Yauheni Kaliuta) [1874006]
- bpf: Avoid iterating duplicated files for task_file iterator (Yauheni Kaliuta) [1874006]
- bpf: {cpu,dev}map: Change various functions return type from int to void (Yauheni Kaliuta) [1874006]
- module: Fix up module_notifier return values (Yauheni Kaliuta) [1874006]
- bpf: Remove bpf_lsm_file_mprotect from sleepable list. (Yauheni Kaliuta) [1874006]
- samples/bpf: Fix to xdpsock to avoid recycling frames (Yauheni Kaliuta) [1874006]
- samples/bpf: Optimize l2fwd performance in xdpsock (Yauheni Kaliuta) [1874006]
- samples/bpf: Add new sample xsk_fwd.c (Yauheni Kaliuta) [1874006]
- libbpf: Support shared umems between queues and devices (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add sleepable tests (Yauheni Kaliuta) [1874006]
- libbpf: Support sleepable progs (Yauheni Kaliuta) [1874006]
- bpf: Add bpf_copy_from_user() helper. (Yauheni Kaliuta) [1874006]
- bpf: Fix build without BPF_LSM. (Yauheni Kaliuta) [1874006]
- bpf: Introduce sleepable BPF programs (Yauheni Kaliuta) [1874006]
- mm/error_inject: Fix allow_error_inject function signatures. (Yauheni Kaliuta) [1874006]
- mm/filemap.c: enable error injection at add_to_page_cache() (Yauheni Kaliuta) [1874006]
- mm/page_alloc.c: allow error injection (Yauheni Kaliuta) [1874006]
- bpf: selftests: Add test for different inner map size (Yauheni Kaliuta) [1874006]
- bpf: Relax max_entries check for most of the inner map types (Yauheni Kaliuta) [1874006]
- bpf: Add map_meta_equal map ops (Yauheni Kaliuta) [1874006]
- bpf: Make bpf_link_info.iter similar to bpf_iter_link_info (Yauheni Kaliuta) [1874006]
- tools, bpf/build: Cleanup feature files on make clean (Yauheni Kaliuta) [1874006]
- libbpf: Fix compilation warnings for 64-bit printf args (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add verifier tests for xor operation (Yauheni Kaliuta) [1874006]
- libbpf: Fix unintentional success return code in bpf_object__load (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix spelling mistake "scoket" -> "socket" (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix open call in trigger_fstat_events (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add set test to resolve_btfids (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add test for d_path helper (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add verifier test for d_path helper (Yauheni Kaliuta) [1874006]
- bpf: Update .BTF_ids section in btf.rst with sets info (Yauheni Kaliuta) [1874006]
- bpf: Add d_path helper (Yauheni Kaliuta) [1874006]
- bpf: Add BTF_SET_START/END macros (Yauheni Kaliuta) [1874006]
- bpf: Add btf_struct_ids_match function (Yauheni Kaliuta) [1874006]
- bpf: Factor btf_struct_access function (Yauheni Kaliuta) [1874006]
- bpf: Remove recursion call in btf_struct_access (Yauheni Kaliuta) [1874006]
- bpf: Add type_id pointer as argument to __btf_resolve_size (Yauheni Kaliuta) [1874006]
- bpf: Add elem_id pointer as argument to __btf_resolve_size (Yauheni Kaliuta) [1874006]
- bpf: Move btf_resolve_size into __btf_resolve_size (Yauheni Kaliuta) [1874006]
- tools resolve_btfids: Add support for set symbols (Yauheni Kaliuta) [1874006]
- tools resolve_btfids: Add size check to get_id function (Yauheni Kaliuta) [1874006]
- bpf: Disallow BPF_PRELOAD in allmodconfig builds (Yauheni Kaliuta) [1874006]
- bpf: Add selftests for local_storage (Yauheni Kaliuta) [1874006]
- bpf: Allow local storage to be used from LSM programs (Yauheni Kaliuta) [1874006]
- bpf: Implement bpf_local_storage for inodes (Yauheni Kaliuta) [1874006]
- bpf: Split bpf_local_storage to bpf_sk_storage (Yauheni Kaliuta) [1874006]
- bpf: Generalize bpf_sk_storage (Yauheni Kaliuta) [1874006]
- bpf: Generalize caching for sk_storage. (Yauheni Kaliuta) [1874006]
- bpf: Renames in preparation for bpf_local_storage (Yauheni Kaliuta) [1874006]
- selftests/bpf: Enable tc verbose mode for test_sk_assign (Yauheni Kaliuta) [1874006]
- samples: bpf: Refactor tracepoint tracing programs with libbpf (Yauheni Kaliuta) [1874006]
- samples: bpf: Refactor kprobe tracing programs with libbpf (Yauheni Kaliuta) [1874006]
- selftests: bpf: Fix sockmap update nits (Yauheni Kaliuta) [1874006]
- libbpf: Fix type compatibility check copy-paste error (Yauheni Kaliuta) [1874006]
- libbpf: Avoid false unuinitialized variable warning in bpf_core_apply_relo (Yauheni Kaliuta) [1874006]
- bpf: sk_lookup: Add user documentation (Yauheni Kaliuta) [1874006]
- docs: Correct subject prefix and update LLVM info (Yauheni Kaliuta) [1874006]
- libbpf: Normalize and improve logging across few functions (Yauheni Kaliuta) [1874006]
- libbpf: Skip well-known ELF sections when iterating ELF (Yauheni Kaliuta) [1874006]
- libbpf: Add __noinline macro to bpf_helpers.h (Yauheni Kaliuta) [1874006]
- libbpf: Factor out common ELF operations and improve logging (Yauheni Kaliuta) [1874006]
- libbpf: Add perf_buffer APIs for better integration with outside epoll loop (Yauheni Kaliuta) [1874006]
- bpftool: Implement link_query for bpf iterators (Yauheni Kaliuta) [1874006]
- bpf: Implement link_query callbacks in map element iterators (Yauheni Kaliuta) [1874006]
- bpf: Implement link_query for bpf iterators (Yauheni Kaliuta) [1874006]
- selftests/bpf: List newest Clang built-ins needed for some CO-RE selftests (Yauheni Kaliuta) [1874006]
- selftests/bpf: Fix two minor compilation warnings reported by GCC 4.9 (Yauheni Kaliuta) [1874006]
- libbpf: Fix libbpf build on compilers missing __builtin_mul_overflow (Yauheni Kaliuta) [1874006]
- libbpf: Fix detection of BPF helper call instruction (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add bpffs preload test. (Yauheni Kaliuta) [1874006]
- bpf: Add kernel module with user mode driver that populates bpffs. (Yauheni Kaliuta) [1874006]
- bpf: Add BPF program and map iterators as built-in BPF programs. (Yauheni Kaliuta) [1874006]
- bpf: Factor out bpf_link_by_id() helper. (Yauheni Kaliuta) [1874006]
- libbpf: Simplify the return expression of build_map_pin_path() (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add tests for ENUMVAL_EXISTS/ENUMVAL_VALUE relocations (Yauheni Kaliuta) [1874006]
- libbpf: Implement enum value-based CO-RE relocations (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add CO-RE relo test for TYPE_ID_LOCAL/TYPE_ID_TARGET (Yauheni Kaliuta) [1874006]
- selftests/bpf: Test TYPE_EXISTS and TYPE_SIZE CO-RE relocations (Yauheni Kaliuta) [1874006]
- libbpf: Implement type-based CO-RE relocations support (Yauheni Kaliuta) [1874006]
- libbpf: Centralize poisoning and poison reallocarray() (Yauheni Kaliuta) [1874006]
- tools/bpftool: Remove libbpf_internal.h usage in bpftool (Yauheni Kaliuta) [1874006]
- selftests/bpf: Add test validating failure on ambiguous relocation value (Yauheni Kaliuta) [1874006]
- libbpf: Improve relocation ambiguity detection (Yauheni Kaliuta) [1874006]
- libbpf: Detect minimal BTF support and skip BTF loading, if missing (Yauheni Kaliuta) [1874006]
- libbpf: Sanitize BPF program code for bpf_probe_read_{kernel, user}[_str] (Yauheni Kaliuta) [1874006]
- libbpf: Factor out common logic of testing and closing FD (Yauheni Kaliuta) [1874006]
- libbpf: Make kernel feature probing lazy (Yauheni Kaliuta) [1874006]
- libbpf: Disable -Wswitch-enum compiler warning (Yauheni Kaliuta) [1874006]
- libbpf: Convert comma to semicolon (Yauheni Kaliuta) [1874006]
- samples: bpf: Fix broken bpf programs due to removed symbol (Yauheni Kaliuta) [1874006]
- bpf: Replace GPLv2 boilerplate/reference with SPDX - rule 206 (Jiri Benc) [1874006]
- LSM: Make some functions static (Yauheni Kaliuta) [1874006]
- LSM: Make lsm_early_cred() and lsm_early_task() local functions. (Yauheni Kaliuta) [1874006]
- LSM: Infrastructure management of the ipc security blob (Yauheni Kaliuta) [1874006]
- LSM: Infrastructure management of the task security (Yauheni Kaliuta) [1874006]
- LSM: Infrastructure management of the inode security (Yauheni Kaliuta) [1874006]
- LSM: Infrastructure management of the file security (Yauheni Kaliuta) [1874006]
- Infrastructure management of the cred security blob (Yauheni Kaliuta) [1874006]
- kernel: move CONFIG_TASKS_TRACE_RCU fields to task_struct_rh (Jiri Benc) [1874006]
- kernel: store pointer to task_struct in task_struct_rh (Jiri Benc) [1874006]

* Wed Jun 02 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-311.el8]
- igb: Add double-check MTA_REGISTER for i210 and i211 (Corinna Vinschen) [1920286]
- igb: Redistribute memory for transmit packet buffers when in Qav mode (Corinna Vinschen) [1920286]
- igb: Fix fall-through warnings for Clang (Corinna Vinschen) [1920286]
- intel: clean up mismatched header comments (Corinna Vinschen) [1920286]
- igb: check timestamp validity (Corinna Vinschen) [1920286]
- igb: Fix duplicate include guard (Corinna Vinschen) [1920286]
- intel: Update drivers to use ethtool_sprintf (Corinna Vinschen) [1920286]
- net: ethernet: intel: igb: Typo fix in the file igb_main.c (Corinna Vinschen) [1920286]
- igb: avoid premature Rx buffer reuse (Corinna Vinschen) [1920286]
- igb: remove h from printk format specifier (Corinna Vinschen) [1920286]
- igb: Enable RSS for Intel I211 Ethernet Controller (Corinna Vinschen) [1920286]
- igb: fix TDBAL register show incorrect value (Corinna Vinschen) [1920286]
- net: igb: use skb_csum_is_sctp instead of protocol check (Corinna Vinschen) [1920286]
- igb: avoid transmit queue timeout in xdp path (Corinna Vinschen) [1920286]
- igb: use xdp_do_flush (Corinna Vinschen) [1920286]
- igb: skb add metasize for xdp (Corinna Vinschen) [1920286]
- igb: XDP extack message on error (Corinna Vinschen) [1920286]
- igb: take VLAN double header into account (Corinna Vinschen) [1920286]
- igb: XDP xmit back fix error code (Corinna Vinschen) [1920286]
- igb: add XDP support (Corinna Vinschen) [1920286]
- r8169: don't advertise pause in jumbo mode (Josef Oskera) [1935346]
- r8169: fix DMA being used after buffer free if WoL is enabled (Josef Oskera) [1935346]
- r8169: fix r8168fp_adjust_ocp_cmd function (Josef Oskera) [1935346]
- r8169: use macro pm_ptr (Josef Oskera) [1935346]
- r8169: fix resuming from suspend on RTL8105e if machine runs on battery (Josef Oskera) [1935346]
- r8169: handle tx before rx in napi poll (Josef Oskera) [1935346]
- r8169: disable detection of bogus xid's 308/388 (Josef Oskera) [1935346]
- r8169: re-configure WOL settings on resume from hibernation (Josef Oskera) [1935346]
- r8169: don't try to disable interrupts if NAPI is scheduled already (Josef Oskera) [1935346]
- r8169: Add support for another RTL8168FP (Josef Oskera) [1935346]
- r8169: fix WoL on shutdown if CONFIG_DEBUG_SHIRQ is set (Josef Oskera) [1935346]
- r8169: work around RTL8125 UDP hw bug (Josef Oskera) [1935346]
- r8169: remove not needed call to rtl_wol_enable_rx from rtl_shutdown (Josef Oskera) [1935346]
- r8169: improve DASH support (Josef Oskera) [1935346]
- r8169: improve rtl8169_rx_csum (Josef Oskera) [1935346]
- r8169: align rtl_wol_suspend_quirk with vendor driver and rename it (Josef Oskera) [1935346]
- r8169: clean up rtl_pll_power_down/up functions (Josef Oskera) [1935346]
- r8169: improve handling D3 PLL power-down (Josef Oskera) [1935346]
- r8169: enable PLL power-down for chip versions 34, 35, 36, 42 (Josef Oskera) [1935346]
- r8169: make use of the unaligned access helpers (Josef Oskera) [1935346]
- r8169: tweak max read request size for newer chips also in jumbo mtu mode (Josef Oskera) [1935346]
- r8169: align RTL8168e jumbo pcie read request size with vendor driver (Josef Oskera) [1935346]
- r8169: don't wakeup-enable device on shutdown if WOL is disabled (Josef Oskera) [1935346]
- r8169: improve rtl_ocp_reg_failure (Josef Oskera) [1935346]
- r8169: replace BUG_ON with WARN in _rtl_eri_write (Josef Oskera) [1935346]
- r8169: improve RTL8168g PHY suspend quirk (Josef Oskera) [1935346]
- r8169: move ERI access functions to avoid forward declaration (Josef Oskera) [1935346]
- r8169: work around power-saving bug on some chip versions (Josef Oskera) [1935346]
- r8169: make NUM_RX_DESC a signed int (Josef Oskera) [1935346]
- r8169: improve rtl_rx (Josef Oskera) [1935346]
- r8169: set tc_offset only if tally counter reset isn't supported (Josef Oskera) [1935346]
- net: phy: realtek: read actual speed on rtl8211f to detect downshift (Josef Oskera) [1935346]
- r8169: use dev_err_probe in rtl_get_ether_clk (Josef Oskera) [1935346]
- r8169: reduce number of workaround doorbell rings (Josef Oskera) [1935346]
- r8169: remove not needed check in rtl8169_start_xmit (Josef Oskera) [1935346]
- r8169: remove nr_frags argument from rtl_tx_slots_avail (Josef Oskera) [1935346]
- r8169: improve rtl8169_start_xmit (Josef Oskera) [1935346]
- r8169: improve rtl_tx (Josef Oskera) [1935346]
- r8169: use READ_ONCE in rtl_tx_slots_avail (Josef Oskera) [1935346]
- net: phy: realtek: support paged operations on RTL8201CP (Josef Oskera) [1935346]
- r8169: disable hw csum for short packets on all chip versions (Josef Oskera) [1935346]
- r8169: fix potential skb double free in an error path (Josef Oskera) [1935346]
- r8169: work around short packet hw bug on RTL8125 (Josef Oskera) [1935346]
- net: phy: realtek: Add support for RTL8221B-CG series (Josef Oskera) [1935346]
- r8169: set IRQF_NO_THREAD if MSI(X) is enabled (Josef Oskera) [1935346]
- r8169: align number of tx descriptors with vendor driver (Josef Oskera) [1935346]
- r8169: use pm_runtime_put_sync in rtl_open error path (Josef Oskera) [1935346]
- r8169: remove unneeded memory barrier in rtl_tx (Josef Oskera) [1935346]
- r8169: remove no longer needed private rx/tx packet/byte counters (Josef Oskera) [1935346]
- r8169: use struct pcpu_sw_netstats for rx/tx packet/byte counters (Josef Oskera) [1935346]
- treewide: Use fallthrough pseudo-keyword (Josef Oskera) [1935346]
- r8169: fix jumbo packet handling on RTL8168e (Josef Oskera) [1915314]
- sctp: delay auto_asconf init until binding the first addr (Xin Long) [1952020]
- netfilter: x_tables: fix compat match/target pad out-of-bound write (Florian Westphal) [1950014]
- netfilter: arp_tables: add pre_exit hook for table unregister (Florian Westphal) [1950014]
- netfilter: bridge: add pre_exit hooks for ebtable unregistration (Florian Westphal) [1950014]
- netfilter: nft_limit: avoid possible divide error in nft_limit_init (Florian Westphal) [1950014]
- netfilter: conntrack: do not print icmpv6 as unknown via /proc (Florian Westphal) [1950014]
- netfilter: flowtable: fix NAT IPv6 offload mangling (Florian Westphal) [1950014]
- netfilter: flowtable: Make sure GC works periodically in idle system (Florian Westphal) [1950014]
- netfilter: nftables: report EOPNOTSUPP on unsupported flowtable flags (Florian Westphal) [1950014]
- netfilter: ctnetlink: fix dump of the expect mask attribute (Florian Westphal) [1950014]
- netfilter: x_tables: gpf inside xt_find_revision() (Florian Westphal) [1950014]
- netfilter: conntrack: skip identical origin tuple in same zone only (Florian Westphal) [1950014]
- flow_dissector: fix byteorder of dissected ICMP ID (Paolo Abeni) [1950288]
- flow_dissector: fix TTL and TOS dissection on IPv4 fragments (Paolo Abeni) [1950288]
- net: Disable NETIF_F_HW_TLS_RX when RXCSUM is disabled (Paolo Abeni) [1950288]
- net: core: devlink: use right genl user_ptr when handling port param get/set (Paolo Abeni) [1950288]
- net_sched: gen_estimator: support large ewma log (Paolo Abeni) [1950288]
- net: Allow NETIF_F_HW_TLS_TX if IP_CSUM && IPV6_CSUM (Paolo Abeni) [1950288]
- net: Disable NETIF_F_HW_TLS_TX when HW_CSUM is disabled (Paolo Abeni) [1950288]
- devlink: Add devlink port documentation (Petr Oros) [1935708]
- devlink: Support get and set state of port function (Petr Oros) [1935708]
- devlink: Support add and delete devlink port (Petr Oros) [1935708]
- devlink: Introduce PCI SF port flavour and port attribute (Petr Oros) [1935708]
- devlink: Prepare code to fill multiple port function attributes (Petr Oros) [1935708]
- net: core: devlink: simplify the return expression of devlink_nl_cmd_trap_set_doit() (Petr Oros) [1935708]
- video: hyperv_fb: Add ratelimit on error message (Mohammed Gamal) [1957804]
- ovl: allow upperdir inside lowerdir (Miklos Szeredi) [1903647]
- i2c: i801: Add support for Intel Alder Lake PCH-P (Steve Best) [1929488]
- pinctrl: tigerlake: Add Alder Lake-P ACPI ID (Steve Best) [1929492]
- xfs: set aside allocation btree blocks from block reservation (Brian Foster) [1911738]
- xfs: introduce in-core global counter of allocbt blocks (Brian Foster) [1911738]
- xfs: unconditionally read all AGFs on mounts with perag reservation (Brian Foster) [1911738]
- xfs: drop unnecessary setfilesize helper (Brian Foster) [1942348]
- xfs: drop unused ioend private merge and setfilesize code (Brian Foster) [1942348]
- xfs: open code ioend needs workqueue helper (Brian Foster) [1942348]
- xfs: drop submit side trans alloc for append ioends (Brian Foster) [1942348]
- blk-mq: Use request queue-wide tags for tagset-wide sbitmap (Ming Lei) [1932145]
- blk-mq: Some tag allocation code refactoring (Ming Lei) [1932145]
- blk-mq: Swap two calls in blk_mq_exit_queue() (Ming Lei) [1932145]
- blk-mq: plug request for shared sbitmap (Ming Lei) [1932145]
- blk-mq: set default elevator as deadline in case of hctx shared tagset (Ming Lei) [1932145]
- powerpc/perf: Fix PMU constraint check for EBB events (Diego Domingos) [1926573]
- s390/idle: fix suspicious RCU usage (Waiman Long) [1963142]
- x86/cpu: Add another Alder Lake CPU to the Intel family (Steve Best) [1962033]
- x86/split_lock: Don't write MSR_TEST_CTRL on CPUs that aren't whitelisted (Prarit Bhargava) [1837350]
- Documentation/admin-guide: Change doc for split_lock_detect parameter (Prarit Bhargava) [1837350]
- x86/traps: Handle #DB for bus lock (Prarit Bhargava) [1837350]
- x86/cpufeatures: Enumerate #DB for bus lock detection (Prarit Bhargava) [1837350]
- x86/split_lock: Enable the split lock feature on another Alder Lake CPU (Prarit Bhargava) [1925373]
- x86/cpu: Add another Alder Lake CPU to the Intel family (Prarit Bhargava) [1925373]
- x86/split_lock: Enable the split lock feature on Sapphire Rapids and Alder Lake CPUs (Prarit Bhargava) [1925373]

* Thu May 27 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-310.el8]
- libbpf: Add user-space variants of BPF_CORE_READ() family of macros (Jiri Olsa) [1896362]
- libbpf: Switch tracing and CO-RE helper macros to bpf_probe_read_kernel() (Jiri Olsa) [1896362]
- selftests/bpf: Fix test_vmlinux test to use bpf_probe_read_user() (Jiri Olsa) [1896362]
- samples/bpf, selftests/bpf: Use bpf_probe_read_kernel (Jiri Olsa) [1896362]
- bpf: Restrict bpf_probe_read{, str}() only to archs where they work (Jiri Olsa) [1896362]
- perf/x86/intel/uncore: Add Rocket Lake support (Michael Petlan) [1837502]
- perf/x86/cstate: Add Rocket Lake CPU support (Michael Petlan) [1837497]
- perf/x86/intel: Add Rocket Lake CPU support (Michael Petlan) [1837497]
- perf/x86/msr: Add Rocket Lake CPU support (Michael Petlan) [1837497]
- tick/nohz: Kick only _queued_ task whose tick dependency is updated (Waiman Long) [1922901]
- tick/nohz: Change signal tick dependency to wake up CPUs of member tasks (Waiman Long) [1922901]
- tick/nohz: Only wake up a single target cpu when kicking a task (Waiman Long) [1922901]
- tick/nohz: Narrow down noise while setting current task's tick dependency (Waiman Long) [1922901]
- sched/fair: Fix shift-out-of-bounds in load_balance() (Phil Auld) [1958730]
- redhat/configs: Update torture test configs (Waiman Long) [1941211]
- rcu: Do not report strict GPs for outgoing CPUs (Waiman Long) [1941211]
- rcu,ftrace: Fix ftrace recursion (Waiman Long) [1941211]
- arm64: smp: Tell RCU about CPUs that fail to come online (Waiman Long) [1941211]
- rcu: Don't invoke try_invoke_on_locked_down_task() with irqs disabled (Waiman Long) [1941211]
- rcu-tasks: Enclose task-list scan in rcu_read_lock() (Waiman Long) [1941211]
- rcu-tasks: Fix low-probability task_struct leak (Waiman Long) [1941211]
- rcu-tasks: Fix grace-period/unlock race in RCU Tasks Trace (Waiman Long) [1941211]
- rcu-tasks: Shorten per-grace-period sleep for RCU Tasks Trace (Waiman Long) [1941211]
- rcu-tasks: Selectively enable more RCU Tasks Trace IPIs (Waiman Long) [1941211]
- rcu-tasks: Use more aggressive polling for RCU Tasks Trace (Waiman Long) [1941211]
- rcu-tasks: Mark variables static (Waiman Long) [1941211]
- rcu-tasks: Prevent complaints of unused show_rcu_tasks_classic_gp_kthread() (Waiman Long) [1941211]
- rcu: Remove unused "cpu" parameter from rcu_report_qs_rdp() (Waiman Long) [1941211]
- torture: Add gdb support (Waiman Long) [1941211]
- rcuperf: Change rcuperf to rcuscale (Waiman Long) [1941211]
- sched,rcuperf: Convert to sched_set_fifo_low() (Waiman Long) [1941211]
- rcu/rcuperf: Convert to SPDX license identifier (Waiman Long) [1941211]
- rcutorture: Allow pointer leaks to test diagnostic code (Waiman Long) [1941211]
- rcu: Report QS for outermost PREEMPT=n rcu_read_unlock() for strict GPs (Waiman Long) [1941211]
- rcu: locking and unlocking need to always be at least barriers (Waiman Long) [1941211]
- rcu: Execute RCU reader shortly after rcu_core for strict GPs (Waiman Long) [1941211]
- rcu: Provide optional RCU-reader exit delay for strict GPs (Waiman Long) [1941211]
- rcu: Make FQS more aggressive in complaining about offline CPUs (Waiman Long) [1941211]
- rcu: Clarify comments about FQS loop reporting quiescent states (Waiman Long) [1941211]
- rcu: IPI all CPUs at GP end for strict GPs (Waiman Long) [1941211]
- rcu: IPI all CPUs at GP start for strict GPs (Waiman Long) [1941211]
- rcu: Attempt QS when CPU discovers GP for strict GPs (Waiman Long) [1941211]
- rcu: Do full report for .need_qs for strict GPs (Waiman Long) [1941211]
- rcu: Always set .need_qs from __rcu_read_lock() for strict GPs (Waiman Long) [1941211]
- rcu: Force DEFAULT_RCU_BLIMIT to 1000 for strict RCU GPs (Waiman Long) [1941211]
- rcu: Restrict default jiffies_till_first_fqs for strict RCU GPs (Waiman Long) [1941211]
- rcu: Reduce leaf fanout for strict RCU grace periods (Waiman Long) [1941211]
- rcu: Add Kconfig option for strict RCU grace periods (Waiman Long) [1941211]
- rcu/nocb: Add a warning for non-GP kthread running GP code (Waiman Long) [1941211]
- rcu: Move rcu_cpu_started per-CPU variable to rcu_data (Waiman Long) [1941211]
- scftorture: Add cond_resched() to test loop (Waiman Long) [1941211]
- rcutorture: Hoist OOM registry up one level (Waiman Long) [1941211]
- rcutorture: Properly synchronize with OOM notifier (Waiman Long) [1941211]
- rcutorture: Properly set rcu_fwds for OOM handling (Waiman Long) [1941211]
- torture: Add kvm.sh --help and update help message (Waiman Long) [1941211]
- refperf: Avoid null pointer dereference when buf fails to allocate (Waiman Long) [1941211]
- rcutorture: Add CONFIG_PROVE_RCU_LIST to TREE05 (Waiman Long) [1941211]
- torture: Update initrd documentation (Waiman Long) [1941211]
- rcutorture: Replace HTTP links with HTTPS ones (Waiman Long) [1941211]
- rcu: Fix kerneldoc comments in rcupdate.h (Waiman Long) [1941211]
- scftorture: Adapt memory-ordering test to UP operation (Waiman Long) [1941211]
- torture: document --allcpus argument added to the kvm.sh script (Waiman Long) [1941211]
- scftorture: Block scftorture_invoker() kthreads for offline CPUs (Waiman Long) [1941211]
- scftorture: Check unexpected "switch" statement value (Waiman Long) [1941211]
- scftorture: Make symbol 'scf_torture_rand' static (Waiman Long) [1941211]
- scftorture: Prevent compiler from reducing race probabilities (Waiman Long) [1941211]
- scftorture: Flag errors in torture-compatible manner (Waiman Long) [1941211]
- scftorture: Consolidate scftorture_invoke_one() scf_check initialization (Waiman Long) [1941211]
- scftorture: Consolidate scftorture_invoke_one() check and kfree() (Waiman Long) [1941211]
- scftorture: Add smp_call_function() memory-ordering checks (Waiman Long) [1941211]
- scftorture: Add smp_call_function_many() memory-ordering checks (Waiman Long) [1941211]
- scftorture: Add smp_call_function_single() memory-ordering checks (Waiman Long) [1941211]
- scftorture: Summarize per-thread statistics (Waiman Long) [1941211]
- torture: Declare parse-console.sh independence from rcutorture (Waiman Long) [1941211]
- tick-sched: Clarify "NOHZ: local_softirq_pending" warning (Waiman Long) [1941211]
- scftorture: Implement weighted primitive selection (Waiman Long) [1941211]
- torture: Add scftorture to the rcutorture scripting (Waiman Long) [1941211]
- scftorture: Add smp_call_function() torture test (Waiman Long) [1941211]
- rcu: Add READ_ONCE() to rcu_do_batch() access to rcu_cpu_stall_ftrace_dump (Waiman Long) [1941211]
- rcu: Add READ_ONCE() to rcu_do_batch() access to rcu_kick_kthreads (Waiman Long) [1941211]
- rcu: Add READ_ONCE() to rcu_do_batch() access to rcu_resched_ns (Waiman Long) [1941211]
- rcu: Add READ_ONCE() to rcu_do_batch() access to rcu_divisor (Waiman Long) [1941211]
- nocb: Remove show_rcu_nocb_state() false positive printout (Waiman Long) [1941211]
- rcu/tree: Remove CONFIG_PREMPT_RCU check in force_qs_rnp() (Waiman Long) [1941211]
- nocb: Clarify RCU nocb CPU error message (Waiman Long) [1941211]
- rcu/tree: Force quiescent state on callback overload (Waiman Long) [1941211]
- rcu/trace: Use gp_seq_req in acceleration's rcu_grace_period tracepoint (Waiman Long) [1941211]
- rcu/trace: Print negative GP numbers correctly (Waiman Long) [1941211]
- rcutorture: Output number of elapsed grace periods (Waiman Long) [1941211]
- rcu: Initialize at declaration time in rcu_exp_handler() (Waiman Long) [1941211]
- rcu: Remove KCSAN stubs from update.c (Waiman Long) [1941211]
- rcu: Remove KCSAN stubs (Waiman Long) [1941211]
- srcu: Remove KCSAN stubs (Waiman Long) [1941211]
- rcutorture: Remove KCSAN stubs (Waiman Long) [1941211]
- refscale: Change --torture type from refperf to refscale (Waiman Long) [1941211]
- refperf: Rename refperf.c to refscale.c and change internal names (Waiman Long) [1941211]
- refperf: Rename RCU_REF_PERF_TEST to RCU_REF_SCALE_TEST (Waiman Long) [1941211]
- torture: Remove obsolete "cd $KVM" (Waiman Long) [1941211]
- torture: Avoid duplicate specification of qemu command (Waiman Long) [1941211]
- torture: Dump ftrace at shutdown only if requested (Waiman Long) [1941211]
- torture: Add kvm-tranform.sh script for qemu-cmd files (Waiman Long) [1941211]
- torture: Add more tracing crib notes to kvm.sh (Waiman Long) [1941211]
- torture: Improve diagnostic for KCSAN-incapable compilers (Waiman Long) [1941211]
- torture: Correctly summarize build-only runs (Waiman Long) [1941211]
- torture: Pass --kmake-arg to all make invocations (Waiman Long) [1941211]
- kernel/rcu/tree.c: Fix kernel-doc warnings (Waiman Long) [1941211]
- rcu-tasks: Fix synchronize_rcu_tasks_trace() header comment (Waiman Long) [1941211]
- rcu: grpnum just records group number (Waiman Long) [1941211]
- rcu: grplo/grphi just records CPU number (Waiman Long) [1941211]
- rcu: gp_max is protected by root rcu_node's lock (Waiman Long) [1941211]
- rcutorture: Check for unwatched readers (Waiman Long) [1941211]
- torture: Abstract out console-log error detection (Waiman Long) [1941211]
- torture: Add a stop-run capability (Waiman Long) [1941211]
- rcu-tasks: Fix code-style issues (Waiman Long) [1941211]
- torture: Create qemu-cmd in --buildonly runs (Waiman Long) [1941211]
- rcu: Stop shrinker loop (Waiman Long) [1941211]
- refperf: Add test for RCU Tasks readers (Waiman Long) [1941211]
- refperf: Add test for RCU Tasks Trace readers. (Waiman Long) [1941211]
- rcu: Replace 1 with true (Waiman Long) [1941211]
- rcu/rcutorture: Replace 0 with false (Waiman Long) [1941211]
- refperf: Change readdelay module parameter to nanoseconds (Waiman Long) [1941211]
- refperf: Work around 64-bit division (Waiman Long) [1941211]
- doc: Document rcuperf's module parameters (Waiman Long) [1941211]
- refperf: Adjust refperf.loop default value (Waiman Long) [1941211]
- rcu-tasks: Conditionally compile show_rcu_tasks_gp_kthreads() (Waiman Long) [1941211]
- rcu-tasks: Add #include of rcupdate_trace.h to update.c (Waiman Long) [1941211]
- rcu-tasks: Make rcu_tasks_postscan() be static (Waiman Long) [1941211]
- refperf: Add read-side delay module parameter (Waiman Long) [1941211]
- lockdep: Complain only once about RCU in extended quiescent state (Waiman Long) [1941211]
- refperf: Simplify initialization-time wakeup protocol (Waiman Long) [1941211]
- rcu: Allow for smp_call_function() running callbacks from idle (Waiman Long) [1941211]
- refperf: Output per-experiment data points (Waiman Long) [1941211]
- refperf: Label experiment-number column "Runs" (Waiman Long) [1941211]
- refperf: Add warmup and cooldown processing phases (Waiman Long) [1941211]
- refperf: More closely synchronize reader start times (Waiman Long) [1941211]
- refperf: Convert reader_task structure's "start" field to int (Waiman Long) [1941211]
- refperf: Tune reader measurement interval (Waiman Long) [1941211]
- refperf: Make functions static (Waiman Long) [1941211]
- srcu: Avoid local_irq_save() before acquiring spinlock_t (Waiman Long) [1941211]
- refperf: Dynamically allocate thread-summary output buffer (Waiman Long) [1941211]
- refperf: Dynamically allocate experiment-summary output buffer (Waiman Long) [1941211]
- refperf: Provide module parameter to specify number of experiments (Waiman Long) [1941211]
- refperf: Convert nreaders to a module parameter (Waiman Long) [1941211]
- refperf: Allow decimal nanoseconds (Waiman Long) [1941211]
- refperf: Hoist function-pointer calls out of the loop (Waiman Long) [1941211]
- rcu: Introduce single argument kvfree_rcu() interface (Waiman Long) [1941211]
- refperf: Add holdoff parameter to allow CPUs to come online (Waiman Long) [1941211]
- torture: Add refperf to the rcutorture scripting (Waiman Long) [1941211]
- rcuperf: Add comments explaining the high reader overhead (Waiman Long) [1941211]
- refperf: Add a test to measure performance of read-side synchronization (Waiman Long) [1941211]
- rcuperf: Remove useless while loops around wait_event (Waiman Long) [1941211]
- rcu: Provide rcu_irq_exit_check_preempt() (Waiman Long) [1941211]
- sched,rcu,tracing: Avoid tracing before in_nmi() is correct (Waiman Long) [1941211]
- hardirq/nmi: Allow nested nmi_enter() (Waiman Long) [1941211]
- powerpc/64s: Include <asm/nmi.h> header file to fix a warning (Waiman Long) [1941211]
- powerpc/64s: system reset interrupt preserve HSRRs (Waiman Long) [1941211]
- powerpc/64s: Fix HV NMI vs HV interrupt recoverability test (Waiman Long) [1941211]
- powerpc/traps: fix recoverability of machine check handling on book3s/32 (Waiman Long) [1941211]
- powerpc/traps: restore recoverability of machine_check interrupts (Waiman Long) [1941211]
- rcu: Mark rcu_nmi_enter() call to rcu_cleanup_after_idle() noinstr (Waiman Long) [1941211]
- rcu: Remove initialized but unused rnp from check_slow_task() (Waiman Long) [1941211]
- rcu: Update comment from rsp->rcu_gp_seq to rsp->gp_seq (Waiman Long) [1941211]
- rcu: Provide __rcu_is_watching() (Waiman Long) [1941211]
- doc: Tasks RCU must protect instructions before trampoline (Waiman Long) [1941211]
- fs/btrfs: Add cond_resched() for try_release_extent_mapping() stalls (Waiman Long) [1941211]
- rcu-tasks: Convert sleeps to idle priority (Waiman Long) [1941211]
- rcu: Expedited grace-period sleeps to idle priority (Waiman Long) [1941211]
- rcu: No-CBs-related sleeps to idle priority (Waiman Long) [1941211]
- rcu: Priority-boost-related sleeps to idle priority (Waiman Long) [1941211]
- rcu: Grace-period-kthread related sleeps to idle priority (Waiman Long) [1941211]
- torture: Add --allcpus argument to the kvm.sh script (Waiman Long) [1941211]
- torture: Remove whitespace from identify_qemu_vcpus output (Waiman Long) [1941211]
- torture: Add script to smoke-test commits in a branch (Waiman Long) [1941211]
- trace: events: rcu: Change description of rcu_dyntick trace event (Waiman Long) [1941211]
- rcu: Add comment documenting rcu_callback_map's purpose (Waiman Long) [1941211]
- rcu: Provide rcu_irq_exit_preempt() (Waiman Long) [1941211]
- rcu: Add callbacks-invoked counters (Waiman Long) [1941211]
- doc: Timer problems can cause RCU CPU stall warnings (Waiman Long) [1941211]
- torture:  Remove qemu dependency on EFI firmware (Waiman Long) [1941211]
- rcutorture: NULL rcu_torture_current earlier in cleanup code (Waiman Long) [1941211]
- rcutorture: Handle non-statistic bang-string error messages (Waiman Long) [1941211]
- torture: Set configfile variable to current scenario (Waiman Long) [1941211]
- rcutorture: Add races with task-exit processing (Waiman Long) [1941211]
- docs: RCU: Don't duplicate chapter names in rculist_nulls.rst (Waiman Long) [1941211]
- docs: RCU: Convert stallwarn.txt to ReST (Waiman Long) [1941211]
- docs: RCU: Convert rcuref.txt to ReST (Waiman Long) [1941211]
- docs: RCU: Convert torture.txt to ReST (Waiman Long) [1941211]
- docs: RCU: Convert rculist_nulls.txt to ReST (Waiman Long) [1941211]
- docs: RCU: Convert lockdep.txt to ReST (Waiman Long) [1941211]
- docs: RCU: Convert lockdep-splat.txt to ReST (Waiman Long) [1941211]
- docs: RCU: Convert checklist.txt to ReST (Waiman Long) [1941211]
- doc: Add rcutorture scripting to torture.txt (Waiman Long) [1941211]
- doc: Fix typo "deference" to "dereference" (Waiman Long) [1941211]
- docs: locking: convert docs to ReST and rename to *.rst (Waiman Long) [1941211]
- doc/rcuref: Document real world examples in kernel (Waiman Long) [1941211]
- Documentation/locking/lockdep: Drop last two chars of sample states (Waiman Long) [1941211]
- doc: rcu: Suspicious RCU usage is a warning (Waiman Long) [1941211]
- RCU/torture.txt: Remove section MODULE PARAMETERS (Waiman Long) [1941211]
- Documentation/lockstat: Fix trivial typo (Waiman Long) [1941211]
- rcu: Simplify the calculation of rcu_state.ncpus (Waiman Long) [1941211]
- srcu: Fix a typo in comment "amoritized"->"amortized" (Waiman Long) [1941211]
- rcuperf: Fix printk format warning (Waiman Long) [1941211]
- mm/mmap.c: Add cond_resched() for exit_mmap() CPU stalls (Waiman Long) [1941211]
- rcu: Initialize and destroy rcu_synchronize only when necessary (Waiman Long) [1941211]
- rcu: Add KCSAN stubs to update.c (Waiman Long) [1941211]
- rcu: Add KCSAN stubs (Waiman Long) [1941211]
- srcu: Add KCSAN stubs (Waiman Long) [1941211]
- rcutorture: Add KCSAN stubs (Waiman Long) [1941211]
- rcu/rcutorture: Convert to SPDX license identifier (Waiman Long) [1941211]
- rcu: Convert rcu_nohz_full_cpu() ULONG_CMP_LT() to time_before() (Waiman Long) [1941211]
- rcu: Convert rcu_initiate_boost() ULONG_CMP_GE() to time_after() (Waiman Long) [1941211]
- rcutorture: Convert ULONG_CMP_LT() to time_before() (Waiman Long) [1941211]
- rcu: Convert ULONG_CMP_GE() to time_after() for jiffy comparison (Waiman Long) [1941211]
- rcu: Add rcu_gp_might_be_stalled() (Waiman Long) [1941211]
- torture: Add a --kasan argument (Waiman Long) [1941211]
- torture: Save a few lines by using config_override_param initially (Waiman Long) [1941211]
- torture: Allow scenario-specific Kconfig options to override CFcommon (Waiman Long) [1941211]
- torture: Allow --kconfig options to override --kcsan defaults (Waiman Long) [1941211]
- torture: Abstract application of additional Kconfig options (Waiman Long) [1941211]
- torture: Eliminate duplicate #CHECK# from ConfigFragment (Waiman Long) [1941211]
- torture: Make --kcsan argument also create a summary (Waiman Long) [1941211]
- rcutorture: Make rcu_fwds and rcu_fwd_emergency_stop static (Waiman Long) [1941211]
- torture: Add --kcsan argument to top-level kvm.sh script (Waiman Long) [1941211]
- rcu-tasks: Allow standalone use of TASKS_{TRACE_,}RCU (Waiman Long) [1941211]
- rcu: Remove self-stack-trace when all quiescent states seen (Waiman Long) [1941211]
- ftrace: Use synchronize_rcu_tasks_rude() instead of ftrace_sync() (Waiman Long) [1941211]
- ftrace: Protect ftrace_graph_hash with ftrace_sync (Waiman Long) [1941211]
- ftrace: Add comment to why rcu_dereference_sched() is open coded (Waiman Long) [1941211]
- tracing: Annotate ftrace_graph_notrace_hash pointer with __rcu (Waiman Long) [1941211]
- tracing: Annotate ftrace_graph_hash pointer with __rcu (Waiman Long) [1941211]
- tracing/fgraph: Fix set_graph_function from showing interrupts (Waiman Long) [1941211]
- rcu: Allow rcutorture to starve grace-period kthread (Waiman Long) [1941211]
- rcu: When GP kthread is starved, tag idle threads as false positives (Waiman Long) [1941211]
- rcutorture: Right-size TREE10 CPU consumption (Waiman Long) [1941211]
- rcu: Replace 1 by true (Waiman Long) [1941211]
- rcutorture: Add test of holding scheduler locks across rcu_read_unlock() (Waiman Long) [1941211]
- rcu: Replace assigned pointer ret value by corresponding boolean value (Waiman Long) [1941211]
- rcu-tasks: Add IPI failure count to statistics (Waiman Long) [1941211]
- rcutorture: Add TRACE02 scenario enabling RCU Tasks Trace IPIs (Waiman Long) [1941211]
- rcu-tasks: Add count for idle tasks on offline CPUs (Waiman Long) [1941211]
- rcu-tasks: Add rcu_dynticks_zero_in_eqs() effectiveness statistics (Waiman Long) [1941211]
- rcu-tasks: Make RCU tasks trace also wait for idle tasks (Waiman Long) [1941211]
- rcu-tasks: Handle the running-offline idle-task special case (Waiman Long) [1941211]
- rcu-tasks: Disable CPU hotplug across RCU tasks trace scans (Waiman Long) [1941211]
- rcu: Mark rcu_state.gp_seq to detect more concurrent writes (Waiman Long) [1941211]
- rcu-tasks: Allow rcu_read_unlock_trace() under scheduler locks (Waiman Long) [1941211]
- rcu-tasks: Avoid IPIing userspace/idle tasks if kernel is so built (Waiman Long) [1941211]
- rcu-tasks: Add Kconfig option to mediate smp_mb() vs. IPI (Waiman Long) [1941211]
- rcu-tasks: Add grace-period and IPI counts to statistics (Waiman Long) [1941211]
- rcu-tasks: Split ->trc_reader_need_end (Waiman Long) [1941211]
- rcu-tasks: Provide boot parameter to delay IPIs until late in grace period (Waiman Long) [1941211]
- rcu-tasks: Add a grace-period start time for throttling and debug (Waiman Long) [1941211]
- rcu: Get rid of some doc warnings in update.c (Waiman Long) [1941211]
- rcu-tasks: Make RCU Tasks Trace make use of RCU scheduler hooks (Waiman Long) [1941211]
- rcu-tasks: Use context-switch hook for PREEMPT=y kernels (Waiman Long) [1941211]
- rcu-tasks: Make rcutorture writer stall output include GP state (Waiman Long) [1941211]
- rcu-tasks: Add RCU tasks to rcutorture writer stall output (Waiman Long) [1941211]
- rcuperf: Add ability to increase object allocation size (Waiman Long) [1941211]
- rcu-tasks: Move #ifdef into tasks.h (Waiman Long) [1941211]
- rcutorture: Add flag to produce non-busy-wait task stalls (Waiman Long) [1941211]
- rcu: Add per-task state to RCU CPU stall warnings (Waiman Long) [1941211]
- sched/core: Add function to sample state of locked-down task (Waiman Long) [1941211]
- rcu-tasks: Add stall warnings for RCU Tasks Trace (Waiman Long) [1941211]
- rcutorture: Add torture tests for RCU Tasks Trace (Waiman Long) [1941211]
- rcu-tasks: Add an RCU Tasks Trace to simplify protection of tracing hooks (Waiman Long) [1941211]
- rcutorture: Add a test for synchronize_rcu_mult() (Waiman Long) [1941211]
- rcu: Reinstate synchronize_rcu_mult() (Waiman Long) [1941211]
- rcu-tasks: Code movement to allow more Tasks RCU variants (Waiman Long) [1941211]
- rcu: Fix the (t=0 jiffies) false positive (Waiman Long) [1941211]
- rcu-tasks: Further refactor RCU-tasks to allow adding more variants (Waiman Long) [1941211]
- rcu-tasks: Use unique names for RCU-Tasks kthreads and messages (Waiman Long) [1941211]
- rcutorture: Add torture tests for RCU Tasks Rude (Waiman Long) [1941211]
- rcu-tasks: Add an RCU-tasks rude variant (Waiman Long) [1941211]
- rcu-tasks: Refactor RCU-tasks to allow variants to be added (Waiman Long) [1941211]
- rcu-tasks: Create struct to hold state information (Waiman Long) [1941211]
- rcu-tasks: Move Tasks RCU to its own file (Waiman Long) [1941211]
- locktorture.c: Fix if-statement empty body warnings (Waiman Long) [1941211]
- rcu: Expedite first two FQS scans under callback-overload conditions (Waiman Long) [1941211]
- rcu: Mark rcu_state.gp_seq to detect concurrent writes (Waiman Long) [1941211]
- rcutorture: Mark data-race potential for rcu_barrier() test statistics (Waiman Long) [1941211]
- rcu: Don't use negative nesting depth in __rcu_read_unlock() (Waiman Long) [1941211]
- rcu: Remove unused ->rcu_read_unlock_special.b.deferred_qs field (Waiman Long) [1941211]
- rcu: Don't set nesting depth negative in rcu_preempt_deferred_qs() (Waiman Long) [1941211]
- rcu: Make rcu_read_unlock_special() safe for rq/pi locks (Waiman Long) [1941211]
- rcutorture: Make kvm-recheck-rcu.sh handle truncated lines (Waiman Long) [1941211]
- rcu: Make RCU IRQ enter/exit functions rely on in_nmi() (Waiman Long) [1941211]
- rcu/tree: Mark the idle relevant functions noinstr (Waiman Long) [1941211]
- rcu: Add comments marking transitions between RCU watching and not (Waiman Long) [1941211]
- rcu: Update __call_rcu() comments (Waiman Long) [1941211]
- rcu: Mark rcu_state.ncpus to detect concurrent writes (Waiman Long) [1941211]
- rcu: Use data_race() for RCU CPU stall-warning prints (Waiman Long) [1941211]
- rcutorture: Manually clean up after rcu_barrier() failure (Waiman Long) [1941211]
- rcutorture: Make rcu_torture_barrier_cbs() post from corresponding CPU (Waiman Long) [1941211]
- rcu: Make nocb_gp_wait() double-check unexpected-callback warning (Waiman Long) [1941211]
- rcu: Tighten rcu_lockdep_assert_cblist_protected() check (Waiman Long) [1941211]
- rcu: Optimize and protect atomic_cmpxchg() loop (Waiman Long) [1941211]
- rcu/nocb: Add missing annotation for rcu_nocb_bypass_unlock() (Waiman Long) [1941211]
- rcu: Don't flag non-starting GPs before GP kthread is running (Waiman Long) [1941211]
- rcu: Fix rcu_barrier_callback() race condition (Waiman Long) [1941211]
- rcu: Add WRITE_ONCE() to rcu_state ->gp_start (Waiman Long) [1941211]
- rcu: Add missing annotation for rcu_nocb_bypass_lock() (Waiman Long) [1941211]
- rcu: Add missing annotation for exit_tasks_rcu_finish() (Waiman Long) [1941211]
- rcu: Add missing annotation for exit_tasks_rcu_start() (Waiman Long) [1941211]
- rculist: Add brackets around cond argument in __list_check_rcu macro (Waiman Long) [1941211]
- rcu: Remove dead code from rcu_segcblist_insert_pend_cbs() (Waiman Long) [1941211]
- rcutorture: Annotation lockless accesses to rcu_torture_current (Waiman Long) [1941211]
- rcu: Add READ_ONCE() to rcu_data ->gpwrap (Waiman Long) [1941211]
- rcu-tasks: *_ONCE() for rcu_tasks_cbs_head (Waiman Long) [1941211]
- rcu: Add *_ONCE() for grace-period progress indicators (Waiman Long) [1941211]
- rcu: Add WRITE_ONCE() to rcu_node ->boost_tasks (Waiman Long) [1941211]
- srcu: Add data_race() to ->srcu_lock_count and ->srcu_unlock_count arrays (Waiman Long) [1941211]
- rcutorture: Add READ_ONCE() to rcu_torture_count and rcu_torture_batch (Waiman Long) [1941211]
- rcu: Add READ_ONCE() to rcu_segcblist ->tails[] (Waiman Long) [1941211]
- rcu: Add WRITE_ONCE() to rcu_node ->qsmaskinitnext (Waiman Long) [1941211]
- rcu: Add READ_ONCE and data_race() to rcu_node ->boost_tasks (Waiman Long) [1941211]
- rcu: Add WRITE_ONCE() to rcu_state ->gp_req_activity (Waiman Long) [1941211]
- rcu: Add READ_ONCE() to rcu_node ->gp_seq (Waiman Long) [1941211]
- rcu: Add *_ONCE() and data_race() to rcu_node ->exp_tasks plus locking (Waiman Long) [1941211]
- rcu: Add *_ONCE() to rcu_node ->boost_kthread_status (Waiman Long) [1941211]
- rcu: Add WRITE_ONCE to rcu_node ->exp_seq_rq store (Waiman Long) [1941211]
- srcu: Hold srcu_struct ->lock when updating ->srcu_gp_seq (Waiman Long) [1941211]
- rcu: Add WRITE_ONCE() to rcu_node ->qsmask update (Waiman Long) [1941211]
- rcu: Provide debug symbols and line numbers in KCSAN runs (Waiman Long) [1941211]
- rcu: Fix exp_funnel_lock()/rcu_exp_wait_wake() datarace (Waiman Long) [1941211]
- srcu: Fix process_srcu()/srcu_batches_completed() datarace (Waiman Long) [1941211]
- srcu: Fix __call_srcu()/srcu_get_delay() datarace (Waiman Long) [1941211]
- srcu: Fix __call_srcu()/process_srcu() datarace (Waiman Long) [1941211]
- rcutorture: Fix stray access to rcu_fwd_cb_nodelay (Waiman Long) [1941211]
- rcutorture: Fix rcu_torture_one_read()/rcu_torture_writer() data race (Waiman Long) [1941211]
- rcutorture: Make kvm-find-errors.sh abort on bad directory (Waiman Long) [1941211]
- rcutorture: Summarize summary of build and run results (Waiman Long) [1941211]
- rcutorture: Add 100-CPU configuration (Waiman Long) [1941211]
- rcu: Warn on for_each_leaf_node_cpu_mask() from non-leaf (Waiman Long) [1941211]
- rcu: Fix spelling mistake "leval" -> "level" (Waiman Long) [1941211]
- torture: Allow disabling of boottime CPU-hotplug torture operations (Waiman Long) [1941211]
- rcutorture: Suppress boottime bad-sequence warnings (Waiman Long) [1941211]
- rcutorture: Allow boottime stall warnings to be suppressed (Waiman Long) [1941211]
- torture: Forgive -EBUSY from boottime CPU-hotplug operations (Waiman Long) [1941211]
- rcutorture: Refrain from callback flooding during boot (Waiman Long) [1941211]
- torture: Make results-directory date format completion-friendly (Waiman Long) [1941211]
- rcutorture: Suppress forward-progress complaints during early boot (Waiman Long) [1941211]
- timer: Use hlist_unhashed_lockless() in timer_pending() (Waiman Long) [1941211]
- rcu: React to callback overload by boosting RCU readers (Waiman Long) [1941211]
- rcu: React to callback overload by aggressively seeking quiescent states (Waiman Long) [1941211]
- rcu: Add and update docbook header comments in list.h (Waiman Long) [1941211]
- list: Add hlist_unhashed_lockless() (Waiman Long) [1941211]
- list: Don't use WRITE_ONCE() in hlist_add_behind() (Waiman Long) [1941211]
- kcsan: Introduce ASSERT_EXCLUSIVE_BITS(var, mask) (Waiman Long) [1941211]
- kcsan: Add kcsan_set_access_mask() support (Waiman Long) [1941211]
- kcsan: Introduce kcsan_value_change type (Waiman Long) [1941211]
- kcsan: Move interfaces that affects checks to kcsan-checks.h (Waiman Long) [1941211]
- kcsan: Fix misreporting if concurrent races on same address (Waiman Long) [1941211]
- kcsan: Expose core configuration parameters as module params (Waiman Long) [1941211]
- kcsan: Add test to generate conflicts via debugfs (Waiman Long) [1941211]
- kcsan: Introduce ASSERT_EXCLUSIVE_*() macros (Waiman Long) [1941211]
- kcsan: Introduce KCSAN_ACCESS_ASSERT access type (Waiman Long) [1941211]
- kcsan: Fix 0-sized checks (Waiman Long) [1941211]
- kcsan: Clean up the main KCSAN Kconfig option (Waiman Long) [1941211]
- kcsan: Clarify Kconfig option KCSAN_IGNORE_ATOMICS (Waiman Long) [1941211]
- kcsan: Add option to assume plain aligned writes up to word size are atomic (Waiman Long) [1941211]
- kcsan: Add docbook header for data_race() (Waiman Long) [1941211]
- kcsan: Address missing case with KCSAN_REPORT_VALUE_CHANGE_ONLY (Waiman Long) [1941211]
- include/linux: Add instrumented.h infrastructure (Waiman Long) [1941211]
- kcsan: Make KCSAN compatible with lockdep (Waiman Long) [1941211]
- kcsan: Rate-limit reporting per data races (Waiman Long) [1941211]
- kcsan: Show full access type in report (Waiman Long) [1941211]
- kcsan: Prefer __always_inline for fast-path (Waiman Long) [1941211]
- kcsan: Improve various small stylistic details (Waiman Long) [1941211]
- x86, kcsan: Enable KCSAN for x86 (Waiman Long) [1941211]
- build, kcsan: Add KCSAN build exceptions (Waiman Long) [1941211]
- x86/uaccess, kcov: Disable stack protector (Waiman Long) [1941211]
- doc: Convert to rcubarrier.txt to ReST (Waiman Long) [1941211]
- doc: Convert to rcu_dereference.txt to rcu_dereference.rst (Waiman Long) [1941211]
- doc: Convert whatisRCU.txt to .rst (Waiman Long) [1941211]
- doc: Converted NMI-RCU.txt to NMI-RCU.rst. (Waiman Long) [1941211]
- doc: Convert arrayRCU.txt to arrayRCU.rst (Waiman Long) [1941211]
- doc: Update list_for_each_entry_rcu() documentation (Waiman Long) [1941211]
- Restore docs "rcu: Restore barrier() to rcu_read_lock() and rcu_read_unlock()" (Waiman Long) [1941211]
- Restore docs "treewide: Rename rcu_dereference_raw_notrace() to _check()" (Waiman Long) [1941211]
- docs: rcu: Increase toctree to 3 (Waiman Long) [1941211]
- docs: rcu: Correct links referring to titles (Waiman Long) [1941211]
- docs: rcu: convert some articles from html to ReST (Waiman Long) [1941211]
- Revert docs from "rcu: Restore barrier() to rcu_read_lock() and rcu_read_unlock()" (Waiman Long) [1941211]
- Documentation: RCU: Add TOC tree hooks (Waiman Long) [1941211]
- doc: Describe choice of rcu_dereference() APIs and __rcu usage (Waiman Long) [1941211]
- doc: Remove obsolete RCU update functions from RCU documentation (Waiman Long) [1941211]
- scsi: ibmvfc: Free channel_setup_buf during device tear down (Steve Best) [1938102]
- blk-mq: clearing flush request reference in tags->rqs[] (Ming Lei) [1811030]
- blk-mq: clear stale request in tags->rq[] before freeing one request pool (Ming Lei) [1811030]
- blk-mq: grab rq->refcount before calling ->fn in blk_mq_tagset_busy_iter (Ming Lei) [1811030]
- vsock/vmci: log once the failed queue pair allocation (Stefano Garzarella) [1958248]
- VMCI: Stop log spew when qp allocation isn't possible (Stefano Garzarella) [1958248]
- vsock/virtio: update credit only if socket is not closed (Stefano Garzarella) [1921694]
- powerpc/security: Fix debugfs data leak on 32-bit (Steve Best) [1935661]
- powerpc/security: Show powerpc_security_features in debugfs (Steve Best) [1935661]
- redhat/configs: Add CONFIG_SYSTEM_REVOCATION_KEYS and CONFIG_SYSTEM_REVOCATION_LIST (Vladis Dronov) [1893793] {CVE-2020-26541}
- certs: add 'x509_revocation_list' to gitignore (Vladis Dronov) [1893793] {CVE-2020-26541}
- integrity: Load mokx variables into the blacklist keyring (Vladis Dronov) [1893793] {CVE-2020-26541}
- certs: Add ability to preload revocation certs (Vladis Dronov) [1893793] {CVE-2020-26541}
- certs: Move load_system_certificate_list to a common function (Vladis Dronov) [1893793] {CVE-2020-26541}
- certs: Add EFI_CERT_X509_GUID support for dbx entries (Vladis Dronov) [1893793] {CVE-2020-26541}
- pNFS/flexfiles: Fix array overflow when flexfiles mirroring is enabled (Benjamin Coddington) [1949575]
- net/sunrpc: fix useless comparison in proc_do_xprt() (Benjamin Coddington) [1949575]
- net/sunrpc: return 0 on attempt to write to "transports" (Benjamin Coddington) [1949575]
- NFSD: MKNOD should return NFSERR_BADTYPE instead of NFSERR_INVAL (Benjamin Coddington) [1949575]
- NFSD: NFSv3 PATHCONF Reply is improperly formed (Benjamin Coddington) [1949575]
- nfsroot: Default mount option should ask for built-in NFS version (Benjamin Coddington) [1949575]
- nfsd: remove unneeded break (Benjamin Coddington) [1949575]
- net/sunrpc: Fix return value for sysctl sunrpc.transports (Benjamin Coddington) [1949575]
- NFSv4.2: support EXCHGID4_FLAG_SUPP_FENCE_OPS 4.2 EXCHANGE_ID flag (Benjamin Coddington) [1949575]
- NFSv4: Fix up RCU annotations for struct nfs_netns_client (Benjamin Coddington) [1949575]
- NFS: Only reference user namespace from nfs4idmap struct instead of cred (Benjamin Coddington) [1949575]
- NFSD: Hoist status code encoding into XDR encoder functions (Benjamin Coddington) [1949575]
- NFSv4: Use the net namespace uniquifier if it is set (Benjamin Coddington) [1949575]
- NFSv4: Clean up initialisation of uniquified client id strings (Benjamin Coddington) [1949575]
- SUNRPC: Add an xdr_align_data() function (Benjamin Coddington) [1949575]
- SUNRPC: Add the ability to expand holes in data pages (Benjamin Coddington) [1949575]
- SUNRPC: Split out _shift_data_right_tail() (Benjamin Coddington) [1949575]
- SUNRPC: Split out xdr_realign_pages() from xdr_align_pages() (Benjamin Coddington) [1949575]
- NFS: Use xdr_page_pos() in NFSv4 decode_getacl() (Benjamin Coddington) [1949575]
- SUNRPC: Implement a xdr_page_pos() function (Benjamin Coddington) [1949575]
- NFS: fix nfs_path in case of a rename retry (Benjamin Coddington) [1949575]
- NFSD: Map nfserr_wrongsec outside of nfsd_dispatch (Benjamin Coddington) [1949575]
- NFSD: Remove the RETURN_STATUS() macro (Benjamin Coddington) [1949575]
- NFSD: Call NFSv2 encoders on error returns (Benjamin Coddington) [1949575]
- NFSD: Fix .pc_release method for NFSv2 (Benjamin Coddington) [1949575]
- NFSD: Remove vestigial typedefs (Benjamin Coddington) [1949575]
- NFSD: Refactor nfsd_dispatch() error paths (Benjamin Coddington) [1949575]
- NFSD: Clean up nfsd_dispatch() variables (Benjamin Coddington) [1949575]
- NFSD: Clean up stale comments in nfsd_dispatch() (Benjamin Coddington) [1949575]
- NFSD: Clean up switch statement in nfsd_dispatch() (Benjamin Coddington) [1949575]
- NFSD: Encoder and decoder functions are always present (Benjamin Coddington) [1949575]
- NFSACL: Replace PROC() macro with open code (Benjamin Coddington) [1949575]
- lockd: Replace PROC() macro with open code (Benjamin Coddington) [1949575]
- NFSD: Add missing NFSv2 .pc_func methods (Benjamin Coddington) [1949575]
- fs: nfs: return per memcg count for xattr shrinkers (Benjamin Coddington) [1949575]
- nfs: remove incorrect fallthrough label (Benjamin Coddington) [1949575]
- nfsd: rq_lease_breaker cleanup (Benjamin Coddington) [1949575]
- sunrpc: simplify do_cache_clean (Benjamin Coddington) [1949575]
- sunrpc: cache : Replace seq_printf with seq_puts (Benjamin Coddington) [1949575]
- silence nfscache allocation warnings with kvzalloc (Benjamin Coddington) [1949575]
- nfsd: fix comparison to bool warning (Benjamin Coddington) [1949575]
- NFSD: Correct type annotations in user xattr XDR functions (Benjamin Coddington) [1949575]
- NFSD: Correct type annotations in user xattr helpers (Benjamin Coddington) [1949575]
- SUNRPC/NFSD: Implement xdr_reserve_space_vec() (Benjamin Coddington) [1949575]
- nfsd: rename delegation related tracepoints to make them less confusing (Benjamin Coddington) [1949575]
- nfsd: Remove unnecessary assignment in nfs4xdr.c (Benjamin Coddington) [1949575]
- net: sunrpc: delete repeated words (Benjamin Coddington) [1949575]
- nfsd: Fix typo in comment (Benjamin Coddington) [1949575]
- nfsd: give up callbacks on revoked delegations (Benjamin Coddington) [1949575]
- nfs: fix spellint typo in pnfs.c (Benjamin Coddington) [1949575]
- fs,nfs: lift compat nfs4 mount data handling into the nfs code (Benjamin Coddington) [1949575]
- nfs: simplify nfs4_parse_monolithic (Benjamin Coddington) [1949575]
- xprtrdma: drop double zeroing (Benjamin Coddington) [1949575]
- NFSv4.2: xattr cache: remove unused cache struct field (Benjamin Coddington) [1949575]
- nfs: Convert to use the preferred fallthrough macro (Benjamin Coddington) [1949575]
- NFS4: Fix oops when copy_file_range is attempted with NFS4.0 source (Benjamin Coddington) [1949575]
- Replace HTTP links with HTTPS ones: NFS, SUNRPC, and LOCKD clients (Benjamin Coddington) [1949575]
- sunrpc: fix duplicated word in <linux/sunrpc/cache.h> (Benjamin Coddington) [1949575]
- SUNRPC: Remove remaining dprintks from sched.c (Benjamin Coddington) [1949575]
- SUNRPC: Remove dprintk call sites in RPC queuing functions (Benjamin Coddington) [1949575]
- SUNRPC: Clean up RPC scheduler tracepoints (Benjamin Coddington) [1949575]
- SUNRPC: Replace rpcbind dprintk call sites with tracepoints (Benjamin Coddington) [1949575]
- SUNRPC: Remove more dprintks in rpcb_clnt.c (Benjamin Coddington) [1949575]
- SUNRPC: Remove dprintk call sites in rpcbind XDR functions (Benjamin Coddington) [1949575]
- SUNRPC: Hoist trace_xprtrdma_op_setport into generic code (Benjamin Coddington) [1949575]
- SUNRPC: Remove rpcb_getport_async dprintk call sites (Benjamin Coddington) [1949575]
- SUNRPC: Clean up call_bind_status() observability (Benjamin Coddington) [1949575]
- SUNRPC: Remove dprintk call site in call_decode (Benjamin Coddington) [1949575]
- SUNRPC: Trace call_refresh events (Benjamin Coddington) [1949575]
- SUNRPC: Add trace_rpc_timeout_status() (Benjamin Coddington) [1949575]
- SUNRPC: Mitigate cond_resched() in xprt_transmit() (Benjamin Coddington) [1949575]
- SUNRPC: Replace connect dprintk call sites with a tracepoint (Benjamin Coddington) [1949575]
- SUNRPC: Remove dprintk call site in call_start() (Benjamin Coddington) [1949575]
- SUNRPC: Remove the dprint_status() macro (Benjamin Coddington) [1949575]
- SUNRPC: Replace dprintk() call site in xs_nospace() (Benjamin Coddington) [1949575]
- SUNRPC: Replace dprintk() call site in xprt_prepare_transmit (Benjamin Coddington) [1949575]
- SUNRPC: Update debugging instrumentation in xprt_do_reserve() (Benjamin Coddington) [1949575]
- SUNRPC: Remove debugging instrumentation from xprt_release (Benjamin Coddington) [1949575]
- SUNRPC: Hoist trace_xprtrdma_op_allocate into generic code (Benjamin Coddington) [1949575]
- SUNRPC: Remove trace_xprt_complete_rqst() (Benjamin Coddington) [1949575]
- SUNRPC dont update timeout value on connection reset (Benjamin Coddington) [1949575]
- nfs4: strengthen error check to avoid unexpected result (Benjamin Coddington) [1949575]
- NFS: remove redundant pointer clnt (Benjamin Coddington) [1949575]
- sunrpc: Avoid comma separated statements (Benjamin Coddington) [1949575]

* Tue May 25 2021 Bruno Meneguele <bmeneg@redhat.com> [4.18.0-309.el8]
- Revert "sctp: Fix SHUTDOWN CTSN Ack in the peer restart case" (Xin Long) [1953839]
- sctp: do asoc update earlier in sctp_sf_do_dupcook_b (Xin Long) [1953839]
- sctp: do asoc update earlier in sctp_sf_do_dupcook_a (Xin Long) [1953839]
- NFSv4: Fixes for nfs4_bitmask_adjust() (Steve Dickson) [1926961]
- powerpc/paravirt: Use is_kvm_guest() in vcpu_is_preempted() (Desnes A. Nunes do Rosario) [1918720]
- powerpc: Reintroduce is_kvm_guest() as a fast-path check (Desnes A. Nunes do Rosario) [1918720]
- powerpc: Rename is_kvm_guest() to check_kvm_guest() (Desnes A. Nunes do Rosario) [1918720]
- powerpc: Refactor is_kvm_guest() declaration to new header (Desnes A. Nunes do Rosario) [1918720]
- powerpc/pseries: Move some PAPR paravirt functions to their own file (Desnes A. Nunes do Rosario) [1918720]
- powerpc/shared: Use static key to detect shared processor (Desnes A. Nunes do Rosario) [1918720]
- powerpc/spinlocks: Fix oops in __spin_yield() on bare metal (Desnes A. Nunes do Rosario) [1918720]
- powerpc/spinlocks: Rename SPLPAR-only spinlocks (Desnes A. Nunes do Rosario) [1918720]
- powerpc/spinlocks: Refactor SHARED_PROCESSOR (Desnes A. Nunes do Rosario) [1918720]
- Revert "ACPICA: Interpreter: fix memory leak by using existing buffer" (Mark Langsdorf) [1935928]
- x86/efi: Remove EFI PGD build time checks (Mark Langsdorf) [1935928]
- ACPI: scan: Fix battery devices sometimes never binding (Mark Langsdorf) [1935928]
- ACPI/IORT: Do not blindly trust DMA masks from firmware (Mark Langsdorf) [1935928]
- ACPI: thermal: Do not call acpi_thermal_check() directly (Mark Langsdorf) [1935928]
- ACPI: sysfs: Prefer "compatible" modalias (Mark Langsdorf) [1935928]
- ACPI: scan: Make acpi_bus_get_device() clear return pointer on error (Mark Langsdorf) [1935928]
- ACPI: scan: Harden acpi_device_add() against device ID overflows (Mark Langsdorf) [1935928]
- ACPI: Update Kconfig help text for items that are no longer modular (Mark Langsdorf) [1935928]
- ACPI: PM: s2idle: Drop unused local variables and related code (Mark Langsdorf) [1935928]
- ACPI: PM: s2idle: Move x86-specific code to the x86 directory (Mark Langsdorf) [1935928]
- ACPI: scan: Add Intel Baytrail Mailbox Device to acpi_ignore_dep_ids (Mark Langsdorf) [1935928]
- ACPI: scan: Avoid unnecessary second pass in acpi_bus_scan() (Mark Langsdorf) [1935928]
- ACPI: scan: Defer enumeration of devices with _DEP lists (Mark Langsdorf) [1935928]
- ACPI: scan: Evaluate _DEP before adding the device (Mark Langsdorf) [1935928]
- ACPI: PM: s2idle: Add AMD support to handle _DSM (Mark Langsdorf) [1935928]
- ACPI: processor: fix