(conditional)=

# Conditionals

```{eval-rst}
.. tags:: Intermediate
```

Flytekit elevates conditions to a first-class construct named `conditional`, providing a powerful mechanism for selectively
executing branches in a workflow. Conditions leverage static or dynamic data generated by tasks or
received as workflow inputs. While conditions are highly performant in their evaluation,
it's important to note that they are restricted to specific binary and logical operators
and are applicable only to primitive values.

```{note}
To clone and run the example code on this page, see the [Flytesnacks repo][flytesnacks].
```

To begin, import the necessary libraries.

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:lines: 1-4
```

## Simple branch

In this example, we introduce two tasks, `calculate_circle_circumference` and
`calculate_circle_area`. The workflow dynamically chooses between these tasks based on whether the input
falls within the fraction range (0-1) or not.

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:lines: 12-38
```

## Multiple branches

We establish an `if` condition with multiple branches, which will result in a failure if none of the conditions is met.
It's important to note that any `conditional` statement in Flyte is expected to be complete,
meaning that all possible branches must be accounted for.

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:pyobject: shape_properties_with_multiple_branches
```

:::{note}
Take note of the usage of bitwise operators (`&`). Due to Python's PEP-335,
the logical `and`, `or` and `not` operators cannot be overloaded.
Flytekit employs bitwise `&` and `|` as equivalents for logical `and` and `or` operators,
a convention also observed in other libraries.
:::

## Consuming the output of a conditional
Here, we write a task that consumes the output returned by a `conditional`.

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:lines: 67-85
```

## Using the output of a previous task in a conditional

You can check if a boolean returned from the previous task is `True`,
but unary operations are not supported directly. Instead, use the `is_true`,
`is_false` and `is_none` methods on the result.

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:lines: 93-123
```

:::{note}
*How do output values acquire these methods?* In a workflow, direct access to outputs is not permitted.
Inputs and outputs are automatically encapsulated in a special object known as {py:class}`flytekit.extend.Promise`.
:::

## Using boolean workflow inputs in a conditional
You can directly pass a boolean to a workflow.

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:pyobject: boolean_input_wf
```

:::{note}
Observe that the passed boolean possesses a method called `is_true`.
This boolean resides within the workflow context and is encapsulated in a specialized Flytekit object.
This special object enables it to exhibit additional behavior.
:::

You can run the workflows locally as follows:

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:lines: 133-139
```

## Nested conditionals

You can nest conditional sections arbitrarily inside other conditional sections.
However, these nested sections can only be in the `then` part of a `conditional` block.

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:lines: 146-168
```

## Using the output of a task in a conditional

Let's write a fun workflow that triggers the `calculate_circle_circumference` task in the event of a "heads" outcome,
and alternatively, runs the `calculate_circle_area` task in the event of a "tail" outcome.

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:pyobject: consume_task_output
```

You can run the workflow locally as follows:

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:lines: 216-225
```

## Running a noop task in a conditional

In some cases, you may want to skip the execution of a conditional workflow if a certain condition is not met.
You can achieve this by using the `echo` task, which simply returns the input value.

:::{note}
To enable the echo plugin in the backend, add the plugin to Flyte's configuration file.
```yaml
task-plugins:
  enabled-plugins:
    - echo
```
:::

```{literalinclude} /examples/advanced_composition/advanced_composition/conditional.py
:caption: advanced_composition/conditional.py
:lines: 200-212
```

## Run the example on the Flyte cluster

To run the provided workflows on the Flyte cluster, use the following commands:

```
pyflyte run --remote \
  https://raw.githubusercontent.com/flyteorg/flytesnacks/656e63d1c8dded3e9e7161c7af6425e9fcd43f56/examples/advanced_composition/advanced_composition/conditional.py \
  shape_properties --radius 3.0
```

```
pyflyte run --remote \
  https://raw.githubusercontent.com/flyteorg/flytesnacks/656e63d1c8dded3e9e7161c7af6425e9fcd43f56/examples/advanced_composition/advanced_composition/conditional.py \
  shape_properties_with_multiple_branches --radius 11.0
```

```
pyflyte run --remote \
  https://raw.githubusercontent.com/flyteorg/flytesnacks/656e63d1c8dded3e9e7161c7af6425e9fcd43f56/examples/advanced_composition/advanced_composition/conditional.py \
  shape_properties_accept_conditional_output --radius 0.5
```

```
pyflyte run --remote \
  https://raw.githubusercontent.com/flyteorg/flytesnacks/656e63d1c8dded3e9e7161c7af6425e9fcd43f56/examples/advanced_composition/advanced_composition/conditional.py \
  boolean_wf
```

```
pyflyte run --remote \
  https://raw.githubusercontent.com/flyteorg/flytesnacks/656e63d1c8dded3e9e7161c7af6425e9fcd43f56/examples/advanced_composition/advanced_composition/conditional.py \
  boolean_input_wf --boolean_input
```

```
pyflyte run --remote \
  https://raw.githubusercontent.com/flyteorg/flytesnacks/656e63d1c8dded3e9e7161c7af6425e9fcd43f56/examples/advanced_composition/advanced_composition/conditional.py \
  nested_conditions --radius 0.7
```

```
pyflyte run --remote \
  https://raw.githubusercontent.com/flyteorg/flytesnacks/656e63d1c8dded3e9e7161c7af6425e9fcd43f56/examples/advanced_composition/advanced_composition/conditional.py \
  consume_task_output --radius 0.4 --seed 7
```

```
pyflyte run --remote \
  https://raw.githubusercontent.com/flyteorg/flytesnacks/656e63d1c8dded3e9e7161c7af6425e9fcd43f56/examples/advanced_composition/advanced_composition/conditional.py \
  noop_in_conditional --radius 0.4 --seed 5
```

[flytesnacks]: https://github.com/flyteorg/flytesnacks/tree/master/examples/advanced_composition
