package sandbox

import (
	"fmt"

	"github.com/flyteorg/flyte/flytectl/pkg/docker"
)

// Config holds configuration flags for sandbox command.
type Config struct {
	DeprecatedSource string `json:"source" pflag:",deprecated, path of your source code, please build images with local daemon"`

	// Flytectl sandbox only supports Flyte version available in Github release https://github.com/flyteorg/flyte/tags.
	// Flytectl sandbox will only work for v0.10.0+.
	// Default value dind represents the latest release.
	Version string `json:"version" pflag:",Version of flyte. Only supports flyte releases greater than v0.10.0"`

	// Optionally it is possible to specify a specific fqn for the docker image with the tag. This should be
	// Flyte compliant sandbox image. Usually useful, if you want to push the image to your own registry and relaunch
	// from there.
	Image string `json:"image" pflag:",Optional. Provide a fully qualified path to a Flyte compliant docker image."`

	// Default value false represents that Flytectl will not use the latest pre-release if it exists.
	Prerelease bool `json:"pre" pflag:",Optional. Pre release Version of flyte will be used for sandbox."`

	// Agent Service
	DisableAgent bool `json:"disable-agent" pflag:",Optional. Disable the agent service."`

	// Optionally it is possible to pass in environment variables to sandbox container.
	Env []string `json:"env" pflag:",Optional. Provide Env variable in key=value format which can be passed to sandbox container."`

	// Optionally it is possible to use local sandbox image
	// Flytectl will not pull the image from the registry if the local flag passes. It is usually useful while testing your local images without pushing them to a registry.
	ImagePullPolicy docker.ImagePullPolicy `json:"imagePullPolicy" pflag:",Optional. Defines the image pull behavior [Always/IfNotPresent/Never]"`

	ImagePullOptions docker.ImagePullOptions `json:"imagePullOptions" pflag:",Optional. Defines image pull options (e.g. auth)"`

	// It's used for development. Users are able to start flyte locally via single binary and save the data to the minio or postgres in the sandbox.
	Dev    bool `json:"dev" pflag:",Optional. Only start minio and postgres in the sandbox."`
	DryRun bool `json:"dryRun" pflag:",Optional. Only print the docker commands to bring up flyte sandbox/demo container.This will still call github api's to get the latest flyte release to use'"`

	Force bool `json:"force" pflag:",Optional. Forcefully delete existing sandbox cluster if it exists."`

	// Allow user to specify the port for the sandbox
	Port string `json:"port" pflag:",Optional. Specify the port for the Kubernetes in the sandbox."`
}

//go:generate pflags Config --default-var DefaultConfig --bind-default-var
var (
	DefaultConfig = &Config{
		Port: "6443", // Default port for the sandbox
	}
)

func (c Config) GetK8sEndpoint() string {
	return fmt.Sprintf("https://127.0.0.1:%s", c.Port)
}
